##############################################################################
#
# Copyright (c) 2002 Ingeniweb SARL
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE
#
##############################################################################

"""
Plone Article core classes for images and attachments and layout models
"""

from AccessControl import ClassSecurityInfo
import string
import random
import os
from Products.CMFCore import CMFCorePermissions
from AccessControl import Permissions, getSecurityManager, ClassSecurityInfo, Unauthorized
from OFS import Folder, Image, ObjectManager
from Products.ZAttachmentAttribute import ZAttachmentAttribute
from global_symbols import ArticleCore_editPermission
from Products.CMFCore.utils import getToolByName
from Products.PythonScripts.standard import url_quote


class ArticleAttachments(ObjectManager.ObjectManager):
    """
    ArticleAttachments => Base class for attachment features
    """
    # Standard security settings
    security = ClassSecurityInfo()
    security.declareObjectProtected(CMFCorePermissions.View)            # $$$ Is this clever ? Isn't it better to make the object private ?
                                                                        # This method makes all class properties & methods protected by View by default
    def __init__(self, ):
        """
        __init__(self,) => init method
        """
        self.attachments_ids = []               # Attachments list

    security.declareProtected(ArticleCore_editPermission, 'removeAttachment')
    def removeAttachment(self, index = 0, REQUEST = {}):
        """
        removeAttachment(self, index = 0, REQUEST = {}) 
        - remove the attachment from the list
        """
        # Check lock status
        self.checkLockStatus()

        # Ensure the index is reachable
        if index >= len(self.attachments_ids):
            return
        
        # Shorten the list
        id = self.attachments_ids[index]
        del self.attachments_ids[index]
        self.attachments_ids = self.attachments_ids             # Ensure persistancy
        self.manage_delObjects(ids = [id],)

        # Reindex catalog
        self.reindexObject()

    security.declareProtected(ArticleCore_editPermission, 'removeAttachments')
    def removeAttachments(self, indexes = [], REQUEST = {}):
        """
        removeAttachments(self, indexes = [], REQUEST = {}) 
        - remove the attachments from the list
        """
        # Check lock status
        self.checkLockStatus()

        # Sort indexes
        indexes.sort()
        indexes.reverse()

        # Remove each one (starting from the end)
        for idx in indexes:
            self.removeAttachment(idx)

    security.declareProtected(ArticleCore_editPermission, 'changeAttachment')
    def changeAttachment(self, index = 0, title = '', file = '', REQUEST = {}):
        """
        changeAttachment(self, index = 0, title = '', file = '', REQUEST = {})
        """
        # Check lock status
        self.checkLockStatus()

        # Ensure the index is reachable
        if index >= len(self.attachments_ids):
            return

        # Get attachment object
        attach = getattr(self, self.attachments_ids[index])
        
        # If we have to change the file, manage it
        if file:
            attach.delete()
            attach.upload(file)

        # Change / update title
        attach.setTitle(title, )

        # Reindex catalog
        self.reindexObject()

    security.declareProtected(ArticleCore_editPermission, 'changeAttachment')
    def changeAttachmentTitle(self, index = 0, title = ''):
        """
        changeAttachmentTitle(self, index = 0, title = '') 
        - index is the images index in the list
        """
        # Check lock status
        self.checkLockStatus()

        # Ensure the index is reachable
        if index >= len(self.attachments_ids):
            return

        # Get attachment object
        attach = getattr(self, self.attachments_ids[index])
        attach.setTitle(title,)

        # Reindex catalog
        self.reindexObject()


    security.declareProtected(ArticleCore_editPermission, 'appendAttachment')
    def appendAttachment(self, title = '', file = '', REQUEST = {}):
        """
        appendAttachment(self, title = '', file = '', REQUEST = {}) 
        - Append a new attachments file at the end of the list.
        """
        # Check lock status
        self.checkLockStatus()

        # Find a random identifier
        id = "Attachment%08d" % (int(random.random() * 100000))
        while id in self.attachments_ids:
            id = "Attachment%08d" % (int(random.random() * 100000))

        # Changed because ZAtt. creation without an id is deprecated
        attach = ZAttachmentAttribute.ZAttachmentAttribute(id = id, title = title)
        self._setObject(id, attach)     # Attach the object
        self.attachments_ids.append(id) # Appending
        self.attachments_ids = self.attachments_ids             # Ensure persistancy
        self.changeAttachment(len(self.attachments_ids) - 1, title, file,)

    security.declareProtected(CMFCorePermissions.View, "listAttachmentIds")
    def listAttachmentIds(self,):
        """
        listAttachmentIds(self,)
        - return all attachment ids from this object
        """
        return self.attachments_ids


    security.declareProtected(CMFCorePermissions.View, "listAttachments")
    def listAttachments(self, ):
        """
        listAttachments(self, )
        - return attachments as a list
        """
        attachlist = []
        for attach in self.listAttachmentIds():
            attachlist.append(getattr(self, attach))
        return attachlist

    
    security.declareProtected(CMFCorePermissions.View, 'getAttachment')
    def getAttachment(self, index, ):
        """
        getAttachment(self, index, )
        """
        return getattr(self, self.attachments_ids[index])       # $$$ POSSIBLE SECURITY HOLE
                                                                # WE HAVE TO USE restrictedTraverse() THERE


    security.declareProtected(CMFCorePermissions.View, 'getAttachmentId')
    def getAttachmentId(self, index):
        """
        getAttachmentId(self, index)
        - return the id of the attachment
        """
        return self.attachments_ids[index]


    security.declareProtected(CMFCorePermissions.View, 'getAttachment')
    def getAttachmentURL(self, index, ):
        """
        getAttachmentURL(self, index, ) => string

        Return a suitable absolute URL for this attachment, including the filename.
        """
        attach = self.getAttachment(index)
        id = self.getAttachmentId(index)
        return "%s/%s/%s" % (self.absolute_url(), id, url_quote(attach.getFilename()), )

