#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - US English header
 *   
 *   This file defines common properties, macros, and other identifiers used
 *   throughout the US English modules of the TADS 3 library.  
 */


/* ------------------------------------------------------------------------ */
/*
 *   Vocabulary properties for parts of speech we use in the English parser.
 *   
 *   These properties are defined in the English-specific header because
 *   other languages might classify their parts of speech differently; for
 *   example, a French parser might distinguish nouns according to gender,
 *   and thus require masculineNoun and feminineNoun properties instead of a
 *   single noun property.
 *   
 *   possessiveAdj is for possessive adjectives.  Note that it's never
 *   necessary to include a "'s" word as a possessive adjective, because we
 *   handle those automatically in the tokenizer.  possessiveAdj is
 *   primarily for pronouns, such as "my" for the player character.  Note
 *   that the third-person possessive pronoun adjectives (his, her, its,
 *   their) are explicitly defined in the English grammar, so these don't
 *   need to be defined in the dictionary.
 *   
 *   possessiveNoun is used for possessive nouns and pronouns.  As with
 *   possessive adjectives, it's never necessary to include "'s" words, so
 *   this is mostly useful for pronouns, such as "mine" for the player
 *   character.  Note that, as with the third-person possessive pronoun
 *   adjectives, the third-person possessive pronouns (his, hers, its,
 *   theirs) are explicitly defined in the English grammar and thus don't
 *   need to be entered into the dictionary.  
 */
dictionary property noun, adjective, plural, possessiveAdj, possessiveNoun;


/*
 *   Numeric parts of speech:
 *   
 *   digitWord - a name for a digit (zero through nine)
 *   
 *   teenWord - a name for a 'teen' (ten through nineteen) - we distinguish
 *   these because there's no general grammatical rule for constructing
 *   names for these numbers out of smaller lexical units.  We include ten
 *   in this set, rather than in the teenWord set, because ten cannot be
 *   combined with digit words to form other numbers.
 *   
 *   tensWord - a name for a multiple of ten (twenty, thirty, ..., ninety).
 *   The words in this set can be used to construct the numbers from 20 to
 *   99 by combining them with digit words using a hyphen (as in
 *   "twenty-two").
 *   
 *   ordinalWord - an ordinal, such as '1st' or 'first'.  
 */
dictionary property digitWord, teenWord, tensWord;
dictionary property ordinalWord;


/* ------------------------------------------------------------------------ */
/*
 *   Define some templates that apply to ordinary objects (descendants of
 *   Thing). 
 */
Thing template 'vocabWords_' 'name';
Thing template 'vocabWords_' 'name' "desc";
Thing template 'vocabWords_' 'name' @location;
Thing template 'vocabWords_' 'name' @location "desc";

/*
 *   For rooms, we normally have no vocabulary words, but we do have a name
 *   and description, and optionally a "destination name" to use to describe
 *   connectors from adjoining rooms.  
 */
Room template 'name' "desc";
Room template 'name' 'destName' "desc";

/*
 *   For passages, allow special syntax to point to the master side of the
 *   passage. 
 */
Passage template ->masterObject inherited;

/* 
 *   For one-way room connectors, provide special syntax to point to the
 *   destination room. 
 */
OneWayRoomConnector template ->destination;

/*
 *   For enterables, allow special syntax to point to the connector which an
 *   actor uses to traverse into the enterable. 
 */
Enterable template ->connector inherited;

/*
 *   For TravelMessage connectors, provide special syntax to specify the
 *   message and point to the destination. 
 */
TravelMessage template ->destination "travelDesc";
NoTravelMessage template "travelDesc";

/*
 *   For TravelBarrier objects, provide special syntax to specify the
 *   underlying connector of the barrier. 
 */
TravelBarrier template ->connector;

/*
 *   Templates for ThingState objects.
 */
ThingState template 'listName';
ThingState template 'listName' @listingOrder;


/* ------------------------------------------------------------------------ */
/*
 *   Convenience macros for defining command grammar.
 *   
 *   A command's grammar is defined via a 'grammar' rule definition for the
 *   'predicate' production; the VerbRule macro can be used for better
 *   readability.
 *   
 *   Within a command grammar rule, there are several commonly-used object
 *   roles assigned to single-noun or noun-list phrases.  We provide the
 *   singleDobj, dobjList, singleIobj, and iobjList macros to make these
 *   assignments more readable.  In addition, number, topic, literal, and
 *   direction phrases can be assigned with singleNumber, singleTopic,
 *   singleLiteral, and singleDir, respectively.  
 */

#define VerbRule(tag)  grammar predicate(tag):
#define singleDobj     singleNoun->dobjMatch
#define singleIobj     singleNoun->iobjMatch
#define dobjList       nounList->dobjMatch
#define iobjList       nounList->iobjMatch
#define singleNumber   numberPhrase->numMatch
#define singleTopic    topicPhrase->topicMatch
#define singleLiteral  literalPhrase->literalMatch
#define singleDir      directionName->dirMatch
