#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library: command reports
 *   
 *   This module defines the "command report" classes, which the command
 *   execution engine uses to keep track of the status of a command.  
 */

#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Command report objects.  The library uses these to control how the
 *   text from a command is displayed.  Game code can also use report
 *   objects to show and control command results, but this isn't usually
 *   necessary; game code can usually simply display messages directly.
 *   
 *   Reports are divided into two broad classes: "default" and "full"
 *   reports.
 *   
 *   A "default" report is one that simply confirms that an action was
 *   performed, and provides little additional information.  The library
 *   uses default reports for simple commands whose full implications
 *   should normally be obvious to a player typing such commands: take,
 *   drop, put in, and the like.  The library's default reports are
 *   usually quite terse: "Taken", "Dropped", "Done".
 *   
 *   A "full" report is one that gives the player more information than a
 *   simple confirmation.  These reports typically describe either the
 *   changes to the game state caused by a command or surprising side
 *   effects of a command.  For example, if the command is "push button,"
 *   and pushing the button opens the door next to the button, a full
 *   report would describe the door opening.
 *   
 *   Note that a full report is warranted any time a command describes
 *   anything beyond a simple confirmation.  In our door-opening button
 *   example, opening the door by pushing the button always warrants a
 *   full report, even if the player has already seen the effects of the
 *   button a hundred times before, and even if the button is labeled
 *   "push to open door."  It doesn't matter whether or not the
 *   consequences of the command ought to be obvious to the player; what
 *   matters is that the command warrants a report beyond a simple
 *   confirmation.  Any time a report is more than a simple confirmation,
 *   it is a full report, no matter how obvious to the player the effects
 *   of the action.
 *   
 *   Full reports are further divided into three subcategories by time
 *   ordering: "main," "before," and "after."  "Before" and "after"
 *   reports are ordered before and after (respectively) a main report.  
 */
class CommandReport: object
    construct()
    {
        /* 
         *   remember the action with which we're associated, unless a
         *   subclass already specifically set the action 
         */
        if (action_ == nil)
            action_ = gAction;
    }

    /* get my action */
    getAction() { return action_; }

    /* check to see if my action is implicit */
    isActionImplicit() { return action_ != nil && action_.isImplicit; }

    /* check to see if my action is nested in the other report's action */
    isActionNestedIn(other)
    {
        return (action_ != nil
                && other.getAction() != nil
                && action_.isNestedIn(other.getAction()));
    }

    /*
     *   Flag: if this property is true, this report indicates a failure.
     *   By default, a report does not indicate failure.  
     */
    isFailure = nil

    /* iteration number current when we were added to the transcript */
    iter_ = 0

    /* the action I'm associated with */
    action_ = nil

    /*
     *   Am I part of the same action as the given report?  Returns true if
     *   this action is part of the same iteration and part of the same
     *   action as the other report.  
     */
    isPartOf(report)
    {
        /* 
         *   if I don't have an action, or the other report doesn't have an
         *   action, we're not related 
         */
        if (action_ == nil || report.action_ == nil)
            return nil;

        /* if our iterations don't match, we're not related */
        if (iter_ != report.iter_)
            return nil;

        /* check if I'm part of the other report's action */
        return action_.isPartOf(report.action_);
    }
;

/*
 *   Group separator.  This simply displays separation between groups of
 *   messages - that is, between one set of messages associated with a
 *   single action and a set of messages associated with a different
 *   action.  
 */
class GroupSeparatorMessage: CommandReport
    construct(report)
    {
        /* use the same action and iteration as the given report */
        action_ = report.getAction();
        iter_ = report.iter_;
    }

    /* show the normal command results separator */
    showMessage() { say(libMessages.complexResultsSeparator); }
;

/*
 *   Internal separator.  This displays separation within a group of
 *   messages for a command, to visually separate the results from an
 *   implied command from the results for the enclosing command. 
 */
class InternalSeparatorMessage: CommandReport
    construct(report)
    {
        /* use the same action and iteration as the given report */
        action_ = report.getAction();
        iter_ = report.iter_;
    }

    /* show the normal command results separator */
    showMessage() { say(libMessages.internalResultsSeparator); }
;

/*
 *   End-of-description marker.  This is a pseudo-report; it doesn't
 *   display anything, but rather serves as a marker in the transcript
 *   stream to let us know where the descriptive reports for a given action
 *   end. 
 */
class EndOfDescReport: CommandReport
    showMessage() { }
;    

/*
 *   Simple MessageResult-based command report 
 */
class CommandReportMessage: CommandReport, MessageResult
    construct([params])
    {
        /* invoke our base class constructors */
        inherited CommandReport();
        inherited MessageResult(params...);
    }
;

/* 
 *   default report 
 */
class DefaultCommandReport: CommandReportMessage
;

/*
 *   extra information report 
 */
class ExtraCommandReport: CommandReportMessage
;

/*
 *   default descriptive report 
 */
class DefaultDescCommandReport: CommandReportMessage
;

/*
 *   cosmetic spacing report 
 */
class CosmeticSpacingCommandReport: CommandReportMessage
;

/*
 *   base class for all "full" reports 
 */
class FullCommandReport: CommandReportMessage
    /* 
     *   a full report has a sequence number that tells us where the
     *   report goes relative to the others - the higher this number, the
     *   later the report goes 
     */
    seqNum = nil
;

/*
 *   "before" report - these come before the main report 
 */
class BeforeCommandReport: FullCommandReport
    seqNum = 1
;

/*
 *   main report 
 */
class MainCommandReport: FullCommandReport
    seqNum = 2
;

/*
 *   failure report 
 */
class FailCommandReport: FullCommandReport
    seqNum = 2
    isFailure = true
;

/*
 *   "after" report - these come after the main report 
 */
class AfterCommandReport: FullCommandReport
    seqNum = 3
;

/* ------------------------------------------------------------------------ */
/*
 *   Announcements.  We use these to track announcements to be made as
 *   part of an action's results. 
 */
class CommandAnnouncement: CommandReport
    construct([params])
    {
        /* inherit default handling */
        inherited();

        /* remember our text */
        messageText_ = getMessageText(params...);
    }

    /* 
     *   Get our message text.  By default, we simply get the libMessages
     *   message given by the property. 
     */
    getMessageText([params])
    {
        /* get the library message */
        return libMessages.(messageProp_)(params...);
    }

    /* 
     *   Show our message.  Our default implementation shows the library
     *   message given by our messageProp_ property, using the parameters
     *   we stored in our constructor.  
     */
    showMessage()
    {
        /* call libMessages to show our message */
        say(messageText_);
    }

    /* our libMessages property */
    messageProp_ = nil

    /* our message text */
    messageText_ = ''
;

class MultiObjectAnnouncement: CommandAnnouncement
    /* show the announceMultiActionObject message */
    messageProp_ = &announceMultiActionObject
;

class AmbigObjectAnnouncement: CommandAnnouncement
    /* show the announceAmbigObject announcement */
    messageProp_ = &announceAmbigActionObject
;

class DefaultObjectAnnouncement: CommandAnnouncement
    construct(obj, whichObj, action, allResolved)
    {
        /* remember our object */
        obj_ = obj;

        /* remember the message parameters */
        whichObj_ = whichObj;
        allResolved_ = allResolved;

        /* remember my action */
        action_ = action;

        /* inherit default handling */
        inherited();
    }

    /* get our message text */
    getMessageText()
    {
        /* get the announcement message from our object */
        return obj_.announceDefaultObject(whichObj_, action_, allResolved_);
    }

    /* our defaulted object */
    obj_ = nil

    /* our message parameters */
    whichObj_ = nil
    allResolved_ = nil
;

class RemappedActionAnnouncement: CommandAnnouncement
    constuct()
    {
        /* use the action as the message parameter */
        inherited(gAction);
    }

    messageProp_ = &announceRemappedAction
;

class ImplicitActionAnnouncement: CommandAnnouncement
    construct(msg)
    {
        /* use the given message property */
        messageProp_ = msg;

        /* inherit default, passing the action as the message parameter */
        inherited(gAction);
    }
;

class CommandSepAnnouncement: CommandAnnouncement
    construct()
    {
        /* we're not associated with an iteration or action */
        action_ = nil;
        iter_ = nil;
    }

    showMessage()
    {
        /* show a command separator */
        "<.commandsep>";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Command Transcript.  This is a "semantic transcript" of the results of
 *   a command.  This provides a list of CommandReport objects describing
 *   the results of the command.  
 */
class CommandTranscript: OutputFilter
    construct()
    {
        /* set up a vector to hold the reports */
        reports_ = new Vector(5);
    }

    /* 
     *   flag: the command has failed (i.e., at least one failure report
     *   has been generated) 
     */
    isFailure = nil

    /* 
     *   flag: I'm active; when this is nil, we'll pass text through our
     *   filter routine unchanged 
     */
    isActive = true

    /* activate - set up to capture output */
    activate()
    {
        /* make myself active */
        isActive = true;
    }

    /* deactivate - stop capturing output */
    deactivate()
    {
        /* make myself inactive */
        isActive = nil;
    }

    /* 
     *   Count an iteration.  An Action should call this once per iteration
     *   if it's a top-level (non-nested) command.
     */
    newIter() { ++iter_; }

    /* show our reports */
    showReports(deact)
    {
        local wasActive;
        
        /* 
         *   remember whether we were active or not originally, then
         *   deactivate (maybe just temporarily) so that we can write out
         *   our reports without recursively intercepting them
         */
        wasActive = isActive;
        deactivate();

        /* first, apply all defined transformations to our transcript */
        applyTransforms();
        
        /*
         *   Temporarily cancel any sense context message blocking.  We
         *   have already taken into account for each report whether or
         *   not the report was visible when it was generated, so we can
         *   display each report that made it past that check without any
         *   further conditions.  
         */
        callWithSenseContext(nil, nil, new function()
        {
            /* show the reports */            
            foreach (local cur in reports_)
            {
                /* if we're allowed to show this report, show it */
                if (canShowReport(cur))
                    cur.showMessage();
            }
        });
            
        /* 
         *   if we were active and we're not being asked to deactivate,
         *   re-activate now that we're finished showing our reports 
         */
        if (wasActive && !deact)
            activate();
    }

    /*
     *   Add a report. 
     */
    addReport(report)
    {
        /* check for a failure report */
        if (report.isFailure)
        {
            /* set the failure flag for the entire command */
            isFailure = true;

            /* 
             *   If this is an implied command, and the actor is in "NPC
             *   mode", suppress the report.  When an implied action fails
             *   in NPC mode, we act as though we never attempted the
             *   action. 
             */
            if (gAction.isImplicit && gActor.impliedCommandMode() == ModeNPC)
                return;
        }

        /* 
         *   Do not queue reports made while the sense context is blocking
         *   output because the player character cannot sense the locus of
         *   the action.  Note that this check comes before we queue the
         *   report, but after we've noted any effect on the status of the
         *   overall action; even if we're not going to show the report,
         *   its status effects are still valid.  
         */
        if (senseContext.isBlocking)
            return;

        /* note the current iteration in the report */
        report.iter_ = iter_;

        /* append the report */
        reports_.append(report);
    }

    /*
     *   End the description section of the report.  This adds a marker
     *   report that indicates that anything following (and part of the
     *   same action) is no longer part of the description; this can be
     *   important when we apply the default description suppression
     *   transformation, because it tells us not to consider the
     *   non-descriptive messages following this marker when, for example,
     *   suppressing default descriptive messages.  
     */
    endDescription()
    {
        /* add an end-of-description report */
        addReport(new EndOfDescReport());
    }

    /*
     *   Announce that the action is implicit
     */
    announceImplicit(msgProp)
    {
        /* 
         *   If the actor performing the command is not in "player" mode,
         *   save an implicit action announcement; for NPC mode, we treat
         *   implicit command results like any other results, so we don't
         *   want a separate announcement.  
         */
        if (gActor.impliedCommandMode() == ModePlayer)
            addReport(new ImplicitActionAnnouncement(msgProp));
    }

    /*
     *   Announce a remapped action 
     */
    announceRemappedAction()
    {
        /* save a remapped-action announcement */
        addReport(new RemappedActionAnnouncement());
    }

    /*
     *   Announce one of a set of objects to a multi-object action.  We'll
     *   record this announcement for display with our report list.  
     */
    announceMultiActionObject(obj, whichObj)
    {
        /* save a multi-action object announcement */
        addReport(new MultiObjectAnnouncement(obj, whichObj, gAction));
    }

    /*
     *   Announce an object that was resolved with slight ambiguity. 
     */
    announceAmbigActionObject(obj, whichObj)
    {
        /* save an ambiguous object announcement */
        addReport(new AmbigObjectAnnouncement(obj, whichObj, gAction));
    }

    /*
     *   Announce a default object. 
     */
    announceDefaultObject(obj, whichObj, action, allResolved)
    {
        /* save the default object announcement */
        addReport(new DefaultObjectAnnouncement(
            obj, whichObj, action, allResolved));
    }

    /*
     *   Add a command separator. 
     */
    addCommandSep()
    {
        /* add a command separator announcement */
        addReport(new CommandSepAnnouncement());
    }

    /*
     *   clear our reports 
     */
    clearReports()
    {
        /* forget all of the reports in the main list */
        if (reports_.length() != 0)
            reports_.removeRange(1, reports_.length());
    }

    /*
     *   Can we show a given report?  By default, we always return true,
     *   but subclasses might want to override this to suppress certain
     *   types of reports.  
     */
    canShowReport(report) { return true; }

    /*
     *   Filter text.  If we're active, we'll turn the text into a command
     *   report and add it to our report list, blocking the text from
     *   reaching the underlying stream; otherwise, we'll pass it through
     *   unchanged.  
     */
    filterText(ostr, txt)
    {
        /* if we're inactive, pass text through unchanged */
        if (!isActive)
            return txt;
        
        /* 
         *   If the current sense context doesn't allow any messages to be
         *   generated, block the generated text entirely.  We want to
         *   block text or not according to the sense context in effect
         *   now; so we must note it now rather than wait until we
         *   actually display the report, since the context could be
         *   different by then.  
         */
        if (senseContext.isBlocking)
            return nil;

        /* add a main report to our list if the text is non-empty */
        if (txt != '')
            addReport(new MainCommandReport(txt));

        /* capture the text - send nothing to the underlying stream */
        return nil;
    }

    /* apply transformations */
    applyTransforms()
    {
        /* apply each defined transformation */
        foreach (local cur in transforms_)
            cur.applyTransform(self, reports_);
    }

    /* 
     *   check to see if the current action has a report matching the given
     *   criteria 
     */
    currentActionHasReport(func)
    {
        /* 
         *   Check for an action that's part of the current iteration and
         *   which matches the given function's criteria.  If we can find
         *   such an action, return true; otherwise, return nil.  
         */
        return (reports_.indexWhich({x: x.iter_ == iter_ && (func)(x)})
                != nil);
    }

    /* 
     *   iteration number - for an iterated top-level command, this helps
     *   us keep the results for a particular iteration grouped together 
     */
    iter_ = 1

    /* our vector of reports */
    reports_ = nil

    /* our list of transformations */
    transforms_ = [defaultReportTransform,
                   reportOrderTransform, complexMultiTransform]
;

/* ------------------------------------------------------------------------ */
/*
 *   Transcript Transform. 
 */
class TranscriptTransform: object
    /*
     *   Apply our transform to the transcript vector.  By default, we do
     *   nothing; each subclass must override this to manipulate the vector
     *   to make the change it wants to make.  
     */
    applyTransform(trans, vec) { }
;

/* ------------------------------------------------------------------------ */
/*
 *   Transcript Transform: set before/main/after report order.  We'll look
 *   for any before/after reports that are out of order with respect to
 *   their main reports, and move them into the appropriate positions. 
 */
reportOrderTransform: TranscriptTransform
    applyTransform(trans, vec)
    {
        /* scan for before/after reports */
        for (local i = 1, local len = vec.length() ; i <= len ; ++i)
        {
            /* get this item */
            local cur = vec[i];

            /* if this is a before/after report, consider moving it */
            if (cur.ofKind(FullCommandReport) && cur.seqNum != nil)
            {
                local idx;
                
                /* 
                 *   This item cares about its sequencing, so it could be
                 *   out of order with respect to other items from the same
                 *   sequence.  Find the first item with a higher sequence
                 *   number from the same group, and make sure this item is
                 *   before the first such item.  
                 */
                for (idx = 1 ; idx < i ; ++idx)
                {
                    local x;
                    
                    /* get this item */
                    x = vec[idx];

                    /* if x should come after cur, we need to move cur */
                    if (x.ofKind(FullCommandReport)
                        && x.seqNum > cur.seqNum
                        && x.isPartOf(cur))
                    {
                        /* remove cur and reinsert it before x */
                        vec.removeElementAt(i);
                        vec.insertAt(idx, cur);

                        /* adjust our scan index for the removal */
                        --i;
                        
                        /* no need to look any further */
                        break;
                    }
                }
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Transcript Transform: remove unnecessary default reports.  We'll scan
 *   the transcript for default reports for actions which also have
 *   implicit announcements or non-default reports, and remove those
 *   default reports.  We'll also remove default descriptive reports which
 *   also have non-default reports in the same action.  
 */
defaultReportTransform: TranscriptTransform
    applyTransform(trans, vec)
    {
        /* scan for default reports */
        for (local i = 1, local len = vec.length() ; i <= len ; ++i)
        {
            local cur;
            
            /* get this item */
            cur = vec[i];
            
            /* 
             *   if this is a default report, check to see if we want to
             *   keep it 
             */
            if (cur.ofKind(DefaultCommandReport))
            {
                /* 
                 *   check for a main report or an implicit announcement
                 *   associated with the same action; if we find anything,
                 *   we don't need to keep the default report 
                 */
                if (vec.indexWhich(
                    {x: (x != cur
                         && cur.isPartOf(x)
                         && (x.ofKind(FullCommandReport)
                             || x.ofKind(ImplicitActionAnnouncement)))
                    }) != nil)
                {
                    /* we don't need this default report */
                    vec.removeElementAt(i);

                    /* adjust our scan index for the removal */
                    --i;
                    --len;
                }
            }

            /*
             *   if this is a default descriptive report, check to see if
             *   we want to keep it 
             */
            if (cur.ofKind(DefaultDescCommandReport))
            {
                local fullIdx;
                
                /* 
                 *   check for a main report associated with the same
                 *   action
                 */
                fullIdx = vec.indexWhich(
                    {x: (x != cur
                         && cur.isPartOf(x)
                         && x.ofKind(FullCommandReport))});
                
                /* 
                 *   if we found another report, check to see if it comes
                 *   before or after any 'end of description' for the same
                 *   action 
                 */
                if (fullIdx != nil)
                {
                    local endIdx;

                    /* find the 'end of description' report, if any */
                    endIdx = vec.indexWhich(
                        {x: (x != cur
                             && cur.isPartOf(x)
                             && x.ofKind(EndOfDescReport))});

                    /* 
                     *   if we found a full report before the
                     *   end-of-description report, then the full report is
                     *   part of the description and thus should suppress
                     *   the default report; otherwise, the description
                     *   portion includes only the default report and the
                     *   default report should thus remain 
                     */
                    if (endIdx == nil || fullIdx < endIdx)
                    {
                        /* don't keep the default descriptive report */
                        vec.removeElementAt(i);

                        /* adjust our indices for the removal */
                        --i;
                        --len;
                    }
                }
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Transcript Transform: Complex Multi-object Separation.  If we have an
 *   action that's being applied to one of a bunch of iterated objects, and
 *   the action has any implied command announcements associated with it,
 *   we'll set off the result for this command from its preceding and
 *   following commands by a paragraph separator.  
 */
complexMultiTransform: TranscriptTransform
    applyTransform(trans, vec)
    {
        /* scan the list for multi-object announcements */
        foreach (local cur in vec)
        {
            /* if it's a multi-object announcement, check it */
            if (cur.ofKind(MultiObjectAnnouncement))
            {
                /* 
                 *   Check for implied action announcements in the same
                 *   iteration - if we find any, then this group of
                 *   commands is complex and must be set off from its
                 *   neighbors.  
                 */
                if (vec.indexWhich(
                    {x: (x != cur
                         && x.iter_ == cur.iter_
                         && (x.ofKind(ImplicitActionAnnouncement)))
                    }) != nil)
                {
                    local idx;
                    
                    /* 
                     *   This is indeed a complex iterated item.  Set it
                     *   off by paragraph breaks before and after the
                     *   iteration.
                     *   
                     *   First, find the first item in this iteration.  If
                     *   it's not the first item in the whole transcript,
                     *   insert a separator before it.  
                     */
                    idx = vec.indexWhich({x: x.iter_ == cur.iter_});
                    if (idx != 1)
                        vec.insertAt(idx, new GroupSeparatorMessage(cur));

                    /* 
                     *   Next, find the last item in this iteration.  If
                     *   it's no the last item in the entire transcript,
                     *   add a separator after it. 
                     */
                    idx = vec.lastIndexWhich({x: x.iter_ == cur.iter_});
                    if (idx != vec.length())
                        vec.insertAt(idx + 1, new GroupSeparatorMessage(cur));
                }
            }

            /* 
             *   if it's a command result from an implied command, and we
             *   have another command result following from the enclosing
             *   command, add a separator between this result and the next
             *   result 
             */
            if (cur.ofKind(CommandReportMessage) && cur.isActionImplicit)
            {
                local idx;

                /* get the index of this element */
                idx = vec.indexOf(cur);

                /* 
                 *   if there's another element following, check to see if
                 *   it's a command report for an enclosing action (i.e.,
                 *   an action that initiated this implied action) 
                 */
                if (idx < vec.length())
                {
                    local nxt;

                    /* get the next element */
                    nxt = vec[idx + 1];

                    /* 
                     *   if it's another command report, and it's for an
                     *   action that encloses this action, then put a
                     *   separator before it 
                     */
                    if (nxt.ofKind(CommandReportMessage)
                        && nxt.getAction() != cur.getAction()
                        && cur.isActionNestedIn(nxt))
                    {
                        /* add a separator */
                        vec.insertAt(idx + 1,
                                     new InternalSeparatorMessage(cur));
                    }
                }
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Invoke a callback function using a transcript of the given class.
 *   We'll return the transcript instance when we're done.  
 */
withCommandTranscript(transcriptClass, func)
{
    local transcript;

    /* 
     *   Create a transcript of the given class.  Make the transcript
     *   transient, since it's effectively part of the output stream state
     *   and thus shouldn't be saved or undone. 
     */
    transcript = transcriptClass.createTransientInstance();

    /* make this transcript the current global transcript */
    gTranscript = transcript;

    /* install the transcript as a filter on the main output stream */
    mainOutputStream.addOutputFilter(transcript);

    /* make sure we undo our global changes before we leave */
    try
    {
        /* invoke the callback */
        (func)();
    }
    finally
    {
        /* uninstall the transcript output filter */
        mainOutputStream.removeOutputFilter(transcript);

        /* it's no longer the active global transcript */
        gTranscript = nil;
    }

    /* show the transcript results */
    transcript.showReports(true);

    /* return the transcript object */
    return transcript;
}

