#ifdef RCSID
static char RCSid[] =
"$Header$";
#endif

/* Copyright (c) 1999, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  vmsave.cpp - T3 save/restore state
Function
  
Notes
  
Modified
  08/02/99 MJRoberts  - Creation
*/

#include "t3std.h"
#include "os.h"
#include "vmglob.h"
#include "vmsave.h"
#include "vmfile.h"
#include "vmimage.h"
#include "vmobj.h"
#include "vmrun.h"
#include "vmstack.h"
#include "vmundo.h"
#include "vmmeta.h"


/* ------------------------------------------------------------------------ */
/*
 *   Saved state signature.  This signature is tied to a specific format
 *   version; it should be changed whenever the format version is modified
 *   so that it is incompatible with older versions.
 *   
 *   Note that incompatible changes to intrinsic class serialization formats
 *   will require updating the version number.
 *   
 *   Incompatible changes to the format are not a particularly big deal.
 *   Saved states tend to be local to a particular machine, since they're
 *   mostly used to suspend sessions for later resumption and to "branch"
 *   the state evolution (i.e., to allow playing a game from a particular
 *   point, then returning later to that same point to play again, but doing
 *   different things this time; this is used particularly to save "good"
 *   positions as a precaution against later getting into unwinnable
 *   states).  
 */
#define VMSAVEFILE_SIG "T3-state-v0008\015\012\032"


/* ------------------------------------------------------------------------ */
/*
 *   Compute the checksum of the contents of a file stream.  We'll compute
 *   the checksum of the given file, starting at the current seek position
 *   and running for the requested number of bytes.  
 */
static unsigned long compute_checksum(CVmFile *fp, unsigned long len)
{
    unsigned long val;
    static unsigned long tab[] = {
        0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
        0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
        0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
        0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
        0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
        0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
        0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
        0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
        0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
        0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
        0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
        0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
        0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
        0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
        0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
        0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
        0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
        0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
        0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
        0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
        0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
        0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
        0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
        0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
        0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
        0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
        0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
        0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
        0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
        0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
        0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
        0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
        0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
        0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
        0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
        0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
        0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
        0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
        0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
        0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
        0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
        0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
        0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
        0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
        0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
        0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
        0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
        0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
        0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
        0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
        0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
        0x2d02ef8dL
    };

    /* compute the CRC value */
    for (val = 0 ; len != 0 ; )
    {
        char buf[256];
        unsigned char *p;
        size_t cur_len;
        
        /* figure out how much we can load from the file */
        cur_len = sizeof(buf);
        if (cur_len > len)
            cur_len = (size_t)len;

        /* load the data from the file */
        fp->read_bytes(buf, cur_len);

        /* deduct the amount we read from the overall file length remaining */
        len -= cur_len;

        /* continue computing the checksum with this block */
        for (p = (unsigned char *)buf ; cur_len != 0 ; ++p, --cur_len)
            val = tab[(val ^ *p) & 0xff] ^ (val >> 8);
    }

    /* return the computed value */
    return val;
}


/* ------------------------------------------------------------------------ */
/*
 *   Save VM state to a file 
 */
void CVmSaveFile::save(VMG_ CVmFile *fp)
{
    const char *fname;
    size_t fname_len;
    long startpos;
    long endpos;
    unsigned long crcval;
    unsigned long datasize;
    
    /* 
     *   We cannot save the state from within a recursive VM invocation --
     *   the full machine state in a recursive invocation includes native
     *   machine state, which we are not equipped to save.  We could thus
     *   never restore such a state, so there's no point in saving it.  
     */
    if (G_interpreter->is_recursive_invocation(vmg0_))
        err_throw(VMERR_RECURSIVE_VM_CALL);

    /* write the signature */
    fp->write_bytes(VMSAVEFILE_SIG, sizeof(VMSAVEFILE_SIG)-1);

    /* note the seek position of the start of the file header */
    startpos = fp->get_pos();

    /* write a placeholder for the stream size and checksum */
    fp->write_int4(0);
    fp->write_int4(0);

    /* write the image file's timestamp */
    fp->write_bytes(G_image_loader->get_timestamp(), 24);

    /* get the image filename */
    fname = G_image_loader->get_filename();
    fname_len = strlen(fname);

    /* 
     *   write the image filename, so we can figure out what image file to
     *   load if we start the interpreter specifying only the saved state
     *   to be restored 
     */
    fp->write_int2(strlen(fname));
    fp->write_bytes(fname, fname_len);

    /* save all modified object state */
    G_obj_table->save(vmg_ fp);

    /* save the synthesized exports */
    G_image_loader->save_synth_exports(vmg_ fp);

    /* remember where the file ends */
    endpos = fp->get_pos();

    /* 
     *   compute the size of the data stream - this includes everything
     *   after the size/checksum fields 
     */
    datasize = endpos - startpos - 8;

    /* 
     *   seek back to just after the size/checksum header - this is the
     *   start of the section of the file for which we must compute the
     *   checksum 
     */
    fp->set_pos(startpos + 8);

    /* compute the checksum */
    crcval = compute_checksum(fp, datasize);

    /* 
     *   seek back to the size/checksum header, and fill in those fields now
     *   that we know their values 
     */
    fp->set_pos(startpos);
    fp->write_int4(datasize);
    fp->write_int4(crcval);

    /* seek back to the end of the file */
    fp->set_pos(endpos);
}

/* ------------------------------------------------------------------------ */
/*
 *   Given a saved state file, get the name of the image file that was
 *   loaded when the state file was created. 
 */
int CVmSaveFile::restore_get_image(osfildef *fp,
                                   char *fname_buf, size_t fname_buf_len)
{
    char buf[128];
    size_t len;

    /* read the signature, size/checksum, and timestamp fields */
    if (osfrb(fp, buf, sizeof(VMSAVEFILE_SIG)-1 + 8 + 24))
        return VMERR_READ_FILE;

    /* check the signature */
    if (memcmp(buf, VMSAVEFILE_SIG, sizeof(VMSAVEFILE_SIG)-1) != 0)
        return VMERR_NOT_SAVED_STATE;

    /* read the length of the image file name */
    if (osfrb(fp, buf, 2))
        return VMERR_READ_FILE;

    /* get the length from the buffer */
    len = osrp2(buf);

    /* if it won't fit in the buffer, return an error */
    if (len + 1 > fname_buf_len)
        return VMERR_READ_FILE;

    /* read the name into the caller's buffer */
    if (osfrb(fp, fname_buf, len))
        return VMERR_READ_FILE;

    /* null-terminate the name */
    fname_buf[len] = '\0';

    /* success */
    return 0;
}

/* ------------------------------------------------------------------------ */
/*
 *   Restore VM state from a file.  Returns zero on success, non-zero on
 *   error.  
 */
int CVmSaveFile::restore(VMG_ CVmFile *fp)
{
    char buf[128];
    int err;
    unsigned long datasize;
    unsigned long old_crcval;
    unsigned long new_crcval;
    long startpos;
    int old_gc_enabled;
    CVmObjFixup *fixups;

    /* we don't have a fixup table yet (the object loader will create one) */
    fixups = 0;

    /*
     *   We cannot restore state during a recursive VM invocation.  Part
     *   of our current machine state is in the native machine stack
     *   (i.e., the recursive call state); we are not equipped to modify
     *   the native state, hence we cannot restore in the current context.
     */
    if (G_interpreter->is_recursive_invocation(vmg0_))
        return VMERR_RECURSIVE_VM_CALL;

    /* read the file's signature */
    fp->read_bytes(buf, sizeof(VMSAVEFILE_SIG)-1);

    /* check the signature */
    if (memcmp(buf, VMSAVEFILE_SIG, sizeof(VMSAVEFILE_SIG)-1) != 0)
        return VMERR_NOT_SAVED_STATE;

    /* read the size/checksum fields */
    datasize = fp->read_int4();
    old_crcval = fp->read_int4();

    /* note the starting position of the datastream */
    startpos = fp->get_pos();

    /* compute the checksum of the file data */
    new_crcval = compute_checksum(fp, datasize);

    /* 
     *   if the checksum we computed doesn't match the one stored in the
     *   file, the file is corrupted 
     */
    if (new_crcval != old_crcval)
        return VMERR_BAD_SAVED_STATE;

    /* seek back to the starting position */
    fp->set_pos(startpos);

    /* check the timestamp */
    fp->read_bytes(buf, 24);
    if (memcmp(buf, G_image_loader->get_timestamp(), 24) != 0)
        return VMERR_WRONG_SAVED_STATE;

    /* 
     *   skip the image filename - since we already have an image file
     *   loaded, this information is of no use to us here (it's only
     *   useful when we want to restore a saved state before we know what
     *   the image file is) 
     */
    fp->set_pos_from_cur(fp->read_int2());

    /* 
     *   discard all undo information - any undo information we currently
     *   have obviously can't be applied to the restored state 
     */
    G_undo->drop_undo(vmg0_);

    /* 
     *   Disable garbage collection while restoring.  This is necessary
     *   because there are possible intermediate states where we have
     *   restored some of the objects but not all of them, so objects that
     *   are reachable from the fully restored state won't necessarily appear
     *   to be reachable from all possible intermediate states. 
     */
    old_gc_enabled = G_obj_table->enable_gc(vmg_ FALSE);

    err_try
    {
        /* forget any IntrinsicClass instances we created at startup */
        G_meta_table->forget_intrinsic_class_instances(vmg0_);

        /* load the object data from the file */
        if ((err = G_obj_table->restore(vmg_ fp, &fixups)) != 0)
            goto read_done;
        
        /* load the synthesized exports from the file */
        err = G_image_loader->restore_synth_exports(vmg_ fp, fixups);
        if (err != 0)
            goto read_done;

        /* 
         *   re-link to the exports and synthesized exports loaded from the
         *   saved session 
         */
        G_image_loader->do_dynamic_link(vmg0_);

        /* create any missing IntrinsicClass instances */
        G_meta_table->create_intrinsic_class_instances(vmg0_);

        /* perform any requested post-load object initializations */
        G_obj_table->do_all_post_load_init(vmg0_);

    read_done: ;
    }
    err_catch(exc)
    {
        /* remember the error code */
        err = exc->get_error_code();
    }
    err_end;

    /* we're done with the fixup table, so delete it if we created one */
    if (fixups != 0)
        delete fixups;

    /* restore the garbage collector's enabled state */
    G_obj_table->enable_gc(vmg_ old_gc_enabled);

    /* 
     *   explicitly run garbage collection, since any dynamic objects that
     *   were reachable before the restore only through non-transient
     *   references will no longer be reachable, all of the non-transient
     *   references having been replaced now 
     */
    G_obj_table->gc_full(vmg0_);

    /* if any error occurred, throw the error */
    if (err != 0)
        err_throw(err);

    /* success */
    return 0;
}

/* ------------------------------------------------------------------------ */
/*
 *   Reset to initial image file state 
 */
void CVmSaveFile::reset(VMG0_)
{
    /* 
     *   We cannot reset the state from within a recursive VM invocation
     *   -- the full machine state in a recursive invocation includes
     *   native machine stack state that we cannot modify.
     */
    if (G_interpreter->is_recursive_invocation(vmg0_))
        err_throw(VMERR_RECURSIVE_VM_CALL);

    /* 
     *   discard undo information, since it applies only to the current VM
     *   state and obviously is no longer relevant after we reset to the
     *   initial state 
     */
    G_undo->drop_undo(vmg0_);

    /* 
     *   discard all synthesized exports, since we want to dynamically link
     *   to the base image file state 
     */
    G_image_loader->discard_synth_exports();

    /* forget any IntrinsicClass instances we created at startup */
    G_meta_table->forget_intrinsic_class_instances(vmg0_);

    /* reset all objects to initial image file load state */
    G_obj_table->reset_to_image(vmg0_);

    /* 
     *   forget the previous dynamic linking information and relink to the
     *   image file again - this will ensure that any objects created after
     *   load are properly re-created now 
     */
    G_image_loader->do_dynamic_link(vmg0_);

    /* create any missing IntrinsicClass instances */
    G_meta_table->create_intrinsic_class_instances(vmg0_);

    /* perform any requested post-load object initializations */
    G_obj_table->do_all_post_load_init(vmg0_);

    /* 
     *   explicitly run garbage collection to clean up dynamic objects that
     *   are no longer reachable from the initial state
     */
    G_obj_table->gc_full(vmg0_);

    /* run the static initializers */
    G_image_loader->run_static_init(vmg0_);
}

