/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: updatedlg.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2006/04/28 14:53:20 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
 
#ifndef _COM_SUN_STAR_LANG_XMultiServiceFactory_HPP_
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_SYSTEM_XSYSTEMSHELLEXECUTE_HPP_
#include <com/sun/star/system/XSystemShellExecute.hpp>
#endif 
#ifndef _COM_SUN_STAR_SYSTEM_SYSTEMSHELLEXECUTEFLAGS_HPP_
#include <com/sun/star/system/SystemShellExecuteFlags.hpp>
#endif

#include <rtl/bootstrap.hxx>
#include <rtl/ustrbuf.hxx>
#include <rtl/uri.hxx>
#include <vos/thread.hxx>
#include <vos/process.hxx>
#include <vos/mutex.hxx>
#include <comphelper/processfactory.hxx>
#include <unotools/bootstrap.hxx>
#include <vcl/msgbox.hxx>
#include <vcl/svapp.hxx>

#include "ucbhelper/content.hxx"
#include "ucbhelper/commandenvironment.hxx"
#include "ucbhelper/activedatasink.hxx"
#include "com/sun/star/ucb/PostCommandArgument2.hpp"
#include "com/sun/star/beans/NamedValue.hpp"
#include "comphelper/seqstream.hxx"
#include "rtl/strbuf.hxx"
#include "app.hrc"
#include "sfxresid.hxx"
#include "updatedlg.hxx"

using namespace ::com::sun::star;
//using namespace ::ucb;
namespace css = ::com::sun::star;
namespace cssu = ::com::sun::star::uno;


class OCheckForUpdate;
class BaseMutex
{
    protected:
	    mutable ::osl::Mutex m_aMutex;
};

class CheckUpdateInfo : public BaseMutex, ::rtl::IReference
{
    friend class OCheckForUpdate;

    public:
        CheckUpdateInfo( const ::rtl::OUString aVersion ) :
          m_aVersion( aVersion )
          , m_bInProgress( false )
          , m_bIsUpdateAvailable( false )
          , m_bErrorOccured( false )
          , m_nRefCount( 0 ) {}
        ~CheckUpdateInfo() {}
        
        // IReference
        virtual oslInterlockedCount SAL_CALL acquire();
	    virtual oslInterlockedCount SAL_CALL release();

        const ::rtl::OUString getVersionString() const
        {
            return m_aVersion;
        }
        
        bool isCheckInProgress() const
        {
            osl::MutexGuard guard( m_aMutex );
            return m_bInProgress;
        }

        bool isUpdateAvailable() const
        {
            osl::MutexGuard guard( m_aMutex );
            return m_bIsUpdateAvailable;
        }

        bool hasErrorOccured() const
        {
            osl::MutexGuard guard( m_aMutex );
            return m_bErrorOccured;
        }
        
        const ::rtl::OUString& getURL() const
        {
            osl::MutexGuard guard( m_aMutex );
            return m_aURL;
        }

    private:
        void setInProgress( bool bInProgress )
        {
            osl::MutexGuard guard( m_aMutex );
            m_bInProgress = bInProgress;
        }

        void setUpdateAvailable( bool bUpdateAvailable )
        {
            osl::MutexGuard guard( m_aMutex );
            m_bIsUpdateAvailable = bUpdateAvailable;
        }
        
        void setErrorOccured( bool bErrorOccured )
        {
            osl::MutexGuard guard( m_aMutex );
            m_bErrorOccured = bErrorOccured;
        }
        
        void setUpdateURL( const ::rtl::OUString& aUpdateURL )
        {
            osl::MutexGuard guard( m_aMutex );
            m_aURL = aUpdateURL;
        }
        
        bool                m_bInProgress;
        bool                m_bIsUpdateAvailable;
        bool                m_bErrorOccured;
        ::rtl::OUString     m_aVersion;
        ::rtl::OUString     m_aURL;
        oslInterlockedCount m_nRefCount;
};

oslInterlockedCount CheckUpdateInfo::acquire()
{
    osl_incrementInterlockedCount( &m_nRefCount );
    return m_nRefCount;
}

oslInterlockedCount CheckUpdateInfo::release()
{
    if ( !osl_decrementInterlockedCount( &m_nRefCount ))
    {
        oslInterlockedCount nCount( m_nRefCount );
		delete this;
        return nCount;
    }

    return m_nRefCount;
} 

// ------------------------------------------------------------------


bool parseUpdateData( const ::rtl::OUString& data, 
                      ::rtl::OUString& url, 
                      cssu::Sequence< ::rtl::OUString >& retData )
{
    bool ret = true;
    ::rtl::OUString retUpdateUrl;
    cssu::Sequence< ::rtl::OUString > seqData(10);
    sal_Int32 dataLength = data.getLength();

    //The strings has the form yes$$$URL$$$key=value\0xa...
    //or no if no update is available.	
    if (data.equals(::rtl::OUString::createFromAscii("no")) == sal_True)
    {
        //no update available
		url = ::rtl::OUString();
        retData = cssu::Sequence< ::rtl::OUString >();
    }
    else if (data.indexOf(::rtl::OUString::createFromAscii("yes$$$"), 0) == 0)
    {
        if (dataLength < 7)
        {
            OSL_ASSERT(0);
            return false;
        }
        //get the index of the second $$$ and extract the URL
        sal_Int32 iSep = data.indexOf(::rtl::OUString::createFromAscii("$$$"), 6);
        if (iSep > 0)
        {
            url = data.copy(6, iSep - 6);
            //parse the key/value pairs.
            if (!(dataLength > iSep + 3))
            {
                OSL_ASSERT(0);
                return false;
            }
            ::rtl::OUString sKeyValue = data.copy(iSep + 3);
	
            sal_Int32 nIndex = 0;
			sal_Int32 nTokens = 0;
			::rtl::OUString aToken;
			do
			{
				aToken = sKeyValue.getToken( 0, '\xa', nIndex );
				if (aToken.getLength() > 0)
				{
					nTokens++;
					if (seqData.getLength() < nTokens)
						seqData.realloc(nTokens * 2);
					seqData[nTokens - 1] = aToken;
				}
			}
			while (nIndex >=0);

			OSL_ENSURE(nTokens > 0,"http message from server does not contain version.ini data");
			retData.realloc(nTokens);
			::rtl::OUString *arStrings = retData.getArray();
			for (int i = 0; i < nTokens; i++)
				arStrings[i] = seqData[i];
        }
        else
        {
            OSL_ENSURE(0, "The data returned from the server has the wrong format");
            ret = false;
        }		
    }
    else
    {
        ret = false;
    }
    return ret;
}

//We encode key value pairs so they correspond to the content type
//application/x-www-form-urlencoded according to 
//http://www.w3.org/TR/html4/interact/forms.html#h-17.13.4.
//The only difference is that wie encode space to %20 and not to '+'
//but this does not matter here.
::rtl::OUString encodeForPost(const css::beans::NamedValue& keyValue)
{
    // Uric derived char class. Additionally & = + are encoded
    static sal_Bool const aCharClass[] =
        { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
          0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          0, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, /* !"#$%&'()*+,-./*/
          1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, /*0123456789:;<=>?*/
          1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
          1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, /*PQRSTUVWXYZ[\]^_*/
          0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
          1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
        };

    ::rtl::OUString sKeyEncoded =	
          rtl::Uri::encode(keyValue.Name,
                           aCharClass,
                           rtl_UriEncodeIgnoreEscapes,
                           RTL_TEXTENCODING_UTF8);
    ::rtl::OUString sValue;
    OSL_VERIFY(keyValue.Value >>= sValue);
    
    ::rtl::OUString sValueEncoded =
          rtl::Uri::encode(sValue,
                           aCharClass,
                           rtl_UriEncodeIgnoreEscapes,
                           RTL_TEXTENCODING_UTF8);

    return sKeyEncoded + ::rtl::OUString::createFromAscii("=") + sValueEncoded;
}
/*

 All out parameters are only valid if the function returns true.
 @param xfac
 [in]

 @param serverUrl
 [in] url used to contact the server, which determines if an update is available.
 
 @param data
 [in] key/value pairs containing selected entries from versionrc and bootstraprc

 @param updateUrl
 [out] the URL that points to the place where the update can be obtained.
 If this param contains a value after return AND the function returns true
 then an update is available.

 @param seqRetProps
 [out] key/value pairs containing selected entries from versionrc and bootstraprc
 of the new or updated office.

 @return false - error occurred during processing.
*/
bool getUpdateInformation(
    const cssu::Reference< css::lang::XMultiServiceFactory >& xfac,
	const ::rtl::OUString& serverUrl,
    const cssu::Sequence< css::beans::NamedValue >& data,
    ::rtl::OUString& updateUrl,
    cssu::Sequence< ::rtl::OUString >& seqRetProps)
{
    bool ret = true;
	if (serverUrl.getLength() == 0)
	{
		OSL_ENSURE(0, "Did not get UpdateURL from version.ini");
		return false;
	}
    try
    {
        cssu::Reference< css::task::XInteractionHandler >
        xIH( xfac->createInstance(
            rtl::OUString::createFromAscii("com.sun.star.task.InteractionHandler")), 
            cssu::UNO_QUERY);
        ::ucb::CommandEnvironment *  pEnvironment = new ::ucb::CommandEnvironment( 
            xIH, cssu::Reference< css::ucb::XProgressHandler >() );
        cssu::Reference< css::ucb::XCommandEnvironment > xEnv(
            static_cast< css::ucb::XCommandEnvironment* >(pEnvironment));
		::ucb::Content aContent(serverUrl,	xEnv);
        ::rtl::OUString aCommandName = ::rtl::OUString::createFromAscii("post");
        css::ucb::PostCommandArgument2 aPostArgument;
        //create the XInputStream
        //first create a Sequence<sal_Int8> from data
        ::rtl::OUStringBuffer buff (4096);
		for (sal_Int32 i = 0; i < data.getLength(); i ++)
		{
			const ::rtl::OUString sEncoded = encodeForPost(data[i]);
            buff.append(sEncoded);
			if ( i < data.getLength() - 1)
				buff.appendAscii("&");	
        }
        ::rtl::OUString sData = buff.makeStringAndClear();
        ::rtl::OString sOStringData = ::rtl::OUStringToOString(sData, RTL_TEXTENCODING_UTF8);

		cssu::Sequence< sal_Int8 > seqData(
            reinterpret_cast< sal_Int8* >(const_cast< sal_Char* >(sOStringData.getStr())),
            sOStringData.getLength());
        cssu::Reference< css::io::XInputStream > xInput(
            static_cast< css::io::XInputStream* >(new comphelper::SequenceInputStream(seqData)));

        aPostArgument.Source = xInput;
        cssu::Reference<css::io::XActiveDataSink> aSink(new ::ucb::ActiveDataSink);
        aPostArgument.Sink = aSink;
		aPostArgument.MediaType = ::rtl::OUString::createFromAscii("application/x-www-form-urlencoded");
        aPostArgument.Referer = ::rtl::OUString();
        cssu::Any aCommandArgument;
        aCommandArgument <<= aPostArgument;
        aContent.executeCommand( aCommandName, aCommandArgument);
		
        cssu::Reference< css::io::XInputStream > xResultStream = aSink->getInputStream();
        if (xResultStream.is())
        {
            //Put the stream content into a string that we can parse.
            cssu::Sequence< sal_Int8 > seqBytes;
            const sal_Int32 bytesToRead = 4096;
            sal_Int32 bytesRead = 0;
            ::rtl::OStringBuffer buff (4096);
            do
            {
                bytesRead = xResultStream->readBytes(seqBytes, bytesToRead);
                buff.append(reinterpret_cast< const sal_Char* >(seqBytes.getConstArray()), bytesRead);
            }
            while(bytesRead == bytesToRead);

            ::rtl::OUString sRetData = ::rtl::OStringToOUString(
                buff.makeStringAndClear(),RTL_TEXTENCODING_UTF8);
            sRetData = sRetData.trim();
            if (sRetData.getLength() > 0)
            {
                ::rtl::OUString sUrl;
                cssu::Sequence< ::rtl::OUString > _seqData;
                if (parseUpdateData(sRetData, sUrl, _seqData) == false)
                {
                    ret = false;
				}
                else
                {
                    updateUrl = sUrl;
                    seqRetProps = _seqData;
                }
            }
            else
            {
                ret = false;
            }
        }
        else
        {
            ret = false;
        }
    }
    catch (cssu::Exception& )
    {
        OSL_ENSURE(sal_False, "Exception during UCB operatration.");
        ret = false;
    }
   
    return ret;
}

// ------------------------------------------------------------------

// class OThread
class OCheckForUpdate : public ::vos::OThread
{
	public:
        OCheckForUpdate( const ::rtl::Reference< CheckUpdateInfo >& aUpdateInfo,
                         const uno::Reference< lang::XMultiServiceFactory >& rSrvMgr );
        virtual ~OCheckForUpdate();

		virtual void SAL_CALL run();
		virtual void SAL_CALL onTerminated();

	private:
		void retrieveUpdateInfo( ::rtl::OUString& serverUrl, uno::Sequence< css::beans::NamedValue >& aUpdateInfo );

        ::rtl::Reference< CheckUpdateInfo > m_aUpdateInfo;
        uno::Reference< lang::XMultiServiceFactory > m_xSrvMgr;
};

OCheckForUpdate::OCheckForUpdate( const ::rtl::Reference< CheckUpdateInfo >& aUpdateInfo,
                                 const uno::Reference< lang::XMultiServiceFactory >& rSrvMgr ) :
    m_aUpdateInfo( aUpdateInfo ),
    m_xSrvMgr( rSrvMgr )
{
    m_aUpdateInfo->setInProgress( true );
}

OCheckForUpdate::~OCheckForUpdate()
{
}

void OCheckForUpdate::retrieveUpdateInfo(
    ::rtl::OUString& serverUrl, uno::Sequence< css::beans::NamedValue >& aUpdateInfo )
{
    ::rtl::OUString aEmptyString;
    ::rtl::OUString aProductKey( RTL_CONSTASCII_USTRINGPARAM( "ProductKey" ));
    ::rtl::OUString aBuildId( RTL_CONSTASCII_USTRINGPARAM( "buildid" ));
    ::rtl::OUString aProductPatch( RTL_CONSTASCII_USTRINGPARAM( "ProductPatch" ));
    ::rtl::OUString aProductSource( RTL_CONSTASCII_USTRINGPARAM( "ProductSource" ));
    ::rtl::OUString aAllLanguages( RTL_CONSTASCII_USTRINGPARAM( "AllLanguages" ));
    ::rtl::OUString aOS( RTL_CONSTASCII_USTRINGPARAM( "_OS" ));
    ::rtl::OUString aArch( RTL_CONSTASCII_USTRINGPARAM( "_ARCH" ));
	::rtl::OUString aUpdateUrl( RTL_CONSTASCII_USTRINGPARAM( "UpdateURL" ));
    
    // Read keys from version.ini|rc
    ::rtl::OUString         aIniName;
    ::rtl::OUString         aValue;
    ::vos::OStartupInfo     aInfo;
//    ::rtl::OUStringBuffer   aKeyValueBuffer;
    
    aInfo.getExecutableFile( aIniName );
    sal_uInt32 lastIndex = aIniName.lastIndexOf('/');
    if ( lastIndex > 0 )
    {
        aIniName    = aIniName.copy( 0, lastIndex+1 );
        aIniName    += ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "version" ));
#ifdef WNT
        aIniName    += ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( ".ini" ));
#else
        aIniName    += ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "rc" ));
#endif
    }
        
    aUpdateInfo.realloc( 7 );
    
    ::rtl::OUString  aOSStr;
    ::rtl::OUString  aArchStr;
    ::rtl::Bootstrap aVersionIni( aIniName );
    
    // operating system and architecture
    ::rtl::Bootstrap::get( rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "_OS" )), aOSStr );
    ::rtl::Bootstrap::get( rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "_ARCH" )), aArchStr );

    // product key from bootstrap.ini|rc
    aUpdateInfo[0].Name = aProductKey;
    aUpdateInfo[0].Value <<= utl::Bootstrap::getProductKey( aEmptyString );

    // build id
    aVersionIni.getFrom( aBuildId, aValue, aEmptyString );
    aUpdateInfo[1].Name = aBuildId;
    aUpdateInfo[1].Value <<= aValue;
    
    // product patch
    aVersionIni.getFrom( aProductPatch, aValue, aEmptyString );
    aUpdateInfo[2].Name = aProductPatch;
    aUpdateInfo[2].Value <<= aValue;
    
    // product source
    aVersionIni.getFrom( aProductSource, aValue, aEmptyString );
    aUpdateInfo[3].Name = aProductSource;
    aUpdateInfo[3].Value <<= aValue;
    
    // all languages
    aVersionIni.getFrom( aAllLanguages, aValue, aEmptyString );
    aUpdateInfo[4].Name = aAllLanguages;
    aUpdateInfo[4].Value <<= aValue;
    
    // OS
    aUpdateInfo[5].Name = aOS;
    aUpdateInfo[5].Value <<= aOSStr;
    
    // architecture
    aUpdateInfo[6].Name = aArch;
    aUpdateInfo[6].Value <<= aArchStr;
	//UpdateURL
	aVersionIni.getFrom(aUpdateUrl, serverUrl, aEmptyString);
}

void SAL_CALL OCheckForUpdate::run()
{
    // make blocking call here
    uno::Sequence< css::beans::NamedValue > aUpdateInfo;
    ::rtl::OUString aUpdateURL;
	uno::Sequence< ::rtl::OUString > aNewVersionRC;
	::rtl::OUString aServerURL;
    retrieveUpdateInfo( aServerURL, aUpdateInfo );
    bool bResult = getUpdateInformation( m_xSrvMgr,
		                                 aServerURL,
                                         aUpdateInfo,
                                         aUpdateURL,
                                         aNewVersionRC );

    if ( bResult && ( aUpdateURL.getLength() > 0 ))
    {
        m_aUpdateInfo->setUpdateURL( aUpdateURL );
        m_aUpdateInfo->setUpdateAvailable( true );
    }
    else if ( !bResult )
        m_aUpdateInfo->setErrorOccured( true );
    
    m_aUpdateInfo->setInProgress( false );
}

void SAL_CALL OCheckForUpdate::onTerminated()
{
	delete this;
}

// ------------------------------------------------------------------

SfxUpdateDialog::SfxUpdateDialog( Window* pParent ) :
    ModalDialog( pParent, SfxResId( DLG_CHECKFORONLINEUPDATE ) ),
    m_aCancelPB( this, ResId( RID_CHECKFORONLINEUPDATE_PB_CANCEL )),
    m_aMessageFT( this, ResId( RID_CHECKFORONLINEUPDATE_FT_MSG )),
    m_pProgress( 0 ),
    m_nProgress( 0 )
{
    FreeResource();

    m_pProgress = new StatusBar( this );

    Size aProgressSize;
    Size aWinSize = GetSizePixel();
    
    const long nHeight = 20;
    
    long nSpace = long(( aWinSize.Width() * 24 ) / 100 );
    long nXPos  = ( nSpace / 2 );
    long nYPos  = ::std::max( long( 0 ), ( aWinSize.Height() / 2 ) - ( nHeight / 2 ) + ( nHeight / 8 ));

    aProgressSize.Width()   = aWinSize.Width() - nSpace;
    aProgressSize.Height()  = nHeight;
    
    m_pProgress->SetPosSizePixel( Point( nXPos, nYPos ),
                                  aProgressSize );
    m_pProgress->StartProgressMode( String() );
    m_pProgress->Show();

	// Create CheckUpdateInfo instance
    CheckUpdateInfo* pCheckUpdateInfo = new CheckUpdateInfo( ::rtl::OUString() );
    m_aUpdateInfo.set( pCheckUpdateInfo );

    // Create thread for blocking update check
    // Thread instance will destroy itself after leaving run() method
    // Communication will be done through the refcounted CheckUpdateInfo instance
    OCheckForUpdate* pCheckForUpdate = 
        new OCheckForUpdate( m_aUpdateInfo, 
                             ::comphelper::getProcessServiceFactory() );
    pCheckForUpdate->create();
    
    // Start progress timer
    m_aProgressTimer.SetTimeout( 250 );
	m_aProgressTimer.SetTimeoutHdl( LINK( this, SfxUpdateDialog, ProgressHdl_Impl ) );
    m_aProgressTimer.Start();
}

SfxUpdateDialog::~SfxUpdateDialog()
{
    delete m_pProgress;
}

IMPL_LINK( SfxUpdateDialog, CancelHdl_Impl, PushButton *, pButton )
{
    m_aProgressTimer.Stop();
    m_pProgress->SetProgressValue( 100 );
    
    return 1;
}
	
IMPL_LINK( SfxUpdateDialog, ProgressHdl_Impl, Timer *, pTimer )
{
    m_nProgress += 5;
    m_pProgress->SetProgressValue( m_nProgress );
    
    if ( m_nProgress == 100 )
        m_nProgress = 0;

    if ( !m_aUpdateInfo->isCheckInProgress() )
    {
        m_pProgress->SetProgressValue( 100 );
        
        if ( m_aUpdateInfo->isUpdateAvailable() )
        {
            InfoBox aInfoBox( this, SfxResId( INFOBOX_NEW_UPDATE_AVAILABLE ));
            short nRet = aInfoBox.Execute();
            if ( nRet == RET_OK )
            {
                // open URL in system web brower
                ::rtl::OUString aURLString = m_aUpdateInfo->getURL();

                uno::Reference< ::com::sun::star::system::XSystemShellExecute > xSystemShellExecute( 
                    ::comphelper::getProcessServiceFactory()->createInstance(
                        ::rtl::OUString::createFromAscii( "com.sun.star.system.SystemShellExecute" )), 
                    uno::UNO_QUERY );
                if ( xSystemShellExecute.is() )
                {
                    try
                    {
                        // start browser
                        xSystemShellExecute->execute( aURLString, 
                                                      ::rtl::OUString(), 
                                                      ::com::sun::star::system::SystemShellExecuteFlags::DEFAULTS );
                    }
                    catch ( ::com::sun::star::lang::IllegalArgumentException& )
                    {
						vos::OGuard aGuard( Application::GetSolarMutex() );
						ErrorBox( this, SfxResId( MSG_ERR_NO_WEBBROWSER_FOUND )).Execute();
                    }
                    catch ( ::com::sun::star::system::SystemShellExecuteException& )
                    {
						vos::OGuard aGuard( Application::GetSolarMutex() );
						ErrorBox( this, SfxResId( MSG_ERR_NO_WEBBROWSER_FOUND )).Execute();
                    }
                }
            }
        }
        else if ( !m_aUpdateInfo->hasErrorOccured() )
        {
            InfoBox aInfoBox( this, SfxResId( INFOBOX_NO_NEW_UPDATE_AVAILABLE ));
            aInfoBox.Execute();
        }
        
        Close();
    }
    else
    {
        m_aProgressTimer.Start();
    }

    return 1;
}
