/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: fflink.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 14:48:36 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include "fastfsys.hxx"
#include "shortcut.hxx"

#include <prewin.h>
#include <shutil.h>
#include <postwin.h>

#include <parser.hxx>

#include <rtl/ustring.hxx>

using namespace rtl;

#ifdef UNICODE
#define lstrchr		wcschr
#define lstrrchr	wcsrchr
#else
#define lstrchr		strchr
#define lstrrchr	strrchr
#endif

#define FlushPrivateProfile( pszFile ) \
	WritePrivateProfileString( NULL, NULL, NULL, pszFile )


//---------------------------------------------------------------------------
// Helper functions
//---------------------------------------------------------------------------

static BOOL IsFileProtocol( const String & crURL )
{
	USHORT	nIndex = 0;
	String	aProtocol = crURL.GetToken( 0, ':', nIndex );

	if ( !aProtocol.Len() )
		// No protocol -> must be file
		return TRUE;
	else if ( 1 == aProtocol.Len() && isalpha(aProtocol.GetChar(0)) )
		// Protocol is drive letter -> file
		return TRUE;
	else if ( COMPARE_EQUAL == aProtocol.ICompare( "file" ) )
		// File protocol
		return TRUE;
	else
		// No stuff for LNK files
		return FALSE;
}

//---------------------------------------------------------------------------

static String URL2Path( const String & crURL )
{
	TCHAR	szPath[MAX_PATH] = "";

	WIN_SHGetPathFromURL( crURL, szPath );
	return szPath;
}

//---------------------------------------------------------------------------

static String Path2URL( const String & crPath, const WIN32_FIND_DATA *pfdInfo )
{
	TCHAR	szURL[MAX_PATH + sizeof("file://")] = "";

	if ( WIN_SHGetURLFromPath( crPath, szURL ) && pfdInfo )
	{
		if ( pfdInfo->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
		{
			if ( szURL[lstrlen(szURL) - 1] != '/' )
				lstrcat( szURL, "/" );
		}
		
	}

	return szURL;
}

//---------------------------------------------------------------------------
// KeyValue implementation
//---------------------------------------------------------------------------

// No default constructor allowed

KeyValue::KeyValue()
{
	throw;
}

//---------------------------------------------------------------------------

KeyValue::~KeyValue()
{
}

//---------------------------------------------------------------------------

KeyValue::KeyValue( const String & crName, const String & crValue ) :
m_aName( crName ),
m_aValue( crValue )
{
}

//---------------------------------------------------------------------------

KeyValue::KeyValue( const KeyValue & crKeyValue ) :
m_aName( crKeyValue.GetValue())
{
}

//---------------------------------------------------------------------------

KeyValue &KeyValue::operator = ( const KeyValue & crKeyValue )
{
	SetName( crKeyValue.GetName() );
	SetValue( crKeyValue.GetValue() );
	return *this;
}

//---------------------------------------------------------------------------

void KeyValue::SetName( const String & crName )
{
	m_aName = crName;
}

//---------------------------------------------------------------------------

void KeyValue::SetValue( const String & crValue )
{
	m_aValue = crValue;
}

//---------------------------------------------------------------------------

const String & KeyValue::GetName() const
{
	return m_aName;
}

//---------------------------------------------------------------------------

const String & KeyValue::GetValue() const
{
	return m_aValue;
}

//---------------------------------------------------------------------------
// Impl_Shortcut implementation
//---------------------------------------------------------------------------

// No default constructors

//---------------------------------------------------------------------------

Impl_Shortcut::Impl_Shortcut()
{
	throw;
}

Impl_Shortcut::Impl_Shortcut( const Impl_Shortcut & )
{
	throw;
}

Impl_Shortcut &Impl_Shortcut::operator = ( const Impl_Shortcut & )
{
	throw;

	return *this;
}

//---------------------------------------------------------------------------

Impl_Shortcut::Impl_Shortcut( const String & crLanguage, const ItemIDPath & crFolderPath ) :
m_eLoadedFormat( SHORTCUT_FORMAT_BESTFIT ),
m_aFolderIDPath( crFolderPath ),
m_aFolderPath( crFolderPath.GetHostNotationPath() ),
m_aLanguage( crLanguage )
{
}

//---------------------------------------------------------------------------

Impl_Shortcut::~Impl_Shortcut()
{
	// Delete list items

	DeleteListItems();
}

//---------------------------------------------------------------------------

void Impl_Shortcut::DeleteListItems()
{
	KeyValue	*pKey = m_aList.First();

	while ( pKey )
	{
		delete pKey;
		pKey = m_aList.Next();
	}

	m_aList.Clear();
}

//---------------------------------------------------------------------------

KeyValue *Impl_Shortcut::FindKeyByName( const String & crName ) const
{
	KeyValue	*pKey = ((Impl_Shortcut *)this)->m_aList.First();
	KeyValue	*pKeyFound = NULL;

	while ( pKey && !pKeyFound )
	{
		if ( COMPARE_EQUAL == pKey->GetName().ICompare( crName ) )
			pKeyFound = pKey;

		pKey = ((Impl_Shortcut *)this)->m_aList.Next();
	}

	return pKeyFound;
}

//---------------------------------------------------------------------------

void Impl_Shortcut::SetValueContent( const String & crName, const String & crValue )
{
	KeyValue	*pKeyFound = FindKeyByName( crName );

	if ( pKeyFound )
		pKeyFound->SetValue( crValue );
	else
		m_aList.Insert( new KeyValue( crName, crValue ) );
}

//---------------------------------------------------------------------------

const String & Impl_Shortcut::GetValueContent( const String & crName ) const
{
	KeyValue	*pKeyFound = FindKeyByName( crName );

	return pKeyFound ? pKeyFound->GetValue() : "";
}

//---------------------------------------------------------------------------

UINT32 Impl_Shortcut::GetValueNameCount() const
{
	return m_aList.Count();
}

//---------------------------------------------------------------------------

const String & Impl_Shortcut::GetValueName( UINT32 nIndex ) const
{
	KeyValue	*pKey = m_aList.GetObject( nIndex );

	return pKey ? pKey->GetName() : "";
}

//---------------------------------------------------------------------------

FSysError Impl_Shortcut::Store( const String & crTitle, ItemIDPath & rPath, ShortcutFormat eFormat )
{
	// Store using forced format

	switch ( eFormat )
	{
	case SHORTCUT_FORMAT_URL:
		return StoreURL( crTitle, rPath );
	case SHORTCUT_FORMAT_SYSTEM:
		return StoreLink( crTitle, rPath );
	case SHORTCUT_FORMAT_BESTFIT:
	default:
		break;
	}

	// Store using last loaded format

	switch ( m_eLoadedFormat )
	{
	case SHORTCUT_FORMAT_URL:
		return StoreURL( crTitle, rPath );
	case SHORTCUT_FORMAT_SYSTEM:
		return StoreLink( crTitle, rPath );
	case SHORTCUT_FORMAT_BESTFIT:
	default:
		break;
	}

	// Determine format by data

	if ( !HasExtendedKeys() )
		 return StoreLink( crTitle, rPath );
	else
		return StoreURL( crTitle, rPath );
}

//---------------------------------------------------------------------------

FSysError Impl_Shortcut::Load( const ItemIDPath & crPath )
{
	// Delete list items

	DeleteListItems();

	// Determine file path

	ItemIDPath	aFull( m_aFolderIDPath );
	aFull += crPath;

	// Determine file extension

	String	aFileExt;
	String	aFilePath = aFull.GetHostNotationPath();
	USHORT	nIndex = aFilePath.SearchBackward( '.' );

	if ( nIndex != STRING_NOTFOUND )
		aFileExt = aFilePath.Copy( nIndex );

	if ( COMPARE_EQUAL == aFileExt.ICompare( ".url" ) )
		return LoadURL( aFilePath );
	else
		return LoadLink( aFilePath );
}

//---------------------------------------------------------------------------

static HRESULT WINAPI SHCoCreateInstance( LPVOID lpszReserved, REFCLSID clsid, LPUNKNOWN pUnkUnknown, REFIID iid, LPVOID *ppv )
{
	HRESULT	hResult = E_NOTIMPL;
	HMODULE	hModShell = GetModuleHandle( "SHELL32" );

	if ( hModShell != NULL )
	{
		typedef	HRESULT (WINAPI *SHCoCreateInstance_PROC)( LPVOID lpszReserved, REFCLSID clsid, LPUNKNOWN pUnkUnknwon, REFIID iid, LPVOID *ppv );

		SHCoCreateInstance_PROC	lpfnSHCoCreateInstance = (SHCoCreateInstance_PROC)GetProcAddress( hModShell, MAKEINTRESOURCE(102) );

		if ( lpfnSHCoCreateInstance )
			hResult = lpfnSHCoCreateInstance( lpszReserved, clsid, pUnkUnknown, iid, ppv );
	}
	return hResult;
}

#define CoCreateInstance( clsid, pUnk, clsctx, iid, ppv ) SHCoCreateInstance( NULL, clsid, pUnk, iid, ppv )

FSysError Impl_Shortcut::LoadLink( const String & crFileName )
{
	BOOL		bOffice2000Hack = FALSE;
	HRESULT		hResult;
	IShellLink	*pLink = NULL;
	FSysError	nError = ERRCODE_IO_UNKNOWN;

	hResult = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, IID_IShellLink, (LPVOID *)&pLink);

	// Because there is no proxy for Shell interfaces use undocumented SHELL export 102

	if ( FAILED(hResult) )
		hResult = SHCoCreateInstance(NULL, CLSID_ShellLink, NULL, IID_IShellLink, (LPVOID *)&pLink);


	if ( SUCCEEDED(hResult) )
	{
		IPersistFile	*pFile = NULL;

		hResult = pLink->QueryInterface(IID_IPersistFile, (LPVOID *)&pFile);

		if ( SUCCEEDED(hResult) )
		{
			UniString	aOlePath( crFileName );

			if ( SUCCEEDED(pFile->Load( aOlePath.GetStr(), TRUE ) ) )
			{
				// URL

				TCHAR	szPath[MAX_PATH] = "";
				TCHAR	szExpanded[MAX_PATH] = "";
				WIN32_FIND_DATA	fdInfo;

				ZeroMemory( &fdInfo, sizeof(fdInfo) );
				hResult = pLink->GetPath( szPath, MAX_PATH, &fdInfo, 0 );

				if ( SUCCEEDED(hResult) && szPath[0] )
				{
					WIN32_FIND_DATA	fdInfo2;


					ZeroMemory( &fdInfo2, sizeof(fdInfo2) );
					HANDLE	hFind = FindFirstFile( szPath, &fdInfo2 );

					// HRO #75283# Detect Office 2000
					// link. The fdInfo is filled with
					// rubbish after IShellLink::GetPath

					if ( hFind != INVALID_HANDLE_VALUE && !(fdInfo2.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && stricmp(fdInfo2.cFileName, fdInfo.cFileName) )
						bOffice2000Hack = TRUE;

					FindClose( hFind );

					// Zuerst muessen Environment-Strings aufgeloest werden (%windir% etc.)

					DWORD	dwResultingChars = ExpandEnvironmentStrings( szPath, szExpanded, MAX_PATH );

					// Wenn wir keinen absoluten Pfad haben, dann wird der Dateiname ohne Protocol
					// als TargetURL verkauft. Das fuehrt dazu, dass der SFX damit nichts anfangen kann
					// und dann wiederum FastFSys zum Ausfuehren benutzt.

					if ( !dwResultingChars || !strchr( szExpanded, '\\' ) )
					{
						String	aURL = szExpanded;
						SetValueContent( "URL", aURL );
					}
					else
					{
						String	aURL = Path2URL( szExpanded, &fdInfo );
						SetValueContent( "URL", aURL );
					}
				}
				else
				{
					LPITEMIDLIST	pidl = NULL;

					hResult = pLink->GetIDList( &pidl );

					if ( SUCCEEDED(hResult) && pidl )
					{
						CItemIDList aIDList( pidl );

						String aURL = "file:///";
						aURL += CreateStringFromItemIDList( pidl );
						SetValueContent( "URL", aURL );

						WIN_SHFree( pidl );
					}
					else
					{
						// Wie oben, wenn auch der PIDL leer ist. Wieso zum Teufel liefern die Methoden bei
						// nicht qualifizierten Pfade als Target nix zurck und setzen trotzdem NOERROR ???
						String	aURL = szPath;
						SetValueContent( "URL", aURL );
					}
				}

				// Arguments

				TCHAR	szArguments[MAX_PATH + 1024] = "";

				pLink->GetArguments( szArguments, 1024 );

				// HRO: #75283#
				// Hack for Office2000 Links
				// Somehow we have to pack the of the Link-Path
				// 0x07 (Backspace) won't appear in an argument List so we Use this

				if ( bOffice2000Hack)
				{
					strcat( szArguments, "\x07" );
					strcat( szArguments, crFileName );
				}

				SetValueContent( "Arguments", szArguments );

				// WorkingDirectory

				TCHAR	szWorkingDirectory[2 * MAX_PATH] = "";

				pLink->GetWorkingDirectory( szWorkingDirectory, MAX_PATH );
				SetValueContent( "WorkingDirectory", szWorkingDirectory );


				// Icon

				TCHAR	szIconFile[MAX_PATH] = "";
				int		nIcon = 0;
				BOOL	bIconLocation = FALSE;

				pLink->GetIconLocation( szIconFile, MAX_PATH, &nIcon );

				// Auf den Rueckgabewert kann man nichts geben. Der ist auch NOERROR, wenn kein Icon geliefert
				// wird. Also gucken, ob etwas in den String geschrieben wurde.

				if ( !szIconFile[0] )
				{
					if ( szExpanded[0] )
						lstrcpy( szIconFile, szExpanded );
					else
					{
						SHFILEINFO	sfi;

						ZeroMemory( &sfi, sizeof(sfi) );

						SHGetFileInfo( crFileName.GetStr(), 0, &sfi, sizeof(sfi), SHGFI_ICONLOCATION );

						if ( sfi.szDisplayName[0] )
						{
							nIcon = sfi.iIcon;
							lstrcpy( szIconFile, sfi.szDisplayName );
							bIconLocation = TRUE;
						}
						else
						{
							// No icon location, so take the linkfile itself
							lstrcpy( szIconFile, crFileName.GetStr() );
						}
					}
				}
				else
					bIconLocation = TRUE;

				SetValueContent( "IconFile", szIconFile );

				if ( bIconLocation )
				{
					if ( nIcon == -1 )
						nIcon = 0;
					SetValueContent( "IconIndex", String(nIcon) );
				}

				// Title = Description

#if 0	// Don't use Description of links as a title
				TCHAR	szDescription[MAX_PATH] = "";

				pLink->GetDescription( szDescription, MAX_PATH );
				SetValueContent( "Title", szDescription );
#endif

				m_eLoadedFormat = SHORTCUT_FORMAT_SYSTEM;
				m_aCurrentFilePath = crFileName;

				nError = ERRCODE_NONE;
			}

			pFile->Release();
		}

		pLink->Release();
	}

	return nError;
}

//---------------------------------------------------------------------------

FSysError Impl_Shortcut::LoadURL( const String & crFileName )
{
	FSysError		nError	= ERRCODE_IO_UNKNOWN;
	WIN32_FIND_DATA	fdInfo;
	HANDLE			hFind;

	hFind = FindFirstFile( crFileName, &fdInfo );
	if ( hFind )
	{
		TCHAR	szBuffer[16384] = "";
		LPCTSTR	pszLines = szBuffer;
		String	aSection = "InternetShortcut";

		// Collect keys

		GetPrivateProfileSection( aSection.GetStr(), szBuffer, 16384, crFileName.GetStr() );
		while ( *pszLines )
		{
			LPTSTR	pAssign = lstrchr( pszLines, '=' );

			if ( pAssign )
			{
				*pAssign = '\0';
				SetValueContent( pszLines, pAssign + 1 );
				*pAssign = '=';
			}

			pszLines += lstrlen(pszLines) + 1;
		}

		// Read UNICODE

		String	aSectionW( aSection );
		aSectionW += ".W";

		GetPrivateProfileSection( aSectionW.GetStr(), szBuffer, 16384, crFileName.GetStr() );

		pszLines = szBuffer;
		while ( *pszLines )
		{
			LPTSTR	pAssign = lstrchr( pszLines, '=' );

			if ( pAssign )
			{
				*pAssign = '\0';
				LPCSTR	pValue = pAssign + 1;
#if 1
				OString strValue = OUStringToOString( OStringToOUString( OString(pValue), RTL_TEXTENCODING_UTF7 ), GetSystemCharSet() );

				SetValueContent( pszLines, strValue.getStr() );
#else
				WCHAR	wszValue[2048];
				CHAR	szValue[2048];


				MultiByteToWideChar( CP_UTF7, 0, pValue, -1, wszValue, 2048 );
				WideCharToMultiByte( CP_ACP, 0, wszValue, -1, szValue, 2048, NULL, FALSE );

				SetValueContent( pszLines, szValue );
#endif
				*pAssign = '=';

			}

			pszLines += lstrlen(pszLines) + 1;
		}

		// Read Title from language specific section

#if	0
		if ( m_aLanguage.Len() )
		{
			TCHAR	szTitle[512] = "";

			aSection += '-';
			aSection += m_aLanguage;

			GetPrivateProfileString( aSection.GetStr(), "Title", "", szTitle, 512, crFileName.GetStr() );

			KeyValue	*pKey = FindKeyByName( "Title" );

			if ( pKey )
			{
				if ( szTitle[0] )
					pKey->SetValue( szTitle );
			}
			else
				SetValueContent( "Title", szTitle );
		}
#else

		// HRO #67395#: Overwrite all availiable keys from language specific section

		aSection += '-';
		aSection += m_aLanguage;

		GetPrivateProfileSection( aSection.GetStr(), szBuffer, 16384, crFileName.GetStr() );

		pszLines = szBuffer;
		while ( *pszLines )
		{
			LPTSTR	pAssign = lstrchr( pszLines, '=' );

			if ( pAssign )
			{
				*pAssign = '\0';
				SetValueContent( pszLines, pAssign + 1 );
				*pAssign = '=';
			}

			pszLines += lstrlen(pszLines) + 1;
		}
#endif

		// Read UNICODE

		aSection += ".W";

		GetPrivateProfileSection( aSection.GetStr(), szBuffer, 16384, crFileName.GetStr() );

		pszLines = szBuffer;
		while ( *pszLines )
		{
			LPTSTR	pAssign = lstrchr( pszLines, '=' );

			if ( pAssign )
			{
				*pAssign = '\0';
				LPCSTR	pValue = pAssign + 1;
#if 1
				OString strValue = OUStringToOString( OStringToOUString( OString(pValue), RTL_TEXTENCODING_UTF7 ), GetSystemCharSet() );

				SetValueContent( pszLines, strValue.getStr() );
#else
				WCHAR	wszValue[2048];
				CHAR	szValue[2048];


				MultiByteToWideChar( CP_UTF7, 0, pValue, -1, wszValue, 2048 );
				WideCharToMultiByte( CP_ACP, 0, wszValue, -1, szValue, 2048, NULL, FALSE );

				SetValueContent( pszLines, szValue );
#endif
				*pAssign = '=';

			}

			pszLines += lstrlen(pszLines) + 1;
		}




		FindClose( hFind );

		m_eLoadedFormat = SHORTCUT_FORMAT_URL;
		m_aCurrentFilePath = crFileName;

		nError = ERRCODE_NONE;
	}


	return nError;
}

//---------------------------------------------------------------------------

FSysError Impl_Shortcut::StoreLink( const String & crTitle, ItemIDPath & rNewIDPath )
{
	HRESULT		hResult;
	IShellLink	*pLink = NULL;
	FSysError	nError = ERRCODE_IO_UNKNOWN;

	hResult = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, IID_IShellLink, (LPVOID *)&pLink);

	if ( SUCCEEDED(hResult) )
	{
		IPersistFile	*pFile = NULL;

		hResult = pLink->QueryInterface(IID_IPersistFile, (LPVOID *)&pFile);

		if ( SUCCEEDED(hResult) )
		{
			KeyValue	*pKey;

			// Set URL

			pKey = FindKeyByName( "URL" );
			if ( pKey )
			{
				String aPath = pKey->GetValue();
				pLink->SetPath( URL2Path(aPath.GetStr()) );
			}

			// Set Working Directory

			pKey = FindKeyByName( "WorkingDirectory" );
			if ( pKey )
			{
				String aWorkingDirectory = pKey->GetValue();
				pLink->SetWorkingDirectory( aWorkingDirectory.GetStr() );
			}

			// Set Arguments

			pKey = FindKeyByName( "Arguments" );
			if ( pKey )
			{
				String aArguments = pKey->GetValue();
				pLink->SetArguments( aArguments.GetStr() );
			}

			// Set Icon

			pKey = FindKeyByName( "IconFile" );
			if ( pKey )
			{
				String	aIconFile = pKey->GetValue();
				int		nIcon = 0;

				pKey = FindKeyByName( "IconIndex" );
				if ( pKey )
					nIcon = pKey->GetValue();

				pLink->SetIconLocation( aIconFile.GetStr(), nIcon );
			}

			// Set Title

			pKey = FindKeyByName( "Title" );
			if ( pKey )
			{
				String	aTitle = pKey->GetValue();
				pLink->SetDescription( aTitle.GetStr() );
			}

			// Save the file

			String	aFilePath( GetFilePathFromTitle( crTitle, SHORTCUT_FORMAT_SYSTEM ) );
			UniString	aOlePath( aFilePath );

			if ( SUCCEEDED(pFile->Save( aOlePath.GetStr(), TRUE ) ) )
			{
				if ( SUCCEEDED(pFile->SaveCompleted( aOlePath.GetStr() ) ) )
				{
					ItemIDPath	aFull( aFilePath );
					ItemIDPath	aParent;
					aFull.Split( aParent, rNewIDPath );

					nError = ERRCODE_NONE;
				}
			}

			pFile->Release();
		}

		pLink->Release();
	}

	return nError;
}

//---------------------------------------------------------------------------

FSysError Impl_Shortcut::StoreURL( const String & crTitle, ItemIDPath & rNewIDPath )
{
	FSysError	nError = ERRCODE_IO_UNKNOWN;
	KeyValue	*pKey = m_aList.First();
	BOOL		bIconIndex = FALSE, bIconFile = FALSE;
	String		aTargetURL;

	String	aPath( GetFilePathFromTitle( crTitle, SHORTCUT_FORMAT_URL ) );

	while ( pKey )
	{
		String	aSection = "InternetShortcut";

		WritePrivateProfileString( aSection, pKey->GetName(), pKey->GetValue(), aPath.GetStr() );

		String aSectionA( aSection );
		aSectionA += ".A";

		WritePrivateProfileString( aSectionA, pKey->GetName(), pKey->GetValue(), aPath.GetStr() );

		String aSectionW( aSection );
		aSectionW += ".W";

		WritePrivateProfileString( aSectionW, pKey->GetName(), OUStringToOString( OStringToOUString( OString(pKey->GetValue()), GetSystemCharSet() ), RTL_TEXTENCODING_UTF7 ), aPath.GetStr() );

		// HRO #67395#: Write every key into language specific section

#if 0
		if ( COMPARE_EQUAL == pKey->GetName().ICompare( "Title" ) )
#endif
		{
			if ( m_aLanguage.Len() )
			{
				String	aLangSection( aSection );
				aLangSection += '-';
				aLangSection += m_aLanguage;
				WritePrivateProfileString( aLangSection, pKey->GetName(), pKey->GetValue(), aPath.GetStr() );

				// Write UNICODE

				aLangSection += ".W";

#if 1
				OString strValue = OUStringToOString( OStringToOUString( OString(pKey->GetValue()), GetSystemCharSet() ), RTL_TEXTENCODING_UTF7 );

				WritePrivateProfileString( aLangSection, pKey->GetName(), strValue.getStr(), aPath.GetStr() );
#else
				WCHAR	wszValue[2048];
				CHAR	szValue[2048];

				MultiByteToWideChar( CP_ACP, 0, pKey->GetValue(), -1, wszValue, 2048 );
				WideCharToMultiByte( CP_UTF7, 0, wszValue, -1, szValue, 2048, NULL, FALSE );

				WritePrivateProfileString( aLangSection, pKey->GetName(), szValue, aPath.GetStr() );
#endif
			}
		}
		
		if ( COMPARE_EQUAL == pKey->GetName().ICompare( "IconFile" ) )
			bIconFile = TRUE;
		else if ( COMPARE_EQUAL == pKey->GetName().ICompare( "IconIndex" ) )
			bIconIndex = TRUE;
		else if ( COMPARE_EQUAL == pKey->GetName().ICompare( "URL" ) )
			aTargetURL = pKey->GetValue();

		pKey = m_aList.Next();
	}

	// if no iconfile specified determine it from target if possible

	if ( !bIconFile && IsFileProtocol( aTargetURL) )
	{
		ItemIDPath	aIDPath( aTargetURL );
		String		aIconFile;
		String		aIconIndex;
		SHFILEINFO	sfi;

		ZeroMemory( &sfi, sizeof(sfi) );
		BOOL fSuccess = SHGetFileInfo( (LPCSTR)aIDPath.GetDataPtr(), 0, &sfi, sizeof(sfi), SHGFI_ICONLOCATION | SHGFI_PIDL );

		if ( fSuccess )
		{
			aIconFile = sfi.szDisplayName;

			if ( sfi.iIcon == -1 )
				sfi.iIcon = 0;

			aIconIndex = sfi.iIcon;
		}

		if ( aIconFile.Len() )
		{
			WritePrivateProfileString( "InternetShortcut", "IconFile", aIconFile, aPath.GetStr() );
			WritePrivateProfileString( "InternetShortcut", "IconIndex", aIconIndex, aPath.GetStr() );
			bIconFile = TRUE;
			bIconIndex = TRUE;
		}
	}

	// if no iconindex specified default to 0

	if ( !bIconIndex && bIconFile )
		WritePrivateProfileString( "InternetShortcut", "IconIndex", "0", aPath.GetStr() );

	// HRO: Force flushing the file content for Win 9x

	FlushPrivateProfile( aPath.GetStr() );

	WIN32_FIND_DATA	fdInfo;
	HANDLE	hFind	= FindFirstFile( aPath.GetStr(), &fdInfo );
	if ( INVALID_HANDLE_VALUE != hFind )
	{
		ItemIDPath	aFull( aPath );
		ItemIDPath	aParent;
		aFull.Split( aParent, rNewIDPath );
		FindClose( hFind );

		nError = ERRCODE_NONE;
	}

	return nError;
}

//---------------------------------------------------------------------------

String Impl_Shortcut::GetFilePathFromTitle( const String & crTitle, ShortcutFormat eFormat )
{
	String	aPath;

	if ( crTitle.Len() )
	{
		aPath = m_aFolderPath;

		if ( aPath.GetChar( aPath.Len() - 1 ) != '\\' )
			aPath += '\\';
		aPath += crTitle;
	}
	else	// Use last loaded file name
	{
		TCHAR	szDrive[_MAX_DRIVE];
		TCHAR	szDir[_MAX_DIR];
		TCHAR	szFName[_MAX_FNAME];
		TCHAR	szExt[_MAX_EXT];

		_splitpath( m_aCurrentFilePath.GetStr(), szDrive, szDir, szFName, szExt );
		aPath = szDrive;
		aPath += szDir;
		aPath += szFName;
	}

		switch( eFormat )
		{
		case SHORTCUT_FORMAT_SYSTEM:
			aPath += ".lnk";
			break;
		case SHORTCUT_FORMAT_URL:
			aPath += ".url";
			break;
		default:
			aPath += ".";
			break;
		}

	return aPath;
}

//---------------------------------------------------------------------------

BOOL Impl_Shortcut::HasExtendedKeys() const
{
	BOOL		bExtended = FALSE;
	KeyValue	*pKey = ((Impl_Shortcut *)this)->m_aList.First();

	while ( pKey && !bExtended )
	{
		String	aKeyName = pKey->GetName();

		bExtended = 
			aKeyName.ICompare( "Title" ) != COMPARE_EQUAL &&
			aKeyName.ICompare( "URL" ) != COMPARE_EQUAL &&
			aKeyName.ICompare( "IconFile" ) != COMPARE_EQUAL &&
			aKeyName.ICompare( "IconIndex" ) != COMPARE_EQUAL &&
			aKeyName.ICompare( "WorkingDirectory" ) != COMPARE_EQUAL &&
			aKeyName.ICompare( "Arguments" ) != COMPARE_EQUAL;

		pKey = ((Impl_Shortcut *)this)->m_aList.Next();
	}

	// Check for file protocol

	if ( !bExtended )
	{
		pKey = FindKeyByName( "URL" );
		if ( pKey )
			bExtended = !IsFileProtocol( pKey->GetValue() );
	}

	return bExtended;
}


