/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: bytarray.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 09:06:12 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/


#include <rtl/alloc.h>
#include <rtl/memory.h>

#include <vos/diagnose.hxx>
#include <vos/object.hxx>
#include <vos/macros.hxx>
#include <vos/bytarray.hxx>

#ifdef _USE_NAMESPACE
using namespace vos;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//  class ByteArray
//

VOS_IMPLEMENT_CLASSINFO(VOS_CLASSNAME(OByteArray, vos), VOS_NAMESPACE(OByteArray, vos), VOS_NAMESPACE(OObject, vos), 0);

OByteArray::OByteArray()
{
    m_Size    = 0L;
    m_pArray  = 0L;
    m_bResize = sal_True;
	m_bOwner  = sal_True;
}

// Constructor: assume that the array of bytes in question begins
// at the given address, and is of the given size
OByteArray::OByteArray (void* pbuf, sal_uInt32 size, sal_Bool resize)
{
    m_pArray  = (sal_uInt8 *)pbuf;
    m_Size    = size;
    m_bResize = resize;
	m_bOwner  = sal_False;
}

OByteArray::OByteArray (const OByteArray& b)
{
    m_pArray  = b.m_pArray;
    m_Size    = b.m_Size;
    m_bResize = b.m_bResize;
	m_bOwner  = sal_False;
}

OByteArray::~OByteArray ()
{
	if (m_bOwner)
		rtl_freeMemory(m_pArray);
}

void OByteArray::setAllBytesTo(sal_uInt8 b)
{
    if (m_Size > 0 && m_pArray != NULL) 
        rtl_fillMemory(m_pArray, m_Size, b);
}

sal_uInt8 OByteArray::operator[](sal_uInt32 index)
{
    VOS_ASSERT(index >= 0 && index < m_Size);

    if (! m_pArray) 
	{
        sal_uInt8 c = 0;

        return (c);
    }

    return m_pArray[index];
}

// Comparison methods
// For the methods taking character pointer arguments, it is
// assumed that the size of the byte array is the same as our
// size.	   

sal_Bool OByteArray::operator== (const void* b) const
{
    return rtl_compareMemory(m_pArray, b, m_Size) == 0;
}

sal_Bool OByteArray::operator== (const OByteArray& o) const
{
    return ((o.m_Size == m_Size) && 
		    (rtl_compareMemory(m_pArray, o.m_pArray, m_Size) == 0));
}

sal_Bool OByteArray::operator!= (const OByteArray& o) const
{
    return ((o.m_Size != m_Size) || 
		    (rtl_compareMemory(m_pArray, o.m_pArray, m_Size) != 0));
}

sal_Bool OByteArray::operator!= (const void* b) const
{
    return (rtl_compareMemory(m_pArray, b, m_Size) != 0);
}

sal_Bool OByteArray::operator>= (const OByteArray& o) const
{
    register sal_Int32 sz = VOS_MIN(m_Size, o.m_Size);
    return (rtl_compareMemory(m_pArray, o.m_pArray, sz) >= 0);
}

sal_Bool OByteArray::operator>= (const void* b) const
{
    return (rtl_compareMemory(m_pArray, b, m_Size) >= 0);
}

sal_Bool OByteArray::operator<= (const OByteArray& o) const
{
    register sal_Int32 sz = VOS_MIN(m_Size, o.m_Size);
    return (rtl_compareMemory(m_pArray, o.m_pArray, sz) <= 0);
}

sal_Bool OByteArray::operator<= (const void* b) const
{
    return (rtl_compareMemory(m_pArray, b, m_Size) <= 0);
}

sal_Bool OByteArray::operator> (const OByteArray& o) const
{
    register sal_Int32 sz = VOS_MIN(m_Size, o.m_Size);
    return (rtl_compareMemory(m_pArray, o.m_pArray, sz) > 0);
}

sal_Bool OByteArray::operator> (const void* b) const
{
    return (rtl_compareMemory(m_pArray, b, m_Size) > 0);
}

sal_Bool OByteArray::operator< (const OByteArray& o) const
{
    register sal_Int32 sz = VOS_MIN(m_Size, o.m_Size);
    return (rtl_compareMemory(m_pArray, o.m_pArray, sz) < 0);
}

sal_Bool OByteArray::operator< (const void* b) const
{
    return (rtl_compareMemory(m_pArray, b, m_Size) < 0);
}

sal_Int16 OByteArray::Compare (const OByteArray& o) const
{
    return (rtl_compareMemory(m_pArray, o.m_pArray, m_Size));
}

// Sub-array extraction
//
// Return the sub-array beginning at the given index (using
// 0-origin indexing) and of the given size. Note that the
// returned OByteArray uses (part of) the same memory as we do.
//
OByteArray OByteArray::operator() (sal_uInt32 start, sal_uInt32 size) const
{
    OByteArray p(m_pArray + start, size);
    
	return (p);
}

sal_uInt32 OByteArray::getCommonPrefixLength(const sal_uInt8* p)
{
    if (! m_pArray)
        return (0); // No memory

    sal_uInt32 i;
    
	for (i = 0; i < m_Size; i++) 
        if (m_pArray[i] != p[i]) return (i);
    
	return (m_Size);
}

OByteArray OByteArray::getSuffix(sal_uInt32 index) const
{
    return OByteArray((*this)(index, (m_Size > index) ?
										(m_Size - index) : 0L));
}

sal_Int32 OByteArray::getLongValueAt(sal_uInt32 index) const
{
    if (!m_pArray)
        return (0); // No memory

    sal_Int32 value;
    
    rtl_copyMemory(&value, &m_pArray[index], sizeof(sal_Int32));
    
	return (value);
}

sal_Int16 OByteArray::getShortValueAt(sal_uInt32 index) const
{
    if (!m_pArray)
        return (0); // No memory

    sal_Int16 value;

    rtl_copyMemory(&value, &m_pArray[index], sizeof(sal_Int16));
    
	return (value);
}

sal_uInt32 OByteArray::copyTo(void* buffer, sal_uInt32 count, sal_uInt32 pos)
{
    if ((count > 0) && m_pArray && (pos >= 0) && (pos < m_Size)) 
	{
        sal_Int32 n = VOS_MIN(m_Size - pos, count);
        rtl_copyMemory(buffer, m_pArray + pos, n);

        return (n);
    }
    
	return (0);
}

sal_uInt32 OByteArray::copyFrom(const void* buffer, sal_uInt32 count, sal_uInt32 pos) const
{
    if ((count > 0) && m_pArray && (pos >= 0) && (pos < m_Size)) 
	{
        sal_Int32 n = VOS_MIN(m_Size - pos, count);
        rtl_copyMemory(m_pArray + pos, buffer, n);
        
		return (n);
    }

    return (0);
}

// Assignment

void OByteArray::operator= (const OByteArray& b)
{
    if (this == &b)
        return;

    if (! m_pArray)
        return; // No memory
    if (! b.m_pArray)
        return; // Null second array

    if (m_pArray != b.m_pArray)  
        rtl_copyMemory(m_pArray, b.asPtr(), VOS_MIN(m_Size, b.getSize()));
}

void OByteArray::operator= (const void* p)
{
    if (!m_pArray)
        return;
    if (!p) return;

    if (m_pArray != p)  
        rtl_copyMemory(m_pArray, p, m_Size);
}

OByteArray& OByteArray::operator= (sal_Int16 value)
{
    if (! m_pArray)
        return *this; // No memory

    rtl_copyMemory(m_pArray, &value, VOS_MIN(m_Size, sizeof value));

    return *this;
}

OByteArray& OByteArray::operator= (sal_Int32 value)
{
    if (!m_pArray)
        return *this; // No memory
    
	rtl_copyMemory(m_pArray, &value, VOS_MIN(m_Size, sizeof value));

    return *this;
}

sal_Bool OByteArray::readFrom(const OStream& s)
{
	return (s.read(m_pArray, m_Size) > 0);
}

sal_Bool OByteArray::writeTo(OStream& s) const
{
    return (s.write(m_pArray, m_Size) == (sal_Int32)m_Size);
}

sal_Bool OByteArray::changeSize(sal_uInt32 Size)
{
	if (m_bResize)
	{		
		sal_uInt8 *pArray = m_pArray;
		m_pArray = (sal_uInt8 *)rtl_allocateMemory(Size);

		if (m_pArray != NULL)
		{
			rtl_copyMemory(m_pArray, pArray, VOS_MIN(m_Size, Size));

			m_Size = Size;

			if (m_bOwner)
				rtl_freeMemory(pArray);

			m_bOwner = sal_True;

			return (sal_True);
		}

		m_pArray = pArray;
	}

    return (sal_False);
}

