//
//  MWDocument.mm
//  MySQL Workbench
//
//  Created by Alfredo Kojima on 05/6/17.
//  Copyright MySQL AB 2005 . All rights reserved.
//

#import "MWDocument.h"
#import "MWCanvasViewTab.h"
#import "MWLayerTree.h"

#import <WBCanvas/MCanvasScrollView.h>
#import <WBCanvas/MGenericCanvasView.h>
#import <MySQLToolsCommon/MTabView.h>
#import <MySQLToolsCommon/MPreferences.h>
#import <MySQLToolsCommon/mxUtils.h>
#import <MySQLToolsCommon/NSView_extras.h>
#import <MySQLToolsCommon/MDialogs.h>
#import <MySQLGRT/MGRT.h>
#import <MySQLGRT/MGRTModuleLoader.h>
#import <MySQLGRT/MGRTTableEditor.h>
#import <MySQLGRT/MGRTViewEditor.h>
#import <MySQLGRT/MGRTRoutineGroupEditor.h>
#import <MySQLGRT/MGRTSchemaEditor.h>

#include <WBCanvas/myx_gc_datatypes.h>
#include <WBCanvas/myx_gc_canvas.h>
#include <WBCanvas/myx_gc_texture.h>

#include <WBCanvas/myx_grt_wb_public_interface.h>

#import "MWReverseEngineering.h"
#import "MWSynchronizeDB.h"
#import "MWDocumentProperties.h"

#import "MWorkbench.h"

@interface MSolidColorImageRep : NSImageRep
{
  NSColor *_color;
}
- (id)initWithColor:(NSColor*)color;
@end


@implementation MSolidColorImageRep
- (id)initWithColor:(NSColor*)color
{
  self= [super init];
  if (self)
    _color= [color retain];
  return self;
}

- (void)dealloc
{
  [_color release];
  [super dealloc];
}

- (BOOL)draw
{
  NSRect rect;
  rect.origin= NSMakePoint(0, 0);
  rect.size= [self size];
  [_color set];
  NSRectFill(rect);
  [[NSColor lightGrayColor] set];
  [NSBezierPath strokeRect:rect];
  return YES;
}
@end



static const NSSize MWDefaultCanvasSize= {10000.0, 7000.0};

@implementation MWDocument


- (void)viewAdded:(id)aView
{
  CGCView *view= (CGCView*)[aView pointerValue];
  string name= view->propertyGet("name",0);

  [_canvas createCanvasViewWithView:view];

  MWCanvasViewTab *cview= [[MWCanvasViewTab alloc] initWithContent:_scrollView
                                                            canvas:_canvas
                                                              view:view
                                                              name:[NSString stringWithUTF8String:name.c_str()]];
  [tabView addTabViewItem:cview];

  [cview release];
}



- (MWCanvasViewTab*)canvasTabForView:(CGCView*)view
{
  unsigned int i, c= [tabView numberOfPages];
  
  for (i= 0; i < c; i++)
  {
    id page= [tabView pageAtIndex:i];
    if ([page gcView] == view)
      return page;
  }
  return nil;
}


- (void)tabView:(MTabView*)aTabView closeTab:(id)tab
{
  
}


- (void)viewChanged:(id)aView
{
  CGCView *view= (CGCView*)[aView pointerValue];
  string name= view->propertyGet("name",0);
  
  MWCanvasViewTab *cview= [self canvasTabForView:view];
  if (cview && strcmp(name.c_str(), [[cview label] UTF8String]) != 0)
    [cview setLabel:[NSString stringWithUTF8String:name.c_str()]];  
}


- (void)refreshWhenIdle:(id)dummy
{
  if ([[[NSThread currentThread] threadDictionary] objectForKey:@"mainThread"])
  {
    if (!_postedIdle)
    {
      _postedIdle= YES;

      NSNotification* notification = [NSNotification notificationWithName:@"MWBIdleNotification" object:nil];
      [[NSNotificationQueue defaultQueue] enqueueNotification:notification postingStyle:NSPostWhenIdle];
    }
  }
  else
    [self performSelectorOnMainThread:@selector(refreshWhenIdle:)
                           withObject:nil
                        waitUntilDone:NO];
}


- (void)switchToView:(id)viewValue
{
  CGCView* view= (CGCView*)[viewValue pointerValue];
  
  for (int i= 0; i < [tabView numberOfPages]; i++)
  {
    MWCanvasViewTab *tab= (MWCanvasViewTab*)[tabView pageAtIndex:i];
    
    if ([tab gcView] == view)
    {
      [tabView selectPage:i];
      break;
    }
  }
}


- (void)layoutChanged:(id)newType
{
  if (!newType || [newType isEqualToString:@"default"])
  {
    
  }
  else if ([newType isEqualToString:@"eer"])
  {
  }
}


static void viewCallback(MYX_GRT *grt, MYX_GRT_VALUE *grtView, CGCView *view, MYX_WB_GC_CHANGE change, void *data)
{
  MWDocument *self= (MWDocument*)data;
  
  switch (change)
  {
    case MYX_WBGC_ADDED:
      [self performSelectorOnMainThread:@selector(viewAdded:)
                             withObject:[NSValue valueWithPointer:view]
                          waitUntilDone:NO];
      break;
    case MYX_WBGC_SWITCHED:
      [self performSelectorOnMainThread:@selector(switchToView:)
                             withObject:[NSValue valueWithPointer:view]
                          waitUntilDone:NO];
      break;
    case MYX_WBGC_MODIFIED:
      [self performSelectorOnMainThread:@selector(viewChanged:)
                             withObject:[NSValue valueWithPointer:view]
                          waitUntilDone:NO];
      break;
    case MYX_WBGC_SELECTION_CHANGE:
      [self searchPlugins:nil];
      break;
    case MYX_WBGC_RUBBERBAND_STARTED:
    case MYX_WBGC_RUBBERBAND_STOPPED:
      [self refreshWhenIdle:nil];
      break;
    case MYX_WBGC_LAYOUT_CHANGED:
    {
      MGRTValue value(MGRTValue::fromGlobal([self->_grt grt], "/workbench/model/currentView/connectionLayoutClass"));
      if (value.isValid())
      {
        [self performSelectorOnMainThread:@selector(layoutChanged:)
                               withObject:[NSString stringWithUTF8String:value.asString()]
                            waitUntilDone:NO];
      }
      break;
    }
    default:
      break;
  }
  [self updateChangeCount: NSChangeDone];
}


static void layerCallback(MYX_GRT *grt, MYX_GRT_VALUE *layer, CFigureInstance *gcLayer, MYX_WB_GC_CHANGE change, void *data)
{
  MWDocument *self= (MWDocument*)data;
  id newNode= nil;

  switch (change)
  {
    case MYX_WBGC_ADDED:
      newNode= [self->_layerDS handleLayerAdd:layer];
      break;
    case MYX_WBGC_MODIFIED:
      [self->_layerDS handleLayerChange:layer];
      break;
    case MYX_WBGC_REMOVED:
      [self->_layerDS handleLayerDelete:layer];
      break;
  }
  [self->layerTree reloadData];
  if (newNode)
    [self->layerTree expandItem:newNode];
  
  [self refreshWhenIdle:nil];
  
  [self updateChangeCount: NSChangeDone];
}


- (void)refreshAfterElementChange:(id)unused
{
  [layerTree reloadData];
  [schemaTree reloadData];

  [self refreshWhenIdle:nil];

  [self updateChangeCount: NSChangeDone];  
}


static void elementCallback(MYX_GRT *grt, MYX_GRT_VALUE *element, CFigureInstance *gcElem, MYX_WB_GC_CHANGE change, void *data)
{
  MWDocument *self= (MWDocument*)data;

  switch (change)
  {
    case MYX_WBGC_ADDED:
      [self->_layerDS handleElementAdd:element];
      [self performSelectorOnMainThread:@selector(refreshAfterElementChange:) withObject:nil waitUntilDone:YES];
      break;
    case MYX_WBGC_REMOVED:
      [self->_layerDS handleElementDelete:element];
      [self performSelectorOnMainThread:@selector(refreshAfterElementChange:) withObject:nil waitUntilDone:YES];
      break;
    case MYX_WBGC_MODIFIED:
      if (element)
      {
        [self->_layerDS handleElementChange:element];
        [self performSelectorOnMainThread:@selector(refreshAfterElementChange:) withObject:nil waitUntilDone:YES];
      }
      break;
  }
}


// for obj-c module support
- (BOOL)exportsMethod:(NSString*)method
{
  if ([method isEqualTo:@"editSchema:"])
    return YES;
  if ([method isEqualTo:@"editTable:"])
    return YES;
  if ([method isEqualTo:@"editView:"])
    return YES;
  if ([method isEqualTo:@"editRoutineGroup:"])
    return YES;
  if ([method isEqualTo:@"popupPluginsMenu:"])
    return YES;
  return NO;
}


- (void)toggleDrawer:(id)sender
{
  if ([drawer state] == NSDrawerOpenState)
  {
    [drawer close];
    [sender setTitle:NSLocalizedString(@"Open Drawer",nil)];
  }
  else
  {
    [drawer open];
    [sender setTitle:NSLocalizedString(@"Close Drawer",nil)];
  }
}


- (IBAction)createSchema:(id)sender
{
  MGRTSchemaEditor *editor= [[MGRTSchemaEditor alloc] init];
  
  [editor setDelegate:self];
  [editor setMGRT:_grt catalogs:MGRTValue([_grt globalValue:"/workbench/catalog"])];
  [editor createNew];
  [editor showWindow:nil];
  
  [_openEditors setObject:editor forKey:NSStr([editor objectId])];
  [editor release];
}


- (void)editSelectedSchemaObject:(id)sender
{
  MYX_GRT_VALUE *object= [catalogDS objectAtNode:[schemaTree itemAtRow:[schemaTree selectedRow]]];

  if (myx_grt_value_get_type(object) == MYX_DICT_VALUE)
  {
    [_grt performModule:@"Base"
              procedure:@"editObj"
              arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:object]]];
  }
}


- (MYX_GRT_VALUE*)editSchema:(NSArray*)args
{
  MYX_GRT_VALUE *object= (MYX_GRT_VALUE*)[[args objectAtIndex:0] pointerValue];
  
  if (object)
  {
    NSString *schemaId= [NSString stringWithUTF8String:myx_grt_dict_id_item_as_string(object)];
    MGRTSchemaEditor *editor;
    
    if ((editor= [_openEditors objectForKey:schemaId]))
    {
      [[editor window] makeKeyAndOrderFront:nil];
    }
    else
    {
      editor= [[MGRTSchemaEditor alloc] init];
      [editor setDelegate:self];
      [editor setMGRT:_grt catalogs:MGRTValue([_grt globalValue:"/workbench/catalog"])];
      [editor editObject:object];
      [editor showWindow:nil];
      
      [_openEditors setObject:editor forKey:schemaId];
      [editor release];
    }
  }
  return NULL;
}


- (MYX_GRT_VALUE*)editTable:(NSArray*)args
{
  MYX_GRT_VALUE *object= (MYX_GRT_VALUE*)[[args objectAtIndex:0] pointerValue];

  if (object)
  {
    NSString *tableId= [NSString stringWithUTF8String:myx_grt_dict_id_item_as_string(object)];
    MGRTTableEditor *editor;
    
    if ((editor= [_openEditors objectForKey:tableId]))
    {
      [[editor window] makeKeyAndOrderFront:nil];
    }
    else
    {
      editor= [[MGRTTableEditor alloc] init];
      [editor setDelegate:self];
      [editor setMGRT:_grt catalogs:MGRTValue([_grt globalValue:"/workbench/catalog"])];
      [editor editObject:object];
      
      if (0)
      {
        [self newDockTabWithContent:[editor contentView] label:[NSString stringWithFormat:@"Table %@",[editor editedObjectName]]];
      }
      else
        [editor showWindow:nil];
      
      [_openEditors setObject:editor forKey:tableId];
      [editor release];
    }
  }
  return NULL;
}


- (MYX_GRT_VALUE*)editView:(NSArray*)args
{
  MYX_GRT_VALUE *object= (MYX_GRT_VALUE*)[[args objectAtIndex:0] pointerValue];
  
  if (object)
  {
    NSString *viewId= [NSString stringWithUTF8String:myx_grt_dict_id_item_as_string(object)];
    MGRTViewEditor *editor;
    
    if ((editor= [_openEditors objectForKey:viewId]))
    {
      [[editor window] makeKeyAndOrderFront:nil];
    }
    else
    {
      editor= [[MGRTViewEditor alloc] init];
      [editor setDelegate:self];
      [editor setMGRT:_grt catalogs:MGRTValue([_grt globalValue:"/workbench/catalog"])];
      [editor editObject:object];
      [editor showWindow:nil];
      
      [_openEditors setObject:editor forKey:viewId];
      [editor release];
    }
  }
  return NULL;
}


- (MYX_GRT_VALUE*)editRoutineGroup:(NSArray*)args
{
  MYX_GRT_VALUE *object= (MYX_GRT_VALUE*)[[args objectAtIndex:0] pointerValue];
  
  if (object)
  {
    NSString *viewId= [NSString stringWithUTF8String:myx_grt_dict_id_item_as_string(object)];
    MGRTViewEditor *editor;
    
    if ((editor= [_openEditors objectForKey:viewId]))
    {
      [[editor window] makeKeyAndOrderFront:nil];
    }
    else
    {
      editor= [[MGRTRoutineGroupEditor alloc] init];
      [editor setDelegate:self];
      [editor setMGRT:_grt catalogs:MGRTValue([_grt globalValue:"/workbench/catalog"])];
      [editor editObject:object];
      [editor showWindow:nil];
      
      [_openEditors setObject:editor forKey:viewId];
      [editor release];
    }
  }
  return NULL;
}


- (void)setupDefaultAppOptions
{
  MGRTValue options(MGRTValue::createTypedDict());
#define SET_DEFAULT(options, key, value) if (!options.isSet(key)) options.set(key, value)
  SET_DEFAULT(options, "DefaultModelBaseWidth", (float)10000.0);
  SET_DEFAULT(options, "DefaultModelBaseHeight", (float)7000.0);
  SET_DEFAULT(options, "PkColumnNameTemplate", "id%tablename%");
  SET_DEFAULT(options, "FKNameTemplate", "FK%table%");
  SET_DEFAULT(options, "FKColumnNameTemplate", "FK%table%%column%");
  SET_DEFAULT(options, "Rel1nSourceOptional", (int)1);
  SET_DEFAULT(options, "Rel1nTargetOptional", 0);
  SET_DEFAULT(options, "Rel11SourceOptional", 1);
  SET_DEFAULT(options, "Rel11TargetOptional", 0);
  SET_DEFAULT(options, "RelnmSourceOptional", 1);
  SET_DEFAULT(options, "RelnmTargetOptional", 1);
  SET_DEFAULT(options, "DockGrtShell", 1);
  SET_DEFAULT(options, "DockGrtShellAutoAddSnippets", 0);
  SET_DEFAULT(options, "DockEditors", 1);
  SET_DEFAULT(options, "SideBarOneWidth", 1);
  SET_DEFAULT(options, "SideBarWidthView", 230);
  SET_DEFAULT(options, "SideBarWidthGrtShell", 350);
  SET_DEFAULT(options, "EditorPanelHeight", 280);
  SET_DEFAULT(options, "InitialZoom", (float)0.75);
  SET_DEFAULT(options, "DefaultRdbms", "Mysql");
  SET_DEFAULT(options, "DefaultRdbmsVersion", "5.1.6");
  SET_DEFAULT(options, "FKDeleteRule", "NO ACTION");
  SET_DEFAULT(options, "FKUpdateRule", "NO ACTION");
  SET_DEFAULT(options, "AuxTableTemplate", "%stable%_has_%dtable%");
#undef SET_DEFAULT
  [_grt setGlobalValue:options.grtValue() forPath:"/app/options"];
}


- (id)init
{
  self = [super init];
  if (self) 
  {    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(performIdleTasks:)
                                                 name:@"MWBIdleNotification"
                                               object:nil];
    
    _openEditors= [[NSMutableDictionary alloc] init];
    
    _pluginsPopup= [[NSMenu alloc] initWithTitle:@"Plugins"];
    
    _grt= [[MGRT alloc] init];
    
    [_grt setInputHandler:self selector:@selector(askInput:)];
    
    _grtShell= [[MGRTShell alloc] initWithMGRT:_grt];
    [_grt initializeGRT:[[NSBundle bundleForClass:[MGRT class]] resourcePath]];
    [_grt scanStructsInPath:[[NSBundle mainBundle] resourcePath]];
    myx_register_grt_wb_module([_grt grt]);

    [_grt globalAppDict]; // this will create /app in case it doesnt exist

    [_grt loadSubtree:"/app/options"
             fromFile:[[MPreferences preferences] pathForFile:@"workbench_app_options.xml"]];
    [self setupDefaultAppOptions];
    
    [_grt scanModulesInPath:[[NSBundle mainBundle] resourcePath]];

    [[_grt objCLoader] registerObject:self asModule:@"WorkbenchUi"];

    [_grt performModule:@"Workbench"
              procedure:@"registerEditors"
              arguments:nil];

    _toolCursors= [[NSArray arrayWithObjects:
      [NSCursor arrowCursor],
      [NSCursor openHandCursor],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"rubber"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"table"] hotSpot:NSMakePoint(4,4)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"view"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"routine"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"res11"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"res1n"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"resnm"] hotSpot:NSMakePoint(0,0)] autorelease],      
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"image"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"note"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"layer"] hotSpot:NSMakePoint(0,0)] autorelease],
      nil] retain];

    _currentElement= new MGRTValue();
    _currentObject= new MGRTValue();
    
    _layerIcon= [[NSImage imageNamed:@"db.workbench.Layer.16x16"] retain];
    _tableIcon= [[NSImage imageNamed:@"db.workbench.TableElement.16x16"] retain];
    _viewIcon= [[NSImage imageNamed:@"db.workbench.ViewElement.16x16"] retain];
    _routinesIcon= [[NSImage imageNamed:@"db.workbench.RoutinesElement.16x16"] retain];
  }
  return self;
}

- (void)dealloc
{
  [_grt saveSubtree:"/app/options"
             toFile:[[MPreferences preferences] pathForFile:@"workbench_app_options.xml"]];

  myx_grt_wb_bridge_set_callbacks([_grt grt], NULL, NULL, NULL, NULL);

  [_grt performModule:@"Workbench"
            procedure:@"shutdownWorkbench"
            arguments:nil];
  
  delete _currentElement;
  delete _currentObject;

  [_objectForPlugin release];
  [_pluginsPopup release];
  [[NSNotificationCenter defaultCenter] removeObserver:self];
  [_layerIcon release];
  [_tableIcon release];
  [_viewIcon release];
  [_routinesIcon release];
  [_toolCursors release];
  [_canvas release];
  [_grtShell hide:nil];
  [_grtShell release];
  [_grt release];
  [_scrollView release];
  [_pluginArray release];
  [super dealloc];
}

- (NSString *)windowNibName
{
  return @"DocumentWindow";
}


- (void)performIdleTasks:(NSNotification*)notif
{
  [[_canvas openGLContext] makeCurrentContext];
  myx_grt_wb_bridge_process_pending([_grt grt]);
  _postedIdle= NO;
  [_canvas setNeedsDisplay:YES];
  
  [self searchPlugins:nil];
}


- (void)setupGRT
{
  MYX_GRT_VALUE *result;
  //g_log_set_always_fatal(G_LOG_LEVEL_CRITICAL);
  
  if (!(result= [_grt performModule:@"Workbench"
                           function:@"initWorkbench"
                          arguments:[NSArray arrayWithObject:[[NSBundle mainBundle] resourcePath]]]))
  {
    NSRunAlertPanel(@"Error",
                    @"The GRT environment for the Workbench could not be initialized.\nPlease verify your installation.",
                    @"OK",nil,nil);
  }
  else
    myx_grt_value_release(result);

  myx_grt_wb_bridge_initialize([_grt grt], [_canvas canvas], "/workbench/model");

  myx_grt_wb_bridge_set_callbacks([_grt grt], self,
                                  viewCallback,
                                  layerCallback,
                                  elementCallback);
}


- (void)showGRTShell:(id)sender
{
  [_grtShell show:nil];
}


- (void)setupMenu:(NSMenu*)menu withColors:(NSArray*)colors
{
  int i;
  
  while ([menu numberOfItems])
    [menu removeItemAtIndex:0];
  
  for (i= 0; i < [colors count]; i+=2)
  {
    id item;
    unsigned int color;
    NSImage *image= [[NSImage alloc] initWithSize:NSMakeSize(20,15)];
    NSImageRep *imageRep;
    NSScanner *scanner= [NSScanner scannerWithString:[colors objectAtIndex:i+1]];
    [scanner setScanLocation:1];
    [scanner scanHexInt:&color];
    imageRep= [[MSolidColorImageRep alloc] initWithColor:[NSColor colorWithDeviceRed:((color>>16)&0xff)/255.0
                                                                               green:((color>>8)&0xff)/255.0
                                                                                blue:(color&0xff)/255.0
                                                                               alpha:1.0]];
    [image addRepresentation:imageRep];
    [imageRep release];
        
    item= [[NSMenuItem alloc] init];
    [item setTitle:@""];
    [item setImage:image];
    [menu addItem:item];
    [item setRepresentedObject:[colors objectAtIndex:i]];
    [image release];
    [item release];
  }
}


- (void)setupMenu:(NSMenu*)menu 
        withItems:(MYX_GRT_VALUE*)items
         titleKey:(const char*)key
{
  int i, c= myx_grt_list_item_count(items);
  
  while ([menu numberOfItems])
    [menu removeItemAtIndex:0];
  
  for (i= 0; i < c; i++)
  {
    MYX_GRT_VALUE *item= myx_grt_list_item_get(items, i);
    
    [menu addItemWithTitle:key ? [NSString stringWithUTF8String:myx_grt_dict_item_get_as_string(item, key)] : [NSString stringWithUTF8String:myx_grt_value_as_string(item)]
                    action:nil
             keyEquivalent:@""];
    
    [[menu itemAtIndex:i] setRepresentedObject:[NSValue valueWithPointer:item]];
  }
}


- (void)setupMenu:(NSMenu*)menu 
 withCharsetItems:(MYX_GRT_VALUE*)items
{
  int i, c= myx_grt_list_item_count(items);
  
  while ([menu numberOfItems])
    [menu removeItemAtIndex:0];
  
  for (i= 0; i < c; i++)
  {
    MYX_GRT_VALUE *item= myx_grt_list_item_get(items, i);
    NSString *name= [NSString stringWithUTF8String:myx_grt_dict_item_get_as_string(item, "name")];
    MYX_GRT_VALUE *collations= myx_grt_dict_item_get_value(item, "collations");
    int j, jc;
    
    jc= myx_grt_list_item_count(collations);
    for (j= 0; j < jc; j++)
    {
      const char *coll= myx_grt_value_as_string(myx_grt_list_item_get(collations, j));
      
      [menu addItemWithTitle:NSStr(coll)
                      action:nil
               keyEquivalent:@""];
      
      [[menu itemAtIndex:[menu numberOfItems]-1] setRepresentedObject:name];
    }
  }
}


- (void)setSplitted:(BOOL)flag
{
  if (!flag)
  {
    NSRect frame= [splitView frame];
    NSView *parent= [splitView superview];
///this is broken, needs to fix (or just remove and put the shell etc in a drawer)
    [tabView retain];
    [tabView removeFromSuperview];
    [splitView retain];
    [splitView removeFromSuperview];
    [parent addSubview:tabView];
    [tabView setFrame:frame];
    [tabView release];
  }
}


- (void)windowControllerDidLoadNib:(NSWindowController *)aController
{
  CGenericCanvas *canvas;
  NSBundle *bundle= [NSBundle mainBundle];
  NSRect rect;
  NSArray *zoomSteps;
  unsigned int i;

  [super windowControllerDidLoadNib:aController];

  [tabView setMaxTabSize:100];
  [tabView setDelegate:self];
  
  // setup canvas
  rect.origin= NSMakePoint(0, 0);
  rect.size= [tabView contentSize];
  _scrollView= [[MCanvasScrollView alloc] initWithFrame:rect];
  rect.size= [_scrollView contentSize];
  
  _canvas= [[MWEditCanvasView alloc] initWithFrame:rect];
  [_canvas setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];
  [_scrollView setContentCanvas:_canvas];
  [_canvas setNextResponder:[modelWindow contentView]];
  [_canvas setMGRT:_grt];
  [_canvas setStatusText:statusText];
  [_canvas registerForDraggedTypes:[NSArray arrayWithObject:MGRTTreeNodePboardType]];

  
  myx_grt_wb_load_style_file([_canvas canvas],
                             [[bundle pathForResource:@"layout.styles.db.connection"
                                               ofType:@"xml"] UTF8String],
                             NULL);
  myx_grt_wb_load_style_file([_canvas canvas],
                             [[bundle pathForResource:@"layout.styles.db.routine"
                                               ofType:@"xml"] UTF8String],
                             NULL);
  myx_grt_wb_load_style_file([_canvas canvas],
                             [[bundle pathForResource:@"layout.styles.db.table"
                                               ofType:@"xml"] UTF8String],
                             NULL);
  myx_grt_wb_load_style_file([_canvas canvas],
                             [[bundle pathForResource:@"layout.styles.db.view"
                                               ofType:@"xml"] UTF8String],
                             NULL);
  myx_grt_wb_load_style_file([_canvas canvas],
                             [[bundle pathForResource:@"layout.styles.db"
                                               ofType:@"xml"] UTF8String],
                             NULL);
  {
    MYX_GRT_VALUE *presets= [_grt performModule:@"Workbench"
                                       function:@"getColorPresets"
                                      arguments:nil];
    if (!presets)
      NSLog(@"Error getting color presets.");
    else
    {
      NSMutableArray *colors= [[NSMutableArray alloc] init];
      MYX_GRT_VALUE *colorList;
      int i, j;
      struct {
        const char *name;
        NSString *path;
        NSMenu *menu;
      } colorPresetInfo[] = 
      {
      {"table", @"layout.styles.db.table.tmpl", [[tableOptions viewWithTag:1] menu]},
      {"view", @"layout.styles.db.view.tmpl", [[viewOptions viewWithTag:1] menu]},
      {"layer", @"layout.styles.layer.tmpl", [[layerOptions viewWithTag:1] menu]},
      {"note", @"layout.styles.note.tmpl", [[noteOptions viewWithTag:1] menu]},
//      {"package", @"layout.styles.package.tmpl", [[routineOptions viewWithTag:1] menu]},
      {NULL, NULL, nil}
      };

      for (j= 0; colorPresetInfo[j].name; j++)
      {
        [colors removeAllObjects];
        colorList= myx_grt_bridge_dict_item_get_value(presets, colorPresetInfo[j].name, 0);
        myx_grt_wb_load_style_file([_canvas canvas],
                                   [[bundle pathForResource:colorPresetInfo[j].path
                                                     ofType:@"xml"] UTF8String],
                                   colorList);
        for (i= 0; i < myx_grt_bridge_list_item_count(colorList); i++)
        {
          MYX_GRT_VALUE *item= myx_grt_bridge_list_item_get(colorList, i, 0);
          [colors addObject:[NSString stringWithUTF8String:myx_grt_bridge_value_as_string(myx_grt_bridge_dict_item_get_value(item, "colorName", 0))]];
          [colors addObject:[NSString stringWithUTF8String:myx_grt_bridge_value_as_string(myx_grt_bridge_dict_item_get_value(item, "color", 0))]];
        }      
        [self setupMenu:colorPresetInfo[j].menu
             withColors:colors];
      }
      [colors release];
      myx_grt_value_release(presets);
    }
  }
  
  if (![_canvas loadLayoutsFromFile:[bundle pathForResource:@"layout.figures.db"
                                                     ofType:@"xml"]])
    NSLog(@"Error reading figures file.");

  canvas= [_canvas canvas];

  [_canvas setDelegate:self];
  [_canvas setBaseSize:MWDefaultCanvasSize];
  [_scrollView setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];

  NSNib *nib= [[NSNib alloc] initWithNibNamed:@"ScrollTools" bundle:[NSBundle mainBundle]];
  [nib instantiateNibWithOwner:self
               topLevelObjects:nil];
  [nib release];
  
  [zoomPop removeAllItems];
  zoomSteps= [_canvas getZoomSteps];
  for (i= 0; i < [zoomSteps count]; i++)
  {
    [zoomPop addItemWithTitle:[zoomSteps objectAtIndex:i]];
  }
  
  [_scrollView setHAccessory:scrollTools];
  
  [self setupGRT];

  _layerDS= [[MWLayerDataSource alloc] initWithGRT:[_grt grt]];
  [layerTree setDataSource:_layerDS];
  [layerTree setDelegate:self];
//  [_layerDS setRootLayerPath:@"/workbench/model/views/0/layers"];
  [layerTree reloadData];
  
  if (!_loadedDocumentData)
  {
    MYX_GRT_VALUE *result;
    if (!(result= [_grt performModule:@"Workbench"
                             function:@"newDocument"
                            arguments:[NSArray arrayWithObjects:[_grt stringAppOption:@"DefaultRdbms"],
                              [_grt stringAppOption:@"DefaultRdbmsVersion"], nil]]))
    {
      
    }
    else
      myx_grt_value_release(result);
  }
  else
    
  {
    MYX_GRT_VALUE *result;
    
    result= [_grt performModule:@"Workbench"
                       function:@"loadDocumentData"
                      arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:_loadedDocumentData]]];
    if (result)
    {
      myx_grt_value_release(result);
    }
    
    [statusText setStringValue:
      [NSString stringWithFormat:NSLocalizedString(@"Loaded Workbench model '%@'",nil),
        [self displayName]]];
    
    myx_grt_value_release(_loadedDocumentData);
    _loadedDocumentData= 0;    
  }

  [catalogDS setMGRT:_grt];
  [catalogDS setRootValue:[_grt globalValue:"/workbench/catalog/schemata"]];
  [catalogDS setDisplayInfo:[NSDictionary dictionaryWithObjectsAndKeys:
    [NSArray arrayWithObjects:@"tables",@"views",@"routineGroups",@"routines",nil], @"db.mysql.Schema",
    nil]];
  [catalogDS setIconInfo:[NSDictionary dictionaryWithObjectsAndKeys:
    [NSImage imageNamed:@"db.Schema.16x16"], @"db.mysql.Schema",
    [NSImage imageNamed:@"db.Table.16x16"], @"db.mysql.Table",
    [NSImage imageNamed:@"db.View.16x16"], @"db.mysql.View",
    [NSImage imageNamed:@"db.Routine.16x16"], @"db.mysql.Routine",
    nil]];
  [schemaTree registerForDraggedTypes:[NSArray arrayWithObject:MGRTTreeNodePboardType]];
  [schemaTree reloadData];
  [schemaTree setDoubleAction:@selector(editSelectedSchemaObject:)];
  [schemaTree setTarget:self];
  
  [pluginList setDoubleAction:@selector(executeSelectedPlugin:)];
  [pluginList setTarget:self];

  [self updateChangeCount:NSChangeCleared];
  
  [self setupMenu:[[tableOptions viewWithTag:2] menu]
        withItems:[_grt globalValue:"/workbench/catalog/schemata"]
         titleKey:"name"];
  
  [self setupMenu:[[viewOptions viewWithTag:2] menu]
        withItems:[_grt globalValue:"/workbench/catalog/schemata"]
         titleKey:"name"];

  [self setupMenu:[[routineOptions viewWithTag:2] menu]
        withItems:[_grt globalValue:"/workbench/catalog/schemata"]
         titleKey:"name"];

  
  [self setupMenu:[[tableOptions viewWithTag:4] menu]
 withCharsetItems:[_grt globalValue:"/rdbmsMgmt/rdbms/Mysql/characterSets"]];

  
  [[tableOptions viewWithTag:4] selectItemWithTitle:@"utf8_general_ci"];

  [self updateToolOptions];
//  [self toggleToolHelp:YES];
  
  [[lowerSideTab tabViewItemAtIndex:1] setView:basePropsView];
  
//broken  [self setSplitted:NO];
  
  [drawer open];
}


- (BOOL)readFromFile:(NSString *)fileName ofType:(NSString *)docType
{
  _loadedDocumentData= myx_grt_retrieve_from_file([_grt grt],
                                                  [fileName UTF8String]);
  if (_loadedDocumentData != 0)
    return YES;
  else
    return NO;
}


- (BOOL)writeToFile:(NSString *)fileName ofType:(NSString *)docType
{
  MYX_GRT_ERROR error;
  
  error= myx_grt_store_to_file([_grt grt],
                               [_grt globalValue:"/workbench"],
                               [fileName UTF8String]);
  if (error != MYX_GRT_NO_ERROR)
  {
    NSRunAlertPanel(NSLocalizedString(@"Error Saving Model",nil),
                    [MGRT errorText:error],
                    @"OK",nil,nil);
    return NO;
  }
  return YES;
}


- (BOOL)writeToURL:(NSURL *)absoluteURL ofType:(NSString *)typeName error:(NSError **)outError
{
  return [self writeToFile:[absoluteURL path] ofType:typeName];
}


- (BOOL)readFromURL:(NSURL *)absoluteURL ofType:(NSString *)typeName error:(NSError **)outError
{
  if ([absoluteURL isFileURL])
  {
    return [self readFromFile:[absoluteURL path] ofType:typeName];
  }
  else
    return NO;
}


- (IBAction)arrangeObjects:(id)sender
{
  [_grt performModule:@"WorkbenchController"
            procedure:@"autoArrangeObjects"
            arguments:[NSArray arrayWithObjects:
              [NSValue valueWithPointer:[_grt globalValue:"/workbench/model/currentView"]],
              [NSNumber numberWithInt:1],
              nil]];
}


- (IBAction)reverseEngineer:(id)sender
{
  MWReverseEngineering *reveng= [[MWReverseEngineering alloc] initWithMGRT:_grt];

  _currentWindow= reveng;
  
  [reveng run];

  _currentWindow= nil;
  
  [schemaTree reloadData];

  [self setupMenu:[[tableOptions viewWithTag:2] menu]
        withItems:[_grt globalValue:"/workbench/catalog/schemata"]
         titleKey:"name"];
  
  [self setupMenu:[[viewOptions viewWithTag:2] menu]
        withItems:[_grt globalValue:"/workbench/catalog/schemata"]
         titleKey:"name"];

  [self setupMenu:[[routineOptions viewWithTag:2] menu]
        withItems:[_grt globalValue:"/workbench/catalog/schemata"]
         titleKey:"name"];
  
  [_grt saveSubtree:"/app/options"
             toFile:[[MPreferences preferences] pathForFile:@"workbench_app_options.xml"]];
}


- (IBAction)synchronizeDB:(id)sender
{
  MWSynchronizeDB *sync= [[MWSynchronizeDB alloc] initWithMGRT:_grt];

  _currentWindow= sync;
  
  [sync run];

  _currentWindow= nil;

  [schemaTree reloadData];

  [_grt saveSubtree:"/app/options"
             toFile:[[MPreferences preferences] pathForFile:@"workbench_app_options.xml"]];
}


- (IBAction)toggleOverview:(id)sender
{
  if (sender != [[modelWindow contentView] viewWithTag:200])
    [[[modelWindow contentView] viewWithTag:200] performClick:nil];
  else
    [_canvas toggleOverview:nil];
}


- (void)generateScript:(id)sender
{
  NSSavePanel *panel= [NSSavePanel savePanel];
  
  [panel setTitle:NSLocalizedString(@"Export as Database Create SQL Script",nil)];
  [panel setRequiredFileType:@"sql"];
  
  if ([panel runModal] == NSFileHandlingPanelOKButton)
  {
    NSString *data= [_grt performModule:@"Workbench"
                         stringFunction:@"generateSqlCreateScript"
                              arguments:nil];
    
    [data writeToFile:[panel filename] atomically:NO];
    
    [statusText setStringValue:[NSString stringWithFormat:NSLocalizedString(@"Wrote SQL script to %@",nil), [panel filename]]];
  }
}


- (void)tabViewChangedPage:(MTabView*)aTabView
{
  MWCanvasViewTab *tab= (MWCanvasViewTab*)[aTabView selectedPage];
  
  [tab makeVisible];
  
  [zoomPop selectItemAtIndex:[_canvas zoomLevel]];
}


static int toggles[]= {
  10, 11,
  20,
  30, 31, 32, 33, 34, 35,
  40, 41, 42
};

/*  
static NSString *toggleKeys[]= {
  @"esc",
  @"H",
  
  @"E",
  
  @"T",
  @"V",
  @"G",
  
  @"R",
  @"r",
  @"^R",
  
  @"I",
  @"N",
  @"L"
};
*/

- (BOOL)editCanvas:(MWEditCanvasView*)canvas keyDown:(NSEvent*)event
{
  int mods= [event modifierFlags] & (NSShiftKeyMask|NSControlKeyMask|NSCommandKeyMask|NSAlternateKeyMask);
  NSString *key= [event charactersIgnoringModifiers];

  if ([key isEqualTo:@"\e"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[0]] performClick:nil];
  }
  else if ([key isEqualTo:@"h"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[1]] performClick:nil];    
  }
  else if ([key isEqualTo:@"e"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[2]] performClick:nil];
  }
  else if ([key isEqualTo:@"t"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[3]] performClick:nil];
  }
  else if ([key isEqualTo:@"v"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[4]] performClick:nil];
  }
  else if ([key isEqualTo:@"g"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[5]] performClick:nil];
  }
  else if ([key isEqualTo:@"r"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[6]] performClick:nil];
  }    
  else if ([key isEqualTo:@"R"] && (mods & NSShiftKeyMask) == NSShiftKeyMask)
  {
    [[paletteView viewWithTag:toggles[7]] performClick:nil];
  }    
  else if ([key isEqualTo:@"r"] && (mods & NSControlKeyMask) == NSControlKeyMask)
  {
    [[paletteView viewWithTag:toggles[8]] performClick:nil];
  }    
  else if ([key isEqualTo:@"i"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[9]] performClick:nil];
  }    
  else if ([key isEqualTo:@"n"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[10]] performClick:nil];
  }    
  else if ([key isEqualTo:@"l"] && mods == 0)
  {
    [[paletteView viewWithTag:toggles[11]] performClick:nil];
  }   
  else if ([key isEqualTo:@" "] && (mods & NSShiftKeyMask) == NSShiftKeyMask)
  {
    [self toggleOverview:nil];
  }
  else
    return NO;
  return YES;
}  

/*
- (void)toggleToolHelp:(BOOL)show
{
  NSTextField *tip;
  NSFont *font= [NSFont systemFontOfSize:8.0];
  int i;
  
  if (show)
  {
    for (i= 0; i < sizeof(toggles)/sizeof(int); i++)
    {
      NSRect rect= [[paletteView viewWithTag:toggles[i]] frame];
      
      //rect.origin.y= 0;
      rect.origin.x= 1;
      rect.size.width= 30;
      rect.size.height= 10;
      
      tip= [[[NSTextField alloc] initWithFrame:rect] autorelease];
      [tip setDrawsBackground:NO];
      [tip setBordered:NO];
      [tip setAutoresizingMask:NSViewMinXMargin|NSViewMaxYMargin];
      [tip setFont:font];
      [tip setEditable:NO];
      [tip setStringValue:toggleKeys[i]];
      [tip setAlignment:NSLeftTextAlignment];
      [tip setTag:500+toggles[i]];
      [tip sizeToFit];
      [tip setAutoresizingMask:NSViewWidthSizable|NSViewMaxYMargin];
      [paletteView addSubview:tip];
    }
  }
  else
  {
    for (i= 0; i < sizeof(toggles)/sizeof(int); i++)
    {
      [[paletteView viewWithTag:500+toggles[i]] removeFromSuperview];
    }
  }
}
*/

- (IBAction)changeZoom:(id)sender
{
  switch ([sender tag])
  {
    case 10:
      [_canvas setZoomLevel:DEFAULT_CANVAS_ZOOM_LEVEL];
      break;
    case 11:
      [_canvas setZoomLevel:[_canvas zoomLevel]+1];
      break;
    case 12:
      [_canvas setZoomLevel:[_canvas zoomLevel]-1];
      break;
    default:
      [_canvas setZoomLevel:[sender indexOfSelectedItem]];
      break;
  }
}


- (IBAction)pickImageFile:(id)sender
{
  NSOpenPanel *panel= [NSOpenPanel openPanel];
  
  [panel setTitle:@"Choose PNG Image File"];
  [panel setRequiredFileType:@"png"];
  if ([panel runModal] == NSFileHandlingPanelOKButton)
    [[imageOptions viewWithTag:1] setStringValue:[panel filename]];
}


- (void)updateToolOptions
{
  [_currentOptionsView removeFromSuperview];
  
  switch (_currentTool)
  {
    case MWToolCursor:
      _currentOptionsView= cursorOptions;
      break;
    case MWToolLayer:
      _currentOptionsView= layerOptions;
      break;
    case MWToolTable:
      _currentOptionsView= tableOptions;
      break;
    case MWToolImage:
      _currentOptionsView= imageOptions;
      break;
    case MWToolNote:
      _currentOptionsView= noteOptions;
      break;
    case MWToolView:
      _currentOptionsView= viewOptions;
      break;      
    case MWToolRoutine:
      _currentOptionsView= routineOptions;
      break;            
    default:
      _currentOptionsView= nil;
      break;
  }

  if (_currentOptionsView)
  {
    [[modelWindow contentView] addSubview:_currentOptionsView];
    [_currentOptionsView setFrameOrigin:NSMakePoint(260, NSHeight([[modelWindow contentView] frame]) - NSHeight([_currentOptionsView frame]))];
  }
}


- (IBAction)toggleGrid:(id)sender
{
  if (![_canvas gridEnabled])
  {
    [_canvas setGridEnabled:YES];
    if (!sender)
      [[[modelWindow contentView] viewWithTag:201] setState:NSOnState];
    [[[[[NSApp mainMenu] itemWithTag:4] submenu] itemWithTag:60] setState:NSOnState];
  }
  else
  {
    [_canvas setGridEnabled:NO];
    if (!sender)
      [[[modelWindow contentView] viewWithTag:201] setState:NSOffState];
    [[[[[NSApp mainMenu] itemWithTag:4] submenu] itemWithTag:60] setState:NSOffState];
  }
}


- (void)centerObjects:(id)sender
{
  float x, y;
  NSRect r;
  
  r= [_canvas actualVisibleRect];
  
  x= NSMinX(r) + NSWidth(r) / 2;
  y= NSMinY(r) + NSHeight(r) / 2;
  
  if (![_grt performModule:@"Workbench"
                 procedure:@"centerElementsAt"
                 arguments:[NSArray arrayWithObjects:
                   [NSNumber numberWithFloat:x],
                   [NSNumber numberWithFloat:y],
                   nil]])
    NSLog(@"error");  
}


- (IBAction)alignObjects:(id)sender
{
  BOOL vert= NO;
  BOOL align= YES;
  NSString *side= nil;
  
  switch ([sender tag])
  {
    case 1:
      vert= YES;
      side= @"T";
      break;
    case 2:
      vert= YES;
      side= @"C";
      break;
    case 3:
      vert= YES;
      side= @"B";
      break;
    case 4:
      vert= NO;
      side= @"L";
      break;
    case 5:
      vert= NO;
      side= @"C";
      break;
    case 6:
      vert= NO;
      side= @"R";
      break;
      
    case 10:
      align= NO;
      vert= YES;
      break;
    case 11:
      align= NO;
      vert= NO;
      break;
  }
  
  if (align)
  {
    if (vert)
    {
      if (![_grt performModule:@"Workbench"
                     procedure:@"verticalAlign"
                     arguments:[NSArray arrayWithObject:side]])
        NSLog(@"error");
    }
    else
    {
      if (![_grt performModule:@"Workbench"
                     procedure:@"horizontalAlign"
                     arguments:[NSArray arrayWithObject:side]])
        NSLog(@"error");
    }
  }
  else
  {
    if (![_grt performModule:@"Workbench"
                   procedure:@"autoSpaceElements"
                   arguments:[NSArray arrayWithObjects:
                     [NSNumber numberWithInt:vert], 
                     [NSNumber numberWithInt:-1], nil]])
      NSLog(@"error");
  }
}


- (IBAction)toggleTool:(id)sender
{
  int i;
  int tag= [sender tag];
  MWToolType oldTool;
  
  [[paletteView viewWithTag:toggles[_currentTool]] setState:NSOffState];
  
  oldTool= _currentTool;
  
  [sender setState:NSOnState];
  for (i= 0; i < sizeof(toggles)/sizeof(int); i++)
  {
    if (toggles[i] == tag)
    {
      [_canvas setCursor:[_toolCursors objectAtIndex:i]];
      _currentTool= (MWToolType)i;
      break;
    }
  }
  
  
  if (_relationshipContext)
  {
    [_grt performModule:@"Workbench"
              procedure:@"relationshipCancel"
              arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:_relationshipContext]]];
    
    myx_grt_value_release(_relationshipContext);
    _relationshipContext= NULL;
  }
  
  switch (_currentTool)
  {
    case MWToolCursor:
      [_canvas setGrabPanning:NO];
      [statusText setStringValue:NSLocalizedString(@"Cursor Tool",nil)];
      break;
    case MWToolPanning:
      [_canvas setGrabPanning:YES];
      [statusText setStringValue:NSLocalizedString(@"Panning Tool",nil)];
      break;
    case MWToolTable:
      [_canvas setGrabPanning:NO];
      [statusText setStringValue:NSLocalizedString(@"Table Tool: select position to place a new table.",nil)];
      break;
      
    case MWToolView:
      [_canvas setGrabPanning:NO];
      [statusText setStringValue:NSLocalizedString(@"View Tool: select position to place a new view.",nil)];
      break;

    case MWToolRoutine:
      [_canvas setGrabPanning:NO];
      [statusText setStringValue:NSLocalizedString(@"Routine Tool: select position to place new routine package.",nil)];
      break;      
      
    case MWToolRelationship1:
      
    case MWToolRelationship2:
    case MWToolRelationship3:
      _relationshipContext= [_grt performModule:@"Workbench"
                                       function:@"relationshipStart"
                                      arguments:[NSArray arrayWithObjects:
                           (_currentTool == MWToolRelationship1) ? @"11" : ((_currentTool == MWToolRelationship2) ? @"1n" : @"nm"),
                                        NSInt([[relationshipOptions viewWithTag:1] state] == NSOnState ? 1 : 0),
                                        NSInt([[relationshipOptions viewWithTag:2] state] == NSOnState ? 1: 0),
                                        nil
                                        ]];
      
      [statusText setStringValue:NSLocalizedString(@"Relationship Tool: select the source table.",nil)];
      break;

    case MWToolLayer:
      [statusText setStringValue:NSLocalizedString(@"Layer Tool: select an area for the new layer.",nil)];
      break;
  }
  [self updateToolOptions];
}


- (void)outlineView:(NSOutlineView *)outlineView willDisplayCell:(id)cell forTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
  if (outlineView == layerTree)
  {
    if ([item isKindOfClass:[MWLayerTreeNode class]])
    {
      [cell setImage:_layerIcon];
    }
    else if (item)
    {
      MGRTValue value([(MWElementTreeNode*)item value]);
      
      if (value.isKindOf([_grt grt], "db.workbench.TableElement"))
        [cell setImage:_tableIcon];
      else if (value.isKindOf([_grt grt], "db.workbench.ViewElement"))
        [cell setImage:_viewIcon];
      else if (value.isKindOf([_grt grt], "db.workbench.RoutinesElement"))
        [cell setImage:_routinesIcon];  
      else
        [cell setImage:nil];
    }
  }
}



- (void)outlineViewSelectionDidChange:(NSNotification *)notification
{
  NSOutlineView *outline= [notification object];
  NSIndexSet *rows= [outline selectedRowIndexes];
  unsigned int i;

  [_grt performModule:@"WorkbenchController"
            procedure:@"unselectObject"
            arguments:nil];

  for (i= [rows firstIndex]; i != NSNotFound; i= [rows indexGreaterThanIndex:i])
  {
    id item= [outline itemAtRow:i];
    [_grt performModule:@"WorkbenchController"
              procedure:@"selectObject"
              arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:[item value]]]];
  }
}



- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  return [_pluginArray count];
}


- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
  id object= [_pluginArray objectAtIndex:rowIndex];
  if ([object isKindOfClass:[NSString class]])
    return object;
  else
    return [NSString stringWithUTF8String:MGRTValue([object grtValue])["caption"].asString()];
}


- (void)canvas:(MGenericCanvasView*)canvas handleChange:(id)change
{
  TGCChangeReason reason= (TGCChangeReason)(long)change;
  switch (reason)
  {
    case GC_CHANGE_VIEW_ZOOM:
      [zoomPop selectItemAtIndex:[canvas zoomLevel]];
      break;
    default:
      break;
  }
}


- (BOOL)canvas:(MGenericCanvasView*)canvas handleMouseDrag:(NSPoint)pos modifiers:(int)mods
{
  return NO;
}


- (BOOL)canvas:(MGenericCanvasView*)canvas handleMouseDown:(int)button atPoint:(NSPoint)pos modifiers:(int)mods
{
  if (button == 0 && _currentTool == MWToolLayer)
  {
    [_canvas startAreaSelectionAt:pos];
  }
  else if (button == 0 && _currentTool == MWToolNote)
  {
    [_canvas startAreaSelectionAt:pos];
  }
  else if (button == 0 && _currentTool == MWToolCursor)
  {
    MYX_GRT_VALUE *result;
    result= [_grt performModule:@"WorkbenchController"
                       function:@"objectAtPoint"
                      arguments:[NSArray arrayWithObjects:[NSNumber numberWithFloat:floor(pos.x)], 
                        [NSNumber numberWithFloat:floor(pos.y)],nil]];
    if (result)
    {
      [self displayObjectProperties:MGRTValue(result)["object"]
                            element:MGRTValue(result)["element"]];

      myx_grt_value_release(result);
    }
    else
      [self displayObjectProperties:MGRTValue()
                            element:MGRTValue()];
  }

  if (_currentTool != MWToolCursor)
    return YES;
  else
    return NO;
}


- (BOOL)canvas:(MGenericCanvasView*)canvas handleMouseUp:(int)button atPoint:(NSPoint)pos modifiers:(int)mods
{
  MYX_GRT_VALUE *result;
  
  if (button == 0 && _currentTool == MWToolTable)
  {
    result= [_grt performModule:@"Workbench"
                       function:@"placeNewTableElement"
                      arguments:[NSArray arrayWithObjects:[NSNumber numberWithFloat:floor(pos.x)], 
                        [NSNumber numberWithFloat:floor(pos.y)],
                        [NSString stringWithFormat:@"table_%i",++_tableIndex],
                        [[[tableOptions viewWithTag:2] selectedItem] representedObject],
                        [[tableOptions viewWithTag:3] titleOfSelectedItem],
                        [[tableOptions viewWithTag:4] titleOfSelectedItem],
                        [[[tableOptions viewWithTag:1] selectedItem] representedObject],
                        nil]];
    if (result)
      myx_grt_value_release(result);
    
    [[paletteView viewWithTag:toggles[0]] performClick:nil];
  }
  else if (button == 0 && _currentTool == MWToolView)
  {
    result= [_grt performModule:@"Workbench"
                       function:@"placeNewViewElement"
                      arguments:[NSArray arrayWithObjects:[NSNumber numberWithFloat:floor(pos.x)], 
                        [NSNumber numberWithFloat:floor(pos.y)],
                        [NSString stringWithFormat:@"view_%i",++_viewIndex],
                        [[[viewOptions viewWithTag:2] selectedItem] representedObject],
                        [[[viewOptions viewWithTag:1] selectedItem] representedObject],
                        nil]];
    if (result)
      myx_grt_value_release(result);
    
    [[paletteView viewWithTag:toggles[0]] performClick:nil];
  }  
  else if (button == 0 && _currentTool == MWToolRoutine)
  {
    result= [_grt performModule:@"Workbench"
                       function:@"placeNewRoutineGroupElement"
                      arguments:[NSArray arrayWithObjects:[NSNumber numberWithFloat:floor(pos.x)], 
                        [NSNumber numberWithFloat:floor(pos.y)],
                        [NSString stringWithFormat:@"routineGroup_%i",++_routineIndex],
                        [[[routineOptions viewWithTag:2] selectedItem] representedObject],
//                        [[[routineOptions viewWithTag:1] selectedItem] representedObject],
                        nil]];
    if (result)
      myx_grt_value_release(result);
    
    [[paletteView viewWithTag:toggles[0]] performClick:nil];
  }    
  else if (button == 0 && _currentTool == MWToolLayer)
  {
    NSRect rect= [_canvas finishAreaSelection];
    result= [_grt performModule:@"Workbench"
                       function:@"placeNewLayerElement"
                      arguments:[NSArray arrayWithObjects:
                        [NSNumber numberWithFloat:floor(rect.origin.x)],
                        [NSNumber numberWithFloat:floor(rect.origin.y)],
                        [NSNumber numberWithFloat:floor(rect.size.width)], 
                        [NSNumber numberWithFloat:floor(rect.size.height)],
                        [NSString stringWithFormat:@"Layer %i", ++_layerIndex],
                        [[[layerOptions viewWithTag:1] selectedItem] representedObject],
                        nil]];
    if (result)
      myx_grt_value_release(result);
    
    [[paletteView viewWithTag:toggles[0]] performClick:nil];
  }
  else if (button == 0 && _currentTool == MWToolNote)
  {
    NSRect rect= [_canvas finishAreaSelection];
    result= [_grt performModule:@"Workbench"
                       function:@"placeNewNoteElement"
                      arguments:[NSArray arrayWithObjects:
                        [NSNumber numberWithFloat:floor(rect.origin.x)], 
                        [NSNumber numberWithFloat:floor(rect.origin.y)],
                        [NSNumber numberWithFloat:floor(rect.size.width)], 
                        [NSNumber numberWithFloat:floor(rect.size.height)],
                        [[[noteOptions viewWithTag:1] selectedItem] representedObject],
                        nil]];
    if (result)
      myx_grt_value_release(result);
    
    [[paletteView viewWithTag:toggles[0]] performClick:nil];
  }
  else if (button == 0 && _currentTool == MWToolEraser)
  {
    MYX_GRT_VALUE *result;
    result= [_grt performModule:@"Workbench"
                       function:@"deleteObjectAtPoint"
                      arguments:[NSArray arrayWithObjects:[NSNumber numberWithFloat:floor(pos.x)], 
                        [NSNumber numberWithFloat:floor(pos.y)],
                        nil]];
    if (result)
    {
      myx_grt_value_release(result);
      
      [[paletteView viewWithTag:toggles[0]] performClick:nil];
    }
  }    
  else if (button == 0 && (_currentTool == MWToolRelationship1
                           || _currentTool == MWToolRelationship2
                           || _currentTool == MWToolRelationship3))
  {
    NSString *message;
    
    message= [_grt performModule:@"Workbench"
                  stringFunction:@"relationshipClickedPoint"
                       arguments:[NSArray arrayWithObjects:
                         [NSValue valueWithPointer:_relationshipContext],
                         [NSNumber numberWithFloat:floor(pos.x)],
                         [NSNumber numberWithFloat:floor(pos.y)],
                         nil]];
    
    if (message)
      [statusText setStringValue: message];
    
    if (myx_grt_dict_item_get_as_int(_relationshipContext, "step") == -1)
    {
      [[paletteView viewWithTag:toggles[0]] performClick:nil];
    }
  }
  
  // handle any eventual layer/element relocations
  if (_currentTool == MWToolCursor)
    myx_grt_wb_bridge_process_relocations([_grt grt]);
    
  return NO;
}


- (void)objectEditorSaved:(MGRTObjectEditor*)editor
{        
  if ([editor isKindOfClass:[MGRTTableEditor class]])
  {
    [_grt performModule:@"Workbench"
               function:@"createRelationshipsFromFKs"
              arguments:[NSArray arrayWithObject:[NSValue valueWithPointer:[_grt globalValue:"/workbench/model/views/0"]]]];
  }
}


- (void)objectEditorClosed:(MGRTObjectEditor*)editor
{
  [_openEditors removeObjectForKey:[NSString stringWithUTF8String:myx_grt_dict_id_item_as_string([editor editedObject])]];
}


- (void)pluginFinished:(NSDictionary*)data
{
  if ([data objectForKey:@"error"])
  {
    [_canvas setStatus:
      [NSString stringWithFormat:NSLocalizedString(@"Error executing plugin '%s'.", nil),
        MGRTValue([[data objectForKey:@"userData"] grtValue])["name"].asString()]];
    [self pluginTextOut:[NSString stringWithFormat:@"Error executing plugin '%s'.",
      MGRTValue([[data objectForKey:@"userData"] grtValue])["name"].asString()]];
    [self pluginTextOut:[data objectForKey:@"message"]];
  }
  else
    [_canvas setStatus:
      [NSString stringWithFormat:NSLocalizedString(@"Plugin '%s' executed.", nil),
      MGRTValue([[data objectForKey:@"userData"] grtValue])["name"].asString()]];
  
  [_grt resetOutputHandler];
}


- (void)invokePlugin:(MGRTValue)plugin
{
  NSMutableArray *args= [NSMutableArray array];
  MYX_GRT_VALUE **selection;
  int count, i;
  
  [pluginOutput setString:@""];
  
  selection= myx_grt_wb_get_selected_objects([_grt grt], &count);
  if (selection)
  {
    for (i= 0; i < count; i++)
    {
      MGRTValue okTypes= plugin["objectStructNames"];
      unsigned int j, tc= okTypes.count();
      for (j= 0; j < tc; j++)
      {
        if (MGRTValue(selection[i]).isKindOf([_grt grt], okTypes[j].asString()))
        {
          [args addObject:[NSGRTValue grtValueWithValue:selection[i]]];
          break;
        }
      }
    }
    g_free(selection);
  }
  else if (_objectForPlugin)
    [args addObject:_objectForPlugin];

  [_grt setOutputHandler:self selector:@selector(pluginTextOut:)];
  [_grt performAsyncModule:[NSString stringWithUTF8String:plugin["moduleName"].asString()]
                  function:[NSString stringWithUTF8String:plugin["moduleFunctionName"].asString()]
                 arguments:[NSArray arrayWithObject:args]
          finishedSelector:@selector(pluginFinished:)
                    target:self
                  userData:[NSGRTValue grtValueWithValue:plugin.grtValue()]];
  [_canvas setStatus:[NSString stringWithFormat:NSLocalizedString(@"Executing plugin '%s'...",nil), plugin["name"].asString()]];
}



- (void)invokeMenuPlugin:(id)sender
{
  [self invokePlugin: MGRTValue([[sender representedObject] grtValue])];
}


- (void)menu:(NSMenu*)menu needsUpdate:(MWorkbench*)sender
{
  int tag= [[[menu supermenu] itemAtIndex:[[menu supermenu] indexOfItemWithSubmenu:menu]] tag];
  
  if (_requestingInput)
    return;
  
  switch (tag)
  {
    case 1: // File
      [[menu itemWithTag:15] setEnabled:NO];
      [[menu itemWithTag:20] setEnabled:NO];
      [[[[menu itemWithTag:30] submenu] itemAtIndex:0] setEnabled:NO];
      [[menu itemWithTag:40] setEnabled:NO];
      [[menu itemWithTag:41] setEnabled:NO];
      break;
    case 4: // Model
      for (int i= [[[menu itemWithTag:2] submenu] numberOfItems]-1; i>=0; --i)
        [[[[menu itemWithTag:2] submenu] itemAtIndex:i] setEnabled:NO];
      [[menu itemWithTag:51] setEnabled:[_canvas hasSelection]];
      [[menu itemWithTag:61] setEnabled:NO];
      break;
    case 6: // Plugins
      [sender refreshPluginsMenu:_grt action:@selector(invokeMenuPlugin:) target:self];
      break;
  }
}


- (void)displayObjectProperties:(MGRTValue)object
                        element:(MGRTValue)element
{
  if (!element.isValid())
  {
    if ([[lowerSideTab tabViewItemAtIndex:1] view] != basePropsView)
    {
      NSView *view= [[lowerSideTab tabViewItemAtIndex:1] view];
      [view retain];
      [view removeFromSuperview];
      [[lowerSideTab tabViewItemAtIndex:1] setView:basePropsView];
    }
    [basePropsView setEnabledRecursive:NO];
    [[basePropsView viewWithTag:1001] setStringValue:@""];
    [[basePropsView viewWithTag:1002] setStringValue:@""];
    [[basePropsView viewWithTag:1003] setStringValue:@""];
    [[basePropsView viewWithTag:1004] setStringValue:@""];
    
    [basePropsText replaceCharactersInRange:NSMakeRange(0, [[basePropsText textStorage] length])
                                 withString:@""];
  }
  else if (element.isKindOf([_grt grt], "db.workbench.Relationship"))
  {
    int idx;
    if ([[lowerSideTab tabViewItemAtIndex:1] view] != relPropsView)
    {
      NSView *view= [[lowerSideTab tabViewItemAtIndex:1] view];
      [view retain];
      [view removeFromSuperview];
      [[lowerSideTab tabViewItemAtIndex:1] setView:relPropsView];
    }
    
    if (!element["startMany"].asInt() && element["endMany"].asInt())
      idx= 0;
    else if (element["startMany"].asInt() && !element["endMany"].asInt())
      idx= 1;
    else if (!element["startMany"].asInt() && !element["endMany"].asInt())
      idx= 2;
    else
      idx= 3;
    
    [[relPropsView viewWithTag:1101] setStringValue:[NSString stringWithUTF8String:element["caption"].asString()]];
    [[relPropsView viewWithTag:1102] selectItemAtIndex:idx];
    [[relPropsView viewWithTag:1103] setState:element["startMandatory"].asInt()?NSOffState:NSOnState];
    [[relPropsView viewWithTag:1104] setState:element["endMandatory"].asInt()?NSOffState:NSOnState];
    
    if (element["comment"].isValid())
      [[[relPropsText textStorage] mutableString] setString:[NSString stringWithUTF8String:element["comment"].asString()]];
    else
      [relPropsText setString:@""];    
  }
  else
  {
    if (element.isKindOf([_grt grt], "db.workbench.NoteElement"))
    {
      [[basePropsView viewWithTag:1005] setStringValue:NSLocalizedString(@"Note Text", nil)];
      [basePropsText setString:[NSString stringWithUTF8String:element["text"].asString()]];
    }
    else
    {
      [[basePropsView viewWithTag:1005] setStringValue:NSLocalizedString(@"Comments", nil)];

      if (object.isValid() && object["comment"].isValid())
        [[[basePropsText textStorage] mutableString] setString:[NSString stringWithUTF8String:object["comment"].asString()]];
      else
        [basePropsText setString:@""];
    }
    [basePropsView setEnabledRecursive:YES];
    [[basePropsView viewWithTag:1001] setFloatValue:floor(element["left"].asDouble())];
    [[basePropsView viewWithTag:1002] setFloatValue:floor(element["top"].asDouble())];
    [[basePropsView viewWithTag:1003] setFloatValue:floor(element["width"].asDouble())];
    [[basePropsView viewWithTag:1004] setFloatValue:floor(element["height"].asDouble())];

    if ([[lowerSideTab tabViewItemAtIndex:1] view] != basePropsView)
    {
      NSView *view= [[lowerSideTab tabViewItemAtIndex:1] view];
      [view retain];
      [view removeFromSuperview];
      [[lowerSideTab tabViewItemAtIndex:1] setView:basePropsView];
    }    
  }
  
  *_currentElement= element;
  *_currentObject= object;
}


- (void)textDidChange:(NSNotification*)notif
{
  if ([notif object] == basePropsText)
  {
    if (_currentElement->isKindOf([_grt grt], "db.workbench.NoteElement"))
      _currentElement->set("text", [[basePropsText string] UTF8String]);
    else if (_currentObject->isValid())
      _currentObject->set("comment", [[basePropsText string] UTF8String]);
  }
  else if ([notif object] == relPropsText)
  {
    if (_currentElement->isValid())
      _currentElement->set("comment", [[relPropsText string] UTF8String]);
  }
}


- (IBAction)changedProperty:(id)sender
{
  switch ([sender tag])
  {
    case 1102:
      if (_currentElement->isValid())
      {
        switch ([sender indexOfSelectedItem])
        {
          case 0:
            _currentElement->set("startMany", 0);
            _currentElement->set("endMany", 1);
            break;
          case 1:
            _currentElement->set("startMany", 1);
            _currentElement->set("endMany", 0);          
            break;
          case 2:
            _currentElement->set("startMany", 0);
            _currentElement->set("endMany", 0);
            break;
          case 3:
            _currentElement->set("startMany", 1);
            _currentElement->set("endMany", 1);
            break;
        }
      }
      break;
    case 1103:
      _currentElement->set("startMandatory", [sender state] == NSOffState);
      break;
    case 1104:
      _currentElement->set("endMandatory", [sender state] == NSOffState);
      break;
  }
}


- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{
  id obj= [aNotification object];
  if (obj == [relPropsView viewWithTag:1101])
  {
    if (_currentElement->isValid())
      _currentElement->set("caption", [[obj stringValue] UTF8String]);
  } 
  else if (obj == [basePropsView viewWithTag:1001])
  {
    _currentElement->set("left", [obj floatValue]);
  }
  else if (obj == [basePropsView viewWithTag:1002])
  {
    _currentElement->set("top", [obj floatValue]);
  }
  else if (obj == [basePropsView viewWithTag:1003])
  {
    _currentElement->set("width", [obj floatValue]);
  }
  else if (obj == [basePropsView viewWithTag:1004])
  {
    _currentElement->set("height", [obj floatValue]);
  }
}


- (void)askInput:(NSMutableDictionary*)options
{
  MStringRequestSheet *sheet= [MStringRequestSheet sheetWithTitle:[options objectForKey:@"caption"]
                                                            label:NSLocalizedString(@"Input:",nil)];
  id result;
  _requestingInput= YES;
  result= [sheet runModal:_currentWindow ? [_currentWindow window] : modelWindow];
  _requestingInput= NO;  
  [options setObject:[result objectAtIndex:0] forKey:@"result"];
}


- (void)importFinished:(NSDictionary*)data
{
  if ([data objectForKey:@"errorCode"])
    [_canvas setStatus:NSLocalizedString(@"Error importing DBDesigner4 document.", nil)];
  else
    [_canvas setStatus:NSLocalizedString(@"Imported DBDesigner4 document.", nil)];
}


- (void)importFile:(id)sender
{
  NSOpenPanel *panel= [NSOpenPanel openPanel];
  
  if ([sender tag] == 10)
  {
    // SQL Script
  }
  else if ([sender tag] == 11)
  {
    // DBD4
    [panel setTitle:NSLocalizedString(@"Import DBDesigner4 Model", nil)];
    if ([panel runModalForTypes:[NSArray arrayWithObject:@"xml"]] == NSOKButton)
    {
      [_canvas setStatus:NSLocalizedString(@"Importing DBDesigner4 document...", nil)];

      [_grt performAsyncModule:@"WorkbenchImport"
                      function:@"importDbd4"
                     arguments:[NSArray arrayWithObject:[panel filename]]
              finishedSelector:@selector(importFinished:)
                        target:self
                      userData:nil];
    }
    else
      [_canvas setStatus:NSLocalizedString(@"Import cancelled.", nil)];
  }
}


- (void)showDocumentProperties:(id)sender
{
  [[MWDocumentProperties alloc] initWithValue:MGRTValue([_grt globalValue:"/workbench/model/properties"])];
}


- (void)expandElements:(id)sender
{
  [_grt performAsyncModule:@"Workbench"
                  function:@"expandElements"
                 arguments:[NSArray arrayWithObject:[NSNumber numberWithInt:[sender tag] >= 22]]
          finishedSelector:nil
                    target:nil
                  userData:nil];
}


- (void)collapseElements:(id)sender
{
  [_grt performAsyncModule:@"Workbench"
                  function:@"collapseElements"
                 arguments:[NSArray arrayWithObject:[NSNumber numberWithInt:[sender tag] >= 22]]
          finishedSelector:nil
                    target:nil
                  userData:nil];  
}


- (void)changeNotation:(id)sender
{
  NSMenu *menu= [sender menu];
  int i;
  for (i= 0; i < [menu numberOfItems]; i++)
    [[menu itemAtIndex:i] setState:NSOffState];
  
  switch ([sender tag])
  {
    case 1:
      [_grt setGlobalValue:MGRTValue("eer").grtValue()
                   forPath:"/workbench/model/currentView/connectionLayoutClass"];
      [sender setState:NSOnState];
      break;
    case 2:
      [_grt setGlobalValue:MGRTValue("default").grtValue()
                   forPath:"/workbench/model/currentView/connectionLayoutClass"];
      [sender setState:NSOnState];
      break;
  }
}


- (void)changeTableVisualization:(id)sender
{
}


- (void)changeGridOption:(id)sender
{
  if ([sender tag] == 60)
  {
    [self toggleGrid:nil];
  }
  else if ([sender tag] == 61)
  {
    // snap to grid
  }
}


- (NSArray*)selectedObjectTypes
{
  NSMutableArray *types= [NSMutableArray array];
  int selcount;
  MYX_GRT_VALUE **selection= myx_grt_wb_get_selected_objects([_grt grt], &selcount);
  
  if (selection)
  {
    for (int i= 0; i < selcount; i++)
    {
      NSString *s= [NSString stringWithUTF8String:MGRTValue(selection[i]).contentStruct()];
      if ([types indexOfObjectIdenticalTo:s] == NSNotFound)
        [types addObject:s];
    }
    
    g_free(selection);
  }
  return types;
}


- (BOOL)plugin:(MGRTValue)plugin matchesSelection:(NSArray*)selection
{
  MGRTValue structNames(plugin["objectStructNames"]);
  
  if (structNames.count() == 0)
    return YES;
  
  if ([selection count] == 0)
    return NO;
  
  for (int j= [selection count]-1; j>=0; --j)
  {
    NSString *s= [selection objectAtIndex:j];
    BOOL ok= NO;
    for (int k= structNames.count()-1; k>=0; --k)
    {
      if (myx_grt_struct_is_or_inherits_from([_grt grt], [s UTF8String],
                                             structNames[k].asString()))
      {
        ok= YES;
        break;
      }
    }
    if (!ok)
      return NO;
  }
  return YES;
}


- (void)listPlugins:(MGRTValue)list
           matching:(NSString*)str 
       forSelection:(NSArray*)types
{
  unsigned int c= list.count();

  for (unsigned int i= 0; i < c; i++)
  {
    MGRTValue group(list[i]);
    
    if (group["plugins"].count() > 0)
    {
      MGRTValue plugins(group["plugins"]);
      for (unsigned int j= 0; j < plugins.count(); j++)
      {
        MGRTValue plugin(MGRTValue::refObject([_grt grt], plugins[j].asString()));
        NSString *caption= [NSString stringWithUTF8String:plugin["caption"].asString()];
        if ([caption rangeOfString:str options:NSCaseInsensitiveSearch].location != NSNotFound)
        {
          if ([self plugin:plugin matchesSelection:types])
            [_pluginArray addObject:[NSGRTValue grtValueWithValue:plugin.grtValue()]];
          else
            [_pluginArray addObject:[NSString stringWithUTF8String:plugin["caption"].asString()]];
        }
      }
    }
    [self listPlugins:group["subGroups"] matching:str forSelection:types];
  }
}


- (void)executeSelectedPlugin:(id)sender
{
  id object= [_pluginArray objectAtIndex: [pluginList selectedRow]];
  
  if ([object isKindOfClass:[NSString class]])
  {
    NSBeep();
    [_canvas setStatus:NSLocalizedString(@"Invalid selection for plugin.", nil)];
  }
  else
    [self invokePlugin:MGRTValue([object grtValue])];
}


- (IBAction)searchPlugins:(id)sender
{
  NSString *str= [pluginSearch stringValue];

  if (!_pluginArray)
    _pluginArray= [[NSMutableArray alloc] init];
  else
  {
    [_pluginArray removeAllObjects];
    [pluginList reloadData];
  }
  
  if ([str length] > 0)
  {
    [self listPlugins:MGRTValue([_grt globalValue:"/app/pluginGroups"])
             matching:str
         forSelection:[self selectedObjectTypes]];
    [pluginList reloadData];
  }
}


- (void)populatePluginsPopup:(MGRTValue)list
               forObjectType:(NSString*)type
{
  unsigned int c= list.count();
  NSArray *selection= [NSArray arrayWithObject:type];
  for (unsigned int i= 0; i < c; i++)
  {
    MGRTValue group(list[i]);
    
    if (group["plugins"].count() > 0)
    {
      MGRTValue plugins(group["plugins"]);
      for (unsigned int j= 0; j < plugins.count(); j++)
      {
        MGRTValue plugin(MGRTValue::refObject([_grt grt], plugins[j].asString()));
        NSString *caption= [NSString stringWithUTF8String:plugin["caption"].asString()];
        if ([self plugin:plugin matchesSelection:selection])
        {
          [[_pluginsPopup addItemWithTitle:caption
                                    action:@selector(invokeMenuPlugin:)
                             keyEquivalent:@""] setRepresentedObject:[NSGRTValue grtValueWithValue:plugin.grtValue()]];
        }
      }
    }
    [self populatePluginsPopup:group["subGroups"] forObjectType:type];
  }
}


- (void)popupPluginsMenu:(NSArray*)args
{
  MGRTValue object((MYX_GRT_VALUE*)[[args objectAtIndex:1] pointerValue]);
  
  while ([_pluginsPopup numberOfItems] > 0)
    [_pluginsPopup removeItemAtIndex:0];
  [self populatePluginsPopup:MGRTValue([_grt globalValue:"/app/pluginGroups"])
               forObjectType:[NSString stringWithUTF8String:object.contentStruct()]];
  [NSMenu popUpContextMenu:_pluginsPopup
                 withEvent:[_canvas lastEvent]
                   forView:_canvas];

  [_objectForPlugin release];
  _objectForPlugin= [[NSGRTValue grtValueWithValue:object.grtValue()] retain];
}


- (void)pluginTextOut:(NSString*)text
{
  [[pluginOutput textStorage] appendAttributedString:[[[NSAttributedString alloc] initWithString:text] autorelease]];
}


@end
