(*pp camlp4r *)

(*
  Copyright (C) 2003  Bardur Arantsson

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(** Device type. *)
type t = 'a;

(** Which kind of driver? *)
type driver_kind_t = [= `LIVE | `FILE | `UNKNOWN ];

(** Byte format specifier. *)
type byte_format_t = [= `LITTLE_ENDIAN | `BIG_ENDIAN | `NATIVE | `UNKNOWN ];

(** Driver type (private). *)
type driver_t = 'b;

(** Get default driver *)
value get_default_driver : unit -> driver_t;

(** [drivers] is a list of available drivers. *)
value drivers : list driver_t;

(** [driver_kind driver] gets the driver type of the given driver. *)
value driver_kind : driver_t -> driver_kind_t;

(** [driver_name driver] gets the name of the given driver. *)
value driver_name : driver_t -> string;

(** [driver_short_name driver] gets the short name of the given
  driver. *)
value driver_short_name : driver_t -> string;

(** [driver_comment driver] gets the comment associated with the given
  driver. *)
value driver_comment : driver_t -> string;

(** [driver_author driver] gets the author associated with the given
  driver. *)
value driver_author : driver_t -> string;

(** [driver_priority driver] gets the priority associated with the
  given driver. *)
value driver_priority : driver_t -> int;

(** [driver_preferred_byte_format driver] retrieves the preferred byte
  format of the given driver. *)
value driver_preferred_byte_format : driver_t -> byte_format_t;

(** [driver_options driver] retrieves the list of supported options
  for the given driver. *)
value driver_options : driver_t -> list string;

(** [open_live TODO]. *)
value open_live : ?bits:int -> 
   ?rate:int -> 
   ?channels:int ->
   ?byte_format:byte_format_t ->
   ?options:(list (string*string)) -> 
   ?driver:driver_t -> unit -> t;

(** [open_file TODO]. *)
value open_file : ?bits:int -> 
   ?rate:int -> 
   ?channels:int ->
   ?byte_format:byte_format_t ->
   ?options:(list (string*string)) -> 
   ?driver:driver_t ->
   ?overwrite:bool ->
   string -> t;

(** [find_driver name] returns the driver associated with the given
  {b short} name. *)
value find_driver : string -> driver_t;

(** [play device buf] plays the sequence of samples in [buf]. *)
value play : t -> string -> unit;

(** [close device] closes the given device. *)
value close : t -> unit;
