/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: BasicPaneFactory.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: ihi $ $Date: 2007/08/17 14:26:11 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "precompiled_sd.hxx"

#include "BasicPaneFactory.hxx"

#include "FrameWindowPane.hxx"
#include "ChildWindowPane.hxx"

#include "framework/FrameworkHelper.hxx"
#include "ViewShellBase.hxx"
#ifndef _COM_SUN_STAR_DRAWING_FRAMEWORK_XCONTROLLERMANAGER_HPP_
#include <com/sun/star/drawing/framework/XControllerManager.hpp>
#endif
#include "PaneChildWindows.hxx"
#include "DrawController.hxx"
#include <boost/bind.hpp>


using namespace ::com::sun::star;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star::drawing::framework;

using ::rtl::OUString;
using ::sd::framework::FrameworkHelper;

namespace {
    enum PaneId {
        CenterPane,
        LeftImpressPane,
        LeftDrawPane,
        RightPane
    };

    static const sal_Int32 gnConfigurationUpdateStartEvent(0);
    static const sal_Int32 gnConfigurationUpdateEndEvent(1);
}

namespace sd { namespace framework {


/** Store URL, XPane reference and (local) PaneId for every pane factory
    that is registered at the PaneController.
*/
class BasicPaneFactory::PaneDescriptor
{
public:
    OUString msPaneURL;
    Reference<XPane> mxPane;
    PaneId mePaneId;
    /** The mbReleased flag is set when the pane has been released.  Some
        panes are just hidden and destroyed.  When the pane is reused this
        flag is reset.
    */
    bool mbIsReleased;
    bool CompareURL (const OUString& rsPaneURL) { return msPaneURL.equals(rsPaneURL); }
    bool ComparePane (const Reference<XPane>& rxPane) { return mxPane==rxPane; }
};


class BasicPaneFactory::PaneContainer
    : public ::std::vector<PaneDescriptor>
{
public:
    PaneContainer (void) {}
};



Reference<XInterface> SAL_CALL BasicPaneFactory_createInstance (
    const Reference<XComponentContext>& rxContext)
{
    return Reference<XInterface>(static_cast<XWeak*>(new BasicPaneFactory(rxContext)));
}




::rtl::OUString BasicPaneFactory_getImplementationName (void) throw(RuntimeException)
{
    return ::rtl::OUString(
        RTL_CONSTASCII_USTRINGPARAM("com.sun.star.comp.Draw.framework.BasicPaneFactory"));
}




Sequence<rtl::OUString> SAL_CALL BasicPaneFactory_getSupportedServiceNames (void)
    throw (RuntimeException)
{
	static const ::rtl::OUString sServiceName(
        ::rtl::OUString::createFromAscii("com.sun.star.drawing.framework.BasicPaneFactory"));
	return Sequence<rtl::OUString>(&sServiceName, 1);
}




//===== PaneFactory ===========================================================

BasicPaneFactory::BasicPaneFactory (
    const Reference<XComponentContext>& rxContext)
    : BasicPaneFactoryInterfaceBase(MutexOwner::maMutex),
      mxConfigurationController(),
      mxPaneController(),
      mpViewShellBase(NULL),
      mpPaneContainer(new PaneContainer),
      mbFirstUpdateSeen(false),
      mpUpdateLockManager()
{
    (void)rxContext;
}





BasicPaneFactory::~BasicPaneFactory (void)
{
}
    



void SAL_CALL BasicPaneFactory::disposing (void)
{
    Reference<XPaneController> xPaneController (mxPaneController);
    if (xPaneController.is())
    {
        Reference<XComponent> xComponent (mxPaneController, UNO_QUERY);
        if (xComponent.is())
            xComponent->removeEventListener(this);
        mxPaneController = NULL;

        for (PaneContainer::const_iterator iDescriptor = mpPaneContainer->begin();
             iDescriptor != mpPaneContainer->end();
             ++iDescriptor)
        {
            xPaneController->removePaneFactoryForURL(iDescriptor->msPaneURL);
            // When the pane has been released then the previous call did
            // not lead (will lead on the next configuration update) to the
            // destruction of the pane.  In this case we have to do it now.
            if (iDescriptor->mbIsReleased)
            {
                xComponent = Reference<XComponent>(iDescriptor->mxPane, UNO_QUERY);
                if (xComponent.is())
                {
                    xComponent->removeEventListener(this);
                    xComponent->dispose();
                }
            }
        }

        if (mxConfigurationController.is())
            mxConfigurationController->removeConfigurationChangeListener(this);
    }
}




void SAL_CALL BasicPaneFactory::initialize (const Sequence<Any>& aArguments)
    throw (Exception, RuntimeException)
{
    if (aArguments.getLength() > 0)
    {
        try
        {
            // Get the XController from the first argument.
            Reference<frame::XController> xController (aArguments[0], UNO_QUERY_THROW);

            // Tunnel through the controller to obtain access to the ViewShellBase.
            try
            {
                Reference<lang::XUnoTunnel> xTunnel (xController, UNO_QUERY_THROW);
                DrawController* pController
                    = reinterpret_cast<DrawController*>(
                        (sal::static_int_cast<sal_uIntPtr>(
                            xTunnel->getSomething(DrawController::getUnoTunnelId()))));
                mpViewShellBase = pController->GetViewShellBase();
                mpUpdateLockManager = mpViewShellBase->GetUpdateLockManager();
            }
            catch(RuntimeException&)
            {}

            Reference<XControllerManager> xControllerManager(xController, UNO_QUERY_THROW);
            mxPaneController = xControllerManager->getPaneController();

            // Add pane factories for the two left panes (one for Impress and one for
            // Draw), the center pane, and the right pane.
            if (mxPaneController.is() && xController.is())
            {
                Reference<XComponent> xComponent (mxPaneController, UNO_QUERY);
                if (xComponent.is())
                    xComponent->addEventListener(this);

                PaneDescriptor aDescriptor;
                aDescriptor.msPaneURL = FrameworkHelper::msCenterPaneURL;
                aDescriptor.mePaneId = CenterPane;
                aDescriptor.mbIsReleased = false;
                mpPaneContainer->push_back(aDescriptor);
                mxPaneController->addPaneFactory(aDescriptor.msPaneURL, this);

                aDescriptor.msPaneURL = FrameworkHelper::msLeftImpressPaneURL;
                aDescriptor.mePaneId = LeftImpressPane;
                mpPaneContainer->push_back(aDescriptor);
                mxPaneController->addPaneFactory(aDescriptor.msPaneURL, this);

                aDescriptor.msPaneURL = FrameworkHelper::msLeftDrawPaneURL;
                aDescriptor.mePaneId = LeftDrawPane;
                mpPaneContainer->push_back(aDescriptor);
                mxPaneController->addPaneFactory(aDescriptor.msPaneURL, this);

                aDescriptor.msPaneURL = FrameworkHelper::msRightPaneURL;
                aDescriptor.mePaneId = RightPane;
                mpPaneContainer->push_back(aDescriptor);
                mxPaneController->addPaneFactory(aDescriptor.msPaneURL, this);
            }

            // Register as configuration change listener.
            mxConfigurationController = xControllerManager->getConfigurationController();
            mxConfigurationController->addConfigurationChangeListener(
                this,
                FrameworkHelper::msConfigurationUpdateStartEvent,
                makeAny(gnConfigurationUpdateStartEvent));
            mxConfigurationController->addConfigurationChangeListener(
                this,
                FrameworkHelper::msConfigurationUpdateEndEvent,
                makeAny(gnConfigurationUpdateEndEvent));
        }
        catch (RuntimeException&)
        {
            Reference<XPaneController> xPaneController (mxPaneController);
            if (mxPaneController.is())
            {
                mxPaneController = NULL;
                for (PaneContainer::const_iterator iDescriptor = mpPaneContainer->begin();
                     iDescriptor != mpPaneContainer->end();
                     ++iDescriptor)
                {
                    xPaneController->removePaneFactoryForURL(iDescriptor->msPaneURL);
                }
            }
        }
    }
}




//===== XPaneFactory ==========================================================

Reference<XPane> SAL_CALL BasicPaneFactory::createPane (
    const Reference<XResourceId>& rxPaneId)
    throw (RuntimeException)
{
    ThrowIfDisposed();

    Reference<XPane> xPane;

    // Based on the ResourceURL of the given ResourceId look up the
    // corresponding factory descriptor.
    PaneContainer::iterator iDescriptor (
        ::std::find_if (
            mpPaneContainer->begin(),
            mpPaneContainer->end(),
            ::boost::bind(&PaneDescriptor::CompareURL, _1, rxPaneId->getResourceURL())));

    if (iDescriptor != mpPaneContainer->end())
    {
        if (iDescriptor->mxPane.is())
        {
            // The pane has already been created and is still active (has
            // not yet been released).  This should not happen.
            xPane = iDescriptor->mxPane;
        }
        else
        {
            // Create a new pane.
            switch (iDescriptor->mePaneId)
            {
                case CenterPane:
                    xPane = CreateFrameWindowPane(rxPaneId);
                    break;

                case LeftImpressPane:
                case LeftDrawPane:
                case RightPane:
                    xPane = CreateChildWindowPane(
                        rxPaneId,
                        *iDescriptor);
                    break;
            }
            iDescriptor->mxPane = xPane;

            // Listen for the pane being disposed.
            Reference<lang::XComponent> xComponent (xPane, UNO_QUERY);
            if (xComponent.is())
                xComponent->addEventListener(this);
        }
        iDescriptor->mbIsReleased = false;
    }
    else
    {
        // The requested pane can not be created by any of the factories
        // managed by the called BasicPaneFactory object.
        throw lang::IllegalArgumentException(
            ::rtl::OUString::createFromAscii(
                "BasicPaneFactory::createPane() called for unknown resource id"),
            NULL,
            0);
    }

    return xPane;
}





void SAL_CALL BasicPaneFactory::releasePane (
    const Reference<XPane>& rxPane)
    throw (RuntimeException)
{
    ThrowIfDisposed();

    // Based on the given XPane reference look up the corresponding factory
    // descriptor.
    PaneContainer::iterator iDescriptor (
        ::std::find_if(
            mpPaneContainer->begin(),
            mpPaneContainer->end(),
            ::boost::bind(&PaneDescriptor::ComparePane, _1, rxPane)));

    if (iDescriptor != mpPaneContainer->end())
    {
        // The given pane was created by one of the factories.  Child
        // windows are just hidden and will be reused when requested later.
        // Other windows are disposed and their reference is reset so that
        // on the next createPane() call for the same pane type the pane is
        // created anew.
        ChildWindowPane* pChildWindowPane = dynamic_cast<ChildWindowPane*>(rxPane.get());
        if (pChildWindowPane != NULL)
        {
            iDescriptor->mbIsReleased = true;
            pChildWindowPane->Hide();
        }
        else
        {
            iDescriptor->mxPane = NULL;
            Reference<XComponent> xComponent (rxPane, UNO_QUERY);
            if (xComponent.is())
            {
                // We are disposing the pane and do not have to be informed of
                // that.
                xComponent->removeEventListener(this);
                xComponent->dispose();
            }
        }
    }
    else
    {
        // The given XPane reference is either empty or the pane was not
        // created by any of the factories managed by the called
        // BasicPaneFactory object.
        throw lang::IllegalArgumentException(
            ::rtl::OUString::createFromAscii(
                "BasicPaneFactory::releasePane() called for pane that that was not created by same factory."),
            NULL,
            0);
    }
}




//===== XConfigurationChangeListener ==========================================

void SAL_CALL BasicPaneFactory::notifyConfigurationChange (
    const ConfigurationChangeEvent& rEvent)
    throw (RuntimeException)
{
    if (mxConfigurationController.is())
    {
        sal_Int32 nEventType = 0;
        rEvent.UserData >>= nEventType;
        switch (nEventType)
        {
            case gnConfigurationUpdateStartEvent:
                // Lock UI updates while we are switching the views except
                // for the first time after creation.  Outherwise this leads
                // to problems after reload (missing resizes for the side
                // panes).
                if (mbFirstUpdateSeen)
                {
                    if (mpUpdateLockManager.get()!=NULL)
                    {
                        ::osl::Guard< ::osl::Mutex > aGuard (::osl::Mutex::getGlobalMutex());
                        mpUpdateLockManager->Lock();
                    }
                }
                else
                    mbFirstUpdateSeen = true;
                break;

            case gnConfigurationUpdateEndEvent:
                // Unlock the update lock here when only the visibility of
                // windows but not the view shells displayed in them have
                // changed.  Otherwise the UpdateLockManager takes care of
                // unlocking at the right time.
                if (mpUpdateLockManager.get() != NULL)
                {
                    ::osl::Guard< ::osl::Mutex > aGuard (::osl::Mutex::getGlobalMutex());
                    if (mpUpdateLockManager->IsLocked())
                        mpUpdateLockManager->Unlock();
                }
                break;
        }
    }
}



                
//===== lang::XEventListener ==================================================

void SAL_CALL BasicPaneFactory::disposing (
    const lang::EventObject& rEventObject)
    throw (RuntimeException)
{
    if (rEventObject.Source == mxPaneController)
    {
        mxPaneController = NULL;
    }
    else if (rEventObject.Source == mxConfigurationController)
    {
        mxConfigurationController = NULL;
    }
    else
    {
        // Has one of the panes been disposed?  If so, then release the
        // reference to that pane, but not the pane descriptor.
        PaneContainer::iterator iDescriptor (
            ::std::find_if (
                mpPaneContainer->begin(),
                mpPaneContainer->end(),
                ::boost::bind(&PaneDescriptor::ComparePane, _1,
                    Reference<XPane>(rEventObject.Source, UNO_QUERY))));
        if (iDescriptor != mpPaneContainer->end())
        {
            iDescriptor->mxPane = NULL;
        }
    }
}





Reference<XPane> BasicPaneFactory::CreateFrameWindowPane (const Reference<XResourceId>& rxPaneId)
{
    Reference<XPane> xPane;
    
    if (mpViewShellBase != NULL)
    {
        xPane = new FrameWindowPane(
            rxPaneId,
            mpViewShellBase->GetViewWindow(),
            *mpViewShellBase);
    }

    return xPane;
}




Reference<XPane> BasicPaneFactory::CreateChildWindowPane (
    const Reference<XResourceId>& rxPaneId,
    const PaneDescriptor& rDescriptor)
{
    Reference<XPane> xPane;

    if (mpViewShellBase != NULL)
    {
        // Create the corresponding shell and determine the id of the child window.
        USHORT nChildWindowId = 0;
        ::std::auto_ptr<SfxShell> pShell;
        switch (rDescriptor.mePaneId)
        {
            case LeftImpressPane:
                pShell.reset(new LeftImpressPaneShell());
                nChildWindowId = ::sd::LeftPaneImpressChildWindow::GetChildWindowId();
                break;
                
            case LeftDrawPane:
                pShell.reset(new LeftDrawPaneShell());
                nChildWindowId = ::sd::LeftPaneDrawChildWindow::GetChildWindowId();
                break;
            
            case RightPane:
                pShell.reset(new RightPaneShell());
                nChildWindowId = ::sd::RightPaneChildWindow::GetChildWindowId();
                break;

            default:
                break;
        }

        // With shell and child window id create the ChildWindowPane
        // wrapper.
        if (pShell.get() != NULL)
        {
            xPane = new ChildWindowPane(
                rxPaneId,
                nChildWindowId,
                *mpViewShellBase,
                pShell);
        }
    }

    return xPane;
}




//-----------------------------------------------------------------------------

void BasicPaneFactory::ThrowIfDisposed (void) const
    throw (lang::DisposedException)
{
	if (rBHelper.bDisposed || rBHelper.bInDispose)
	{
        throw lang::DisposedException (
            ::rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(
                "BasicPaneFactory object has already been disposed")),
            const_cast<uno::XWeak*>(static_cast<const uno::XWeak*>(this)));
    }
}


} } // end of namespace sd::framework
