/*
 *  Copyright (C) 2001 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distrnibuted in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 * Saving and loading bookmarks in galeon own xml format 
 */

#include "bookmarks_io.h"
#include "misc_general.h"

#include <string.h>
#include <glade/glade-xml.h>
#include <gnome-xml/xmlmemory.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>

/* This is the only exported function. */
BookmarkTranslator *bookmarks_io_own_format_init (void);


/* private functions */
static BookmarkItem *load_from_file (const gchar *filename, 
				     BookmarkItem **new_default_folder);
static BookmarkItem *load_from_string (const gchar *data, 
				       BookmarkItem **new_default_folder);
static gboolean save_to_file (BookmarkItem *root,
			      BookmarkItem *default_folder,
			      const gchar *file);
static gboolean save_to_string (BookmarkItem *root,
				BookmarkItem *default_folder,
				gchar **data);
static BookmarkItem *bookmarks_xml_read (xmlNodePtr item,
					 BookmarkItem **default_bookmarks);
static BookmarkItem *bookmarks_xml_read_item (BookmarkLoaderInfo *li,
					      xmlNodePtr item);
static void bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root, 
					BookmarkItem *b,
					BookmarkItem *default_bookmarks_root);


/**
 * Info about the translator
 */

static gchar *extensions[] = { "xml", NULL };

static BookmarkTranslator own_format = {
	N_("Galeon's old bookmarks format"),
	extensions,
	load_from_file,
	load_from_string,
	save_to_file,
	save_to_string
};

BookmarkTranslator *
bookmarks_io_own_format_init (void)
{
	if (g_list_find (bookmarks_translators, &own_format) == NULL)
		bookmarks_translators = g_list_prepend (bookmarks_translators,
							&own_format);
	return &own_format;
}

static BookmarkItem *
load_from_file (const char *file, BookmarkItem **default_bookmarks_root)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookmarkItem *b = NULL;

	if (!(g_file_exists (file))) {
		/* no bookmarks */
		return NULL;
	}

	doc = xmlParseFile (file);	
	
	if (doc) {
		item = doc->root;
		b = bookmarks_xml_read (item, default_bookmarks_root);
		xmlFreeDoc (doc);
		return b;
	} else {
		g_warning ("unable to parse bookmarks file: %s", file);
		return NULL;
	}
}

/**
 * bookmarks_xml_read: read all items from xml file, doing alias resolution
 * @item: the root of the tree to read bookmarks from
 * @default_bookmarks_root: if not NULL, it is set to the bookmark marked 
 * as default_bookmarks_root in the file
 */
static BookmarkItem * 
bookmarks_xml_read (xmlNodePtr item, BookmarkItem **default_bookmarks_root) 
{
	BookmarkLoaderInfo *li = bookmarks_io_loader_info_new ();
	BookmarkItem *result;
	item = item->childs;
	
	li->root = bookmarks_xml_read_item (li, item);
	
	bookmarks_io_resolve_loaded_alias (li);

	if (default_bookmarks_root != NULL) 
	{
		*default_bookmarks_root = li->default_bookmarks_root;
	}
	result = li->root;
	bookmarks_io_loader_info_free (li);
	return result;
}

/**
 * bookmarks_xml_read_item: read an item from xml file. Does not resolve 
 * aliases. Use with care.
 */
static BookmarkItem *
bookmarks_xml_read_item (BookmarkLoaderInfo *li, xmlNodePtr item)
{
	BookmarkItem *b = NULL;
	BookmarkItem *b2;
	gchar *ct, *expanded, *ccm, *default_root, *tbs, *ak, *am;
	guint idval = 0;

	gchar *name = xmlGetRawProp (item, "name");	
	gchar *notes = xmlGetRawProp (item, "notes");
	gchar *nick = xmlGetRawProp (item, "nick");
	gchar *pixmap_file = xmlGetProp (item, "pixmap");
	gchar *idstr = xmlGetProp (item, "id");
	gchar *time_added_str = xmlGetProp (item, "time_added");
	gchar *time_modified_str = xmlGetProp (item, "time_modified");
	gchar *time_visited_str = xmlGetProp (item, "time_visited");
  
	if (idstr) idval = strtoul (idstr, NULL, 10);

	if (strcmp (item->name, "folder") == 0 ||
	    strcmp (item->name, "category") == 0) 
	{
		b = bookmarks_new_bookmark (BM_FOLDER, FALSE, name, NULL,
					    nick, notes, pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		tbs = xmlGetProp (item, "toolbar_style");
		ccm = xmlGetProp (item, "create_context_menu");
		expanded = xmlGetProp (item, "expanded");
	        default_root = xmlGetProp (item, "default_bookmarks_root");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE") ? 
				     TRUE : FALSE);
		if (tbs)
			b->toolbar_style = atoi(tbs);
		if (b->toolbar_style > 2) /* backwards compatiblity FIXME */
			b->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE") ? 
					  TRUE : FALSE);
		b->expanded = (expanded && !strcmp (expanded, "TRUE") ? 
			       TRUE : FALSE);
		if (default_root && !strcmp (default_root, "TRUE"))
		{
			li->default_bookmarks_root = b;
		}
	        xmlFree (ct);		
		xmlFree (tbs);
		xmlFree (ccm);
		xmlFree (expanded);
		xmlFree (default_root);
		item = item->childs;
		while (item != NULL)
		{
			xmlNodePtr item2 = item;
			b2 = bookmarks_xml_read_item (li, item2);
			if (b2) 
			{
				b->list = g_list_append (b->list, b2);
				b2->parent = b;
			}
			item = item->next;
		}
	} 
	else if (!strcmp (item->name, "separator"))
	{
		b = bookmarks_new_bookmark (BM_SEPARATOR, FALSE, name, NULL,
					    nick, notes, pixmap_file);
	} 
	else if (!strcmp (item->name, "autobookmarks"))
	{
		/* FIXME: remove code duplication */
		b = bookmarks_new_bookmark (BM_AUTOBOOKMARKS, FALSE, name, 
					    NULL, nick, notes, pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		tbs = xmlGetProp (item, "toolbar_style");
		ccm = xmlGetProp (item, "create_context_menu");
		expanded = xmlGetProp (item, "expanded");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE")) ? TRUE 
			: FALSE;
		if (tbs)
			b->toolbar_style = atoi(tbs);
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE")) ? TRUE
			: FALSE;
		b->expanded = (expanded && !strcmp (expanded, "TRUE")) ? TRUE
			: FALSE;
		xmlFree (ct);
		xmlFree (tbs);
		xmlFree (ccm);
		xmlFree (expanded);
	} 
	else if (!strcmp (item->name, "alias")) 
	{
		b = bookmarks_new_alias (NULL);
		li->unresolved_aliases = g_list_prepend 
			(li->unresolved_aliases, 
			 bookmarks_io_unresolved_alias_info_new (b, idval));

		/* free the idstr pointer to avoid adding this bookmark 
		   to the id to bookmark mapping, because it would overwrite 
		   the real bookmark (and we'd have an alias to itself) */
		g_free (idstr);
		idstr = NULL;

		ak = xmlGetProp (item, "accel_key");
		if (ak)
			b->accel_key = atoi (ak);
		xmlFree (ak);
		am = xmlGetProp (item, "accel_mods");
		if (am)
			b->accel_mods = atoi (am);
		xmlFree (am);

		ct = xmlGetProp (item, "create_toolbar");
		tbs = xmlGetProp (item, "toolbar_style");
		ccm = xmlGetProp (item, "create_context_menu");

		b->create_toolbar = (ct && !strcmp (ct, "TRUE") ? 
				     TRUE : FALSE);
		if (tbs)
			b->toolbar_style = atoi(tbs);
		if (b->toolbar_style > 2) /* backwards compatiblity FIXME */
			b->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE") ? 
					  TRUE : FALSE);

		xmlFree (ct);
		xmlFree (tbs);
		xmlFree (ccm);

	} 
	else  if (!strcmp (item->name, "site")) 
	{
		/* site */
		gchar *url = xmlGetProp (item, "url");
		gchar *smarturl = xmlGetProp (item, "smarturl");
		b = bookmarks_new_bookmark (BM_SITE, FALSE, name, url, 
					    nick, notes, pixmap_file);
		ccm = xmlGetProp (item, "create_context_menu");
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE") ? 
					  TRUE : FALSE);
		xmlFree (ccm);
		if (url) g_free (url);
		ak = xmlGetProp (item, "accel_key");
		if (ak)
			b->accel_key = atoi (ak);
		xmlFree (ak);
		am = xmlGetProp (item, "accel_mods");
		if (am)
			b->accel_mods = atoi (am);
		xmlFree (am);
		if (smarturl)
		{
			b->smarturl = g_strdup (smarturl);
			xmlFree (smarturl);
		}
		/* FIXME backwards compat */
		else if (b->url 
			 && strstr (b->url, "%s")
			 && !b->smarturl)
		{
			b->smarturl = g_strdup (b->url);
		}
	}
	else
	{
		/* error: unexpected node. Maybe only a harmless text node  */
	}
	if (b)
	{
		if (time_added_str)
		{
			b->time_added = strtol (time_added_str, NULL, 10);
			g_free (time_added_str);
		}
		if (time_modified_str)
		{
			b->time_modified = strtol (time_modified_str, 
						   NULL, 10);
			g_free (time_modified_str);
		}
		if (time_visited_str)
		{
			b->time_visited = strtol (time_visited_str, NULL, 10);
			g_free (time_visited_str);
		}
		
		if (idstr != NULL)  
		{
			/* if the id attribute was present, add this item to 
			   the bookmarks id map */
			g_hash_table_insert (li->id_bookmark,
					     GINT_TO_POINTER (idval), b);
		}
	}

	g_free (name);
	g_free (notes);
	g_free (nick);
	g_free (pixmap_file);
	g_free (idstr);
	
	return b;
}

static gboolean
save_to_file (BookmarkItem *bookmarks_root,
	      BookmarkItem *default_bookmarks_folder,
	      const gchar *filename)
{
	xmlNodePtr node;
	xmlDocPtr doc;
	int size;
	
	/* build an XML document and save it to the filename */
	doc = xmlNewDoc ("1.0");
	node = xmlNewDocNode (doc, NULL, "bookmarks", NULL);
	xmlDocSetRootElement (doc, node);
	bookmarks_save_recursively (doc, node, bookmarks_root, 
				    default_bookmarks_folder);
	size = xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);     

	return size > 0;
}

/**
 * bookmarks_save_recursively: recursively save a bookmarks tree to 
 * the xml file 
 */
static void
bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root, BookmarkItem *b,
			    BookmarkItem *default_bookmarks_root)
{
	BookmarkItem *b2;
	xmlNodePtr node;
	GList *li;
	gchar *idstr;
	GString *s = g_string_new ("");
	gchar *tmp;

	if (b->alias_of)
	{
		idstr = g_strdup_printf
			("%lu", (gulong) bookmarks_find_real_bookmark (b));
		node = xmlNewDocNode (doc, NULL, "alias", NULL);
		xmlSetProp (node, "id", idstr);
		if (b->create_toolbar)
			xmlSetProp (node, "create_toolbar", "TRUE");
		if (b->toolbar_style != TOOLBAR_STYLE_HORIZONTAL)
		{
			tmp = g_strdup_printf("%d", b->toolbar_style);
			xmlSetProp (node, "toolbar_style", tmp);
			g_free (tmp);
		}
		if (b->create_context_menu)
			xmlSetProp (node, "create_context_menu", "TRUE");
		if (b->accel_key) {
			g_string_sprintf (s, "%d", b->accel_key);
			xmlSetProp (node, "accel_key", s->str);
		}
		if (b->accel_mods) {
			g_string_sprintf (s, "%d", b->accel_mods);
			xmlSetProp (node, "accel_mods", s->str);
		}
		if (b->time_added) {
			g_string_sprintf (s, "%d", b->time_added);
			xmlSetProp (node, "time_added", s->str);
		}
		xmlAddChild (root, node);
	} 
	else
	{
		idstr = g_strdup_printf ("%lu", (gulong) b);
		switch (b->type)
		{
		case BM_FOLDER:
			node = xmlNewDocNode (doc, NULL, "folder", NULL);
			if (b->alias)
				xmlSetProp (node, "id", idstr);
			xmlSetRawProp (node, "name", b->name);
			if (b->create_toolbar)
				xmlSetProp (node, "create_toolbar", "TRUE");
			if (b->toolbar_style != TOOLBAR_STYLE_HORIZONTAL)
			{
				tmp = g_strdup_printf("%d", b->toolbar_style);
				xmlSetProp (node, "toolbar_style", tmp);
				g_free (tmp);
			}
			if (b->create_context_menu)
				xmlSetProp (node, "create_context_menu", 
					    "TRUE");
			if (b->expanded)
				xmlSetProp (node, "expanded", "TRUE");
			if (*b->notes)
				xmlSetRawProp (node, "notes", b->notes);
			if (default_bookmarks_root == b) {
				xmlSetProp (node, "default_bookmarks_root", 
					    "TRUE");
			}
			if (b->pixmap_file && *b->pixmap_file)
				xmlSetProp (node, "pixmap", b->pixmap_file);
			if (b->time_added) {
				g_string_sprintf (s, "%d", b->time_added);
				xmlSetProp (node, "time_added", s->str);
			}
			if (b->time_modified) {
				g_string_sprintf (s, "%d", b->time_modified);
				xmlSetProp (node, "time_modified", s->str);
			}
			if (b->time_visited) {
				g_string_sprintf (s, "%d", b->time_visited);
				xmlSetProp (node, "time_visited", s->str);
			}
			xmlAddChild (root, node);
			
			for (li = b->list; li != NULL; li = li->next) {
				b2 = li->data;
				bookmarks_save_recursively 
					(doc, node, b2,
					 default_bookmarks_root);
		}
			break;
			
		case BM_SEPARATOR:
			node = xmlNewDocNode(doc, NULL, "separator", NULL);
			xmlAddChild(root, node);
			break;

		case BM_SITE:
			node = xmlNewDocNode(doc, NULL, "site", NULL);
			if (b->alias)
				xmlSetProp (node, "id", idstr);
			xmlSetRawProp (node, "name", b->name);
			xmlSetProp (node, "url", b->url);    
			if (*b->nick)
				xmlSetRawProp (node, "nick", b->nick);
			if (*b->notes)
				xmlSetRawProp (node, "notes", b->notes);
			if (b->create_context_menu)
				xmlSetProp (node, "create_context_menu",
					    "TRUE");
			if (b->accel_key) {
				g_string_sprintf (s, "%d", b->accel_key);
				xmlSetProp (node, "accel_key", s->str);
			}
			if (b->accel_mods) {
				g_string_sprintf (s, "%d", b->accel_mods);
				xmlSetProp (node, "accel_mods", s->str);
			}
			if (b->pixmap_file && *b->pixmap_file)
				xmlSetProp (node, "pixmap", b->pixmap_file);
			if (b->time_added) {
				g_string_sprintf (s, "%d", b->time_added);
				xmlSetProp (node, "time_added", s->str);
			}
			if (b->time_modified) {
				g_string_sprintf (s, "%d", b->time_modified);
				xmlSetProp (node, "time_modified", s->str);
			}
			if (b->time_visited) {
				g_string_sprintf (s, "%d", b->time_visited);
				xmlSetProp (node, "time_visited", s->str);
			}
			if (b->smarturl) {
				xmlSetProp (node, "smarturl", b->smarturl);
			}
			xmlAddChild(root, node);
			break;
			
		case BM_AUTOBOOKMARKS:
			node = xmlNewDocNode(doc, NULL, "autobookmarks", NULL);
			if (b->alias)
				xmlSetProp (node, "id", idstr);
			xmlSetRawProp (node, "name", b->name);
			if (b->create_toolbar)
				xmlSetProp (node, "create_toolbar", "TRUE");
			if (b->toolbar_style != TOOLBAR_STYLE_HORIZONTAL)
			{
				tmp = g_strdup_printf("%d", b->toolbar_style);
				xmlSetProp (node, "toolbar_style", tmp);
				g_free (tmp);
			}
			if (b->create_context_menu)
				xmlSetProp (node, "create_context_menu", 
					    "TRUE");
			if (b->expanded)
				xmlSetProp (node, "expanded", "TRUE");
			if (*b->notes)
				xmlSetRawProp (node, "notes", b->notes);
			if (b->pixmap_file && *b->pixmap_file)
				xmlSetProp(node, "pixmap", b->pixmap_file);
			xmlAddChild(root, node);
			break;
		}
	}
        g_string_free (s, TRUE);
	g_free (idstr);
}


static gboolean
save_to_string (BookmarkItem *root,
		BookmarkItem *default_folder,
		gchar **data)
{
	xmlNodePtr node;
	xmlDocPtr doc;
	xmlChar *mem;
	int size;

	g_return_val_if_fail (data != NULL, FALSE);

	doc = xmlNewDoc ("1.0");
	node = xmlNewDocNode (doc, NULL, "single_bookmark", NULL);
	bookmarks_save_recursively (doc, node, root, default_folder);
	xmlDocSetRootElement (doc, node);
	xmlDocDumpMemory (doc, &mem, &size);
	xmlFreeDoc (doc);
	
	*data = mem;

	return TRUE;
}

BookmarkItem *
load_from_string (const gchar *data, 
		  BookmarkItem **new_default_folder)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookmarkItem *b = NULL;

	doc = xmlParseMemory ((gchar *) data, strlen (data));	
	
	if (doc)
	{
		item = doc->root;
		b = bookmarks_xml_read (item, new_default_folder);
		xmlFreeDoc (doc);
	} 
	else
	{
		return NULL;
	}
	return b;
}
