/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozcallbacks.h"
#include "mozilla_prefs.h"
#include "embed.h"
#include "misc_general.h"
#include "window.h"
#include "history.h"
#include "spinner.h"
#include "session.h"
#include "mozilla.h"
#include "context.h"
#include "prefs.h"
#include "downloader.h"
#include "menubar.h"
#include "link_interfaces.h"
#include "bookmarks.h"
#include "favicon.h"
#include "stylesheets.h"
#include "stroke.h"
#include "gestures.h"

#include <time.h>
#include <string.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-triggers.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <gdk/gdkkeysyms.h>

WrapperMouseEventInfo *mouse_down_info = NULL;
static void mozembed_gesture_connect (GaleonEmbed *embed,
				      guint button);
static gboolean mozembed_gesture_motion_cb (GtkWidget *widget,
					    GdkEventMotion *e,
					    GaleonEmbed *embed);
static gboolean mozembed_gesture_press_cb (GtkWidget *widget,
					   GdkEventButton *e,
					   GaleonEmbed *embed);
static gboolean mozembed_gesture_release_cb (GtkWidget *widget,
					     GdkEventButton *e,
					     GaleonEmbed *embed);

/**
 * mozembed_new_window_cb: GTKMOZEMBED SIGNAL, emitted any time a new 
 * window is requested by the document 
 */
void 
mozembed_new_window_cb (GtkMozEmbed *dummy, GtkMozEmbed **retval, 
			guint chrome_mask, GaleonEmbed *embed) 
{
	GaleonEmbed *new_embed;
	GaleonWindow *window;
	GaleonWindow *new_window;
	gboolean open_in_tab;

	/* check callback args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;
	g_assert (retval != NULL);

	/* check config */
	open_in_tab = eel_gconf_get_boolean (CONF_TABS_TABBED_POPUPS);

	/* override if it's some sort of XUL (must be some sort of dialog) */
	if ((chrome_mask & GTK_MOZ_EMBED_FLAG_OPENASCHROME) != 0)
	{
		open_in_tab = FALSE;
	}

#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed))
	{
		/* no tabs inside nautilus */
		open_in_tab = FALSE;
	}
#endif

	/* create a new browser */
	new_embed = embed_create_after_embed_chrome (embed, !open_in_tab,
						     NULL,
						     EMBED_CREATE_RAISE_WINDOW |
						     EMBED_CREATE_DONT_SHOW,
						     chrome_mask);

	if ((chrome_mask & GTK_MOZ_EMBED_FLAG_OPENASCHROME) != 0)
	{
		new_window = new_embed->parent_window;
		return_if_not_window (new_window);
		gtk_window_set_transient_for (GTK_WINDOW(new_window->wmain),
					      GTK_WINDOW(window->wmain));
	}

	/* set the popup flag */
	if (new_embed->parent_window != embed->parent_window)
	{
		new_embed->parent_window->is_popup = TRUE;
	}

	/* return the new browser to gtkmozembed */
	*retval = GTK_MOZ_EMBED (new_embed->mozembed);
}

void new_window_orphan_cb (GtkMozEmbedSingle *embed,
			   GtkMozEmbed **retval, guint chrome_mask,
			   gpointer data)
{
	GaleonEmbed *new_embed;

	/* create a new browser */
	new_embed = embed_create_in_window_chrome (NULL, NULL, NULL,
						   EMBED_CREATE_RAISE_WINDOW |
						   EMBED_CREATE_DONT_SHOW,
						   chrome_mask);

	/* set the popup flag */
	new_embed->parent_window->is_popup = TRUE;

	/* return the new browser to gtkmozembed */
	*retval = GTK_MOZ_EMBED (new_embed->mozembed);
}

/**
 * mozembed_visibility_cb: GTKMOZEMBED SIGNAL, emitted when the toplevel
 * window need to be showed or hidden
 */
void 
mozembed_visibility_cb (GtkMozEmbed *dummy, gboolean visibility, 
			GaleonEmbed *embed) 
{
        GaleonWindow *window;
	return_if_not_sane_embed (embed);

#if ENABLE_NAUTILUS_VIEW
	/* we can't raise the window when we are embedded in Nautilus */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif
	
	window = embed->parent_window;

	/* set visiblity of this embed */
	embed_set_visibility (embed, visibility);

	if (visibility && 
	    window->is_popup &&
	    !window->layered)
	{
	        window_set_layer (window);
	}

	/* do embed init if necessary */
	if (visibility && !embed->wrapper)
	{
		embed_wrapper_init (embed);
	}
}

/**
 * mozembed_destroy_brsr_cb: GTKMOZEMBED SIGNAL, emitted when the document
 * has requested that the toplevel window be closed
 */
void 
mozembed_destroy_brsr_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	return_if_not_embed (embed);

	if (embed->being_closed) return;

	/* close the GaleonEmbed */
	embed_close (embed);
}

/**
 * mozembed_location_changed_cb: GTKMOZEMBED SIGNAL, emitted any time that
 * the location of the document has changed
 */
void
mozembed_location_changed_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	GaleonWindow *window;
	gint zoom;

 	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* update in embed (and window if necessary) */
	embed_update_location (embed);

	/* update the buttons view */
	if (embed->is_active && window)
		window_update_nav_controls (window);

	/* autosave the updated session, if about:blank hasn't been loaded */
	if (strcmp ("about:blank", embed->location))
	{
		/* make sure we're not recovering from a crash (in which
		 * case, we don't want to autosave the session yet) */
		if (!embed->dont_autosave_session)
			session_autosave ();
		else embed->dont_autosave_session = FALSE;
	}

	/* check zoom setting for the new site, unless it has 
	   a javascript URI */
	if (embed->location 
	    && strncmp ("javascript:", embed->location, 11))
	{
		zoom = history_get_zoom (embed->location);
		if (zoom == 0 && embed->zoom_auto_set)
		{
			embed_set_zoom (embed, 100, FALSE);
		}
		else if (zoom != 0 && zoom != embed->zoom)
		{
			embed_set_zoom (embed, zoom, FALSE);
			embed->zoom_auto_set = TRUE;
		}
		/* update the spinner */
		if (window != NULL)
		{
			window_update_zoom (window);
		}
	}
}

/**
 * mozembed_title_changed_cb: GTKMOZEMBED SIGNAL, emitted any time that the 
 * title of the document has changed
 */
void
mozembed_title_changed_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	return_if_not_embed (embed);

	/* update the embed's title (and the window's, if appropriate) */
	embed_update_title (embed);
}

/**
 * mozembed_load_started_cb: GTKMOZEMBED SIGNAL, emitted any time that the 
 * load of a document has been started
 */
void
mozembed_load_started_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* check callback args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* initialise embed whenever a document is first loaded */
	if (embed->wrapper == NULL)
		embed_wrapper_init (embed);
	
	embed->load_started++;

	/* if we haven't already, start the window spinner, set the
	 * load_started tag, and clear the info about the progress */
	if (embed->load_started == 1)
	{
		embed_update_tab_status (embed);
		embed_progress_clear (embed);
		embed->when_started = time (NULL);

		/* start spinner if this is the active embed */
		if (embed->is_active && window)
			spinner_start (window);
#if ENABLE_NAUTILUS_VIEW
		if (EMBED_IN_NAUTILUS (embed))
			galeon_nautilus_view_report_load_underway 
				(embed->nautilus_view);
#endif
	}

	/* update the buttons view */
	if (embed->is_active && window)
	{
		window_update_nav_controls (window);
		window_statusbar_update_message (window);
		window_statusbar_update_progress_bar (window);
		if (window->csslist != NULL)
			gtk_widget_set_sensitive (window->csslist, FALSE);
	}

	/* update drag pixmap */
	favicon_update_drag_handle (embed, FALSE);

	/* no user sheet has been applied yet */
	embed->usercss_applied = FALSE;
}

/**
 * mozembed_load_finished_cb: GTKMOZEMBED SIGNAL, emitted any time that 
 * the load of a document has finished
 */
void
mozembed_load_finished_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	GaleonWindow *window;
	gchar *realurl;
	
	/* check callback args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* set in the embed */
	embed->load_started--;
	embed_progress_clear (embed);

	if (embed->load_started == 0)
	{
		/* stop spinner if this is the active embed */
		if (embed->is_active && window)
			spinner_stop (window);
#if ENABLE_NAUTILUS_VIEW
		if (EMBED_IN_NAUTILUS (embed))
			galeon_nautilus_view_report_load_complete
				(embed->nautilus_view);
#endif

		/* new content, so only viewed if this is already active */
		embed->has_been_viewed = embed->is_active;

		/* update tab status */
		embed_update_tab_status (embed);
		
		/* load saved alternate stylesheet */
		stylesheets_load_authorsheet (embed);

		/* load user stylesheet */
		stylesheets_load_usersheet (embed);

		/* update drag pixmap */
		realurl = mozilla_get_real_url (embed);
		favicon_update_drag_handle (embed,
			!(realurl && embed->location &&
			  strcmp (embed->location, realurl)));
		g_free (realurl);

		/* Update the recent menu */
		if (window && embed->location)
		{
			window_add_recent_menu_entry
				(window, 
				 embed->location, embed->title);
		}
	}

	/* update the buttons view */
	if (embed->is_active)
	{
		if (window)
		{
			window_update_nav_controls (window);
			window_statusbar_update_message (window);
			window_statusbar_update_progress_bar (window);
		}

#ifdef ENABLE_NAUTILUS_VIEW
		if (!EMBED_IN_NAUTILUS (embed))
		{
#endif
			/* check if the wrapper is already initialized */
			stylesheets_menu_build (embed);
			link_interfaces_set_sensitivity (embed);
#ifdef ENABLE_NAUTILUS_VIEW
		}
#endif
	}

	/* hack to workaround a bunch of problems when no document is loaded */
	if (embed->realurl != NULL)
	{
		/* Nullify before loading the url,
		   I dont get how that happen but the finish callback appear to be
		   called again before finish_cb exit */
		gchar *tmp = g_strdup (embed->realurl);
		g_free (embed->realurl);
		embed->realurl = NULL;

		embed_load_url (embed, tmp);

		if ((embed->create_flags & EMBED_CREATE_GRAB_FOCUS) &&
		    (window->visible_embeds == 1 || 
		     eel_gconf_get_boolean (CONF_TABS_TABBED_AUTOJUMP)))
		{
			embed_grab_focus (embed);
		}

		g_free (tmp);
	}
}

/**
 * mozembed_net_status_change_cb: GTKMOZEMBED SIGNAL, emitted any time that
 * there is a change in the status of network loading
 */
void mozembed_net_status_change_cb (GtkMozEmbed *dummy,
				    const char *uri, gint flags, 
				    guint status, GaleonEmbed *embed) 
{
	GaleonWindow *window;

	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* if this is a new embed, set modified_location, so the location
	 * will show up in the entry */
	if (!embed->location && !embed->modified_location && uri &&
	    strcmp (uri, "about:blank") &&
	    strcmp (uri, "about:parser-dummy-request") &&
	    strcmp (uri, "about:layout-dummy-request"))
	{
		embed->modified_location = g_strdup (uri);
		embed->location_is_modified = TRUE;

#ifdef ENABLE_NAUTILUS_VIEW
		if (EMBED_IN_NAUTILUS (embed))
		{
			galeon_nautilus_view_set_location (
						embed->nautilus_view,
						embed->modified_location);
		}
		else
#endif
		{
			return_if_not_window (window);
			
			/* update the text */
			if (embed->is_active)
				window_update_location_entry_text (window);
		}
	}

	/* clear temporary message, if any */
	embed_set_temp_statusbar_message (embed, NULL);

	if (flags & GTK_MOZ_EMBED_FLAG_IS_REQUEST)
	{
		if (flags & GTK_MOZ_EMBED_FLAG_START)
		{
			/* ignore */
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_REDIRECTING)
		{
			embed->load_status_message =
				_("Redirecting to site...");
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_TRANSFERRING)
		{
			embed->load_status_message = 
				_("Transferring data from site...");
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_NEGOTIATING)
		{
			embed->load_status_message = 
				_("Waiting for authorization...");
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_STOP)
		{
			/* ignore */
		}
	}

	if (flags & GTK_MOZ_EMBED_FLAG_IS_DOCUMENT)
	{
		if (flags & GTK_MOZ_EMBED_FLAG_START)
		{
			embed->load_status_message = _("Loading site...");
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_STOP)
		{
			embed->load_status_message = _("Done.");
		}
	}

	switch (status)
	{
	case GTK_MOZ_EMBED_STATUS_FAILED_DNS:
		embed->load_status_message = _("Site not found.");
		break;
		
	case GTK_MOZ_EMBED_STATUS_FAILED_CONNECT:
		embed->load_status_message =
			_("Failed to connect to site.");
		break;

	case GTK_MOZ_EMBED_STATUS_FAILED_TIMEOUT:
		embed->load_status_message =
			_("Failed due to connection timeout.");
		break;

	case GTK_MOZ_EMBED_STATUS_FAILED_USERCANCELED:
		/* don't do a message for this */
		break;

	default:
/* FIXME: find out what these are, but for now it's best not to
 * alarm users with lots of warning messages -- MattA
		g_warning ("unhandled status message 0x%08x\n", status);
 */
		break;
	}
#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed))
	{
		const gchar *msg = embed->temp_statusbar_message;
		if (msg == NULL)
		{
			msg = embed->load_status_message;
		}
		if (msg == NULL)
		{
			msg = "";
		}
		galeon_nautilus_view_set_statusbar (embed->nautilus_view, msg);
		if (embed->load_started)
		{
			galeon_nautilus_view_report_load_progress 
				(embed->nautilus_view, 
				 embed->load_percent);
		}
	}
#endif

	/* display the update */
	if (embed->is_active && window)
	{
		window_statusbar_update_message (window);
		window_statusbar_update_progress_bar (window);
	}
	// NAUTIFIXME: report the failed load
}

/**
 * mozembed_progress_change_cb: GTKMOZEMBED SIGNAL, emitted any time that 
 * there is a change in the progress of loading a document.
 */
void mozembed_progress_change_cb (GtkMozEmbed *dummy, gint cur, gint max,
				  GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* check callback args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* compute percentages */
	if (max == -1)
	{
		embed->load_percent = 0;
		embed->bytes_loaded = cur;
		embed->max_bytes_loaded = 0;
	}
	else if (cur > max)
	{
		/* sometimes length of the downloaded document is 
		 * greater than the length of the document */
		embed->load_percent = 100;
		embed->bytes_loaded = cur;
		embed->max_bytes_loaded = max;
	} 
	else
	{
		/* normal conditions */
		embed->bytes_loaded = cur;
		embed->max_bytes_loaded = max;
		embed->load_percent = (!max) ? 0 : (cur * 100) / max;
	}

	/* update view */
#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed))
	{
		galeon_nautilus_view_report_load_progress 
			(embed->nautilus_view,
			 ((double) embed->load_percent) / 100.0);
	}
	else
#endif
	if (embed->is_active && window)
	{
		window_statusbar_update_message (window);
		window_statusbar_update_progress_bar (window);
	}

}

/**
 * mozembed_link_message_cb: GTKMOZEMBED SIGNAL, emitted when the 
 * link message changes
 */
void
mozembed_link_message_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	char *message;

	return_if_not_embed (embed);
	
	/* get the link message */
	message = mozilla_get_link_message (embed, NULL);
	embed_set_temp_statusbar_message (embed, message);

	if (message) g_free (message);
}

/**
 * mozembed_js_status_cb: GTKMOZEMBED SIGNAL, emitted when the Javascript 
 * message status changes
 */
void
mozembed_js_status_cb (GtkMozEmbed *dummy, GaleonEmbed *embed)
{
	char *message;

	return_if_not_sane_embed (embed);

	if (eel_gconf_get_boolean (CONF_FILTERING_STATUSBAR_REWRITE))
  	{
		/* get the javascript message */
		if (embed->is_active)
		{
			message = mozilla_get_js_status (embed, NULL);
			embed_set_temp_statusbar_message (embed, message);

			if (message != NULL) g_free (message);
		}
	}
}

/**
 * mozembed_dom_mouse_down_cb: emitted on a button press event
 */
gint
mozembed_dom_mouse_down_cb (GtkMozEmbed *dummy, gpointer dom_event, 
			    GaleonEmbed *embed)
{
	gint rmb_action = 0;

	return_val_if_not_embed (embed, FALSE);

	/* free old info */
	if (mouse_down_info)
	{
		mozilla_free_context_info_sub (&mouse_down_info->ctx);
		g_free (mouse_down_info);
	}

	/* get new info */
	mouse_down_info = g_new0 (WrapperMouseEventInfo, 1);
	if (!mozilla_get_mouse_event_info (embed, dom_event, mouse_down_info)) 
	{
		mozilla_free_context_info_sub (&mouse_down_info->ctx);
		g_free (mouse_down_info);
		mouse_down_info = NULL;
		mozilla_pop_target_document (embed);
		return FALSE;
	}

	switch (mouse_down_info->button)
	{
		case 1:
		if (!(mouse_down_info->ctx.context & CONTEXT_LINK) &&
		    !(mouse_down_info->ctx.context & CONTEXT_INPUT))
		{
			gint mmb_action;

			mmb_action = eel_gconf_get_integer (
				CONF_MOUSE_MMB_ACTION);

			switch (mmb_action)
			{
			case 0:
				/* show the bookmarks menu */
				context_show_bookmark_menu (embed, 3,
					mouse_down_info->timestamp);
				break;
			case 3:
				/* perform gesture */
				/* don't do anything if we already have
				 * attached signals (avoids double-clicks) */
				if (embed->gesture_release_signal) break;

				/* start the gesture */
				mozembed_gesture_connect (embed, 2);
				break;
			default:
				/* handled elsewhere */
				break;
			}

		}
		break;
		case 2:
		{
			rmb_action = eel_gconf_get_integer (
				CONF_MOUSE_RMB_ACTION);

			/* show the context menu */
			if (rmb_action == 0)
			{
				context_show_menu (embed,
						   &mouse_down_info->ctx, 3,
						   mouse_down_info->timestamp);
			}
			/* perform gesture */
			else
			{
				/* don't do anything if we already have
				 * attached signals (avoids double-clicks) */
				if (embed->gesture_release_signal) break;

				/* start the gesture */
				mozembed_gesture_connect (embed, 3);
			}
		}
		break;
	}
	
	/* For gestures the target will be popped later, on the signal */
	if (rmb_action == 0 && embed->wrapper && !embed->being_closed)
	{
		mozilla_pop_target_document (embed);
	}

	return FALSE;
}

static guint
mozembed_modifier_to_gdk_state (guint modifier)
{
	guint state = 0;
	if (modifier & CTRL_KEY)
		state |= GDK_CONTROL_MASK;
	if (modifier & SHIFT_KEY)
		state |= GDK_SHIFT_MASK;
	if (modifier & ALT_KEY)
		state |= GDK_MOD1_MASK;
	return state;
}

/**
 * mozembed_dom_mouse_click_cb: GTKMOZEMBED SIGNAL, emitted when user 
 * clicks on the document
 */
gint 
mozembed_dom_mouse_click_cb (GtkMozEmbed *dummy, gpointer dom_event, 
			     GaleonEmbed *embed)
{
	GaleonWindow *window;
	WrapperMouseEventInfo *info;
	gboolean handled = FALSE;

	return_val_if_not_sane_embed (embed, FALSE);
	window = embed->parent_window;

	info = g_new0 (WrapperMouseEventInfo,1);
	if (!mozilla_get_mouse_event_info (embed, dom_event, info)) 
	{
		mozilla_free_context_info_sub(&info->ctx);
		g_free(info);
		mozilla_pop_target_document (embed);
		return FALSE;
	}

	/* make a sound if a link was clicked */
	if ((info->ctx.context & CONTEXT_LINK) && 
	    (info->ctx.link && info->button != 3))
	{
		gnome_triggers_do ("", "program", "galeon", 
				   "url_clicked", NULL);
	}

	switch (info->button)
	{
	case 2:
		/* this will never be reached, as far as i can tell - dan
		context_show_menu (embed, &info->ctx, -1, 0);
		handled = TRUE;
		*/
		break;

	case 1:
		if (!(info->ctx.context & CONTEXT_LINK) &&
		    !(info->ctx.context & CONTEXT_INPUT))
		{
			gint mmb_action;

			mmb_action = eel_gconf_get_integer (
				CONF_MOUSE_MMB_ACTION);

			switch (mmb_action)
			{
				/* show the bookmarks menu */
				case 0:
					/* handled elsewhere */
					break;
				/* load the url from the clipboard */
				case 1:
					/* NAUTIFIXME: why are we using the
					   wmain instead of the mozembed? */
					if (window)
						gtk_selection_convert
							(window->wmain,
							 GDK_SELECTION_PRIMARY,
							 GDK_SELECTION_TYPE_STRING,
							 GDK_CURRENT_TIME);
					break;
				/* go back */
				case 2:
					gtk_moz_embed_go_back (dummy);
					break;
				/* perform gestures */
				case 3:
					/* we do this on the mouse_down
					 * event, not here */
					break;
				/* do nothing */
				default:
					break;
			}

			handled = TRUE;
		}
		break;

	case 0: 
		if ((info->ctx.context & CONTEXT_IMAGE) &&
		    (info->modifier & CTRL_KEY))
		{
			embed_save_image (embed, info->ctx.img, FALSE);
			handled = TRUE;
		}
		break;

	/* perform back/forward for extra buttons */
	case 5:
		gtk_moz_embed_go_back (GTK_MOZ_EMBED (embed->mozembed));
		handled = TRUE;
		break;

	case 6:
		gtk_moz_embed_go_forward (GTK_MOZ_EMBED (embed->mozembed));
		handled = TRUE;
		break;

	default:
		break;
	}

	/* if not handled already handle with generic click handler (only if it
	 * is not a non modified left click, since we need to let moz handle
	 * that for frames to work properly ) */
	if (!handled && (info->ctx.context & CONTEXT_LINK)
	    && !(info->button == 0
		 && mozembed_modifier_to_gdk_state (info->modifier) == 0))
	{
		LinkState link_state;

		link_state = misc_general_mouse_state_to_link_state 
			(info->button + 1, 
			 mozembed_modifier_to_gdk_state (info->modifier));

		handled = embed_activate_link (embed, NULL, info->ctx.link,
					       link_state);
	}
	mozilla_free_context_info_sub (&info->ctx);
	g_free (info);

	if (!embed->being_closed && embed->wrapper)
	{
		mozilla_pop_target_document (embed);
	}

	return handled;
}

/**
 * mozembed_dom_key_press_cb: GTKMOZEMBED SIGNAL, emitted when a key is 
 * pressed
 */
gint
mozembed_dom_key_press_cb (GtkMozEmbed *dummy, gpointer dom_event, 
			   GaleonEmbed *embed)
{
	GaleonWindow *window;
	int state = 0;
	gboolean handled = FALSE;
	WrapperKeyEventInfo *info;

	return_val_if_not_sane_embed (embed, FALSE);
	window = embed->parent_window;

	info = g_new0 (WrapperKeyEventInfo,1);
	if (!mozilla_get_key_event_info(embed, dom_event, info))
	{
		mozilla_free_context_info_sub (&info->ctx);
		g_free (info);
		return FALSE;
	}

	state = mozembed_modifier_to_gdk_state (info->modifier);

	/* g_print("key = %ld  modifier = %d\n",info->key, info->modifier); */

	/* Keypresses that are not handled here are now passed on to the main
	   window's acceltable, so you should *not* handle any functions here
	   that have an accelerator key in the menubar. --Josh */

	if ((info->ctx.context & CONTEXT_LINK) && 
	    info->gdk_keyval == GDK_Return)
	{
		handled = embed_activate_link_keyboard (embed, NULL, 
							info->ctx.link, state);
	}
	else if ((info->modifier & KEY_CODE) && (info->modifier & SHIFT_KEY))
	{
		switch(info->gdk_keyval)
		{
		case GDK_F10:
			context_show_menu (embed, &info->ctx, -1, 0);
			handled = TRUE;
			break;
		default:
			break;
		}
	}
	else if ((info->modifier & KEY_CODE) && ((info->modifier & CTRL_KEY) ||
						 (info->modifier & ALT_KEY)))
	{
		switch(info->gdk_keyval)
		{
		case GDK_Left:
		case GDK_KP_Left:
			gtk_moz_embed_go_back (GTK_MOZ_EMBED 
					       (embed->mozembed));
			handled = TRUE;
			break;
		case GDK_KP_Right:
		case GDK_Right:
			gtk_moz_embed_go_forward (GTK_MOZ_EMBED
						  (embed->mozembed));
			handled = TRUE;
			break;
		default:
			break;
		}
	}
	else if ((info->modifier & ALT_KEY) && 
		 !(info->modifier & (CTRL_KEY | SHIFT_KEY)))
	{
		switch(info->gdk_keyval)
		{
		case GDK_1:
		case GDK_2:
		case GDK_3:
		case GDK_4:
		case GDK_5:
		case GDK_6:
		case GDK_7:
		case GDK_8:
		case GDK_9:
			gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook),
					       info->gdk_keyval - GDK_1);
			handled = TRUE;
			break;
		case GDK_0:
			gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook),
					       9);
			handled = TRUE;
			break;
		default:
			break;

		}
	}
	else if (info->modifier & CTRL_KEY)
	{		
		switch(info->gdk_keyval)
		{
		case GDK_l:
		case GDK_L:
			if (window && window->location_entry != NULL)
			{
				gtk_editable_select_region
					(GTK_EDITABLE (window->location_entry),
					 0, -1);
				gtk_window_set_focus 
					(GTK_WINDOW (window->wmain),
					 window->location_entry);
			}
			handled = TRUE;
			break;
		default:
			break;

		}
	}

	/* do not pass keypresses without modifiers in textareas to the
	 * gtk accel table */
	if ((info->modifier == 0) && (info->ctx.context & CONTEXT_INPUT))
		handled = TRUE;

	if ((handled == FALSE) && (info->modifier == 0))
	{
		switch (info->gdk_keyval)
		{
		case GDK_k:
			mozilla_scroll_up (embed);
			handled = TRUE;
			break;
		case GDK_j:
			mozilla_scroll_down (embed);
			handled = TRUE;
			break;
		case GDK_h:
			mozilla_scroll_left (embed);
			handled = TRUE;
			break;
		case GDK_l:
			mozilla_scroll_right (embed);
			handled = TRUE;
			break;
		default:
			break;
		}
	}

	/* We haven't specifically handled the keypress, so send it to the
	   main window in case it is a menu accelerator key */
	if (!handled && window)
	// NAUTIFIXME: don't know what to do in the bonobo case
	{
		handled = gtk_accel_groups_activate 
			(GTK_OBJECT (window->wmain), info->gdk_keyval, state);
	}
	
	mozilla_free_context_info_sub (&info->ctx);
	g_free (info);
				
	/* If it wasn't an accelerator, let Mozilla handle it */
	return handled;
}

/**
 * mozembed_size_to_cb: GTKMOZEMBED SIGNAL, emitted when a  size change 
 * is requested
 */
void 
mozembed_size_to_cb (GtkMozEmbed *dummy, gint width, gint height,
		     GaleonEmbed *embed)
{
	GaleonWindow *window;

	return_if_not_sane_embed (embed);
	window = embed->parent_window;

#if ENABLE_NAUTILUS_VIEW
	/* We can't change the size of the nautilus window, I thnk */
	if (EMBED_IN_NAUTILUS (embed)) return;
#endif

	/* when window hasn't been shown... */
	if (window->visible_embeds == 0)
	{
		/* resize the embed */
		gtk_widget_set_usize (GTK_WIDGET (embed->mozembed), 
				      width, height);
		
		/* don't override this by changing the main window size! */
		window->set_size = TRUE;
	}
	else
	{
		/* forcibly resize the window */
		gtk_widget_set_usize (GTK_WIDGET (window->wmain), 
				      width, height);
	}
}

void
mozembed_security_change_cb (GtkMozEmbed *dummy, gpointer request,
			     guint state, GaleonEmbed *embed)
{	
	GaleonWindow *window;
	gboolean security_state = (state!=0);

	/* get parent window */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	if (embed->secure != security_state || security_state)
	{
		embed->secure = security_state;
#if ENABLE_NAUTILUS_VIEW
		if (EMBED_IN_NAUTILUS (embed))
		{
			/* NAUTIFIXME: Probably we want to change the 
			   icon displayed by nautilus in the sidebar
			   and inform nautilus somehow */
		}
		else
#endif
		{
			gchar *tooltip;
			gchar *tmp =
				mozilla_make_security_tooltip (request);
			gchar *level = mozilla_security_level_string (state);
			
			if (security_state && tmp != NULL)
			{
				tooltip = g_strdup_printf
					(_("Security level: %s\n%s"),
					 level, tmp);
			}
			else
			{
				tooltip = g_strdup_printf
					(_("Security level: %s"), level);
			}
			
			window_statusbar_set_security_icon 
				(window, security_state, tooltip);
			embed->security_tooltip = tooltip;
			g_free (tmp);
			g_free (level);
		}
	}
}

/**
 * mozembed_destroy_cb: gtkmozembed component destroying
 */
void
mozembed_destroy_cb (GtkObject *object, GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* get parent window */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* no longer the active embed */
	if (embed->is_active)
	{
		embed->is_active = FALSE;
		if (window)
			window->active_embed = NULL;
	}

	/* free memory */
	if (embed->location != NULL)
	{
		g_free (embed->location);
		embed->location = NULL;
	}
	if (embed->modified_location)
	{
		g_free (embed->modified_location);
		embed->modified_location = NULL;
	}
	if (embed->title != NULL)
	{
		g_free (embed->title);
		g_free (embed->title_utf8);
		embed->title = NULL;
		embed->title_utf8 = NULL;
	}
	if (embed->temp_statusbar_message != NULL)
	{
		g_free (embed->temp_statusbar_message);
		embed->temp_statusbar_message = NULL;
	}

	/* no longer visible */
	if (embed->is_visible && window)
	{
		window->visible_embeds--;
	}

	/* destroy C++ wrapper */
	if (embed->wrapper)
	{
		mozilla_wrapper_destroy (embed);
	}

	/* remove from list of embeds */
	all_embeds = g_list_remove (all_embeds, embed);

	/* from list of embeds in parent */
	if (window)
		window->embed_list = g_list_remove (window->embed_list, embed);

	/* if that's the last embed in parent window, destroy it */
	if (window && (g_list_length (window->embed_list) == 0))
	{
		window_close (window);
	}
	/* window_close will autosave the updated session, but if it isn't
	 * called, we should save ourselves */
	else
	{
		session_autosave ();
	}

	/* scrub and free the GaleonEmbed structure */
	if (embed->security_tooltip) g_free (embed->security_tooltip);
	memset (embed, 0, sizeof (GaleonEmbed));
	g_free (embed);
	embed = NULL;

	/* start the quit timeout */
	if (all_embeds == NULL)
		session_server_start_timeout ();
}

/** 
 * mozembed_drag_drop_cb:
 */
gboolean
mozembed_drag_drop_cb (GtkWidget * widget, GdkDragContext *context, 
		       gint x, gint y, GtkSelectionData *selection_data, 
		       guint info, guint time)
{
	g_warning ("unexpected mozembed_drag_drop_cb\n");
	return FALSE;
}

/**
 * mozembed_gesture_connect: connects gesture-related signals and grabs the
 * pointer
 */
static void
mozembed_gesture_connect (GaleonEmbed *embed, guint button)
{
	static GdkCursor *cursor = NULL;
	GtkWidget *mozembed;

	return_if_not_embed (embed);
	mozembed = GTK_WIDGET (embed->mozembed);

	/* init libstroke */
	stroke_init ();

	/* get a new cursor, if necessary */
	if (!cursor) cursor = gdk_cursor_new (GDK_PENCIL);

	/* set button */
	embed->gesture_button = button;

	/* attach signals */
	embed->gesture_motion_signal =
		gtk_signal_connect (GTK_OBJECT (mozembed),
			"motion_notify_event",
			GTK_SIGNAL_FUNC (mozembed_gesture_motion_cb), embed);
	embed->gesture_press_signal =
		gtk_signal_connect (GTK_OBJECT (mozembed),
			"button_press_event",
			GTK_SIGNAL_FUNC (mozembed_gesture_press_cb), embed);
	embed->gesture_release_signal =
		gtk_signal_connect (GTK_OBJECT (mozembed),
			"button_release_event",
			GTK_SIGNAL_FUNC (mozembed_gesture_release_cb), embed);

	/* grab the pointer */
	gtk_grab_add (mozembed);

	if (button == 2)
	{
		gdk_pointer_grab (mozembed->window, FALSE,
				  GDK_BUTTON2_MOTION_MASK |
				  GDK_BUTTON_PRESS_MASK |
				  GDK_BUTTON_RELEASE_MASK,
				  NULL, cursor, GDK_CURRENT_TIME);
	}
	else
	{
		gdk_pointer_grab (mozembed->window, FALSE,
				  GDK_BUTTON3_MOTION_MASK |
				  GDK_BUTTON_PRESS_MASK |
				  GDK_BUTTON_RELEASE_MASK,
				  NULL, cursor, GDK_CURRENT_TIME);
	}
}

/**
 * mozembed_gesture_disconnect: disconnects gesture-related signals and
 * ungrabs the pointer
 */
void
mozembed_gesture_disconnect (GaleonEmbed *embed)
{
	GtkWidget *mozembed;

	return_if_not_embed (embed);
	mozembed = GTK_WIDGET (embed->mozembed);

	/* ungrab the pointer if it's grabbed */
	if (gdk_pointer_is_grabbed ())
	{
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (mozembed);
	}

	/* disconnect signals */
	if (embed->gesture_motion_signal)
	{
		gtk_signal_disconnect (GTK_OBJECT (mozembed),
				       embed->gesture_motion_signal);
		embed->gesture_motion_signal = 0;
	}
	if (embed->gesture_press_signal)
	{
		gtk_signal_disconnect (GTK_OBJECT (mozembed),
				       embed->gesture_press_signal);
		embed->gesture_press_signal = 0;
	}
	if (embed->gesture_release_signal)
	{
		gtk_signal_disconnect (GTK_OBJECT (mozembed),
				       embed->gesture_release_signal);
		embed->gesture_release_signal = 0;
	}
}

/**
 * mozembed_gesture_motion_cb: handles gesture-related motion events in the
 * mozembed by passing samples to libstroke
 */
static gboolean
mozembed_gesture_motion_cb (GtkWidget *widget, GdkEventMotion *e,
			    GaleonEmbed *embed)
{
	/* record the location */
	stroke_record (e->x_root, e->y_root);
	return TRUE;
}

/**
 * mozembed_gesture_press_cb: handles button presses during gestures (to go
 * back in the history)
 */
static gboolean
mozembed_gesture_press_cb (GtkWidget *widget, GdkEventButton *e,
			   GaleonEmbed *embed)
{
	if (e->button != 1 || e->type != GDK_BUTTON_PRESS)
		return FALSE;

	return_val_if_not_embed (embed, TRUE);

	/* end gesture */
	mozembed_gesture_disconnect (embed);

	/* go back in history */
	gtk_moz_embed_go_back 
		(GTK_MOZ_EMBED (embed->mozembed));

	return TRUE;
}

/**
 * mozembed_gesture_release_cb: handles button releases during gestures,
 * and reads the gesture at the end and performs the appropriate action.
 * displays the context menu if the user did not move the mouse far enough
 * to complete a gesture.
 */
static gboolean
mozembed_gesture_release_cb (GtkWidget *widget, GdkEventButton *e,
			     GaleonEmbed *embed)
{
	char sequence[STROKE_MAX_SEQUENCE + 1];
	gint tabbed_mode;
	GaleonGestureAction action;
	gchar *homepage;
	GaleonEmbed *new = NULL;
	gboolean closing = FALSE;

	if (e->type != GDK_BUTTON_RELEASE)
		return FALSE;

	return_val_if_not_embed (embed, TRUE);

	/* make sure that this is the same mouse button that started the
	 * gesture */
	if (e->button != embed->gesture_button)
		return FALSE;

	/* ungrab and disconnect */
	mozembed_gesture_disconnect (embed);

	/* handle gestures */
	if (stroke_trans (sequence) == TRUE)
	{
		/* get the action */
		action = gestures_lookup (sequence);
		tabbed_mode = eel_gconf_get_boolean (CONF_TABS_TABBED);

		switch (action)
		{
			case UNRECOGNIZED:
				break;
			case NEW_WINDOW:
				/* if the gesture started over a link... */
				if (mouse_down_info->ctx.context ==
				    (CONTEXT_LINK | CONTEXT_DOCUMENT) ||
				    mouse_down_info->ctx.context ==
				    (CONTEXT_LINK | CONTEXT_DOCUMENT |
				     CONTEXT_IMAGE))
				{
					new = embed_create_after_embed (embed,
						TRUE,
						mouse_down_info->ctx.link,
						EMBED_CREATE_RAISE_WINDOW);
				}
				else
				{
					new = embed_create_after_embed (embed,
						TRUE, NULL,
						EMBED_CREATE_LOAD_DEFAULT_URL |
						EMBED_CREATE_FORCE_JUMP |
						EMBED_CREATE_RAISE_WINDOW);
				}
				g_assert (new != NULL);
				window_grab_location_focus
					(new->parent_window);
				break;
			case NEW_TAB:
				/* if the gesture started over a link... */
				if (mouse_down_info->ctx.context ==
				    (CONTEXT_LINK | CONTEXT_DOCUMENT) ||
				    mouse_down_info->ctx.context ==
				    (CONTEXT_LINK | CONTEXT_DOCUMENT |
				     CONTEXT_IMAGE))
				{
					new = embed_create_after_embed (embed,
						FALSE,
						mouse_down_info->ctx.link,
						EMBED_CREATE_RAISE_WINDOW);
				}
				else
				{
					new = embed_create_after_embed (embed,
						FALSE, NULL,
						EMBED_CREATE_LOAD_DEFAULT_URL |
						EMBED_CREATE_FORCE_JUMP |
						EMBED_CREATE_RAISE_WINDOW);
				}
				g_assert (new != NULL);
				window_grab_location_focus
					(new->parent_window);
				break;
			case RELOAD:
				embed_reload (embed,
					GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
				break;
			case RELOAD_BYPASS:
				embed_reload (embed,
					GTK_MOZ_EMBED_FLAG_RELOADBYPASSPROXYANDCACHE);
				break;
			case HOMEPAGE:
				homepage = eel_gconf_get_string 
						(CONF_GENERAL_HOMEPAGE);

				if (homepage != NULL &&
				    strlen (homepage) != 0)
				{
					embed_load_url (embed, homepage);
				}
				if (homepage != NULL) g_free (homepage);

				break;
			case CLONE_WINDOW:
				embed_create_after_embed (embed,
					TRUE, embed->location,
					EMBED_CREATE_RAISE_WINDOW);
				break;
			case CLONE_TAB:
				embed_create_after_embed (embed,
					FALSE, embed->location,
					EMBED_CREATE_RAISE_WINDOW);
				break;
			case UP:
				embed_go_up (embed, 0, 0);
				break;
			case CLOSE:
#if ENABLE_NAUTILUS_VIEW		       
				/* it seems that a nautilus view can't
				 * close a nautilus window */
				if (!EMBED_IN_NAUTILUS (embed))
#endif
				{
					embed_close (embed);
					closing = TRUE;
				}
				break;
			case BACK:
				gtk_moz_embed_go_back 
					(GTK_MOZ_EMBED (embed->mozembed));
				break;
			case FORWARD:
				gtk_moz_embed_go_forward 
					(GTK_MOZ_EMBED (embed->mozembed));
				break;
			case FULLSCREEN:
#if ENABLE_NAUTILUS_VIEW		       
				if (!EMBED_IN_NAUTILUS (embed))
#endif
					window_toggle_fullscreen_mode
						(embed->parent_window);
				break;
			case NEXT_TAB:
#if ENABLE_NAUTILUS_VIEW
				if (!EMBED_IN_NAUTILUS (embed))
#endif
				{
					GaleonWindow *window;
					gint current, last;  

					window = embed->parent_window;

					current =
						gtk_notebook_get_current_page (
							GTK_NOTEBOOK (
							window->notebook));
					last = g_list_length (GTK_NOTEBOOK (
						window->notebook)->children) -1;

					if (current == last)
						gtk_notebook_set_page (
							GTK_NOTEBOOK (
							window->notebook), 0);
					else
						gtk_notebook_next_page (
							GTK_NOTEBOOK (
							window->notebook));
				}
				break;
			case PREV_TAB:
#if ENABLE_NAUTILUS_VIEW
				if (!EMBED_IN_NAUTILUS (embed))
#endif
				{
					GaleonWindow *window;
					gint current, last;  

					window = embed->parent_window;

					current =
						gtk_notebook_get_current_page (
							GTK_NOTEBOOK (
							window->notebook));
					last = g_list_length (GTK_NOTEBOOK (
						window->notebook)->children) -1;

					if (current == 0)
						gtk_notebook_set_page (
							GTK_NOTEBOOK (
							window->notebook),
							last);
					else
						gtk_notebook_prev_page (
							GTK_NOTEBOOK (
							window->notebook));
				}
				break;
			case VIEW_SOURCE:
				embed_view_source (embed, TRUE,
						   !tabbed_mode);
				break;
			case PREV_LINK:
				embed_go_prev_link (embed);
				break;
			case NEXT_LINK:
				embed_go_next_link (embed);
				break;
			case CONTENTS_LINK:
				embed_go_contents_link (embed);
				break;
			case STOP:
				gtk_moz_embed_stop_load (
					GTK_MOZ_EMBED (embed->mozembed));
				break;
			default:
				g_warning ("unknown gesture action: %d\n",
					   action);
		}
	}
	/* if the gesture failed and this is the right button, display
	 * the context menu */
	else if (embed->gesture_button == 3)
	{
		context_show_menu (embed, &mouse_down_info->ctx, -1, 0);
	}

	if (!closing && !embed->being_closed && embed->wrapper)
	{
		mozilla_pop_target_document (embed);
	}

	return TRUE;
}
