/* CConfigFile.cpp */
#include "CConfigFile.h"

#include <qfile.h>
#include <iostream.h>
#include <qregexp.h>

#define MAX_LINE_LENGTH 1500

CConfigFile::CConfigFile(const QString& filename)
{
  setName(filename);
  // create default group
  setGroup("Default");
}

CConfigFile::CConfigFile()
{
}

CConfigFile::~CConfigFile()
{
  clear();
}

void CConfigFile::clear()
{
  ConfigMap::Iterator it;
    for(it = map.begin(); it != map.end(); ++it)
    delete it.data();
}

void CConfigFile::setName(const QString& filename)
{
// assign file name
  fname = filename;
}

void CConfigFile::setGroup(const QString& group)
{
  if (map.find(group) == map.end())
    {
      // new group
      currentgroup = new CConfigGroup(group);
      CHECK_PTR(currentgroup);

      map.insert(currentgroup->name(), currentgroup);
    }
  else currentgroup = map[group];
}

const QString& CConfigFile::group() const
{
  return currentgroup->name();
}

void CConfigFile::writeEntry(const QString& key, const QString& value)
{
//  qDebug("config write: key %s, value %s", key.latin1(), value.latin1());
  currentgroup->writeEntry(key, value);
}

void CConfigFile::writeEntry(const QString& key, int value)
{
  QString tmp;
  tmp.setNum(value);

  currentgroup->writeEntry(key, tmp);
}

void CConfigFile::writeEntry(const QString& key, double value)
{
  QString tmp;
  tmp.setNum(value);

  currentgroup->writeEntry(key, tmp);
}

void CConfigFile::writeBoolEntry( const QString& key, bool b_value)
{
  QString tmp;
  if(b_value)
    tmp="TRUE";
  else
    tmp= "FALSE";
  currentgroup->writeEntry(key, tmp);
}

void CConfigFile::writeNumEntry( const QString& key,  int number)
{
  QString tmp;
  tmp.setNum(number,10);
  currentgroup->writeEntry(key, tmp);
}
/////////////////////////////////////////////////////////

const QString& CConfigFile::readEntry(const QString& key, const QString& default_value)
{
 //   qDebug("config read: key %s, value %s", key.latin1(), default_value.latin1());
  return currentgroup->readEntry(key, default_value);
}

int CConfigFile::readEntry(const QString& key, int default_value)
{
    QString tmp;
    tmp = currentgroup->readEntry(key);
    if (tmp == QString::null)
        return default_value;
    else
        return tmp.toInt();
}

double CConfigFile::readEntry(const QString& key, double default_value)
{
    QString tmp;
    tmp = currentgroup->readEntry(key);
    if (tmp == QString::null)
        return default_value;
    else
        return tmp.toDouble();
}

bool CConfigFile::readBoolEntry(const QString& key, bool b_default_value)
{
    QString tmp;
    tmp = currentgroup->readEntry(key);
    if(tmp=="TRUE")
        return true;
    else if (tmp=="FALSE")
        return false;
    else
        return b_default_value;
}

int CConfigFile::readNumEntry(const QString& key, int idefault)
{
    QString tmp;
    tmp = currentgroup->readEntry(key);
    bool ok;
    if (tmp == QString::null)
        return idefault;
    else
        return tmp.toInt(&ok,10);
}

void CConfigFile::write()
{
  if (fname.isEmpty()) return;
  // write to file
  QFile file(name());
  if( file.open(IO_WriteOnly)) {
      ConfigMap::Iterator it;
      for(it = map.begin(); it != map.end(); ++it)
          if (!it.data()->isEmpty())
          {  // write group name
              file.writeBlock("[" + it.key() + "]\n", 3 + it.key().length());
                // write group data
              GroupMap::Iterator git;
              for (git = it.data()->map.begin(); git != it.data()->map.end(); ++git)
                  file.writeBlock(git.key() + " = " + git.data() + "\n",
                                  git.key().length() + 3 + git.data().length() + 1);
          }
      file.close();
//      qDebug("CConfigFile- Writing to config file.");
  }  else
  printf("Opening config file failed.\n");
}

void CConfigFile::read()
{
    if (fname.isEmpty()) return;
    QFile file(name());
    file.open(IO_ReadOnly);
    QString line;
    while(file.readLine(line, MAX_LINE_LENGTH) != -1) {
          // is it a group name?
        int index = line.find("[");
        if (index != -1) {
              // get group name
            int index2 = line.findRev("]");
            setGroup(line.mid(index + 1, index2 - index - 1));
        }// is it a key = value string
        if (index == -1)  {
              // simplify white space
            line = line.simplifyWhiteSpace();
            index = line.find("=");
            if (index != -1) {
                  // get key and value
                QString key = line.left(index - 1);
                QString value = line.right(line.length() - (index + 2));
                writeEntry(key, value);
            }
        }
    }
    file.close();
      // switch to default group
    setGroup("Default");
}

QStringList CConfigFile::groupList()
{
  QStringList tmp;
  ConfigMap::Iterator it;
  for(it = map.begin(); it != map.end(); ++it)
    tmp.append(it.key());
  return tmp;
}

QFont CConfigFile::readFontEntry( const QString& key, const QFont* pDefault )
{
    QFont fnt;
    fnt = currentgroup->readFontEntry(key, pDefault);
    qDebug("Reading font config\n");
    return fnt;
}

void CConfigFile::writeFontEntry( const QString& key, const QFont& rFont )
{
    currentgroup->writeFontEntry(key, rFont);
}

