// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "decoder_mp3.h"
#include "constants.h"
#include "buffer.h"
#include "output.h"

#include <qobject.h>
#include <qiodevice.h>

#include <math.h>

const int DecoderMP3::maxDecodeRetries = 64;
const int DecoderMP3::maxFrameSize     = 1441;
const int DecoderMP3::maxFrameCheck    = 3;
const int DecoderMP3::initialFrameSize = maxFrameCheck * maxFrameSize;

static int extractI4(unsigned char *buf)
{
    int x;
    // big endian extract

    x = buf[0];
    x <<= 8;
    x |= buf[1];
    x <<= 8;
    x |= buf[2];
    x <<= 8;
    x |= buf[3];

    return x;
}


DecoderMP3::DecoderMP3(DecoderFactory *d, QIODevice *i, Output *o)
    : Decoder(d, i, o)
{
    done = finish = almost_done = inited = user_stop = derror = FALSE;
    stat = 0;
    bks = 0;
    seekTime = -1.0;
    totalTime = msPerFrame = 0.0;
    input_buf = 0;
    input_bytes = input_at = 0;
    input_trigger = 4096;
    output_buf = 0;
    output_bytes = output_at = 0;

    mpeg_init(&mpeg, 1);
    fb = 0;
    br = vr = ly = bad_syncs = 0;
    xingheader = 0;
}

DecoderMP3::~DecoderMP3() {
    deinit();

    if (input_buf) {
	delete [] input_buf;
	input_buf = 0;
    }

    if (output_buf) {
	delete [] output_buf;
	output_buf = 0;
    }

    mpeg_cleanup(&mpeg);
}

bool DecoderMP3::nextFrame()
{
    unsigned char *ibufferbase;
    unsigned char *ibuffer;
    int i;

    do {
	fill();
	ibufferbase = input_buf + input_at;

	if (almost_done && input_bytes < fb)
	    return FALSE; // end of stream

	ibuffer = ibufferbase + 1;
	for (i = 0;
	     i < DecoderMP3::maxFrameSize - 1 &&
		 ! (*ibuffer == 0xff &&
		    ((*(ibuffer + 1) & 0xf0) == 0xf0 ||
		     (*(ibuffer + 1) & 0xf0) == 0xe0));
	     ibuffer++, i++);

	input_at += i + 1;
	input_bytes -= i + 1;

	if (i != 0 && i < DecoderMP3::maxFrameSize - 1)
	    break;
    } while (true);

    return TRUE;
}

bool DecoderMP3::findHeader()
{
    int retries, frame, offset, lastsr, lastoption, lastid, lastmode;
    unsigned int forward;

    offset = lastsr = lastoption = lastid = lastmode = 0;

    for (retries = 0; retries < DecoderMP3::maxDecodeRetries; retries++) {
	fill();

	for (frame = 0, offset = 0; frame < maxFrameCheck; frame++) {
	    fb = head_info3(input_buf + input_at + offset,
			    DecoderMP3::initialFrameSize - offset,
			    &head, &br, &forward);

	    if (fb > 0 && frame == 0) {
		offset += fb + forward + head.pad;

		lastsr = head.sr_index;
		lastoption = head.option;
		lastid = head.id;
		lastmode = head.mode;

		continue;
	    }

	    if (fb > 0 && fb < DecoderMP3::maxFrameSize &&
		(head.option == 1 || head.option == 2) &&
		lastsr     == head.sr_index &&
		lastoption == head.option &&
		lastid     == head.id &&
		lastmode   == head.mode) {
		offset += fb + forward + head.pad;

		lastsr = head.sr_index;
		lastoption = head.option;
		lastid = head.id;
		lastmode = head.mode;

		if (frame < DecoderMP3::maxFrameCheck - 1)
		    continue;

		if (xingheader) {
		    delete [] xingheader->toc;
		    delete xingheader;
		    xingheader = 0;
		}

		xingheader = new XHEADDATA;
		xingheader->toc = new unsigned char[100];
		if (! findXingHeader()) {
		    delete xingheader->toc;
		    delete xingheader;
		    xingheader = 0;
		}

		continue;
	    }

	    if (! nextFrame())
		return FALSE;

	    break;
	}

	if (frame == DecoderMP3::maxFrameCheck)
	    return TRUE;
    }

    return FALSE;
}

bool DecoderMP3::findXingHeader()
{
    int i, head_flags;
    int h_id, h_mode, h_sr_index;
    static int sr_table[4] = { 44100, 48000, 32000, 99999 };
    unsigned char *buf = input_buf + input_at;

    // get Xing header data
    xingheader->flags = 0;     // clear to null incase fail

    // get selected MPEG header data
    h_id       = (buf[1] >> 3) & 1;
    h_sr_index = (buf[2] >> 2) & 3;
    h_mode     = (buf[3] >> 6) & 3;

    // determine offset of header
    if( h_id ) {
        // mpeg1
        if (h_mode != 3)
	    buf+=(32 + 4);
        else
	    buf+=(17 + 4);
    } else {
	// mpeg2
        if (h_mode != 3)
	    buf += (17 + 4);
        else
	    buf += (9 + 4);
    }

    char foo[5];
    foo[0] = buf[0];
    foo[1] = buf[1];
    foo[2] = buf[2];
    foo[3] = buf[3];
    foo[4] = 0;

    if (buf[0] != 'X') return false; // fail
    if (buf[1] != 'i') return false; // header not found
    if (buf[2] != 'n') return false;
    if (buf[3] != 'g') return false;
    buf += 4;

    xingheader->h_id = h_id;
    xingheader->samprate = sr_table[h_sr_index];
    if (h_id == 0)
	xingheader->samprate >>= 1;

    // get flags
    head_flags = xingheader->flags = extractI4(buf);
    buf += 4;

    if (head_flags & FRAMES_FLAG) {
	xingheader->frames   = extractI4(buf);
	buf += 4;
    }
    if (head_flags & BYTES_FLAG) {
	xingheader->bytes = extractI4(buf);
	buf += 4;
    }

    if (head_flags & TOC_FLAG) {
        if (xingheader->toc) {
            for(i=0; i<100; i++)
		xingheader->toc[i] = buf[i];
        }
        buf += 100;
    }

    xingheader->vbr_scale = -1;
    if (head_flags & VBR_SCALE_FLAG)  {
	xingheader->vbr_scale = extractI4(buf);
	buf += 4;
    }

    return true;       // success
}

void DecoderMP3::fill() {
    if (seekTime >= 0.0) {
	long seek_frame = long((seekTime * 1000) / msPerFrame);
	long seek_pos;

	if (xingheader)
	    seek_pos = seekPoint(seek_frame, xingheader->frames);
	else
	    seek_pos = seek_frame * fb;

	input()->at(seek_pos);
	output_size = long(seekTime) *
		      long(decinfo.samprate * decinfo.channels * decinfo.bits / 2);
	input_at = input_bytes = 0;
	almost_done = false;
    }

    if (input_bytes < input_trigger && ! almost_done) {
        if (input_bytes)
            memmove(input_buf, input_buf + input_at, input_bytes);

	int len = input()->readBlock((char *) (input_buf + input_bytes),
				     globalBufferSize - input_bytes);
        if (len >= 0) {
            input_bytes += len;
            input_at = 0;
        }

        if (input()->atEnd())
            almost_done = TRUE;
    }

    if (seekTime >= 0.0) {
	seekTime = -1.0;
	nextFrame();
    }
}

void DecoderMP3::flush(bool final)
{
    long min = final ? 0 : bks;

    while ((! done && ! finish && ! derror) && output_bytes > min) {
	output()->recycler()->mutex()->lock();

	while ((! done && ! finish && ! derror) && output()->recycler()->full()) {
	    mutex()->unlock();

	    output()->recycler()->cond()->wait(output()->recycler()->mutex());

	    mutex()->lock();
	    done = user_stop;
	}

	if (user_stop || finish || derror) {
	    inited = FALSE;
	    done = TRUE;
	} else {
	    long sz = output_bytes < bks ? output_bytes : bks;
	    Buffer *b = output()->recycler()->get();

	    memcpy(b->data, output_buf, sz);
	    if (sz != bks)
		memset(b->data + sz, 0, bks - sz);

	    b->nbytes = bks;
	    b->rate = br;
	    output_size += b->nbytes;
	    output()->recycler()->add();

	    output_bytes -= sz;
	    memmove(output_buf, output_buf + sz, output_bytes);
	    output_at = output_bytes;
	}

	if (output()->recycler()->full())
	    output()->recycler()->cond()->wakeOne();

	output()->recycler()->mutex()->unlock();
    }
}

void DecoderMP3::stop() {
    user_stop = 1;
}

bool DecoderMP3::initialize()
{
    bks = blockSize();

    done = finish = derror = FALSE;
    inited = almost_done = user_stop = FALSE;
    br = vr = ly = fb = bad_syncs = 0;
    output_size = 0;
    seekTime = -1.0;
    totalTime = msPerFrame = 0.0;

    if (! input()) {
	error("DecoderMP3: cannot initialize.  No input.");
	return FALSE;
    }

    if (! input()->isOpen()) {
	if (! input()->open(IO_ReadOnly)) {
	    error("DecoderMP3: Failed to open input.  Error " +
		  QString::number(input()->status()) + ".");
	    return FALSE;
	}
    }

    mpeg_init(&mpeg, 0);

    if (! input_buf)
	input_buf = new unsigned char[globalBufferSize];
    input_at = 0;
    input_bytes = 0;

    if (! output_buf)
	output_buf = new unsigned char[globalBufferSize];
    output_at = 0;
    output_bytes = 0;

    if (! findHeader()) {
	error("DecoderMP3: Cannot find a valid MPEG header.");
	return FALSE;
    }

    br /= 1000;
    vr = (head.id) ? 10 : (head.sync & 1 == 0) ? 25 : 20;
    switch(head.option) { case 1: ly++; case 2: ly++; case 3: ly++; }

    if (! audio_decode_init(&mpeg, &head, fb, 0, 0, 0, 24000)) {
	error("DecoderMP3: The decoding engine failed to initalize.");
	return FALSE;
    }

    audio_decode_info(&mpeg, &decinfo);

    calcLength();

    if (output())
	output()->configure(decinfo.samprate, decinfo.channels,
			    decinfo.bits, br);

    inited = TRUE;
    return TRUE;
}

double DecoderMP3::lengthInSeconds()
{
    if (! inited)
	return 0;

    return totalTime;
}

void DecoderMP3::calcLength()
{
    if (! input() || ! input()->isDirectAccess())
	return;

    static int sample_rate_table[8] = {
	22050L, 24000L, 16000L, 1L, 44100L, 48000L, 32000L, 1L
    };

    double totalFrames;
    int sampleRate, sampleRateIndex;
    long filesize = input()->size();

    sampleRateIndex = 4 * head.id + head.sr_index;
    sampleRate = sample_rate_table[sampleRateIndex];

    if (head.id == 1)
	msPerFrame = double(1152 * 1000) / double(sampleRate);
    else
	msPerFrame = double( 576 * 1000) / double(sampleRate);

    if (filesize > 0) {
	if (xingheader)
	    totalFrames = double(xingheader->frames);
	else
	    totalFrames = double(filesize) / double(fb);

	totalTime = long(totalFrames * msPerFrame / double(1000.0));
    }

    totalTime = totalTime < 0.0 ? 0.0 : totalTime;
}

long DecoderMP3::seekPoint(long frame, long count)
{
    double percent = double(frame) * 100.0 / double(count);
    double fa, fb, fx;
    long a, seekpoint;

    if (percent < 0.0)
	percent = 0.0;
    if (percent > 100.0)
	percent = 100.0;

    a = int(percent);
    if (a > 99)
	a = 99;

    fa = xingheader->toc[a];
    if (a < 99)
        fb = xingheader->toc[a+1];
    else
        fb = 256.0;

    fx = fa + (fb - fa) * (percent - a);

    seekpoint = long((1.0 / 256.0) * fx * input()->size());
    return seekpoint;
}

void DecoderMP3::seek(double pos)
{
    seekTime = pos;
}

void DecoderMP3::deinit()
{
    if (input())
	input()->close();

    done = finish = FALSE;
    inited = almost_done = user_stop = FALSE;
    br = vr = ly = fb = bad_syncs = 0;
    output_size = 0;
    setInput(0);
    setOutput(0);

    if (xingheader) {
	delete [] xingheader->toc;
	delete xingheader;
	xingheader = 0;
    }
}

void DecoderMP3::setEQEnabled(bool enable)
{
    mpeg.eq->enableEQ = enable;
}

void DecoderMP3::setEQ(const EqPreset &preset)
{
    Decoder::setEQ(preset);

    static unsigned char const map[32] = {
	0, 1, 2, 3, 4, 5, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8,
	8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9
    };
    double preamp = preset.preamp();
    for (int i = 0; i < 32; i++) {
	double val = preamp + preset.band(map[i]);
	mpeg.eq->equalizer[i] =
	    pow(2., (val > 18. ? 18. : val < -18. ? -18. : val) / 20.);
    }
    mpeg.eq->EQ_gain_adjust = 0.0;
}

void DecoderMP3::run() {
    mutex()->lock();

    if (! inited) {
	mutex()->unlock();

	return;
    }

    stat = DecoderEvent::Decoding;

    mutex()->unlock();

    {
	DecoderEvent e((DecoderEvent::Type) stat);
	dispatch(e);
    }

    int newbr;
    long len;
    long left, newfb;
    unsigned char *ibuffer = 0;

    while (! done && ! finish && ! derror) {
	mutex()->lock();

	// decode
	fill();
	ibuffer = input_buf + input_at;
	left = input_bytes;

	if (left >= fb) {
	    x = audio_decode(&mpeg, ibuffer, (short *) (output_buf + output_at));
	    if (x.in_bytes > 0) {
		// decoded data len
		input_at += x.in_bytes;
		input_bytes -= x.in_bytes;
		ibuffer = input_buf + input_at;
		left = input_bytes;

		len = x.out_bytes;

		newbr = br;
		newfb = head_info2(ibuffer, left, &head, &newbr);
		if (newfb && newfb != fb) {
		    fb = newfb;
		    br = newbr / 1000;
		}

		bad_syncs = 0;
	    } else {
		// decoder error
		len = -1;
	    }
	} else {
	    // EOS
	    len = 0;
	}

	// put data into buffer
	if (len > 0) {
	    output_at += len;
	    output_bytes += len;

	    if (output())
		flush();
	} else if (len == 0) {
	    // end of stream
	    flush(TRUE);

	    if (output()) {
		output()->recycler()->mutex()->lock();
		while (! output()->recycler()->empty() && ! user_stop) {
		    output()->recycler()->cond()->wakeOne();
		    mutex()->unlock();
		    output()->recycler()->cond()->
			wait(output()->recycler()->mutex());
		    mutex()->lock();
		}
		output()->recycler()->mutex()->unlock();
	    }

	    done = TRUE;
	    if (! user_stop)
		finish = TRUE;
	} else {
	    // error in stream... ie. lost sync... try to get it back
	    if (bad_syncs > 10) {
		error("DecoderMP3: Excessive sync loss. Giving up on this "
		      "stream.");
		derror = TRUE;
	    } else {
		if (! nextFrame()) {
		    error("DecoderMP3: Cannot regain sync in stream. Giving up.");
		    derror = TRUE;
		}
		bad_syncs++;
	    }
	}

	mutex()->unlock();
    }

    mutex()->lock();

    if (finish)
	stat = DecoderEvent::Finished;
    else if (user_stop || derror)
	stat = DecoderEvent::Stopped;

    mutex()->unlock();

    if (! derror) {
	DecoderEvent e((DecoderEvent::Type) stat);
	dispatch(e);
    }

    deinit();
}


// DecoderMP3Factory

bool DecoderMP3Factory::supports(const QString &source) const
{
    return (source.right(extension().length()).lower() == extension());
}

const QString &DecoderMP3Factory::extension() const
{
    static QString ext(".mp3");
    return ext;
}

const QString &DecoderMP3Factory::description() const
{
    static QString desc("MPEG Layer 1/2/3 Audio (XingTech decoder)");
    return desc;
}

Decoder *DecoderMP3Factory::create(QIODevice *input,
				   Output *output,
				   bool deletable)
{
    if (deletable)
	return new DecoderMP3(this, input, output);

    static DecoderMP3 *decoder = 0;

    if (! decoder) {
	decoder = new DecoderMP3(this, input, output);
    } else {
	decoder->setInput(input);
	decoder->setOutput(output);
    }

    return decoder;
}
