#include "song.h"
#include "collection.h"
#include "collectionview.h"
#include "group.h"
#include "mediaview.h"
#include "organizer.h"

#include <qdom.h>
#include <qpainter.h>
#include <qregexp.h>

#if defined(Q_OS_UNIX)
#  include <openssl/md5.h>
#endif


int Song::RTTI = 141832;

Song::Song(Group *parent, const QString &t, const QString &l, const QString &e, int pr)
    : QCheckListItem(parent, t, CheckBox),
      group(parent), coltext(t), loc(l), eq(e), pri(pr), len(-1.0), curr(false)
{
    group->addSong();
    calcSig();
}

Song::Song(Group *parent, const QString &t, const QString &l,
	   const QString &e, const Signature &s, int pr)
    : QCheckListItem(parent, t, CheckBox),
      group(parent), coltext(t), loc(l), eq(e), pri(pr), len(-1.0), curr(false), sig(s)
{
    group->addSong();
    calcSig();
}

void Song::setup()
{
    QListViewItem::setup();
    setExpandable(false);
    setSelectable(true);
    setPixmap(0, MediaView::instance()->iconset.pixmap(QIconSet::Small,
						       (curr ? QIconSet::Active :
							QIconSet::Normal)));
}

Song::~Song()
{
    if (Organizer::instance()->isDestructing())
	return;

    group->remSong();

    // remove song from internal lists in mediaview
    MediaView::instance()->lengthlist.remove(this);

    // and collectionview
    QListViewItem *item = CollectionView::instance()->firstChild();
    while (item) {
	if (item->rtti() == Collection::RTTI) {
	    Collection *c = (Collection *) item;
	    c->remove(this);
	    if (c->currentSong == this)
		c->setCurrentSong(0, false);
	}

	item = item->nextSibling();
    }
}

void Song::setCurrent(bool c)
{
    curr = c;
    setPixmap(0,
	      MediaView::instance()->iconset.pixmap(QIconSet::Small,
						    (curr ? QIconSet::Active :
						     QIconSet::Normal)));
}

void Song::paintCell(QPainter *p, const QColorGroup &cg, int a, int b, int c)
{
    if (curr) {
	QFont font(p->font());
	font.setBold(true);
	p->setFont(font);
    }
    QCheckListItem::paintCell(p, cg, a, b, c);
}

void Song::setLocation(const QString &l)
{
    loc = l;
    setPriority(pri);
}

void Song::setText(int column, const QString &text)
{
    if (column != 0 || text.isEmpty() || text.isNull())
	return;
    coltext = text;
    ttl = QString::null;
    MediaView::instance()->setModified(true);
}

QString Song::title() const
{
    if (ttl.isNull()) {
	Song *that = (Song *) this;
	QListViewItem *item = that->parent();
	while (item) {
	    that->ttl = item->text(0) + " - " + that->ttl;
	    item = item->parent();
	}
	that->ttl += that->coltext;
    }

    return ttl;
}

QString Song::text(int column) const
{
    QString ret;
    switch (column) {
    case 0:
	ret = coltext;
	break;

    case 1:
	if (len > 0.0) {
	    long m, s;
	    m = long(len) / 60;
	    s = long(len) % 60;
	    ret.sprintf("%02ld:%02ld", m, s);
	} else
	    ret = QString("--:--");
	break;

    case 2:
	if (isOn())
	    ret = QString::number(pri);
	else
	    ret = QString("-");
	break;

    case 3:
	ret = eq;
	break;

    default:
	break;
    }

    return ret;
}

void Song::setOn(bool on)
{
    if (isOn() == on)
	return;

    if (! CollectionView::instance()->currentCollection()) {
	// create collection?
	return;
    }

    QCheckListItem::setOn(on);

    group->enableSong(on);

    Collection *c = CollectionView::instance()->currentCollection();
    if (c->loading)
	return;

    c->remove(this);
    if (on)
	c->add(this, pri);

    CollectionView::instance()->setModified(true);
}

void Song::setPriority(int p)
{
    if (! CollectionView::instance()->currentCollection()) {
	// create collection?
	return;
    }

    if (! isOn())
	return;

    pri = p;

    Collection *c = CollectionView::instance()->currentCollection();
    if (c->loading)
	return;

    c->remove(this);
    c->add(this, pri);

    CollectionView::instance()->setModified(true);
}

void Song::setEqualizer(const QString &e)
{
     eq = e;
     MediaView::instance()->setModified(true);
     repaint();
}

void Song::calcSig()
{
    if (sig.isValid())
	return;

#if defined(Q_OS_UNIX)
    // use openssl API to get the MD5 sum of the location...
    MD5_CTX ctx;
    MD5_Init(&ctx);
    MD5_Update(&ctx, loc.unicode(), loc.length() * sizeof(QChar));
    MD5_Final((unsigned char *) &sig, &ctx);
#endif
}

int Song::rtti() const
{
    return Song::RTTI;
}

void Song::xmlDescription(QTextStream &stream) const {
    stream << "<song>";
    QString s = coltext;
    s = (s.replace(QRegExp("&"), "&amp;").
	 replace(QRegExp("<"), "&lt;").
	 replace(QRegExp(">"), "&gt;"));
    stream << "<name>" << s << "</name>";
    s = loc;
    s = (s.replace(QRegExp("&"), "&amp;").
	 replace(QRegExp("<"), "&lt;").
	 replace(QRegExp(">"), "&gt;"));
    stream << "<path>" << s << "</path>";
    stream << "<length>" << QString::number(len) << "</length>";
    stream << "<equalizer>" << eq << "</equalizer>";
    stream << "<signature>" << sig.toString() << "</signature>";
    stream << "</song>" << endl;
}

void Song::create(Group *parent, const QDomElement &element)
{
    if (! parent)
	return;

    QString name, path, eq, s;
    Song::Signature sig;
    bool enabled = false;
    int priority = 1;
    double length = -1.0;

    if (element.tagName() == "song") {
	QDomElement e = element.firstChild().toElement();
	while (! e.isNull()) {
	    if (e.tagName() == "name")
		name = e.firstChild().toText().data();
	    else if (e.tagName() == "path")
		path = e.firstChild().toText().data();
	    else if (e.tagName() == "length")
		length = e.firstChild().toText().data().toDouble();
	    else if (e.tagName() == "equalizer")
		eq = e.firstChild().toText().data();
	    else if (e.tagName() == "signature") {
		s = e.firstChild().toText().data();
		sig.fromString(s);
	    }

	    e = e.nextSibling().toElement();
	}

	if (! name.isNull() && ! path.isNull() && priority > 0) {
	    if (eq.isNull())
		eq = "Default";
	    Song *song;
	    song = new Song(parent, name, path, eq, sig, priority);
	    song->setOn(enabled);

	    if (length == -1.0)
		MediaView::instance()->lengthlist.append(song);
	    else
		song->setLength(length);
	}
    }
}
