/*
 * sdl.cxx            
 *                           
 *This file contains the class heirachy for 1)creating windows under sdl.
 *                                          2)displaying data on those windows
 *
 * Copyright (c) 1999-2000 Indranet Technologies ltd
 * 
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Open H323
 * 
 * Contributor(s): Author Derek J Smithies (derek@indranet.co.nz)
 *
 * $Log: sdlvid.cxx,v $
 * Revision 1.4  2001/10/23 02:21:39  dereks
 * Initial release CU30 video codec.
 * Add --videotest option, to display raw video, but not invoke a call.
 *
 * Revision 1.3  2001/08/02 05:13:39  dereks
 * Added PTRACE statments for debugging.
 *
 * Revision 1.2  2001/05/25 01:14:44  dereks
 * Alter SetFrameSize & OpenWindo to use unsigned variables. Change type of
 * other variables to eliminate compiler warnings.
 *
 * Revision 1.1  2001/03/07 01:47:45  dereks
 * Initial release of SDL (Simple DirectMedia Layer, a cross-platform multimedia library),
 * a video library code.
 *
 *
 *
 */

#include <ptlib.h>

#ifdef HAS_SDL
#include "sdlvid.h"


PMutex          SDLVideoDevice::videoMutex;
SDL_Surface    *SDLVideoDevice::screen=NULL;
SDL_Overlay    *SDLVideoDevice::overlay[2];
SDL_Rect        SDLVideoDevice::displayPosn[2];
unsigned        SDLVideoDevice::wide[2];
unsigned        SDLVideoDevice::high[2];


void WriteOutDisplay(void);
void WriteOutDisplay(void)
{
  return;
  
  cout <<"SDL\t 0 x,y="<<SDLVideoDevice::displayPosn[0].x<<" "
       <<SDLVideoDevice::displayPosn[0].y << endl;
  
  cout <<"SDL\t 0 w,h="<<SDLVideoDevice::displayPosn[0].w<<" "
       <<SDLVideoDevice::displayPosn[0].h << endl;
  
  cout <<"SDL\t 1 x,y="<<SDLVideoDevice::displayPosn[1].x<<" "
       <<SDLVideoDevice::displayPosn[1].y << endl;
  
  cout <<"SDL\t 1 w,h="<<SDLVideoDevice::displayPosn[1].w<<" "
       <<SDLVideoDevice::displayPosn[1].h << endl;  
}





SDLVideoDevice::SDLVideoDevice(const PString & _remoteName,BOOL _isEncoding, 
                                                 BOOL _videoPIP)
{
       videoMutex.Wait();
       PTRACE(3,"Generic\t SDL video constructor start.");
       
       remoteName = _remoteName; 
       forceDepth = 0;

       isEncoding= _isEncoding;
       videoPIP  = _videoPIP;

       width=0;
       height=0;
       
       overlay[xIndex()]= NULL;


       videoMutex.Signal();
}

SDLVideoDevice::~SDLVideoDevice()
{ 
  Close();
}

BOOL SDLVideoDevice::Close()
{
  PWaitAndSignal vMutex(videoMutex);
  PTRACE(3,"SDL\t Close Start");
  CloseWindow();
  PTRACE(3,"SDL\t Close DONE");
  return TRUE;
}

BOOL SDLVideoDevice::SetFrameSize (unsigned _width, unsigned _height)
{
  PWaitAndSignal vMutex(videoMutex);
  PTRACE(3,"SDL\t SetFrameSize Start for the "<<DirectionStr()<<" window" );
  if ((width != _width) || (height != _height)) {
    PTRACE(3,"SDL Requested internal size of "<<_width<<"x"<<_height);
    PTRACE(3,"SDL did not match internal size of "<<width<<"x"<<height);
    
    if (overlay[xIndex()]!=NULL)
      SDL_FreeYUVOverlay(overlay[xIndex()]);
    
    if((_width!=0)&&(_height!=0)) {
        // save parameters
      wide[xIndex()] = _width;
      high[xIndex()] = _height;
      InitDisplayPosn();
      if(OpenScreen()) {        
        CreateOverlay();
      }      
    }    
  }
  PTRACE(3,"SDL SetSize DONE for the "<<DirectionStr()<<" window" );
  return TRUE;
}


void SDLVideoDevice::InitializeStaticVariables(void)
{
  for(int i=0;i<2;i++) {
    high[i]=0;
    wide[i]=0;
    overlay[i]=NULL;
    displayPosn[i].x=0;
    displayPosn[i].y=0;
    displayPosn[i].w=0;
    displayPosn[i].h=0;
  }  
}



 // open screen for SDL to display images on.
BOOL SDLVideoDevice::OpenScreen()
{
   if (!IsOpen()) {
      /* Initialize the SDL library */
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
      PTRACE(0,"Couldn't initialize SDL: "<<SDL_GetError());
      return FALSE;      
    }
   }
   
   /* Get the required width and height*/
   int w = displayPosn[0].w + displayPosn[1].w;
   int h = PMAX(displayPosn[0].h,displayPosn[1].h);

   PTRACE(3,"SDL Open SDL Screen of size "<<w<<"x"<<h);
    
   if ((w==0) || (h==0)) {
     PTRACE(3,"SDL Quit, refuse to Open SDL Screen of size "<<w<<"x"<<h);
     SDL_Quit();      
     return FALSE;     //refuse to open SDL screen containg 0 pixels.
   }
   
   screen = SDL_SetVideoMode(w, h, 0,
                              SDL_SWSURFACE | SDL_RESIZABLE );

   if ( screen == NULL ) {
     PTRACE(0,"Could not open screen to display window: "<<SDL_GetError());
     SDL_Quit();
     return FALSE;
   }

   PTRACE(3,"SDL successfully opened a SDL screen of size "<<w<<"x"<<h);
   return TRUE;
}

void SDLVideoDevice::InitDisplayPosn()
{  
  if(videoPIP) {
    displayPosn[encodeIndex()].x = (short)wide[remoteIndex()]<<1;
    displayPosn[encodeIndex()].y = (short)(high[remoteIndex()]<<1)-(high[encodeIndex()]>>1);
    displayPosn[encodeIndex()].w = (short)wide[encodeIndex()]>>1;
    displayPosn[encodeIndex()].h = (short)high[encodeIndex()]>>1;    

    displayPosn[remoteIndex()].x = 0;
    displayPosn[remoteIndex()].y = 0;
    displayPosn[remoteIndex()].w = wide[remoteIndex()]<<1;
    displayPosn[remoteIndex()].h = high[remoteIndex()]<<1;    
    WriteOutDisplay();    
  } else {
    displayPosn[encodeIndex()].x = wide[remoteIndex()];
    displayPosn[encodeIndex()].y = 0;
    displayPosn[encodeIndex()].w = wide[encodeIndex()];
    displayPosn[encodeIndex()].h = high[encodeIndex()];    

    displayPosn[remoteIndex()].x = 0;
    displayPosn[remoteIndex()].y = 0;
    displayPosn[remoteIndex()].w = wide[remoteIndex()];
    displayPosn[remoteIndex()].h = high[remoteIndex()];
    WriteOutDisplay();    
  }
}


BOOL  SDLVideoDevice::CreateOverlay()
{
  unsigned w = wide[xIndex()];
  unsigned h = high[xIndex()];
  
 if((overlay[xIndex()]!=NULL) && (w==width) && (h==height)   )
   return TRUE;

 if( (w==0) || (h==0)) {
   if(overlay[xIndex()]!=NULL)
     SDL_FreeYUVOverlay(overlay[xIndex()]);
   overlay[xIndex()]= NULL;
   return TRUE;
 }
 
 PTRACE(3,"SDL\t YUV420P overlay ("<<w<<"x"<<h<<") Screen:"<<::hex<<(uint)screen<<::dec);


 overlay[xIndex()] = SDL_CreateYUVOverlay(w, h, SDL_IYUV_OVERLAY, screen);
  if( overlay[xIndex()]==NULL ) {
      PTRACE(0,"Could not open overlay to display window: "<<SDL_GetError());
      return FALSE;
    }

  width  = w;
  height = h;
  
  PTRACE(3,"SDL\t Successfully create the YUV overlay");
  return TRUE;
}


void SDLVideoDevice::CloseWindow()
{
  PTRACE(3,"SDL\t Close the "<<DirectionStr()<<" window.");
  if(!(width||height)) {
    PTRACE(3,"SDL\t Attempting to close 0x0 sized window. So do nothing");
    return;
  }

  if(overlay[xIndex()] != NULL) {
    PTRACE(3,"SDL\t close the overlay for "<<DirectionStr()<<" window.");
    SDL_FreeYUVOverlay(overlay[xIndex()]);
    overlay[xIndex()] = NULL;    
  }
  
  
  if(overlay[1-xIndex()] == NULL)
    CloseScreen();

  width=0;
  height=0;
}

void SDLVideoDevice::CloseScreen(void)
{ 
  if(screen!=NULL) {    
    PTRACE(3,"SDL\t close the SDL display screen.");
    SDL_FreeSurface(screen); //screen!=NULL, as IsOpen() returned TRUE.
    screen = NULL;
    SDL_Quit();
  }  
}


  

void SDLVideoDevice::ProcessSdlEvents(void) 
{

  if(!IsOpen()) {
    PTRACE(6,"SDL\t Screen not open, so dont process events");
    return;
  }

  SDL_Event event;  
  while ( SDL_PollEvent(&event)) {
    if (event.type == SDL_QUIT) {//User selected cross
      
      PTRACE(3,"SDL\t user selected cross on window, close window");
      if(overlay[0])
        SDL_FreeYUVOverlay(overlay[0]);
      if(overlay[1])
        SDL_FreeYUVOverlay(overlay[1]);
      overlay[0]= NULL;
      overlay[1]= NULL;
      CloseScreen();
      return;
    }
    if (event.type == SDL_VIDEORESIZE){
      PTRACE(3,"SDL\t Resize window to "<<event.resize.w<<"x"<<event.resize.h);
      if(videoPIP) {
        PTRACE(3,"SDL\t videoPIP is on for resize");      
        displayPosn[encodeIndex()].w = event.resize.w/5;
        displayPosn[encodeIndex()].h = event.resize.h/4;
        displayPosn[encodeIndex()].x = event.resize.w*4/5;        
        displayPosn[encodeIndex()].y = event.resize.h*3/4;        
        displayPosn[remoteIndex()].w = event.resize.w*4/5;
        displayPosn[remoteIndex()].h = event.resize.h;
        displayPosn[remoteIndex()].x = 0;        
        displayPosn[remoteIndex()].y = 0;        
        WriteOutDisplay();        
      } else {
        displayPosn[encodeIndex()].w = event.resize.w/2;
        displayPosn[encodeIndex()].h = event.resize.h;
        displayPosn[encodeIndex()].x = event.resize.w/2;        
        displayPosn[encodeIndex()].y = 0;        
        displayPosn[remoteIndex()].w = event.resize.w/2;
        displayPosn[remoteIndex()].h = event.resize.h;
        displayPosn[remoteIndex()].x = 0;        
        displayPosn[remoteIndex()].y = 0;   
        WriteOutDisplay();        
      }
      OpenScreen();
    }    //end event.type==SDL_VIDEORESIZE
  }      //end while (SDL_PollEvent())
}


BOOL SDLVideoDevice::Redraw (const void *frame)
{
  PWaitAndSignal vMutex(videoMutex);
  PTRACE(6,"SDL\t Redraw starts now for real. "<<DirectionStr());  
//  cout <<"redraw starts now for "<<DirectionStr()<<endl;
  
  ProcessSdlEvents();

  if(!IsOpen()) {   //Some task somewhere has closed our display device.
    PTRACE(6,"SDL\t Screen is closed. ::Redraw() returning immediately"); 
    return TRUE;   //(could have been user selecting cross on the window)
  }
      
  SDL_Overlay *yuvOverlay= overlay[xIndex()];
//  cout <<"redraw continues now for "<< (int)yuvOverlay<<endl;
  if (yuvOverlay==NULL) {
    PTRACE(6,"SDL\t Redraw end prematurely, no overlay to use. "
           <<DirectionStr()<<" window.");
    return TRUE;       //Again, we lost the overlay, just exit quickly.
  }
  
  yuvOverlay->pixels[0] = (unsigned char *) frame;
  yuvOverlay->pixels[1] = (unsigned char *) frame + (width*height);
  yuvOverlay->pixels[2] = (unsigned char *) frame + (width*height*5/4);
  
  SDL_LockYUVOverlay(yuvOverlay);
  SDL_DisplayYUVOverlay(yuvOverlay,displayPosn + xIndex() );    
  SDL_UnlockYUVOverlay(yuvOverlay);

  return TRUE;  
}


BOOL  SDLVideoDevice::IsOpen()
{
  return screen != NULL;
}

 

#endif

// HAS_SDL
