%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id$
%%
-module(mnesia_late_loader).

-export([
	 async_late_disc_load/3,
	 maybe_async_late_disc_load/3,
	 init/1,
	 start/0
	]).

%% sys callback functions
-export([
	 system_continue/3,
	 system_terminate/4,
	 system_code_change/4
	]).

-define(SERVER_NAME, ?MODULE).

-record(state, {supervisor}).

async_late_disc_load(Node, Tabs, Reason) ->
    Msg = {async_late_disc_load, Tabs, Reason},
    catch ({?SERVER_NAME, Node} ! {self(), Msg}).

maybe_async_late_disc_load(Node, Tabs, Reason) ->
    Msg = {maybe_async_late_disc_load, Tabs, Reason},
    catch ({?SERVER_NAME, Node} ! {self(), Msg}).

start() ->
    mnesia_monitor:start_proc(?SERVER_NAME, ?MODULE, init, [self()]).

init(Parent) ->
    %% Trap exit omitted intentionally
    register(?SERVER_NAME, self()),
    mnesia_controller:merge_schema(),
    proc_lib:init_ack(Parent, {ok, self()}),
    loop(#state{supervisor = Parent}).

loop(State) ->
    receive
	{From, {async_late_disc_load, Tabs, Reason}} ->
	    mnesia_controller:schedule_late_disc_load(Tabs, Reason),
	    loop(State);

	{From, {maybe_async_late_disc_load, Tabs, Reason}} ->
	    GoodTabs =
		[T || T <- Tabs,
		      lists:member(node(),
				   mnesia_recover:get_master_nodes(T))],
	    mnesia_controller:schedule_late_disc_load(GoodTabs, Reason),
	    loop(State);

	{system, From, Msg} ->
	    mnesia_lib:dbg_out("~p got {system, ~p, ~p}~n",
			       [?SERVER_NAME, From, Msg]),
	    Parent = State#state.supervisor,
	    sys:handle_system_msg(Msg, From, Parent, ?MODULE, [], State);
	    
	Msg ->
	    mnesia_lib:error("~p got unexpected message: ~p~n",
			     [?SERVER_NAME, Msg]),
	    loop(State)
    end.

%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% System upgrade

system_continue(Parent, Debug, State) ->
    loop(State).

system_terminate(Reason, Parent, Debug, State) ->
    exit(Reason).

system_code_change(State, Module, OldVsn, Extra) ->
    exit(not_supported).
