/*-
 * Copyright (c) 1992, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 * Copyright (c) 1992, 1993, 1994, 1995, 1996
 *	Keith Bostic.  All rights reserved.
 *
 * See the LICENSE file for redistribution information.
 */

#include "config.h"

#ifndef lint
static const char sccsid[] = "@(#)v_ch.c	10.8 (Berkeley) 3/6/96";
#endif /* not lint */

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/time.h>

#include <bitstring.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>

#include "../common/common.h"
#include "vi.h"

#ifndef MULTIBYTE
static void notfound __P((SCR *, ARG_CHAR_T));
#else
static void notfound __P((SCR *, CHAR_T *));
#endif
static void noprev __P((SCR *));

/*
 * v_chrepeat -- [count];
 *	Repeat the last F, f, T or t search.
 *
 * PUBLIC: int v_chrepeat __P((SCR *, VICMD *));
 */
int
v_chrepeat(sp, vp)
	SCR *sp;
	VICMD *vp;
{
#ifndef MULTIBYTE
	vp->character = VIP(sp)->lastckey;
#else
	memcpy(vp->characters, VIP(sp)->lastckeys, sizeof(vp->characters));
#endif

	switch (VIP(sp)->csearchdir) {
	case CNOTSET:
		noprev(sp);
		return (1);
	case FSEARCH:
		return (v_chF(sp, vp));
	case fSEARCH:
		return (v_chf(sp, vp));
	case TSEARCH:
		return (v_chT(sp, vp));
	case tSEARCH:
		return (v_cht(sp, vp));
	default:
		abort();
	}
	/* NOTREACHED */
}

/*
 * v_chrrepeat -- [count],
 *	Repeat the last F, f, T or t search in the reverse direction.
 *
 * PUBLIC: int v_chrrepeat __P((SCR *, VICMD *));
 */
int
v_chrrepeat(sp, vp)
	SCR *sp;
	VICMD *vp;
{
	cdir_t savedir;
	int rval;

#ifndef MULTIBYTE
	vp->character = VIP(sp)->lastckey;
#else
	memcpy(vp->characters, VIP(sp)->lastckeys, sizeof(vp->characters));
#endif
	savedir = VIP(sp)->csearchdir;

	switch (VIP(sp)->csearchdir) {
	case CNOTSET:
		noprev(sp);
		return (1);
	case FSEARCH:
		rval = v_chf(sp, vp);
		break;
	case fSEARCH:
		rval = v_chF(sp, vp);
		break;
	case TSEARCH:
		rval = v_cht(sp, vp);
		break;
	case tSEARCH:
		rval = v_chT(sp, vp);
		break;
	default:
		abort();
	}
	VIP(sp)->csearchdir = savedir;
	return (rval);
}

/*
 * v_cht -- [count]tc
 *	Search forward in the line for the character before the next
 *	occurrence of the specified character.
 *
 * PUBLIC: int v_cht __P((SCR *, VICMD *));
 */
int
v_cht(sp, vp)
	SCR *sp;
	VICMD *vp;
{
#ifdef MULTIBYTE
	CHAR_T *p;
#endif

	if (v_chf(sp, vp))
		return (1);

#ifdef MULTIBYTE
	/* we need content of the line. */
	if (db_eget(sp, vp->m_start.lno, &p, NULL, NULL))
		return (1);
#endif

	/*
	 * v_chf places the cursor on the character, where the 't'
	 * command wants it to its left.  We know this is safe since
	 * we had to move right for v_chf() to have succeeded.
	 */
#ifndef MULTIBYTE
	--vp->m_stop.cno;
#else
	vp->m_stop.cno -= multi_back(sp, p, &p[vp->m_stop.cno]);
#endif

	/*
	 * Make any necessary correction to the motion decision made
	 * by the v_chf routine.
	 */
	if (!ISMOTION(vp))
		vp->m_final = vp->m_stop;

	VIP(sp)->csearchdir = tSEARCH;
	return (0);
}

/*
 * v_chf -- [count]fc
 *	Search forward in the line for the next occurrence of the
 *	specified character.
 *
 * PUBLIC: int v_chf __P((SCR *, VICMD *));
 */
int
v_chf(sp, vp)
	SCR *sp;
	VICMD *vp;
{
	size_t len;
	u_long cnt;
#ifndef MULTIBYTE
	int isempty, key;
	char *endp, *p, *startp;
#else
	int isempty;
	CHAR_T *key;
	size_t blen;
	CHAR_T *endp, *p, *startp;
#endif /*MULTIBYTE*/

	/*
	 * !!!
	 * If it's a dot command, it doesn't reset the key for which we're
	 * searching, e.g. in "df1|f2|.|;", the ';' searches for a '2'.
	 */
#ifndef MULTIBYTE
	key = vp->character;
	if (!F_ISSET(vp, VC_ISDOT))
		VIP(sp)->lastckey = key;
#else
	key = vp->characters;
	blen = KEY_BLEN(sp, key[0]);
	if (!F_ISSET(vp, VC_ISDOT))
		memcpy(VIP(sp)->lastckeys, key, sizeof(VIP(sp)->lastckeys));
#endif /*MULTIBYTE*/
	VIP(sp)->csearchdir = fSEARCH;

	if (db_eget(sp, vp->m_start.lno, &p, &len, &isempty)) {
		if (isempty)
			goto empty;
		return (1);
	}

	if (len == 0) {
empty:		notfound(sp, key);
		return (1);
	}

	endp = (startp = p) + len;
	p += vp->m_start.cno;
	for (cnt = F_ISSET(vp, VC_C1SET) ? vp->count : 1; cnt--;) {
#ifndef MULTIBYTE
		while (++p < endp && *p != key);
#else
		do {
			p += KEY_BLEN(sp, p[0]);
		} while (p < endp && (*p != key[0] || memcmp(p, key, blen)));
#endif /*MULTIBYTE*/
		if (p == endp) {
			notfound(sp, key);
			return (1);
		}
	}

	vp->m_stop.cno = p - startp;

	/*
	 * Non-motion commands move to the end of the range.
	 * Delete and yank stay at the start, ignore others.
	 */
	vp->m_final = ISMOTION(vp) ? vp->m_start : vp->m_stop;
	return (0);
}

/*
 * v_chT -- [count]Tc
 *	Search backward in the line for the character after the next
 *	occurrence of the specified character.
 *
 * PUBLIC: int v_chT __P((SCR *, VICMD *));
 */
int
v_chT(sp, vp)
	SCR *sp;
	VICMD *vp;
{
#ifdef MULTIBYTE
	CHAR_T *p;
#endif

	if (v_chF(sp, vp))
		return (1);

#ifdef MULTIBYTE
	/* we need content of the line. */
	if (db_eget(sp, vp->m_start.lno, &p, NULL, NULL))
		return (1);
#endif

	/*
	 * v_chF places the cursor on the character, where the 'T'
	 * command wants it to its right.  We know this is safe since
	 * we had to move left for v_chF() to have succeeded.
	 */
#ifndef MULTIBYTE
	++vp->m_stop.cno;
#else
	vp->m_stop.cno += KEY_BLEN(sp, p[vp->m_stop.cno]);
#endif
	vp->m_final = vp->m_stop;

	VIP(sp)->csearchdir = TSEARCH;
	return (0);
}

/*
 * v_chF -- [count]Fc
 *	Search backward in the line for the next occurrence of the
 *	specified character.
 *
 * PUBLIC: int v_chF __P((SCR *, VICMD *));
 */
int
v_chF(sp, vp)
	SCR *sp;
	VICMD *vp;
{
	size_t len;
	u_long cnt;
#ifndef MULTIBYTE
	int isempty, key;
	char *endp, *p;
#else
	int isempty;
	CHAR_T *key;
	size_t blen;
	CHAR_T *endp, *p;
#endif

	/*
	 * !!!
	 * If it's a dot command, it doesn't reset the key for which
	 * we're searching, e.g. in "df1|f2|.|;", the ';' searches
	 * for a '2'.
	 */
#ifndef MULTIBYTE
	key = vp->character;
	if (!F_ISSET(vp, VC_ISDOT))
		VIP(sp)->lastckey = key;
#else
	key = vp->characters;
	blen = KEY_BLEN(sp, key[0]);
	if (!F_ISSET(vp, VC_ISDOT))
		memcpy(VIP(sp)->lastckeys, key, sizeof(VIP(sp)->lastckeys));
#endif /*MULTIBYTE*/
	VIP(sp)->csearchdir = FSEARCH;

	if (db_eget(sp, vp->m_start.lno, &p, &len, &isempty)) {
		if (isempty)
			goto empty;
		return (1);
	}

	if (len == 0) {
empty:		notfound(sp, key);
		return (1);
	}

	endp = p - 1;
	p += vp->m_start.cno;
	for (cnt = F_ISSET(vp, VC_C1SET) ? vp->count : 1; cnt--;) {
#ifndef MULTIBYTE
		while (--p > endp && *p != key);
#else
		do {
			p -= multi_back(sp, endp + 1, p);
		} while (p > endp && (*p != key[0] || memcmp(p, key, blen)));
#endif
		if (p == endp) {
			notfound(sp, key);
			return (1);
		}
	}

	vp->m_stop.cno = (p - endp) - 1;

	/*
	 * All commands move to the end of the range.  Motion commands
	 * adjust the starting point to the character before the current
	 * one.
	 */
	vp->m_final = vp->m_stop;
	if (ISMOTION(vp)) {
#ifndef MULTIBYTE
		--vp->m_start.cno;
#else
		vp->m_start.cno -= multi_back(sp, p, &p[vp->m_start.cno]);
#endif
	}
	return (0);
}

static void
noprev(sp)
	SCR *sp;
{
	msgq(sp, M_BERR, "178|No previous F, f, T or t search");
}

#ifndef MULTIBYTE
static void
notfound(sp, ch)
	SCR *sp;
	ARG_CHAR_T ch;
{
	msgq(sp, M_BERR, "179|%s not found", KEY_NAME(sp, ch));
}
#else
static void
notfound(sp, p)
	SCR *sp;
	CHAR_T *p;
{
#if 0
	msgq(sp, M_BERR, "179|%s not found", multi_name(sp, p));
#else
	/*
	 * currently msg routines do not support multibyte chars.
	 */
	if (1 < KEY_BLEN(sp, p[0]))
		msgq(sp, M_BERR, "179|%s not found", "character");
	else
		msgq(sp, M_BERR, "179|%s not found", KEY_NAME(sp, p));
#endif
}
#endif /*MULTIBYTE*/
