/*
 *   xmcd - Motif(tm) CD Audio Player
 *
 *   Copyright (C) 1993-1999  Ti Kan
 *   E-mail: ti@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef LINT
static char *_wwwwarp_c_ident_ = "@(#)wwwwarp.c	6.45 99/04/18";
#endif

#include "common_d/appenv.h"
#include "common_d/patchlevel.h"
#include "common_d/util.h"
#include "xmcd_d/xmcd.h"
#include "xmcd_d/widget.h"
#include "cddb_d/cddb.h"
#include "xmcd_d/cdfunc.h"
#include "xmcd_d/dbprog.h"
#include "xmcd_d/wwwwarp.h"
#include "libdi_d/libdi.h"


#define CHARS_PER_TAB	8			/* # chars in a tab */

#ifdef BUGGY_BROWSER
/* Should always use &nbsp; but some versions of NCSA Mosaic don't grok it */
#define HTML_ESC_SPC	"&#20;"			/* HTML for blank space */
#else
#define HTML_ESC_SPC	"&nbsp;"		/* HTML for blank space */
#endif

extern widgets_t	widgets;
extern appdata_t	app_data;
extern char		*cdisplay;
extern FILE		*errfp;

STATIC int		website_idx;		/* Web site select */
STATIC urltmpl_ent_t	*cur_srcheng;		/* Current search engine */
STATIC bool_t		wwwwarp_initted;	/* Initialization done */

/* URL protocols */
STATIC struct {
	char		*protocol;
	bool_t		islocal;
} url_protolist[] = {
	{ "http:",	FALSE	},
	{ "https:",	FALSE	},
	{ "ftp:",	FALSE	},
	{ "file:",	TRUE	},
	{ "mailto:",	FALSE	},
	{ "news:",	FALSE	},
	{ "snews:",	FALSE	},
	{ "nntp:",	FALSE	},
	{ "gopher:",	FALSE	},
	{ "wais:",	FALSE	},
	{ "telnet:",	FALSE	},
#ifdef WWWWARP_URL_EXTPROTOS		/* Define this if desired */
	{ "rlogin:",	FALSE	},
	{ "tn3270:",	FALSE	},
	{ "data:",	FALSE	},
	{ "ldap:",	FALSE	},
	{ "ldaps:",	FALSE	},
	{ "castanet:",	FALSE	},
#endif
	{ NULL,		FALSE	}
};


/* Forward declaration */
STATIC char	*wwwwarp_gen_url(curstat_t *, urltmpl_ent_t *);


/***********************
 *  internal routines  *
 ***********************/


#define IS_UNKN_URL	0
#define IS_LOCAL_URL	1
#define IS_REMOTE_URL	2

/*
 * wwwwarp_urlchk
 *	Check a URL to see if it's local or remote.  Return
 *	a pointer to the actual start of the path name (past the
 *	protocol, hostname and port portion if it's local).
 *
 * Args:
 *	url - Ths URL string to check
 *	filepath - The character pointer to set to the beginning of the
 *		   file path.
 *
 * Return:
 *	IS_UNKN_URL if the URL is invalid
 *	IS_LOCAL_URL if the URL is local
 *	IS_REMOTE_URL if the URL is remote
 */
STATIC int
wwwwarp_urlchk(char *url, char **filepath)
{
	int	i;
	char	*p,
		*q;
	bool_t	local;

	p = url;
	local = TRUE;

	for (i = 0; url_protolist[i].protocol != NULL; i++) {
		if (util_strncasecmp(url, url_protolist[i].protocol,
			strlen(url_protolist[i].protocol)) == 0) {

			if (!url_protolist[i].islocal)
				local = FALSE;
			break;
		}
	}

	if (url_protolist[i].protocol == NULL) {
		*filepath = p;
		return IS_UNKN_URL;
	}

	if (local) {
		if (util_strncasecmp(url, "file://localhost", 16) == 0)
			p = url + 16;
		else if (util_strncasecmp(url, "file://", 7) == 0) {
			p = url + 7;
			if ((q = strchr(p, '/')) != NULL)
				p = q + 1;
			else
				p += strlen(p);
		}
		else if (util_strncasecmp(url, "file:", 5) == 0)
			p = url + 5;
	}

	*filepath = p;

	return (local ? IS_LOCAL_URL : IS_REMOTE_URL);
}


/*
 * wwwwarp_html_fputs
 *	Similar to fputs(3), but handles HTML escape sequence, newline and
 *	tab translation.  If specified, it also tries to recognizes URLs
 *	in the string and turns them into links.
 *
 * Args:
 *	str - The string to write
 *	fp - The file pointer
 *	urldet - Whether to look for URLs and turn them into links
 *	fontname - The font name to use, or NULL to use default font
 *	fontsize - The font size to use. or 0 to use default size
 *
 * Return:
 *	Nothing.
 */
STATIC void
wwwwarp_html_fputs(
	char	*str,
	FILE	*fp,
	bool_t	urldet,
	char	*fontname,
	int	fontsize
)
{
	int	i,
		j,
		k;
	char	*cp,
		*p1,
		*p2,
		*p3,
		*p4,
		*p5,
		*endstr,
		sav;

	endstr = str + strlen(str);

	if (fontname != NULL || fontsize != 0) {
		(void) fputs("<FONT", fp);
		if (fontname != NULL)
			(void) fprintf(fp, " FACE=\"%s\"", fontname);
		if (fontsize != 0)
			(void) fprintf(fp, " SIZE=\"%d\"", fontsize);
		(void) fputs(">\n", fp);
	}

	i = k = 0;
	for (cp = str; *cp != '\0'; cp++) {
		switch (*cp) {
		case ' ':
			(void) fprintf(fp, "%s",
				(k > 0) ?
				    HTML_ESC_SPC :
				    ((*(cp + 1) == ' ') ? HTML_ESC_SPC : " ")
			);
			k++;
			break;
		case '\n':
			(void) fputs("<BR>\n", fp);
			i = -1;
			k = 0;
			break;
		case '\t':
			for (j = CHARS_PER_TAB; j > i; j--)
				(void) fputs(HTML_ESC_SPC, fp);
			i = -1;
			k = 0;
			break;
		case '<':
			(void) fputs("&lt;", fp);
			k = 0;
			break;
		case '>':
			(void) fputs("&gt;", fp);
			k = 0;
			break;
		case '&':
			(void) fputs("&amp;", fp);
			k = 0;
			break;
		case '"':
			(void) fputs("&quot;", fp);
			k = 0;
			break;
		default:
			/* Check to see if it's a URL */
			if (!urldet ||
			    wwwwarp_urlchk(cp, &p1) == IS_UNKN_URL) {
				(void) fputc(*cp, fp);
				k = 0;
				break;
			}

			/* Check for incomplete URL */
			if ((p1 = strchr(cp, ':')) == NULL ||
			     *(p1 + 1) == ' ' || *(p1 + 1) == '\r' ||
			     *(p1 + 1) == '\n') {
				(void) fputc(*cp, fp);
				k = 0;
				break;
			}

			/* Look for the end of the URL */
			sav = '\0';
			if ((p2 = strchr(p1, ' ')) == NULL)
				p2 = endstr;
			if ((p3 = strchr(p1, '\t')) == NULL)
				p3 = endstr;
			if ((p4 = strchr(p1, '\r')) == NULL)
				p4 = endstr;
			if ((p5 = strchr(p1, '\n')) == NULL)
				p5 = endstr;

			p1 = (p2 < p3) ? p2 : p3;
			p2 = (p4 < p5) ? p4 : p5;
			if (p2 < p1)
				p1 = p2;

			sav = *p1;
			*p1 = '\0';

			/* Output */
			(void) fprintf(fp, "<A HREF=\"%s\">%s</A>", cp, cp);

			i = ((i + (int) strlen(cp)) % CHARS_PER_TAB) - 1;

			*p1 = sav;
			cp = p1 - 1;
			k = 0;
			break;
		}
		if (++i == CHARS_PER_TAB)
			i = 0;
	}

	if (fontname != NULL || fontsize != 0)
		(void) fputs("\n</FONT>\n", fp);
}


#ifdef __VMS

#define UNIX_2_VMS	1
#define VMS_2_UNIX	2

/*
 * wwwwarp_vms_urlconv
 *	URL format translation function between UNIX-style and VMS-style
 *	path name syntax.  This is a hack to work around the inconsistent
 *	URL formats required by the browser under different circumstances.
 *	The caller should MEM_FREE() the return string buffer.
 *
 * Args:
 *	url - The input URL string
 *	dir - Conversion direction: UNIX_2_VMS or VMS_2_UNIX
 *
 * Return:
 *	The converted URL string
 */
STATIC char *
wwwwarp_vms_urlconv(char *url, int dir)
{
	int	i;
	char	*p1,
		*p2,
		*p3,
		*p4,
		*buf;
	bool_t	first;

	if (wwwwarp_urlchk(url, &p1) == IS_REMOTE_URL)
		return (url);	/* Remote URL: don't touch it */

	buf = (char *) MEM_ALLOC(
		"wwwwarp_vms_urlconv",
		(strlen(url) * 2) + FILE_PATH_SZ
	);
	if (buf == NULL) {
		CD_FATAL(app_data.str_nomemory);
		return NULL;
	}
	buf[0] = '\0';

	switch (dir) {
	case UNIX_2_VMS:
		first = TRUE;
		if (*p1 == '/')
			p1++;

		for (i = 0; (p2 = strchr(p1, '/')) != NULL; i++) {
			*p2 = '\0';

			if (i == 0 && (strchr(p1, '$') != NULL)) {
				(void) sprintf(buf, "%s:", p1);
			}
			else if (first) {
				first = FALSE;
				(void) sprintf(buf, "%s[%s", buf, p1);
			}
			else {
				(void) sprintf(buf, "%s.%s", buf, p1);
			}

			*p2 = '/';
			p1 = p2 + 1;
		}

		if (first) {
			if (strchr(p1, '$') != NULL)
				(void) sprintf(buf, "%s:", p1);
			else
				(void) sprintf(buf, "%s%s", buf, p1);
		}
		else
			(void) sprintf(buf, "%s]%s", buf, p1);

		break;

	case VMS_2_UNIX:
		strcpy(buf, "file:");

		if ((p2 = strchr(p1, ':')) != NULL) {
			*p2 = '\0';
			(void) sprintf(buf, "%s//localhost/%s", buf, p1);
			*p2 = ':';
			p1 = p2 + 1;
		}

		if ((p3 = strchr(p1, '[')) != NULL) {
			p3++;
			if ((p4 = strchr(p3, ']')) != NULL)
				*p4 = '\0';

			while ((p1 = strchr(p3, '.')) != NULL) {
				*p1 = '\0';
				(void) sprintf(buf, "%s/%s", buf, p3);
				*p1 = '.';
				p3 = p1 + 1;
			}
			(void) sprintf(buf, "%s/%s", buf, p3);

			if (p4 != NULL) {
				*p4 = ']';
				p1 = p4 + 1;
			}
		}

		if (p2 == NULL && p3 == NULL)
			(void) sprintf(buf, "%s%s", buf, p1);
		else
			(void) sprintf(buf, "%s/%s", buf, p1);

		break;

	default:
		MEM_FREE(buf);
		buf = NULL;
		break;
	}

	return (buf);
}

#endif	/* __VMS */


/*
 * wwwwarp_site
 *	Invoke the web browser to the selected web site.
 *
 * Args:
 *	url - The destination URL string
 *
 * Return:
 *	Error code
 */
STATIC void
wwwwarp_site(char *url)
{
	int		a,
			b,
			n,
			ret;
	char		*r,
			*s,
			*env,
			*cmd;
	static bool_t	first = TRUE;

	/* If user specified the -display option when starting xmcd,
	 * force this display specification into the environment so
	 * that we will manage the browser on the same display as
	 * xmcd.
	 */
	if (first && cdisplay != NULL) {
		first = FALSE;
		env = (char *) MEM_ALLOC(
			"wwwwarp_site_env",
			strlen(cdisplay) + 12
		);
		if (env == NULL) {
			CD_FATAL(app_data.str_nomemory);
			return;
		}
		(void) sprintf(env, "DISPLAY=%s", cdisplay);
		if (putenv(env) != 0) {
			CD_FATAL(app_data.str_nomemory);
			return;
		}
		/* Do not MEM_FREE(env)! */
	}

	a = strlen(app_data.browser_rmt);
	b = strlen(app_data.browser_dir);
	n = ((a > b) ? a : b) + strlen(url) + 12;

	if ((cmd = (char *) MEM_ALLOC("wwwwarp_site_cmd", n)) == NULL) {
		CD_FATAL(app_data.str_nomemory);
		return;
	}

	/* Set up remote control of web browser */
	r = cmd;
	for (s = app_data.browser_rmt; s != NULL && *s != '\0'; s++) {
		switch (*s) {
		case '%':
			switch (*(++s)) {
			case 'U':
				(void) strcpy(r, url);
				r += strlen(url);
				break;
			case '%':
				*r++ = '%';
				break;
			default:
				*r++ = '%';
				*r++ = *s;
				break;
			}
			break;
		default:
			*r++ = *s;
			break;
		}
	}
	*r = '\0';

	/* Attempt to remote control an existing browser running on
	 * this display.
	 */
	if ((ret = util_runcmd(cmd, event_loop, 0)) == 0) {
		/* Success */
		MEM_FREE(cmd);
		return;
	}

	if (ret == EXE_ERR) {
		/* There is a configuration error: cannot invoke browser */
		CD_INFO(app_data.str_browser_fail);
		MEM_FREE(cmd);
		return;
	}

	/*
	 * Remote control of running browser failed.
	 * Try starting a new session.
	 */

	/* Set up direct browser command */
	r = cmd;
	for (s = app_data.browser_dir; s != NULL && *s != '\0'; s++) {
		switch (*s) {
		case '%':
			switch (*(++s)) {
			case 'U':
				(void) strcpy(r, url);
				r += strlen(url);
				break;
			case '%':
				*r++ = '%';
				break;
			default:
				*r++ = '%';
				*r++ = *s;
				break;
			}
			break;
		default:
			*r++ = *s;
			break;
		}
	}
	*r = '\0';

	CD_INFO_AUTO(app_data.str_browser_start);

	/* Do the direct browser invocation */
	if ((ret = util_runcmd(cmd, event_loop, 0)) != 0)
		CD_INFO(app_data.str_browser_fail);

	MEM_FREE(cmd);
}


/*
 * wwwwarp_txtreduce
 *	Transform a text string into a CGI search string
 *
 * Args:
 *	text - Input text string
 *      reduce - Whether to reduce text (remove punctuations,
 *		 exclude words  from the excludeWords list).
 *
 * Return:
 *	Output string.  The storage is internally allocated per call,
 *	and should be freed by the caller with MEM_FREE() after the
 *	buffer is no longer needed.  If there is no sufficient memory
 *	to allocate the buffer, NULL is returned.
 */
STATIC char *
wwwwarp_txtreduce(char *text, bool_t reduce)
{
	char	*p,
		*q;

	if (reduce) {
		if ((p = util_text_reduce(text)) == NULL)
		return NULL;
	}
	else
		p = text;

	if ((q = util_cgi_xlate(p)) == NULL)
		return NULL;

	if (reduce)
		MEM_FREE(p);

	return (q);
}


/*
 * wwwwarp_url_len
 *	Determine a string buffer size large enough to hold a
 *	URL after it is expanded from its template.
 *
 * Args:
 *	tmpl - Template string
 *	up - Pointer to the associated URL attributes structure
 *	trkpos - Pointer to the track position for %T or %t.
 *	s - Pointer to the curstat_t structure.
 *
 * Return:
 *	The buffer length.
 */
STATIC int
wwwwarp_url_len(char *tmpl, url_attrib_t *up, int *trkpos, curstat_t *s)
{
	int		n,
			dtitle_len;
	cddb_incore_t	*dbp;

	dbp = dbprog_curdb(s);
	dtitle_len = (dbp->dtitle != NULL) ? strlen(dbp->dtitle) : STR_BUF_SZ;

	n = 16 + strlen(tmpl) + ((up->acnt + up->dcnt) * dtitle_len);

	if (*trkpos < 0 || dbp->trklist[*trkpos] == NULL) {
		*trkpos = -1;
		n += (up->tcnt * dtitle_len);
	}
	else
		n += (cur_srcheng->a_attrib.tcnt *
		      strlen(dbp->trklist[*trkpos]));

	n += (up->xcnt * strlen(PROGNAME));
	n += (up->vcnt * strlen(VERSION));
	n += (up->ncnt * STR_BUF_SZ);
	n += (up->hcnt * MAXHOSTNAMELEN);
	n += (up->lcnt * strlen(app_data.libdir));
	n += (up->ccnt * strlen(dbp->category));
	n += (up->icnt * 8);

	return (n);
}


/*
 * wwwwarp_tmpl_to_url
 *	Make a URL string from a template.
 *
 * Args:
 *	tmpl - Template string
 *	url - Buffer to store resultant URL string
 *	trkpos - The track position for %T or %t
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing.
 */
STATIC void
wwwwarp_tmpl_to_url(char *tmpl, char *url, int trkpos, curstat_t *s)
{
	int		n;
	char		*p,
			*q,
			*r,
			*a;
	char		sav;
	cddb_incore_t	*dbp;

	dbp = dbprog_curdb(s);

	r = url;
	for (a = tmpl; *a != '\0'; a++) {
		switch (*a) {
		case '%':
			switch (*(++a)) {
			case 'X':
				(void) strcpy(r, PROGNAME);
				r += strlen(PROGNAME);
				break;

			case 'V':
				(void) strcpy(r, VERSION);
				r += strlen(VERSION);
				break;

			case 'N':
				q = util_loginname();
				(void) strcpy(r, q);
				r += strlen(q);
				break;

			case 'H':
				q = cd_hostname();
				(void) strcpy(r, q);
				r += strlen(q);
				break;

			case 'L':
				(void) strcpy(r, app_data.libdir);
				r += strlen(app_data.libdir);
				break;

			case 'C':
				if (dbp->category[0] == '\0')
					q = "unknown";
				else
					q = dbp->category;

				(void) strcpy(r, q);
				r += strlen(q);
				break;

			case 'I':
				(void) sprintf(r, "%08x", dbp->discid);
				r += 8;
				break;

			case 'A':
			case 'a':
				sav = '\0';

				/* Translate disk artist into a
				 * keyword string in CGI form
				 */
				p = dbp->dtitle;
				while ((p = strchr(p, '/')) != NULL)
				{
					if (p > dbp->dtitle &&
					    *(p-1) == ' ' && *(p+1) == ' ')
						break;
					else
						p++;
				}
				if (p != NULL) {
					/* Check for '/' at beginning of
					 * string.
					 */
					n = 0;
					for (q = dbp->dtitle; q < p; q++) {
						if (!isspace(*q)) {
							n++;
							break;
						}
					}
					if (n > 0) {
						--p;
						sav = *p;
						*p = '\0';
					}
				}

				q = wwwwarp_txtreduce(
					dbp->dtitle,
					(bool_t) (*a == 'a')
				);
				if (q == NULL) {
					CD_FATAL(app_data.str_nomemory);
					return;
				}

				if (sav != '\0')
					*p = sav;

				(void) strcpy(r, q);
				r += strlen(q);
				MEM_FREE(q);
				break;

			case 'T':
			case 't':
				/* Translate track title into a
				 * keyword string in CGI form
				 */
				if (trkpos >= 0) {
					q = wwwwarp_txtreduce(
						dbp->trklist[trkpos],
						(bool_t) (*a == 't')
					);
					if (q == NULL) {
					    CD_FATAL(app_data.str_nomemory);
					    return;
					}
					strcpy(r, q);
					r += strlen(q);
					MEM_FREE(q);
					break;
				}
				/*FALLTHROUGH*/

			case 'D':
			case 'd':
				/* Translate disc title into a
				 * keyword string in CGI form
				 */
				p = dbp->dtitle;
				while ((p = strchr(p, '/')) != NULL)
				{
					if (p > dbp->dtitle &&
					    *(p-1) == ' ' && *(p+1) == ' ')
						break;
					else
						p++;
				}
				if (p == NULL)
					/* '/' not found, just use the
					 * whole string.
					 */
					p = dbp->dtitle;
				else {
					q = ++p;
					while (*q != '\0' && isspace(*q))
						q++;

					if (*q == '\0') {
						/* '/' at end-of-string: just
						 * use the whole string
						 */
						p = dbp->dtitle;
					}
				}

				q = wwwwarp_txtreduce(p, (bool_t) (*a == 'd'));
				if (q == NULL) {
					CD_FATAL(app_data.str_nomemory);
					return;
				}

				(void) strcpy(r, q);
				r += strlen(q);
				MEM_FREE(q);
				break;

			case 'B':
			case 'b':
				/* Translate artist and disc title into a
				 * keyword string in CGI form
				 */
				q = wwwwarp_txtreduce(
					dbp->dtitle,
					(bool_t) (*a == 'b')
				);
				if (q == NULL) {
					CD_FATAL(app_data.str_nomemory);
					return;
				}

				(void) strcpy(r, q);
				r += strlen(q);
				MEM_FREE(q);
				break;

			case '%':
				*r++ = '%';
				break;

			default:
				*r++ = '%';
				*r++ = *a;
				break;
			}
			break;
		default:
			*r++ = *a;
			break;
		}
	}
	*r = '\0';
}


/*
 * wwwwarp_db2html
 *	Perform HTML output formatting.  Called by wwwwarp_gen_discogidx().
 *
 * Args:
 *	fp - Output file pointer.
 *	s - Pointer to the curstat_t structure.
 *	baseurl - The URL to the output file.
 *	outdir - The path name to the output directory.
 *
 * Return:
 *	Nothing.
 */
STATIC void
wwwwarp_db2html(FILE *fp, curstat_t *s, char *baseurl, char *outdir)
{
	int		i;
	char		*p,
			*tblparms1,
			*tblparms2,
			*tdparms1,
			*tdparms2,
			*name,
			*srchact,
			filepath[FILE_PATH_SZ * 2];
	XmString	xs;
	DIR		*dp;
	struct dirent	*de;
	cddb_incore_t	*dbp;
	cddb_path_t	*pp;
	urltmpl_ent_t	*t;
	bool_t		first,
			first_categ;
	Boolean		ret;

	dbp = dbprog_curdb(s);

	/* Look for CDDB search engine definition */
	srchact = NULL;
	for (t = di_srcheng_list(); t != NULL; t = t->next) {
		if (strcmp(t->name, "CDDB") == 0) {
			srchact = (char *) MEM_ALLOC(
				"srchact",
				strlen(t->action) + 1
			);
			if (srchact == NULL) {
				CD_FATAL(app_data.str_nomemory);
				return;
			}
			(void) strcpy(srchact, t->action);

			if ((p = strchr(srchact, '%')) != NULL)
				*p = '\0';
			else {
				MEM_FREE(srchact);
				srchact = NULL;
			}
			break;
		}
	}

	/* HTML output */
	tblparms1 = "CELLSPACING=\"0\" CELLPADDING=\"1\" BORDER=\"0\"";
	tblparms2 = "CELLSPACING=\"0\" CELLPADDING=\"3\" BORDER=\"1\"";
	tdparms1 = "ALIGN=\"center\"";
	tdparms2 = "ALIGN=\"left\"";

	(void) fprintf(fp, "<!-- %s Local Discography\n", PROGNAME);
	(void) fprintf(fp, "     DO NOT EDIT: Generated by %s %s%s PL%d\n",
		PROGNAME, VERSION, VERSION_EXT, PATCHLEVEL);
	(void) fprintf(fp, "     %s\n     URL: %s E-mail: %s -->\n",
		COPYRIGHT, XMCD_URL, EMAIL);
	(void) fputs("<!-- tItLe: ", fp);
	wwwwarp_html_fputs(
		(dbp->dtitle == NULL || dbp->dtitle[0] == '\0') ?
			app_data.str_unkndisc : dbp->dtitle,
		fp,
		FALSE,
		NULL,
		0
	);
	(void) fputs(" -->\n", fp);

	(void) fputs("<HTML>\n<HEAD>\n", fp);
	(void) fprintf(fp, "<TITLE>\n%s: ", PROGNAME);
	wwwwarp_html_fputs(
		(dbp->dtitle == NULL || dbp->dtitle[0] == '\0') ?
			app_data.str_unkndisc : dbp->dtitle,
		fp,
		FALSE,
		NULL,
		0
	);
	(void) fputs("\n</TITLE>\n", fp);

#ifdef __VMS
	(void) sprintf(filepath, "%s.discog]bkgnd.gif", app_data.libdir);
	if ((p = wwwwarp_vms_urlconv(filepath, VMS_2_UNIX)) != NULL) {
		(void) strncpy(filepath, p, sizeof(filepath)-1);
		filepath[sizeof(filepath)-1] = '\0';
		MEM_FREE(p);
	}
#else
	(void) strcpy(filepath, "../../bkgnd.gif");
#endif
	(void) fprintf(fp,
		"</HEAD>\n<BODY BGCOLOR=\"%s\" BACKGROUND=\"%s\">\n",
		"#FFFFFF",		/* Background color */
		filepath
	);

	(void) fputs("<DIV ALIGN=\"center\">\n", fp);

	/* xmcd logo */
	(void) fprintf(fp, "<A HREF=\"%s\">\n", XMCD_URL);
#ifdef __VMS
	(void) sprintf(filepath, "%s.discog]xmcdlogo.gif", app_data.libdir);
	if ((p = wwwwarp_vms_urlconv(filepath, VMS_2_UNIX)) != NULL) {
		(void) strncpy(filepath, p, sizeof(filepath)-1);
		filepath[sizeof(filepath)-1] = '\0';
		MEM_FREE(p);
	}
#else
	(void) strcpy(filepath, "../../xmcdlogo.gif");
#endif
	(void) fprintf(fp, "<IMG SRC=\"%s\" ALT=\"%s\" BORDER=\"0\">",
		filepath, PROGNAME);
	(void) fprintf(fp, "</A><P>\n<H4>%s</H4><P>\n",
		app_data.str_localdiscog);

	/* Disc artist / title */
	if (dbp->dtitle == NULL || dbp->dtitle[0] == '\0') {
		(void) fprintf(fp, "<H3>(%s)</H3>\n", app_data.str_unkndisc);
	}
	else {
		(void) fputs("<H3>\n", fp);
		if (srchact != NULL) {
			p = wwwwarp_txtreduce(dbp->dtitle, TRUE),
			(void) fprintf(fp, "<A HREF=\"%s%s\">",
				srchact,
				p
			);
			MEM_FREE(p);
		}
		wwwwarp_html_fputs(dbp->dtitle, fp, FALSE, NULL, 0);
		if (srchact != NULL)
			(void) fputs("</A>\n", fp);
		(void) fputs("</H3>\n", fp);
	}

	/* Disc ID */
	(void) fprintf(fp, "<P>\n<TABLE %s>\n", tblparms1);
	(void) fprintf(fp,
		"<TR><TH %s>Disc ID:</TH><TD %s>%s %08x<BR></TD></TR>\n",
		tdparms2, tdparms2, dbp->category, dbp->discid);
	(void) fprintf(fp,
		    "<TR><TH %s>Total:</TH><TD %s>%02d:%02d<BR></TD></TR>\n",
		    tdparms2, tdparms2, s->tot_min, s->tot_sec);
	(void) fputs("</TABLE>\n", fp);

	/* Tracks */
	(void) fprintf(fp, "<P>\n<TABLE %s>\n", tblparms2);
	(void) fprintf(fp, "<TR>\n<TD %s>Track</TD>\n", tdparms1);
	(void) fprintf(fp, "<TD %s>Start</TD>\n", tdparms1);
	(void) fprintf(fp, "<TD %s>Length</TD>\n", tdparms1);
	(void) fprintf(fp, "<TD %s>Description<BR></TD>\n</TR>\n", tdparms1);

	for (i = 0; i < (int) s->tot_trks; i++) {
		int	min,
			sec,
			secs;

		secs = ((s->trkinfo[i+1].min * 60 + s->trkinfo[i+1].sec) -
			(s->trkinfo[i].min * 60 + s->trkinfo[i].sec));
		min = (byte_t) (secs / 60);
		sec = (byte_t) (secs % 60);

		(void) fputs("<TR>\n", fp);
		(void) fprintf(fp, "<TD %s>%d</TD>\n",
			tdparms1,
			s->trkinfo[i].trkno
		);
		(void) fprintf(fp, "<TD %s>%02d:%02d</TD>\n",
			tdparms1, s->trkinfo[i].min, s->trkinfo[i].sec);
		(void) fprintf(fp, "<TD %s>%02d:%02d</TD>\n",
			tdparms1, min, sec);
		if (dbp->trklist[i] == NULL || dbp->trklist[i][0] == '\0') {
			(void) fprintf(fp, "<TD %s><B>(%s)</B><BR></TD>\n",
				tdparms2,
				app_data.str_unkntrk
			);
		}
		else {
			(void) fprintf(fp, "<TD %s><B>", tdparms2);
			if (srchact != NULL) {
				p = wwwwarp_txtreduce(dbp->trklist[i], TRUE),
				(void) fprintf(fp, "<A HREF=\"%s%s\">",
					srchact,
					p
				);
				MEM_FREE(p);
			}
			wwwwarp_html_fputs(
				dbp->trklist[i],
				fp,
				FALSE,
				NULL,
				0
			);
			if (srchact != NULL)
				(void) fputs("</A>", fp);
			(void) fputs("</B><BR></TD>\n", fp);
		}
		(void) fputs("</TR>\n", fp);
	}

	(void) fputs("</TABLE>\n", fp);

	/* end of <DIV ALIGN="center"> */
	(void) fputs("</DIV>\n<P>\n", fp);

	/* Extended disc info */
	if (dbp->extd != NULL && dbp->extd[0] != '\0') {
		(void) fputs(
			"<H4>Disc Extended Information</H4>\n<P>\n<DIR>\n",
			fp
		);
		wwwwarp_html_fputs(dbp->extd, fp, TRUE, "courier", -1);
		(void) fputs("\n</DIR>\n", fp);
	}

	/* Extended track info */
	first = TRUE;
	for (i = 0; i < (int) s->tot_trks; i++) {
		if (dbp->extt[i] != NULL && dbp->extt[i][0] != '\0') {
			if (first) {
			    first = FALSE;
			    (void) fputs(
				    "<H4>Track Extended Information</H4>\n",
				    fp
			    );
			}
			(void) fputs("<DIR>\n", fp);
			(void) fprintf(fp, "<P>\n<B>Track %d: ",
			    s->trkinfo[i].trkno);
			wwwwarp_html_fputs(
				(dbp->trklist[i] == NULL ||
				 dbp->trklist[i][0] == '\0') ?
				    app_data.str_unkntrk : dbp->trklist[i],
				fp,
				FALSE,
				NULL,
				0
			);
			(void) fputs("</B>\n<P>\n<DIR>\n", fp);
			wwwwarp_html_fputs(
				dbp->extt[i],
				fp,
				TRUE,
				"courier",
				-1
			);
			(void) fputs("\n</DIR>\n</DIR>\n", fp);
		}
	}

	/* Local discography */
	(void) fputs("<H4>Local Discography</H4>\n<P>\n<UL>\n", fp);

	DBGPRN(errfp, "Checking directory: %s\n", outdir);

	if (wwwwarp_urlchk(baseurl, &p) != IS_REMOTE_URL &&
	    (dp = OPENDIR(outdir)) != NULL) {
		while ((de = READDIR(dp)) != NULL) {
			if (strcmp(de->d_name, ".") == 0 ||
			    strcmp(de->d_name, "..") == 0 ||
			    util_strcasecmp(de->d_name,
					    util_basename(baseurl)) == 0)
				continue;
#ifdef __VMS
			/* Discard version number */
			if ((p = strrchr(de->d_name, ';')) != NULL)
				*p = '\0';
#endif

			DBGPRN(errfp, "\tAdding to Local Discography: %s\n",
				de->d_name);

#ifdef __VMS
			p = wwwwarp_vms_urlconv(outdir, VMS_2_UNIX);
			if (p != NULL) {
				(void) strncpy(filepath, p,
					       sizeof(filepath)-1);
				filepath[sizeof(filepath)-1] = '\0';
				MEM_FREE(p);
			}
#else
			filepath[0] = '\0';
#endif
			(void) fprintf(fp,
				"<LI><A HREF=\"%s%s\">%s: %s</A></LI>\n",
				filepath,
				de->d_name,
				app_data.str_file,
				de->d_name
			);
		}

		(void) CLOSEDIR(dp);
	}

	if ((pp = cddb_pathlist()) != NULL) {
		first_categ = TRUE;

		for (; pp != NULL; pp = pp->next) {
			if (pp->category == NULL)
				continue;

#ifdef __VMS
			(void) sprintf(filepath, "%s.discog.%s]index.html",
				app_data.libdir,
				pp->category
			);
			p = wwwwarp_vms_urlconv(filepath, VMS_2_UNIX);
			if (p != NULL) {
				(void) strncpy(filepath, p,
					sizeof(filepath)-1);
				filepath[sizeof(filepath)-1] = '\0';
				MEM_FREE(p);
			}
#else
			(void) sprintf(filepath, "../../%s/index.html",
				pp->category);
#endif
			if (first_categ) {
				first_categ = FALSE;
				(void) fprintf(fp, "<LI>%s:\n<UL>\n",
					app_data.str_browse_categ);
			}

			(void) fprintf(fp, "<LI><A HREF=\"%s\">%s</A></LI>\n",
				filepath,
				pp->category
			);
		}

		if (!first_categ)
			(void) fputs("</UL>\n</LI>\n", fp);
	}

#ifdef __VMS
	(void) sprintf(filepath, "%s.discog]discog.html", app_data.libdir);
	if ((p = wwwwarp_vms_urlconv(filepath, VMS_2_UNIX)) != NULL) {
		(void) strncpy(filepath, p, sizeof(filepath)-1);
		filepath[sizeof(filepath)-1] = '\0';
		MEM_FREE(p);
	}
#else
	(void) strcpy(filepath, "../../discog.html");
#endif
	(void) fprintf(fp, "<LI><A HREF=\"%s\">%s %s</A></LI>\n</UL>\n",
		filepath,
		app_data.str_howto_setup,
		app_data.str_localdiscog);

	/* Links */
	(void) fputs("<H4>Links</H4>\n<P>\n<UL>\n", fp);

	for (i = 0; i < 3; i++) {
		for (t = di_sites_list(); t != NULL; t = t->next) {
			if (t->idx == i)
				break;
		}

		if (t != NULL && t->action != NULL && t->aux != NULL) {
			XtVaGetValues((Widget) t->aux,
				XmNlabelString, &xs,
				NULL
			);
			ret = XmStringGetLtoR(
				xs,
				XmSTRING_DEFAULT_CHARSET,
				&name
			);

			if (!ret)
				name = t->name;

			if ((p = wwwwarp_gen_url(s, t)) != NULL) {
				(void) fprintf(fp,
					"<LI><A HREF=\"%s\">%s</A></LI>\n",
					p, name
				);

				MEM_FREE(p);
			}
		}
	}
	(void) fputs("</UL>\n</BODY>\n</HTML>\n", fp);

	if (srchact != NULL)
		MEM_FREE(srchact);
}


/*
 * wwwwarp_gen_discogidx
 *	Output local discographies index file for the current CD.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *	baseurl - The URL to the output file.
 *	outfile - The path name to the output file.
 *
 * Return:
 *	0 for success, errno for failure.
 */
STATIC int
wwwwarp_gen_discogidx(curstat_t *s, char *baseurl, char *outfile)
{
	int		ret;
	unsigned int	dmode,
			fmode;
	FILE		*fp;
	char		*p,
			outdir[FILE_PATH_SZ],
			catdir[FILE_PATH_SZ];
#ifndef __VMS
	pid_t		cpid;
	waitret_t	stat_val;

	switch (cpid = FORK()) {
	case 0:
		/* Child process */
		break;

	case -1:
		/* Fork failed */
		return (errno);

	default:
		/* Parent process: wait for child to exit */
		while ((ret = WAITPID(cpid, &stat_val, 0)) != cpid) {
			if (ret < 0) {
				DBGPRN(errfp, "waitpid() failed (errno=%d)\n",
				       errno);
				return 0;
			}

			/* Handle some X events */
			event_loop(0);
		}

		if (WIFEXITED(stat_val))
			ret = WEXITSTATUS(stat_val);
		else
			ret = 0xff;

		return (ret);
	}

	/* Force uid and gid to original setting */
	if (!util_set_ougid())
		exit(errno);
#endif	/* __VMS */

	(void) sscanf(app_data.cddb_filemode, "%o", &fmode);
	/* Make sure file is at least accessible by user */
	fmode |= S_IRUSR | S_IWUSR;
	fmode &= ~(S_ISUID | S_ISGID);

	/* Set directory perm based on file perm */
	dmode = (fmode | S_IXUSR);
	if (fmode & S_IRGRP)
		dmode |= (S_IRGRP | S_IXGRP);
	if (fmode & S_IWGRP)
		dmode |= (S_IWGRP | S_IXGRP);
	if (fmode & S_IROTH)
		dmode |= (S_IROTH | S_IXOTH);
	if (fmode & S_IWOTH)
		dmode |= (S_IWOTH | S_IXOTH);

	if ((p = util_dirname(outfile)) != NULL) {
		(void) strcpy(outdir, p);
		if ((p = util_dirname(outdir)) != NULL)
			(void) strcpy(catdir, p);
		else
			catdir[0] = '\0';
	}
	else {
		DBGPRN(errfp, "Directory path error: %s\n", outdir);
#ifdef __VMS
		return EINVAL;
#else
		exit(EINVAL);
#endif
	}

	/* Make directories */
	if (catdir[0] != '\0')
		(void) mkdir(catdir, (mode_t) dmode);
	if ((ret = mkdir(outdir, (mode_t) dmode)) < 0 && errno != EEXIST) {
		ret = errno;
		DBGPRN(errfp, "Cannot create directory %s (errno=%d)\n",
			outdir, errno);
#ifdef __VMS
		return (ret);
#else
		exit(ret);
		/*NOTREACHED*/
#endif	/* __VMS */
	}
	if (ret == 0) {
		DBGPRN(errfp, "Created directory: %s\n", outdir);
	}
	(void) chmod(outdir, (mode_t) dmode);

	(void) unlink(outfile);

	if ((fp = fopen(outfile, "w")) == NULL) {
		ret = errno;
		DBGPRN(errfp, "Cannot open file for writing: %s\n", outfile);
		return (ret);
	}
	(void) chmod(outfile, (mode_t) fmode);

	DBGPRN(errfp, "Writing Local Discography: %s\n", outfile);

	/* Write out HTML file */
	wwwwarp_db2html(fp, s, baseurl, outdir);

	(void) fclose(fp);

#ifdef __VMS
	return (0);
#else
	exit(0);
	/*NOTREACHED*/
#endif	/* __VMS */
}


/*
 * wwwwarp_gen_url
 *	Generate an appropriate URL from the template.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *	t - Pointer to the site's urltmpl_ent_t structure.
 *
 * Return:
 *	Pointer to The URL string.  The string buffer should be freed
 *	via MEM_FREE by the caller.
 */
STATIC char *
wwwwarp_gen_url(curstat_t *s, urltmpl_ent_t *t)
{
	int		n,
			trkpos;
	char		*url,
			*path;
	int		sel_pos;

	sel_pos = dbprog_curseltrk(s);

	if (sel_pos > 0)
		/* A track is selected: use it for the track title */
		trkpos = sel_pos - 1;
	else
		/* Use current playing track for the track title, if playing */
		trkpos = di_curtrk_pos(s);

	n = wwwwarp_url_len(t->action, &t->a_attrib, &trkpos, s);

	url = (char *) MEM_ALLOC("wwwwarp_genurl_url", n);
	if (url == NULL) {
		CD_FATAL(app_data.str_nomemory);
		return NULL;
	}

	/* Make the URL from template */
	wwwwarp_tmpl_to_url(t->action, url, trkpos, s);

	if (wwwwarp_urlchk(url, &path) != IS_REMOTE_URL) {
		if (t->idx == WWWWARP_DISCOG)
			/* Generate new local discography index page */
			(void) wwwwarp_gen_discogidx(s, url, path);

		if (access(path, 0 /* F_OK */) < 0 && t->fallback != NULL) {
			/* Cannot access local doc: try fallback */

			DBGPRN(errfp, "File not accessible: %s\n", url);
			MEM_FREE(url);

			n = wwwwarp_url_len(
				t->fallback,
				&t->f_attrib,
				&trkpos,
				s
			);

			url = (char *) MEM_ALLOC("wwwwarp_genurl_url", n);
			if (url == NULL) {
				CD_FATAL(app_data.str_nomemory);
				return NULL;
			}

			/* Make the URL from template */
			wwwwarp_tmpl_to_url(t->fallback, url,
					    trkpos, s);

			DBGPRN(errfp, "Using fallback: %s\n", url);
		}
	}

	return (url);
}


/***********************
 *   public routines   *
 ***********************/


/*
 * wwwwarp_init
 *	Initialize the wwwWarp subsystem.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
void
wwwwarp_init(curstat_t *s)
{
	int		i;
	Widget		w;
	XmString	xs,
			xs2,
			xs3;
	urltmpl_ent_t	*p;
	Arg		arg[10];
	char		str[12];
	bool_t		first;

	/* Set up web sites list */
	for (p = di_sites_list(); p != NULL; p = p->next) {
		switch (p->idx) {
		case WWWWARP_1:
			p->aux = (void *) widgets.wwwwarp.site1_btn;
			break;
		case WWWWARP_2:
			p->aux = (void *) widgets.wwwwarp.site2_btn;
			break;
		case WWWWARP_3:
			p->aux = (void *) widgets.wwwwarp.site3_btn;
			break;
		case WWWWARP_DISCOG:
			p->aux = (void *) widgets.wwwwarp.discog_btn;
			break;
		default:
			DBGPRN(errfp,
				"INTERNAL ERROR: idx=%d %s on di_sites_list\n",
				p->idx, p->name != NULL ? p->name : "unkn");
			break;
		}
	}

	/* Set up search engines list */
	first = TRUE;
	for (p = di_srcheng_list(); p != NULL; p = p->next) {
		if (strcmp(p->name, "-") == 0) {
			/* Separator line */
			i = 0;
			XtSetArg(arg[i], XmNseparatorType,
				 XmSINGLE_DASHED_LINE); i++;
			w = XmCreateSeparator(
				widgets.wwwwarp.srchsite_menu,
				"menuSeparator",
				arg,
				i
			);
			XtManageChild(w);
			continue;
		}

		/* Real menu entry */
		(void) sprintf(str, "[%s%s%s]   ",
			(p->a_attrib.acnt > 0) ? "a" : "-",
			(p->a_attrib.dcnt > 0) ? "d" : "-",
			(p->a_attrib.tcnt > 0) ? "t" : "-");
		xs2 = XmStringCreateLtoR(str, CHSET3);
		xs3 = XmStringCreateLtoR(p->name, CHSET2);
		xs = XmStringConcat(xs2, xs3);

		i = 0;
		XtSetArg(arg[i], XmNlabelString, xs); i++;
		w = XmCreatePushButton(
			widgets.wwwwarp.srchsite_menu,
			p->name,
			arg,
			i
		);
		XmStringFree(xs);
		XmStringFree(xs2);
		XmStringFree(xs3);
		XtManageChild(w);

		p->idx = WWWWARP_SRCHENG;
		p->aux = (void *) w;

		if (first) {
			XtVaSetValues(widgets.wwwwarp.srchsite_opt,
				XmNmenuHistory, w,
				NULL
			);

			/* Set current selection to the first entry */
			cur_srcheng = p;
			first = FALSE;
		}

		XtAddCallback(w,
			XmNactivateCallback,
			(XtCallbackProc) wwwwarp_srcheng_sel,
			(XtPointer) p
		);
	}

	wwwwarp_initted = TRUE;

	/* Configure the wwwWarp menu entries */
	wwwwarp_sel_cfg(s);
}


/*
 * wwwwarp_sel_cfg
 *	Set the sensitivity of the wwwWarp menu entries
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 * 
 * Return:
 *	Nothing.
 */
void
wwwwarp_sel_cfg(curstat_t *s)
{
	urltmpl_ent_t	*p;
	cddb_incore_t	*dbp;
	Boolean		sens,
			srchsens,
			gosens1,
			gosens2;

	if (!wwwwarp_initted)
		return;

	dbp = dbprog_curdb(s);
	gosens1 = gosens2 = False;

	for (p = di_sites_list(); p != NULL; p = p->next) {
		sens = True;

		if ((p->a_attrib.acnt > 0 || p->a_attrib.dcnt > 0 ||
		     p->a_attrib.tcnt > 0) &&
		    (dbp->dtitle == NULL || dbp->dtitle[0] == '\0'))
			sens = False;

		if (p->a_attrib.ccnt > 0 && dbp->category[0] == '\0')
			sens = False;

		if (p->a_attrib.icnt > 0 && dbp->discid == 0)
			sens = False;

		XtSetSensitive(p->aux, sens);

		if (website_idx == p->idx)
			gosens1 = sens;
	}

	srchsens = False;
	for (p = di_srcheng_list(); p != NULL; p = p->next) {
		if (strcmp(p->name, "-") == 0)
			continue;

		sens = True;

		if ((p->a_attrib.acnt > 0 || p->a_attrib.dcnt > 0 ||
		     p->a_attrib.tcnt > 0) &&
		    (dbp->dtitle == NULL || dbp->dtitle[0] == '\0'))
			sens = False;

		if (p->a_attrib.ccnt > 0 && dbp->category[0] == '\0')
			sens = False;

		if (p->a_attrib.icnt > 0 && dbp->discid == 0)
			sens = False;

		XtSetSensitive(p->aux, sens);

		if (sens)
			srchsens = True;

		if (website_idx == WWWWARP_SRCHENG && cur_srcheng == p)
			gosens2 = sens;
	}

	XtSetSensitive(widgets.wwwwarp.srchweb_btn, srchsens);
	XtSetSensitive(widgets.wwwwarp.go_btn, (Boolean) (gosens1 || gosens2));
}


/**************** vv Callback routines vv ****************/


/*
 * wwwwarp_popup
 *	wwwWarp button callback function
 */
void
wwwwarp_popup(Widget w, XtPointer client_data, XtPointer call_data)
{
	if (XtIsManaged(widgets.wwwwarp.form)) {
		/* Pop wwwwarp window down */
		wwwwarp_popdown(w, client_data, call_data);
		return;
	}

	/* Pop wwwWarp window up */
	XtManageChild(widgets.wwwwarp.form);

	/* Set keyboard focus on the Go button */
	XmProcessTraversal(
		widgets.wwwwarp.go_btn,
		XmTRAVERSE_CURRENT
	);
}


/*
 * wwwwarp_popdown
 *	wwwWarp cancel button callback function
 */
/*ARGSUSED*/
void
wwwwarp_popdown(Widget w, XtPointer client_data, XtPointer call_data)
{
	if (XtIsManaged(widgets.wwwwarp.form))
		XtUnmanageChild(widgets.wwwwarp.form);
}


/*
 * wwwwarp_select
 *	wwwWarp selection callback function
 */
/*ARGSUSED*/
void
wwwwarp_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XmRowColumnCallbackStruct	*p =
		(XmRowColumnCallbackStruct *)(void *) call_data;
	curstat_t			*s = (curstat_t *)(void *) client_data;
	XmToggleButtonCallbackStruct	*q;

	if (p == NULL)
		return;

	q = (XmToggleButtonCallbackStruct *)(void *) p->callbackstruct;

	if (!q->set)
		return;

	if (p->widget == widgets.wwwwarp.site1_btn)
		website_idx = WWWWARP_1;
	else if (p->widget == widgets.wwwwarp.site2_btn)
		website_idx = WWWWARP_2;
	else if (p->widget == widgets.wwwwarp.site3_btn)
		website_idx = WWWWARP_3;
	else if (p->widget == widgets.wwwwarp.discog_btn)
		website_idx = WWWWARP_DISCOG;
	else if (p->widget == widgets.wwwwarp.srchweb_btn)
		website_idx = WWWWARP_SRCHENG;

	DBGPRN(errfp, "\n* WWWWARP selected %s (idx=%d)\n",
		XtName(p->widget), website_idx);

	if (website_idx == WWWWARP_SRCHENG) {
		XtSetSensitive(widgets.wwwwarp.srchsite_opt, True);

		/* Configure the wwwWarp menu entries */
		wwwwarp_sel_cfg(s);
	}
	else {
		XtSetSensitive(widgets.wwwwarp.srchsite_opt, False);
		XtSetSensitive(widgets.wwwwarp.go_btn, True);
	}
}


/*
 * wwwwarp_srcheng_sel
 *	wwwWarp search engine selector menu callback function
 */
/*ARGSUSED*/
void
wwwwarp_srcheng_sel(Widget w, XtPointer client_data, XtPointer call_data)
{
	urltmpl_ent_t	*p = (urltmpl_ent_t *)(void *) client_data;

	if (w != (Widget) p->aux)
		return;

	cur_srcheng = p;
	XtSetSensitive(widgets.wwwwarp.go_btn, True);

	DBGPRN(errfp, "\n* WWWWARP search engine set to \"%s\"\n", p->name);
}


/*
 * wwwwarp_go
 *	wwwWarp OK button callback function
 */
/*ARGSUSED*/
void
wwwwarp_go(Widget w, XtPointer client_data, XtPointer call_data)
{
	curstat_t	*s = (curstat_t *)(void *) client_data;
	urltmpl_ent_t	*t;
	char		*url;

	DBGPRN(errfp, "\n* WWWWARP Go!\n");

	/* Change to watch cursor */
	cd_busycurs(TRUE, CURS_WWWWARP);

	if (website_idx == WWWWARP_SRCHENG)
		t = cur_srcheng;
	else {
		for (t = di_sites_list(); t != NULL; t = t->next) {
			if (t->idx == website_idx)
				break;
		}
	}

	if (t != NULL && t->action != NULL &&
	    (url = wwwwarp_gen_url(s, t)) != NULL) {
		/* Go to the URL */
		wwwwarp_site(url);

		MEM_FREE(url);
	}
	else
		cd_beep();

	/* Change to normal cursor */
	cd_busycurs(FALSE, CURS_WWWWARP);
}


/**************** ^^ Callback routines ^^ ****************/

