// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.exportPath('multidevice_setup');

/** @enum {string} */
multidevice_setup.PageName = {
  FAILURE: 'setup-failed-page',
  PASSWORD: 'password-page',
  SUCCESS: 'setup-succeeded-page',
  START: 'start-setup-page',
};

cr.define('multidevice_setup', function() {
  const PageName = multidevice_setup.PageName;

  const MultiDeviceSetup = Polymer({
    is: 'multidevice-setup',

    behaviors: [WebUIListenerBehavior],

    properties: {
      /**
       * Delegate object which performs differently in OOBE vs. non-OOBE mode.
       * @type {!multidevice_setup.MultiDeviceSetupDelegate}
       */
      delegate: Object,

      /**
       * Text to be shown on the forward navigation button.
       * @type {string|undefined}
       */
      forwardButtonText: {
        type: String,
        computed: 'getForwardButtonText_(visiblePage_)',
        notify: true,
      },

      /** Whether the forward button should be disabled. */
      forwardButtonDisabled: {
        type: Boolean,
        computed: 'shouldForwardButtonBeDisabled_(' +
            'passwordPageForwardButtonDisabled_, visiblePageName_)',
        notify: true
      },

      /**
       * Text to be shown on the cancel button.
       * @type {string|undefined}
       */
      cancelButtonText: {
        type: String,
        computed: 'getCancelButtonText_(visiblePage_)',
        notify: true,
      },

      /**
       * Text to be shown on the backward navigation button.
       * @type {string|undefined}
       */
      backwardButtonText: {
        type: String,
        computed: 'getBackwardButtonText_(visiblePage_)',
        notify: true,
      },

      /**
       * Element name of the currently visible page.
       *
       * @private {!multidevice_setup.PageName}
       */
      visiblePageName_: {
        type: String,
        value: PageName.START,
        notify: true,  // For testing purposes only.
      },

      /**
       * DOM Element corresponding to the visible page.
       *
       * @private {!PasswordPageElement|!StartSetupPageElement|
       *           !SetupSucceededPageElement|!SetupFailedPageElement}
       */
      visiblePage_: Object,

      /**
       * Authentication token, which is generated by the password page.
       * @private {string}
       */
      authToken_: {
        type: String,
      },

      /**
       * Array of objects representing all potential MultiDevice hosts.
       *
       * @private {!Array<!chromeos.deviceSync.mojom.RemoteDevice>}
       */
      devices_: Array,

      /**
       * Unique identifier for the currently selected host device.
       *
       * Undefined if the no list of potential hosts has been received from mojo
       * service.
       *
       * @private {string|undefined}
       */
      selectedDeviceId_: String,

      /**
       * Whether the password page reports that the forward button should be
       * disabled. This field is only relevant when the password page is
       * visible.
       * @private {boolean}
       */
      passwordPageForwardButtonDisabled_: Boolean,

      /**
       * Provider of an interface to the MultiDeviceSetup Mojo service.
       * @private {!multidevice_setup.MojoInterfaceProvider}
       */
      mojoInterfaceProvider_: Object
    },

    listeners: {
      'backward-navigation-requested': 'onBackwardNavigationRequested_',
      'cancel-requested': 'onCancelRequested_',
      'forward-navigation-requested': 'onForwardNavigationRequested_',
    },

    /** @override */
    created: function() {
      this.mojoInterfaceProvider_ =
          multidevice_setup.MojoInterfaceProviderImpl.getInstance();
    },

    /** @override */
    ready: function() {
      this.addWebUIListener(
          'multidevice_setup.initializeSetupFlow',
          this.initializeSetupFlow.bind(this));
    },

    initializeSetupFlow: function() {
      this.mojoInterfaceProvider_.getInterfacePtr()
          .getEligibleHostDevices()
          .then((responseParams) => {
            if (responseParams.eligibleHostDevices.length == 0) {
              console.warn('Potential host list is empty.');
              return;
            }

            this.devices_ = responseParams.eligibleHostDevices;
          })
          .catch((error) => {
            console.warn('Mojo service failure: ' + error);
          });
    },

    /** @private */
    onCancelRequested_: function() {
      this.exitSetupFlow_();
    },

    /** @private */
    onBackwardNavigationRequested_: function() {
      // The back button is only visible on the password page.
      assert(this.visiblePageName_ == PageName.PASSWORD);

      this.$$('password-page').clearPasswordTextInput();
      this.visiblePageName_ = PageName.START;
    },

    /** @private */
    onForwardNavigationRequested_: function() {
      if (this.forwardButtonDisabled)
        return;

      this.visiblePage_.getCanNavigateToNextPage().then((canNavigate) => {
        if (!canNavigate)
          return;
        this.navigateForward_();
      });
    },

    /** @private */
    navigateForward_: function() {
      switch (this.visiblePageName_) {
        case PageName.FAILURE:
          this.visiblePageName_ = PageName.START;
          return;
        case PageName.PASSWORD:
          this.$$('password-page').clearPasswordTextInput();
          this.setHostDevice_();
          return;
        case PageName.SUCCESS:
          this.exitSetupFlow_();
          return;
        case PageName.START:
          if (this.delegate.isPasswordRequiredToSetHost())
            this.visiblePageName_ = PageName.PASSWORD;
          else
            this.setHostDevice_();
          return;
      }
    },

    /** @private */
    setHostDevice_: function() {
      // An authentication token must be set if a password is required.
      assert(this.delegate.isPasswordRequiredToSetHost() == !!this.authToken_);

      let deviceId = /** @type {string} */ (this.selectedDeviceId_);
      this.delegate.setHostDevice(deviceId, this.authToken_)
          .then((responseParams) => {
            if (!responseParams.success) {
              console.warn('Failure setting host with device ID: ' + deviceId);
              return;
            }

            if (this.delegate.shouldExitSetupFlowAfterSettingHost()) {
              this.exitSetupFlow_();
              return;
            }

            this.visiblePageName_ = PageName.SUCCESS;
            this.fire('forward-button-focus-requested');
          })
          .catch((error) => {
            console.warn('Mojo service failure: ' + error);
          });
    },

    /** @private */
    onUserSubmittedPassword_: function() {
      this.onForwardNavigationRequested_();
    },

    /**
     * @return {string|undefined} The forward button text, which is undefined
     *     if no button should be displayed.
     * @private
     */
    getForwardButtonText_: function() {
      if (!this.visiblePage_)
        return undefined;
      return this.visiblePage_.forwardButtonText;
    },

    /**
     * @return {boolean} Whether the forward button should be disabled.
     * @private
     */
    shouldForwardButtonBeDisabled_: function() {
      return (this.visiblePageName_ == PageName.PASSWORD) &&
          this.passwordPageForwardButtonDisabled_;
    },

    /**
     * @return {string|undefined} The cancel button text, which is undefined
     *     if no button should be displayed.
     * @private
     */
    getCancelButtonText_: function() {
      if (!this.visiblePage_)
        return undefined;
      return this.visiblePage_.cancelButtonText;
    },

    /**
     * @return {string|undefined} The backward button text, which is undefined
     *     if no button should be displayed.
     * @private
     */
    getBackwardButtonText_: function() {
      if (!this.visiblePage_)
        return undefined;
      return this.visiblePage_.backwardButtonText;
    },

    /**
     * @return {boolean}
     * @private
     */
    shouldPasswordPageBeIncluded_: function() {
      return this.delegate.isPasswordRequiredToSetHost();
    },

    /**
     * @return {boolean}
     * @private
     */
    shouldSetupSucceededPageBeIncluded_: function() {
      return !this.delegate.shouldExitSetupFlowAfterSettingHost();
    },

    /**
     * Notifies observers that the setup flow has completed.
     *
     * @private
     */
    exitSetupFlow_: function() {
      this.fire('setup-exited');
    },
  });

  return {
    MultiDeviceSetup: MultiDeviceSetup,
  };
});
