// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/browsing_data/browsing_data_indexed_db_helper.h"

#include "base/strings/utf_string_conversions.h"
#include "base/threading/thread_task_runner_handle.h"
#include "chrome/test/base/testing_profile.h"
#include "content/public/browser/browser_context.h"
#include "content/public/browser/storage_partition.h"
#include "content/public/test/test_browser_thread_bundle.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace {

class CannedBrowsingDataIndexedDBHelperTest : public testing::Test {
 public:
  content::IndexedDBContext* IndexedDBContext() {
    return content::BrowserContext::GetDefaultStoragePartition(&profile_)->
        GetIndexedDBContext();
  }

 private:
  content::TestBrowserThreadBundle thread_bundle_;
  TestingProfile profile_;
};

TEST_F(CannedBrowsingDataIndexedDBHelperTest, Empty) {
  const GURL origin("http://host1:1/");
  scoped_refptr<CannedBrowsingDataIndexedDBHelper> helper(
      new CannedBrowsingDataIndexedDBHelper(IndexedDBContext()));

  ASSERT_TRUE(helper->empty());
  helper->AddIndexedDB(origin);
  ASSERT_FALSE(helper->empty());
  helper->Reset();
  ASSERT_TRUE(helper->empty());
}

TEST_F(CannedBrowsingDataIndexedDBHelperTest, Delete) {
  const GURL origin1("http://host1:9000");
  const GURL origin2("http://example.com");

  scoped_refptr<CannedBrowsingDataIndexedDBHelper> helper(
      new CannedBrowsingDataIndexedDBHelper(IndexedDBContext()));

  EXPECT_TRUE(helper->empty());
  helper->AddIndexedDB(origin1);
  helper->AddIndexedDB(origin2);
  EXPECT_EQ(2u, helper->GetIndexedDBCount());
  helper->DeleteIndexedDB(origin2);
  EXPECT_EQ(1u, helper->GetIndexedDBCount());
}

TEST_F(CannedBrowsingDataIndexedDBHelperTest, IgnoreExtensionsAndDevTools) {
  const GURL origin1("chrome-extension://abcdefghijklmnopqrstuvwxyz/");
  const GURL origin2("chrome-devtools://abcdefghijklmnopqrstuvwxyz/");

  scoped_refptr<CannedBrowsingDataIndexedDBHelper> helper(
      new CannedBrowsingDataIndexedDBHelper(IndexedDBContext()));

  ASSERT_TRUE(helper->empty());
  helper->AddIndexedDB(origin1);
  ASSERT_TRUE(helper->empty());
  helper->AddIndexedDB(origin2);
  ASSERT_TRUE(helper->empty());
}

}  // namespace
