/*
 *  Copyright (C) 2001 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __bookmarks_io_h
#define __bookmarks_io_h

#include "bookmarks.h"
#include "gfilepicker.h"

typedef struct
{
	/**
	 * The name of the file format, as the user will see it
	 */
	gchar *format_name;

	/**
	 * tipical suffixes of files
	 */ 
	gchar **extensions;

	/**
	 * Load bookmarks from a file
	 * @filename: where to load bookmarks from
	 * @new_default_folder: if != NULL it is set to the new default folder
	 */
	BookmarkItem *(*load_from_file) (const gchar *filename, 
					 BookmarkItem **new_default_folder);
	/**
	 * Load bookmarks from a string
	 * @data: where to load files from
	 * @new_default_folder: if != NULL it is set to the new default folder
	 */
	BookmarkItem *(*load_from_string) (const gchar *data, 
					   BookmarkItem **new_default_folder);

	/**
	 * Save the bookmarks to a file
	 * @file: filename to save the bookmarks to. It will be overwritten
	 * @default_folder: This folder is marked as the default folder
	 * @retuns TRUE if succeds
	 */
	gboolean (*save_to_file) (BookmarkItem *root,
				  BookmarkItem *default_folder,
				  const gchar *file);
				    
	/**
	 * Save the bookmarks to a string
	 * @data: it is set to the buffer containing the saved data (needs
	 *        to be freed)
	 * @default_folder: This folder is marked as the default folder
	 * @retuns TRUE if succeds
	 */
	gboolean (*save_to_string) (BookmarkItem *root,
				    BookmarkItem *default_folder,
				    gchar **data);
				    
} BookmarkTranslator;

/**
 * List of avaible BookmarkTranslators
 */
extern GList *bookmarks_translators;
extern BookmarkTranslator *bookmarks_default_translator;

/**
 * private info and functions needed by loaders and savers 
 */

/**
 * Struct useful to loaders only
 */
typedef struct 
{
	/** 
	 * Aliases that have been loaded and are to be connected with 
	 * their bookmarks
	 */
	GList *unresolved_aliases; /* list of UnresolvedAliasInfo* */
	
	/**
	 * Map from ids (as in the file) to loaded bookmarks
	 */
	GHashTable *id_bookmark;

	/** 
	 * The root of the loaded bookmarks
	 */
	BookmarkItem *root;

	/**
	 * The folder that was marked as the default_bookmarks_root
	 */
	BookmarkItem *default_bookmarks_root;
} BookmarkLoaderInfo;

typedef struct
{
	BookmarkItem *alias;
	gulong alias_of_id;
} UnresolvedAliasInfo;	

BookmarkLoaderInfo *bookmarks_io_loader_info_new (void);
void bookmarks_io_loader_info_free (BookmarkLoaderInfo *li);
UnresolvedAliasInfo *
bookmarks_io_unresolved_alias_info_new (BookmarkItem *unresolved,
					guint alias_of_id);
void bookmarks_io_unresolved_alias_info_free (UnresolvedAliasInfo *uai);
void bookmarks_io_resolve_loaded_alias (BookmarkLoaderInfo *li);
BookmarkItem *bookmarks_io_error_bookmarks_new (const gchar *msg_title, 
						const gchar *msg_notes);
FileFormat *bookmarks_io_get_fileformats (void);

#endif
