/*!**************************************************************************

  module      : Kernel_FileIO.hpp

  -------------------------------------------------------------------------

  responsible : TorstenS

  special area: 
  description : This module is used to write a kernel page into the 
                filesystem of the operating system.


  last changed: 2001-05-30  15:43
  see also    : 

  -------------------------------------------------------------------------

  copyright:    (c) 2001-2004 SAP AG

  ========== licence begin  GPL
  Copyright (c) 2001-2004 SAP AG

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  ========== licence end

*****************************************************************************/


#ifndef KERNEL_FILEIO_HPP
#define KERNEL_FILEIO_HPP



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "gsp00.h"
#include "KernelCommon/Kernel_Common.hpp"
#include "KernelCommon/Kernel_IPage.hpp"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

 /*!----------------------------------------------------------------------
 declarations: Predefined labels and extensions
-----------------------------------------------------------------------*/

#define KERNEL_CONV_FILE            "Conv"
#define KERNEL_DATA_FILE            "Data"
#define KERNEL_LOG_FILE             "Log"

#define KERNEL_BAD_FILE_EXTENSION   "bad"
#define KERNEL_COR_FILE_EXTENSION   "cor"

/*!----------------------------------------------------------------------
enddeclarations:
-----------------------------------------------------------------------*/

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!*****************************************************************************

   class: Kernel_FileIO

   description: This module is used to write a kernel page into the filesystem 
                of the operating system.
              
*******************************************************************************/

class Kernel_FileIO
{
public:

    /*!-------------------------------------------------------------------------
    function:     Kernel_FileIO()
    description:  The constructor is used to generate a file handle for kernel
                  page write operations into a OS file. The given file name 
                  determines the name of the generated file. The contructor 
                  opens implizit the given file. If no file is present it will
                  be generated, in other cases the file is overridden.
    arguments:    FileName [in] Name for the OS file 
    return value: none
    --------------------------------------------------------------------------*/

    Kernel_FileIO( const tsp00_VFilename  &FileName );
    
    /*!-------------------------------------------------------------------------
    function:     Kernel_FileIO()
    description:  The constructor is used to generate a file handle for kernel
                  page write operations into a OS file. The filename is given 
                  in parts, so that the call is able to build a individual 
                  filename very effectiv. For example the generated filename 
                  looks like this "Data14888.bad" or "Conv-1-20876.bad". (See
                  the predefined labels and extensions in this module)           
                  The contructor opens implizit the given file. If no file is 
                  present it will be generated, in other cases the file is 
                  overridden.
    arguments:    Label      [in] Suffix of the filename
                  Extension  [in] Prefix of the filename
                  FirstName  [in] First part of the filename
                  SecondName [in] If given(optional) second part of the filename
    return value: none
    --------------------------------------------------------------------------*/

    Kernel_FileIO(
        const SAPDB_Char    *Label,
        const SAPDB_Char    *Extenstion,
        const SAPDB_Char    *FirstName,
        const SAPDB_Char    *SecondName = NULL);

    /*!-------------------------------------------------------------------------
    function:     ~Kernel_FileIO()
    description:  This destructor closes the current file if it is still in 
    mode open.
    arguments:    none 
    return value: none
    --------------------------------------------------------------------------*/

    ~Kernel_FileIO()
    {
        Close();
    }

    /*--------------------------------------------------------------------------
    function:     Close()
    description:  This method is used to close a file if it is in open mode.
    arguments:    none 
    return value: none
    --------------------------------------------------------------------------*/

    void Close();

    /*!-------------------------------------------------------------------------
    function:     Write()
    description:  This method is used to write the memory block of a kernel
                  page into the run directtory of the kernel.
    arguments:    Page [in] Kernel page
    return value: true if no error has occured
    --------------------------------------------------------------------------*/

    SAPDB_Bool  Write( Kernel_IPage    &Page );

    /*!-------------------------------------------------------------------------
    function:     ErrorOccured()
    description:  This method is used to get the file internal error state.
    arguments:    none
    return value: true if no error has occured
    --------------------------------------------------------------------------*/

    SAPDB_Bool  ErrorOccured()
    {
        return( vf_ok != m_FileError );
    }

    /*!-------------------------------------------------------------------------
    function:     GetFileNo()
    description:  This method returns the OS file identifier.
    arguments:    none
    return value: (SAPDB_UInt4) FileNo
    --------------------------------------------------------------------------*/

    SAPDB_UInt4 GetFileNo() const
    {
        return( m_FileNo );
    }

private:

    /*_-------------------------------------------------------------------------
    function:     Open()
    description:  This method is used to create or open the current file. If 
                  the file is already open nothing happens.
    arguments:    none 
    return value: true is file is in mode open.
    --------------------------------------------------------------------------*/

    SAPDB_Bool  Open();

    /*--------------------------------------------------------------------------
    function:     IsOpen()
    description:  This method returns true if the file is in mode open.
    arguments:    none 
    return value: true if the file is in mode open
    --------------------------------------------------------------------------*/

    SAPDB_Bool  IsOpen() const
    {
        return( m_IsOpen );
    }

    SAPDB_Bool IsLocked() const
    {
        return( 0 != m_TaskId );
    }

    /*--------------------------------------------------------------------------
    function:     Lock()
    description:  Lock resource, if it isn't locked yet.
    arguments:    none 
    return value: none
    --------------------------------------------------------------------------*/

    void Lock();

    /*--------------------------------------------------------------------------
    function:     Unlock()
    description:  Unlock resource, if it is locked
    arguments:    none 
    return value: none
    --------------------------------------------------------------------------*/

    void Unlock();

private:

    /*-------------------------------------------------------------------------
    declaration: m_TaskId
    description: current taskId for locking.
    --------------------------------------------------------------------------*/

    tsp00_TaskId m_TaskId;

    /*-------------------------------------------------------------------------
    declaration: m_IsOpen
    description: file state. true is file is in mode open.
    --------------------------------------------------------------------------*/

    SAPDB_Bool  m_IsOpen;

    /*-------------------------------------------------------------------------
    declaration: m_FileError
    description: contains the error if one has occured.
    --------------------------------------------------------------------------*/

    tsp00_VfReturn  m_FileError;

    /*-------------------------------------------------------------------------
    declaration: m_FileNo
    description: contains the internal Id of the OS file.
    --------------------------------------------------------------------------*/

    SAPDB_Int4  m_FileNo;

    /*-------------------------------------------------------------------------
    declaration: m_FileName
    description: contains the given OS file name.
    --------------------------------------------------------------------------*/

    tsp00_VFilename m_FileName;
};

/*!*****************************************************************************

   endclass: Kernel_FileIO

*******************************************************************************/

#endif  /* KERNEL_FILEIO_HPP */
