/*!
 * \file    OMS_Exceptions.hpp
 * \author  IvanS, MarkusS, PeterG
 * \brief   Exceptions
 */
/*

    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/

#ifndef __OMS_EXCEPTIONS_HPP
#define __OMS_EXCEPTIONS_HPP

#include "Oms/OMS_DbpError.hpp"

/*---------------------------------------------------------------*/
/// Exception for duplicate key
/*!
** The OmsDuplicateKey class is derived from DbpError, and is thrown 
** by the OMS if the uniqueness of the key is violated when the new 
** operator of a class derived from OmsKeyedObject is executed.
**
** \see \ref example_OmsDuplicateKey
*/
class AFX_EXT_CLASS OmsDuplicateKey : virtual public DbpError
{
public:
  /// Constructor
  OmsDuplicateKey(long e, const OmsObjectId& oid, const char* pFileName=0, unsigned int line=0); 
  OmsDuplicateKey(long e, const OmsObjectId& oid, const char* msg, const char* pFileName=0, unsigned int line=0); 
};


/*---------------------------------------------------------------*/
/// Exception for lock-timeout
/*!
** The OmsLockTimeout class is derived from DbpError and is thrown  
** by the OMS if the permissible wait time is exceeded when an 
** instance lock is requested.
**
** \see \ref example_OmsLockTimeout
*/
class AFX_EXT_CLASS OmsLockTimeout : virtual public DbpError
{
public:
  /// Constructor
  OmsLockTimeout(const OmsObjectId& oid, const char* pFileName=0, unsigned int line=0);
  OmsLockTimeout(const OmsObjectId& oid, const char* msg, const char* pFileName=0, unsigned int line=0);
};


/*---------------------------------------------------------------*/
/// Exception for duplicate key or lock timeout
/*!
** The OmsKeyInUse class is derived from OmsDuplicateKey and
** OmsLockTimeout and is thrown by the OMS if the uniqueness of the key 
** is violated. In contrast to the duplicate key exception, this exception
** is also thrown, when the object with the key is created in a concurrent
** transaction and this transaction is not committed, yet.
**
*/
class AFX_EXT_CLASS OmsKeyInUse : public OmsLockTimeout, public OmsDuplicateKey 
{
public:
  /// Constructor
  OmsKeyInUse(long e, const OmsObjectId& oid, const char* pFileName=0, unsigned int line=0); 
  OmsKeyInUse(long e, const OmsObjectId& oid, const char* msg, const char* pFileName=0, unsigned int line=0); 
};


/*---------------------------------------------------------------*/
/// Exception for out-of-date
/*!
** The OmsOutOfDate class is derived from DbpError and is thrown 
** by the OMS if, when an instance lock is requested, it is determined 
** that the consistent view of the instance differs from the current 
** state in the liveCache.
**
** \see \ref example_OmsOutOfDate
*/
class AFX_EXT_CLASS OmsOutOfDate : public DbpError
{
public:
  /// Constructor
  OmsOutOfDate(const char* pFileName=0, unsigned int line=0);
  OmsOutOfDate(const OmsObjectId& oid, const char* pFileName=0, unsigned int line=0);
  OmsOutOfDate(const OmsObjectId& oid, const char* msg, const char* pFileName=0, unsigned int line=0);
};


/*---------------------------------------------------------------*/
/// Exception for object not found
/*!
** The OmsObjectNotFound class is derived from DbpError, and is thrown 
** by the OMS if the deref fails. There are several reasons for such an
** error (object does not exist, consistent view has been dropped,...)
** and therefore the origional error code is supplied to the constructor.
** So it is possible either to catch the OmsObjectNotFound exception or
** for a more detailed error handling to catch the error as DpbError and 
** check the error-code.
**
*/
class AFX_EXT_CLASS OmsObjectNotFound : public DbpError
{
public:
  /// Constructor
  OmsObjectNotFound(long e, const OmsObjectId& oid, const char* pFileName=0, unsigned int line=0); 
  OmsObjectNotFound(long e, const OmsObjectId& oid, const char* msg, const char* pFileName=0, unsigned int line=0); 
};

#endif // __OMS_EXCEPTIONS_HPP
