/*!
    @file     SharedSQL_IParseIDCache.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    Cache for ParseIDs, session specific
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/


#ifndef SHAREDSQL_IPARSEIDCACHE_HPP
#define SHAREDSQL_IPARSEIDCACHE_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_IHandles.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_IParseIDCache
    @brief          Interface to SharedSQL_ParseIDCache
 */
class SharedSQL_IParseIDCache
{
public:

    /*!
        @brief  Get an interface to a SharedSQL_ExecuteHandle for some ParseID
        @param  Alloc           [in]
        @param  PID             [in]
        @param  CountExecuting  [in]
        @param  Valid           [out]
        @param  M9              [out]
        @return SharedSQL_IExecuteHandle*
    */
    virtual SharedSQL_IExecuteHandle* NewExecuteHandle( SAPDBMem_IRawAllocator& Alloc, SharedSQL_ParseID PID, bool CountExecuting, SAPDB_Bool& Valid, SAPDB_Bool& M9 ) = 0;

    /*!
        @brief  Destroy the Handle created with NewExecuteHandle
        @param  EH [in]
    */
    virtual void DestroyExecuteHandle( SharedSQL_IExecuteHandle* EH ) = 0;


    /*!
        @brief  Add parse and modulename for some command identified its prepare handle
        @param  SH          [in]
        @param  PID         [in]
        @param  ModuleName  [in]
        @param  M9          [in]
        @return true, if ok, else false
    */
    virtual SAPDB_Bool PutParseID( const SharedSQL_IPrepareHandle* SH, const SharedSQL_ParseID& PID, const SharedSQL_ModuleName& ModuleName, const SAPDB_Bool M9 ) = 0;

    /*!
        @brief  Drop some parseid previously added with PutParseID
        @param  PID [in]
        @return true, if ok, else false
    */
    virtual SAPDB_Bool DropParseID( const SharedSQL_ParseID& PID ) = 0;

    /*!
        @brief  Get the module name for some command identified by parseid
        @param  PID [in]
        @return const SharedSQL_ModuleName*
    */
    virtual const SharedSQL_ModuleName* GetModuleName( const SharedSQL_ParseID& PID ) = 0;

    /*!
        @brief  Set statistic data from execution for some command identified by parseid
        @param  PID                 [in]
        @param  ExecuteTimeSec      [in]
        @param  ExecuteTimeMicroSec [in]
        @param  ReadRowCount        [in]
        @param  QualifiedRowCount   [in]
        @param  VirtualReadCount    [in]
        @param  PhysicalReadCount   [in]
        @param  FetchRowCount       [in]
        @param  SuspendCount        [in]
        @param  WaitCount           [in]
        @return true, if ok, else false
    */
    virtual SAPDB_Bool SetExecuteStatisticData( const SharedSQL_ParseID& PID,
                                        SharedSQL_CommandID& CommandID,
                                        SAPDB_UInt8 ExecuteTimeSec, SAPDB_UInt8 ExecuteTimeMicroSec,
                                        SAPDB_Int8 ReadRowCount, SAPDB_Int8 QualifiedRowCount,
                                        SAPDB_Int8 VirtualReadCount, SAPDB_Int8 PhysicalReadCount,
                                        SAPDB_Int8 FetchRowCount, 
                                        SAPDB_Int8 SuspendCount, SAPDB_Int8 WaitCount ) const = 0;

}; // SharedSQL_ParseIDCache

/* ------------------------------------------------------------------------- */

#endif  /* SHAREDSQL_IPARSEIDCACHE_HPP */

