/*!
  @defgroup       GarbageCollection Remove history records not used anymore
*/
/*!
  @ingroup        GarbageCollection
  @file           gbd930.h
  @author         FerdiF
  @brief          Remove history records not used anymore
*/
/*

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
#ifndef GBD930_H
#define GBD930_H

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include <new.h>

#include "gsp03.h"
#include "gsp03_3.h"
#include "ggg12.h"
#include "hgg01.h" //g01abort,g01opmsg 
#include "hgg01_1.h" //g01abort,g01opmsg 
#include "hgg01_3.h"   //get cserv parameter g01maxuser
#include "heo55k.h"  //vbegexcl
#include "heo56.h"   //vsuspend,vresume
//#include "heo57.h"   //vallocat //PTS 1110854 TS 2001-07-09

#include "Logging/Log_History.hpp"
#include "RunTime/MemoryManagement/RTEMem_Allocator.hpp"        //PTS 1110854 TS 2001-07-09
#include "SAPDBCommon/MemoryManagement/SAPDBMem_NewDestroy.hpp" //PTS 1110854 TS 2001-07-09
/* 7.4.01 -> 7.4.02 Merge FF 29/11/2001                                                  */
#include "liveCache/LVC_WaitForDelObjFileQueue.hpp"            // PTS 1112228 AK 19/10/2001 
#include "GarbageCollector/GC_IGarbageCollector.hpp" // PTS xxxxxxx FF 2002-02-12
#include "GarbageCollector/GC_IGCController.hpp" 
#include "Trace/Trace_Entry.hpp"


/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

#define NIL_HISTORY_FILE_INDEX_BD930  -1

/* local defines uses in this file only */
#define GC_IS_STARTED                 true
#define GC_IS_STOPPED                 false

#if COMPILEMODE_MEO00 >= QUICK_MEO00 
#  define BD930_CHECK_GC  bd930_CheckGC()
#else 
#  define BD930_CHECK_GC
#endif

/* Timeinterval which is used to control starting GCs via Timeout process   */ 
#define TIMEOUT_START_TIME_INTERVAL     30
#define TIMEOUT_START_TIME_INTERVAL_RED  3

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/

/*===========================================================================*
 *  FORWARD DECLARATIONS  OF CLASSES, STRUCTURES, TYPES, UNIONS              *
 *===========================================================================*/

class cbd930_GarbageCollector;
class cbd930_GCState;
class cbd930_GCCounters;
class cbd930_GarbCollController;
class cbd930_HistoryFileInfo;
class cbd930_HistoryFileDir;
class cbd930_HistoryFileIter;

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *==========================================================f=================*/

class cbd930_HistoryFileInfo
{
public: //members
    SAPDB_UInt4             hfiHistoryFileNo_bd930;
    tsp00_Int2              hfiCurrActivGC_bd930;
    tsp00_Bool              hfiHistoryGetsTruncated_bd930;
    tsp00_Bool              hfiFiller1_bd930;

public: //methods
              cbd930_HistoryFileInfo   ();
         bool bd930ClearHistoryFile    (tgg00_TransContext        &Trans,
                                        GC_IGarbageCollector      &IGarbColl);                // PTS 1113185 UH 2001-12-18

private:
    /* this check is necessary for vallocat */
    CHECK_MAPPABLE_CLASS_GIVEN_SIZE_SP00 (cbd930_HistoryFileInfo,  8)
};

/*---------------------------------------------------------------------------*/

class cbd930_HistoryFileDir
{
    friend class cbd930_HistoryFileIter;

public: // methods
              cbd930_HistoryFileDir   ();
  inline void bd930AddHistoryFile     (tgg00_TransContext &Trans, 
                                       SAPDB_UInt4         HistoryFileNo);
  inline void bd930ClearAll           ();
         void bd930DumpHistoryFileDir (tgg00_VfFileref     &HostFile,
                                       tsp00_Int4            BufferSize,
                                       tsp00_Page           &Buffer,
                                       tsp00_Int4           &OutPno,
                                       tsp00_Int4           &OutPos,
                                       tsp00_VfReturn       &HostError,
                                       tsp00_ErrText        &ErrText);
  inline void bd930Init               ();
private: // members
  cbd930_HistoryFileInfo *hfdFirstHFInfoPtr_bd930;    // pointer to the first entry in the directory
  tsp00_Int4              hfdNumHFInfos_bd930;        // number of HistoryFileInfo in the directory
  tsp00_Int4              hfdMaxNumHFInfos_bd930;     // maximum number of HistoryFileInfo which can be stored  
};

/*---------------------------------------------------------------------------*/

class cbd930_HistoryFileIter
{
public: // methods
  inline cbd930_HistoryFileInfo* operator () ();
  inline cbd930_HistoryFileInfo* operator ++ ();
  inline cbd930_HistoryFileInfo* bd930First  ();
         cbd930_HistoryFileInfo* bd930SetToHistFileWithOldestTrans(tgg00_TransContext &Trans);
         cbd930_HistoryFileIter (cbd930_HistoryFileDir & HistoryFileDir);

private: // members
  cbd930_HistoryFileDir  &hitHFDir_bd930;
  cbd930_HistoryFileInfo *hitCurrHFInfoPtr_bd930;
  int                     hitCurrHFIndex_bd930;
};

/*-----------------------------------------------------------*/

class cbd930_GarbageCollector : public GC_IGarbageCollector
{
public:
  /* methods defined by GC_IGarbageCollector interface */
  virtual SAPDB_UInt1 GetHistOption() const;
  virtual void        IncHistCreateObjFileCount();
  virtual void        IncHistDeleteObjCount();
  virtual void        IncHistDropObjFileCount();
  virtual void        IncHistEntryReleaseCount();
  virtual void        IncHistInsertObjCount();
  virtual void        IncHistLockObjCount();
  virtual void        IncHistNewObjCount();
  virtual void        IncHistUpdateObjCount();
  virtual void        IncObjReleaseCount();
  virtual void        IncReleaseEmptyPageCount();
  virtual bool        IsInterrrupted() const;
  virtual void        SetHistOption(SAPDB_UInt1 option);

public:
  cbd930_GarbageCollector (tgg00_TransContext          &Trans,
                           cbd930_GarbCollController   &GarbCollController);
  
private: // members
    tgg00_TransContext         &gcoTrans_bd930;
    const int                   gcoIndex_bd930;
    cbd930_GarbCollController  &gcoController_bd930;

private: // methods
  inline void bd930_HistoryErrorHandler              ();
         void bd930_PageErrorHandler                 (cbd900_ObjFileInfo  &ObjFileInfo);
  inline void bd930_WriteMessageIntoVtrace           (const bool bStartStopFlag);
         void bd930_WritePageStatusIntoVtrace        (tgg00_TransContext    &Trans,
                                                      tbd_nodeptr            pPage,
                                                      tgg00_VtraceType_Enum  TraceObject);
         bool bd930_ReleaseUnusedPagesOfAllFiles     ();
         void bd930_ReleaseUnusedPagesOfOneFile      (tgg00_FileId        &ObjFileId,
                                                      cbd900_ObjFileInfo  &ObjFileInfo,
                                                      bool                &bSomethingDone,
                                                      bool                &bFileIsEmpty);
         void bd930_ReleaseUnusedPagesOfOneChain     (tgg00_FileId        &ObjFileId,
                                                      cbd900_ObjFileInfo  &ObjFileInfo,
                                                      tsp00_Int4           ChainNo);
         void bd930_ReleaseAllPagesOfOneChain        (tgg00_FileId        &ObjFileId,
                                                      cbd900_ObjFileInfo  &ObjFileInfo,
                                                      tsp00_Int4           ChainNo);
         void bd930_RemoveFile                       (tgg00_FileId        &ObjFileId,   //PTS 1107306 AK 01/08/2000
                                                      bool                &bSomethingDone);
         bool bd930_TryToReleaseUnusedObjDataPages   ();
         void bd930_RemoveEntryFromPersFileDirectory (tgg00_FileId        &ObjFileId);
};

/*---------------------------------------------------------------------------*/

class cbd930_GCCounters
{
public:
  inline cbd930_GCCounters();
public:
  tsp00_8ByteCounter      gccntActivateCount;     
  tsp00_8ByteCounter      gccntObjectsDeleted;     
  tsp00_8ByteCounter      gccntHistEntriesReleased;
  tsp00_8ByteCounter      gccntEmtyPagesReleased;            // pages released because of empty-page handling
  tsp00_8ByteCounter      gccntDropPagesReleased;            // pages released because of drop page handling
  tsp00_8ByteCounter      gccntDropFiles;                    // number of files dropped 
  tsp00_8ByteCounter      gccntHistCreateObjFile;
  tsp00_8ByteCounter      gccntHistDeleteObj;
  tsp00_8ByteCounter      gccntHistDropObjFile;
  tsp00_8ByteCounter      gccntHistLockObj;
  tsp00_8ByteCounter      gccntHistInsertObj;
  tsp00_8ByteCounter      gccntHistNewObj;
  tsp00_8ByteCounter      gccntHistUpdateObj;
};


/*---------------------------------------------------------------------------*/

class cbd930_GCState
{
public:
  inline cbd930_GCState();
public:
  tsp00_TaskId gcsTaskId_bd930;
  tsp00_Bool   gcsActive_bd930;
  SAPDB_UInt1  gcsHistoryOption_bd930;
};

/*---------------------------------------------------------------------------*/

class  cbd930_GarbCollController : public GC_IGCController
{
  friend class cbd930_GarbageCollector;
  
private://members
  SAPDB_UInt8                  gccLastTimeGCsStarted;
  tsp00_Int4                   gccNumGC_bd930;                    // # of all garbage collectors (GC) 
  tsp00_Int4                   gccNumGCUsed_bd930;                // # of used GCs  
  tsp00_Int4                   gccNumGCUsedToReleasePages_bd930;  // # of GCs used to remove empty pages or deleted files
  tsp00_Int4                   gccNumGCReqToReleasePages_bd930;   // # of chains which request the GC to release empty pages
  tsp00_TaskId                 gccStopGCManagerTID_bd930;         // Id of the task coordinating the suspension of all GCs
  cbd900_ObjFileDir           *gccObjFileDirPtr_bd930;
  cbd900_ContObjFileDir       *gccContObjFileDirPtr_bd930;
  cbd930_HistoryFileDir        gccHistoryFileDir_bd930;
  tsp00_Bool                   gccStopGC_bd930;                   // flag that displays whether all GCs have to stoped
  tsp00_Bool                   gccSpecialOperationIsActive_bd930; // savepoint/b01restart_filesystem or     PTS 1113552 FF
  tsp00_Bool                   gccGCStartRequested_bd930;         // flag: all GCs have to be started after Timeintervall PTS 1116157 FF                                                                        
  // "Cold Verify" or "DIAGNOSE REPAIR" is active 
  tsp00_Bool                   gccCheckSavepointNeeded_bd930;       
  tsp00_Bool                   gccRestartOfAllGCsInAction;
  tsp00_Bool				   gccPhaseRed_bd930;				  // strict checking for filling of db needed
  
  LVC_WaitForDelObjFileQueue  *m_pWaitForDelObjFileQueue;         // PTS 1112228 AK 19/10/2001 7.4.01 -> 7.4.02 Merge FF 29/11/2001
public:
 
  ARRAY_VAR_MSP00 (gccGCState_bd930, cbd930_GCState,   MAX_OBJ_GARB_COLL_BD00);
  ARRAY_VAR_MSP00 (gccGCCounters_bd930, cbd930_GCCounters,   MAX_OBJ_GARB_COLL_BD00);
  
public://methods
  inline void TriggerGarbageCollection() { gccGCStartRequested_bd930 = true; }
  inline void RegisterHistoryFile              (tgg00_TransContext &Trans, 
                                                SAPDB_UInt4         HistoryFileNo);
  inline void ClearAll                         (); 
  inline SAPDB_UInt8 GetGCLastTimeStarted      ();
  inline bool bd930CheckIfTaskIsAGC            (const tsp00_TaskId        TaskId) const;
         void bd930DumpGCController            (tgg00_VfFileref     &HostFile,
                                                tsp00_Int4            BufferSize,
                                                tsp00_Page           &Buffer,
                                                tsp00_Int4           &OutPno,
                                                tsp00_Int4           &OutPos,
                                                tsp00_VfReturn       &HostError,
                                                tsp00_ErrText        &ErrText);
  inline tsp00_Int4 bd930GetNumGCs             () const;
  inline void bd930Init                        (cbd900_ObjFileDir     &ObjFileDir,
                                                cbd900_ContObjFileDir &ContObjFileDir);
  void bd930StartAllGCs                        (tsp00_TaskId   TaskId,
                                                bool           bIfGCWaitTimeDone = true);
  inline void bd930StartAllGCsAfterSpecialOp   (tsp00_TaskId   TaskId);
         void bd930StopAllGCsForSpecialOp      (tsp00_TaskId   TaskId);
         void bd930StartGCsIfRequired          (tsp00_TaskId   TaskId);
         void bd930StartGCToRemoveFile         (tgg00_TransContext   &Trans, 
                                                cbd900_ObjFileInfo   &ObjFileInfo,
                                                bool                  bWithinRegion = false);  // PTS 1115596 FF 2002-05-08  
  inline void InsertIntoWaitForDelObjFileQueue (tsp00_TaskId    TaskId,     // PTS 1112228 AK 19/10/2001 
	                                             tgg00_ObjFileNo ObjFileNo);
  inline void WakeUpTasksWaitingForDelObjFile  (tsp00_TaskId    TaskId,     // PTS 1112228 AK 19/10/2001 
	                                             tgg00_ObjFileNo ObjFileNo);

private : // methods used by the friend class cbd930_GarbageCollector
  inline tsp00_Int4 bd930_AddGC        (tgg00_TransContext &Trans);
  inline void       bd930_CheckGC      ();   
  inline void       bd930_FindUnusedGC (tsp00_TaskId       &UnusedGCTaskId);
};

/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_GarbCollController                       */
/*---------------------------------------------------------------------------*/

/* PTS 1111248 AK 3/08/20001                                                 */
inline void
cbd930_GarbCollController::RegisterHistoryFile (tgg00_TransContext &Trans, 
                                                SAPDB_UInt4         HistoryFileNo)
{
  gccHistoryFileDir_bd930.bd930AddHistoryFile (Trans, HistoryFileNo);
}

/*---------------------------------------------------------------------------*/

inline void
cbd930_GarbCollController::ClearAll ()
{
  gccHistoryFileDir_bd930.bd930ClearAll();
}

/*---------------------------------------------------------------------------*/

inline SAPDB_UInt8 
cbd930_GarbCollController::GetGCLastTimeStarted()
{
     return gccLastTimeGCsStarted;
}

/*---------------------------------------------------------------------------*/

inline bool
cbd930_GarbCollController::bd930CheckIfTaskIsAGC (const tsp00_TaskId  TaskId) const
{
  /* check if the given task id belongs to a Garabage Collector */
  for  (int iGC=0; iGC< gccNumGC_bd930; ++iGC)
  {
    if(gccGCState_bd930 [iGC].gcsTaskId_bd930 == TaskId)
      return true;
  }
  return false;
}
  
/*---------------------------------------------------------------------------*/

inline tsp00_Int4 
cbd930_GarbCollController::bd930GetNumGCs() const
{
  return gccNumGC_bd930;
};

/*---------------------------------------------------------------------------*/

inline void
cbd930_GarbCollController::bd930Init (cbd900_ObjFileDir     &ObjFileDir,
                                      cbd900_ContObjFileDir &ContObjFileDir)
{
  gccLastTimeGCsStarted            = 0;
  gccNumGC_bd930                   = 0;
  gccNumGC_bd930                   = 0;
  gccNumGCUsed_bd930               = 0;                    
  gccNumGCUsedToReleasePages_bd930 = 0;
  gccNumGCReqToReleasePages_bd930  = 0;
  gccStopGCManagerTID_bd930        = cgg_nil_pid;
  gccObjFileDirPtr_bd930           = &ObjFileDir;
  gccContObjFileDirPtr_bd930       = &ContObjFileDir;
  gccStopGC_bd930                  = false;
  gccSpecialOperationIsActive_bd930 = true; // PTS 1113552 FF  
  gccGCStartRequested_bd930        = true;  // PTS 1116157 FF
  gccCheckSavepointNeeded_bd930    = false;
  
 	// PTS 1112228 AK 19/10/2001 FF 7.4.01 -> 7.4.02 Merge 29/11/2001
  SAPDBMem_IRawAllocator &Allocator = RTEMem_Allocator::Instance();
  m_pWaitForDelObjFileQueue = new (Allocator) LVC_WaitForDelObjFileQueue (Allocator, g01kbfunction_tasks());
  
  gccRestartOfAllGCsInAction       = false;
  gccPhaseRed_bd930				   = false;
  gccHistoryFileDir_bd930.bd930Init();
}

/*---------------------------------------------------------------------------*/

inline tsp00_Int4
cbd930_GarbCollController::bd930_AddGC(tgg00_TransContext   &Trans)
{
  vbegexcl (Trans.trTaskId_gg00, g08garbage); 
  
  /* update number of registerd GCs */
  const int IndexGC = gccNumGC_bd930++;
  
  /* check Number of  garbage collectores */
  if (gccNumGC_bd930 > MAX_OBJ_GARB_COLL_BD00)
    g01abort (csp3_bd_msg, csp3_n_obj, "BD930RegGC:No more slots", IndexGC);
  
  /* check that this slot is really free */
  if (cgg_nil_pid != gccGCState_bd930[IndexGC].gcsTaskId_bd930)
    g01abort (csp3_bd_msg, csp3_n_obj, "BD930RegGC:Slot not free", IndexGC);
  
  vendexcl (Trans.trTaskId_gg00, g08garbage); 
  
  /* assign TaskIds */ 
  gccGCState_bd930[IndexGC].gcsTaskId_bd930 = Trans.trTaskId_gg00;
  
  return IndexGC;   
}

/*-----------------------------------------------------------*/

inline void
cbd930_GarbCollController::bd930_FindUnusedGC (tsp00_TaskId  &UnusedGCTaskId)      
{
  
  ROUTINE_DBG_MEO00 ("bd930_FindUnusedGC");
  
  /* This function looks for an unused GC  */
  /* If one found it is marked and         */
  /* bUnusedGCExists ist set to true       */
  
  /* check if one is exclusively in the g08garbage region */
# if COMPILEMODE_MEO00 >= SLOW_MEO00 
  g08check_excl (g08garbage);
# endif
  
  UnusedGCTaskId = cgg_nil_pid;
  
  /* decide if there is still an unused GC which can be started  */
  if (gccNumGCUsed_bd930 < gccNumGC_bd930 && ! gccSpecialOperationIsActive_bd930) // PTS 1113552 FF
  {               
    /* find free GC */
    for  (int iGC=0; iGC< gccNumGC_bd930; ++iGC)
    {
      if(!gccGCState_bd930 [iGC].gcsActive_bd930)
      {
        gccGCState_bd930 [iGC].gcsActive_bd930 = true;
        UnusedGCTaskId = gccGCState_bd930 [iGC].gcsTaskId_bd930;
        ++gccNumGCUsed_bd930;
        break;
      }
    }
  }
}

/*---------------------------------------------------------------------------*/

inline void
cbd930_GarbCollController::bd930StartAllGCsAfterSpecialOp (tsp00_TaskId   TaskId)
{ 
    const bool startUnconditionally = false;

    gccRestartOfAllGCsInAction        = true;      // avoid dropping consistent view
    gccSpecialOperationIsActive_bd930 = false;     // PTS 1113552 FF
    gccGCStartRequested_bd930         = true;
    bd930StartAllGCs (TaskId, startUnconditionally);
    
    g01opmsg (sp3p_knldiag, sp3m_info, csp3_bd_msg, csp3_n_obj,
                  "Restarted Garbage coll  ", gccNumGCUsed_bd930);

    gccRestartOfAllGCsInAction = false;
}

/*---------------------------------------------------------------------------*/
// PTS 1112228 AK 19/10/2001 FF 7.4.01 -> 7.4.02 Merge 29/11/2001
inline void
cbd930_GarbCollController::InsertIntoWaitForDelObjFileQueue (tsp00_TaskId    TaskId,
                                                             tgg00_ObjFileNo ObjFileNo)
{
  m_pWaitForDelObjFileQueue->Insert (TaskId, ObjFileNo);
}

/*---------------------------------------------------------------------------*/
// PTS 1112228 AK 19/10/2001 FF 7.4.01 -> 7.4.02 Merge 29/11/2001
inline void
cbd930_GarbCollController::WakeUpTasksWaitingForDelObjFile (tsp00_TaskId    TaskId,
                                                            tgg00_ObjFileNo ObjFileNo)
{
  m_pWaitForDelObjFileQueue->WakeUpWaitingTasks (TaskId, ObjFileNo);
}
    
/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_GarbageCollector                         */
/*---------------------------------------------------------------------------*/

inline void
cbd930_GarbageCollector::bd930_WriteMessageIntoVtrace (const bool bStart)
{
    if (g01vtrace.vtrOmsFree_gg00 )
    {
        tgg12_OmsGarbCollTrace TrLineGC;
        
        TrLineGC.ogrbTrType_gg12.becomes(otGarbColl_egg12);
        TrLineGC.ogrbTrType2_gg12.becomes(ot2Nil_egg12);
        TrLineGC.ogrbCollectorIndex_gg12 = gcoIndex_bd930;
        TrLineGC.ogrbIsActiv_gg12        = bStart;
        
        Trace_CommonEntry( gcoTrans_bd930, tgg00_Debug::fromConst(bd), 
            tgg00_VtraceType::fromConst(b930Garbcol), 
            REINTERPRET_CAST( tsp00_BytePtr, &TrLineGC ), sizeof (TrLineGC) );
    }
}

/*---------------------------------------------------------------------------*/

inline void 
cbd930_GarbageCollector::bd930_PageErrorHandler (cbd900_ObjFileInfo  &ObjFileInfo)
{
  switch (gcoTrans_bd930.trError_gg00)
  {
  case e_ok            : /* nothing to do                                    */
    break;
  case e_file_not_found: /* the file has been removed and the ObjFileInfo    */
    /* has to be updated                                */
    ObjFileInfo.bd900ResetFileInfo();
    gcoTrans_bd930.trError_gg00 = e_ok;
    break;
  default :             /* unknown error, shutdown because of safety reasons */ 
    g01opmsg (sp3p_knldiag, sp3m_error, csp3_bd_msg, csp3_n_obj, 
      "BD930GC_Error: Index    ", gcoIndex_bd930);
    g01abort (csp3_bd_msg, csp3_n_obj,"EMERGENCY HALT          ", gcoTrans_bd930.trError_gg00);
  }
}

/*---------------------------------------------------------------------------*/

inline void 
cbd930_GarbageCollector::bd930_HistoryErrorHandler()
{
  if (e_ok != gcoTrans_bd930.trError_gg00)
  {
    g01opmsg (sp3p_knldiag, sp3m_error, csp3_bd_msg, csp3_n_obj, 
      "BD930HistGC: Error      ", gcoTrans_bd930.trError_gg00);
    g01abort (csp3_bd_msg, csp3_n_obj,"BD930HistGC: Index      ", gcoIndex_bd930);
  }
}

/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_GCCounters                               */
/*---------------------------------------------------------------------------*/

inline 
cbd930_GCCounters::cbd930_GCCounters()
:
gccntActivateCount (0),
gccntObjectsDeleted (0),
gccntHistEntriesReleased (0),
gccntEmtyPagesReleased (0),
gccntDropPagesReleased (0),
gccntDropFiles (0),
gccntHistCreateObjFile (0),
gccntHistDeleteObj (0),
gccntHistDropObjFile (0),
gccntHistLockObj (0),
gccntHistInsertObj (0),
gccntHistNewObj (0),
gccntHistUpdateObj (0)
{};

/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_GCState                                  */
/*---------------------------------------------------------------------------*/

inline 
cbd930_GCState::cbd930_GCState()
:
gcsTaskId_bd930 (cgg_nil_pid),
gcsActive_bd930 (false),
gcsHistoryOption_bd930( Log_History::Normal )
{};

/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_HistoryFileInfo                          */
/*---------------------------------------------------------------------------*/

inline
cbd930_HistoryFileInfo::cbd930_HistoryFileInfo()
:
hfiHistoryFileNo_bd930        (SAPDB_MAX_UINT4),
hfiCurrActivGC_bd930          (NIL_GC_INDEX_BD900),
hfiHistoryGetsTruncated_bd930 (false),
hfiFiller1_bd930              (false) 
{}

/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_HistoryFileDir                           */
/*---------------------------------------------------------------------------*/

inline void
cbd930_HistoryFileDir::bd930Init()
{
  if (g01is_livecache ()) /* PTS 1106791 Alexander Kley 01/06/2000 */
  {
    hfdNumHFInfos_bd930    = 0;
    hfdMaxNumHFInfos_bd930 = g01maxuser();
    
    /* write note about allocation into knldiag */
    g01allocate_msg (csp3_n_dynpool, "HIST_DIR  item size    :", sizeof (cbd930_HistoryFileInfo));
    g01allocate_msg (csp3_n_dynpool, "HIST_DIR  num items    :", hfdMaxNumHFInfos_bd930);
    g01allocate_msg (csp3_n_dynpool, "DYNP_B930_HIST_DIR     :", hfdMaxNumHFInfos_bd930 * sizeof (cbd930_HistoryFileInfo));
    
    // PTS 1110854 TS 2001-07-09 
    /* allocate memory for the history file entries */
    //vallocat (hfdMaxNumHFInfos_bd930 * sizeof (cbd930_HistoryFileInfo), (void **) &hfdFirstHFInfoPtr_bd930, &bOk);
    
    newarray (hfdFirstHFInfoPtr_bd930, hfdMaxNumHFInfos_bd930, RTEMem_Allocator::Instance());
    
    if ( NULL == hfdFirstHFInfoPtr_bd930 ) 
      g01abort (csp3_sysbuf_storage_exceeded, csp3_n_config,"DYNP_B930_HIST_DIR      ", 0);
    
    /* initialize allocated memory */
    /*
    if(bOk)
    
      new (hfdFirstHFInfoPtr_bd930) cbd930_HistoryFileInfo [hfdMaxNumHFInfos_bd930]; 
      else
      g01abort (csp3_sysbuf_storage_exceeded, csp3_n_config,"DYNP_B930_HIST_DIR      ", 0);
    */
  }
}

/*---------------------------------------------------------------------------*/
/* PTS 1111248 AK 3/08/20001                                                 */
inline void
cbd930_HistoryFileDir::bd930AddHistoryFile (tgg00_TransContext &Trans, 
                                            SAPDB_UInt4         HistoryFileNo)
{
  vbegexcl (Trans.trTaskId_gg00, g08garbage);
  if (hfdNumHFInfos_bd930 < hfdMaxNumHFInfos_bd930)
  {
    (hfdFirstHFInfoPtr_bd930 + hfdNumHFInfos_bd930)->hfiHistoryFileNo_bd930 = HistoryFileNo;
    hfdNumHFInfos_bd930 ++;
  }
  else
    g01abort (csp3_bd_msg, csp3_n_obj,"BD930:Too Many HistFiles", hfdNumHFInfos_bd930);
  
  vendexcl (Trans.trTaskId_gg00, g08garbage); 
}

/*---------------------------------------------------------------------------*/

inline void
cbd930_HistoryFileDir::bd930ClearAll ()
{
  // vbegexcl (Trans.trTaskId_gg00, g08garbage);
  hfdNumHFInfos_bd930 = 0;
  // vendexcl (Trans.trTaskId_gg00, g08garbage); 
}

/*---------------------------------------------------------------------------*/

inline
cbd930_HistoryFileDir::cbd930_HistoryFileDir()
:
hfdFirstHFInfoPtr_bd930 (NULL),
hfdNumHFInfos_bd930     (0),
hfdMaxNumHFInfos_bd930  (0)
{};

/*---------------------------------------------------------------------------*/
/* METHOD IMPLEMENTATION FOR cbd930_HistoryFileIter                          */
/*---------------------------------------------------------------------------*/

inline 
cbd930_HistoryFileIter::cbd930_HistoryFileIter (cbd930_HistoryFileDir & HistoryFileDir)
:
hitHFDir_bd930         (HistoryFileDir),
hitCurrHFInfoPtr_bd930 (NULL),
hitCurrHFIndex_bd930   (NIL_HISTORY_FILE_INDEX_BD930)
{};

/*---------------------------------------------------------------------------*/

inline cbd930_HistoryFileInfo*
cbd930_HistoryFileIter::operator () ()
{
  return hitCurrHFInfoPtr_bd930;
}

/*---------------------------------------------------------------------------*/

inline cbd930_HistoryFileInfo*
cbd930_HistoryFileIter::operator ++ ()
{
  hitCurrHFIndex_bd930 ++;
  
  if (hitCurrHFIndex_bd930 < hitHFDir_bd930.hfdNumHFInfos_bd930)
    hitCurrHFInfoPtr_bd930 = hitHFDir_bd930.hfdFirstHFInfoPtr_bd930 + hitCurrHFIndex_bd930;
  else
    hitCurrHFInfoPtr_bd930 = NULL;
  
  return hitCurrHFInfoPtr_bd930;
}

/*---------------------------------------------------------------------------*/

inline cbd930_HistoryFileInfo*
cbd930_HistoryFileIter::bd930First()
{
  if (hitHFDir_bd930.hfdNumHFInfos_bd930 > 0)
  {
    hitCurrHFIndex_bd930   = 0;
    hitCurrHFInfoPtr_bd930 = hitHFDir_bd930.hfdFirstHFInfoPtr_bd930;
  }
  else
  {
    hitCurrHFIndex_bd930   = NIL_HISTORY_FILE_INDEX_BD930;
    hitCurrHFInfoPtr_bd930 = NULL;
  }
  
  return hitCurrHFInfoPtr_bd930;
}

/*---------------------------------------------------------------------------*/

#endif  /* GBD930_H */
