﻿// Authors:
//      Marek Habersack <mhabersack@novell.com>
//
// Copyright (C) 2010 Novell Inc. http://novell.com
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using System.Web;
using System.Web.Caching;
using System.Web.Configuration;

using NUnit.Framework;

namespace MonoTests.System.Web.Caching
{
	public class TestOutputCacheProvider : OutputCacheProvider
	{
		Dictionary<string, object> data = new Dictionary<string, object> ();

		public override object Add (string key, object entry, DateTime utcExpiry)
		{
			if (data.ContainsKey (key))
				data[key] = entry;
			else
				data.Add (key, entry);

			return entry;
		}

		public override object Get (string key)
		{
			object o;

			if (data.TryGetValue (key, out o))
				return o;

			return null;
		}

		public override void Remove (string key)
		{
			if (data.ContainsKey (key))
				data.Remove (key);
		}

		public override void Set (string key, object entry, DateTime utcExpiry)
		{
			if (data.ContainsKey (key))
				data[key] = entry;
		}
	}

	[Serializable]
	class TestResponseElement : ResponseElement
	{
		public string Text
		{
			get;
			private set;
		}

		public TestResponseElement (string text)
		{
			this.Text = text;
		}
	}

	[TestFixture]
	[Serializable]
	public class OutputCacheTest
	{
		[Test]
		public void DefaultProviderName ()
		{
			Assert.AreEqual ("AspNetInternalProvider", OutputCache.DefaultProviderName, "#A1");

			var ocs = WebConfigurationManager.GetWebApplicationSection ("system.web/caching/outputCache") as OutputCacheSection;
			Assert.AreEqual (ocs.DefaultProviderName, OutputCache.DefaultProviderName, "#B1");
		}

		[Test]
		public void Providers ()
		{
			var ocs = WebConfigurationManager.GetWebApplicationSection ("system.web/caching/outputCache") as OutputCacheSection;
			OutputCacheProviderCollection coll = OutputCache.Providers;

			Assert.IsNull (coll, "#A1");
			Assert.IsNotNull (ocs.Providers, "#B1-1");
			Assert.AreEqual (0, ocs.Providers.Count, "#B1-2");
		}

		string DummyBadSubstitutionCallback (HttpContext ctx)
		{
			return "string";
		}

		static string DummySubstitutionCallback (HttpContext ctx)
		{
			return "string";
		}

		// Serialized elements for the Serialize test below
#if TARGET_DOTNET
		static readonly byte[] memoryResponseElement = { 0x0, 0x1, 0x0, 0x0, 0x0, 0xFF, 0xFF, 0xFF, 0xFF, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xC, 0x2, 0x0, 0x0, 0x0, 0x4D, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2C, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x34, 0x2E, 0x30, 0x2E, 0x30, 0x2E, 0x30, 0x2C, 0x20, 0x43, 0x75, 0x6C, 0x74, 0x75, 0x72, 0x65, 0x3D, 0x6E, 0x65, 0x75, 0x74, 0x72, 0x61, 0x6C, 0x2C, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63, 0x4B, 0x65, 0x79, 0x54, 0x6F, 0x6B, 0x65, 0x6E, 0x3D, 0x62, 0x30, 0x33, 0x66, 0x35, 0x66, 0x37, 0x66, 0x31, 0x31, 0x64, 0x35, 0x30, 0x61, 0x33, 0x61, 0x5, 0x1, 0x0, 0x0, 0x0, 0x28, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2E, 0x43, 0x61, 0x63, 0x68, 0x69, 0x6E, 0x67, 0x2E, 0x4D, 0x65, 0x6D, 0x6F, 0x72, 0x79, 0x52, 0x65, 0x73, 0x70, 0x6F, 0x6E, 0x73, 0x65, 0x45, 0x6C, 0x65, 0x6D, 0x65, 0x6E, 0x74, 0x2, 0x0, 0x0, 0x0, 0x7, 0x5F, 0x62, 0x75, 0x66, 0x66, 0x65, 0x72, 0x7, 0x5F, 0x6C, 0x65, 0x6E, 0x67, 0x74, 0x68, 0x7, 0x0, 0x2, 0x9, 0x2, 0x0, 0x0, 0x0, 0x9, 0x3, 0x0, 0x0, 0x0, 0xC, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xF, 0x3, 0x0, 0x0, 0x0, 0xC, 0x0, 0x0, 0x0, 0x2, 0x45, 0x6E, 0x63, 0x6F, 0x64, 0x65, 0x64, 0x20, 0x64, 0x61, 0x74, 0x61, 0xB, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
		static readonly byte[] fileResponseElement = { 0x0, 0x1, 0x0, 0x0, 0x0, 0xFF, 0xFF, 0xFF, 0xFF, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xC, 0x2, 0x0, 0x0, 0x0, 0x4D, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2C, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x34, 0x2E, 0x30, 0x2E, 0x30, 0x2E, 0x30, 0x2C, 0x20, 0x43, 0x75, 0x6C, 0x74, 0x75, 0x72, 0x65, 0x3D, 0x6E, 0x65, 0x75, 0x74, 0x72, 0x61, 0x6C, 0x2C, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63, 0x4B, 0x65, 0x79, 0x54, 0x6F, 0x6B, 0x65, 0x6E, 0x3D, 0x62, 0x30, 0x33, 0x66, 0x35, 0x66, 0x37, 0x66, 0x31, 0x31, 0x64, 0x35, 0x30, 0x61, 0x33, 0x61, 0x5, 0x1, 0x0, 0x0, 0x0, 0x26, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2E, 0x43, 0x61, 0x63, 0x68, 0x69, 0x6E, 0x67, 0x2E, 0x46, 0x69, 0x6C, 0x65, 0x52, 0x65, 0x73, 0x70, 0x6F, 0x6E, 0x73, 0x65, 0x45, 0x6C, 0x65, 0x6D, 0x65, 0x6E, 0x74, 0x3, 0x0, 0x0, 0x0, 0x5, 0x5F, 0x70, 0x61, 0x74, 0x68, 0x7, 0x5F, 0x6F, 0x66, 0x66, 0x73, 0x65, 0x74, 0x7, 0x5F, 0x6C, 0x65, 0x6E, 0x67, 0x74, 0x68, 0x1, 0x0, 0x0, 0x9, 0x9, 0x2, 0x0, 0x0, 0x0, 0x6, 0x3, 0x0, 0x0, 0x0, 0x11, 0x2F, 0x70, 0x61, 0x74, 0x68, 0x2F, 0x74, 0x6F, 0x2F, 0x66, 0x69, 0x6C, 0x65, 0x2E, 0x74, 0x78, 0x74, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xD2, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xB, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
		static readonly byte[] substitutionResponseElement = { 0x0, 0x1, 0x0, 0x0, 0x0, 0xFF, 0xFF, 0xFF, 0xFF, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xC, 0x2, 0x0, 0x0, 0x0, 0x4D, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2C, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x34, 0x2E, 0x30, 0x2E, 0x30, 0x2E, 0x30, 0x2C, 0x20, 0x43, 0x75, 0x6C, 0x74, 0x75, 0x72, 0x65, 0x3D, 0x6E, 0x65, 0x75, 0x74, 0x72, 0x61, 0x6C, 0x2C, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63, 0x4B, 0x65, 0x79, 0x54, 0x6F, 0x6B, 0x65, 0x6E, 0x3D, 0x62, 0x30, 0x33, 0x66, 0x35, 0x66, 0x37, 0x66, 0x31, 0x31, 0x64, 0x35, 0x30, 0x61, 0x33, 0x61, 0x5, 0x1, 0x0, 0x0, 0x0, 0x2E, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2E, 0x43, 0x61, 0x63, 0x68, 0x69, 0x6E, 0x67, 0x2E, 0x53, 0x75, 0x62, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x69, 0x6F, 0x6E, 0x52, 0x65, 0x73, 0x70, 0x6F, 0x6E, 0x73, 0x65, 0x45, 0x6C, 0x65, 0x6D, 0x65, 0x6E, 0x74, 0x2, 0x0, 0x0, 0x0, 0xF, 0x5F, 0x74, 0x61, 0x72, 0x67, 0x65, 0x74, 0x54, 0x79, 0x70, 0x65, 0x4E, 0x61, 0x6D, 0x65, 0xB, 0x5F, 0x6D, 0x65, 0x74, 0x68, 0x6F, 0x64, 0x4E, 0x61, 0x6D, 0x65, 0x1, 0x1, 0x2, 0x0, 0x0, 0x0, 0x6, 0x3, 0x0, 0x0, 0x0, 0x41, 0x4D, 0x6F, 0x6E, 0x6F, 0x54, 0x65, 0x73, 0x74, 0x73, 0x2E, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2E, 0x43, 0x61, 0x63, 0x68, 0x69, 0x6E, 0x67, 0x2E, 0x4F, 0x75, 0x74, 0x70, 0x75, 0x74, 0x43, 0x61, 0x63, 0x68, 0x65, 0x54, 0x65, 0x73, 0x74, 0x2C, 0x20, 0x44, 0x6F, 0x74, 0x4E, 0x65, 0x74, 0x34, 0x5F, 0x4E, 0x55, 0x6E, 0x69, 0x74, 0x5F, 0x54, 0x65, 0x73, 0x74, 0x73, 0x6, 0x4, 0x0, 0x0, 0x0, 0x19, 0x44, 0x75, 0x6D, 0x6D, 0x79, 0x53, 0x75, 0x62, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x69, 0x6F, 0x6E, 0x43, 0x61, 0x6C, 0x6C, 0x62, 0x61, 0x63, 0x6B, 0xB, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
		static readonly byte[] badSubstitutionResponseElement = { 0x0, 0x1, 0x0, 0x0, 0x0, 0xFF, 0xFF, 0xFF, 0xFF, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xC, 0x2, 0x0, 0x0, 0x0, 0x4D, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2C, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x34, 0x2E, 0x30, 0x2E, 0x30, 0x2E, 0x30, 0x2C, 0x20, 0x43, 0x75, 0x6C, 0x74, 0x75, 0x72, 0x65, 0x3D, 0x6E, 0x65, 0x75, 0x74, 0x72, 0x61, 0x6C, 0x2C, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63, 0x4B, 0x65, 0x79, 0x54, 0x6F, 0x6B, 0x65, 0x6E, 0x3D, 0x62, 0x30, 0x33, 0x66, 0x35, 0x66, 0x37, 0x66, 0x31, 0x31, 0x64, 0x35, 0x30, 0x61, 0x33, 0x61, 0x5, 0x1, 0x0, 0x0, 0x0, 0x2E, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2E, 0x43, 0x61, 0x63, 0x68, 0x69, 0x6E, 0x67, 0x2E, 0x53, 0x75, 0x62, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x69, 0x6F, 0x6E, 0x52, 0x65, 0x73, 0x70, 0x6F, 0x6E, 0x73, 0x65, 0x45, 0x6C, 0x65, 0x6D, 0x65, 0x6E, 0x74, 0x2, 0x0, 0x0, 0x0, 0xF, 0x5F, 0x74, 0x61, 0x72, 0x67, 0x65, 0x74, 0x54, 0x79, 0x70, 0x65, 0x4E, 0x61, 0x6D, 0x65, 0xB, 0x5F, 0x6D, 0x65, 0x74, 0x68, 0x6F, 0x64, 0x4E, 0x61, 0x6D, 0x65, 0x1, 0x1, 0x2, 0x0, 0x0, 0x0, 0x6, 0x3, 0x0, 0x0, 0x0, 0x41, 0x4D, 0x6F, 0x6E, 0x6F, 0x54, 0x65, 0x73, 0x74, 0x73, 0x2E, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x57, 0x65, 0x62, 0x2E, 0x43, 0x61, 0x63, 0x68, 0x69, 0x6E, 0x67, 0x2E, 0x4F, 0x75, 0x74, 0x70, 0x75, 0x74, 0x43, 0x61, 0x63, 0x68, 0x65, 0x54, 0x65, 0x73, 0x74, 0x2C, 0x20, 0x44, 0x6F, 0x74, 0x4E, 0x65, 0x74, 0x34, 0x5F, 0x4E, 0x55, 0x6E, 0x69, 0x74, 0x5F, 0x54, 0x65, 0x73, 0x74, 0x73, 0x6, 0x4, 0x0, 0x0, 0x0, 0x1C, 0x44, 0x75, 0x6D, 0x6D, 0x79, 0x42, 0x61, 0x64, 0x53, 0x75, 0x62, 0x73, 0x74, 0x69, 0x74, 0x75, 0x74, 0x69, 0x6F, 0x6E, 0x43, 0x61, 0x6C, 0x6C, 0x62, 0x61, 0x63, 0x6B, 0xB, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
#else
		static readonly byte[] memoryResponseElement = {0x0,0x1,0x0,0x0,0x0,0xFF,0xFF,0xFF,0xFF,0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xC,0x2,0x0,0x0,0x0,0x4D,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x34,0x2E,0x30,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x62,0x30,0x33,0x66,0x35,0x66,0x37,0x66,0x31,0x31,0x64,0x35,0x30,0x61,0x33,0x61,0x5,0x1,0x0,0x0,0x0,0x28,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4D,0x65,0x6D,0x6F,0x72,0x79,0x52,0x65,0x73,0x70,0x6F,0x6E,0x73,0x65,0x45,0x6C,0x65,0x6D,0x65,0x6E,0x74,0x2,0x0,0x0,0x0,0x17,0x3C,0x42,0x75,0x66,0x66,0x65,0x72,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x17,0x3C,0x4C,0x65,0x6E,0x67,0x74,0x68,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x7,0x0,0x2,0x9,0x2,0x0,0x0,0x0,0x9,0x3,0x0,0x0,0x0,0xC,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xF,0x3,0x0,0x0,0x0,0xC,0x0,0x0,0x0,0x2,0x45,0x6E,0x63,0x6F,0x64,0x65,0x64,0x20,0x64,0x61,0x74,0x61,0xB,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
		static readonly byte[] fileResponseElement = {0x0,0x1,0x0,0x0,0x0,0xFF,0xFF,0xFF,0xFF,0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xC,0x2,0x0,0x0,0x0,0x4D,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x34,0x2E,0x30,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x62,0x30,0x33,0x66,0x35,0x66,0x37,0x66,0x31,0x31,0x64,0x35,0x30,0x61,0x33,0x61,0x5,0x1,0x0,0x0,0x0,0x26,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x46,0x69,0x6C,0x65,0x52,0x65,0x73,0x70,0x6F,0x6E,0x73,0x65,0x45,0x6C,0x65,0x6D,0x65,0x6E,0x74,0x3,0x0,0x0,0x0,0x17,0x3C,0x4C,0x65,0x6E,0x67,0x74,0x68,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x17,0x3C,0x4F,0x66,0x66,0x73,0x65,0x74,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x15,0x3C,0x50,0x61,0x74,0x68,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x0,0x0,0x1,0x9,0x9,0x2,0x0,0x0,0x0,0xD2,0x4,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x6,0x3,0x0,0x0,0x0,0x11,0x2F,0x70,0x61,0x74,0x68,0x2F,0x74,0x6F,0x2F,0x66,0x69,0x6C,0x65,0x2E,0x74,0x78,0x74,0xB,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
		static readonly byte[] substitutionResponseElement = {0x0,0x1,0x0,0x0,0x0,0xFF,0xFF,0xFF,0xFF,0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xC,0x2,0x0,0x0,0x0,0x4D,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x34,0x2E,0x30,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x62,0x30,0x33,0x66,0x35,0x66,0x37,0x66,0x31,0x31,0x64,0x35,0x30,0x61,0x33,0x61,0x5,0x1,0x0,0x0,0x0,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x52,0x65,0x73,0x70,0x6F,0x6E,0x73,0x65,0x45,0x6C,0x65,0x6D,0x65,0x6E,0x74,0x3,0x0,0x0,0x0,0x8,0x74,0x79,0x70,0x65,0x4E,0x61,0x6D,0x65,0xA,0x6D,0x65,0x74,0x68,0x6F,0x64,0x4E,0x61,0x6D,0x65,0x19,0x3C,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x1,0x1,0x3,0x22,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2,0x0,0x0,0x0,0x6,0x3,0x0,0x0,0x0,0x88,0x1,0x4D,0x6F,0x6E,0x6F,0x54,0x65,0x73,0x74,0x73,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4F,0x75,0x74,0x70,0x75,0x74,0x43,0x61,0x63,0x68,0x65,0x54,0x65,0x73,0x74,0x2C,0x20,0x6e,0x65,0x74,0x5f,0x34,0x5f,0x78,0x5f,0x53,0x79,0x73,0x74,0x65,0x6d,0x2e,0x57,0x65,0x62,0x5f,0x74,0x65,0x73,0x74,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x31,0x2E,0x33,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x33,0x31,0x62,0x66,0x33,0x38,0x35,0x36,0x61,0x64,0x33,0x36,0x34,0x65,0x33,0x35,0x6,0x4,0x0,0x0,0x0,0x19,0x44,0x75,0x6D,0x6D,0x79,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x9,0x5,0x0,0x0,0x0,0x4,0x5,0x0,0x0,0x0,0x22,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2,0x0,0x0,0x0,0x8,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x7,0x6D,0x65,0x74,0x68,0x6F,0x64,0x30,0x3,0x3,0x30,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2B,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x2F,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x52,0x65,0x66,0x6C,0x65,0x63,0x74,0x69,0x6F,0x6E,0x2E,0x4D,0x65,0x6D,0x62,0x65,0x72,0x49,0x6E,0x66,0x6F,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x9,0x6,0x0,0x0,0x0,0x9,0x7,0x0,0x0,0x0,0x4,0x6,0x0,0x0,0x0,0x30,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2B,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x7,0x0,0x0,0x0,0x4,0x74,0x79,0x70,0x65,0x8,0x61,0x73,0x73,0x65,0x6D,0x62,0x6C,0x79,0x6,0x74,0x61,0x72,0x67,0x65,0x74,0x12,0x74,0x61,0x72,0x67,0x65,0x74,0x54,0x79,0x70,0x65,0x41,0x73,0x73,0x65,0x6D,0x62,0x6C,0x79,0xE,0x74,0x61,0x72,0x67,0x65,0x74,0x54,0x79,0x70,0x65,0x4E,0x61,0x6D,0x65,0xA,0x6D,0x65,0x74,0x68,0x6F,0x64,0x4E,0x61,0x6D,0x65,0xD,0x64,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x1,0x1,0x2,0x1,0x1,0x1,0x3,0x30,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2B,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x6,0x8,0x0,0x0,0x0,0x2B,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x48,0x74,0x74,0x70,0x52,0x65,0x73,0x70,0x6F,0x6E,0x73,0x65,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x6,0x9,0x0,0x0,0x0,0x4D,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x34,0x2E,0x30,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x62,0x30,0x33,0x66,0x35,0x66,0x37,0x66,0x31,0x31,0x64,0x35,0x30,0x61,0x33,0x61,0xA,0x6,0xA,0x0,0x0,0x0,0x5A,0x6e,0x65,0x74,0x5f,0x34,0x5f,0x78,0x5f,0x53,0x79,0x73,0x74,0x65,0x6d,0x2e,0x57,0x65,0x62,0x5f,0x74,0x65,0x73,0x74,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x31,0x2E,0x33,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x33,0x31,0x62,0x66,0x33,0x38,0x35,0x36,0x61,0x64,0x33,0x36,0x34,0x65,0x33,0x35,0x6,0xB,0x0,0x0,0x0,0x2C,0x4D,0x6F,0x6E,0x6F,0x54,0x65,0x73,0x74,0x73,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4F,0x75,0x74,0x70,0x75,0x74,0x43,0x61,0x63,0x68,0x65,0x54,0x65,0x73,0x74,0x9,0x4,0x0,0x0,0x0,0xA,0x4,0x7,0x0,0x0,0x0,0x2F,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x52,0x65,0x66,0x6C,0x65,0x63,0x74,0x69,0x6F,0x6E,0x2E,0x4D,0x65,0x6D,0x62,0x65,0x72,0x49,0x6E,0x66,0x6F,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x7,0x0,0x0,0x0,0x4,0x4E,0x61,0x6D,0x65,0xC,0x41,0x73,0x73,0x65,0x6D,0x62,0x6C,0x79,0x4E,0x61,0x6D,0x65,0x9,0x43,0x6C,0x61,0x73,0x73,0x4E,0x61,0x6D,0x65,0x9,0x53,0x69,0x67,0x6E,0x61,0x74,0x75,0x72,0x65,0xA,0x53,0x69,0x67,0x6E,0x61,0x74,0x75,0x72,0x65,0x32,0xA,0x4D,0x65,0x6D,0x62,0x65,0x72,0x54,0x79,0x70,0x65,0x10,0x47,0x65,0x6E,0x65,0x72,0x69,0x63,0x41,0x72,0x67,0x75,0x6D,0x65,0x6E,0x74,0x73,0x1,0x1,0x1,0x1,0x1,0x0,0x3,0x8,0xD,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x54,0x79,0x70,0x65,0x5B,0x5D,0x9,0x4,0x0,0x0,0x0,0x9,0xA,0x0,0x0,0x0,0x9,0xB,0x0,0x0,0x0,0x6,0xF,0x0,0x0,0x0,0x3F,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x53,0x74,0x72,0x69,0x6E,0x67,0x20,0x44,0x75,0x6D,0x6D,0x79,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x28,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x48,0x74,0x74,0x70,0x43,0x6F,0x6E,0x74,0x65,0x78,0x74,0x29,0x6,0x10,0x0,0x0,0x0,0x3F,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x53,0x74,0x72,0x69,0x6E,0x67,0x20,0x44,0x75,0x6D,0x6D,0x79,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x28,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x48,0x74,0x74,0x70,0x43,0x6F,0x6E,0x74,0x65,0x78,0x74,0x29,0x8,0x0,0x0,0x0,0xA,0xB,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
		static readonly byte[] badSubstitutionResponseElement = {0x0,0x1,0x0,0x0,0x0,0xFF,0xFF,0xFF,0xFF,0x1,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xC,0x2,0x0,0x0,0x0,0x4D,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x34,0x2E,0x30,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x62,0x30,0x33,0x66,0x35,0x66,0x37,0x66,0x31,0x31,0x64,0x35,0x30,0x61,0x33,0x61,0x5,0x1,0x0,0x0,0x0,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x52,0x65,0x73,0x70,0x6F,0x6E,0x73,0x65,0x45,0x6C,0x65,0x6D,0x65,0x6E,0x74,0x3,0x0,0x0,0x0,0x8,0x74,0x79,0x70,0x65,0x4E,0x61,0x6D,0x65,0xA,0x6D,0x65,0x74,0x68,0x6F,0x64,0x4E,0x61,0x6D,0x65,0x19,0x3C,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x3E,0x6B,0x5F,0x5F,0x42,0x61,0x63,0x6B,0x69,0x6E,0x67,0x46,0x69,0x65,0x6C,0x64,0x1,0x1,0x3,0x22,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2,0x0,0x0,0x0,0x6,0x3,0x0,0x0,0x0,0x88,0x1,0x4D,0x6F,0x6E,0x6F,0x54,0x65,0x73,0x74,0x73,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4F,0x75,0x74,0x70,0x75,0x74,0x43,0x61,0x63,0x68,0x65,0x54,0x65,0x73,0x74,0x2C,0x20,0x6e,0x65,0x74,0x5f,0x34,0x5f,0x78,0x5f,0x53,0x79,0x73,0x74,0x65,0x6d,0x2e,0x57,0x65,0x62,0x5f,0x74,0x65,0x73,0x74,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x31,0x2E,0x33,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x33,0x31,0x62,0x66,0x33,0x38,0x35,0x36,0x61,0x64,0x33,0x36,0x34,0x65,0x33,0x35,0x6,0x4,0x0,0x0,0x0,0x1C,0x44,0x75,0x6D,0x6D,0x79,0x42,0x61,0x64,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x9,0x5,0x0,0x0,0x0,0xC,0x6,0x0,0x0,0x0,0x5A,0x6e,0x65,0x74,0x5f,0x34,0x5f,0x78,0x5f,0x53,0x79,0x73,0x74,0x65,0x6d,0x2e,0x57,0x65,0x62,0x5f,0x74,0x65,0x73,0x74,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x31,0x2E,0x33,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x33,0x31,0x62,0x66,0x33,0x38,0x35,0x36,0x61,0x64,0x33,0x36,0x34,0x65,0x33,0x35,0x4,0x5,0x0,0x0,0x0,0x22,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x3,0x0,0x0,0x0,0x8,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x7,0x74,0x61,0x72,0x67,0x65,0x74,0x30,0x7,0x6D,0x65,0x74,0x68,0x6F,0x64,0x30,0x3,0x4,0x3,0x30,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2B,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x2C,0x4D,0x6F,0x6E,0x6F,0x54,0x65,0x73,0x74,0x73,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4F,0x75,0x74,0x70,0x75,0x74,0x43,0x61,0x63,0x68,0x65,0x54,0x65,0x73,0x74,0x6,0x0,0x0,0x0,0x2F,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x52,0x65,0x66,0x6C,0x65,0x63,0x74,0x69,0x6F,0x6E,0x2E,0x4D,0x65,0x6D,0x62,0x65,0x72,0x49,0x6E,0x66,0x6F,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x9,0x7,0x0,0x0,0x0,0x9,0x8,0x0,0x0,0x0,0x9,0x9,0x0,0x0,0x0,0x4,0x7,0x0,0x0,0x0,0x30,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2B,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x7,0x0,0x0,0x0,0x4,0x74,0x79,0x70,0x65,0x8,0x61,0x73,0x73,0x65,0x6D,0x62,0x6C,0x79,0x6,0x74,0x61,0x72,0x67,0x65,0x74,0x12,0x74,0x61,0x72,0x67,0x65,0x74,0x54,0x79,0x70,0x65,0x41,0x73,0x73,0x65,0x6D,0x62,0x6C,0x79,0xE,0x74,0x61,0x72,0x67,0x65,0x74,0x54,0x79,0x70,0x65,0x4E,0x61,0x6D,0x65,0xA,0x6D,0x65,0x74,0x68,0x6F,0x64,0x4E,0x61,0x6D,0x65,0xD,0x64,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x1,0x1,0x2,0x1,0x1,0x1,0x3,0x30,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x2B,0x44,0x65,0x6C,0x65,0x67,0x61,0x74,0x65,0x45,0x6E,0x74,0x72,0x79,0x6,0xA,0x0,0x0,0x0,0x2B,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x48,0x74,0x74,0x70,0x52,0x65,0x73,0x70,0x6F,0x6E,0x73,0x65,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x6,0xB,0x0,0x0,0x0,0x4D,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x34,0x2E,0x30,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x62,0x30,0x33,0x66,0x35,0x66,0x37,0x66,0x31,0x31,0x64,0x35,0x30,0x61,0x33,0x61,0x6,0xC,0x0,0x0,0x0,0x7,0x74,0x61,0x72,0x67,0x65,0x74,0x30,0x6,0xD,0x0,0x0,0x0,0x5A,0x6e,0x65,0x74,0x5f,0x34,0x5f,0x78,0x5f,0x53,0x79,0x73,0x74,0x65,0x6d,0x2e,0x57,0x65,0x62,0x5f,0x74,0x65,0x73,0x74,0x2C,0x20,0x56,0x65,0x72,0x73,0x69,0x6F,0x6E,0x3D,0x31,0x2E,0x33,0x2E,0x30,0x2E,0x30,0x2C,0x20,0x43,0x75,0x6C,0x74,0x75,0x72,0x65,0x3D,0x6E,0x65,0x75,0x74,0x72,0x61,0x6C,0x2C,0x20,0x50,0x75,0x62,0x6C,0x69,0x63,0x4B,0x65,0x79,0x54,0x6F,0x6B,0x65,0x6E,0x3D,0x33,0x31,0x62,0x66,0x33,0x38,0x35,0x36,0x61,0x64,0x33,0x36,0x34,0x65,0x33,0x35,0x6,0xE,0x0,0x0,0x0,0x2C,0x4D,0x6F,0x6E,0x6F,0x54,0x65,0x73,0x74,0x73,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4F,0x75,0x74,0x70,0x75,0x74,0x43,0x61,0x63,0x68,0x65,0x54,0x65,0x73,0x74,0x9,0x4,0x0,0x0,0x0,0xA,0x5,0x8,0x0,0x0,0x0,0x2C,0x4D,0x6F,0x6E,0x6F,0x54,0x65,0x73,0x74,0x73,0x2E,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x43,0x61,0x63,0x68,0x69,0x6E,0x67,0x2E,0x4F,0x75,0x74,0x70,0x75,0x74,0x43,0x61,0x63,0x68,0x65,0x54,0x65,0x73,0x74,0x0,0x0,0x0,0x0,0x6,0x0,0x0,0x0,0x4,0x9,0x0,0x0,0x0,0x2F,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x52,0x65,0x66,0x6C,0x65,0x63,0x74,0x69,0x6F,0x6E,0x2E,0x4D,0x65,0x6D,0x62,0x65,0x72,0x49,0x6E,0x66,0x6F,0x53,0x65,0x72,0x69,0x61,0x6C,0x69,0x7A,0x61,0x74,0x69,0x6F,0x6E,0x48,0x6F,0x6C,0x64,0x65,0x72,0x7,0x0,0x0,0x0,0x4,0x4E,0x61,0x6D,0x65,0xC,0x41,0x73,0x73,0x65,0x6D,0x62,0x6C,0x79,0x4E,0x61,0x6D,0x65,0x9,0x43,0x6C,0x61,0x73,0x73,0x4E,0x61,0x6D,0x65,0x9,0x53,0x69,0x67,0x6E,0x61,0x74,0x75,0x72,0x65,0xA,0x53,0x69,0x67,0x6E,0x61,0x74,0x75,0x72,0x65,0x32,0xA,0x4D,0x65,0x6D,0x62,0x65,0x72,0x54,0x79,0x70,0x65,0x10,0x47,0x65,0x6E,0x65,0x72,0x69,0x63,0x41,0x72,0x67,0x75,0x6D,0x65,0x6E,0x74,0x73,0x1,0x1,0x1,0x1,0x1,0x0,0x3,0x8,0xD,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x54,0x79,0x70,0x65,0x5B,0x5D,0x9,0x4,0x0,0x0,0x0,0x9,0xD,0x0,0x0,0x0,0x9,0xE,0x0,0x0,0x0,0x6,0x12,0x0,0x0,0x0,0x42,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x53,0x74,0x72,0x69,0x6E,0x67,0x20,0x44,0x75,0x6D,0x6D,0x79,0x42,0x61,0x64,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x28,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x48,0x74,0x74,0x70,0x43,0x6F,0x6E,0x74,0x65,0x78,0x74,0x29,0x6,0x13,0x0,0x0,0x0,0x42,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x53,0x74,0x72,0x69,0x6E,0x67,0x20,0x44,0x75,0x6D,0x6D,0x79,0x42,0x61,0x64,0x53,0x75,0x62,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x43,0x61,0x6C,0x6C,0x62,0x61,0x63,0x6B,0x28,0x53,0x79,0x73,0x74,0x65,0x6D,0x2E,0x57,0x65,0x62,0x2E,0x48,0x74,0x74,0x70,0x43,0x6F,0x6E,0x74,0x65,0x78,0x74,0x29,0x8,0x0,0x0,0x0,0xA,0xB,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0};
#endif

		static readonly object[] serializeObjects = {
			(object)new object (),
			(bool)true,
			(char)'a',
			(sbyte)-128,
			(byte)128,
			(short)-32768,
			(ushort)65535,
			(int)2147483647,
			(uint)4294967295,
			(long)9223372036854775807,
			(ulong)18446744073709551615,
			(float)3.402823E+38F,
			(double)1.79769313486232E+307D,
			(decimal)79228162514264337593543950335M,
			(string)"string",
			(DateTime)DateTime.Now,
			(DBNull)DBNull.Value,
			(TestResponseElement)new TestResponseElement ("text"),
			(HeaderElement)new HeaderElement ("name", "value")
		};

		[Test]
		public void Serialize ()
		{
			ResponseElement data = new MemoryResponseElement (new byte[10], 10);

			Assert.Throws<ArgumentNullException> (() => {
				OutputCache.Serialize (null, data);
			}, "#A1");

			var ms = new MemoryStream ();
			Assert.Throws<ArgumentException> (() => {
				OutputCache.Serialize (ms, null);
			}, "#A2");

			foreach (object o in serializeObjects) {
				Assert.Throws<ArgumentException> (() => {
					OutputCache.Serialize (ms, o);
				}, String.Format ("#A3-{0}", o.GetType ()));
			}
			
			MemoryResponseElement mre;
			FileResponseElement fre;
			SubstitutionResponseElement sre, sreBad;

			CreateElements (out mre, out fre, out sre, out sreBad);
#if false
			WriteSerializedBytes ("/tmp/bytes.txt", "memoryResponseElement", mre);
			WriteSerializedBytes ("/tmp/bytes.txt", "fileResponseElement", fre);
			WriteSerializedBytes ("/tmp/bytes.txt", "substitutionResponseElement", sre);
			WriteSerializedBytes ("/tmp/bytes.txt", "badSubstitutionResponseElement", sreBad);
#endif
			
			byte[] bytes = SerializeElement (mre);
			Assert.AreEqual (bytes, memoryResponseElement, "#B1");

			bytes = SerializeElement (fre);
			Assert.AreEqual (bytes, fileResponseElement, "#B2");

			bytes = SerializeElement (sre);
			Assert.AreEqual (bytes, substitutionResponseElement, "#B3");

			bytes = SerializeElement (sreBad);
			Assert.AreEqual (bytes, badSubstitutionResponseElement, "#B4");
		}

		[Test]
		public void Deserialize ()
		{
			Assert.Throws<ArgumentNullException> (() => {
				OutputCache.Deserialize (null);
			}, "#A1");

			foreach (object o in serializeObjects) {
				using (var m = new MemoryStream (SerializeToBinary (o))) {
					Assert.Throws<ArgumentException> (() => {
						OutputCache.Deserialize (m);
					}, String.Format ("#A2-{0}", o.GetType ()));
				}
			}

			MemoryResponseElement mre, mreOrig;
			FileResponseElement fre, freOrig;
			SubstitutionResponseElement sre, sreOrig, sreBad, sreBadOrig;
			MemoryStream ms;

			CreateElements (out mre, out fre, out sre, out sreBad);

			using (ms = new MemoryStream (memoryResponseElement))
				mreOrig = OutputCache.Deserialize (ms) as MemoryResponseElement;
			Assert.IsNotNull (mreOrig, "#B1-1");
			Assert.AreEqual (mreOrig.Buffer, mre.Buffer, "#B1-2");
			Assert.AreEqual (mreOrig.Length, mre.Length, "#B1-3");

			using (ms = new MemoryStream (fileResponseElement))
				freOrig = OutputCache.Deserialize (ms) as FileResponseElement;
			Assert.IsNotNull (freOrig, "#C1-1");
			Assert.AreEqual (freOrig.Length, fre.Length, "#C1-2");
			Assert.AreEqual (freOrig.Offset, fre.Offset, "#C1-3");
			Assert.AreEqual (freOrig.Path, fre.Path, "#C1-4");

			using (ms = new MemoryStream (substitutionResponseElement))
				sreOrig = OutputCache.Deserialize (ms) as SubstitutionResponseElement;
			Assert.IsNotNull (sreOrig);
			Assert.AreEqual (sreOrig.Callback, sre.Callback, "#D1");

			// Callback here is not a static method - Substitution delegates must be static
			Assert.Throws<ArgumentException> (() => {
				using (ms = new MemoryStream (badSubstitutionResponseElement))
					sreBadOrig = OutputCache.Deserialize (ms) as SubstitutionResponseElement;
			}, "#E1");
		}

		void CreateElements (out MemoryResponseElement mre, out FileResponseElement fre, out SubstitutionResponseElement sre,
			out SubstitutionResponseElement sreBad)
		{
			byte[] bytes = Encoding.UTF8.GetBytes ("Encoded data");
			mre = new MemoryResponseElement (bytes, bytes.Length);
			fre = new FileResponseElement ("/path/to/file.txt", 0, 1234);
			sre = new SubstitutionResponseElement (new HttpResponseSubstitutionCallback (DummySubstitutionCallback));
			sreBad = new SubstitutionResponseElement (new HttpResponseSubstitutionCallback (DummyBadSubstitutionCallback));
		}

		byte[] SerializeToBinary (object o)
		{
			using (var ms = new MemoryStream ()) {
				var fmt = new BinaryFormatter ();
				fmt.Serialize (ms, o);
				ms.Flush ();
				return ms.GetBuffer ();
			}
		}

		byte[] SerializeElement (ResponseElement e)
		{
			using (var ms = new MemoryStream ()) {
				OutputCache.Serialize (ms, e);
				ms.Flush ();
				return ms.GetBuffer ();
			}
		}

		// A helper method should one want to add new serialized type arrays
		void WriteSerializedBytes (string filePath, string varName, object data)
		{
			byte[] bytes;
			using (var ms = new MemoryStream ()) {
				OutputCache.Serialize (ms, data);
				ms.Flush ();
				bytes = ms.GetBuffer ();

				if (bytes == null)
					return;

				using (var sw = File.AppendText (filePath)) {
					sw.Write ("byte[] {0} = {{", varName);

					bool first = true;
					foreach (byte b in bytes) {
						if (first)
							first = false;
						else
							sw.Write (",");
						sw.Write ("0x{0:X}", b);
					}
					sw.WriteLine ("};");
				}
			}
		}
	}
}
