/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, 1991, 1993, David Koblas.  (koblas@netcom.com)    | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */


#include <string.h>
#include "pnm.h"

#define GIFMAXVAL 255
#define MAXCOLORMAPSIZE 256

#define CM_RED   0
#define CM_GREEN 1
#define CM_BLUE  2

#define MAX_LWZ_BITS  12

#define INTERLACE      0x40
#define LOCALCOLORMAP  0x80
#define BitSet(byte, bit)      (((byte) & (bit)) == (bit))

#define        ReadOK(file,buffer,len) (fread(buffer, len, 1, file) != 0)

#define LM_to_uint(a,b)                        (((b)<<8)|(a))

struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filespec;  /* Filespecs of input files */
    int verbose;    /* -verbose option */
    int comments;   /* -comments option */
    int image_no;   /* image number he wants from input */
    char *alpha_filename;
    int alpha_stdout;
};


static void
parse_command_line(int argc, char ** argv,
                   struct cmdline_info *cmdline_p) {
/*----------------------------------------------------------------------------
   Note that the file spec array we return is stored in the storage that
   was passed to us as the argv array.
-----------------------------------------------------------------------------*/
    optStruct *option_def = malloc(100*sizeof(optStruct));
        /* Instructions to OptParseOptions2 on how to parse our options.
         */
    optStruct2 opt;

    unsigned int option_def_index;

    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENTRY(0,   "verbose",     OPT_FLAG,   &cmdline_p->verbose,         0);
    OPTENTRY(0,   "comments",    OPT_FLAG,   &cmdline_p->comments,        0);
    OPTENTRY(0,   "image",       OPT_UINT,   &cmdline_p->image_no,        0);
    OPTENTRY(0,   "alphaout",   OPT_STRING, &cmdline_p->alpha_filename, 0);

    /* Set the defaults */
    cmdline_p->verbose = FALSE;
    cmdline_p->comments = FALSE;
    cmdline_p->image_no = 1;
    cmdline_p->alpha_filename = NULL;

    opt.opt_table = option_def;
    opt.short_allowed = FALSE;  /* We have no short (old-fashioned) options */
    opt.allowNegNum = FALSE;  /* We have no parms that are negative numbers */

    pm_optParseOptions2(&argc, argv, opt, 0);
        /* Uses and sets argc, argv, and some of *cmdline_p and others. */

    if (argc-1 == 0) 
        cmdline_p->input_filespec = "-";
    else if (argc-1 != 1)
        pm_error("Program takes zero or one argument (filename).  You "
                 "specified %d", argc-1);
    else
        cmdline_p->input_filespec = argv[1];

    if (cmdline_p->alpha_filename && 
        strcmp(cmdline_p->alpha_filename, "-") == 0 )
        cmdline_p->alpha_stdout = 1;
    else 
        cmdline_p->alpha_stdout = 0;
}



struct GifScreen {
    unsigned int    Width;
    unsigned int    Height;
    unsigned char   ColorMap[3][MAXCOLORMAPSIZE];
    unsigned int    ColorMapSize;
    unsigned int    ColorResolution;
    unsigned int    Background;
    unsigned int    AspectRatio;
        /* Aspect ratio of each pixel, times 64, minus 15.  (i.e. 1 => 1:4).
           But Zero means 1:1.
        */
    int      has_gray;  
        /* Boolean: global colormap has at least one gray color
           (not counting black and white) 
        */
    int      has_color;
        /* Boolean: global colormap has at least one non-gray,
           non-black, non-white color 
        */
};

static struct GifScreen GifScreen;

static struct {
       int     transparent;
       int     delayTime;
       int     inputFlag;
       int     disposal;
} Gif89 = { -1, -1, -1, 0 };

static int verbose;
int    showComment;



static void
ReadColorMap(FILE *fd, const int colormapsize, 
             unsigned char colormap[3][MAXCOLORMAPSIZE],
             int *has_grayP, int * const has_colorP) {

    int             i;
    unsigned char   rgb[3];

    *has_grayP = FALSE;  /* initial assumption */
    *has_colorP = FALSE;  /* initial assumption */

    for (i = 0; i < colormapsize; ++i) {
        if (! ReadOK(fd, rgb, sizeof(rgb)))
            pm_error("Unable to read Color %n from colormap", i);

        colormap[CM_RED][i] = rgb[0] ;
        colormap[CM_GREEN][i] = rgb[1] ;
        colormap[CM_BLUE][i] = rgb[2] ;

        if (rgb[0] == rgb[1] && rgb[1] == rgb[2]) {
            if (rgb[0] != 0 && rgb[0] != GIFMAXVAL)
                *has_grayP = TRUE;
        } else
            *has_colorP = TRUE;
    }
}



static bool ZeroDataBlock = FALSE;
    /* the most recently read DataBlock was an EOD marker, i.e. had
       zero length */

static void
GetDataBlock(FILE *          const fd, 
             unsigned char * const buf, 
             bool *          const eofP,
             unsigned int *  const lengthP) {
/*----------------------------------------------------------------------------
   Read a DataBlock from file 'fd', return it at 'buf'.

   The first byte of the datablock is the length, in pure binary, of the
   rest of the datablock.  We return the data portion (not the length byte)
   of the datablock at 'buf', and its length as *lengthP.

   Except that if we hit EOF or have an I/O error reading the first
   byte (size field) of the DataBlock, we return *eofP == TRUE and
   *lengthP == 0.

   We return *eofP == FALSE if we don't hit EOF or have an I/O error.

   If we hit EOF or have an I/O error reading the data portion of the
   DataBlock, we exit the program with pm_error().
-----------------------------------------------------------------------------*/
    unsigned char   count;
    bool successfulRead;
    
    successfulRead = ReadOK(fd, &count, 1);
    if (!successfulRead) {
        pm_message("EOF or error in reading DataBlock size from file" );
        *eofP = TRUE;
        *lengthP = 0;
    } else {
        *eofP = FALSE;
        *lengthP = count;

        if (count == 0) 
            ZeroDataBlock = TRUE;
        else {
            bool successfulRead;

            successfulRead = ReadOK(fd, buf, count); 
            
            if (!successfulRead) 
                pm_error("EOF or error reading data portion of %d byte "
                         "DataBlock from file", count);
        }
    }
}



static void
ReadThroughEod(FILE * const fd) {
    unsigned char buf[260];
    bool eod;

    eod = FALSE;  /* initial value */
    while (!eod) {
        bool eof;
        unsigned int count;

        GetDataBlock(fd, buf, &eof, &count);
        if (eof)
            pm_message("EOF encountered before EOD marker.  The GIF "
                       "file is malformed, but we are proceeding "
                       "anyway as if an EOD marker were at the end "
                       "of the file.");
        if (eof || count == 0)
            eod = TRUE;
    }
}



static void
DoCommentExtension(FILE * const fd) {
/*----------------------------------------------------------------------------
   Read the rest of a comment extension from the input file 'fd' and handle
   it.
   
   We ought to deal with the possibility that the comment is not text.  I.e.
   it could have nonprintable characters or embedded nulls.  I don't know if
   the GIF spec requires regular text or not.
-----------------------------------------------------------------------------*/
    char buf[255+1];
    unsigned int blocklen;  
    bool done;

    done = FALSE;
    while (!done) {
        bool eof;
        GetDataBlock(fd, (unsigned char*) buf, &eof, &blocklen); 
        if (blocklen == 0 || eof)
            done = TRUE;
        else {
            buf[blocklen] = '\0';
            if (showComment) {
                pm_message("gif comment: %s", buf);
            }
        }
    }
}



static void DoGraphicControlExtension(FILE * const fd) {

    bool eof;
    unsigned int length;
    static unsigned char buf[256];

    GetDataBlock(fd, buf, &eof, &length);
    if (eof)
        pm_error("EOF/error encountered reading "
                 "1st DataBlock of Graphic Control Extension.");
    else if (length < 4) 
        pm_error("graphic control extension 1st DataBlock too short.  "
                 "It must be at least 4 bytes; it is %d bytes.",
                 length);
    else {
        Gif89.disposal = (buf[0] >> 2) & 0x7;
        Gif89.inputFlag = (buf[0] >> 1) & 0x1;
        Gif89.delayTime = LM_to_uint(buf[1],buf[2]);
        if ((buf[0] & 0x1) != 0)
            Gif89.transparent = buf[3];
        ReadThroughEod(fd);
    }
}



static void
DoExtension(FILE * const fd, int const label) {
    char * str;
    
    switch (label) {
    case 0x01:              /* Plain Text Extension */
        str = "Plain Text Extension";
#ifdef notdef
        GetDataBlock(fd, (unsigned char*) buf, &eof, &length);
        
        lpos   = LM_to_uint(buf[0], buf[1]);
        tpos   = LM_to_uint(buf[2], buf[3]);
        width  = LM_to_uint(buf[4], buf[5]);
        height = LM_to_uint(buf[6], buf[7]);
        cellw  = buf[8];
        cellh  = buf[9];
        foreground = buf[10];
        background = buf[11];
        
        while (GetDataBlock(fd, (unsigned char*) buf) != 0) {
            PPM_ASSIGN(image[ypos][xpos],
                       cmap[CM_RED][v],
                       cmap[CM_GREEN][v],
                       cmap[CM_BLUE][v]);
            ++index;
        }
        return;
#else
        break;
#endif
    case 0xff:              /* Application Extension */
        str = "Application Extension";
        break;
    case 0xfe:              /* Comment Extension */
        DoCommentExtension(fd);
        return;
    case 0xf9:              /* Graphic Control Extension */
        DoGraphicControlExtension(fd);
        return;
    default: {
        static char buf[256];
        str = buf;
        sprintf(buf, "UNKNOWN (0x%02x)", label);
        }
        break;
    }

    pm_message("got a '%s' extension", str );

    ReadThroughEod(fd);
}



static int
GetCode(FILE * const fd, 
        int    const code_size, 
        bool   const first)
{
       static unsigned char    buf[280];
       static int              curbit, lastbit, last_byte;
       static bool             done;
       int                     i, j, ret;

       if (first) {
               /* Fake a previous data block */
               buf[0] = 0;
               buf[1] = 0;
               last_byte  = 2;
               curbit = 16;
               lastbit = 16;

               done = FALSE;
               return 0;
       }

       if ( (curbit+code_size) >= lastbit) {
               unsigned int count;
               unsigned int assumed_count;
               bool eof;

               if (done) {
                       if (curbit >= lastbit)
                               pm_error("ran off the end of my bits" );
                       return -1;
               }
               buf[0] = buf[last_byte-2];
               buf[1] = buf[last_byte-1];

               GetDataBlock(fd, &buf[2], &eof, &count);
               if (eof) {
                       pm_message("EOF encountered in image "
                                  "before EOD marker.  The GIF "
                                  "file is malformed, but we are proceeding "
                                  "anyway as if an EOD marker were at the end "
                                  "of the file.");
                       assumed_count = 0;
               } else
                       assumed_count = count;
               if (assumed_count == 0)
                       done = TRUE;

               last_byte = 2 + assumed_count;
               curbit = (curbit - lastbit) + 16;
               lastbit = (2+assumed_count)*8 ;
       }

       ret = 0;
       for (i = curbit, j = 0; j < code_size; ++i, ++j)
               ret |= ((buf[ i / 8 ] & (1 << (i % 8))) != 0) << j;

       curbit += code_size;

       return ret;
}



static int
LWZReadByte(FILE * const fd, bool const first, int const input_code_size) {
/*----------------------------------------------------------------------------
   Return the next byte of the decompressed image.

   Return -1 if we hit EOF prematurely (i.e. before an "end" code.  We
   forgive the case that the "end" code is followed by EOF instead of
   an EOD marker (zero length DataBlock)).

   Return -2 if there are no more bytes in the image.
-----------------------------------------------------------------------------*/
       static int      fresh = FALSE;
       int             code, incode;
       static int      code_size, set_code_size;
       static int      max_code, max_code_size;
       static int      firstcode, oldcode;
       static int      clear_code, end_code;
       static int      table[2][(1<< MAX_LWZ_BITS)];
       static int      stack[(1<<(MAX_LWZ_BITS))*2], *sp;
       register int    i;

       if (first) {
               set_code_size = input_code_size;
               code_size = set_code_size+1;
               clear_code = 1 << set_code_size ;
               end_code = clear_code + 1;
               max_code_size = 2*clear_code;
               max_code = clear_code+2;

               GetCode(fd, 0, TRUE);
               
               fresh = TRUE;

               for (i = 0; i < clear_code; ++i) {
                       table[0][i] = 0;
                       table[1][i] = i;
               }
               for (; i < (1<<MAX_LWZ_BITS); ++i)
                       table[0][i] = table[1][i] = 0;

               sp = stack;

               return 0;
       } else if (fresh) {
               fresh = FALSE;
               do {
                       firstcode = oldcode =
                               GetCode(fd, code_size, FALSE);
               } while (firstcode == clear_code);
               return firstcode;
       }

       if (sp > stack)
               return *--sp;

       while ((code = GetCode(fd, code_size, FALSE)) >= 0) {
               if (code == clear_code) {
                       for (i = 0; i < clear_code; ++i) {
                               table[0][i] = 0;
                               table[1][i] = i;
                       }
                       for (; i < (1<<MAX_LWZ_BITS); ++i)
                               table[0][i] = table[1][i] = 0;
                       code_size = set_code_size+1;
                       max_code_size = 2*clear_code;
                       max_code = clear_code+2;
                       sp = stack;
                       firstcode = oldcode =
                                       GetCode(fd, code_size, FALSE);
                       return firstcode;
               } else if (code == end_code) {
                       if (ZeroDataBlock)
                               return -2;

                       ReadThroughEod(fd);

                       return -2;
               }

               incode = code;

               if (code >= max_code) {
                       *sp++ = firstcode;
                       code = oldcode;
               }

               while (code >= clear_code) {
                       *sp++ = table[1][code];
                       if (code == table[0][code])
                               pm_error("circular table entry BIG ERROR");
                       code = table[0][code];
               }

               *sp++ = firstcode = table[1][code];

               if ((code = max_code) <(1<<MAX_LWZ_BITS)) {
                       table[0][code] = oldcode;
                       table[1][code] = firstcode;
                       ++max_code;
                       if ((max_code >= max_code_size) &&
                               (max_code_size < (1<<MAX_LWZ_BITS))) {
                               max_code_size *= 2;
                               ++code_size;
                       }
               }

               oldcode = incode;

               if (sp > stack)
                       return *--sp;
       }
       return code;
}



static void
ReadImage(FILE *fd, pixel ** const image, const int len, const int height, 
          unsigned char cmap[3][MAXCOLORMAPSIZE], const int interlace) {

       unsigned char   c;      
       int             v;
       int             xpos = 0, ypos = 0, pass = 0;

       /*
       **  Initialize the Compression routines
       */
       if (! ReadOK(fd,&c,1))
               pm_error("EOF / read error on image data" );

       if (LWZReadByte(fd, TRUE, c) < 0)
               pm_error("error reading image" );

       if (verbose)
               pm_message("reading %d by %d%s GIF image",
                       len, height, interlace ? " interlaced" : "" );

       while ((v = LWZReadByte(fd,FALSE,c)) >= 0 ) {
               PPM_ASSIGN(image[ypos][xpos], cmap[CM_RED][v],
                                       cmap[CM_GREEN][v], cmap[CM_BLUE][v]);

               ++xpos;
               if (xpos == len) {
                       xpos = 0;
                       if (interlace) {
                               switch (pass) {
                               case 0:
                               case 1:
                                       ypos += 8; break;
                               case 2:
                                       ypos += 4; break;
                               case 3:
                                       ypos += 2; break;
                               }

                               if (ypos >= height) {
                                       ++pass;
                                       switch (pass) {
                                       case 1:
                                               ypos = 4; break;
                                       case 2:
                                               ypos = 2; break;
                                       case 3:
                                               ypos = 1; break;
                                       default:
                                               goto fini;
                                       }
                               }
                       } else {
                               ++ypos;
                       }
               }
               if (ypos >= height)
                       break;
       }

fini:
       if (LWZReadByte(fd,FALSE,c)>=0)
               pm_message("too much input data, ignoring extra...");

}



static void
WritePnm(FILE *outfile, pixel ** const image, 
         const int cols, const int rows,
         const int has_gray, const int has_color) {
    int format;
    const char *format_name;
           
    if (has_color) {
        format = PPM_FORMAT;
        format_name = "PPM";
    } else if (has_gray) {
        format = PGM_FORMAT;
        format_name = "PGM";
    } else {
        format = PBM_FORMAT;
        format_name = "PBM";
    }
    if (verbose) 
        pm_message("writing a %s file", format_name);
    
    if (outfile) 
        pnm_writepnm(outfile, image, cols, rows,
                     (pixval) GIFMAXVAL, format, FALSE);
}



static void
transparency_message(int transparent_index, 
                     unsigned char cmap[3][MAXCOLORMAPSIZE]) {
/*----------------------------------------------------------------------------
   If user wants verbose output, tell him that the color with index
   'transparent_index' is supposed to be a transparent background color.
   
   If transparent_index == -1, tell him there is no transparent background
   color.
-----------------------------------------------------------------------------*/
    if (verbose) {
        if (transparent_index == -1)
            pm_message("no transparency");
        else
            pm_message("transparent background color: rgb:%02x/%02x/%02x "
                       "Index %d",
                       cmap[CM_RED][transparent_index],
                       cmap[CM_GREEN][transparent_index],
                       cmap[CM_BLUE][transparent_index],
                       transparent_index
                );
    }
}



static void
output_alpha(FILE *alpha_file, pixel ** const image, 
             const int cols, const int rows, const int transparent_index,
             unsigned char cmap[3][MAXCOLORMAPSIZE]) {
/*----------------------------------------------------------------------------
   Output to file 'alpha_file' (unless it is NULL) the alpha mask for the
   image 'image', given that the color whose index in the color map 'cmap' is
   'transparent_index' is the transparent color.  The image, and thus the
   alpha mask have dimensions 'cols' by 'rows'.

   transparent_index == -1 means there are no transparent pixels.
-----------------------------------------------------------------------------*/

    if (alpha_file) {
        bit *alpha_row;  /* malloc'ed */
        xel transparent_color;

        if (transparent_index != -1) 
            PPM_ASSIGN(transparent_color, 
                       cmap[CM_RED][transparent_index],
                       cmap[CM_GREEN][transparent_index],
                       cmap[CM_BLUE][transparent_index]
                );
        
        alpha_row = pbm_allocrow(cols);

        pbm_writepbminit(alpha_file, cols, rows, FALSE);

        {
            int row;
            for (row = 0; row < rows; row++) {
                int col;
                for (col = 0; col < cols; col++) {
                    if (transparent_index != -1 && 
                        PNM_EQUAL(image[row][col], transparent_color))
                        alpha_row[col] = PBM_BLACK;
                    else 
                        alpha_row[col] = PBM_WHITE;
                }
                pbm_writepbmrow(alpha_file, alpha_row, cols, FALSE);
            }
        }
        pbm_freerow(alpha_row);
    }
}



static void
readGifHeader(FILE * const gifFile, struct GifScreen * const GifScreenP) {
/*----------------------------------------------------------------------------
   Read the header off the file gifFile, which is present positioned to the 
   beginning of a GIF file.  Return the info from it as *GifScreenP.
-----------------------------------------------------------------------------*/
    unsigned char   buf[16];
    char     version[4];


    if (! ReadOK(gifFile,buf,6))
        pm_error("error reading magic number" );
    
    if (strncmp((char *)buf,"GIF",3) != 0)
        pm_error("not a GIF file" );
    
    strncpy(version, (char *)buf + 3, 3);
    version[3] = '\0';
    
    if (verbose)
        pm_message("GIF format version is '%s'", version);
    
    if ((strcmp(version, "87a") != 0) && (strcmp(version, "89a") != 0))
        pm_error("bad version number, not '87a' or '89a'" );
    
    if (! ReadOK(gifFile,buf,7))
        pm_error("failed to read screen descriptor" );
    
    GifScreenP->Width           = LM_to_uint(buf[0],buf[1]);
    GifScreenP->Height          = LM_to_uint(buf[2],buf[3]);
    GifScreenP->ColorMapSize    = 2<<(buf[4]&0x07);
    GifScreenP->ColorResolution = (((buf[4]&0x70)>>3)+1);
    GifScreenP->Background      = buf[5];
    GifScreenP->AspectRatio     = buf[6];

    if (verbose) {
        pm_message("GIF Width = %d GIF Height = %d "
                   "Pixel aspect ratio = %d (%f:1)",
                   GifScreenP->Width, GifScreenP->Height, 
                   GifScreenP->AspectRatio, 
                   GifScreenP->AspectRatio == 0 ? 
                   1 : (GifScreenP->AspectRatio + 15) / 64.0);
        pm_message("Colors = %d   Color Resolution = %d",
                   GifScreenP->ColorMapSize, GifScreenP->ColorResolution);
    }           
    if (BitSet(buf[4], LOCALCOLORMAP)) {    /* Global Colormap */
        ReadColorMap(gifFile, GifScreenP->ColorMapSize, GifScreenP->ColorMap,
                     &GifScreenP->has_gray, &GifScreenP->has_color);
        if (verbose) {
            pm_message("Color map %s grays, %s colors", 
                       GifScreenP->has_gray ? "contains" : "doesn't contain",
                       GifScreenP->has_color ? "contains" : "doesn't contain");
        }
    }
    
    if (GifScreenP->AspectRatio != 0 && GifScreenP->AspectRatio != 49) {
        float   r;
        r = ( (float) GifScreenP->AspectRatio + 15.0 ) / 64.0;
        pm_message("warning - input pixels are not square, "
                   "but we are rendering them as square pixels "
                   "in the output.  "
                   "To fix the output, run it through "
                   "'pnmscale -%cscale %g'",
                   r < 1.0 ? 'x' : 'y',
                   r < 1.0 ? 1.0 / r : r );
    }
}



static void
ConvertImages(FILE *fd, const int imageNumber, 
              FILE *imageout_file, FILE *alphafile) {

    unsigned char   buf[16];
    unsigned char   c;
    unsigned char   localColorMap[3][MAXCOLORMAPSIZE];
    int      useGlobalColormap;
    int      imageCount = 0;
       
    readGifHeader(fd, &GifScreen);

    for (;;) {
        pixel **image;  /* The image, in libpnm format */
        int cols, rows;  /* Dimensions of the image */
        int      localColorMapSize;

        if (! ReadOK(fd,&c,1))
            pm_error("EOF / read error on image data" );

        if (c == ';') {         /* GIF terminator */
            if (imageCount < imageNumber)
                pm_error("You requested Image %d, but "
                         "only %d image%s found in file",
                         imageNumber,
                         imageCount, imageCount>1?"s":"" );
            return;
        }

        if (c == '!') {         /* Extension */
            if (! ReadOK(fd,&c,1))
                pm_error("EOF / "
                         "read error on extension function code");
            DoExtension(fd, c);
            continue;
        }

        if (c != ',') {         /* Not a valid start character */
            pm_message("bogus character 0x%02x, ignoring", (int) c );
            continue;
        }

        ++imageCount;

        if (! ReadOK(fd,buf,9))
            pm_error("couldn't read left/top/width/height");

        useGlobalColormap = ! BitSet(buf[8], LOCALCOLORMAP);

        localColorMapSize = 1<<((buf[8]&0x07)+1);

        cols = LM_to_uint(buf[4],buf[5]);
        rows = LM_to_uint(buf[6],buf[7]);

        image = pnm_allocarray(cols, rows);
        if (!image)
            pm_error("couldn't alloc space for image" );

        if (! useGlobalColormap) {
            int has_gray, has_color;

            ReadColorMap(fd, localColorMapSize, localColorMap, 
                         &has_gray, &has_color);
            transparency_message(Gif89.transparent, localColorMap);
            ReadImage(fd, image, cols, rows, localColorMap, 
                      BitSet(buf[8], INTERLACE));
            if (imageCount == imageNumber)
                WritePnm(imageout_file, image, cols, rows,
                         has_gray, has_color);
            output_alpha(alphafile, image, cols, rows, 
                         Gif89.transparent, localColorMap);
        } else {
            transparency_message(Gif89.transparent, GifScreen.ColorMap);
            ReadImage(fd, image, cols, rows, GifScreen.ColorMap, 
                      BitSet(buf[8], INTERLACE));
            if (imageCount == imageNumber)
                WritePnm(imageout_file, image, cols, rows,
                         GifScreen.has_gray, GifScreen.has_color);
            output_alpha(alphafile, image, cols, rows, 
                         Gif89.transparent, GifScreen.ColorMap);
        }

        pnm_freearray(image, rows);
    }
}



int
main(int argc, char **argv) {

    struct cmdline_info cmdline;
    FILE *ifp;
    FILE *alpha_file, *imageout_file;

    pnm_init(&argc, argv);

    parse_command_line(argc, argv, &cmdline);
    verbose = cmdline.verbose;
    showComment = cmdline.comments;
   
    ifp = pm_openr(cmdline.input_filespec);

    if (cmdline.alpha_stdout)
        alpha_file = stdout;
    else if (cmdline.alpha_filename == NULL) 
        alpha_file = NULL;
    else {
        alpha_file = pm_openw(cmdline.alpha_filename);
    }

    if (cmdline.alpha_stdout) 
        imageout_file = NULL;
    else
        imageout_file = stdout;

    ConvertImages(ifp, cmdline.image_no, imageout_file, alpha_file);

    pm_close(ifp);
    if (imageout_file != NULL) 
        pm_close( imageout_file );
    if (alpha_file != NULL)
        pm_close( alpha_file );

    exit(0);
}




