/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
-----------------------------------------
file	:	SAMLEvidence.cpp
project	:	Policy Server
date	:	
author	:	andy
rights	:	(c) Parthenon Computing
notes	:	 extension to openSAML C++ library (based on Java class)
--------------------------------------------
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLEvidence::SAMLEvidence(const Iterator<SAMLAssertion*>& assertions, const Iterator<const XMLCh*>& assertionIDRefs)
{
    init(assertions, assertionIDRefs);
}

SAMLEvidence::SAMLEvidence(const Iterator<const XMLCh*>& assertionIDRefs)
{
    init(EMPTY(SAMLAssertion*), assertionIDRefs);
}

void SAMLEvidence::init(const Iterator<SAMLAssertion*>& assertions, const Iterator<const XMLCh*>& assertionIDRefs)
{
    RTTI(SAMLEvidence);
    while(assertions.hasNext())
        m_assertions.push_back(static_cast<SAMLAssertion*>(assertions.next()->setParent(this)));
    while(assertionIDRefs.hasNext())
        m_assertionIDRefs.push_back(XML::assign(assertionIDRefs.next()));
}

SAMLEvidence::SAMLEvidence(DOMElement* e)
{
    RTTI(SAMLEvidence);
    fromDOM(e);
}

SAMLEvidence::SAMLEvidence(std::istream& in) : SAMLObject(in)
{
    RTTI(SAMLEvidence);
    fromDOM(m_document->getDocumentElement());
}

SAMLEvidence::~SAMLEvidence()
{
    for (vector<SAMLAssertion*>::const_iterator i=m_assertions.begin(); i!=m_assertions.end(); i++)
        delete (*i);

    if(m_bOwnStrings) {
        for (vector<const XMLCh*>::const_iterator i=m_assertionIDRefs.begin(); i!=m_assertionIDRefs.end(); i++) {
            XMLCh* p = const_cast<XMLCh*>(*i);
            XMLString::release(&p);
        }
    }
}

void SAMLEvidence::ownStrings()
{
    if (!m_bOwnStrings) {
        for (vector<const XMLCh*>::iterator i=m_assertionIDRefs.begin(); i!=m_assertionIDRefs.end(); i++)
            (*i)=XML::assign(*i);
        m_bOwnStrings=true;
    }
}

void SAMLEvidence::fromDOM(DOMElement *e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(Evidence)))
        throw MalformedException("SAMLEvidence::fromDOM() requires saml:Evidence at root");

    DOMElement* n=XML::getFirstChildElement(e);
    while (n) {
        if (XML::isElementNamed(n,XML::SAML_NS, L(Assertion))) {
            SAMLAssertion* a=new SAMLAssertion(n);
            m_assertions.push_back(static_cast<SAMLAssertion*>(a->setParent(this)));
        }
        else if (XML::isElementNamed(n,XML::SAML_NS, L(AssertionIDReference))) {
            if (n->hasChildNodes())
                m_assertionIDRefs.push_back(n->getFirstChild()->getNodeValue());
        }
        else
            SAML_log.warn("skipping unrecognized child element");
        n=XML::getNextSiblingElement(n);
    }
    checkValidity();
}

void SAMLEvidence::setAssertions(const Iterator<SAMLAssertion*>& assertions)
{
    while (m_assertions.size())
        removeAssertion(0);
    while (assertions.hasNext())
        addAssertion(assertions.next());
}

void SAMLEvidence::addAssertion(SAMLAssertion* assertion)
{
    if (assertion) {
        m_assertions.push_back(static_cast<SAMLAssertion*>(assertion->setParent(this)));
        ownStrings();
        setDirty();
    }
    else
        throw SAMLException("assertion cannot be null");
}

void SAMLEvidence::removeAssertion(unsigned long index)
{
    SAMLAssertion* kill=m_assertions[index];
    m_assertions.erase(m_assertions.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

void SAMLEvidence::setAssertionIDRefs(const Iterator<const XMLCh*>& assertionIDRefs)
{
    while (m_assertionIDRefs.size())
        removeAssertionIDRef(0);
    while (assertionIDRefs.hasNext())
        addAssertionIDRef(assertionIDRefs.next());
}

void SAMLEvidence::addAssertionIDRef(const XMLCh* ref)
{
    if (XML::isEmpty(ref))
        throw SAMLException("IDRef cannot be null or empty");
    
    ownStrings();
    m_assertionIDRefs.push_back(XML::assign(ref));
    setDirty();
}

void SAMLEvidence::removeAssertionIDRef(unsigned long index)
{
    if (m_bOwnStrings) {
        XMLCh* ch=const_cast<XMLCh*>(m_assertionIDRefs[index]);
        XMLString::release(&ch);
    }
    m_assertionIDRefs.erase(m_assertionIDRefs.begin()+index);
    ownStrings();
    setDirty();
}

DOMElement* SAMLEvidence::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* e=doc->createElementNS(XML::SAML_NS, L(Evidence));
    if (xmlns)
        e->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return e;
}

DOMNode* SAMLEvidence::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc, xmlns);
    DOMElement* e=static_cast<DOMElement*>(m_root);
    doc=m_root->getOwnerDocument();
    
    if (m_bDirty) {
        for (vector<SAMLAssertion*>::const_iterator i=m_assertions.begin(); i!=m_assertions.end(); i++)
            e->appendChild((*i)->toDOM(doc,false));
    
        for (vector<const XMLCh*>::const_iterator j=m_assertionIDRefs.begin(); j!=m_assertionIDRefs.end(); j++)
            e->appendChild(doc->createElementNS(XML::SAML_NS,L(AssertionIDReference)))->appendChild(doc->createTextNode(*j));
        
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(e,XML::SAML_NS);
    }
    return e;
}

void SAMLEvidence::checkValidity() const
{
    if(m_assertions.empty() && m_assertionIDRefs.empty())
        throw MalformedException("SAMLEvidence() requires at least one of either an assertion or an assertion ID reference");
}

SAMLObject* SAMLEvidence::clone() const
{
    return new SAMLEvidence(getAssertions().clone(), m_assertionIDRefs);
}
