#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef linux
#include <net/if_arp.h>
#endif

#include <config.h>
#include <support.h>
#include <osdep.h>
#include <xcio.h>

#include <sysmsg.h>
#include <option.h>
#include <log.h>
#include <phase.h>
#include <env.h>
#include <console.h>
#include <timer.h>
#include <frame.h>
#include <fsm.h>
#include <cps.h>
#include <dev/device.h>

#include "ipsupport.h"
#include "ipcp.h"

struct ipcpopt_s ipcpOpt;

static char *TostrIPaddr();
static char *TostrIPcproto();
static void IpcpInit();

ipatflag_t ipAtFlag;

#define	IPCPCODE_MAX	CPCODE_CODE_REJ

#define	IPCPCONF_ADDRS	1
#define	IPCPCONF_CPROTO	2
#define	IPCPCONF_ADDR	3
#define	IPCPCONF_DNS1	129
#define	IPCPCONF_WINS1	130
#define	IPCPCONF_DNS2	131
#define	IPCPCONF_WINS2	132
#define	IPCPCONF_MAX	133

static struct ipcpreg_s xReg;
struct ipcpreg_s ipcprReg, ipcplReg;

static struct fsmreg_s ipcpFsm;
static struct cpcodeop_s ipcpCop[IPCPCODE_MAX + 1];
static struct cptypeop_s ipcpTop[]={
    {
	"IP-Addresses", NULL,
	NULL, NULL, 0,
	IPCPCONF_ADDRS
    }, {
	"IP-Compression-Protocol", TostrIPcproto,
	&xReg.cproto, NULL, 4,
	IPCPCONF_CPROTO
    }, {
	"IP-Address", TostrIPaddr,
	&xReg.addr, NULL, 4,
	IPCPCONF_ADDR
    }, {
	"Primary-DNS-Address", TostrIPaddr,
	&xReg.dns1, NULL, 4,
	IPCPCONF_DNS1
    }, {
	"Secondary-DNS-Address", TostrIPaddr,
	&xReg.dns2, NULL, 4,
	IPCPCONF_DNS2
    }, {
	"Primary-WINS-Address", TostrIPaddr,
	&xReg.wins1, NULL, 4,
	IPCPCONF_WINS1
    }, {
	"Secondary-WINS-Address", TostrIPaddr,
	&xReg.wins2, NULL, 4,
	IPCPCONF_WINS2
    }
};

static cptype_t remoteOrder[IPCPCONF_MAX], localOrder[IPCPCONF_MAX];

static struct cpstate_s ipcpState={
    ipcpCop,		/* code oprators */
    ipcpTop,		/* type operators */
    (void *)&ipcprReg,	/* remote registry */
    (void *)&ipcplReg,	/* local registry */
    NULL,		/* received buffer pointers */
    NULL,		/* encode order */
    remoteOrder,	/* remote(received) order */
    localOrder,		/* request(local) order */
    {{0}},		/* encode list */
    {{0}},		/* remote(received) list */
    {{0}},		/* request(local) list */
    /* number of configuration types */
    sizeof(ipcpTop)/sizeof(struct cptypeop_s),
    NBO_PROTO_IPCP,
    0, 0		/* remote/local ID */
};

inline bool_t
CheckAndSearchRemoteIp(struct in_addr *addr)
{
    struct in_addr nbo_addr;
    bool_t given=TRUE;		/* given addr is good */
    u_long hbo_sa;
    extern bool_t SysIpAddressCheck();
    extern char *ifName;

    hbo_sa = ntohl(addr->s_addr);
    while (1) {
	nbo_addr.s_addr = htonl(hbo_sa);
	if (SysIpAddressCheck(nbo_addr, ifName))
	    if (!ipcpOpt.r_addr.s_addr ||
		(ipcpOpt.r_addr.s_addr & ipcpOpt.r_mask.s_addr)
		== (xReg.addr.s_addr & ipcpOpt.r_mask.s_addr)) break;
	if (given) {
	    given = FALSE;
/*
	    if (ipcpOpt.r_addr.s_addr) nbo_addr.s_addr =
		ipcpOpt.r_addr.s_addr & ipcpOpt.r_mask.s_addr;
*/
	    hbo_sa = ntohl(nbo_addr.s_addr);
	}
	hbo_sa ++;
    }
    if (given == FALSE) addr->s_addr = nbo_addr.s_addr;
    return(given);
}

static char *
TostrIPcproto(void *cproto)
{
    struct cproto_s *cps;
    static char str[40];

    cps = (struct cproto_s *)cproto;
    switch (cps->nbo_cproto) {
    case NBO_PROTO_VJC:	/* VJ comp */
	sprintf(str, "VJ (Max=%d, Comp=%d)",
		cps->opt.vj.max_slot_id,
		cps->opt.vj.cmp_slot_id);
	break;
    default:
	sprintf(str, "%#04x", ntohs(cps->nbo_cproto));
    }
    return(str);
}

static char *
TostrIPaddr(void *addr)
{
    return(ISLOG(LOG_PRIVATE) ?
	   inet_ntoa(*(struct in_addr *)addr): "x.x.x.x");
}

static int
IpcpEncodeReg(u_char *buf, struct ipcpreg_s *icr)
{
    memcpy(&xReg, icr, sizeof(struct ipcpreg_s));
    return(CpEncodeReg(&ipcpState, buf, FALSE));
}

/* FSM action functions */

static void
TLD(struct fsmreg_s *fsm)
{
    const char *oldpath;

    if ((pppInfo.l_stat & LSTAT_NLINK) && ipcpOpt.stop) {
	oldpath = SetLoadFilePath("ip");
	LoadScriptFile(ipcpOpt.stop);
	SetLoadFilePath(oldpath);
    }
    if (pppInfo.n_stat & NSTAT_IP) {
	if (ipcpOpt.down) {
	    oldpath = SetLoadFilePath("ip");
	    LoadScriptFile(ipcpOpt.down);
	    SetLoadFilePath(oldpath);
	}
	pppInfo.n_stat &= ~NSTAT_IP;
    }
    if (ipcpOpt.restoreroute) SysIpRestoreRoute();
    DnsRestoreConf();
    DnsRelay(NULL);
    if (ipcpOpt.proxyarp) SysProxyArp(0);
}

static void
TLF(struct fsmreg_s *fsm)
{
/*    phaseShift = PHASE_DOWN;*/
    FsmInit(fsm);
    memset(&ipcplReg, 0, sizeof(ipcplReg));
    memset(&ipcprReg, 0, sizeof(ipcprReg));
    VjInit(FALSE);
    IpcpInit();
}

static void
IpDown(bool_t process)
{
    FsmClose(&ipcpFsm);
    if (!process) {
	TLD(&ipcpFsm);
	TLF(&ipcpFsm);
    }
    /*    pppInfo.p[PINFO_IPCP].st = PIST_DOWN;*/
}

void
IpUp()
{
    const char *oldpath;

    if (ipcpOpt.restoreroute) SysIpSaveRoute();
/*    if (!(pppInfo.l_stat & LSTAT_NLINK)) IpcpInit();*/
    if (!(pppInfo.n_stat & NSTAT_IP)) IpcpInit();
    pppInfo.n_stat |= NSTAT_IP;
    if (ipcprReg.cproto.nbo_cproto == NBO_PROTO_VJC) VjInit(TRUE);
    if (ipcpOpt.res_conf != DNSRES_NO || ipcpOpt.dnsrelay) {
	int n;
	struct in_addr dns[MAX_DNSERVER];
	extern int DnsSetupConf();

	memcpy(dns, ipcpOpt.dns, sizeof(dns));
	if (ipcpOpt.res_conf != DNSRES_FIX) {
	    if (ipcplReg.dns1.s_addr)
		dns[0].s_addr = ipcplReg.dns1.s_addr;
	    if (ipcplReg.dns2.s_addr)
		dns[1].s_addr = ipcplReg.dns2.s_addr;
	}
	for (n = 0; n < MAX_DNSERVER; n ++) if (dns[n].s_addr) {
	    if (ipcpOpt.res_conf)
		DnsSetupConf(ipcpOpt.domain, dns, MAX_DNSERVER);
	    if (ipcpOpt.dnsrelay) DnsRelay(dns);
	    break;
	}
    }
    if (ipcpOpt.up) {
	oldpath = SetLoadFilePath("ip");
	LoadScriptFile(ipcpOpt.up);
	SetLoadFilePath(oldpath);
    }
    if (ipcpOpt.proxyarp) SysProxyArp(ipcprReg.addr.s_addr);
    if (ipAtFlag) {
	extern void IpMapperAdd(), IpXFilterRemp();

	IpXFilterRemap();
	if (ipAtFlag == IPAT_ALL) IpMapperAdd();
	RebuildQueue(NBO_PROTO_IP);
	ipAtFlag = IPAT_NONE;
    }
    RegisterDownPhase("IPCP", IpDown);
    if ((pppInfo.l_stat & LSTAT_NLINK) && ipcpOpt.start) {
	oldpath = SetLoadFilePath("ip");
	LoadScriptFile(ipcpOpt.start);
	SetLoadFilePath(oldpath);
    }
}

static void
TLU(struct fsmreg_s *fsm)
{
    bool_t fail=FALSE;

    pppInfo.l_stat |= LSTAT_NLINK;
    pppInfo.n_stat |= NSTAT_IP;
    if (!B256_ISSET(&ipcpState.l_list, IPCPCONF_ADDR)) {
	ConsoleMsg(MS_E_NO_LOCALADDRESS, NULL);
	return;
    }
    if (!B256_ISSET(&ipcpState.r_list, IPCPCONF_ADDR)) {
	ConsoleMsg(MS_E_NO_REMOTEADDRESS, NULL);
	return;
    }
    IpUp();
}

/* FSM event functions */

static int
RCR(u_char *buf, int n)
{
    int i=0;
    cptype_t type;
    b256_t rej, nak;

    memset(&xReg, 0, sizeof(struct ipcpreg_s));
    if (CpDecodeReg(&ipcpState, buf, n) < 0) {
	FsmRUC(&ipcpFsm);
	return(0);
    }
    ipcpFsm.nak_rcr = ipcpFsm.rej_rcr = 0;
    B256_ZERO(&nak);
    B256_ZERO(&rej);
    while ((type = remoteOrder[i++]) < IPCPCONF_MAX) {
	switch (type) {
	case IPCPCONF_ADDR:
	    if (!ipcpOpt.r_addr.s_addr ||
		(ipcpOpt.r_addr.s_addr & ipcpOpt.r_mask.s_addr)
		== (xReg.addr.s_addr & ipcpOpt.r_mask.s_addr)) {
#ifdef	linux
		if (CheckAndSearchRemoteIp(&xReg.addr)) continue;
#else		/* for temp. */
		continue;
#endif
	    } else
		xReg.addr.s_addr = ipcpOpt.r_addr.s_addr;
	    break;
	case IPCPCONF_WINS1:
	case IPCPCONF_WINS2:
	    if (!xReg.wins1.s_addr && !xReg.wins2.s_addr) continue;
	    break;
	case IPCPCONF_DNS1:
	    if (ipcpOpt.res_conf != DNSRES_NO || ipcpOpt.dnsrelay
		|| !ipcpOpt.dns[0].s_addr) goto reject;
	    if (xReg.dns1.s_addr != ipcpOpt.dns[0].s_addr) {
		xReg.dns1.s_addr = ipcpOpt.dns[0].s_addr;
		break;
	    }
	    continue;
	case IPCPCONF_DNS2:
	    if (ipcpOpt.res_conf != DNSRES_NO || ipcpOpt.dnsrelay
		|| !ipcpOpt.dns[1].s_addr) goto reject;
	    if (xReg.dns2.s_addr != ipcpOpt.dns[1].s_addr) {
		xReg.dns2.s_addr = ipcpOpt.dns[1].s_addr;
		break;
	    }
	    continue;
	case IPCPCONF_CPROTO:
	    switch (xReg.cproto.nbo_cproto) {
	    case NBO_PROTO_VJC:
		if (ipcpOpt.vjcmp) continue;
	    }
	reject:
	default:
	    ipcpFsm.rej_rcr = 1;
	    B256_SET(&rej, type);
	    continue;
	}
	ipcpFsm.nak_rcr = 1;
	B256_SET(&nak, type);
    }
    memcpy(&ipcprReg, &xReg, sizeof(struct ipcpreg_s));
    if (ipcpFsm.rej_rcr) B256_CPY(&ipcpState.r_list, &rej);
    else if (ipcpFsm.nak_rcr) B256_CPY(&ipcpState.r_list, &nak);
    FsmRCR(&ipcpFsm);
    return(0);
}

static int
RCA(u_char *buf, int n)
{
    memset(&xReg, 0, sizeof(struct ipcpreg_s));
    CpDecodeReg(&ipcpState, buf, n);
/*    ipcpFsm.good_rcr = localReject ? 0: 1;*/
    FsmRCA(&ipcpFsm);
    return(0);
}

static int
RCNJ(u_char *buf, int n, bool_t rcj)
{
    int i=0;
    cptype_t type;

    memset(&xReg, 0, sizeof(struct ipcpreg_s));
    CpDecodeReg(&ipcpState, buf, n);
    while ((type = remoteOrder[i++]) < IPCPCONF_MAX) {
	if (rcj) B256_CLR(&ipcpState.l_list, type);
	switch (type) {
	case IPCPCONF_ADDR:
	    if (!ipcpOpt.l_addr.s_addr ||
		(ipcpOpt.l_addr.s_addr & ipcpOpt.l_mask.s_addr)
		== (xReg.addr.s_addr & ipcpOpt.l_mask.s_addr)) {
		/* translate only queued packets */
		ipAtFlag = IPAT_QUEUE;
		ipcplReg.addr.s_addr = xReg.addr.s_addr;
/*		if (IpAddressCheck(xReg.addr)) continue;*/
		continue;
	    } else if (ipcpOpt.l_addr.s_addr
		       && (ipcpOpt.l_mask.s_addr == INADDR_NONE)) {
		/* translate all packets */
		ipAtFlag = IPAT_ALL;
		ipcplReg.addr.s_addr = xReg.addr.s_addr;
		continue;
	    }
	    break;
	case IPCPCONF_DNS1:
	case IPCPCONF_DNS2:
	    if (!rcj) {
		ipcplReg.dns1.s_addr = xReg.dns1.s_addr;
		ipcplReg.dns2.s_addr = xReg.dns2.s_addr;
	    }
	    break;
	case IPCPCONF_CPROTO:
	    ipcplReg.cproto.nbo_cproto = 0;
	    break;
	}
    }
    FsmRCN(&ipcpFsm);
    return(0);
}

static int
RCN(u_char *buf, int n)
{
    return(RCNJ(buf, n, FALSE));
}

static int
RCJ(u_char *buf, int n)
{
    return(RCNJ(buf, n, TRUE));
}

static int
RTR(u_char *buf, int n)
{
    return(CpRtr(&ipcpFsm, buf, n));
}

static int
RTA(u_char *buf, int n)
{
    return(CpRta(&ipcpFsm, buf, n));
}

#if 0
static int
RXJ(u_char *buf, int n)
{
    int i=0;
    cptype_t type;

    memset(&xReg, 0, sizeof(struct ipcpreg_s));
    CpDecodeReg(&ipcpState, buf, n);
    ipcpFsm.fatal_rxj = 0;
    while ((type = remoteOrder[i++]) < IPCPCONF_MAX) {
	switch (type) {
	case IPCPCONF_CPROTO:
	    ipcplReg.cproto.nbo_cproto = 0;
	    B256_CLR(&ipcpState.l_list, IPCPCONF_CPROTO);
	    break;
	default:
	    break;
	}
    }
    FsmRXJ(&ipcpFsm);
}
#endif

static bool_t
EnvNSaddr(int argc, char *argv[], char *outs)
{
    int i, max;
    struct in_addr *nslist;
    char *name;

    if ((name = strchr(argv[0], '.')) != NULL) name ++;
    else name = argv[0];
    if (!strcasecmp(name, "WINS")) {
	nslist = ipcpOpt.wins;
	max = MAX_WINSERVER;
    } else {
	nslist = ipcpOpt.dns;
	max = MAX_DNSERVER;
    }
    if (!argc) {
	char *p;

	p = outs;
	for (i = 0; i < max; i ++) if (nslist[i].s_addr)
	    p += SprintF(p, "%s ", inet_ntoa(nslist[i]));
	return FALSE;
    }
    for (i = 0; i < max; i ++)
	nslist[i].s_addr = ((i + 1) < argc) ? inet_addr(argv[i + 1]): 0;
    return TRUE;
}

static bool_t
EnvDomain(int argc, char *argv[], char *outs)
{
    int i;
    char tmp[256];

    if (!argc) {
	if (ipcpOpt.domain) strcpy(outs, ipcpOpt.domain);
	return FALSE;
    }
    if (ipcpOpt.domain) free(ipcpOpt.domain);
    tmp[0] = '\0';
    for (i = 1; i < argc; i ++) {
	strcat(tmp, argv[i]);
	strcat(tmp, " ");
    }
    ipcpOpt.domain = strdup(tmp);
    return TRUE;
}

static bool_t
EnvIPaddr(int argc, char *argv[], char *outs)
{
    struct in_addr *addr, *mask, work;
    char *mp, *name;

    if ((name = strchr(argv[0], '.')) != NULL) name ++;
    else name = argv[0];
    addr = mask = NULL;
    if (!argc) {
	char *p;

	if (!strcasecmp(name, "REMOTE")) {
	    if (pppInfo.n_stat & NSTAT_IP) {
		if ((pppInfo.l_stat & LSTAT_NLINK) && ipcprReg.addr.s_addr)
		  addr = &ipcprReg.addr;
		else
		  addr = &ipcpOpt.r_addr;
	    } else {
		addr = &ipcpOpt.r_addr;
		mask = &ipcpOpt.r_mask;
	    }
	} else if (!strcasecmp(name, "LOCAL")) {
	    if (pppInfo.n_stat & NSTAT_IP) {
		if ((pppInfo.l_stat & LSTAT_NLINK)
		    && ipAtFlag != IPAT_ALL)
		    addr = &ipcplReg.addr;
		else
		    addr = &ipcpOpt.l_addr;
	    } else {
		addr = &ipcpOpt.l_addr;
		mask = &ipcpOpt.l_mask;
	    }
	} else {
	    if ((pppInfo.n_stat & NSTAT_IP) && !ipcpOpt.netmask.s_addr) {
		u_long hbo_addr;

		if (pppInfo.l_stat & LSTAT_NLINK)
		    hbo_addr = ntohl(ipcplReg.addr.s_addr);
		else
		    hbo_addr = ntohl(ipcpOpt.l_addr.s_addr);
		addr = &work;
		if (IN_CLASSA(hbo_addr))
		    work.s_addr = htonl(IN_CLASSA_NET);
		else if (IN_CLASSB(hbo_addr))
		    work.s_addr = htonl(IN_CLASSB_NET);
		else
		    work.s_addr = htonl(IN_CLASSC_NET);
	    } else {
		addr = &ipcpOpt.netmask;
		work.s_addr = addr->s_addr;
	    }
	    if (!strcasecmp(name , "NETWORK")) {
		if (!(pppInfo.n_stat & NSTAT_IP)) return FALSE;
		addr = &work;
		work.s_addr &= ((pppInfo.l_stat & LSTAT_NLINK)
				&& ipcprReg.addr.s_addr) ?
		    ipcprReg.addr.s_addr: ipcpOpt.r_addr.s_addr;
	    }
	    mask = NULL;
	}
	p = outs;
	p += SprintF(p, "%s", inet_ntoa(*addr));
	if (mask) sprintf(p, "/%s", inet_ntoa(*mask));
	return FALSE;
    }
    if (pppInfo.n_stat & NSTAT_IP) return FALSE;
    if (!strcasecmp(name, "REMOTE")) {
	addr = &ipcpOpt.r_addr;
	mask = &ipcpOpt.r_mask;
    } else if (!strcasecmp(name, "LOCAL")) {
	addr = &ipcpOpt.l_addr;
	mask = &ipcpOpt.l_mask;
    } else if (!strcasecmp(name, "NETMASK")) {
	addr = &ipcpOpt.netmask;
	mask = NULL;
    } else return FALSE;
    if (mask) {
	mask->s_addr = 0;
	if ((mp = strchr(argv[1], '/')) != NULL) {
	    char *p;

	    *mp = '\0';
	    mp ++;
	    if ((p = strchr(mp, '.')) != NULL)
		mask->s_addr = inet_addr(mp);
	    else {
		u_int b;

		if ((b = atoi(mp)) != 0)
		    mask->s_addr = htonl(~((1<<(32 - b)) - 1));
	    }
	}
    }
    addr->s_addr = inet_addr(argv[1]);
    return TRUE;
}

static bool_t
EnvResolv(int argc, char **argv, char *outs)
{
    dnsres_t d;
    unsigned l;
    const char *ress[]={
	"no",
	"fix",
	"yes",
    };

    if (!argc) {
	strcpy(outs, ress[ipcpOpt.res_conf]);
	return FALSE;
    }
    l = strlen(argv[1]);
    for (d = DNSRES_NO; d < DNSRES_MAX; d ++) {
	if (!strncasecmp(ress[d], argv[1], l)) {
	    ipcpOpt.res_conf = d;
	    return TRUE;
	}
    }
    ipcpOpt.res_conf = DNSRES_YES;
    return FALSE;
}

static void
IpcpEnable(bool_t sw)
{
    FrameEnable(NBO_PROTO_IPCP, sw);
}

void
IpcpSetup()
{
    static struct env_s envlist[]={
	{"UP", {&ipcpOpt.up}, ENV_STRING, 0, 0, 0666},
	{"DOWN", {&ipcpOpt.down}, ENV_STRING, 0, 0, 0666},
	{"LOCAL", {EnvIPaddr}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"REMOTE", {EnvIPaddr}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"NETMASK", {EnvIPaddr}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"NETWORK", {EnvIPaddr}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"DNS", {EnvNSaddr}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"WINS", {EnvNSaddr}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"DOMAIN", {EnvDomain}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"RESOLV", {EnvResolv}, ENV_SET, 0, 0, 0666},
	/*	{"DNSERVE", {&ipcpOpt.dnserve}, ENV_BOOL},*/
	{"DNSRELAY", {&ipcpOpt.dnsrelay}, ENV_BOOL, 0, 0, 0666},
	{"SLOCAL", {&ipcpOpt.s_local}, ENV_BOOL, 0, 0, 0666},
	{"RESTOREROUTE", {&ipcpOpt.restoreroute}, ENV_BOOL, 0, 0, 0666},
	{"PROXYARP", {&ipcpOpt.proxyarp}, ENV_BOOL, 0, 0, 0666},
	{"VJ", {&ipcpOpt.vjcmp}, ENV_BOOL, 0, 0, 0666},
	{"START", {&ipcpOpt.start}, ENV_STRING, 0, 0, 0666},
	{"STOP", {&ipcpOpt.stop}, ENV_STRING, 0, 0, 0666},
	{NULL}
    };

#if !dont_bc
    ipcpOpt.res_conf = DNSRES_YES;
#endif
    ipcpOpt.l_addr.s_addr = SysIpLocalAddress();
    RegisterEnvs(envlist, "IP", IpcpEnable);
    RegisterCp(&ipcpState, &ipcpFsm);
    ipcpFsm.name = "IPCP";
    ipcpFsm.timer.name = "IPCPFSM";
    ipcpFsm.timer.arg = &ipcpFsm;
    ipcpCop[CPCODE_CONF_REQ].fsmevent = RCR;
    ipcpCop[CPCODE_CONF_REQ].encode = IpcpEncodeReg;
    ipcpCop[CPCODE_CONF_ACK].fsmevent = RCA;
    ipcpCop[CPCODE_CONF_ACK].encode = IpcpEncodeReg;
    ipcpCop[CPCODE_CONF_NAK].fsmevent = RCN;
    ipcpCop[CPCODE_CONF_NAK].encode = IpcpEncodeReg;
    ipcpCop[CPCODE_TERM_REQ].fsmevent = RTR;
    ipcpCop[CPCODE_CONF_REJ].encode = IpcpEncodeReg;
    ipcpCop[CPCODE_TERM_ACK].fsmevent = RTA;
    ipcpCop[CPCODE_CONF_REJ].fsmevent = RCJ;
    ipcpFsm.tlu = TLU;
    ipcpFsm.tlf = TLF;
    ipcpFsm.tld = TLD;
    /*    strcpy(pppInfo.p[PINFO_IPCP].name, ipcpFsm.name);*/
}

static void
IpcpInit()
{
    int i, k;

    IpInit();
    ipAtFlag = IPAT_NONE;
    memset(&ipcprReg, 0, sizeof(struct ipcpreg_s));
    ipcpState.r_id = ipcpState.l_id = 0;
    B256_ZERO(&ipcpState.l_list);
    B256_SET(&ipcpState.l_list, IPCPCONF_ADDR);
    ipcplReg.addr.s_addr = ipcpOpt.s_local ? ipcpOpt.l_addr.s_addr: 0;
    if (ipcpOpt.res_conf == DNSRES_YES || ipcpOpt.dnsrelay) {
	B256_SET(&ipcpState.l_list, IPCPCONF_DNS1);
	B256_SET(&ipcpState.l_list, IPCPCONF_DNS2);
/*
	ipcplReg.dns1.s_addr = ipcpOpt.dns[0].s_addr;
	ipcplReg.dns2.s_addr = ipcpOpt.dns[1].s_addr;
*/
	ipcplReg.dns1.s_addr = ipcplReg.dns2.s_addr = 0; /* request */
    }
    /*
    if (ipcpOpt.dns[0].s_addr || ipcpOpt.dns[1].s_addr) {
    }
    */
    if (ipcpOpt.vjcmp) {
	B256_SET(&ipcpState.l_list, IPCPCONF_CPROTO);
	ipcplReg.cproto.nbo_cproto = NBO_PROTO_VJC;
	ipcplReg.cproto.opt.vj.max_slot_id = 15;
	ipcplReg.cproto.opt.vj.cmp_slot_id = 0;
    }
    for (k = 0, i = IPCPCONF_ADDRS; i < IPCPCONF_MAX; i ++)
	if (B256_ISSET(&ipcpState.l_list, i)) localOrder[k++] = i;
    localOrder[k] = ENDOF_LIST;
}

void
IpcpStart()
{
    ipcpFsm.init_r_count = pppOpt.r_count;
    ipcpFsm.init_r_to = pppOpt.r_to;
    TLF(&ipcpFsm);	/* IpcpInit() + FsmInit() */
    FsmUp(&ipcpFsm);
/*    ipcpFsm.opt_p = (pppOpt.mode == RUN_PASSIVE) ? 1: 0;*/
    ipcpFsm.opt_p = 0;
    FsmOpen(&ipcpFsm);
    /*    pppInfo.p[PINFO_IPCP].st = PIST_RUN;*/
}
