/* sa.c */

/* Copyright (C) 1993 Free Software Foundation, Inc.

This file is part of the GNU Accounting Utilities

The GNU Accounting Utilities are free software; you can redistribute
them and/or modify them under the terms of the GNU General Public
License as published by the Free Software Foundation; either version
2, or (at your option) any later version.

The GNU Accounting Utilities are distributed in the hope that they will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Accounting Utilities; see the file COPYING.  If
not, write to the Free Software Foundation, 675 Mass Ave, Cambridge,
MA 02139, USA.  */

#include "config.h"
#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <pwd.h>
#include <math.h>

#include "common.h"
#include "getopt.h"
#include "uid_hash.h"

/* defines, structures, and the like */

#define USERNAME_LEN 8		/* this value is standard now, but we
				   might need to FIXME later */

/* THIS SHOULD REALLY BE IN <sys/acct.h>!  I spent a long time trying
 * to figure out what the format was -- it should be public.
 * Actually, what does it matter?  Nobody else uses these files. */

struct stats {
  unsigned long num_calls;	/* how many times this command called */
  double user_time;		/* user time */
  double sys_time;		/* system time */
  double elapsed_time;		/* real time */
  double mem_usage;		/* total memory usage */
  double disk_io;		/* how many disk I/O operations */
};  

struct savacct {
  char name[CMD_LEN];		/* command name */
  char fork_flag;		/* nonzero if fork but no exec */
  struct stats s;
};

struct usracct {
  char name[USERNAME_LEN];
  struct stats s;
};

struct sum_entry {
  struct sum_entry *next;
  struct savacct s;
  char junked;
};

struct user_entry {
  struct user_entry *next;
  uid_t uid;
  struct usracct u;
};

/* sort types */

enum {
  sort_sys_plus_user,
  sort_sys_plus_user_div_calls,
  sort_avio,
  sort_tio,
  sort_cpu_mem_average,
  sort_cpu_storage,
  sort_num_calls
};

#ifndef HAVE_GETPAGESIZE
#define getpagesize() 4096
#endif

#include "files.h"

/* globals */

static char rcsid[] = "$Id: sa.c,v 1.5 1994/03/23 03:03:24 noel Exp noel $";

#define CMD_TABLE_SIZE 601	/* yet another prime */
struct sum_entry *command_list[CMD_TABLE_SIZE];	/* list of cmds */

struct stats stats_totals;	/* total of all commands */

struct user_entry *user_list[CMD_TABLE_SIZE]; /* list of user info */

char *program_name;		/* name of the program, for usage & errs */

int debugging_enabled;		/* print internal information relating
				   to entries read, data structures,
				   etc. */
double system_page_size;	/* size in kbytes of each page in
				   memory */

#define PAGES_TO_KB(x) ((double) (x) / system_page_size)

int print_seconds;		/* print seconds instead of minutes */
int separate_times;		/* print separate user and sys times */
int dont_read_summary_files;	/* guess! */
int print_ksec;			/* ^^^ */
int print_ratio;		/* ^^ */
int print_users;		/* ^ */
int total_io;			/* print total disk io operations */
int percentages;		/* include percentages in printout */
int user_summary_flag;		/* are we printing a user summary? */
int sort_type;			/* which field to sort by */
int reverse_sort;		/* nonzero if the sort is reversed */
int merge_files;		/* nonzero if we want to write new
				   savacct and usracct files */
int dont_separate_forks;	/* nonzero if we're to ignore the
				   fork-no-exec flag when grouping
				   command names */
int number_sort_options;
int junk_threshold;		/* if a command has this number of
				   calls or less, add it to "***junk" */
int print_all_records;		/* don't lump calls into the
				   "***other" category */
int always_yes;			/* nonzero means always answer yes to
				   a query */

char default_acct_file[] = ACCT_FILE;
char default_savacct_file[] = SAVACCT_FILE;
char default_usracct_file[] = USRACCT_FILE;

char *acct_file_name = default_acct_file; /* the file name */
char *savacct_file_name = default_savacct_file;
char *usracct_file_name = default_usracct_file;

long line_number = 0;		/* the line number of the file, for
				 * error reporting purposes */

/* prototypes */

void main (int, char *[]);
void give_usage (void);
void write_savacct_file (char *);
void write_usracct_file (char *);
void parse_savacct_entries (char *);
void parse_usracct_entries (char *);
void parse_acct_entries (char *);
void init_flags_and_data (void);
void setup_devices (void);
int hash_name (char *);
void update_command_list (char *, long, double, double, double,
			  double, double, char);
void update_user_list (char *, long, double, double, double, double, double);
void update_totals (long, double, double, double, double, double);
int compare_sum_entry (struct sum_entry **, struct sum_entry **);
int compare_user_entry (struct user_entry **, struct user_entry **);
int compare_stats_entry (struct stats *, struct stats *);
void print_command_list (void);
void print_user_list (void);
void print_stats_nicely (struct stats *);
int non_printable (char *, int);
int ask_if_junkable (char *, int);


/* code */

void
main (int argc, char *argv[])
{
  int c;

  init_flags_and_data ();
  program_name = argv[0];
  
  while (1)
    {
      int option_index = 0;
      
      static struct option long_options[] = {
	{ "debug", no_argument, NULL, 1 },
	{ "version", no_argument, NULL, 2 },
	{ "help", no_argument, NULL, 3 },
	{ "other-acct-file", required_argument, NULL, 4 },
	{ "print-seconds", no_argument, NULL, 5 },
	{ "dont-read-summary-files", no_argument, NULL, 6 },
	{ "list-all-names", no_argument, NULL, 7 },
	{ "separate-times", no_argument, NULL, 8 },
	{ "other-savacct-file", 1, NULL, 9 },
	{ "sort-ksec", no_argument, NULL, 10 },
	{ "print-ratio", no_argument, NULL, 11 },
	{ "print-users", no_argument, NULL, 12 },
	{ "sort-tio", no_argument, NULL, 13 },
	{ "percentages", no_argument, NULL, 14 },
	{ "sort-sys-user-div-calls", no_argument, NULL, 15 },
	{ "sort-avio", no_argument, NULL, 16 },
	{ "sort-cpy-avmem", no_argument, NULL, 17 },
	{ "sort-num-calls", no_argument, NULL, 18 },
	{ "not-interactive", no_argument, NULL, 19 },
	{ "user-summary", no_argument, NULL, 20 },
	{ "reverse-sort", no_argument, NULL, 21 },
	{ "merge", no_argument, NULL, 22 },
	{ "threshold", required_argument, NULL, 23 },
	{ "separate-forks", no_argument, NULL, 24 },
	{ "other-usracct-file", required_argument, NULL, 25 },
	{ 0, 0, 0, 0 }
      };
      
      c = getopt_long (argc, argv, "abcdfiljkmnrstuv:DK",
		       long_options, &option_index);
      
      if (c == EOF)
	break;

      switch (c)
	{
	case 1:
	  debugging_enabled = 1;
	  break;
	case 2:
	  fprintf (stderr, "%s\n", rcsid);
	  break;
	case 3:
	  give_usage ();
	  exit (1);
	  break;
	case 4:
	  acct_file_name = optarg;
	  break;
	case 'j':
	case 5:
	  print_seconds = 1;
	  break;
	case 'i':
	case 6:
	  dont_read_summary_files = 1;
	  break;
	case 'a':
	case 7:
	  print_all_records = 1;
	  break;
	case 'l':
	case 8:
	  separate_times = 1;
	  break;
	case 9:
	  savacct_file_name = optarg;
	  break;
	case 'K':
	case 10:
	  sort_type = sort_cpu_storage;
	  print_ksec = 1;
	  number_sort_options++;
	  break;
	case 't':
	case 11:
	  print_ratio = 1;
	  break;
	case 'u':
	case 12:
	  print_users = 1;
	  break;
	case 'D':
	case 13:
	  sort_type = sort_tio;
	  total_io = 1;
	  number_sort_options++;
	  break;
	case 'c':
	case 14:
	  percentages = 1;
	  break;
	case 'b':
	case 15:
	  sort_type = sort_sys_plus_user_div_calls;
	  number_sort_options++;
	  break;
	case 'd':
	case 16:
	  sort_type = sort_avio;
	  number_sort_options++;
	  break;
	case 'k':
	case 17:
	  sort_type = sort_cpu_mem_average;
	  number_sort_options++;
	  break;
	case 'n':
	case 18:
	  sort_type = sort_num_calls;
	  number_sort_options++;
	  break;
	case 'f':
	case 19:
	  always_yes = 1;
	  break;
	case 'm':
	case 20:
	  user_summary_flag = 1;
	  break;
	case 'r':
	case 21:
	  reverse_sort = 1;
	  break;
	case 's':
	case 22:
	  merge_files = 1;
	  break;
	case 'v':
	case 23:
	  junk_threshold = atoi (optarg);

	  if (junk_threshold < 1)
	    {
	      fprintf (stderr, "%s: threshold must be one or greater\n",
		       program_name);
	      exit (1);
	    }

	  break;
	case 24:
	  dont_separate_forks = 0;
	  break;
	case 25:
	  usracct_file_name = optarg;
	  break;
	default:
	  {
	    /* char ts[255];
	    sprintf (ts, "?? getopt returned char code 0%o ??", c);
	    fatal (ts); */
	    give_usage ();
	    exit (1);
	  }
	  break;
	}
    }

  /* check options */

  if (number_sort_options > 1)
    {
      fprintf (stderr, "%s: please specify only ONE sorting option!\n",
	       program_name);
      exit (1);
    }
  
  if (print_users || number_sort_options)
    {
      fprintf (stderr,
	       "%s: sorting options can't be specified with `--print-users'\n",
	       program_name);
      exit (1);
    }

  if (print_users && (merge_files || user_summary_flag))
    {
      fprintf (stderr,
	       "%s: can't specify `--merge' or `--user-summary' with `--print-users'\n", program_name);
      exit (1);
    }

  if (merge_files && user_summary_flag)
    {
      fprintf (stderr,
	       "%s: can't specify `--user-summary' with `--merge'\n",
	       program_name);
      exit (1);
    }

  if (optind == (argc - 1))
    {
      /* if we get here, we're expecting a filename */
      acct_file_name = argv[optind++];
    }
  else if (optind < argc)
    {
      fprintf (stderr, "%s: extra arguments ", program_name);

      while (optind < argc)
	{
	  fprintf (stderr, "`%s'", argv[optind++]);
	  if (optind < argc)
	    fprintf (stderr, ", ");
	}

      fprintf (stderr, " -- aborting\n");
      exit (1);
    }
  
  /* see if the summary file is there -- if so, load it and parse it */

  if (!dont_read_summary_files)
    {
      struct stat st;

#define NEED_SAVACCT_INFO (!print_users && !user_summary_flag)

      if (NEED_SAVACCT_INFO)
	if (stat (savacct_file_name, &st) == 0)
	  parse_savacct_entries (savacct_file_name);

#define NEED_USRACCT_INFO (!print_users && (merge_files || user_summary_flag))

      if (NEED_USRACCT_INFO)
	if (stat (usracct_file_name, &st) == 0)
	  parse_usracct_entries (usracct_file_name);
    }
  
  /* now, do the acct file */
  
  parse_acct_entries (acct_file_name);

  if (merge_files)
    {
      write_savacct_file (savacct_file_name);
      write_usracct_file (usracct_file_name);
    }
  else
    {
      if (user_summary_flag)
	print_user_list ();
      else
	print_command_list ();
    }

  exit (0);			/* guarantee the proper return value */
}


/* guess what this does... */
void
give_usage (void)
{
  char *usage = "\n\
Usage: %s [ options ] [ file ]\n\
\n\
       options: [-abcdfiljkmnrtuDK] [-v <num>] [--version] [--help]\n\
       [--other-acct-file <name>] [--other-usracct-file <name>]\n\
       [--print-seconds] [--dont-read-summary-files] [--debug]\n\
       [--separate-times] [--other-savacct-file <name>]\n\
       [--sort-ksec] [--print-ratio] [--print-users] [--sort-tio]\n\
       [--percentages] [--sort-sys-user-div-calls] [--sort-avio]\n\
       [--sort-cpu-avmem] [--sort-num-calls] [--merge]\n\
       [--user-summary] [--list-all-names] [--not-interactive]\n\
       [--threshold <num>]\n\
\n\
";
  
  fprintf (stderr, usage, program_name);
}

/**********************************************************************

IDEA -- use a TEXT format for savacct and usracct!  If users are going
to replace their system utilities with the GNU utilities, why do we
have to support the OS's files?  That saves everyone the hassle of
discovering the particular format of the usracct and savacct files.
Plus, if the format is text, it is easily readable and processable
WITHOUT having to invoke sa or the like.

Is a text format really good?  Let's see.  We want to record all of
this information:

  char name[CMD_LEN];		CMD_LEN bytes
  char exec_flag;		1
  double user_time;		4
  double sys_time;		4
  double elapsed_time;		4
  double mem_usage;		4
  double disk_io;		4
  long num_calls;		4
			      -----
                                25 + CMD_LEN bytes

Actually, this is NOT cool.  We could get some savings by recording
the name of the command with only as many characters as are in the
name.  That will be at most CMD_LEN characters.  Then, store a NULL
and then the rest of the record.  Worst case, this solution will be
WORSE than just saving CMD_LEN characters, since we store an extra
NULL in there.  On average, however, this should be a lot better,
since most unix commands are shorter than CMD_LEN characters.

FIXME -- haven't added the stuff to optimize the storage of the
command name yet.

**********************************************************************/

void
write_savacct_file (char *filename)
{
  struct sum_entry *np;
  long temp;
  char s[1000];
  FILE *fp;

  sprintf (s, "%s~", filename);

  fp = fopen (s, "w");
  if (fp == NULL)
    {
      fprintf (stderr,
	       "%s (write_savacct_file): Couldn't open temporary file `%s'",
	       program_name, s);
      exit (1);
    }

  for (temp = 0; temp < CMD_TABLE_SIZE; temp++)
    {
      for (np = command_list[temp]; np != NULL; np = np->next)
	{
	  if (fwrite (&(np->s), sizeof (np->s), 1, fp) == 0)
	    {
	      fprintf (stderr,
		       "%s (write_savacct_file): probs writing to file `%s'\n",
		       program_name, s);
	      exit (1);
	    }
	}
    }

  if (rename (s, filename) != 0)
    {
      perror ("sa");
      exit (1);
    }
}

void
write_usracct_file (char *filename)
{
  struct user_entry *np;
  long temp;
  char s[1000];
  FILE *fp;

  sprintf (s, "%s~", filename);

  fp = fopen (s, "w");
  if (fp == NULL)
    {
      fprintf (stderr,
	       "%s (write_usracct_file): Couldn't open temporary file `%s'",
	       program_name, s);
      exit (1);
    }

  for (temp = 0; temp < CMD_TABLE_SIZE; temp++)
    {
      for (np = user_list[temp]; np != NULL; np = np->next)
	{
	  if (fwrite (&(np->u), sizeof (np->u), 1, fp) == 0)
	    {
	      fprintf (stderr,
		       "%s (write_usracct_file): probs writing to file `%s'\n",
		       program_name, s);
	      exit (1);
	    }
	}
    }

  if (rename (s, filename) != 0)
    {
      perror ("sa");
      exit (1);
    }
}

/* parse the entries in a savacct file */

void
parse_savacct_entries (char *filename)
{
  struct savacct rec;
  FILE *fp;

  fp = open_binary (filename);

  if (debugging_enabled)
    fprintf (stddebug, "\
----------------------------------------------------------------------\n\
savacct entries in `%s'\n\
----------------------------------------------------------------------\n\
", filename);

  while (fread (&rec, sizeof (struct savacct), 1, fp))
    {
      update_command_list (rec.name, rec.s.num_calls, rec.s.user_time,
			   rec.s.sys_time, rec.s.elapsed_time, rec.s.disk_io,
			   rec.s.mem_usage, rec.fork_flag);

      update_totals (rec.s.num_calls, rec.s.user_time, rec.s.sys_time,
		     rec.s.elapsed_time, rec.s.disk_io, rec.s.mem_usage);
    }

  fclose (fp);
}


/* parse the entries in a usracct file */

void
parse_usracct_entries (char *filename)
{
  struct usracct rec;
  FILE *fp;

  fp = open_binary (filename);

  if (debugging_enabled)
    fprintf (stddebug, "\
----------------------------------------------------------------------\n\
usracct entries in `%s'\n\
----------------------------------------------------------------------\n\
", filename);

  while (fread (&rec, sizeof (struct usracct), 1, fp))
    {
      update_user_list (rec.name, rec.s.num_calls, rec.s.user_time,
			rec.s.sys_time, rec.s.elapsed_time, rec.s.disk_io,
			rec.s.mem_usage);

      /* only add these records to the total when we're doing a
         per-user summary of the data, because it will only screw up
         the percentages if we view a breakdown by command name */

      if (user_summary_flag)
	update_totals (rec.s.num_calls, rec.s.user_time, rec.s.sys_time,
		       rec.s.elapsed_time, rec.s.disk_io, rec.s.mem_usage);
    }

  fclose (fp);
}


/* parse the entries in an acct file */

void
parse_acct_entries (char *filename)
{
  struct acct *rec;             /* the current record */
  FILE *fp;

  fp = open_binary (filename);

  if (debugging_enabled)
    fprintf (stddebug, "\
----------------------------------------------------------------------\n\
acct entries in `%s'\n\
----------------------------------------------------------------------\n\
", filename);

  rec = (struct acct *) xmalloc (sizeof (struct acct));
  
  /* loop while there are entries to be had */
  while (fread (rec, sizeof (struct acct), 1, fp))
    {
#ifdef AC_UTIME_COMP_T
      double ut = comp_t_2_double (rec->ac_utime) / (double) AHZ;
#else
      double ut = (double) rec->ac_utime / (double) AHZ;
#endif

#ifdef AC_STIME_COMP_T
      double st = comp_t_2_double (rec->ac_stime) / (double) AHZ;
#else
      double st = (double) rec->ac_stime / (double) AHZ;
#endif

#ifdef AC_ETIME_COMP_T
      double et = comp_t_2_double (rec->ac_etime) / (double) AHZ;
#else
      double et = (double) rec->ac_etime / (double) AHZ;
#endif

#ifdef AC_IO_COMP_T
      double di = comp_t_2_double (rec->ac_io) / (double) AHZ;
#else
      double di = (double) rec->ac_io / (double) AHZ;
#endif

      /* Christoph Badura (bad@flatlin.ka.sub.org) says that ac_mem IS
	 NOT a comp_t (for Xenix) -- but some machines have it defined
	 that way, like suns. */

#ifdef 0			/* OLD CODE */
      double mu = (ut + st) * PAGES_TO_KB (comp_t_2_double (rec->ac_mem));
#endif

#ifdef AC_MEM_COMP_T
      double mu = PAGES_TO_KB (comp_t_2_double (rec->ac_mem));
#else
      double mu = PAGES_TO_KB (rec->ac_mem);
#endif
      /* if (debugging_enabled)
	fprintf (stddebug,
		 "%-10.8s ac_mem:%6.2f corr:%6.2f pgs:%6.2f\n",
		 rec->ac_comm, (double) rec->ac_mem,
		 comp_t_2_double (rec->ac_mem), mu); */

      if (print_users)
	{
	  printf ("%-8.8s %6.2f cpu %8.0fk mem %6.0f io %-10.*s%s\n",
		  uid_name (rec->ac_uid), ut + st,
		  ((ut + st) > 0) ? mu / (ut + st) : mu,
		  di,
		  CMD_LEN, rec->ac_comm,
		  (rec->ac_flag & AFORK) ? "*" : "");
	}
      else
	{
	  if (NEED_SAVACCT_INFO)
	    update_command_list (rec->ac_comm, 1, ut, st, et, di, mu,
				 (rec->ac_flag & AFORK));

	  if (NEED_USRACCT_INFO)
	    update_user_list (uid_name (rec->ac_uid), 1, ut, st, et, di, mu);
	  
	  update_totals (1, ut, st, et, di, mu);
	}
    }

  free (rec);

  fclose (fp);

  if (print_users) exit (0);
}


int
hash_name (char *s)
{
  unsigned h = 0, g, q;

  for (q = 0; (s[q] != '\0') && (q < CMD_LEN); q++)
    {
      h = (h << 4) + s[q];
      if ((g = h & 0xf0000000))
	{
	  h = h ^ (g >> 24);
	  h = h ^ g;
	}
    }
  
  return h % CMD_TABLE_SIZE;
}
  

void
update_totals (long num_calls, double user_time, double sys_time,
	       double elapsed_time, double disk_io, double mem_usage)

{
#define ADDIT(x); stats_totals.x += x;

  ADDIT (user_time);
  ADDIT (sys_time);
  ADDIT (elapsed_time);
  ADDIT (disk_io);
  ADDIT (mem_usage);
  ADDIT (num_calls);

#undef ADDIT
}

void
update_command_list (char *name, long num_calls, double user_time,
		     double sys_time, double elapsed_time, double disk_io,
		     double mem_usage, char fork_flag)
{
  /* Look for the command in the list.  If found, add the stats.  If
     not found, create a new entry and add the stats. */

  struct sum_entry *ap;
  int hashval;

  if (debugging_enabled)
    {
      fprintf (stddebug,
       "+:%-10.10s%s %10ld %10.2fu %10.2fs %10.2fe %10.2fio %10.2fmem\n",
	       name, (fork_flag) ? "*" : " ", num_calls,
	       user_time, sys_time, elapsed_time, disk_io, mem_usage);
    }
  
  hashval = hash_name (name);

  for (ap = command_list[hashval]; ap != NULL; ap = ap->next)
    {
      if ((strncmp (ap->s.name, name, CMD_LEN) == 0)
	  && (dont_separate_forks || (fork_flag == ap->s.fork_flag)))
	/* this is the one! */
	break;
    }
  
  /* If we didn't find it, AP will be null (and therefore we know when
   * to create a new record */

  if (ap == NULL)
    {
      ap = (struct sum_entry *) xmalloc (sizeof (struct sum_entry));
      strncpy (ap->s.name, name, CMD_LEN);
      ap->next = command_list[hashval];
      command_list[hashval] = ap;
      ap->junked = 0;
      ap->s.s.user_time = 0.0;
      ap->s.s.sys_time = 0.0;
      ap->s.s.elapsed_time = 0.0;
      ap->s.s.mem_usage = 0.0;
      ap->s.s.disk_io = 0.0;
      ap->s.s.num_calls = 0;
      ap->s.fork_flag = 0;
    }

#define ADDIT(thing); ap->s.s.thing += thing;

  ADDIT (user_time);
  ADDIT (sys_time);
  ADDIT (elapsed_time);
  ADDIT (disk_io);
  ADDIT (mem_usage);
  ADDIT (num_calls);

#undef ADDIT
  
  /* we want to OR the old value of AP->FORK_FLAG because we might
     not be separating statistics based on whether or not the command
     forked.  In that case, we want to record the fact that some of
     the commands forked without calling exec. */

  ap->s.fork_flag |= fork_flag;
}


/* Add the amount of time and resources used for the per-user
   summary. */

void
update_user_list (char *name, long num_calls, double user_time,
		  double sys_time, double elapsed_time, double disk_io,
		  double mem_usage)
{
  /* Look for the uid in the list.  If found, add the stats.  If
     not found, create a new entry and add the stats. */

  struct user_entry *ap;
  int hashval;

  if (debugging_enabled)
    {
      fprintf (stddebug,
	       "+u:%*s %10ld %10.2fu %10.2fs %10.2fe %10.2fio %10.2fmem\n",
	       USERNAME_LEN, name, num_calls,
	       user_time, sys_time, elapsed_time, disk_io, mem_usage);
    }

  hashval = hash_name (name);

  for (ap = user_list[hashval]; ap != NULL; ap = ap->next)
    {
      if (strncmp (ap->u.name, name, USERNAME_LEN) == 0)
	/* this is the one! */
	break;
    }
  
  /* If we didn't find it, AP will be null (and therefore we know when
   * to create a new record */

  if (ap == NULL)
    {
      ap = (struct user_entry *) xmalloc (sizeof (struct user_entry));
      ap->next = user_list[hashval];
      user_list[hashval] = ap;

      strncpy (ap->u.name, name, USERNAME_LEN);
      ap->u.s.user_time = 0.0;
      ap->u.s.sys_time = 0.0;
      ap->u.s.elapsed_time = 0.0;
      ap->u.s.mem_usage = 0.0;
      ap->u.s.disk_io = 0.0;
      ap->u.s.num_calls = 0;
    }

#define ADDIT(thing); ap->u.s.thing += thing;

  ADDIT (user_time);
  ADDIT (sys_time);
  ADDIT (elapsed_time);
  ADDIT (disk_io);
  ADDIT (mem_usage);
  ADDIT (num_calls);

#undef ADDIT
}


/* compare two entries, returning an int less than, equal to, or
 * greater than zero reflecting whether s1 is less than, equal to, or
 * greater than s2. */

int
compare_stats_entry (struct stats *s1, struct stats *s2)
{
  double v1, v2;

  switch (sort_type)
    {
    case sort_sys_plus_user:
      v1 = s1->user_time + s1->sys_time;
      v2 = s2->user_time + s2->sys_time;
      break;
    case sort_sys_plus_user_div_calls:
      v1 = (s1->user_time + s1->sys_time) / (double) s1->num_calls;
      v2 = (s2->user_time + s2->sys_time) / (double) s2->num_calls;
      break;
    case sort_avio:
      v1 = s1->disk_io / (double) s1->num_calls;
      v2 = s2->disk_io / (double) s2->num_calls;
      break;
    case sort_tio:
      v1 = s1->disk_io;
      v2 = s2->disk_io;
      break;
    case sort_cpu_mem_average:
      v1 = s1->mem_usage / (double) s1->num_calls;
      v2 = s2->mem_usage / (double) s2->num_calls;
      break;
    case sort_cpu_storage:
      v1 = ((s1->user_time + s1->sys_time)
	    ? (s1->user_time + s1->sys_time) * s1->mem_usage
	    : s1->mem_usage);
      v2 = ((s2->user_time + s2->sys_time)
	    ? (s2->user_time + s2->sys_time) * s2->mem_usage
	    : s2->mem_usage);
      break;
    case sort_num_calls:
      v1 = (double) s1->num_calls;
      v2 = (double) s2->num_calls;
      break;
    }
  if (v1 < v2) return ((reverse_sort) ? -1 : 1);
  if (v1 > v2) return ((reverse_sort) ? 1 : -1);

#define compareit(x); \
  if (s1->x < s2->x) return ((reverse_sort) ? -1 : 1); \
  if (s1->x > s2->x) return ((reverse_sort) ? 1 : -1);
  
  compareit (num_calls);
  compareit (user_time);
  compareit (sys_time);
  compareit (disk_io);
  compareit (elapsed_time);
  compareit (mem_usage);

#undef compareit

  return 0;
}


int
compare_user_entry (struct user_entry **s1, struct user_entry **s2)
{
  return compare_stats_entry (&((*s1)->u.s), &((*s2)->u.s));
}


int
compare_sum_entry (struct sum_entry **s1, struct sum_entry **s2)
{
  return compare_stats_entry (&((*s1)->s.s), &((*s2)->s.s));
}


void
print_stats_nicely (struct stats *s)
{

#define NC s->num_calls
#define DNC ((double) NC)
#define RE s->elapsed_time
#define U s->user_time
#define S s->sys_time
#define CP (U + S)
#define IO s->disk_io
#define K s->mem_usage

  if (debugging_enabled)
    {
      fprintf (stddebug,
	       "stat:%10ld %10.2fu %10.2fs %10.2fe %10.2fio %10.2fmem\n",
	       s->num_calls, s->user_time, s->sys_time,
	       s->elapsed_time, s->disk_io, s->mem_usage);
    }
  
  if (NC == 0)
    {
      fprintf (stderr,
	       "%s: ERROR -- print_stats_nicely called with num_calls == 0\n",
	       program_name);
      exit (1);
    }

  /* print it out */
  
  printf ("%8ld ", NC);
  
  if (percentages)
    printf ("%7.2f%% ", ((stats_totals.num_calls)
			 ? (DNC
			    / (double) (stats_totals.num_calls)) * 100.0
			 : 0.0));
  
  
  printf ("%10.2fre ", (((DNC && print_seconds) || !print_seconds)
			? RE / ((print_seconds) ? DNC : 60.0)
			: 0.0));
  
  if (percentages)
    printf ("%7.2f%% ", ((stats_totals.elapsed_time)
			 ? (RE / stats_totals.elapsed_time) * 100.0
			 : 0.0));
  
  if (separate_times)
    {
      printf ("%10.2fu ", (((DNC && print_seconds) || !print_seconds)
			   ? U / ((print_seconds) ? DNC : 60.0)
			   : 0.0));
	  
      if (percentages)
	printf ("%7.2f%% ", ((stats_totals.user_time)
			     ? (U / stats_totals.user_time) * 100.0
			     : 0.0));
	  
      printf ("%10.2fs", (((DNC && print_seconds) || !print_seconds)
			  ? S / ((print_seconds) ? DNC : 60.0)
			  : 0.0));
	  
      if (percentages)
	printf (" %7.2f%%", ((stats_totals.sys_time)
			     ? (S / stats_totals.sys_time) * 100.0
			     : 0.0));
    }
  else
    {
      printf ("%10.2fcp", (((DNC && print_seconds) || !print_seconds)
			   ? CP / ((print_seconds) ? DNC : 60.0)
			   : 0.0));
	  
      if (percentages)
	printf (" %7.2f%%",
		((stats_totals.sys_time + stats_totals.user_time)
		 ? ((CP /
		     (stats_totals.sys_time + stats_totals.user_time))
		    * 100.0)
		 : 0.0));
    }
  
  if (print_ratio)
    {
      if (CP == 0.0)
	printf ("*ignore*");
      else
	printf ("%8.1f", ((CP) ? RE / CP : 0.0));
	  
      fputs ("re/cp", stdout);
    }
  
  if (total_io)
    printf ("%10.0ftio ", IO);
  else
    printf ("%10.0favio ", ((DNC) ? IO / DNC : 0.0));
  
#if 0
  if (print_ksec)
    printf ("%9.0fk*sec", K);
  else
    printf ("%9ldk", (long) ((CP) ? (K / CP) : 0.0));
#endif

  /* Christoph says: */
  
  if (print_ksec)
    printf ("%9.0fk*sec", (CP) ? K * CP : K);
  else
    printf ("%9.0fk", K / DNC);
}


void
print_user_list (void)
{
  /* even though the BSD version of sa doesn't support sorting of the
     user summary list, why can't we? */

  struct user_entry *ap, **entry_array, user_totals;
  long num_users, temp, which;

  /* make the summary record */

  user_totals.next = NULL;
  user_totals.u.name[0] = '\0';
  user_totals.u.s = stats_totals;

  /* Count the number of users in the hash table.  Start at one
     because we're including the summary entry in the table. */
  
  for (temp = 0, num_users = 1; temp < CMD_TABLE_SIZE; temp++)
    {
      for (ap = user_list[temp]; ap != NULL; ap = ap->next)
	num_users++;
    }

  entry_array = (struct user_entry **)
    xmalloc (sizeof (struct user_entry *) * num_users);

  which = 0;
  entry_array[which++] = &user_totals;

  for (temp = 0; temp < CMD_TABLE_SIZE; temp++)
    {
      for (ap = user_list[temp]; ap != NULL; ap = ap->next)
	entry_array[which++] = ap;
    }
  
  /* The summary entry should always be first, so don't sort it.
     Remember to correct the number of elements to adjust... */

  qsort (entry_array + 1, (size_t) num_users - 1, sizeof (struct user_entry *),
	 (int (*)()) compare_user_entry);

  /* now we've got a sorted list of user entries -- PRINT IT! */

  for (temp = 0; temp < num_users; temp++)
    {
      ap = entry_array[temp];

      if (debugging_enabled)
	{
	  fprintf (stddebug,
	   "t:%-10.10s %10ld %10.2fu %10.2fs %10.2fe %10.2fio %10.2fmem\n",
		   ap->u.name,
		   ap->u.s.num_calls, ap->u.s.user_time, ap->u.s.sys_time,
		   ap->u.s.elapsed_time, ap->u.s.disk_io, ap->u.s.mem_usage);
	}

      printf ("%-*.*s ", USERNAME_LEN, USERNAME_LEN, ap->u.name);
      
      print_stats_nicely (&(ap->u.s));

      putchar ('\n');
    }

  free (entry_array);
}

int
non_printable (char *s, int len)
{
  int a;

  for (a = 0; (a < len) && (s[a] != '\0'); a++)
    {
      if (!isprint (s[a]))
	return 1;
    }
  
  return 0;
}


int
ask_if_junkable (char *s, int len)
{
  char line[1000];
  char word[1000];

  if (always_yes)
    return 1;

  printf ("Junk `%*s'? ", len, s);
  fflush (stdout);
  
  fgets (line, 1000, stdin);
  sscanf (line, " %s ", word);
  
  if ((word[0] == 'y') || (word[0] == 'Y'))
    return 1;

  return 0;
}

void
print_command_list (void)
{
  struct sum_entry *ap, **entry_array, sum_totals, other_totals, junk_totals;
  long num_commands, temp, which;
  int num_in_other_category, num_in_junk_category;

  /* create the summary entry */

  sum_totals.next = NULL;
  sum_totals.s.name[0] = '\0';
  sum_totals.s.fork_flag = 0;
  sum_totals.s.s = stats_totals;

  other_totals.next = NULL;
  strcpy (other_totals.s.name, "***other");
  other_totals.s.fork_flag = 0;
  other_totals.s.s.user_time = 0.0;
  other_totals.s.s.sys_time = 0.0;
  other_totals.s.s.elapsed_time = 0.0;
  other_totals.s.s.mem_usage = 0.0;
  other_totals.s.s.disk_io = 0.0;
  other_totals.s.s.num_calls = 0;

  junk_totals.next = NULL;
  strcpy (junk_totals.s.name, "**junk**");
  junk_totals.s.fork_flag = 0;
  junk_totals.s.s.user_time = 0.0;
  junk_totals.s.s.sys_time = 0.0;
  junk_totals.s.s.elapsed_time = 0.0;
  junk_totals.s.s.mem_usage = 0.0;
  junk_totals.s.s.disk_io = 0.0;
  junk_totals.s.s.num_calls = 0;

  /* Count the number of commands in the table. */

  for (temp = 0, num_commands = 0, num_in_other_category = 0,
       num_in_junk_category = 0;
       temp < CMD_TABLE_SIZE;
       temp++)
    {
      for (ap = command_list[temp]; ap != NULL; ap = ap->next)
	{
	  if (print_all_records)
	    num_commands++;
	  else
	    {
	      /* if we're junking stuff and the number of calls is
                 under the threshold, do it... */
	      if (junk_threshold
		  && (ap->s.s.num_calls <= junk_threshold)
		  && ask_if_junkable (ap->s.name, CMD_LEN))
		{
		  num_in_junk_category++;
		  ap->junked = 1;
		  
#define ADDIT(x) junk_totals.s.s.x += ap->s.s.x;

		  ADDIT (user_time);
		  ADDIT (sys_time);
		  ADDIT (elapsed_time);
		  ADDIT (disk_io);
		  ADDIT (mem_usage);
		  ADDIT (num_calls);
#undef ADDIT
		  junk_totals.s.fork_flag |= ap->s.fork_flag;
		}

	      /* if the number of calls is one or the characters in
                 the filename are non-printing, we have to add this to
                 the "***other" category */
	      
	      else if ((ap->s.s.num_calls == 1)
		  || non_printable (ap->s.name, CMD_LEN))
		{
		  num_in_other_category++;
		  ap->junked = 1;

#define ADDIT(x) other_totals.s.s.x += ap->s.s.x;

		  ADDIT (user_time);
		  ADDIT (sys_time);
		  ADDIT (elapsed_time);
		  ADDIT (disk_io);
		  ADDIT (mem_usage);
		  ADDIT (num_calls);
#undef ADDIT
		  other_totals.s.fork_flag |= ap->s.fork_flag;
		}
	      else
		num_commands++;
	    }
	}
    }

  num_commands++;		/* one for the summary entry */
  if (num_in_other_category)
    num_commands++;		/* one for the `***other' category */
  if (num_in_junk_category)
    num_commands++;		/* one for the `***junk' category */

  entry_array = (struct sum_entry **)
    xmalloc (sizeof (struct sum_entry *) * num_commands);

  which = 0;
  entry_array[which++] = &sum_totals;

  if (num_in_other_category)
    entry_array[which++] = &other_totals;

  if (num_in_junk_category)
    entry_array[which++] = &junk_totals;

  for (temp = 0; temp < CMD_TABLE_SIZE; temp++)
    {
      for (ap = command_list[temp]; ap != NULL; ap = ap->next)
	{
	  if (print_all_records)
	    entry_array[which++] = ap;
	  else
	    {
	      if (ap->junked)
		{
		  /* skip this one */
		}
	      else
		entry_array[which++] = ap;
	    }
	}
    }
  
  /* the summary entry should always be first, so don't sort it --
     therefore, pass location 1 of the array to qsort and one less
     than the number of commands */

  qsort (entry_array + 1, (size_t) num_commands - 1,
	 sizeof (struct sum_entry *), (int (*)()) compare_sum_entry);

  for (temp = 0; temp < num_commands; temp++)
    {
      ap = entry_array[temp];

      print_stats_nicely (&(ap->s.s));

      if (ap->s.name[0] != '\0')
	{
	  printf ("   %-.10s", ap->s.name);
	  
	  if (ap->s.fork_flag)
	    putchar ('*');
	}

      putchar ('\n');
    }

  free (entry_array);
}


/* clear all global flags & data */
void
init_flags_and_data (void)
{
  long a;

  debugging_enabled = 0;
  print_seconds = 0;
  separate_times = 0;
  dont_read_summary_files = 0;
  print_ksec = 0;
  print_ratio = 0;
  print_users = 0;
  total_io = 0;
  percentages = 0;
  sort_type = sort_sys_plus_user;
  reverse_sort = 0;
  dont_separate_forks = 1;
  user_summary_flag = 0;
  number_sort_options = 0;
  print_all_records = 0;
  junk_threshold = 0;

  /* get the page size of the machine for the PAGES_TO_KB macro */
  system_page_size = (double) getpagesize () / 1024.0;

  /* init the summary record */
  stats_totals.user_time = 0.0;
  stats_totals.sys_time = 0.0;
  stats_totals.elapsed_time = 0.0;
  stats_totals.mem_usage = 0.0;
  stats_totals.disk_io = 0.0;
  stats_totals.num_calls = 0;

  clear_uid_table ();

  /* clear the hash tables */
  for (a = 0; a < CMD_TABLE_SIZE; a++)
    {
      command_list[a] = NULL;
      user_list[a] = NULL;
    }
}

