/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1995, William Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell derivative works of
 * "tgif", and the right to distribute "tgif" for a fee) are
 * reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /u/multimedia/william/X11/TGIF2/RCS/oval.c,v 2.64 1995/07/10 05:48:15 william Exp $";
#endif

#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "attr.e"
#include "auxtext.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "file.e"
#include "grid.e"
#include "mainloop.e"
#include "msg.e"
#include "obj.e"
#ifndef _NO_EXTERN
#include "oval.e"
#endif
#include "pattern.e"
#include "poly.e"
#include "ps.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "xpixmap.e"

int	ovalDrawn = FALSE;

static
void GetPSEllipseStr (xc, yc, a, b, s)
   int	xc, yc, a, b;
   char	* s;
{
   if (preDumpSetup) PSUseEllipse();
#ifdef INVERT_CTM_BUG
   if (preDumpSetup) PSUseMinRadius();
   sprintf (s, "newpath %1d %1d %1d %s %1d %s tgifellipse",
         xc, yc, a, "tgif_min_radius", b, "tgif_min_radius");
#else
   sprintf (s, "newpath %1d %1d %1d %1d tgifellipse", xc, yc, a, b);
#endif
}

void MyOval (window, gc, bbox)
   Window	window;
   GC		gc;
   struct BBRec	bbox;
{
   register int		ltx, lty, w, h;

   if (bbox.ltx > bbox.rbx)
   {
      ltx = bbox.rbx; w = bbox.ltx - ltx;
   }
   else
   {
      ltx = bbox.ltx; w = bbox.rbx - ltx;
   }

   if (bbox.lty > bbox.rby)
   {
      lty = bbox.rby; h = bbox.lty - lty;
   }
   else
   {
      lty = bbox.lty; h = bbox.rby - lty;
   }

   XDrawArc (mainDisplay, window, gc, ltx, lty, w, h, 0, 360*64);
}

static
void DumpOvalPath (FP, ObjPtr, Xc, Yc, A, B, Width, Pen, Dash)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
   int			Xc, Yc, A, B, Width, Pen, Dash;
{
   register int	i;
   char		s[MAXSTRING];

   if (Width != 1) fprintf (FP, "   %1d setlinewidth\n", Width);
   if (Dash != 0)
   {
      fprintf (FP, "   [");
      for (i = 0; i < dashListLength[Dash]-1; i++)
         fprintf (FP, "%1d ", (int)(dashList[Dash][i]));
      fprintf (FP, "%1d] 0 setdash\n",
            (int)(dashList[Dash][dashListLength[Dash]-1]));
   }

   if (!colorDump && Pen > BACKPAT)
   {
      GrayCheck (Pen);
      if (useGray)
         fprintf (FP, "   %s setgray\n", GrayStr(Pen));
      else
      {
         if (preDumpSetup) PSUseBWPattern();
         fprintf (FP, "   pat%1d %s\n", Pen, patternStr);
      }
   }

   GetPSEllipseStr (Xc, Yc, A, B, s);
   fprintf (FP, "   %s ", s);

   switch (Pen)
   {
      case SOLIDPAT: fprintf (FP, "stroke\n"); break;
      case BACKPAT: fprintf (FP, "1 setgray stroke 0 setgray\n"); break;
      default:
         if (colorDump)
         {
            if (preDumpSetup) PSUseColorPattern();
            fprintf (FP, "flattenpath strokepath clip newpath\n");
            DumpPatFill (FP, Pen, 8, ObjPtr->bbox, "   ");
         }
         else
            fprintf (FP, "stroke\n");
         break;
   }
   if (Dash != 0) fprintf (FP, "   [] 0 setdash\n");
   if (Width != 1) fprintf (FP, "   1 setlinewidth\n");
}
 
void DumpOvalObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   int	ltx, lty, rbx, rby, xc, yc, a, b;
   int	fill, width, pen, dash, color_index;
   char	s[MAXSTRING];

   ltx = ObjPtr->obbox.ltx; lty = ObjPtr->obbox.lty;
   rbx = ObjPtr->obbox.rbx; rby = ObjPtr->obbox.rby;
   a = (rbx - ltx) / 2; xc = ltx + a;
   b = (rby - lty) / 2; yc = lty + b;

   fill = ObjPtr->detail.o->fill;
   width = ObjPtr->detail.o->width;
   pen = ObjPtr->detail.o->pen;
   dash = ObjPtr->detail.o->dash;

   if (fill == NONEPAT && pen == NONEPAT) return;

   fprintf (FP, "%% OVAL\n");
   color_index = ObjPtr->color;
   if (colorDump)
      DumpRGBColorLine(FP, color_index, 0, TRUE);
   else
      fprintf (FP, "0 setgray\n");

   GetPSEllipseStr (xc, yc, a, b, s);
   switch (fill)
   {
      case NONEPAT: break;
      case SOLIDPAT:
         /* solid black oval */
         fprintf (FP, "%s fill\n", s);
         break;
      case BACKPAT:
         /* solid white oval */
         fprintf (FP, "%s\n", s);
         fprintf (FP, "closepath 1 setgray fill\n");
         if (colorDump)
            DumpRGBColorLine(FP, color_index, 3, TRUE);
         else
            fprintf (FP, "0 setgray\n");
         break;
      default:
         /* patterned */
         fprintf (FP, "gsave\n");
         if (colorDump)
         {
            if (preDumpSetup) PSUseColorPattern();
            fprintf (FP, "   %s\n", s);
            fprintf (FP, "   closepath 1 setgray fill\n");
            DumpRGBColorLine(FP, color_index, 3, TRUE);
            fprintf (FP, "   %s\n", s);
            fprintf (FP, "   closepath eoclip newpath\n");
            DumpPatFill (FP, fill, 8, ObjPtr->bbox, "   ");
         }
         else
         {
            GrayCheck (fill);
            if (useGray)
               fprintf (FP, "   %s setgray\n", GrayStr(fill));
            else
            {
               if (preDumpSetup) PSUseBWPattern();
               fprintf (FP, "   pat%1d %s\n",fill, patternStr);
            }
            fprintf (FP, "   %s fill\n", s);
         }
         fprintf (FP, "grestore\n");
         break;
   }

   if (pen == NONEPAT) { fprintf (FP, "\n"); return; }

   fprintf (FP, "gsave\n");

   if (colorDump && pen > BACKPAT)
   {
      DumpOvalPath (FP, ObjPtr, xc, yc, a, b, width, BACKPAT, 0);
      if (colorDump)
         DumpRGBColorLine(FP, color_index, 3, TRUE);
   }
   DumpOvalPath (FP, ObjPtr, xc, yc, a, b, width, pen, dash);

   fprintf (FP, "grestore\n");

   fprintf (FP, "\n");
}
 
void DrawOvalObj (window, XOff, YOff, ObjPtr)
   Window		window;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   int			fill, width, pen, dash, pixel, real_x_off, real_y_off;
   struct BBRec		bbox;
   XGCValues		values;

   real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);
   bbox.ltx = ZOOMED_SIZE(ObjPtr->obbox.ltx - real_x_off);
   bbox.lty = ZOOMED_SIZE(ObjPtr->obbox.lty - real_y_off);
   bbox.rbx = ZOOMED_SIZE(ObjPtr->obbox.rbx - real_x_off);
   bbox.rby = ZOOMED_SIZE(ObjPtr->obbox.rby - real_y_off);

   fill = ObjPtr->detail.o->fill;
   width = ObjPtr->detail.o->width;
   pen = ObjPtr->detail.o->pen;
   dash = ObjPtr->detail.o->dash;
   pixel = colorPixels[ObjPtr->color];

   if (fill != 0)
   {
      values.foreground = (fill == 2) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[fill];
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);
      XFillArc (mainDisplay, window, drawGC, bbox.ltx, bbox.lty,
            bbox.rbx-bbox.ltx, bbox.rby-bbox.lty, 0, 360*64);
   }
   if (pen != 0)
   {
      values.foreground = (pen == 2) ? myBgPixel : pixel;
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[pen];
      values.line_width = ZOOMED_SIZE(width);
#ifdef NO_THIN_LINE
      if (values.line_width < 1) values.line_width = 1;
#else
#ifdef THIN_OVAL_AND_ARC
      if (values.line_width <= 1) values.line_width = 0;
#endif
#endif
      if (dash != 0)
      {
         XSetDashes (mainDisplay, drawGC, 0, dashList[dash],
               dashListLength[dash]);
         values.line_style = LineOnOffDash;
      }
      else
         values.line_style = LineSolid;
      XChangeGC (mainDisplay, drawGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple | GCLineWidth |
            GCLineStyle, &values);
      XDrawArc (mainDisplay, window, drawGC, bbox.ltx, bbox.lty,
            bbox.rbx-bbox.ltx, bbox.rby-bbox.lty, 0, 360*64);
   }
}

static
void CreateOvalObj (BBox)
   struct BBRec	BBox;
{
   struct OvalRec	* oval_ptr;
   struct ObjRec	* obj_ptr;
   int			width, w;

   oval_ptr = (struct OvalRec *) calloc (1, sizeof(struct OvalRec));
   oval_ptr->fill = objFill;
   oval_ptr->width = width = curWidthOfLine[lineWidth];
   oval_ptr->pen = penPat;
   oval_ptr->dash = curDash;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));

   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = ABS_X(BBox.ltx);
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = ABS_Y(BBox.lty);
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = ABS_X(BBox.rbx);
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = ABS_Y(BBox.rby);
   w = HALF_W(width);
   obj_ptr->bbox.ltx -= w;
   obj_ptr->bbox.lty -= w;
   obj_ptr->bbox.rbx += w;
   obj_ptr->bbox.rby += w;
   obj_ptr->type = OBJ_OVAL;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->rotation = 0;
   obj_ptr->locked = FALSE;
   obj_ptr->detail.o = oval_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   AddObj (NULL, topObj, obj_ptr);
}

static XComposeStatus	c_stat;
 
static
void ContinueOval (OrigX, OrigY)
   int	OrigX, OrigY;
{
   int 			end_x, end_y, grid_x, grid_y, done=FALSE, abort=FALSE;
   int			xor_pixel;
   char			buf[80], w_buf[80], h_buf[80], x_buf[80], y_buf[80];
   struct BBRec		bbox;
   XGCValues		values;
   XEvent		input, ev;
   XMotionEvent		* motion_ev;

   bbox.ltx = bbox.rbx = OrigX;
   bbox.lty = bbox.rby = OrigY;

   xor_pixel = xorColorPixels[colorIndex];

   values.foreground = xor_pixel;
   values.function = GXxor;
   values.fill_style = FillSolid;
#ifdef NO_THIN_LINE
   values.line_width = 1;
#else
   values.line_width = 0;
#endif
   values.line_style = LineSolid;
   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCLineStyle,
         &values);

   grid_x = end_x = OrigX;
   grid_y = end_y = OrigY; 
   PixelToMeasurementUnit(w_buf, 0);
   PixelToMeasurementUnit(h_buf, 0);
   PixelToMeasurementUnit(x_buf, ABS_X(grid_x));
   PixelToMeasurementUnit(y_buf, ABS_Y(grid_y));
   sprintf (buf, "%sx%s%s%s", w_buf, h_buf, x_buf, y_buf);
   StartShowMeasureCursor (grid_x, grid_y, buf, TRUE);
   BeginIntervalRulers (grid_x, grid_y, grid_x, grid_y);
   XGrabPointer (mainDisplay, drawWindow, FALSE,
         PointerMotionMask | ButtonReleaseMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);
   
   while (!done)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         MyOval (drawWindow, drawGC, bbox);
         EndIntervalRulers (grid_x, grid_y);
         PixelToMeasurementUnit(w_buf, ABS_SIZE(abs(bbox.rbx-OrigX)));
         PixelToMeasurementUnit(h_buf, ABS_SIZE(abs(bbox.rby-OrigY)));
         PixelToMeasurementUnit(x_buf, ABS_X(bbox.rbx));
         PixelToMeasurementUnit(y_buf, ABS_Y(bbox.rby));
         sprintf (buf, "%sx%s%s%s", w_buf, h_buf, x_buf, y_buf);
         EndShowMeasureCursor (bbox.rbx, bbox.rby, buf, TRUE);
         done = TRUE;
      }
      else if (input.type == MotionNotify)
      {
         motion_ev = &(input.xmotion);
         end_x = motion_ev->x;
         end_y = motion_ev->y;
         GridXY (end_x, end_y, &grid_x, &grid_y);
         if (motion_ev->state & (ShiftMask | ControlMask))
         {
            int	w, h, pos_w=TRUE, pos_h=TRUE;

            w = grid_x - bbox.ltx;
            h = grid_y - bbox.lty;
            if (w < 0)
            {
               w = (-w);
               pos_w = FALSE;
            }
            if (h < 0)
            {
               h = (-h);
               pos_h = FALSE;
            }
            if (w > h)
               grid_x = (pos_w ? (bbox.ltx+h) : (bbox.ltx-h));
            else
               grid_y = (pos_h ? (bbox.lty+w) : (bbox.lty-w));
         }
         if (grid_x != bbox.rbx || grid_y != bbox.rby)
         {
            PixelToMeasurementUnit(w_buf, ABS_SIZE(abs(bbox.rbx-OrigX)));
            PixelToMeasurementUnit(h_buf, ABS_SIZE(abs(bbox.rby-OrigY)));
            PixelToMeasurementUnit(x_buf, ABS_X(bbox.rbx));
            PixelToMeasurementUnit(y_buf, ABS_Y(bbox.rby));
            sprintf (buf, "%sx%s%s%s", w_buf, h_buf, x_buf, y_buf);
            ShowMeasureCursor (bbox.rbx, bbox.rby, buf, TRUE);
            MyOval (drawWindow, drawGC, bbox);
            bbox.rbx = grid_x;
            bbox.rby = grid_y;
            MyOval (drawWindow, drawGC, bbox);
            PixelToMeasurementUnit(w_buf, ABS_SIZE(abs(bbox.rbx-OrigX)));
            PixelToMeasurementUnit(h_buf, ABS_SIZE(abs(bbox.rby-OrigY)));
            PixelToMeasurementUnit(x_buf, ABS_X(bbox.rbx));
            PixelToMeasurementUnit(y_buf, ABS_Y(bbox.rby));
            sprintf (buf, "%sx%s%s%s", w_buf, h_buf, x_buf, y_buf);
            ShowMeasureCursor (bbox.rbx, bbox.rby, buf, TRUE);
         }
         DrawIntervalRulers (OrigX, OrigY, grid_x, grid_y);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
      else if (input.type == KeyPress)
      {
         KeySym	key_sym;
         char	s[80];

         XLookupString (&(input.xkey), s, 80-1, &key_sym, &c_stat);
         TranslateKeys (s, &key_sym);
         if (s[0] == '\033' && (key_sym & 0xff) == '\033')
         {
            XUngrabPointer (mainDisplay, CurrentTime);
            MyOval (drawWindow, drawGC, bbox);
            EndIntervalRulers (grid_x, grid_y);
            PixelToMeasurementUnit(w_buf, ABS_SIZE(abs(bbox.rbx-OrigX)));
            PixelToMeasurementUnit(h_buf, ABS_SIZE(abs(bbox.rby-OrigY)));
            PixelToMeasurementUnit(x_buf, ABS_X(bbox.rbx));
            PixelToMeasurementUnit(y_buf, ABS_Y(bbox.rby));
            sprintf (buf, "%sx%s%s%s", w_buf, h_buf, x_buf, y_buf);
            EndShowMeasureCursor (bbox.rbx, bbox.rby, buf, TRUE);
            abort = TRUE;
            done = TRUE;
         }
      }
   }
   if (!abort && OrigX != grid_x && OrigY != grid_y)
   {
      if (bbox.ltx > bbox.rbx)
      {
         end_x = bbox.ltx; bbox.ltx = bbox.rbx; bbox.rbx = end_x;
      }
      if (bbox.lty > bbox.rby)
      {
         end_y = bbox.lty; bbox.lty = bbox.rby; bbox.rby = end_y;
      }
      CreateOvalObj (bbox);
      RecordNewObjCmd ();
      DrawOvalObj (drawWindow, drawOrigX, drawOrigY, topObj);
      ovalDrawn = TRUE;
      SetFileModified (TRUE);
   }
}

void DrawOval (input)
   XEvent	* input;
{
   XButtonEvent	* button_ev;
   int		mouse_x, mouse_y, grid_x, grid_y;

   if (input->type != ButtonPress) return;

   button_ev = &(input->xbutton);
   if (button_ev->button == Button1)
   {
      mouse_x = button_ev->x;
      mouse_y = button_ev->y;
      GridXY (mouse_x, mouse_y, &grid_x, &grid_y);
      ContinueOval (grid_x, grid_y);
   }
}

void SaveOvalObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   if (fprintf (FP, "oval('%s',", colorMenuItems[ObjPtr->color]) == EOF)
      writeFileFailed = TRUE;
   if (fprintf (FP, "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,",
         ObjPtr->obbox.ltx, ObjPtr->obbox.lty, ObjPtr->obbox.rbx,
         ObjPtr->obbox.rby, ObjPtr->detail.o->fill, ObjPtr->detail.o->width,
         ObjPtr->detail.o->pen, ObjPtr->id, ObjPtr->detail.o->dash,
         ObjPtr->rotation, ObjPtr->locked) == EOF)
      writeFileFailed = TRUE;
   SaveAttrs (FP, ObjPtr->lattr);
   if (fprintf (FP, ")") == EOF) writeFileFailed = TRUE;
}

#define GETVALUE(val,name) ScanValue("%d", &(val), name, "oval")

void ReadOvalObj (Inbuf, ObjPtr)
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   struct OvalRec	* oval_ptr;
   char			color_str[40], * s;
   int			ltx, lty, rbx, rby, fill, width, pen, dash, w, id=0;
   int			new_alloc, rotation, locked=FALSE;

   *ObjPtr = NULL;

   s = FindChar ((int)'(', Inbuf);
   s = ParseStr (s, (int)',', color_str, sizeof(color_str));

   InitScan (s, "\n\t, ");

   dash = 0;
   rotation = 0;
   if (fileVersion <= 5)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID)
      {
         return;
      }
      switch (width)
      {
         case 1: width = 3; break;
         case 2: width = 6; break;
      }
      id = objId++;
   }
   else if (fileVersion <= 7)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID)
      {
         return;
      }
      id = objId++;
   }
   else if (fileVersion <= 8)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 13)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 25)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (width,    "width") == INVALID ||
          GETVALUE (pen,      "pen") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (dash,     "dash") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID ||
          GETVALUE (locked,   "locked") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }

   if (ltx > rbx || lty > rby)
   {
      int	tmp_ltx, tmp_lty, tmp_rbx, tmp_rby;

      if (!PRTGIF) Msg ("Bad oval bounding box.  Adjusted.");
      CalcBBox (ltx, lty, rbx, rby, &tmp_ltx, &tmp_lty, &tmp_rbx, &tmp_rby);
      ltx = tmp_ltx; lty = tmp_lty; rbx = tmp_rbx; rby = tmp_rby;
   }

   if (fileVersion <= 16 && width <= 6) width = origWidthOfLine[width];
   fill = UpgradePenFill (fill);
   pen = UpgradePenFill (pen);

   * ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   oval_ptr = (struct OvalRec *) calloc (1, sizeof(struct OvalRec));

   oval_ptr->fill = fill;
   oval_ptr->width = width;
   oval_ptr->pen = pen;
   oval_ptr->dash = dash;
   (*ObjPtr)->x = ltx;
   (*ObjPtr)->y = lty;
   (*ObjPtr)->color = QuickFindColorIndex (color_str, &new_alloc, TRUE);
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->id = id;
   (*ObjPtr)->rotation = rotation;
   (*ObjPtr)->locked = locked;
   (*ObjPtr)->type = OBJ_OVAL;
   (*ObjPtr)->obbox.ltx = ltx;
   (*ObjPtr)->obbox.lty = lty;
   (*ObjPtr)->obbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = rby;
   w = HALF_W(width);
   (*ObjPtr)->bbox.ltx = ltx - w;
   (*ObjPtr)->bbox.lty = lty - w;
   (*ObjPtr)->bbox.rbx = rbx + w;
   (*ObjPtr)->bbox.rby = rby + w;
   (*ObjPtr)->detail.o = oval_ptr;
}

void FreeOvalObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   cfree (ObjPtr->detail.o);
   cfree (ObjPtr);
}
