# Copyright (c) 1993 by Sanjay Ghemawat
##############################################################################
# NoteList
#
#	Maintains list of notices for a certain date.
#
# Description
# ===========
# A NoteList displays notices for a particular date.

class NoteList {name view} {
    set slot(window) $name
    set slot(view) $view
    set slot(date) [date today]
    set slot(items) ""
    set slot(focus) ""
    set slot(width) 100
    set slot(iwidth) 100
    set slot(sbar)  0

    frame $name -bd 0
    scrollbar $name.s -relief raised -bd 1 -orient vertical\
	-command [list $name.c yview]
    canvas $name.c -bd 1 -relief raised -yscroll [list $self sbar_set]

    # Get font for this window
    set slot(font) [option get $name.c itemFont Font]
    if ![string compare $slot(font) ""] {
	set slot(font) [pref itemFont]
    }

    # Get font dimensions.  The text string "00:00AM" is used so that
    # we hit the cache entry created by "ApptList".
    set slot(font_height) [text_height $slot(font) "00:00AM" [pref itemPad]]

    $self configure
    $self background

    pack $name.c -side left -expand 1 -fill both

    # Establish bindings
    $name.c bind bg <1> [list $self new]
    bind $name.c <Configure> [list $self canvas_resize %w %h]

    # Handle triggers
    trigger on add	[list $self change]
    trigger on delete	[list $self remove]
    trigger on change	[list $self change]
    trigger on text	[list $self textchange]
    trigger on flush	[list $self rescan]
    trigger on reconfig	[list $self reconfig]
}

method NoteList set_date {date} {
    set slot(date) $date
    $self rescan
    $slot(window).c yview 0
}

# effects - Cleanup on destruction
method NoteList destructor {} {
    # We have to be very careful here about making sure callbacks do
    # not occur in the wrong place (i.e. on already deleted objects).

    # Remove triggers as soon as possible
    trigger remove add		[list $self change]
    trigger remove delete	[list $self remove]
    trigger remove change	[list $self change]
    trigger remove text		[list $self textchange]
    trigger remove flush	[list $self rescan]
    trigger remove reconfig	[list $self reconfig]

    # Trim item list to avoid unselect callback problems
    set list $slot(items)
    set slot(items) {}

    foreach item $list {
	class_kill $slot(window.$item)
    }

    destroy $slot(window)
}

method NoteList select {item} {
    if [info exists slot(window.$item)] {
	set slot(focus) $item
	$slot(window.$item) select
    }
}

method NoteList unselect {item} {
    if ![string compare $slot(focus) $item] {
	set slot(focus) {}
	$slot(window.$item) unselect
    }
}

method NoteList itemwindow {item} {
    if [info exists slot(window.$item)] {
	return $slot(window.$item)
    }
    return {}
}

##############################################################################
# Internal Procedures

method NoteList reconfig {} {
    set name $slot(window)
    set slot(width)  [winfo pixels $name "[cal option ItemWidth]c"]
    set slot(iwidth) [expr $slot(width) - 2*[pref itemPad]]

    # Set canvas geometry
    $name.c configure\
	-width $slot(width)\
	-height "[cal option NoticeHeight]c"\
	-confine 1\
	-scrollregion [list 0 0 $slot(width) "[cal option NoticeHeight]c"]
    $name.c yview 0

    $self layout
}

# effects - Compute various dimensions for NoteList
method NoteList configure {} {
    set name $slot(window)

    $self reconfig

    # Allow vertical scrolling with middle mouse button
    #$name.c bind all <2> [list $name.c scan mark 0 %y]
    #$name.c bind all <B2-Motion> [list $name.c scan dragto 0 %y]
}

method NoteList background {} {
    $slot(window).c create rectangle 0 0 1 1\
	-fill ""\
	-outline ""\
	-width 0\
	-tags bg
}

method NoteList new {} {
    if {$slot(focus) != ""} {
	# Just unselect
	ical_action $slot(view) unselect
	return
    }

    if [cal readonly] {
	error_notify [winfo toplevel $slot(window)] "Permission denied"
	return
    }

    set id [notice]
    $id date $slot(date)
    $id earlywarning [cal option DefaultEarlyWarning]
    $id own

    cal add $id
    ical_with_view $slot(view) {run-hook item-create $id}

    if [info exists slot(window.$id)] {
	ical_action $slot(view) select $id
    }
}

method NoteList change {item} {
    set list $slot(items)
    if {[$item is note] && [$item contains $slot(date)]} {
        if {[lsearch $list $item] < 0} {
            # Add item
            lappend list $item
            set slot(items) $list
            $self make_window $item
        } else {
            $slot(window.$item) read
        }
        $self layout
        return
    }

    if [lremove list $item] {
        set slot(items) $list
        class_kill $slot(window.$item)
        unset slot(window.$item)
        $self layout
    }
}

method NoteList textchange {item} {
    if {[$item is note] && [$item contains $slot(date)]} {
	$slot(window.$item) read
	$self layout
    }
}

method NoteList remove {item} {
    set list $slot(items)
    if [lremove list $item] {
	set slot(items) $list
	class_kill $slot(window.$item)
	unset slot(window.$item)
	$self layout
    }
}

# args are ignored - they just allow trigger to call us directly.
method NoteList rescan {args} {
    set list $slot(items)
    set slot(items) ""

    foreach appt $list {
        class_kill $slot(window.$appt)
	unset slot(window.$appt)
    }

    set list {}
    cal query $slot(date) $slot(date) item d {
        if [$item is note] {
            lappend list $item
            $self make_window $item
        }
    }
    set slot(items) $list
    $self layout
}

method NoteList layout {} {
    set x [pref itemPad]
    set y 0
    foreach item $slot(items) {
	$slot(window.$item) geometry $x $y $slot(iwidth) 1
	set y [lindex [$slot(window.$item) bbox] 3]
    }
    incr y $slot(font_height)

    # Set canvas geometry
    $slot(window).c configure -scrollregion [list 0 0 $slot(width) $y]
}

# Adjust scrollbar.  Unpack it completely if the whole canvas is visible,
method NoteList sbar_set {total size first last} {
    if {$total > $size} {
	# Need scrollbar
	if !$slot(sbar) {
	    pack $slot(window).s -side right -fill y
	    set slot(sbar) 1
	}
    } else {
	if $slot(sbar) {
	    pack forget $slot(window).s
	    set slot(sbar) 0
	}
    }

    $slot(window).s set $total $size $first $last
}

method NoteList make_window {item} {
    set slot(window.$item) [ItemWindow\
				$slot(view)\
				$slot(window).c\
				$slot(font)\
				$item $slot(date)]
}

method NoteList canvas_resize {w h} {
    $slot(window).c coord bg 0 0 $w [expr $slot(font_height)*100]
}
