# Copyright (c) 1994 by Sanjay Ghemawat
#############################################################################
#
# This file contains code that allows common editing procedures to be
# written for text/entry/canvas-items.  Each one of these editable
# widget provides a set of basic editing operations.  User-level
# editing operations are built out of these operations.
#
#	index   	<w> <index>
#	goto    	<w> <index>
#	insert  	<w> <index> <text>
#	delete  	<w> <index> <index>
#	compare 	<w> <index> <op> <index>
#	extract 	<w> <index> <index>
#
# Each widget specific operation should be implemented as a procedure
# and the mapping from the abstract operation name to the procedure name
# should be stored in a global array.  See the sample implementations of
# these operations below for more detail.  Sample implementations are
# provided for Text/Entry/Canvas.
#

# Global editor state
#
#	kill		Killed text
#	prefix		Current prefix argument
#	killing		User is in the middle of a sequence of kill ops
#	curkilling	Value of "killing" when this keybinding was invoked.

# This is the global editing procedure.  Key strokes should be
# bound to commands that invoke this procedure with appropriate
# parameters.
#
#	class		Identifies the low-level editing operations.
#			Useful values are "Text_Edit", "Entry_Edit",
#			"Canvas_Edit".
#	w		The target window for the editing operation.
#	action		The editor action to invoke.
#	args		Args to supply to the editor action.  Usually
#			no args are needed.  The "self-insert" action
#			needs the key to insert as its arg.
#
# Example bindings --
#
#	bind Text <Any-KeyPress> {edit-action Text_Edit %W self-insert {%A}}
#	bind Text <Control-k>	 {edit-action Text_Edit %W kill-line}

proc edit_action {class w action args} {
    # Completely ignore bogus self-insert actions
    if {![string compare $action self-insert] &&
	![string compare [lindex $args 0] {}]} {return}

    # Update editor state for new action
    global edit
    set edit(curprefix)  $edit(prefix)
    set edit(curkilling) $edit(killing)
    set edit(prefix) 1
    set edit(killing) 0

    upvar #0 $class c
    global edit_code
    eval $edit_code($action)
}

# Define code for a particular editing action
proc edit_proc {action code} {
    global edit_code
    set edit_code($action) $code
}

edit_proc self-insert {
    $c(insert) $w [edit_str_multiply [lindex $args 0] $edit(curprefix)]
}

edit_proc beginning-of-file {
    $c(goto) $w 1.0
}

edit_proc end-of-file {
    $c(goto) $w end
}

edit_proc backward-char {
    $c(goto) $w "insert -[set edit(curprefix)]chars"
}

edit_proc forward-char {
    $c(goto) $w "insert +[set edit(curprefix)]chars"
}

edit_proc backward-word {
    $c(goto) $w [edit_back_word $class $w insert $edit(curprefix)]
}

edit_proc forward-word {
    $c(goto) $w [edit_forw_word $class $w insert $edit(curprefix)]
}

edit_proc delete-word {
    edit_kill_text $class $w\
	insert\
	[edit_forw_word $class $w insert $edit(curprefix)]
}

edit_proc backward-delete-word {
    edit_kill_text $class $w\
	[edit_back_word $class $w insert $edit(curprefix)]\
	insert
}

edit_proc beginning-of-line {
    $c(goto) $w "insert linestart"
}

edit_proc end-of-line {
    $c(goto) $w "insert lineend"
}

edit_proc previous-line {
    $c(goto) $w "insert -[set edit(curprefix)]lines"
}

edit_proc next-line {
    $c(goto) $w "insert +[set edit(curprefix)]lines"
}

edit_proc delete-char {
    if {$edit(curprefix) == 1} {
	$c(delete) $w insert insert+1chars
    } else {
	edit_kill_text $class $w insert insert+[set edit(curprefix)]chars
    }
}

edit_proc delete-char-or-sel {
    if [catch {edit_kill_text $class $w sel.first sel.last}] {
	# No selection
	if {$edit(curprefix) == 1} {
	    $c(delete) $w insert insert+1chars
	} else {
	    edit_kill_text $class $w insert insert+[set edit(curprefix)]chars
	}
    }
}

edit_proc delete-backward {
    if {$edit(curprefix) == 1} {
	$c(delete) $w "insert -1chars" insert
    } else {
	edit_kill_text $class $w "insert -[set edit(curprefix)]chars" insert
    }
}

edit_proc delete-selection {
    catch {edit_kill_text $class $w sel.first sel.last}
}

edit_proc copy-selection {
    catch {edit_copy_text $class $w sel.first sel.last}
}

edit_proc delete-line {
    if {$edit(curprefix) == 1} {
	if [$c(compare) $w insert == "insert lineend"] {
	    set end "insert +1chars"
	} else {
	    set end "insert lineend"
	}
    } else {
	set n [expr $edit(curprefix)-1]
	set end "insert +[set n]lines lineend +1chars"
    }
    edit_kill_text $class $w insert $end
}

edit_proc new-line {
    $c(insert) $w [edit_str_multiply "\n" $edit(curprefix)]
}

edit_proc open-line {
    $c(insert) $w [edit_str_multiply "\n" $edit(curprefix)]
    $c(goto) $w "insert -[set edit(curprefix)]chars"
}

edit_proc yank {
    if [catch {set str [selection get]}] {
	set str $edit(kill)
    }
    $c(insert) $w [edit_str_multiply $str $edit(curprefix)]
}

edit_proc edit-prefix {
    set edit(prefix) [expr $edit(curprefix)*4]
    
    # Do not let <Control-u> interfere with extended kills
    set edit(killing) $edit(curkilling)
}

proc edit_forw_word {class w index num} {
    upvar #0 $class c
    set wc {[a-zA-Z0-9]}
    set pc {[^a-zA-Z0-9]}
    set re "$pc*$wc+"

    set t [$c(extract) $w $index end]
    set s 0
    while {$num > 0} {
	if ![regexp -indices -- $re $t m] {return end}
	set f [lindex $m 1]
	incr f
	set t [string range $t $f end]
	incr s $f
	incr num -1
    }

    return "$index +[set s]chars"
}

proc edit_back_word {class w index num} {
    upvar #0 $class c
    set wc {[a-zA-Z0-9]}
    set pc {[^a-zA-Z0-9]}
    set re "^$pc*($wc+$pc+)*($wc+)$pc*\$";

    set t [$c(extract) $w 1.0 $index]
    while {$num > 0} {
	if ![regexp -indices -- $re $t j1 j2 m] {return 1.0}
	set t [string range $t 0 [expr [lindex $m 0]-1]]
	incr num -1
    }
    return "1.0 + [string length $t]chars"
}

# Kill text and save in buffer
proc edit_kill_text {class w i1 i2} {
    upvar #0 $class c
    edit_copy_text $class $w $i1 $i2
    $c(delete) $w $i1 $i2
}

# Copy text to kill buffer
proc edit_copy_text {class w i1 i2} {
    upvar #0 $class c
    global edit

    # If we are already killing text, make sure we combine with
    # previously killed text.
    if !$edit(curkilling) {set edit(kill) ""}
    set edit(kill) "$edit(kill)[$c(extract) $w $i1 $i2]"

    # Remember that we are killing text
    set edit(killing) 1
}

# Duplicate "string" "n" times
proc edit_str_multiply {s n} {
    set r ""
    while {$n > 0} {
	set r "$r$s"
	incr n -1
    }
    return $r
}

# Return default key map
proc edit_keymap {} {
    global keymap
    return $keymap(default)
}


# Return named key map
proc edit_keymap {name} {
    global keymap
    return $keymap($name)
}

# Clear all key bindings from specified window
proc Edit_Clear {w} {
    foreach b [bind $w] {
	if [regexp Key- $b] {bind $w $b ""}
    }
}

# Assign key bindings from specified keymap to specified window.
# "class" identifies the editing operations to use for "w".
# Some useful values for "class" are "Text_Edit", "Entry_Edit",
# and "Canvas_Edit".
proc Edit_Install {w c keymap} {
    foreach b $keymap {
	set k [lindex $b 0]
	set a [lindex $b 1]
	bind $w $k [concat [list edit_action $c %W] $a]
    }
}

#### The actual low-level editing procedures for text/entry/canvas ####

# Evaluate complex index specification for canvas items.
proc Canvas_iparse {c t index} {
    set ws   "\[ \t\]"
    set sign "\[\+\-\]"
    set num  {[0-9]+}
    set mod  "chars|lines"

    # Canonicalize index
    regsub -all "($sign)$ws*($num)$ws*($mod)" $index { \1\2\3 } index
    regsub -all "$ws+" $index " " index
    set index [string trim $index]
    set list [split $index " \t\n"]
    set p [lindex $list 0]
    set list [lrange $list 1 end]

    # Handle <line>.<char>
    if [regexp "^($num)\.($num)" $p junk line col] {
	set p 0
	set list [linsert $list 0 +[expr $line-1]lines +[set col]chars]
    }

    # Handle "sel.last" because of difference in semantics
    if ![string compare $p "sel.last"] {
	set list [linsert $list 0 +1chars]
    }

    set p [$c index $t $p]
    set e [$c index $t end]
    set v [lindex [$c itemconfigure $t -text] 4]

    foreach m $list {
	if {$p < 0} {set p 0}
	if {$p > $e} {set p $e}

	if [regexp "^($sign$num)chars$" $m junk count] {
	    incr p $count
	    continue
	}
	if [regexp "($sign)($num)lines$" $m junk s count] {
	    if {$count > 0} {
		# Get column number
		set col 0
		while {($p > 0) && ([string index $v [expr $p-1]] != "\n")} {
		    incr col
		    incr p -1
		}

		# "$p" is now at column 0.

		if ![string compare $s +] {
		    # Move forward over "count" newlines
		    while {($count > 0) && ($p < $e)} {
			if {[string index $v $p] == "\n"} {incr count -1}
			incr p
		    }
		} else {
		    # Skip past the newline for this line
		    incr p -1

		    # Move backward to "count" more newlines
		    while {($count > 0) && ($p > 0)} {
			if {[string index $v [expr $p-1]] == "\n"} {
			    incr count -1
			}
			incr p -1
		    }
		    incr p
		}

		# Advance to right column
		while {($col>0) && ($p<$e) && ([string index $v $p] != "\n")} {
		    incr p
		    incr col -1
		}
	    }
	    continue
	}
	switch -exact -- $m {
	    linestart {
		while {$p > 0} {
		    if {[string index $v [expr $p-1]] == "\n"} break
		    incr p -1
		}
	    }
	    lineend {
		while {$p < $e} {
		    if {[string index $v $p] == "\n"} break
		    incr p
		}
	    }
	    default {
		error "unknown canvas index modifier: $m"
	    }
	}
    }

    if {$p < 0} {set p 0}
    if {$p > $e} {set p $e}

    return $p
}

# Evaluate complex index specification for entries
proc Entry_iparse {w index} {
    set ws   "\[ \t\]"
    set sign "\[\+\-\]"
    set num  {[0-9]+}
    set mod  "chars|lines"

    # Canonicalize index
    regsub -all "($sign)$ws*($num)$ws*($mod)" $index { \1\2\3 } index
    regsub -all "$ws+" $index " " index
    set index [string trim $index]
    set list [split $index " \t\n"]
    set p [lindex $list 0]
    set list [lrange $list 1 end]

    # Handle <line>.<char>
    if [regexp "^($num)\.($num)" $p junk line col] {
	set p 0
	set list [linsert $list 0 +[set col]chars]
    }

    # Handle "sel.last" because of difference in semantics
    if ![string compare $p "sel.last"] {
	set list [linsert $list 0 +1chars]
    }

    # Turn "@x,y" into "@x".
    regsub "^@($num),($num)" $p {@\1} p

    set p [$w index $p]
    set e [$w index end]
    set v [$w get]

    foreach m $list {
	if {$p < 0} {set p 0}
	if {$p > $e} {set p $e}

	if [regexp "^($sign$num)chars$" $m junk count] {
	    incr p $count
	    continue
	}
	if [regexp "($sign)($num)lines$" $m junk s count] {
	    if {$count > 0} {
		if ![string compare $s +] {
		    # Move to end of string
		    set p $e
		} else {
		    # Move to beginning
		    set p 0
		}
	    }
	    continue
	}
	switch -exact -- $m {
	    linestart {
		set p 0
	    }
	    lineend {
		set p $e
	    }
	    default {
		error "unknown canvas index modifier: $m"
	    }
	}
    }

    if {$p < 0} {set p 0}
    if {$p > $e} {set p $e}

    return $p
}

proc Text_index {w i} {
    return [$w index $i]
}

proc Text_goto {w i} {
    $w mark set insert $i
    $w yview -pickplace insert
}

proc Text_delete {w i1 i2} {
    $w delete $i1 $i2
}

proc Text_insert {w t} {
    $w insert insert $t
    $w yview -pickplace insert
}

proc Text_compare {w i1 op i2} {
    return [$w compare $i1 $op $i2]
}

proc Text_extract {w i1 i2} {
    return [$w get $i1 $i2]
}

proc Entry_index {w i} {
    return "1.[$w index [Entry_iparse $w $i]]"
}

proc Entry_goto {w i} {
    $w icursor [Entry_iparse $w $i]
    tk_entrySeeCaret $w
}

proc Entry_delete {w i1 i2} {
    $w delete [Entry_iparse $w $i1] [expr [Entry_iparse $w $i2]-1]
}

proc Entry_insert {w t} {
    $w insert insert $t
}

proc Entry_compare {w i1 op i2} {
    return [expr [Entry_iparse $w $i1] $op [Entry_iparse $w $i2]]
}

proc Entry_extract {w i1 i2} {
    set i1 [$w index [Entry_iparse $w $i1]]
    set i2 [expr [$w index [Entry_iparse $w $i2]]-1]
    return [string range [$w get] $i1 $i2]
}

proc Canvas_index {w i} {
    set item [$w focus]
    return "1.[$w index $item [Canvas_iparse $w $item $i]]"
}

proc Canvas_goto {w i} {
    set item [$w focus]
    $w icursor $item [Canvas_iparse $w $item $i]
}

proc Canvas_delete {w i1 i2} {
    set item [$w focus]
    $w dchars $item\
	[Canvas_iparse $w $item $i1]\
	[expr [Canvas_iparse $w $item $i2]-1]
}

proc Canvas_insert {w t} {
    $w insert [$w focus] insert $t
}

proc Canvas_compare {w i1 op i2} {
    set item [$w focus]
    return [expr [Canvas_iparse $w $item $i1] $op [Canvas_iparse $w $item $i2]]
}

proc Canvas_extract {w i1 i2} {
    set item [$w focus]
    set i1 [$w index $item [Canvas_iparse $w $item $i1]]
    set i2 [expr [$w index $item [Canvas_iparse $w $item $i2]]-1]
    return [string range [lindex [$w itemconfigure $item -text] 4] $i1 $i2]
}

#### Initialize default bindings ####
proc edit_init {} {
    global keymap edit Text_Edit Entry_Edit Canvas_Edit

    set edit(kill) ""
    set edit(prefix) 1
    set edit(killing) 0
    set edit(curkilling) 0

    # Default key map that probably nobody should be editing.
    set keymap(default) {
	{ <Any-KeyPress>	{self-insert %A}		}
	{ <Return>		new-line			}
	{ <Delete>		delete-backward			}
	{ <BackSpace>		delete-backward			}
	{ <Up>			previous-line			}
	{ <Down>		next-line			}
	{ <Left>		backward-char			}
	{ <Right>		forward-char			}
    }

    # Emacs style keymap
    set keymap(emacs) {
	{ <Control-b>		backward-char			}
	{ <Control-f>		forward-char			}
	{ <Meta-b>		backward-word			}
	{ <Meta-f>		forward-word			}
	{ <Control-a>		beginning-of-line		}
	{ <Control-e>		end-of-line			}
	{ <Control-p>		previous-line			}
	{ <Control-n>		next-line			}
	{ <Meta-less>		beginning-of-file		}
	{ <Meta-greater>	end-of-file			}
	
	{ <Control-d>		delete-char-or-sel		}
	{ <Control-h>		delete-backward			}
	{ <Meta-d>		delete-word			}
	{ <Meta-Delete>		backward-delete-word		}
	{ <Control-k>		delete-line			}
	{ <Control-o>		open-line			}
	
	{ <Control-y>		yank				}
	{ <Control-w>		delete-selection		}
	{ <Meta-w>		copy-selection			}
	
	{ <Control-u>		edit-prefix			}
    }

    set Text_Edit(index)	Text_index
    set Text_Edit(goto)		Text_goto
    set Text_Edit(insert)	Text_insert
    set Text_Edit(delete)	Text_delete
    set Text_Edit(compare)	Text_compare
    set Text_Edit(extract)	Text_extract

    set Entry_Edit(index)	Entry_index
    set Entry_Edit(goto)	Entry_goto
    set Entry_Edit(insert)	Entry_insert
    set Entry_Edit(delete)	Entry_delete
    set Entry_Edit(compare)	Entry_compare
    set Entry_Edit(extract)	Entry_extract

    set Canvas_Edit(index)	Canvas_index
    set Canvas_Edit(goto)	Canvas_goto
    set Canvas_Edit(insert)	Canvas_insert
    set Canvas_Edit(delete)	Canvas_delete
    set Canvas_Edit(compare)	Canvas_compare
    set Canvas_Edit(extract)	Canvas_extract

    foreach t {Text Entry} {
	Edit_Clear $t
	Edit_Install $t [set t]_Edit $keymap(default)
	Edit_Install $t [set t]_Edit $keymap(emacs)
    }
}
