#include <sys/types.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/uio.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h> /* getenv, exit */
#include <stdarg.h>
#include <string.h> /* strcmp */
#include <ctype.h>
#include <memory.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h> /* setsid */
#include <fcntl.h>
#include <syslog.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rpcsvc/yp_prot.h>
#include <rpcsvc/ypclnt.h>
#include <rpc/pmap_clnt.h> /* for pmap_unset */
#include "ourhdr.h"
#include "ypbind.h"

extern int putenv(const char *);

extern SVCXPRT *udptransp;
extern domainname mydomain;
extern struct binding *ypbindlist;
extern int lockfd;
extern int broken_server;
extern int debug;
extern char **Argv;
extern int Argc;

domainname askdomain;

pid_t
start_slave(void)
{
  struct sigaction sact;
  pid_t pid;
  
  pid = fork();
  if (pid < 0)
    log_sys("fork failed");
  if (0 != pid) /* parent */
    return pid;
  
  inststr(Argv, Argc, "ypbind (slave)");

  sigemptyset(&sact.sa_mask);
  sact.sa_handler = toggle_debug;
  sact.sa_flags = SA_RESTART;
  if (0 != sigaction(SIGUSR1, &sact, NULL))
    log_ret("Could not install signal handler for SIGUSR1");
  sact.sa_handler = handle_hangup;
  if (0 != sigaction(SIGHUP, &sact, NULL))
    log_ret("Could not install signal handler for SIGHUP");
  
  init_binding();
  
  for(;;)
    {
      check_binding();
      sleep(PING_INTERVAL);
    }
}

void
init_binding(void)
{
  struct binding ypdb;

  write_lock_binding();
  bzero(ypbindlist, sizeof(struct binding) * _MAXDOMAIN);
#if !MMAP_SHARED_OK
  lseek(lockfd, 0, SEEK_SET);
  writen(lockfd, ypbindlist, sizeof(struct binding) * _MAXDOMAIN);
#endif
  un_lock_binding();

  strncpy(ypdb.domain, mydomain, YPMAXDOMAIN);
  ypdb.is_bound = TRUE; /* always bind to our NIS domain */
  ypdb.lockfd = -1;
  ypdb.client_handle = NULL;
  ypdb.is_alive = FALSE;
  ypdb.use_broadcast = TRUE; /* default is using broadcast */
  update_entry(&ypdb);
  parse_config_file(_PATH_YPCONF);
  return;
}

void
check_binding(void)
{
  struct binding ypdb;
  time_t t;
  int i;

  time(&t);
  for (i = 0; i < _MAXDOMAIN; i++)
    {
      read_lock_binding();
      memcpy(&ypdb, &ypbindlist[i], sizeof ypdb);
      un_lock_binding();
      if (ypdb.is_bound)
        {
          if (ypdb.is_alive && ypdb.next_check > t)
            {
              if (debug)
                  log_msg("pinging server %s, port %d",
                          inet_ntoa(ypdb.server_addr),
                          ntohs(ypdb.server_port));
              ping_server(&ypdb);
            }
          
          if (!ypdb.is_alive || ypdb.next_check <= t)
            {
              if (ypdb.use_broadcast)
                {
                  if (debug)
                      log_msg("broadcasting for domain %s", ypdb.domain);
                  broadcast(&ypdb);
                }
              else
                {
                  if (debug)
                    log_msg("rebinding to server %s", ypdb.host->h_name);
                  bindto_server(ypdb.domain, ypdb.host);
                }
              continue;
            }
        }
    }
}

void
add_server(char *dom, struct sockaddr_in *raddrp, CLIENT *clnt_handlep, struct hostent *host, bool_t use_broadcast)
{
  struct binding entry;
#if USE_BINDINGDIR
  struct iovec iov[2];
  struct ypbind_resp ybr;
  pid_t lockpid;
  char path[MAXPATHLEN];
  int fd, len, status;
#endif
  
  if (NULL == dom || (0 != get_entry(dom, &entry)))
    return;
  
  if (!entry.is_bound)
    {
      strncpy(entry.domain, dom, YPMAXDOMAIN);
      entry.lockfd = -1;
      entry.client_handle = NULL;
      entry.is_bound = TRUE;
    }
  if (NULL != raddrp)
    {
      if (!broken_server &&
	  (ntohs(raddrp->sin_port) >= IPPORT_RESERVED ||
	   ntohs(raddrp->sin_port) <  IPPORT_RESERVED/2))
	{
	  log_msg("Answer from %s on illegal port", inet_ntoa(raddrp->sin_addr));
	  return;
	}
      memcpy(&entry.server_addr, &raddrp->sin_addr, sizeof entry.server_addr);
      entry.server_port = raddrp->sin_port;
      entry.is_alive = TRUE;
    }
  if (NULL != entry.client_handle)
    clnt_destroy(entry.client_handle);
  entry.client_handle = clnt_handlep;
  if (NULL != host)
    entry.host = host;
  entry.next_check = time(NULL) + REBIND_INTERVAL;
  entry.version = YPVERS;
  entry.use_broadcast = use_broadcast;
  
#if USE_BINDINGDIR
  if (NULL != raddrp)
    {
      if (-1 != entry.lockfd)
        close(entry.lockfd);
      sprintf(path, "%s/%s.%ld", BINDINGDIR,
              entry.domain, entry.version);
      if ((fd = open(path, O_CREAT | O_RDWR | O_TRUNC, FILE_MODE )) == -1)
        {
          if (-1 == mkdir(BINDINGDIR, DIR_MODE))
            log_ret("mkdir");
          if ((fd = open(path, O_CREAT | O_RDWR | O_TRUNC, FILE_MODE)) == -1)
            return;
        }
      
      lockpid = lock_test(fd, F_RDLCK, 0, SEEK_SET, 0);
      if (0 != lockpid) 
        log_quit("%s already locked by pid %d", path, lockpid);
      
      status = read_lock(fd, 0, SEEK_SET, 0);
      if (0 != status) 
        log_sys("set lock");
      
      
          /*
           * ok, if BINDINGDIR exists, and we can create the binding file, then
           * write to it..
           */
      entry.lockfd = fd;
      
      iov[0].iov_base = (caddr_t) &(udptransp->xp_port);
      iov[0].iov_len = sizeof udptransp->xp_port;
      iov[1].iov_base = (caddr_t) &ybr;
      iov[1].iov_len = sizeof ybr;
      
      bzero(&ybr, sizeof ybr);
      ybr.ypbind_status = YPBIND_SUCC_VAL;
      ybr.ypbind_respbody.ypbind_bindinfo.ypbind_binding_addr = raddrp->sin_addr;
      ybr.ypbind_respbody.ypbind_bindinfo.ypbind_binding_port = raddrp->sin_port;
      
      len = iov[0].iov_len + iov[1].iov_len ;
      if (writev(entry.lockfd, iov, 2) != len )
        {
          log_ret("writev");
          close(entry.lockfd);
          (void) unlink(path);
          entry.lockfd = -1;
        }
    }
#endif
  update_entry(&entry);
  return;
}

void
bindto_server(char *ypdomain, struct hostent *host)
{
  struct sockaddr_in server_addr;
  int sock;
  bool_t out;
  enum clnt_stat status;
  struct timeval timeout;
  CLIENT *clnt_handlep = NULL;
  int i = 0;
  
  if (debug)
    log_msg("bindto_server: domain %s, host %s", ypdomain, host->h_name);
  
  bzero((char *)&server_addr, sizeof server_addr);
  server_addr.sin_family = host->h_addrtype;
  server_addr.sin_port = htons(0);
  sock = RPC_ANYSOCK;
  while(NULL != host->h_addr_list[i])
    {
      memcpy(&server_addr.sin_addr, host->h_addr_list[i],
             host->h_length);
      
      timeout.tv_sec = 1;
      timeout.tv_usec = 0;
      clnt_handlep = clntudp_create(&server_addr, YPPROG, YPVERS,
                                    timeout, &sock);
      if (NULL != clnt_handlep)
          break;
      i++;
    }

  if (NULL == clnt_handlep)
    {
      log_msg("clnt_create for server %s failed", host->h_name);
      return; 
    }
  
  timeout.tv_sec = 5;
  timeout.tv_usec = 0;
  status = clnt_call(clnt_handlep, YPPROC_DOMAIN,
                     (xdrproc_t) xdr_domainname, ypdomain,
                     (xdrproc_t) xdr_bool, &out,
                     timeout);
  if (RPC_SUCCESS != status)
    {
      log_msg(clnt_sperror(clnt_handlep, host->h_name));
      clnt_destroy(clnt_handlep);
    }
  else if (TRUE != out)
    {
      log_msg("domain %s not served by %s", ypdomain, host->h_name);
      clnt_destroy(clnt_handlep);
    }
  else
    add_server(ypdomain, &server_addr, clnt_handlep, host, FALSE);

  return;
}

void
ping_server(struct binding *ypdb)
{
  int status;
  struct timeval timeout;
  char *nothing;
  
  if (NULL != ypdb->client_handle)
    {
      bzero((char *)&nothing, sizeof nothing);
      timeout.tv_sec = 2;
      timeout.tv_usec = 0;
      status = clnt_call(ypdb->client_handle, YPPROC_NULL,
                         (xdrproc_t) xdr_void, nothing,
                         (xdrproc_t) xdr_void, &nothing,
                         timeout);
      
      if (RPC_SUCCESS != status)
        {
          log_msg(clnt_sperror(ypdb->client_handle, ypdb->host->h_name));
	  clnt_destroy(ypdb->client_handle);
          ypdb->client_handle = NULL;
          ypdb->is_alive = FALSE;
          ypdb->next_check = time(NULL);
          update_entry(ypdb);
        }
    }
  return;
}

bool_t
eachresult(bool_t *out, struct sockaddr_in *addr)
{
  CLIENT *clnt_handlep;
  struct timeval timeout;
  int sock;
  
  if (*out)
    {
      if(debug)
        {
          struct hostent *hostentp;
          hostentp = gethostbyaddr((char *) &addr->sin_addr.s_addr,
                                   sizeof(addr->sin_addr.s_addr), AF_INET);
          log_msg("Answer from server %s .", hostentp->h_name);
        }
      
      sock = RPC_ANYSOCK;
      timeout.tv_sec = 1;
      timeout.tv_usec = 0;
      clnt_handlep = clntudp_create(addr, YPPROG, YPVERS, timeout, &sock);
      add_server(askdomain, addr, clnt_handlep, NULL, TRUE);
      return 1;
    }
  else
    {
      return 0;
    }
}

void
broadcast(struct binding *ypdb)
{
  bool_t out;
  enum clnt_stat  status;
#if USE_BINDINGDIR
  char path[MAXPATHLEN];
#endif
  
      /* update global variable for eachresult */
  askdomain = ypdb->domain;
  status = clnt_broadcast(YPPROG, YPVERS, YPPROC_DOMAIN_NONACK,
                          (xdrproc_t) xdr_domainname, askdomain,
                          (xdrproc_t) xdr_bool, (void *)&out,
                          (resultproc_t) eachresult);
  if (RPC_SUCCESS != status)
    {
#if USE_BINDINGDIR
      if ( -1 != ypdb->lockfd)
        {
          close(ypdb->lockfd);
          ypdb->lockfd = -1;
          sprintf(path, "%s/%s.%ld", BINDINGDIR,
                  ypdb->domain, ypdb->version);
          unlink(path);
        }
#endif
      log_msg("broadcast: %s.", clnt_sperrno(status));
    }
}


/*
 * Routines for parsing the config file ( /etc/yp.conf )
 *
 */

void
parse_config_file(const char *path)
      /* parse the config file, check bindings */
{
  FILE *fp;
  char buf[1024];
  char *cp, *tmp;
  char tmpserver[81], tmpdomain[YPMAXDOMAIN + 1];
  int count;
  
  fp = fopen(path, "r");
  if (NULL == fp)
    return;

  if (debug)
    log_msg("parsing config file");

  while ((cp = fgets(buf, sizeof(buf), fp)) != NULL)
    {
      tmp = strchr(cp, '#');  /* Kommentare ausblenden */
      if (tmp)
        *tmp = '\0';
      while (isspace(*cp))    /* Leerraum ueberlesen */
        cp++;
      if (*cp == '\0')        /* Leerzeile ignorieren */
        continue;
      
      if (debug)
        log_msg("Trying entry: %s", cp);
      count = sscanf(cp, "domain %64s server %80s", tmpdomain, tmpserver);
      if (2 == count)
        {
          if (debug)
            log_msg("parsed domain %s server %s", tmpdomain, tmpserver);
          check_config_entry(tmpdomain, tmpserver);
          continue;
        }
      count = sscanf(cp, "domain %s broadcast", tmpdomain);
      if (1 == count)
        {
          if (debug)
            log_msg("parsed domain %s broadcast", tmpdomain);
          add_server(mydomain, NULL, NULL, NULL, TRUE);
          continue;
        }
      count = sscanf(cp, "ypserver %80s", tmpserver);
      if (1 == count)
        {
          if (debug)
            log_msg("parsed ypserver %s", tmpserver);
          check_config_entry(mydomain, tmpserver);
          continue;
        }
    }
  fclose(fp);
  return;
}

void
check_config_entry(char *ypdomain, const char *ypserver)
{
  struct hostent *host;
  static char *order = NULL;
  int result;

      /*
       * FIXME: gethostbyname may in turn ask ypbind (entry in 
       * /etc/host.conf)!!! 
       * so much for shooting yourself in the foot.
       * Using RESOLV_SERV_ORDER is a kludge, should use gethostent()
       */

  if (!order)
    order = strdup("RESOLV_SERV_ORDER=hosts");
  result = putenv(order);
  if (0 != result)
    log_ret("putenv failed");
  if (debug)
    log_msg("getenv: %s", getenv("RESOLV_SERV_ORDER"));
  host = gethostbyname(ypserver);
  if (NULL == host)
    {
      switch (h_errno)
        {
        case HOST_NOT_FOUND:
          log_msg("Unknown host: %s", ypserver);
          break;
        case TRY_AGAIN:
          log_msg("Host name lookup failure");
          break;
        case NO_DATA:
          log_msg("No address associated with name: %s", ypserver);
          break;
        case NO_RECOVERY:
          log_msg("Unknown server error");
          break;
        default:
          log_ret("gethostbyname: Unknown error");
          break;
        }
    }
  else
    bindto_server(ypdomain, host);

  return;
}

int
open_lockfile(void)
{
  char name[L_tmpnam];
  int fd;
  
  if (NULL == tmpnam(name))
    log_sys("tmpnam failed");
  
  fd = open(name, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
  if (fd < 0)
    log_sys("cannot open lockfile");
  unlink(name);
  return fd;
}

void 
read_lock_binding(void)
{
  int status;

  status = lock_reg(lockfd, F_SETLKW, F_RDLCK, 0, SEEK_SET, 0);
  if (0 != status) 
    log_sys("set read lock");
#if !MMAP_SHARED_OK
  lseek(lockfd, 0, SEEK_SET);
  readn(lockfd, ypbindlist, sizeof(struct binding) * _MAXDOMAIN);
#endif
  return;
}

void
write_lock_binding(void)
{
  int status;

  status = lock_reg(lockfd, F_SETLKW, F_WRLCK, 0, SEEK_SET, 0); 
  if (0 != status) 
    log_sys("set write lock");
  return;
}

void 
un_lock_binding(void)
{
  int status;
  
  status = lock_reg(lockfd, F_SETLKW, F_UNLCK, 0, SEEK_SET, 0);
  if (0 != status)
    log_sys("unlock");
  return;
}

int
get_entry(char *dom, struct binding *entry)
{
  struct binding *ypdb = NULL;
  
  ypdb = find_entry(dom);
  if (NULL == ypdb)
    return -1;
  read_lock_binding();
  memcpy(entry, ypdb, sizeof (struct binding));
  un_lock_binding();
  return 0;
}

void
update_entry(struct binding *entry)
{
  /* update entry in ypbindlist. Lock while updating */
  struct binding *ypdb = NULL;

  ypdb = find_entry(entry->domain);
  if (NULL == ypdb)
    return;
  write_lock_binding();
  memcpy(ypdb, entry, sizeof (struct binding));
#if !MMAP_SHARED_OK  
  lseek(lockfd, 0, SEEK_SET);
  writen(lockfd, ypbindlist, sizeof(struct binding) * _MAXDOMAIN);
#endif
  un_lock_binding();
  if (debug)
    log_msg("%s entry for domain %s: server %s, port %d",
            entry->is_alive ? "updated" : "cleared", entry->domain,
            inet_ntoa(entry->server_addr), ntohs(entry->server_port));
  return;
}

struct binding *
find_entry(char *dom)
{
  int i;
  struct binding *ypdb = NULL;
  
  read_lock_binding();
      /* Find entry for domain dom */
  for (i = 0; i < _MAXDOMAIN; i++)
    {
      if (ypbindlist[i].is_bound && 0 == strcmp(ypbindlist[i].domain, dom))
        {
          ypdb = &ypbindlist[i];
          break;
        }
    }
  
  if (NULL == ypdb) /* no entry for domain dom */
    {
          /* find empty slot */
      for (i = 0; i < _MAXDOMAIN; i++)
        {
          if (FALSE == ypbindlist[i].is_bound)
            {
              ypdb = &ypbindlist[i];
              break;
            }
        }
    }
  un_lock_binding();
  return ypdb;
}

void
handle_hangup(int sig)
{
  log_msg("rereading config file");

  init_binding();

  return;
}
