/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#define HELP_C

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <ctype.h>
#include <string.h>

static struct binding_t *help_lookupFunction (int op, int menu)
{
  int i;
  struct binding_t *map;

  switch (menu)
  {
    case MENU_MAIN:
      map = OpMain;
      break;

    case MENU_PAGER:
      map = OpPager;
      break;

    default:
      for (i=0; OpGeneric[i].name; i++)
	if (OpGeneric[i].op == op)
	  return (&OpGeneric[i]);    

      switch (menu)
      {
	case MENU_ATTACH:
	  map = OpAttach;
	  break;
	case MENU_POST:
	  map = OpPost;
	  break;
	case MENU_FOLDER:
	  map = OpBrowser;
	  break;
	case MENU_COMPOSE:
	  map = OpCompose;
	  break;
#ifdef _PGPPATH
        case MENU_PGP:
	  map = OpPgp;
	  break;
#endif
        case MENU_ALIAS:
	  map = OpAlias;
	  break;
	
        default:
	  return NULL;
      }

  }
  for (i=0; map[i].name; i++)
    if (map[i].op == op)
      return (&map[i]);
  return (NULL);
}

void mutt_make_help (char *d, size_t dlen, char *txt, int menu, int op)
{
  char buf[SHORT_STRING];

  if (km_expand_key (buf, sizeof(buf), km_find_func (menu, op)))
    snprintf (d, dlen, "%s:%s", buf, txt);
  else
    d[0] = 0;
}

static void print_macro (FILE *f, const char *macro)
{
  for (; *macro; macro++)
  {
    switch (*macro)
    {
      case '\n':
	fputs ("\\n", f);
	break;
      case '\r':
	fputs ("\\r", f);
	break;
      case '\t':
	fputs ("\\t", f);
	break;
      default:
	fputc (*macro, f);
	break;
    }
  }
}

void mutt_help (int menu)
{
  char t[_POSIX_PATH_MAX];
  char buf[SHORT_STRING];
  char *desc;
  struct keymap_t *map;
  struct binding_t *b;
  FILE *f;

  mutt_mktemp (t);
  if ((f = fopen (t, "w")) == NULL)
  {
    mutt_perror (t);
    return;
  }

  switch (menu)
  {
    case MENU_PAGER:
      desc = "pager";
      break;

    case MENU_MAIN:
      desc = "index";
      break;

    case MENU_ATTACH:
      desc = "attach";
      break;

    case MENU_COMPOSE:
      desc = "compose";
      break;

    case MENU_FOLDER:
      desc = "browser";
      break;

    case MENU_ALIAS:
      desc = "alias";
      break;

    case MENU_URL:
      desc = "url";
      break;

    case MENU_POST:
      desc = "postponed";
      break;

    case MENU_PGP:
      desc = "pgp";
      break;

    default:
      desc = "<UNKNOWN MENU>";
  }

  map = Keymaps[menu];

  /* browse through the keymap table */
  while (map)
  {
    km_expand_key (buf, sizeof(buf), map);

    if (map->op == OP_MACRO)
    {
      fprintf (f, "%s\t%-20s\t", buf, "macro");
      print_macro (f, map->macro);
      fputc ('\n', f);
    }
    else if (map->op != OP_NULL)
    {
      b = help_lookupFunction (map->op, menu);
      fprintf (f, "%s\t%-20s\t%s\n", buf,
	      b ? b->name : "UNKNOWN",
	      b ? HelpStrings[b->op] : "ERROR: please report this bug");
    }

    map = map->next;
  }

  fclose (f);

  snprintf (buf, sizeof (buf), "Help for %s", desc);
  mutt_do_pager (buf, t, 0);
}
