/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mx.h"
#include "send.h"
#include "rfc2047.h"

#ifdef _PGPPATH
#include "pgp.h"
#endif

#include <dirent.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <utime.h>

/* HP-UX and ConvexOS don't have this macro */
#ifndef S_ISLNK
#define S_ISLNK(x) (((x) & S_IFMT) == S_IFLNK ? 1 : 0)
#endif

#define mutt_is_spool(s)  (strcmp (Spoolfile, s) == 0)

#define MAXLOCKATTEMPT 5

#ifdef USE_DOTLOCK
static int dotlock_file (const char *path)
{
  const char *pathptr = path;
  char lockfile[_POSIX_PATH_MAX];
  char nfslockfile[_POSIX_PATH_MAX];
  char realpath[_POSIX_PATH_MAX];
  struct stat sb;
  int count = 0;
  int fd;

  /* if the file is a symlink, find the real file to which it refers */
  FOREVER
  {
    dprint(2,(debugfile,"dotlock_file(): locking %s\n", pathptr));

    if (lstat (pathptr, &sb) != 0)
    {
      mutt_perror (pathptr);
      return (-1);
    }

    if (S_ISLNK (sb.st_mode))
    {
      char linkfile[_POSIX_PATH_MAX];
      char linkpath[_POSIX_PATH_MAX];

      if ((count = readlink (pathptr, linkfile, sizeof (linkfile))) == -1)
      {
	mutt_perror (path);
	return (-1);
      }
      linkfile[count] = 0; /* readlink() does not NUL terminate the string! */
      mutt_expand_link (linkpath, pathptr, linkfile);
      strfcpy (realpath, linkpath, sizeof (realpath));
      pathptr = realpath;
    }
    else
      break;
  }

  snprintf (nfslockfile, sizeof (nfslockfile), "%s.%s.%d", pathptr, Hostname, (int) getpid ());
  snprintf (lockfile, sizeof (lockfile), "%s.lock", pathptr);
  unlink (nfslockfile);

  while ((fd = open (nfslockfile, O_WRONLY | O_EXCL | O_CREAT, 0)) < 0)
    if (errno != EAGAIN)
    {
      mutt_perror ("cannot open NFS lock file!");
      return (-1);
    }

  close (fd);

  count = 0;
  FOREVER
  {
    link (nfslockfile, lockfile);
    if (stat (nfslockfile, &sb) != 0)
    {
      mutt_perror ("stat");
      return (-1);
    }

    if (sb.st_nlink == 2)
      break;

    if (++count == MAXLOCKATTEMPT)
    {
      if (mutt_yesorno ("Lock count exceeded, remove lock?", 1) == 1)
      {
	unlink (lockfile);
	count = 0;
	continue;
      }
      else
	return (-1);
    }

    mutt_message ("Waiting for lock attempt #%d...", count);
    sleep (1);
  }

  unlink (nfslockfile);

  return 0;
}

static int undotlock_file (const char *path)
{
  const char *pathptr = path;
  char lockfile[_POSIX_PATH_MAX];
  char realpath[_POSIX_PATH_MAX];
  struct stat sb;
  int n;

  FOREVER
  {
    dprint (2,(debugfile,"undotlock: unlocking %s\n",path));

    if (lstat (pathptr, &sb) != 0)
    {
      mutt_perror (pathptr);
      return (-1);
    }

    if (S_ISLNK (sb.st_mode))
    {
      char linkfile[_POSIX_PATH_MAX];
      char linkpath[_POSIX_PATH_MAX];

      if ((n = readlink (pathptr, linkfile, sizeof (linkfile))) == -1)
      {
	mutt_perror (pathptr);
	return (-1);
      }
      linkfile[n] = 0; /* readlink() does not NUL terminate the string! */
      mutt_expand_link (linkpath, pathptr, linkfile);
      strfcpy (realpath, linkpath, sizeof (realpath));
      pathptr = realpath;
      continue;
    }
    else
      break;
  }

  snprintf (lockfile, sizeof (lockfile), "%s.lock", pathptr);
  unlink (lockfile);
  return 0;
}
#endif /* USE_DOTLOCK */

/* Args:
 *	excl		if excl != 0, request an exclusive lock
 *	dot		if dot != 0, try to dotlock the file
 */
int mx_lock_file (const char *path, int fd, int excl, int dot)
{
#if defined (USE_FCNTL) || defined (USE_FLOCK)
  int count;
#endif
  int r = 0;

#ifdef USE_FCNTL
  struct flock lck;

  memset (&lck, 0, sizeof (struct flock));
  lck.l_type = excl ? F_WRLCK : F_RDLCK;
  lck.l_whence = SEEK_SET;

  count = 0;
  while (fcntl (fd, F_SETLK, &lck) == -1)
  {
    dprint(1,(debugfile, "mx_lock_file(): fcntl errno %d.\n", errno));
    if (errno != EAGAIN && errno != EACCES)
    {
      mutt_perror ("fcntl");
      return (-1);
    }

    if (++count == MAXLOCKATTEMPT)
    {
      mutt_error ("Timeout exceeded while attempting fcntl lock!");
      return (-1);
    }

    mutt_message ("Waiting for fcntl lock... %d", count);
    sleep (1);
  }
#endif /* USE_FCNTL */

#ifdef USE_FLOCK
  count = 0;
  while (flock (fd, (excl ? LOCK_EX : LOCK_SH) | LOCK_NB) == -1)
  {
    if (errno != EWOULDBLOCK)
    {
      mutt_perror ("flock");
      r = -1;
      break;
    }

    if (++count == MAXLOCKATTEMPT)
    {
      mutt_error ("Timeout exceeded while attempting flock lock!");
      r = -1;
      break;
    }

    mutt_message ("Waiting for flock attempt... %d", count);
    sleep (1);
  }
#endif /* USE_FLOCK */

#ifdef USE_DOTLOCK
  if (r == 0 && dot)
    r = dotlock_file (path);
#endif /* USE_DOTLOCK */

  if (r == -1)
  {
    /* release any other locks obtained in this routine */

#ifdef USE_FCNTL
    lck.l_type = F_UNLCK;
    fcntl (fd, F_SETLK, &lck);
#endif /* USE_FCNTL */

#ifdef USE_FLOCK
    flock (fd, LOCK_UN);
#endif /* USE_FLOCK */

    return (-1);
  }

  return 0;
}

int mx_unlock_file (const char *path, int fd)
{
#ifdef USE_FCNTL
  struct flock unlockit = { F_UNLCK, 0, 0, 0 };

  memset (&unlockit, 0, sizeof (struct flock));
  unlockit.l_type = F_UNLCK;
  unlockit.l_whence = SEEK_SET;
  fcntl (fd, F_SETLK, &unlockit);
#endif

#ifdef USE_FLOCK
  flock (fd, LOCK_UN);
#endif

#ifdef USE_DOTLOCK
  undotlock_file (path);
#endif
  
  return 0;
}

/* open a file and lock it */
FILE *mx_open_file_lock (const char *path, const char *mode)
{
  FILE *f;

  if ((f = fopen (path, mode)) != NULL)
  {
    if (mx_lock_file (path, fileno (f), *mode != 'r', 1) != 0)
    {
      fclose (f);
      f = NULL;
    }
  }

  return (f);
}

/* try to figure out what type of mailbox ``path'' is
 *
 * return values:
 *	M_*	mailbox type
 *	0	not a mailbox
 *	-1	error
 */
int mx_get_magic (const char *path)
{
  struct stat st;
  int magic = 0;
  char tmp[_POSIX_PATH_MAX];
  FILE *f;

#ifdef USE_IMAP
  if (*path == '{')
    return M_IMAP;
#endif /* USE_IMAP */

  if (stat (path, &st) == -1)
  {
    dprint (1, (debugfile, "mx_get_magic(): unable to stat %s: %s (errno %d).\n",
		path, strerror (errno), errno));
    mutt_perror (path);
    return (-1);
  }

  if (S_ISDIR (st.st_mode))
  {
    /* check for maildir-style mailbox */

    snprintf (tmp, sizeof (tmp), "%s/cur", path);
    if (stat (tmp, &st) == 0 && S_ISDIR (st.st_mode))
      return (M_MAILDIR);

    /* check for mh-style mailbox */

    snprintf (tmp, sizeof (tmp), "%s/.mh_sequences", path);
    if (access (tmp, F_OK) == 0)
      return (M_MH);

    snprintf (tmp, sizeof (tmp), "%s/.xmhcache", path);
    if (access (tmp, F_OK) == 0)
      return (M_MH);
  }
  else if (st.st_size == 0)
  {
    /* hard to tell what zero-length files are, so assume the default magic */
    return (DefaultMagic);
  }
  else if ((f = fopen (path, "r")) != NULL)
  {
    struct utimbuf times;

    fgets (tmp, sizeof (tmp), f);
    if (strncmp ("From ", tmp, 5) == 0)
      magic = M_MBOX;
    else if (strcmp (MMDF_SEP, tmp) == 0)
      magic = M_MMDF;
    fclose (f);
    /* need to restore the times here, the file was not really accessed,
     * only the type was accessed.  This is important, because detection
     * of "new mail" depends on those times set correctly.
     */
    times.actime = st.st_atime;
    times.modtime = st.st_mtime;
    utime (path, &times);
  }
  else
  {
    dprint (1, (debugfile, "mx_get_magic(): unable to open file %s for reading.\n",
		path));
    mutt_perror (path);
    return (-1);
  }

  return (magic);
}

/*
 * set DefaultMagic to the given value
 */
int mx_set_magic (const char *s)
{
  if (strcasecmp (s, "mbox") == 0)
    DefaultMagic = M_MBOX;
  else if (strcasecmp (s, "mmdf") == 0)
    DefaultMagic = M_MMDF;
  else if (strcasecmp (s, "mh") == 0)
    DefaultMagic = M_MH;
  else if (strcasecmp (s, "maildir") == 0)
    DefaultMagic = M_MAILDIR;
  else
    return (-1);

  return 0;
}

/*
 * open a mailbox and parse it
 *
 * Args:
 *	flags	M_NOSORT	do not sort mailbox
 *		M_APPEND	open mailbox for appending
 *		M_READONLY	open mailbox in read-only mode
 *		M_QUIET		only print error messages
 *	ctx	if non-null, context struct to use
 */
CONTEXT *mx_open_mailbox (const char *path, int flags, CONTEXT *pctx)
{
  CONTEXT *ctx = pctx;
  int rc;

  if (!ctx)
    ctx = safe_malloc (sizeof (CONTEXT));
  memset (ctx, 0, sizeof (CONTEXT));
  ctx->path = safe_strdup (path);

  ctx->msgnotreadyet = -1;
  
  if (flags & M_QUIET)
    ctx->quiet = 1;
  if (flags & M_READONLY)
    ctx->readonly = 1;

  if (flags & M_APPEND)
  {
    /* open mailbox in append-mode (write) */

    ctx->append = 1;
    if (access (path, W_OK) == 0)
    {
      switch (ctx->magic = mx_get_magic (path))
      {
	case 0:
	  mutt_error ("%s is not a mailbox.", path);
	  /* fall through */
	case -1:
	  mx_fastclose_mailbox (ctx);
	  if (!pctx)
	    free (ctx);
	  return (NULL);
      }
    }
    else if (errno == ENOENT)
    {
      ctx->magic = DefaultMagic;

      if (ctx->magic == M_MH || ctx->magic == M_MAILDIR)
      {
	char tmp[_POSIX_PATH_MAX];

	if (mkdir (path, S_IRWXU))
	{
	  mutt_perror (tmp);
	  goto bail;
	}

	if (ctx->magic == M_MAILDIR)
	{
	  snprintf (tmp, sizeof (tmp), "%s/cur", path);
	  if (mkdir (tmp, S_IRWXU))
	  {
	    mutt_perror (tmp);
	    rmdir (path);
	    goto bail;
	  }

	  snprintf (tmp, sizeof (tmp), "%s/new", path);
	  if (mkdir (tmp, S_IRWXU))
	  {
	    mutt_perror (tmp);
	    snprintf (tmp, sizeof (tmp), "%s/cur", path);
	    rmdir (tmp);
	    rmdir (path);
	    goto bail;
	  }
	  snprintf (tmp, sizeof (tmp), "%s/tmp", path);
	  if (mkdir (tmp, S_IRWXU))
	  {
	    mutt_perror (tmp);
	    snprintf (tmp, sizeof (tmp), "%s/cur", path);
	    rmdir (tmp);
	    snprintf (tmp, sizeof (tmp), "%s/new", path);
	    rmdir (tmp);
	    rmdir (path);
	    goto bail;
	  }
	}
	else
	{
	  int i;

	  snprintf (tmp, sizeof (tmp), "%s/.mh_sequences", path);
	  if ((i = creat (tmp, S_IRWXU)) == -1)
	  {
	    mutt_perror (tmp);
	    rmdir (path);
	    goto bail;
	  }
	  close (i);
	}
      }
    }
    else
    {
      mutt_perror (path);
bail:
      mx_fastclose_mailbox (ctx);
      if (!pctx)
	safe_free ((void **) &ctx);
      return (NULL);
    }

    switch (ctx->magic)
    {
      case M_MBOX:
      case M_MMDF:

	if ((ctx->fp = fopen (ctx->path, "a")) == NULL ||
	    mbox_lock_mailbox (ctx, 1) != 0)
	{
	  if (ctx->fp == NULL)
	    mutt_perror (ctx->path);
	  mx_fastclose_mailbox (ctx);
	  if (!pctx)
	    safe_free ((void **) &ctx);
	  return (NULL);
	}
	fseek (ctx->fp, 0, 2);
	break;

      case M_MH:
      case M_MAILDIR:

	/* nothing to do */
	break;

      default:

	mx_fastclose_mailbox (ctx);
	if (!pctx)
	  safe_free ((void **) &ctx);
	break;
    }
  }
  else
  {
    /* read-mode */

    switch (ctx->magic = mx_get_magic (path))
    {
      case 0:

	mutt_error ("%s is not a mailbox.", path);
	/* fall through */

      case -1:

	mx_fastclose_mailbox (ctx);
	if (!pctx)
	  free (ctx);
	return (NULL);
    }

    if (!ctx->quiet)
      mutt_message ("Reading %s...", ctx->path);

    switch (ctx->magic)
    {
      case M_MH:
	rc = mh_read_dir (ctx, NULL);
	break;

      case M_MAILDIR:
	rc = maildir_read_dir (ctx);
	break;

      case M_MMDF:
      case M_MBOX:
	rc = mbox_open_mailbox (ctx);
	break;

#ifdef USE_IMAP
      case M_IMAP:
	rc = imap_open_mailbox (ctx);
	break;
#endif /* USE_IMAP */

      default:
	rc = -1;
	break;
    }
    
    if (rc == 0)
    {
      if ((flags & M_NOSORT) == 0)
	mutt_sort_headers (ctx);
      if (!ctx->quiet)
	mutt_clear_error ();
    }
    else
    {
      mx_fastclose_mailbox (ctx);
      if (!pctx)
	safe_free ((void **) &ctx);
    }
  }

  return (ctx);
}

/* free up memory associated with the mailbox context */
void mx_fastclose_mailbox (CONTEXT *ctx)
{
  int i;
  
#ifdef USE_IMAP
  if (ctx->magic == M_IMAP)
    imap_fastclose_mailbox (ctx);
#endif /* USE_IMAP */
  for (i = 0; i < ctx->msgcount; i++)
    mutt_free_header (&ctx->hdrs[i]);
  safe_free ((void **) &ctx->hdrs);
  safe_free ((void **) &ctx->v2r);
  safe_free ((void **) &ctx->path);
  if (ctx->fp)
    fclose (ctx->fp);
  memset (ctx, 0, sizeof (CONTEXT));
}

/* save changes to disk */
static int sync_mailbox (CONTEXT *ctx)
{
#ifdef BUFFY_SIZE
  BUFFY *tmp = NULL;
#endif
  int rc = -1;

  if (!ctx->quiet)
    mutt_message ("Writing %s...", ctx->path);
  switch (ctx->magic)
  {
    case M_MBOX:
    case M_MMDF:
    
      rc = mbox_sync_mailbox (ctx);
#ifdef BUFFY_SIZE
      tmp = mutt_find_mailbox (ctx->path);
#endif
      break;
      
    case M_MH:
    case M_MAILDIR:
    
      rc = mh_sync_mailbox (ctx);
      break;
      
#ifdef USE_IMAP
    case M_IMAP:
    
      rc = imap_sync_mailbox (ctx);
      break;
#endif /* USE_IMAP */
  }

#ifdef BUFFY_SIZE
  mutt_update_mailbox (tmp);
#endif
  return rc;
}

/* save changes and close mailbox */
int mx_close_mailbox (CONTEXT *ctx)
{
  int i, move_messages = 0, purge = 1, read_msgs = 0;
  int isSpool = 0;
  CONTEXT f;
  char mbox[_POSIX_PATH_MAX];
  char buf[SHORT_STRING];

  if (ctx->readonly || ctx->dontwrite)
  {
    /* mailbox is readonly or we don't want to write */
    mx_fastclose_mailbox (ctx);
    return 0;
  }

  if (ctx->append)
  {
    /* mailbox was opened in write-mode */
    if (ctx->magic == M_MBOX || ctx->magic == M_MMDF)
      mbox_close_mailbox (ctx);
    else
      mx_fastclose_mailbox (ctx);
    return 0;
  }

  for (i = 0; i < ctx->msgcount; i++)
  {
    if (!ctx->hdrs[i]->deleted && ctx->hdrs[i]->read)
      read_msgs++;
  }

  if (read_msgs && !option (OPTHOLD))
  {
    char *p;

    if ((p = mutt_find_hook (M_MBOXHOOK, ctx->path)))
    {
      isSpool = 1;
      strfcpy (mbox, p, sizeof (mbox));
    }
    else
    {
      strfcpy (mbox, Inbox, sizeof (mbox));
      isSpool = mutt_is_spool (ctx->path) && !mutt_is_spool (mbox);
    }
    mutt_expand_path (mbox, sizeof (mbox));

    if (isSpool)
    {
      snprintf (buf, sizeof (buf), "Move read messages to %s?", mbox);
      if ((move_messages = query_quadoption (OPT_MOVE, buf)) == -1)
	return (-1);
    }
    else
      read_msgs = 0; /* no messages will be moved */
  }

  if (ctx->deleted)
  {
    if ((purge = query_quadoption (OPT_DELETE, "Purge deleted messages?")) < 0)
      return (-1);
  }

  for (i = 0; i < ctx->msgcount; i++)
  {
    if (!ctx->hdrs[i]->deleted && !ctx->hdrs[i]->old && option (OPTMARKOLD))
      mutt_set_flag (ctx, ctx->hdrs[i], M_OLD, 1);
  }

  if (move_messages)
  {
    if (mx_open_mailbox (mbox, M_APPEND, &f) == NULL)
      return (-1);

    mutt_message ("Moving read messages to %s...", mbox);

    for (i = 0; i < ctx->msgcount; i++)
    {
      if (ctx->hdrs[i]->read && !ctx->hdrs[i]->deleted)
      {
	mx_append_message (ctx, i, &f);
	ctx->hdrs[i]->deleted = 1;
	ctx->deleted++;
      }
    }

    mx_close_mailbox (&f);
  }
  else if (!ctx->changed && ctx->deleted == 0)
  {
    mutt_message ("Mailbox is unchanged.");
    mx_fastclose_mailbox (ctx);
    return 0;
  }
  
  if (!purge)
  {
    for (i = 0; i < ctx->msgcount; i++)
      ctx->hdrs[i]->deleted = 0;
    ctx->deleted = 0;
  }

  if (ctx->changed || ctx->deleted)
  {
    if (sync_mailbox (ctx) == -1)
      return (-1);
  }

  if (move_messages)
    mutt_message ("%d kept, %d moved, %d deleted.",
		  ctx->msgcount - ctx->deleted, read_msgs, ctx->deleted);
  else
    mutt_message ("%d kept, %d deleted.",
		  ctx->msgcount - ctx->deleted, ctx->deleted);

  if (ctx->msgcount == ctx->deleted &&
      (ctx->magic == M_MMDF || ctx->magic == M_MBOX) &&
      strcmp (ctx->path, Spoolfile) != 0 && !option (OPTSAVEEMPTY))
    unlink (ctx->path);

  mx_fastclose_mailbox (ctx);

  return 0;
}

/* save changes to mailbox
 *
 * return values:
 *	0		success
 *	-1		error
 */
int mx_sync_mailbox (CONTEXT *ctx)
{
  int rc, i, j;

  if (ctx->dontwrite)
  {
    mutt_error ("Mailbox is marked unwritable.");
    return -1;
  }
  else if (ctx->readonly)
  {
    mutt_error ("Mailbox is read-only.");
    return -1;
  }

  if (!ctx->changed && !ctx->deleted)
  {
    mutt_message ("Mailbox is unchanged.");
    return (0);
  }

  if (ctx->deleted)
  {
    if ((rc = query_quadoption (OPT_DELETE, "Purge deleted messages?")) < 0)
      return (-1);
    else if (rc == M_NO)
    {
      if (!ctx->changed)
	return 0; /* nothing to do! */
      for (i = 0 ; i < ctx->msgcount ; i++)
	ctx->hdrs[i]->deleted = 0;
      ctx->deleted = 0;
    }
  }

  if ((rc = sync_mailbox (ctx)) == 0)
  {
    mutt_message ("%d kept, %d deleted.", ctx->msgcount - ctx->deleted,
		ctx->deleted);

    if (ctx->msgcount == ctx->deleted &&
	(ctx->magic == M_MBOX || ctx->magic == M_MMDF) &&
	strcmp (ctx->path, Spoolfile) != 0 && !option (OPTSAVEEMPTY))
    {
      unlink (ctx->path);
      mx_fastclose_mailbox (ctx);
      return 0;
    }

    /* update memory to reflect the new state of the mailbox */
    ctx->vcount = 0;
    ctx->tagged = 0;
    ctx->deleted = 0;
    ctx->new = 0;
    ctx->changed = 0;
    ctx->flagged = 0;
    for (i=0, j=0; i<ctx->msgcount; i++)
    {
      if (!ctx->hdrs[i]->deleted)
      {
	if (i != j)
	{
	  ctx->hdrs[j] = ctx->hdrs[i];
	  ctx->hdrs[i] = NULL;
	}
	ctx->hdrs[j]->msgno = j;
	if (ctx->hdrs[j]->virtual != -1)
	{
	  ctx->v2r[ctx->vcount] = j;
	  ctx->hdrs[j]->virtual = ctx->vcount++;
	}
	ctx->hdrs[j]->changed = 0;
	if (ctx->hdrs[j]->tagged)
	  ctx->tagged++;
	if (ctx->hdrs[j]->flagged)
	  ctx->flagged++;
	if (!ctx->hdrs[j]->read && !ctx->hdrs[j]->old)
	  ctx->new++;
	j++;
      }
      else
	mutt_free_header (&ctx->hdrs[i]);
    }
    ctx->msgcount = j;

    mutt_sort_headers (ctx);
  }

  return (rc);
}

int mh_open_new_message (MESSAGE *msg, CONTEXT *dest, HEADER *hdr)
{
  int hi = 1;
  int fd;
  int n;
  char *cp;
  char path[_POSIX_PATH_MAX];
  DIR *dirp;
  struct dirent *de;

  do
  {
    if ((dirp = opendir (dest->path)) == NULL)
    {
      mutt_perror (dest->path);
      return (-1);
    }

    /* figure out what the next message number is */
    while ((de = readdir (dirp)) != NULL)
    {
      cp = de->d_name;
      while (*cp)
      {
	if (!isdigit (*cp))
	  break;
	cp++;
      }
      if (!*cp)
      {
	n = atoi (de->d_name);
	if (n > hi)
	  hi = n;
      }
    }
    closedir (dirp);
    hi++;
    snprintf (path, sizeof (path), "%s/%d", dest->path, hi);
    if ((fd = open (path, O_WRONLY | O_EXCL | O_CREAT, 0600)) == -1)
    {
      if (errno != EEXIST)
      {
	mutt_perror (path);
	return (-1);
      }
    }
  }
  while (fd < 0);

  if ((msg->fp = fdopen (fd, "w")) == NULL)
    return (-1);

  return 0;
}

int maildir_open_new_message (MESSAGE *msg, CONTEXT *dest, HEADER *hdr)
{
  char tmp[_POSIX_PATH_MAX];
  char path[_POSIX_PATH_MAX];

  maildir_create_filename (dest->path, hdr, path, tmp);
  if ((msg->fp = safe_fopen (tmp, "w")) == NULL)
    return (-1);
  return 0;
}

int mbox_open_new_message (MESSAGE *msg, CONTEXT *dest, HEADER *hdr)
{
  msg->fp = dest->fp;
  if (dest->magic == M_MMDF)
    fputs (MMDF_SEP, msg->fp);
  return 0;
}

/* args:
 *	dest	destintation mailbox
 *	hdr	message being copied (required for maildir support, because
 *		the filename depends on the message flags)
 */
MESSAGE *mx_open_new_message (CONTEXT *dest, HEADER *hdr)
{
  MESSAGE *msg;
  int (*func) (MESSAGE *, CONTEXT *, HEADER *);

  switch (dest->magic)
  {
    case M_MMDF:
    case M_MBOX:
      func = mbox_open_new_message;
      break;

    case M_MAILDIR:
      func = maildir_open_new_message;
      break;

    case M_MH:
      func = mh_open_new_message;
      break;

    default:
      dprint (1, (debugfile, "mx_open_new_message(): function unimplemented for mailbox type %d.\n",
		  dest->magic));
      return (NULL);
  }

  msg = safe_calloc (1, sizeof (MESSAGE));
  msg->magic = dest->magic;
  msg->write = 1;

  if (func (msg, dest, hdr) != 0)
    safe_free ((void **) &msg);

  return msg;
}

/* Called from mx_append_message () */
int mutt_append_message (FILE *fp, HEADER *hdr, int has_from, CONTEXT *dest)
{
  MESSAGE *destmsg;               /* new message stream */
  int chflags;                    /* flags to mutt_copy_header */
  int r = -1;                     /* return value */

  chflags = option (OPTFORWARD) ? (CH_XMIT | CH_MIME) : CH_UPDATE;
  if ((destmsg = mx_open_new_message (dest, hdr)) != NULL)
  {
    if (!option (OPTFORWARD) && (dest->magic == M_MMDF || dest->magic == M_MBOX))
    {
      if (!has_from)
      {
	char tmp[LONG_STRING];
	ADDRESS *p;

	/*
	 * need to generate a "From " line because these formats do not
	 * use one.
	 */
	if (hdr->env->return_path)
	  p = hdr->env->return_path;
	else if (hdr->env->sender)
	  p = hdr->env->sender;
	else
	  p = hdr->env->from;

	tmp[0] = 0;
	mutt_simple_address (tmp, sizeof (tmp), p);

	if (!hdr->received)
	  hdr->received = time (NULL);
	fprintf (destmsg->fp, "From %s %s", tmp[0] ? tmp : "-", ctime (&hdr->received));
      }
      else
	chflags |= CH_FROM;
    }
    else if (dest->magic == M_MAILDIR)
      chflags |= CH_NOSTATUS;

    mutt_copy_header (fp, hdr, destmsg->fp, chflags);
    /* mutt_copy_header() doesn't add '\n' with CH_XMIT */
    if (option (OPTFORWARD))
      fputc ('\n', destmsg->fp);
    fseek (fp, hdr->content->offset, 0);
    mutt_copy_bytes (fp, destmsg->fp, hdr->content->length);

    mx_close_message (&destmsg);
    r = 0;
  }

  return (r);
}

int mutt_reopen_mailbox (CONTEXT *ctx, unsigned short *index_hint)
{
  int (*cmp_headers) (const HEADER *, const HEADER *) = NULL;
  HEADER **old_hdrs;
  unsigned short old_msgcount;
  int msg_mod = 0;
  int index_hint_set;
  int i, j;
  int rc = -1;

  /* silent operations */
  ctx->quiet = 1;
  
  mutt_message ("Reopening mailbox...");
  
  /* our heuristics require the old mailbox to be unsorted */
  if (Sort != SORT_ORDER)
  {
    short old_sort;

    old_sort = Sort;
    Sort = SORT_ORDER;
    mutt_sort_headers (ctx);
    Sort = old_sort;
  }

  /* save the old headers */
  old_msgcount = ctx->msgcount;
  old_hdrs = ctx->hdrs;

  /* simulate a close */
  for (i = 0; i < ctx->msgcount; i++)
  {
    safe_free ((void **) &ctx->v2r);
    if (ctx->readonly)
      mutt_free_header (&(ctx->hdrs[i])); /* nothing to do! */
    else
      ctx->hdrs = NULL;
  }

  if (ctx->readonly)
    safe_free ((void **) &ctx->hdrs);

  ctx->hdrmax = 0;	/* force allocation of new headers */
  ctx->msgcount = 0;
  ctx->vcount = 0;
  ctx->tagged = 0;
  ctx->deleted = 0;
  ctx->new = 0;
  ctx->flagged = 0;

  switch (ctx->magic)
  {
    case M_MBOX:

      fseek (ctx->fp, 0, 0);
      cmp_headers = mbox_strict_cmp_headers;
      rc = mbox_parse_mailbox (ctx);
      break;

    case M_MMDF:
      fseek (ctx->fp, 0, 0);
      cmp_headers = mbox_strict_cmp_headers;
      rc = mmdf_parse_mailbox (ctx);
      break;

    case M_MH:
      /* cmp_headers = mh_strict_cmp_headers; */
      rc = mh_read_dir (ctx, NULL);
      break;

    case M_MAILDIR:
      /* cmp_headers = maildir_strict_cmp_headers; */
      rc = maildir_read_dir (ctx);
      break;

    default:
      rc = -1;
      break;
  }
  
  if (rc == -1)
  {
    /* free the old headers */
    for (j = 0; j < old_msgcount; j++)
      mutt_free_header (&(old_hdrs[j]));
    safe_free ((void **) &old_hdrs);

    ctx->quiet = 0;
    return (-1);
  }

  /* now try to recover the old flags */

  index_hint_set = (index_hint == NULL);

  if (!ctx->readonly)
  {
    for (i = 0; i < ctx->msgcount; i++)
    {
      int found = 0;

      /* some messages have been deleted, and new  messages have been
       * appended at the end; the heuristic is that old messages have then
       * "advanced" towards the beginning of the folder, so we begin the
       * search at index "i"
       */
      for (j = i; j < old_msgcount; j++)
      {
	if (old_hdrs[j] == NULL)
	  continue;
	if (cmp_headers (ctx->hdrs[i], old_hdrs[j]))
	{
	  found = 1;
	  break;
	}
      }
      if (!found)
      {
	for (j = 0; j < i; j++)
	{
	  if (old_hdrs[j] == NULL)
	    continue;
	  if (cmp_headers (ctx->hdrs[i], old_hdrs[j]))
	  {
	    found = 1;
	    break;
	  }
	}
      }

      if (found)
      {
	/* this is best done here */
	if (!index_hint_set && *index_hint == j)
	  *index_hint = i;
	if (old_hdrs[j]->changed || old_hdrs[j]->deleted)
	{
	  /* Only update the flags if the old header was changed;
	   * otherwise, the header may have been modified externally,
	   * and we don't want to lose _those_ changes
	   */
	  if (ctx->hdrs[i]->flagged != old_hdrs[j]->flagged)
	  {
	    ctx->hdrs[i]->flagged = old_hdrs[j]->flagged;
	    if (old_hdrs[j]->flagged)
	      ctx->flagged++;
	    else
	      ctx->flagged--;
	    ctx->hdrs[i]->changed = 1;
	    ctx->changed = 1;
	  }
	  if (ctx->hdrs[i]->old != old_hdrs[j]->old)
	  {
	    ctx->hdrs[i]->old = old_hdrs[j]->old;
	    ctx->hdrs[i]->changed = 1;
	    ctx->changed = 1;
	  }
	  if (ctx->hdrs[i]->read != old_hdrs[j]->read)
	  {
	    ctx->hdrs[i]->read = old_hdrs[j]->read;
	    ctx->hdrs[i]->changed = 1;
	    ctx->changed = 1;
	  }
	  if (ctx->hdrs[i]->replied != old_hdrs[j]->replied)
	  {
	    ctx->hdrs[i]->replied = old_hdrs[j]->replied;
	    ctx->hdrs[i]->changed = 1;
	    ctx->changed = 1;
	  }
	  if (!ctx->hdrs[i]->deleted && old_hdrs[j]->deleted)
	  {
	    ctx->hdrs[i]->deleted = 1;
	    ctx->deleted++;
	    ctx->changed = 1;
	  }
	}
	
	/* tagged messages need to be restored anyway */
	if (old_hdrs[j]->tagged)
	{
	  ctx->hdrs[i]->tagged = 1;
	  ctx->tagged++;
	}

	/* we don't need this header any more */
	mutt_free_header (&(old_hdrs[j]));
      }
    }

    /* update the "new" count and the "changed" flag */
    ctx->new = 0;
    for (i = 0; i < ctx->msgcount; i++)
    {
      if (!ctx->hdrs[i]->old && (!ctx->hdrs[i]->read || option (OPTMARKOLD)))
	ctx->new++;
    }

    /* free the remaining old headers */
    for (j = 0; j < old_msgcount; j++)
    {
      if (old_hdrs[j])
      {
	mutt_free_header (&(old_hdrs[j]));
	msg_mod = 1;
      }
    }
    safe_free ((void **) &old_hdrs);
  }

  ctx->quiet = 0;

  /* 2 = folder modified
   * 1 = new mail has arrived
   */
  rc = (ctx->changed || msg_mod) ? 2 : 1;
  return (rc);
}

/* append a message to another mailbox */
int mx_append_message (CONTEXT *src, int msgno, CONTEXT *dest)
{
  MESSAGE *msg;			/* source message stream */
  int r = -1;			/* return value */

  if ((msg = mx_open_message (src, msgno)) != NULL)
  {
    mutt_parse_mime_message (src->hdrs[msgno]);
    r = mutt_append_message (msg->fp, src->hdrs[msgno],
		      (src->magic == M_MBOX || src->magic == M_MMDF), dest);
    mx_close_message (&msg);
  }
  return r;
}

/* Called from mx_append_decoded() and mutt_view_attachments()
 * to build a new message with a given "flat" body
 */
int mutt_append_decoded (FILE *fph, FILE *fpb, HEADER *hdr, BODY *b, CONTEXT *dest)
{
  char buffer[LONG_STRING];
  char tempfile[_POSIX_PATH_MAX];
  char attfile[_POSIX_PATH_MAX];
  HEADER *hn;			/* destination header */
  MESSAGE *msg;			/* new message stream */
  STATE s;
  FILE *tempfp = NULL;
  struct stat st;
  int r = -1;			/* return value */

#ifdef _PGPPATH
  if(hdr->pgp & PGPENCRYPT && !pgp_valid_passphrase())
    return r;
  unset_option (OPTVERIFYSIG);
#endif

  /* create the temporary file now in order to avoid memory cleanup
   * if the operation fails
   */
  if (dest->magic == M_MMDF || dest->magic == M_MBOX)
  {
    mutt_mktemp (tempfile);
    if ((tempfp = safe_fopen (tempfile, "w+")) == NULL)
    {
      mutt_perror (tempfile);
      return r;
    }
  }

  /* build a header for the new message */
  hn = mutt_new_header ();
  fseek (fph, hdr->offset, 0);
  /* recover the relevant fields from the old header */
  hn->env = _mutt_read_rfc822_header (fph, hn, 1);

  /* we need to make sure we don't write lines with more than 72 characters,
   * so we rely on _mutt_read_rfc822_header() to decode the headers, and
   * we encode them back here
   */
  rfc2047_encode_adrlist (hn->env->from);
  rfc2047_encode_adrlist (hn->env->to);
  rfc2047_encode_adrlist (hn->env->cc);
  rfc2047_encode_adrlist (hn->env->reply_to);
  rfc2047_encode_adrlist (hn->env->return_path);
  rfc2047_encode_adrlist (hn->env->sender);

  if (hn->env->subject)
  {
    rfc2047_encode_string (buffer, (unsigned char *) hn->env->subject,
                           sizeof (buffer) - 1);
    safe_free ((void **) &hn->env->subject);
    hn->env->subject = safe_strdup (buffer);
  }

  /* generate a new Message-Id: */
  hn->env->message_id = mutt_gen_msgid ();

  hn->pgp = 0;
  if (dest->magic == M_MAILDIR)
    hn->read = 1;

  /* decode the attachments */
  memset (&s, 0, sizeof (s));
  s.fpin = fpb;
  mutt_mktemp (attfile);
  s.fpout = safe_fopen (attfile, "w+");

  mutt_body_handler (b, &s);
  fclose (s.fpout);

  /* build a body for the new message */
  hn->content->use_disp = 0;
  hn->content->filename = safe_strdup (attfile);
  /* the new message can still contain accented characters */
  mutt_update_encoding (hn->content);

  if ((msg = mx_open_new_message (dest, hn)) != NULL)
  {
    if (!option (OPTFORWARD) && (dest->magic == M_MMDF || dest->magic == M_MBOX))
    {
      /* we need a "From " line */
      time_t now;
      ADDRESS *p = NULL;

      /* try something that makes sense first */
      if (hdr->env->from)
	p = hdr->env->from;
      else if (hdr->env->sender)
	p = hdr->env->sender;
      else if (hdr->env->return_path)
	p = hdr->env->return_path;

      buffer[0] = 0;
      mutt_simple_address (buffer, sizeof (buffer), p);
      now = time (NULL);

      fprintf (msg->fp, "From %s %s", buffer[0] ? buffer : Username, ctime (&now));
    }

    /* write the new header */
    mutt_write_rfc822_header (msg->fp, hn->env, hn->content, 0);
    if (!option (OPTFORWARD) && dest->magic != M_MAILDIR)
      fputs ("Status: RO\n", msg->fp);

    mutt_make_string (buffer, sizeof (buffer), DecodeFmt, hdr);

    /* add a "Content-Length:" field */
    if (!option (OPTFORWARD) && (dest->magic == M_MMDF || dest->magic == M_MBOX))
    {
      /* write the ID string */
      fprintf (tempfp, "%s\n", buffer);
      
      mutt_write_mime_body (hn->content, tempfp);
      
      /* make sure the last line ends with a newline.  Emacs doesn't ensure
       * this will happen, and it can cause problems parsing the mailbox
       * later.
       */
      fseek (tempfp, -1, 2);
      if (fgetc (tempfp) != '\n')
      {
	fseek (tempfp, 0, 2);
	fputc ('\n', tempfp);
      }
      
      fflush (tempfp);
      rewind (tempfp);
      fstat (fileno (tempfp), &st);
      fprintf (msg->fp, "Content-Length: %ld\n\n", (long) st.st_size);

      /* copy the body */
      mutt_copy_stream (tempfp, msg->fp);
    }
    else
    {
      /* finish off the header and write the ID string */
      fprintf (msg->fp, "\n%s\n", buffer);
      
      /* write the new body */
      mutt_write_mime_body (hn->content, msg->fp);
    }


    mx_close_message (&msg);
    r = 0;
  }
  
  if (tempfp)
  {
    fclose (tempfp);
    unlink (tempfile);
  }

  unlink (hn->content->filename);
  mutt_free_header (&hn);

  return (r);
}

/* append a decoded version of a message to another mailbox */
int mx_append_decoded (CONTEXT *src, int msgno, CONTEXT *dest)
{
  MESSAGE *msg;			/* source message stream */
  int r = -1;			/* return value */
  
  if ((msg = mx_open_message (src, msgno)) != NULL)
  {
    mutt_parse_mime_message (src->hdrs[msgno]);
    r = mutt_append_decoded (msg->fp, msg->fp, src->hdrs[msgno],
				      src->hdrs[msgno]->content, dest);
    mx_close_message (&msg);
  }
  return r;
}

/* check for new mail */
int _mx_check_mailbox (CONTEXT *ctx, unsigned short *index_hint)
{
  if (ctx)
  {
    switch (ctx->magic)
    {
      case M_MBOX:
      case M_MMDF:
	return (mbox_check_mailbox (ctx, index_hint));

      case M_MH:
      case M_MAILDIR:
	return (mh_check_mailbox (ctx, index_hint));

#ifdef USE_IMAP
      case M_IMAP:
	return (imap_check_mailbox (ctx, index_hint));
#endif /* USE_IMAP */
    }
  }

  dprint (1, (debugfile, "mx_check_mailbox: null or invalid context.\n"));
  return (-1);
}

/* return a stream pointer for a message */
MESSAGE *mx_open_message (CONTEXT *ctx, int msgno)
{
  HEADER *cur = ctx->hdrs[msgno];
  MESSAGE *msg;
  char path[_POSIX_PATH_MAX];
  
  msg = safe_calloc (1, sizeof (MESSAGE));
  switch (msg->magic = ctx->magic)
  {
    case M_MBOX:
    case M_MMDF:
      msg->fp = ctx->fp;
      break;

    case M_MH:
    case M_MAILDIR:
      snprintf (path, sizeof (path), "%s/%s", ctx->path, cur->path);
      if ((msg->fp = fopen (path, "r")) == NULL)
      {
	mutt_perror (path);
	dprint (1, (debugfile, "mx_open_message: fopen: %s: %s (errno %d).\n",
		    path, strerror (errno), errno));
	free (msg);
	msg = NULL;
      }
      break;

#ifdef USE_IMAP
    case M_IMAP:
      if (imap_fetch_message (msg, ctx, msgno) != 0)
      {
	free (msg);
	msg = NULL;
      }
      break;
#endif /* USE_IMAP */

    default:

      dprint (1, (debugfile, "mx_open_message(): function not implemented for mailbox type %d.\n", ctx->magic));
      safe_free ((void **) &msg);
      break;
  }
  return (msg);
}

/* close a pointer to a message */
int mx_close_message (MESSAGE **msg)
{
  int r = 0;

  if ((*msg)->write)
  {
    /* add the message terminator */
    switch ((*msg)->magic)
    {
      case M_MMDF:
	fputs (MMDF_SEP, (*msg)->fp);
	break;

      case M_MBOX:
	fputc ('\n', (*msg)->fp);
	break;
    }
  }

  if ((*msg)->magic == M_MH || (*msg)->magic == M_MAILDIR || (*msg)->magic == M_IMAP)
    r = fclose ((*msg)->fp);

  free (*msg);
  *msg = NULL;
  return (r);
}

void mx_alloc_memory (CONTEXT *ctx)
{
  int i;

  if (ctx->hdrs)
  {
    safe_realloc ((void **) &ctx->hdrs, sizeof (HEADER *) * (ctx->hdrmax += 25));
    safe_realloc ((void **) &ctx->v2r, sizeof (short) * ctx->hdrmax);
  }
  else
  {
    ctx->hdrs = safe_malloc (sizeof (HEADER *) * (ctx->hdrmax += 25));
    ctx->v2r = safe_malloc (sizeof (short) * ctx->hdrmax);
  }
  for (i = ctx->msgcount ; i < ctx->hdrmax ; i++)
  {
    ctx->hdrs[i] = NULL;
    ctx->v2r[i] = -1;
  }
}

/* this routine is called to update the counts in the context structure for
 * the last message header parsed.
 */
void mx_update_context (CONTEXT *ctx)
{
  HEADER *h = ctx->hdrs[ctx->msgcount];

#ifdef _PGPPATH
  /* NOTE: this _must_ be done before the check for mailcap! */
  h->pgp = pgp_query (h->content);
  if (!h->pgp)
#endif /* _PGPPATH */
    if (mutt_needs_mailcap (h->content))
      h->mailcap = 1;
  if (h->flagged)
    ctx->flagged++;
  if (h->deleted)
    ctx->deleted++;
  if (!h->read && !h->old)
    ctx->new++;
  if (ctx->vcount == ctx->msgcount)
  {
    ctx->v2r[ctx->vcount] = ctx->msgcount;
    h->virtual = ctx->vcount++;
  }
  else
    h->virtual = -1;
  h->msgno = ctx->msgcount;
  ctx->msgcount++;
}

int mx_copy_message (HEADER *h, FILE *in, FILE *out)
{
  if (mutt_copy_header (in, h, out, CH_FROM | CH_UPDATE) == -1)
    return (-1);

  return (mutt_copy_bytes (in, out, h->content->length));
}
