/* texpars2.cc
   Definitions of classes for parsing TeX.
   Second part: the actual parsing,
                and the writing of Lyx.

    Author: John Collins, collins@phys.psu.edu.
    22 Jan 96

    (C) John Collins & Penn State University.

*/

#include "texparse.h"
#include "utils.h"
#include <string.h>

// Global variables
int FatalError = 0;


// ====== Parsing utilities ====


int FindClose (const BeginGroup &start, BaseTeXObject *&end) {
  /*  Assuming that start is a { object.
      Find its matching }.
      Return: 0 if I succeed.
              >=1 if I get to the end-of-file without a match.
                result is the number of unclosed braces.
  */
  end = start.next;
  // The { starts a group.
  int bracelevel = 1;
  while (end != NULL) {
     if (end->IsA() == IDBG) bracelevel++;
     if (end->IsA() == IDEG) bracelevel--;
     if (bracelevel < 1) return 0; // Success
     end = end->next;
  }
  // No matching } before end-of-file.
  return bracelevel;
}




int FindMatchingEnd (const BeginEnv &start, BaseTeXObject *&end) {
  /*  Assuming that start is a \begin object.
      Find its matching \end.
      Return: 0 if I succeed.
              1 if I find a matching \end, but its name is wrong.
                (E.g., \begin{eqnarray} ... \end{equation}
              2 if there was a } without a corresponding { before I found an
                \end. This could indicate start is in the middle of
                a macro definition (i.e., it is legal), or that the document
                is incorrect. end points to the bad brace.
              3 if I find a matching \end, but there are unclosed {s. This
                should indicate that the document is incorrect.
                end points to the \end.
              4 Same as 3, except that the name of the \end is not equal
                to mine. Again, this should indicate that the document
                is incorrect. end points to the bad brace.
              5 if I don't find a matching \end by the end the file, but
                there are matching braces. end is NULL.
              6 if I don't find a matching \end by the end of the file, and
                there are unclosed opening braces. end is NULL.
              7 if there was a mismatched environment before I found the
                matching \end.
  */

  end = start.next;
  // The \begin starts a group.
  int bracelevel = 1;
  while (end) {
     bracelevel -= end->Closes;
     if (bracelevel < 1) {
        // I have found a group/environment closing that matches my start.
        if (end->IsA() == IDEnd) goto Found;
        else return 2;
     }
     switch (end->IsA()) {
        // Here, I either get out of the loop or point end to the next object
        // to look at.
        case IDBegin: 
              // Enclose the body of this case in a group, to avoid local
              // temporary variables being visible outside.  Accessing later
              // cases of the switch bypasses their initialization, which is
              // a (compile-time) error.
              {  BeginEnv *thisbegin = (BeginEnv *)end;
                 int result = FindMatchingEnd (*thisbegin, end);
                 if (result != 0) { end = thisbegin; return 7;}
                 end = end->next;
	       }
               break;
        case IDEnd:
                 goto Found;
        default:
                 bracelevel += end->Opens;
                 end = end->next;
     }
  }
  // No matching \end before end-of-file.
  if (bracelevel == 0) return 5;
  else return 6;

Found:
  // end points to the matching construct to my \begin
  if (SameName(start, *((EndEnv *) end) )) {
    if (bracelevel == 0) return 0;
    else return 3;
  } else {
    if (bracelevel == 0) return 1;
    else return 4;
  }
}



void Extract (BaseTeXObject *first, BaseTeXObject *last) {
   /* Unlink the list from first to last.
      Set the pointers of the rest of the list correctly.
      first or last ==NULL => do nothing.
   */
   if (!first || !last) {
      // There's nothing to do.
   }
   // Fix up pointers of attached list:
   if (first->prev) (first->prev)->next = last->next;
   if (last->next) (last->next)->prev = first->prev;

   // Now detach my piece
   first->prev = last->next = NULL;
}


void Extract (BaseTeXObject *first) {
   /* Unlink the list from first onwards.
      Set the pointers of the rest of the list correctly.
      first==NULL => do nothing.
   */
   if (!first || !(first->prev) ) {
      // There's nothing to do.
   }
   // The original predecessor of first is now the end of its list:
   (first->prev)->next = NULL;
   // And first is the head of its list.
   first->prev = NULL;
}



void MoveTo (BaseTeXObject &dest,
             BaseTeXObject *source, BaseTeXObject *end) {
   /* Move the list source to end to just after dest.
      If end is NULL, the whole of the rest of the list is to be moved.
   */
   if (end) Extract (source, end);
   else Extract (source);
   if (dest.next) {
      //Patch in source's tail.
      end = source->tail();
      (dest.next)->prev = end;
      end->next = dest.next;
   }
   source->prev = &dest;
   dest.next = source;
}



//=========================================================
//=========================================================

// ======= Now the parsing routines =====

int ParseAll (BaseTeXObject * Current, TeXState &s) {
   /* Starting at current, parse all remaining objects.
      Return 0 if no errors, else
      return 1 for non-fatal errors,
      or return 100 for fatal errors.
   */
   int MyRetCode = 0;
   FatalError = 0;
   BaseTeXObject *next = NULL;
   while (Current) {
      int RetCode = Current->Parse(next, s);
      /* Parse() has many side effects, and it may totally rearrange
         the logical list of which *Current is a part.
      */
      if (FatalError) {
         MyRetCode = 100;
      } else if (RetCode != 0) {
         MyRetCode = 1;
      }
      Current = next;
   }
   return MyRetCode;
}


//=========================================================

int BaseTeXObject::Parse (BaseTeXObject *& next_to_parse, TeXState &s) {
   /* Parse me, so that I am a complete object.
      Rearrange the logical list to which I am attached as needed.
      Set next_to_parse to the next object to parse.
      Return 0 if no errors, else return non-zero error code.
   */
   // The default is to do nothing.
   next_to_parse = next;
   return 0;
}

//===================== TeXObject, BaseTeXSpecial ...=====================


int TeXCS::Parse (BaseTeXObject *& next_to_parse, TeXState &s) {
//   fprintf (stderr, "Parsing TeXCS\n");
   next_to_parse = next;
   return 0;
}



int BeginEnv::Parse (BaseTeXObject *& next_to_parse, TeXState &s) {
//For debugging:   lite_cerr << "Parsing Begin Env\n";
   BaseTeXObject *end;
   int result = FindMatchingEnd(*this, end);

   //For debugging: lite_cerr << "Result=" << result << ". ";
   if (result) {
      Error ("Failed to find matching end.", start);
//      if (end) end->Show();
//      else lite_cerr << "No end\n";
      switch (result) {
   case 1: Error ("The matching \\end had a different name.", end->start);
           break;
   case 2: Error ("Braces closed before I found an \\end.", end->start);
           break;
   case 3: Error ("There were open braces at the matching \\end.");
           break;
   case 4: Error ("There were open braces at the matching \\end.\n"
                  "And the \\end had a different name.", end->start);
           break;
   case 5: Error("No matching \\end by the end-of-file.");
           break;
   case 6: Error("No matching \\end by the end-of-file.\n"
                 "And there were open braces.");
           break;
   case 7: Error("No matching \\end by the end-of-file.");
           break;
       }
      next_to_parse = next;
   } else {
      //For debugging:lite_cerr << "Success\n";
      if (end == next) {
         // Body is empty:
         Body = new ListHead;
      } else {
         Body = new ListHead(next, end->prev);
      }
      next_to_parse = end->next;
      // Remove the \end.  It's superfluous
      Extract (end, end);
      delete end;
      // And fix up my properties
      if (strcmp(name, "document") == 0)
         ID = IDDocumentBody;
      else
         ID = IDEnv;
      Closes = 0;
      Opens = 0;
   }
   return result;
}



//=========================================================
//=========================================================

// ======= Lyx writing =============

void BaseTeXObject::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   // Default is same as TeX.
   WriteTeX (f);
}

void TeXObject::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   // Default is now my text, Lyx-style.
   if (state.InPar == 0) {
      f << "\n\\layout Standard\n";
      state.InPar = 1;
   }
   WriteMyLyxText(f);
}


void TeXObject::WriteMyLyxText (sos &f) const {
   /* Write out the text I point to, but appropriately for Lyx.
      Since Lyx ignores newlines, but TeX treats them as spaces,
      we must put in an extra space.
   */

   Line *curline = start.l;
   unsigned int i1 = start.index;
   while (curline){
      if (curline->len > 0) {
         /* There's something to print on this line.
	    Print chars [i1] to [i2] of current line.
            i1 is determined.
            i2 is either end of line, or the end of the buffer, whichever is
            earlier.
         */
         unsigned int i2 = (curline->len)-1;
         // Note that i2 is the index into the file, and is zero-based.
         if ((curline == end.l) && (i2 > end.index))
            i2 = end.index;

//         for (unsigned int i = i1; i <= i2; i++)
//            f.put ((*curline)[i]);
         // The following is probably faster:
         //write (fileno(f), curline->data+i1, i2-i1+1);
         f.write (curline->data+i1, i2-i1+1);
         if (curline->data[i2] == '\n')
         // After the end of a line, put the extra space Lyx will want
           f.put (' ');
      }
      if (curline == end.l) {
         curline = NULL; // Flag that we are done
      } else {
         curline = curline->next;
         i1 = 0;
      }
   }
}

//===================== TeXObject ... =====================

void TeXText::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   if (state.InPar == 0) {
      f << "\n\\layout Standard\n";
      state.InPar = 1;
   }
   WriteMyLyxText(f);
}

//===================== TeXObject, TeXText ... =====================


void BlankLine::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   // Default is now my text, Lyx-style.
   f << "\n\n";
   state.InPar = 0;
}

void Comment::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   f << "\\latex latex ";
   char *data = (start.l)->data;
   // Note that the comment is all on one line.
   for (int i = start.index; i <= end.index; i++) {
       // Convert backslash:
       if (data[i] == '\\') f << "\\backslash ";
       else f.put(data[i]);
   }
   f << "\\latex default \n";
}


//=========================================================
//===================== TeXObject ... =====================


void BaseTeXSpecial::WriteLyx (LyxState &state, sos &f) const {
   // Write a dummy Lyx representation of this object.
   f << " ===.UNIMPLEMENTED.OBJECT.=== \n";
   WriteMyLyxText(f);
}

//===================== TeXObject, BaseTeXSpecial ... =====================

void TeXCS::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   if (ID == IDPar) {
      f <<"\n\n";
      state.InPar = 0;
      return;
   }
   f << "\\latex latex ";
   for (int i = 1; i<=lenCS; i++) {
       // Convert initial backslash:
       if (start[i-1] == '\\') f << "\\backslash ";
       else f.put(start[i-1]);
   }
   if (chcat[start[lenCS]] == Letter)
      f.put(' ');
   f << "\\latex default ";
}


void DocClass::WriteLyx (LyxState &state, sos &f) const {
   //For debugging: f << " ===DOCUMENTCLASS=== \n";
}


void BeginEnv::WriteLyx (LyxState &state, sos &f) const {
   // Dummy for now, except:
   state.EnvLevel++;
// For debugging:   f << " =====BEGIN{" << name << "}====== \n";
   if (Body)
      Body->WriteLyxAll(state, f);
}

void EndEnv::WriteLyx (LyxState &state, sos &f) const {
   // Dummy for now, except:
   state.EnvLevel--;
// For debugging:      f << " =====END{" << name << "}====== \n";
}


void BaseSimple::WriteLyx (LyxState &state, sos &f) const {
   // Write the Lyx representation of this object.
   f.put(name);
}

//=========================================================
//=========== TeXObject, BaseTeXSpecial, BaseSimple ... ========


void BeginGroup::WriteLyx (LyxState &state, sos &f) const {
//   if (state.InPar == 0) {
     BaseTeXObject *close;
     if ( (FindClose(*this, close) == 0)
         && (close->prev->IsA()==IDPar) ){
        // I am at beginning of Lyx-generated paragraph
        state.InPar = 1;
        return;
     }
//   }
   f.put('{');
}


void EndGroup::WriteLyx (LyxState &state, sos &f) const {
   if ((state.InPar == 0) && prev && (prev->IsA() == IDPar)) {
      // This was generated by Lyx to surround a paragraph
   } else {
      f.put('}');
   }
}



//====================================================
//===================== ListHead =====================


void ListHead::WriteLyxAll (LyxState &state, sos &f) const {
   for (BaseTeXObject const *t = this; t; t = t->next)
      t->WriteLyx (state, f);
}


