/*
 * dmnt.c - FreeBSD mount support functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.12 96/04/05 09:55:55 abe Exp $";
#endif


#include "lsof.h"


/*
 * Local static information
 */

static char *mnt_names[] =

#if	_FREEBSDV<200
	{ "none", "ufs", "nfs", "mfs", "pc", "iso9600", "procfs", "devfs" };
#else	/* _FREEBSDV>=200 */
	INITMOUNTNAMES;
#endif	/* _FREEBSDV<200 */


/*
 * readmnt() - read mount table
 */

int
readmnt()
{
	char *dn = NULL;
	int err = 0;
	char *ln;
	struct statfs *mb = NULL;
	struct mounts *mtp;
	int n;
	struct stat sb;

#if	defined(HASPROCFS)
	unsigned char procfs = 0;
#endif	/* defined(HASPROCFS) */

/*
 * Access mount information.
 */
	if ((n = getmntinfo(&mb, MNT_NOWAIT)) <= 0) {
		(void) fprintf(stderr, "%s: no mount information\n", Pn);
		return(0);
	}
/*
 * Read mount information.
 */
	for (; n; n--, mb++) {
		if (mb->f_type == MOUNT_NONE || mb->f_type >= MOUNT_MAXTYPE)
			continue;
	/*
	 * Interpolate a possible symbolic directory link.
	 */
		if (dn)
			(void) free((FREE_P *)dn);
		if ((dn = (char *)malloc((MALLOC_S)strlen(mb->f_mntonname)+1))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(dn, mb->f_mntonname);
		if ((ln = Readlink(dn)) == NULL) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
			}
			err = 2;
			continue;
		}
		if (ln != dn) {
			(void) free((FREE_P *)dn);
			dn = ln;
		}
	/*
	 * Stat() the directory.
	 */
		if (statsafely(dn, &sb)) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "%s: WARNING: can't stat() %s file system %s\n",
			    Pn, mnt_names[mb->f_type],
			    mb->f_mntonname);
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
		    }
		    err = 2;
		    (void) bzero((char *)&sb, sizeof(sb));
		    sb.st_dev = (dev_t)mb->f_fsid.val[0];
		    sb.st_mode = S_IFDIR | 0777;
		    if (!Fwarn)
			(void) fprintf(stderr,
			    "      assuming \"dev=%x\" from mount table\n",
			    sb.st_dev);
		}
	/*
	 * Allocate and fill a local mount structure.
	 */
		if ((mtp=(struct mounts *)malloc(sizeof(struct mounts))) == NULL
		||  (mtp->fsname = (char *)malloc((MALLOC_S)
			strlen(mb->f_mntfromname)+1)) == NULL)
		{
			err = 1;
			break;
		}
		(void) strcpy(mtp->fsname, mb->f_mntfromname);
		mtp->dir = dn;
		dn = NULL;

#if	defined(HASPROCFS)
		if (mb->f_type == MOUNT_PROCFS) {

		/*
		 * Save information on exactly one procfs file system.
		 */
			if (procfs)
				Mtprocfs = NULL;
			else {
				procfs = 1;
				Mtprocfs = mtp;
			}
		}
#endif	/* defined(HASPROCFS) */

		mtp->next = Mtab;
		mtp->dev = sb.st_dev;
		mtp->rdev = sb.st_rdev;
		mtp->inode = sb.st_ino;
		mtp->mode = sb.st_mode;
		Mtab = mtp;
	}
/*
 * Handle errors.
 */
	switch(err) {
	case 1:
		(void) fprintf(stderr, "%s: no space for mount at %s (%s)\n",
			Pn, mb->f_mntonname, mb->f_mntfromname);
		return(0);
	case 2:
		return(1);
	}
	return(1);
}


/*
 * readvfs() - read vfs structure
 */

struct l_vfs *
readvfs(vm)
	struct mount *vm;		/* mount address in vnode */
{
	struct mount m;
	struct l_vfs *vp;

/*
 * Search for match on existing entry.
 */
	for (vp = Lvfs; vp; vp = vp->next) {
		if (vm == vp->addr)
			return(vp);
	}
/*
 * Read the (new) mount structure, allocate a local entry, and fill it.
 */
	if (kread((KA_T)vm, (char *)&m, sizeof(m)) != 0)
		return(NULL);
	if ((vp = (struct l_vfs *)malloc(sizeof(struct l_vfs))) == NULL) {
		(void) fprintf(stderr, "%s: PID %d, no space for vfs\n",
			Pn, Lp->pid);
		Exit(1);
	}
	if ((vp->dir = (char *)malloc(strlen(m.m_stat.f_mntonname)+1))
	== NULL
	||  (vp->fsname = (char *)malloc(strlen(m.m_stat.f_mntfromname)+1))
	== NULL) {
		(void) fprintf(stderr, "%s: PID %d, no space for mount names\n",
			Pn, Lp->pid);
		Exit(1);
	}
	(void) strcpy(vp->dir, m.m_stat.f_mntonname);
	(void) strcpy(vp->fsname, m.m_stat.f_mntfromname);
	vp->addr = vm;
	vp->fsid = m.m_stat.f_fsid;
	vp->type = m.m_stat.f_type;
	vp->next = Lvfs;
	return(vp);
}
