;*---------------------------------------------------------------------*/
;*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime1.8/Ast/let.scm              */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sun Jan  1 11:37:29 1995                          */
;*    Last change :  Wed Feb  7 08:38:27 1996 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The `let->ast' translator                                        */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module ast_let
   (include "Ast/node.sch"
	    "Tools/trace.sch")
   (import  parse_definition
	    type_env
	    parse_definition
	    tools_location
	    tools_progn
	    tools_shape
	    ast_substitute
	    ast_sexp
	    ast_dump
	    ast_local)
   (export  (let->ast <sexp> <stack> <loc> <site>)))

;*---------------------------------------------------------------------*/
;*    let->ast ...                                                     */
;*---------------------------------------------------------------------*/
(define (let->ast exp stack loc site)
   (match-case exp
      ((?- () . ?-)
       ;; we don't remove explicit user let.
       (let ((loc  (find-location/loc exp loc))
	     (body (normalize-progn (cddr exp))))
	  (ast-let-var loc
		       #f
		       #f
		       '()
		       (sexp->ast body
				  stack
				  #f
				  loc
				  site)
		       #f)))
      ((?- ?bindings . ?-)
       (let ((loc (find-location/loc exp loc)))
          (if (or (not (or (pair? bindings)
                           (null? bindings)))
                  (let loop ((bindings bindings))
                     (if (null? bindings)
                         #f
                         (let ((binding (car bindings)))
                            (match-case binding
                               ((?- ?-)
                                (loop (cdr bindings)))
                               (else
                                #t))))))
              (user-error/location loc
                                   (current-function)
                                   (string-append "Illegal form `"
                                                  (symbol->string (car exp))
                                                  "' form")
                                   exp
                                   ''())
              (make-smart-generic-let (car exp)
                                      (make-generic-let exp stack loc site)))))
      (else
       (user-error/location (find-location/loc exp loc)
                            (current-function)
                            (string-append "Illegal form `"
                                           (symbol->string (car exp))
                                           "' form")
                            exp
                            ''()))))

;*---------------------------------------------------------------------*/
;*    make-generic-let ...                                             */
;*---------------------------------------------------------------------*/
(define (make-generic-let exp stack loc site)
   (let* ((bindings   (cadr exp))
	  (body       (normalize-progn (cddr exp)))
	  (frame      (map (lambda (binding)
			      (let* ((var  (parse-formal-ident (car binding)))
				     (name (car var))
				     (type (cdr var)))
				 (make-local-variable name
						      (find-type type))))
			   bindings))
	  (new-stack  (append frame stack)))
      (let ((body     (sexp->ast body
				 new-stack
				 #f
				 loc
				 'read))
	    (bindings (map (lambda (binding var)
			      (let ((val (sexp->ast (normalize-progn
						     (cdr binding))
						    (if (eq? (car exp) 'let)
							stack
							new-stack)
						    #f
						    (find-location/loc binding
								       loc)
						    'read)))
				 (cons var val)))
			   bindings
			   frame)))
	 (ast-let-var loc #f #f bindings body #t))))

;*---------------------------------------------------------------------*/
;*    make-smart-generic-let ...                                       */
;*    -------------------------------------------------------------    */
;*    We patch bindings which concerns a function and where the        */
;*    variable is never mutated. These bindings are put alltogether    */
;*    in a labels form.                                                */
;*    -------------------------------------------------------------    */
;*    We try to apply the following transformation:                    */
;*    (let (... (f (labels ((aux args body)) aux)) ...) ...)           */
;*       -->                                                           */
;*    (labels ((f args body)) (let (...) ...))                         */
;*---------------------------------------------------------------------*/
(define (make-smart-generic-let let/letrec ast-let)
   (let loop ((bindings (let-var-bindings ast-let))
	      (fun      '())
	      (value    '()))
      (trace init "make-smart-generic-let: " (shape bindings) #\Newline)
      (if (null? bindings)
	  (cond
	     ((null? fun)
	      (let-or-letrec let/letrec ast-let))
	     ((null? value)
	      (let->labels fun (let-var-body ast-let)))
	     (else
	      ;; first we ajust let-var bindings
	      (let-var-bindings-set! ast-let value)
	      ;; then, we send the let form to the `let-or-letrec' function
	      (let ((let (let-or-letrec let/letrec ast-let)))
		 (let-var-body-set! let (let->labels fun (let-var-body let)))
		 let)))
	  (let* ((binding (car bindings))
		 (var     (car binding))
		 (sexp    (cdr binding)))
	     (ast-case sexp
		((let-fun)
		 (let* ((locals (let-fun-locals sexp))
			(body   (let-fun-body   sexp)))
		    (if (or (null? locals) (not (null? (cdr locals))))
			;; several functions are introduced by the let-fun
			;; construction or, the body of the construction
			;; include several forms. We skip ...
			(loop (cdr bindings)
			      fun
			      (cons (car bindings) value))
			(ast-case body
			   ((var)
			    (let ((res (var-variable body))
				  (aux (car locals)))
			       (if (or (not (eq? res aux))
				       ;; the result of the labels
				       ;; construction is not the
				       ;; introduced variable.
				       (eq? (local-access var) 'write))
				   ;; the variable is mutated
				   ;; we skip
				   (loop (cdr bindings)
					 fun
					 (cons (car bindings) value))
				   (begin
				      ;; yes, we have found one
				      (trace init "let->labels: "
					     (shape var)
					     " ["
					     (shape (local-type var))
					     "]" #\Newline)
				      (loop (cdr bindings)
					    (cons (car bindings) fun)
					    value)))))
			   (else
			    (loop (cdr bindings)
				  fun
				  (cons (car bindings) value)))))))
		(else
		 (loop (cdr bindings)
		       fun
		       (cons (car bindings) value))))))))
	      
;*---------------------------------------------------------------------*/
;*    let-or-letrec ...                                                */
;*    -------------------------------------------------------------    */
;*    Let differ from letrec in the sens that in a letrec form all     */
;*    bindings must be introduces by the unspecified value and         */
;*    it must exists an initialization stage which initialize all      */
;*    introduced local variables. This means that in a letrec form     */
;*    all variable have to be bound to unspecified then, they have     */
;*    to be mutated to their correct values.                           */
;*---------------------------------------------------------------------*/
(define (let-or-letrec let/letrec ast-let)
   (if (eq? let/letrec 'let)
       ast-let
       (let* ((bindings (let-var-bindings ast-let))
	      (body     (let-var-body ast-let))
	      (seq      (ast-case body
			   ((sequence)
			    body)
			   (else
			    (ast-sequence (ast-location body)
					  #f
					  #f
					  (list body))))))
	  (let-var-body-set! ast-let seq)
	  (let loop ((bindings  bindings)
		     (nsequence (sequence-exp seq)))
	     (if (null? bindings)
		 (begin 
		    (sequence-exp-set! seq nsequence)
		    ast-let)
		 (let* ((binding (car bindings))
			(var     (car binding))
			(val     (cdr binding)))
		    (let ((init (ast-setq (ast-location val)
					  #f
					  #f
					  (ast-var #f #f #f var)
					  val)))
		       (local-access-set! var 'write)
		       (local-occurrence-set! var
					      (+fx 1 (local-occurrence var)))
		       (set-cdr! binding (sexp->ast #unspecified
						    '()
						    #f
						    (ast-location val)
						    'read))
		       (loop (cdr bindings)
			     (cons init nsequence)))))))))
 
;*---------------------------------------------------------------------*/
;*    let->labels ...                                                  */
;*    -------------------------------------------------------------    */
;*    This function create a `labels' construction for variables       */
;*    introduced in a `let' form which are never mutated and bound     */
;*    to functions.                                                    */
;*---------------------------------------------------------------------*/
(define (let->labels value-bindings ast)
   (let loop ((vbindings value-bindings)
	      (fbindings '()))
      (trace init "let->labels: " (shape vbindings) #\Newline)
      (if (null? vbindings)
	  (ast-let-fun (ast-location ast) #f #f fbindings ast)
	  (let* ((binding (car vbindings))
		 (var     (car binding))
		 (val     (cdr binding)))
	     (let* ((aux (car (let-fun-locals val))))
		;; we transfer some informations from the variable
		;; `aux' to `var'.
		(local-value-set! var (local-value aux))
		(local-occurrence-set! var
				       (+fx (local-occurrence var)
					    (local-occurrence aux)))
		(local-type-set!      var (find-type 'procedure))
		;; we style have to alpha-convert the body of `var'
		(let* ((fun  (local-value var))
		       (body (function-body fun)))
		   (function-body-set! fun
				       (substitute! (list (cons aux var))
                                                    body)))
		;; ok, it is finished, we loop now.
		(loop (cdr vbindings) (cons var fbindings)))))))
