;*---------------------------------------------------------------------*/
;*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime1.9/Ast/sexp.scm             */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Dec 30 13:45:27 1994                          */
;*    Last change :  Thu Apr  4 09:00:26 1996 (serrano)                */
;*    -------------------------------------------------------------    */
;*    We build and `ast' from a `sexp'                                 */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module ast_sexp
   (include "Ast/node.sch"
	    "Tools/trace.sch")
   (import  tools_error
	    tools_shape
	    tools_misc
	    tools_location
	    tools_progn
	    tools_module
	    engine_param
	    parse_definition
	    type_env
	    type_cache
	    ast_env
	    ast_global
            ast_local
	    ast_let
	    ast_labels
	    ast_exit)
   (export  (sexp->ast     <sexp> <stack> <sexp> <loc> <site>)
	    (use-variable! <var>  <loc> <site>)))
   
;*---------------------------------------------------------------------*/
;*    sexp->ast ...                                                    */
;*    -------------------------------------------------------------    */
;*    `exp' is the expression to compile.                              */
;*    `stack' is the lexical environment                               */
;*    `entering' is a information for assertions                       */
;*    `loc' is the current file-position                               */
;*---------------------------------------------------------------------*/
(define (sexp->ast exp stack entering loc site)
   (trace (init loop) "sexp->ast: " exp #\Newline)
   (match-case exp
;*--- () --------------------------------------------------------------*/
      (()
       (user-error/location loc
                            (current-function)
                            "Illegal `()' expression"
                            '()
                            ''()))
;*--- atom ------------------------------------------------------------*/
      ((atom ?atom)
       (cond
	  ((not (symbol? atom))
	   (ast-atom loc #t #f atom))
	  ((lookup atom stack)
	   (let* ((local (lookup atom stack))
		  (var   (ast-var loc #f #f local)))
	      (use-variable! local loc site)
	      var))
	  (else
	   (let ((global (find-global atom)))
	      (cond
		 ((not (global? global))
		  (user-error/location loc
				       (current-function)
				       "Unbound variable"
				       exp
				       (sexp->ast ''()
						  stack
						  entering
						  loc
						  site)))
		 ((eq? (global-import global) 'eval)
		  (sexp->ast `(eval ',atom)
			     stack
			     entering
			     loc
			     site))
		 (else
		  (use-variable! global loc site)
		  (ast-var loc #f #f global)))))))
;*--- qualified global variable ---------------------------------------*/
      ((@ . ?-)
       (let ((loc (find-location/loc exp loc)))
	  (match-case exp
	     ((@ (and (? symbol?) ?name) (and (? symbol?) ?module))
	      (let ((global (find-global name module)))
		 (cond
		    ((not (global? global))
		     (user-error/location loc
					  (current-function)
					  "Unbound variable"
					  exp
					  (sexp->ast ''()
						     stack
						     entering
						     loc
						     site)))
		    ((eq? (global-import global) 'eval)
		     (sexp->ast `(eval ,atom)
				stack
				entering
				loc
				site))
		    (else
		     (use-variable! global loc site)
		     (ast-var loc #f #f global)))))
	     (else
	      (user-error/location loc
                                   (current-function)
				   "Illegal `@' expression"
				   exp
                                   (sexp->ast ''()
						  stack
						  entering
						  loc
						  site))))))
;*--- quote -----------------------------------------------------------*/
      ((quote . ?-)
       (match-case exp
          ((?- ?value)
	   (cond
	      ((null? value)
	       (ast-atom (find-location/loc exp loc) #f #f value))
	      ((or (pair? value)
		   (vector? value)
		   (struct? value)
		   (symbol? value))
	       (ast-kwote (find-location/loc exp loc) #f #f value #f))
	      ((or (integer? value)
		   (real? value)
		   (string? value)
		   (cnst? value)
		   ;; I won't put in the compiler that characters
		   ;; and boolean are constant so I explicitize
		   ;; these tests.
		   (char? value)
		   (boolean? value))
	       (sexp->ast value stack entering loc site))
	      (else
	       (user-error/location (find-location/loc exp loc)
				    (current-function)
				    "Illegal `quote' expression"
				    exp
				    (sexp->ast ''()
					       stack
					       entering
					       loc
					       site)))))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `quote' expression"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- begin -----------------------------------------------------------*/
      ((begin . ?body)
       (let ((body (if (null? body)
		       (list #unspecified)
		       body)))
	  (let* ((loc     (find-location/loc exp loc))
		 (values  (sexp*->ast body stack entering loc site)))
	     (ast-sequence loc #f #f values))))
;*--- if --------------------------------------------------------------*/
      ((if . ?-)
       (match-case exp
          ((?- ?si ?alors ?sinon)
           (match-case si
              ((or (if ?si #f #t)
                   ((kwote not) ?si))
               (let* ((loc       (find-location/loc exp loc))
                      (loc-si    (find-location/loc si loc))
                      (loc-alors (find-location/loc alors loc))
                      (loc-sinon (find-location/loc sinon loc))
                      (si        (sexp->ast si stack #f loc-si 'value))
                      (alors     (sexp->ast alors stack #f loc-alors 'value))
                      (sinon     (sexp->ast sinon stack #f loc-sinon 'value)))
                  (ast-conditional loc #f #f si sinon alors)))
              (else
               (let* ((loc       (find-location/loc exp loc))
                      (loc-si    (find-location/loc si loc))
                      (loc-alors (find-location/loc alors loc))
                      (loc-sinon (find-location/loc sinon loc))
                      (si        (sexp->ast si stack #f loc-si 'value))
                      (alors     (sexp->ast alors stack #f loc-alors 'value))
                      (sinon     (sexp->ast sinon stack #f loc-sinon 'value)))
                  (ast-conditional loc #f #f si alors sinon)))))
          ((?- ?si ?alors)
           (set-cdr! (cddr exp) (list #unspecified))
           (sexp->ast exp stack entering loc site))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `if' form"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- set! ------------------------------------------------------------*/
      ((set! . ?-)
       (match-case exp
          ((?- ?var ?val)
           (let* ((loc     (find-location/loc exp loc))
                  (val-loc (find-location/loc val loc))
                  (var     (let ((ast (sexp->ast var stack #f loc 'set!)))
                              (ast-case ast
                                 ((var)
				  ast)
                                 (else 
                                  (user-error/location
                                   loc
                                   (current-function)
                                   "illegal `set!' expression"
                                   exp
                                   (sexp->ast ''()
                                              stack
                                              entering
                                              loc
                                              site))))))
                  (val     (sexp->ast val stack #f val-loc 'value)))
              (ast-setq loc #f #f var val)))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `set!' form"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- a pattern to improve pattern-matching compilation ---------------*/
      ((((or let letrec labels) ?- ?body) . ?args)
       (let ((let-part (car exp)))
          (set-car! (cddr let-part) `(,body ,@args))
          (sexp->ast let-part stack entering loc site)))
;*--- let & letrec ----------------------------------------------------*/
      (((or let letrec) . ?-)
       (let->ast exp stack loc 'value))
;*--- labels ----------------------------------------------------------*/
      ((labels . ?-)
       (labels->ast exp stack entering loc 'value))
;*--- the direct lambda applications (see match-case ...) -------------*/
      (((lambda ?vars . ?body) . ?args)
       (let ((loc (find-location/loc exp loc)))
	  (sexp->ast `(let ,(let loop ((vars vars)
				       (args args))
			       (cond
				  ((null? vars)
				   (if (null? args)
				       '()
				       (user-error/location
					loc
					(current-function)
					"wrong number of argument"
					exp)))
				  ((not (pair? vars))
				   (list
				    (list
				     vars
				     (let liip ((args args))
					(if (null? args)
					    ''()
					    `(cons ,(car args)
						   ,(liip (cdr args))))))))
				  (else
				   (cons (list (car vars) (car args))
					 (loop (cdr vars) (cdr args))))))
			 ,@body)
		  stack
		  entering
		  loc
		  site)))
;*--- lambda ----------------------------------------------------------*/
      ((lambda . ?-)
       (match-case exp
          ((?- ?args . ?body)
           (let ((loc (find-location/loc exp loc))
                 (fun (gensym 'lambda)))
              (sexp->ast `(labels ((,fun ,args ,(normalize-progn body))) ,fun)
                         stack
                         entering
                         loc
                         'value)))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `lambda' form"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- pragma ----------------------------------------------------------*/
      ((pragma . ?-)
       (pragma/type->ast *unspec* exp stack entering loc site))
;*--- failure ---------------------------------------------------------*/
      ((failure . ?-)
       (match-case exp
          ((?- ?proc ?msg ?obj)
           (let* ((loc      (find-location/loc exp loc))
                  (loc-proc (find-location/loc proc loc))
                  (loc-msg  (find-location/loc msg loc))
                  (loc-obj  (find-location/loc obj loc))
                  (proc     (sexp->ast proc stack #f loc-proc 'value))
                  (msg      (sexp->ast msg stack #f loc-msg 'value))
                  (obj      (sexp->ast obj stack #f loc-obj 'value)))
              (ast-fail loc #f #f proc msg obj)))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `failure' form"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- case ------------------------------------------------------------*/
      ((case . ?-)
       (match-case exp
          ((?- ?test . ?clauses)
           (let* ((loc  (find-location/loc exp loc))
                  (test (sexp->ast test
                                   stack
                                   #f
                                   (find-location/loc test loc)
                                   'read)))
	      (let loop ((cls clauses))
		 (if (null? cls)
		     (ast-switch loc #f #f test clauses)
		     (let ((clause (car cls)))
			;; we build the ast for the then part of the clause
			(set-cdr! clause
				  (sexp->ast (normalize-progn (cdr clause))
					     stack
					     #f
					     (find-location/loc clause loc)
					     'read))
			;; we check that it is not an illegal `else' clause
			(if (and (eq? (car clause) 'else)
				 (not (null? (cdr cls))))
			    (user-error/location (find-location/loc exp loc)
						 (current-function)
						 "Illegal `case' form"
						 exp
						 (sexp->ast ''()
							    stack
							    entering
							    loc
							    site))
			    (loop (cdr cls))))))))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `case' form"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- set-exit --------------------------------------------------------*/
      ((set-exit . ?-)
       (set-exit->ast exp stack entering loc site))
;*--- jump-exit -------------------------------------------------------*/
      ((jump-exit . ?-)
       (jump-exit->ast exp stack entering loc site))
;*--- apply -----------------------------------------------------------*/
      ((apply . ?-)
       (match-case exp
          ((apply ?proc ?arg)
           (let* ((loc  (find-location/loc exp loc))
                  (proc (sexp->ast proc
                                   stack
                                   #f
                                   (find-location/loc proc loc)
                                   'apply))
                  (arg  (sexp->ast arg
                                   stack
                                   #f
                                   (find-location/loc arg loc)
                                   'read)))
              (ast-app-ly loc #f #f proc arg)))
          (else
           (user-error/location (find-location/loc exp loc)
                                (current-function)
                                "Illegal `apply' form"
                                exp
                                (sexp->ast ''()
                                           stack
                                           entering
                                           loc
                                           site)))))
;*--- app -------------------------------------------------------------*/
      (else
       ;; this expression can be a function call or a typed pragma
       ;; form. We first check to see if it is a pragma. If it is not
       ;; we compile a function call. This check is required by the
       ;; form (pragma::??? ...) (because we can't add a branch in the
       ;; match-case to check the node `pragma::???'.
       (let ((caller (car exp)))
	  (let ((pragma-type (and (symbol? caller)
				  (let ((string (symbol->string caller)))
				     (if (and (>fx (string-length string) 8)
					      (eq? (string->symbol
						    (substring string 0 8))
						   'pragma::))
					 (string->symbol
					  (substring string
						     8
						     (string-length string)))
					 #f)))))
	     (if (symbol? pragma-type)
		 (pragma/type->ast (find-type pragma-type)
				   exp
				   stack
				   entering
				   loc
				   site)
		 (app->ast exp stack entering loc site)))))))

;*---------------------------------------------------------------------*/
;*    app->ast ...                                                     */
;*---------------------------------------------------------------------*/
(define (app->ast exp stack entering loc site)
   (let ((loc (find-location/loc exp loc)))
      (let loop ((args  exp)
		 (nargs '()))
	 (cond
	    ((null? args)
	     (set! nargs (reverse! nargs))
	     (let ((fun (sexp->ast (car nargs)
				   stack
				   #f
				   loc
				   'app)))
		(if (and (not (var? fun))
			 (not (and (atom? fun) (eq? (atom-value fun) '()))))
		    ;; (null? (car nargs)) stand for an error, so 
		    ;; we do not warn a second time.
		    (user-error/location loc
					 (current-function)
					 "Illegal `application' form"
					 exp
					 (sexp->ast ''()
						    stack
						    entering
						    loc
						    site))
		    (ast-app loc
			     #f
			     #f
			     fun
			     (map (lambda (a)
				     (sexp->ast a stack #f loc 'value))
				  (cdr nargs))
			     #f
			     #f
			     #f
			     #f))))
	    ((not (pair? args))
	     (user-error/location loc
				  (current-function)
				  "Illegal `application' form"
				  exp
				  (sexp->ast ''()
					     stack
					     entering
					     loc
					     site)))
	    (else
	     (let ((darg (direct-arg (car args))))
		(cond
		   (darg
		    ;; this is a direct argument (see below) no needs to
		    ;; normalization.
		    (loop (cdr args) (cons darg nargs)))
		   (else
		    ;; we first have to normalize this expression
		    (let ((exp (let loop ((args  exp)
					  (nargs '()))
				  (if (null? args)
				      (reverse! nargs)
				      (let ((darg (direct-arg (car args))))
					 (cond
					    (darg
					     (loop (cdr args)
						   (cons darg nargs)))
					    (else
					     (let ((aux (gensym 'aux)))
						`(let ((,aux ,(car args)))
						    ,(loop
						      (cdr args)
						      (cons aux
							    nargs)))))))))))
		       (sexp->ast exp stack entering loc site))))))))))

;*---------------------------------------------------------------------*/
;*    direct-arg ...                                                   */
;*    -------------------------------------------------------------    */
;*    This function checks if an argument (or the function) has to     */
;*    be putted in a let construction or if it can be used directly.   */
;*    If the value can be used directly, the value to be used is       */
;*    returned.                                                        */
;*---------------------------------------------------------------------*/
(define (direct-arg arg)
   (match-case arg
      ((atom ?val)
       (if (not (symbol? val))
	   val
	   (let ((global (find-global val)))
	      (cond
		 ((not (global? global))
		  val)
		 ((eq? (global-import global) 'eval)
		  #f)
		 (else
		  val)))))
      ((and (@ ?name ?module) ?val)
       (let ((global (find-global name module)))
	  (cond
	     ((not (global? global))
	      val)
	     ((eq? (global-class global) 'eval)
	      #f)
	     (else
	      val))))
      ((and (quote ()) ?val)
       val)
      ((and ?val (pragma ?-))
       val)
      ((begin (and ?arg (or (@ ?- ?-) (atom ?-))))
       (direct-arg arg))
      (else
       #f)))
      
;*---------------------------------------------------------------------*/
;*    sexp*->ast ...                                                   */
;*---------------------------------------------------------------------*/
(define (sexp*->ast exp* stack entering loc site)
   (let loop ((exps     exp*)
	      (entering entering)
	      (res      '()))
      (cond
	 ((null? exps)
	  (user-error/location loc
                               (current-function)
			       "Illegal empty sequence"
			       exps
                               (sexp->ast ''()
                                          stack
                                          entering
                                          loc
                                          site)))
	 ((null? (cdr exps))
	  (let ((loc (find-location/loc (car exps) loc)))
	     (reverse! (cons (sexp->ast (car exps)
					stack
					#f
					loc
					site)
			     res))))
	 (else
	  (let ((loc (find-location/loc (car exps) loc)))
	     (loop (cdr exps)
		   #f
		   (cons (sexp->ast (car exps)
				     stack
				     entering
				     loc
				     'value)
			 res)))))))

;*---------------------------------------------------------------------*/
;*    pragma/type->ast ...                                             */
;*---------------------------------------------------------------------*/
(define (pragma/type->ast type exp stack entering loc site)
   (match-case exp
      ((?- (and (? string?) ?string) . ?values)
       (let ((max-index (get-max-index string))
	     (loc       (find-location/loc exp loc)))
	  (if (not (=fx max-index (length values)))
	      (user-error/location (find-location/loc exp loc)
				   (current-function)
				   "Wrong number of arguments in `pragma' form"
				   exp
				   (sexp->ast ''()
					      stack
					      entering
					      loc
					      site))
	      (let loop ((exps values)
			 (asts '()))
		 (if (null? exps)
		     (ast-prag-ma (find-location/loc exp loc)
				  type
				  #f
				  string
				  (reverse! asts))
		     (loop (cdr exps)
			   (cons
			    (sexp->ast (car exps)
				       stack
				       type
				       (find-location/loc (car exps) loc)
				       'value)
			    asts)))))))
      (else
       (user-error/location (find-location/loc exp loc)
			    (current-function)
			    "Illegal `pragma' form"
			    exp
			    (sexp->ast ''()
				       stack
				       entering
				       loc
				       site)))))

;*---------------------------------------------------------------------*/
;*    A 1-line cache lookup machinery                                  */
;*---------------------------------------------------------------------*/
(define *cache-name*  #f)
(define *cache-stack* #f)
(define *cache-res*   #f)

;*---------------------------------------------------------------------*/
;*    lookup ...                                                       */
;*---------------------------------------------------------------------*/
(define (lookup name stack)
   (if (and (eq? name *cache-name*)
	    (eq? stack *cache-stack*))
       *cache-res*
       (begin
	  (set! *cache-name* name)
	  (set! *cache-stack* stack)
	  (let loop ((stack stack))
	     (cond
		((null? stack)
		 (set! *cache-res* #f)
		 #f)
		((eq? (local-name (car stack)) name)
		 (set! *cache-res* (car stack))
		 (car stack))
		(else
		 (loop (cdr stack))))))))
	     
;*---------------------------------------------------------------------*/
;*    use-variable! ...                                                */
;*---------------------------------------------------------------------*/
(define (use-variable! var loc site)
   (trace init "use-variable!: " (shape var) " site: " site
	  (if (global? var)
	      (string-append "  " (symbol->string (global-import var)))
	      "")
	  " occurrence: " (variable-occurrence var)
	  " lib?: " (and (global? var) (global-library? var))
	  #\Newline)
   (if (eq? site 'set!)
       (variable-access-set! var 'write))
   (let ((val (variable-value var)))
      (if (and (eq? site 'set!)
	       (or (function? val)
		   (ffunction? val)
		   (return? val)))
	  (user-error/location loc
			       (current-function)
			       "Illegal mutation"
			       (shape var)
			       (sexp->ast ''()
					  '()
					  #f
					  loc
					  site)))
      (variable-occurrence-set! var (+fx (variable-occurrence var) 1))))

;*---------------------------------------------------------------------*/
;*    get-max-index ...                                                */
;*---------------------------------------------------------------------*/
(define (get-max-index string)
   (let ((parser (regular-grammar ()
		    ((#\$ (+ (>-< #\0 #\9)))
		     (let* ((str (the-string))
			    (len (the-length)))
			(string->number (substring str 1 len))))
		    ((+ (out #\$))
		     (ignore))
		    (else
		     (the-failing-char))))
	 (port   (open-input-string string)))
      (let loop ((exp (read/rp parser port))
		 (max 0))
	 (cond
	    ((eof-object? exp)
	     max)
	    ((char? exp)
	     (loop (read/rp parser port) max))
	    (else
	     (loop (read/rp parser port) (if (>fx exp max) exp max)))))))
