/*---------------------------------------------------------------------*/
/*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \    /  '                              */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome Send them to                                           */
/*        <Manuel.Serrano@inria.fr>                                    */
/*        Manuel Serrano                                               */
/*        INRIA -- Rocquencourt                                        */
/*        Domaine de Voluceau, BP 105                                  */
/*        78153 Le Chesnay Cedex                                       */
/*        France                                                       */
/*---------------------------------------------------------------------*/


/*---------------------------------------------------------------------*/
/*    serrano/prgm/project/bigloo/runtime1.9/Clib/ports.c              */
/*                                                                     */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Thu Jul 23 15:34:53 1992                          */
/*    Last change :  Mon Apr  1 15:27:51 1996 (serrano)                */
/*                                                                     */
/*    La manipulation des ports d'entree et de sortie                  */
/*---------------------------------------------------------------------*/
#include <stdio.h>
#include <errno.h>
#if( !(defined( NeXT ) && defined( mc68000 )) )
#   include <termio.h>
#endif
#if( !defined( sony_news ) && !(defined( NeXT ) && defined( mc68000 )) )
#   include <unistd.h>
#endif
#include <sys/file.h>
/*---------------------------------------------------------------------*/
/*    On definit cette macros pour que le fichier `gc_private.h' ne    */
/*    soit pas inclus. Il faut faire ca car ce fichier declare un      */
/*    prototype de `sbrk' qui n'est pas compatible avec le prototype   */
/*    quelque fois present (sur linux, par exemple) dans `unisys.h'    */
/*---------------------------------------------------------------------*/
#if( defined( i386 ) )
#   define GC_PRIVATE_H
#endif
#include <bigloo.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif

#include <string.h>
/*---------------------------------------------------------------------*/
/*    Des variables globales                                           */
/*---------------------------------------------------------------------*/
obj_t current_output_port, current_input_port, current_error_port;
long default_io_bufsiz;

/*---------------------------------------------------------------------*/
/*    External definitions.                                            */
/*---------------------------------------------------------------------*/
extern obj_t  string_to_bstring();
extern obj_t  string_to_symbol();
extern obj_t  make_real();

/*---------------------------------------------------------------------*/
/*     make_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
make_output_port( name, file )
char *name;
FILE *file;
{
   obj_t new_output_port;

   new_output_port = GC_MALLOC( OUTPUT_PORT_SIZE );
   
   new_output_port->output_port_t.header = MAKE_HEADER( OUTPUT_PORT_TYPE, 0 );
   new_output_port->output_port_t.file   = file;
   new_output_port->output_port_t.name   = name;
   
   return BREF( new_output_port );
}

/*---------------------------------------------------------------------*/
/*    open_output_file ...                                             */
/*---------------------------------------------------------------------*/
obj_t
open_output_file( name )
obj_t name;
{
   FILE *file;
   
   if( !(file = fopen( BSTRING_TO_STRING( name ), "w" )) )
      return BFALSE;

   return make_output_port( BSTRING_TO_STRING( name ), file );
}

/*---------------------------------------------------------------------*/
/*    append_output_file ...                                           */
/*---------------------------------------------------------------------*/
obj_t
append_output_file( name )
obj_t name;
{
   FILE *file;
   
   if( !(file = fopen( BSTRING_TO_STRING( name ), "a+" )) )
      return BFALSE;

   return make_output_port( BSTRING_TO_STRING( name ), file );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_output_string ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_output_string()
{
   char *buffer;
   obj_t port;

   port = GC_MALLOC( OUTPUT_STRING_PORT_SIZE );

   /* on alloue le buffer avec des 0 partout */
   buffer = (char *)(GC_MALLOC( OUTPUT_STRING_PORT_BUFFER_SIZE + 1 ));

   port->output_string_port_t.header = MAKE_HEADER( OUTPUT_STRING_PORT_TYPE,0);
   port->output_string_port_t.buffer = buffer;
   port->output_string_port_t.size   = OUTPUT_STRING_PORT_BUFFER_SIZE;
   port->output_string_port_t.offset = 0;
   
   return BREF( port );
}

/*---------------------------------------------------------------------*/
/*    close_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
close_output_port( port )
obj_t port;
{
   if( OUTPUT_STRING_PORTP( port ) )
   {
      obj_t res;
	   
      res = string_to_bstring( OUTPUT_STRING_PORT( port ).buffer );
      OUTPUT_PORT( port ).file = 0L;

      return res;
   }
   else
   {
      fclose( OUTPUT_PORT( port ).file );
      
      return port;
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_file ...                                              */
/*    -------------------------------------------------------------    */
/*    !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!  */
/*    Quand on ouvre un input_port on est oblige, pour des histoires   */
/*    de coherence de remplir le buffer.                               */
/*---------------------------------------------------------------------*/
obj_t
open_input_file( name, bbufsiz )
obj_t name;
obj_t bbufsiz;
{
   FILE *file;
   obj_t new_input_port;
   long  buflen;
   long  bufsiz = CINT( bbufsiz );
   
   if( !(file = fopen( BSTRING_TO_STRING( name ), "r" )) )
      return BFALSE;
      
   new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

   new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->input_port_t.class    = CLASS_FILE;
   new_input_port->input_port_t.file     = file;
   new_input_port->input_port_t.filepos  = 0L;
   new_input_port->input_port_t.eof      = 0;
   new_input_port->input_port_t.bufsiz   = bufsiz + 1;
   new_input_port->input_port_t.backward = 0L;
   new_input_port->input_port_t.forward  = 0L;
   new_input_port->input_port_t.remember = 0L;
   new_input_port->input_port_t.name     = BSTRING_TO_STRING( name );
   new_input_port->input_port_t.annexe   = 0L;
   new_input_port->input_port_t.anxsiz   = 0L;
	
/*--- On remplis le buffer --------------------------------------------*/
   buflen = fread( BUFFER( BREF( new_input_port ) ),
                   1,
                   bufsiz,
                   file );

/*--- A-t-on lu la fin du fichier ? -----------------------------------*/
   if( buflen < bufsiz )
      new_input_port->input_port_t.eof = 1;

/*--- On n'oublie pas de placer le `\0' de fin de buffer  -------------*/
   BUFFER( BREF( new_input_port ) )[ buflen ] = '\0';

   return BREF( new_input_port );
}

/*---------------------------------------------------------------------*/
/*    open_input_console ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_input_console()
{
   obj_t new_input_port;
   long  bufsiz = default_io_bufsiz;
   
   new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

   new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->input_port_t.class    = CLASS_CONSOLE;
   new_input_port->input_port_t.file     = stdin;
   new_input_port->input_port_t.filepos  = 0L;
   new_input_port->input_port_t.eof      = 0;
   new_input_port->input_port_t.bufsiz   = default_io_bufsiz + 1;
   new_input_port->input_port_t.backward = default_io_bufsiz;
   new_input_port->input_port_t.forward  = default_io_bufsiz;
   new_input_port->input_port_t.remember = default_io_bufsiz;
   new_input_port->input_port_t.name     = "[stdin]";
   new_input_port->input_port_t.annexe   = 0L;
   new_input_port->input_port_t.anxsiz   = 0L;
   
/*--- On ne remplis pas le buffer, on le vide -------------------------*/
   bzero( BUFFER( BREF( new_input_port ) ), bufsiz );

/*--- On vide le buffer de stdin --------------------------------------*/
   fflush( stdin );
   
   return BREF( new_input_port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    file_to_input_port ...                                           */
/*---------------------------------------------------------------------*/
obj_t
file_to_input_port( file )
FILE *file;
{
   if( file == stdin )
      open_input_console();
   else
   {
      obj_t new_input_port;
      long  bufsiz = default_io_bufsiz;
   
      new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

      new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0);
      new_input_port->input_port_t.class    = CLASS_FILE;
      new_input_port->input_port_t.file     = file;
      new_input_port->input_port_t.filepos  = 0L;
      new_input_port->input_port_t.eof      = 0;
      new_input_port->input_port_t.bufsiz   = default_io_bufsiz + 1;
      new_input_port->input_port_t.backward = default_io_bufsiz;
      new_input_port->input_port_t.forward  = default_io_bufsiz;
      new_input_port->input_port_t.remember = default_io_bufsiz;
      new_input_port->input_port_t.name     = "[file]";
      new_input_port->input_port_t.annexe   = 0L;
      new_input_port->input_port_t.anxsiz   = 0L;
   
/*--- On ne remplis pas le buffer, on le vide -------------------------*/
      bzero( BUFFER( BREF( new_input_port ) ), bufsiz );

/*--- On vide le buffer de file ---------------------------------------*/
      fflush( file );
      
      return BREF( new_input_port );
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_string ...                                            */
/*---------------------------------------------------------------------*/
obj_t
open_input_string( string )
obj_t string;
{
   obj_t new_input_port;
   long  bufsiz = STRING_LENGTH( string );

   new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

   new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->input_port_t.class    = CLASS_STRING;
   new_input_port->input_port_t.file     = 0L;
   new_input_port->input_port_t.filepos  = 0L;
   new_input_port->input_port_t.eof      = 0;
   new_input_port->input_port_t.bufsiz   = bufsiz + 1;
   new_input_port->input_port_t.backward = 0L;
   new_input_port->input_port_t.forward  = 0L;
   new_input_port->input_port_t.remember = 0L;
   new_input_port->input_port_t.name     = "[string]";
   new_input_port->input_port_t.annexe   = 0L;
   new_input_port->input_port_t.anxsiz   = 0L;
	
/*--- On remplis le buffer --------------------------------------------*/
   strcpy( BUFFER( BREF( new_input_port ) ), BSTRING_TO_STRING( string ) );
	
/*--- On a lu la fin du fichier (bien sur) ! --------------------------*/
   new_input_port->input_port_t.eof = 1;

   return BREF( new_input_port );   
}

/*---------------------------------------------------------------------*/
/*    input_port_debug ...                                             */
/*---------------------------------------------------------------------*/
obj_t
input_port_debug( s, f )
obj_t s , f;
{
   FILE *file = OUTPUT_PORT( f ).file;
   
   fprintf( file, "====== input_port_debug ===============================\n" );
   fprintf( file, "%s:\n\n", INPUT_PORT( s ).name );
   fprintf( file, "   backward   : %d '%c'\n",
            INPUT_PORT( s ).backward,
            BUFFER( s )[ INPUT_PORT( s ).backward ]);
   fprintf( file, "   forward    : %d '%c'\n",
            INPUT_PORT( s ).forward,
            BUFFER( s )[ INPUT_PORT( s ).forward ]);
   fprintf( file, "   remember   : %d '%c'\n",
            INPUT_PORT( s ).remember,
            BUFFER( s )[ INPUT_PORT( s ).remember ]);
   fprintf( file, "   mark       : %d '%c'\n",
            INPUT_PORT( s ).mark,
            BUFFER( s )[ INPUT_PORT( s ).mark ]);
   fprintf( file, "   eof        : %d\n", INPUT_PORT( s ).eof );
   fprintf( file, "   buffer     : [%s]\n", BUFFER( s ) );
   fprintf( file, "===================================================\n" );

   return s;
}

/*---------------------------------------------------------------------*/
/*    close_input_port ...                                             */
/*---------------------------------------------------------------------*/
obj_t
close_input_port( port )
obj_t port;
{
   if( (INPUT_PORTP( port )) && (INPUT_PORT( port ).class == CLASS_FILE) )
      fclose( INPUT_PORT( port ).file );

   return port;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    buffer_to_annexe ...                                             */
/*    -------------------------------------------------------------    */
/*    On deplace le buffer dans l'annexe. Deux cas peuvent se          */
/*    presenter:                                                       */
/*       - l'annexe n'existe pas, on la creer de la taille du          */
/*         buffer                                                      */
/*       - l'annexe existe, on la fait grandir.                        */
/*---------------------------------------------------------------------*/
void
buffer_to_annexe( port )
obj_t port;
{
   if( !INPUT_PORT( port ).annexe )
   /* on est dans le premier cas, l'annexe n'existe pas */
   {
      char *annexe;
      long  size = INPUT_PORT( port ).bufsiz - 1;
      
      annexe = malloc( size );
      
      memcpy( annexe, BUFFER( port ), size );
      
      INPUT_PORT( port ).anxsiz = size;
      INPUT_PORT( port ).annexe = annexe;
   }
   else
   /* on est dans le deuxieme cas, l'annexe existe deja */
   {
      char *annexe;
      long  size = INPUT_PORT( port ).bufsiz + INPUT_PORT( port ).anxsiz - 1 ;
      
      free( INPUT_PORT( port ).annexe );
      
      annexe = malloc( size );
      
      memcpy( annexe,
	      INPUT_PORT( port ).annexe,
	      INPUT_PORT( port ).anxsiz );
      memcpy( annexe + INPUT_PORT( port ).anxsiz,
	      BUFFER( port ),
	      INPUT_PORT( port ).bufsiz );
      
      INPUT_PORT( port ).anxsiz = size;
      INPUT_PORT( port ).annexe = annexe;
   }
}

/*---------------------------------------------------------------------*/
/*    input_port_fill_buffer ...                                       */
/*    -------------------------------------------------------------    */
/*    Si on a lu un end-of-file, on ne peut rien lire de plus.         */
/*    Idem si le buffer est deja plein : runner == 0                   */
/*    -------------------------------------------------------------    */
/*    S'il est encore possible de lire des choses on commence par      */
/*    reajuster le buffer. C'est a dire le recaler a gauche.           */
/*    -------------------------------------------------------------    */
/*    Quand on tompe sur un end-of-file, on n'a pas besoin de          */
/*    mettre un '\0' dans le buffer car C s'en charge.                 */
/*    -------------------------------------------------------------    */
/*    Voici l'etat du buffer avant de lire quoi que ce soit:           */
/*       +-------------------+-+----------------+-+------------+-+     */
/*       |                   | |                | |            |0|     */
/*       +-------------------+-+----------------+-+------------+-+     */
/*       0               backward              forward      bufsiz     */
/*                                                                     */
/*    Apres le rewind l'etat devient:                                  */
/*       +-+-----------------+-+---------------------------------+     */
/*       | |                 | |           |0|?|?|?|?|?|?|?|?|?|?|     */
/*       +-+-----------------+-+---------------------------------+     */
/*     0 = backward        forward                          bufsiz     */
/*                                                                     */
/*    Et pour finir on a:                                              */
/*       +-+-----------------+-+---------------------------------+     */
/*       | |                 | |                               |0|     */
/*       +-+-----------------+-+---------------------------------+     */
/*     0 = backward        forward                          bufsiz     */
/*---------------------------------------------------------------------*/
bool_t
input_port_fill_buffer( port )
obj_t port;
{
   long backward = INPUT_PORT( port ).backward;

/*--- du debug --------------------------------------------------------*/
#ifdef DEBUG_PORT
   puts( "Avant le fill_buffer: " );
   input_port_debug( port, current_output_port );
#endif
	
/*--- le buffer est plein, on deplace les car lus dans l'annexe -------*/
   if( !INPUT_PORT( port ).eof && !backward )
   {
      if( (INPUT_PORT( port ).class != CLASS_CONSOLE) )
      {
#ifdef DEBUG_PORT
	 puts( "Le buffer est plein" );
#endif 		
	 /* on copie le buffer dans l'annexe */
	 buffer_to_annexe( port );

	 /* on met les marks dans un etat coherent */
	 backward = INPUT_PORT( port ).bufsiz - 1;
	 INPUT_PORT( port ).backward = backward;
      }
      else
      {
	 if( INPUT_PORT( port ).forward == INPUT_PORT( port ).bufsiz )
	 {
	    the_failure( string_to_bstring( "input_port_fill_buffer" ),
		         string_to_bstring( "token too large " ),
		         string_to_bstring( "on console" ) );
	    exit( -1 );
	 }
      }
   }

/*---------------------------------------------------------------------*/
/*    On a un traitement specifique a faire suivant qu'on est en       */
/*    train de lire sur la console ou dans un fichier                  */
/*---------------------------------------------------------------------*/
   if( INPUT_PORT( port ).class == CLASS_FILE )
   {
/*--- On a lu un `fin de fichier', il faut donc s'arretter ------------*/
      if( INPUT_PORT( port ).eof )
         return 0;
      else
      {
         char *buffer  = BUFFER( port );
         long  bufsiz  = INPUT_PORT( port ).bufsiz;
	 long  forward = INPUT_PORT( port ).forward;
         long  buflen;

/*--- On recale le buffer a gauche ------------------------------------*/
	 strcpy( buffer, buffer + backward );

/*--- Ou commence et ou termine le buffer actuel ?  -------------------*/
         buflen = bufsiz - backward - 1;

/*--- On lit pour remplir le buffer -----------------------------------*/
         buflen += fread( buffer + buflen, 1, bufsiz - buflen - 1,
                          INPUT_PORT( port ).file );
			
/*--- A-t-on lu la fin du fichier ? -----------------------------------*/
         if( buflen < (bufsiz - 1) )
            INPUT_PORT( port ).eof = 1;

/*--- On n'oublie pas de placer le `\0' de fin de buffer  -------------*/
         buffer[ buflen ] = '\0';
      }
   }
   else
      if( INPUT_PORT( port ).class == CLASS_CONSOLE )
      {
         char *buffer  = BUFFER( port );
         long  bufsiz  = INPUT_PORT( port ).bufsiz;
	 long  forward = INPUT_PORT( port ).forward;
         long  buflen;

         if( INPUT_PORT( port ).eof )
	    return 0;
         else
         {
/*--- On recale le buffer a gauche ------------------------------------*/
            strcpy( buffer, buffer + backward );
            
            buflen = forward - backward - 1;
            
            fgets( buffer + buflen,
                   bufsiz - buflen - 1,
                   INPUT_PORT( port ).file );
            
            if( feof( INPUT_PORT( port ).file ) )
	    {
	       INPUT_PORT( port ).eof = 1;

               if( buffer[ buflen ] == '\0' )
	       {
		  INPUT_PORT( port ).remember = buflen;
		  
		  return 0;
	       }
	    }
         }
      }
      else
         return 0;
         
/*--- Et ziou, c'est fini... ------------------------------------------*/
   INPUT_PORT( port ).forward  = INPUT_PORT( port ).forward -
                                 INPUT_PORT( port ).backward - 1;
   INPUT_PORT( port ).remember = INPUT_PORT( port ).remember -
                                 INPUT_PORT( port ).backward;
      
   INPUT_PORT( port ).backward = 0L;
      
/*--- du debug --------------------------------------------------------*/
#ifdef DEBUG_PORT
   puts( "Apres le fill_buffer: " );
   input_port_debug( port, current_output_port );
#endif      

   return 1;
}

/*---------------------------------------------------------------------*/
/*    obj                                                              */
/*    reset_console ...                                                */
/*    -------------------------------------------------------------    */
/*    On purge les ports d'entree pour que le ^C marche bien sous      */
/*    l'interprete. Le seul entroit ou cette fonction est utilisee     */
/*    la fonction `repl' (voir Eval/eval.scm).                         */
/*---------------------------------------------------------------------*/
obj_t
reset_console( port )
obj_t port;
{
   long size = INPUT_PORT( port ).bufsiz - 1;

   if( INPUT_PORT( port ).class == CLASS_CONSOLE )
   {
      INPUT_PORT( port ).backward = size;
      INPUT_PORT( port ).forward  = size;
      INPUT_PORT( port ).remember = size;
      bzero( BUFFER( port ) , size );
      fflush( stdin );
   }

   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_string ...                                        */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine `Bigloo' qui est extraite    */
/*    du buffer. La chaine commence en `mark' et fini en `backward'.   */
/*    Pour pouvoir faire un strcpy, on efface momentanement le char    */
/*    suivant `backward', on est donc obliger de le backuper.          */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_string( s )
obj_t s;
{
   char  bck;
   obj_t res;

   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

/* printf( "get_small_string:  mark: %d\n", INPUT_PORT( s ).mark );    */
/* printf( "                 annexe: %s\n", INPUT_PORT( s ).annexe );  */
/* printf( "                 buffer: %s\n", BUFFER( s ) );  */
	
   if( INPUT_PORT( s ).annexe )
   {
      char *aux;

      aux = malloc( INPUT_PORT_GET_LENGTH( s ) );

      memcpy( aux, INPUT_PORT( s ).annexe, INPUT_PORT( s ).anxsiz );
      memcpy( &(aux[ INPUT_PORT( s ).anxsiz ]),
	      BUFFER( s ),
	      INPUT_PORT( s ).backward );
				       
      res = string_to_bstring( aux );

      free( aux );
      free( INPUT_PORT( s ).annexe );
      
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
   else
      res = string_to_bstring( &BUFFER( s )[ INPUT_PORT( s ).mark ] );
	
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_small_string ...                                  */
/*    -------------------------------------------------------------    */
/*    Cette fonction n'existe que pour le lecteur `Bigloo'. Quand      */
/*    le lecteur lit une chaine "...", il veut pouvoir acces tres      */
/*    facilement au coprs de la chaine, sans les guillemets. Cette     */
/*    fonction se charge de ce travail.                                */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine `Bigloo' qui est extraite    */
/*    du buffer. La chaine commence en `mark' + 1 et fini en           */
/*    `backward' - 1.                                                  */
/*    Pour pouvoir faire un strcpy, on efface momentanement le char    */
/*    suivant `backward', on est donc obliger de le backuper.          */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_small_string( s )
obj_t s; 
{
   char  bck;
   obj_t res;
   
   bck = BUFFER( s )[ INPUT_PORT( s ).backward - 1 ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward - 1 ] = '\0';

/* printf( "get_small_string:  mark: %d\n", ( INPUT_PORT( s ).mark ) );*/
/* printf( "                 annexe: %s\n", INPUT_PORT( s ).annexe );  */
/* printf( "                 buffer: %s\n", BUFFER( s ) );  */
	
   if( INPUT_PORT( s ).annexe )
   {
      char *aux;

      aux = malloc( INPUT_PORT_GET_LENGTH( s ) );
      
      memcpy( aux,
	      &INPUT_PORT( s ).annexe[ 1 ],
	      INPUT_PORT( s ).anxsiz - 1 );
      memcpy( &(aux[ INPUT_PORT( s ).anxsiz - 1 ]),
	      BUFFER( s ),
	      INPUT_PORT( s ).backward - 1 );
				       
      res = string_to_bstring( aux );

      free( INPUT_PORT( s ).annexe );
      free( aux );
      
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
   else
      res = string_to_bstring(&BUFFER( s )[ INPUT_PORT( s ).mark + 1 ]);
	
   BUFFER( s )[ INPUT_PORT( s ).backward - 1 ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_symbol ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_symbol( s )
obj_t s;
{
   char  bck;
   obj_t res;
   char  *buf;
	
   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      buf = malloc( INPUT_PORT_GET_LENGTH( s ) );
      
      memcpy( buf, INPUT_PORT( s ).annexe, INPUT_PORT( s ).anxsiz );
      memcpy( &(buf[ INPUT_PORT( s ).anxsiz ]),
	      BUFFER( s ),
	      INPUT_PORT( s ).backward );
   }
   else
      buf = &BUFFER( s )[ INPUT_PORT( s ).mark ];
		
/*---------------------------------------------------------------------*/
/*    Veut-on des symbols `case unsensitive' ?                         */
/*---------------------------------------------------------------------*/
#define SYMBOL_UNCASE_SENSITIVE 1

#if defined( SYMBOL_UNCASE_SENSITIVE )
   {
      char *walk;

      for( walk = buf; *walk; walk++ )
         if( isascii( *walk ) )
            *walk = toupper( *walk );
   }
#endif   

   res = string_to_symbol( buf );
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   if( INPUT_PORT( s ).annexe )
   {
      free( buf );
      free( INPUT_PORT( s ).annexe );
		
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
		
   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_fixnum ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_fixnum( s )
obj_t s;
{
   char bck;
   long res;

   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      FAILURE( string_to_bstring( "get-flonum" ),
	       string_to_bstring( "Token too large" ),
	       BUNSPEC );
   }
   else
      res = atol( &BUFFER( s )[ INPUT_PORT( s ).mark ] );
	
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   return BINT( res );
}

/*---------------------------------------------------------------------*/
/*    input_port_get_flonum ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_flonum( s )
obj_t s;
{
   char  bck;
   obj_t res;

   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      FAILURE( string_to_bstring( "get-flonum" ),
	       string_to_bstring( "Token too large" ),
	       BUNSPEC );
   }
   else
      res = make_real( strtod( &BUFFER( s )[ INPUT_PORT( s ).mark ], 0 ) );
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    intput_port_read_string ...                                      */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne une chaine fraichement allouee           */
/*    -------------------------------------------------------------    */
/*    Cette fonction fonctionne aussi bien pour les `consoles' que pour*/
/*    les fichiers.                                                    */
/*---------------------------------------------------------------------*/
obj_t
intput_port_read_string( port, number )
obj_t port;
long  number;
{
   obj_t res;
   long  len;
   char  aux;
   char *buffer = BUFFER( port );

   /* On verifie que la taille de ce qu'on veut lire n'est pas plus grande */
   /* que la taille des buffers du port.                                   */
   len = INPUT_PORT( port ).bufsiz > number ?
             number : INPUT_PORT( port ).bufsiz;

   /* On regarde si on n'a pas deja lu eof sur le port */
   if( INPUT_PORT( port ).eof )
   {
      long buf_len;
      long offset;

      /* Puisqu'on a lu eof, il faut regarder quelle est la taille de la */
      /* chaine contenue dans le buffer.                                 */
      buf_len = (long)strlen( &BUFFER( port )[ INPUT_PORT( port ).backward ] );

      len = buf_len > len ? len : buf_len;

      /* On peut maintenant construire la nouvelle chaine. */
      offset = INPUT_PORT( port ).backward + len;
      
      aux = buffer[ offset ]; 
      buffer[ offset ] = '\0';
      res = string_to_bstring( &BUFFER( port )[ INPUT_PORT( port ).backward ] );
      buffer[ offset ] = aux;
      
      /* on ajuste les curseurs */
      INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + len;
   }
   else
      /* on regarde si on a assez de chose dans le buffer */
      if( (INPUT_PORT( port ).bufsiz - INPUT_PORT( port ).backward) < number )
      {
         long offset;

         /* on calcule la chaine resultat */
         offset = INPUT_PORT( port ).backward + number;
         
         aux = buffer[ offset ];
         buffer[ offset ] = '\0';
         res = string_to_bstring( &BUFFER( port )[ INPUT_PORT( port ).backward ] );
         buffer[ offset ] = aux;
      
         /* on ajuste les curseurs */
         INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + number;
      }
      else
      {
         long len_aux;
         
         /* on commence par remplir le buffer */
         INPUT_PORT( port ).forward = INPUT_PORT( port ).forward + 1;
         input_port_fill_buffer( port );

         /* On regarde la taille de ce qu'on vient de lire */
         if( (len_aux = strlen( buffer )) > len )
         {
            /* on construit la nouvelle chaine */
            aux = buffer[ len ];
            buffer[ len ] = '\0';
            res = string_to_bstring( BUFFER( port ) );
            buffer[ len ] = aux;
      
            /* on ajuste les curseurs */
            INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + len;
         }
         else
         {
            res = string_to_bstring( BUFFER( port ) );
      
            /* on ajuste les curseurs */
            INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + len_aux;
         }   
      }

   INPUT_PORT_REMEMBER_REF( port );
   INPUT_PORT_AJUST_CURSOR( port );

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_display_error ...                                     */
/*---------------------------------------------------------------------*/
obj_t
input_port_display_error( ip, op )
obj_t ip, op;
{
   FILE *output        = OUTPUT_PORT( op ).file;
   long forward        = INPUT_PORT( ip ).forward - 1;
   long end_of_print   = forward;
   long start_of_print = INPUT_PORT( ip ).backward - 40;
   long count;
   char bck;
   
   /* on calcule l'endroit ou on va commencer a afficher */
   if( (long)start_of_print < 0 )
      start_of_print = 0;

   count = end_of_print - start_of_print;

   /* on calcule la chaine et la longueur de ce qu'on va afficher */
   while( (BUFFER( ip )[ end_of_print ] != '\0') &&
          (BUFFER( ip )[ end_of_print ] != '\n') &&
          (count < 80) )
      end_of_print++, count++;

   /* on trippote le buffer */
   bck = BUFFER( ip )[ end_of_print ];
   BUFFER( ip )[ end_of_print ] = '\0';

   /* on affiche la chaine */ 
   fprintf( output, "%s\n", &BUFFER( ip )[ start_of_print ] );
   
   /* on restore le buffer */
   BUFFER( ip )[ end_of_print ] = bck;

   return ip;
}

/*---------------------------------------------------------------------*/
/*     init_io ...                                                     */
/*---------------------------------------------------------------------*/
init_io()
{
#if( !defined( _SBFSIZ ) )
#   define _SBFSIZ 8
#endif

   default_io_bufsiz = BUFSIZ * _SBFSIZ;
   
   current_output_port = make_output_port( "stdout", stdout );
   current_error_port  = make_output_port( "stderr", stderr );
   current_input_port  = open_input_console();
}

/*---------------------------------------------------------------------*/
/*    fexists ...                                                      */
/*---------------------------------------------------------------------*/
bool_t
fexists( name )
char *name;
{
   return !access( name, R_OK );
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    reset_eof ...                                                    */
/*    -------------------------------------------------------------    */
/*    Quand le port pointe sur la console, cette fonction annule la    */
/*    lecture du `^D'. Cela permet une reprise de lecture.             */
/*---------------------------------------------------------------------*/
bool_t
reset_eof( port )
obj_t port;
{
   if( INPUT_PORT( port ).class == CLASS_CONSOLE )
   {
      long bufsiz = INPUT_PORT( port ).bufsiz;
      
      /* on annule le `eof' */
      INPUT_PORT( port ).eof = 0;
      
      /* on nettoie le buffer */
      bzero( (char *)BUFFER( port ), bufsiz );

      /* on recale les curseurs */
      INPUT_PORT( port ).forward  = default_io_bufsiz;
      INPUT_PORT( port ).remember = default_io_bufsiz;
      INPUT_PORT( port ).backward = default_io_bufsiz;

      /* on flush , et on clear */
      fflush( stdin );
      clearerr( stdin );

      return 1;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strport_flush ...                                                */
/*    -------------------------------------------------------------    */
/*    On flush un string port.                                         */
/*---------------------------------------------------------------------*/
obj_t
strport_flush( port )
obj_t port;
{
   obj_t res;

   res = string_to_bstring( OUTPUT_STRING_PORT( port ).buffer );

   OUTPUT_STRING_PORT( port ).offset = 0;
   OUTPUT_STRING_PORT( port ).buffer[ 0 ] = '\0';

   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strport_grow ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
strport_grow( 	p )
obj_t p;
{
   long  old_size, new_size;
   char *old_buffer, *new_buffer;
	
   old_buffer = OUTPUT_STRING_PORT( p ).buffer;
   old_size   = OUTPUT_STRING_PORT( p ).size;

   new_size   = old_size * 2;
   new_buffer = (char *)( GC_MALLOC( new_size + 1 ) );

   strcpy( new_buffer, old_buffer );

   OUTPUT_STRING_PORT( p ).buffer = new_buffer;
   OUTPUT_STRING_PORT( p ).size   = new_size;

   return p;
}






