/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** scan.c
**
** Deliver tokens one at a time
**
**	METHOD:
**	The input arrives via LSLScanFreshToken ().
**	The output is reported via LSLScanNextToken ().
**
**	The tokens are built in module ScanLine.
**	The tokens are delivered from this module.
**	Meantimes, they are saved in a static array.
**
**	The tokenizing is split off from the delivery of tokens
**	to facilitate incremental scanning at a later date.
**	The essential is that scanline () can be called or not
**	if the input text is dirty or not.  Clean lines cause
**	tokens to be played out from the saved token list (not
**	yet implemented in this version).
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "signature.h"
# include "signature2.h"
# include "scan.h"
# include "scanline.h"

# define MAXLINE 1000		

extern /*@external@*/ /*@unused@*/ YYSTYPE lsllval;

static /*@exposed@*/ ltoken LSLScanLookAhead (void);
static tsource *scanFile;	  /* file to scan */
static o_ltoken TokenList[MAXLINE]; /* available tokens */
static int nextToken;		  /* next available token */
static int lastToken;		  /* next available slot */

static /*@dependent@*/ /*@null@*/ char *line;	/* input text */
static unsigned int lineNumber;	/* current line number */

unsigned int
lsllex (YYSTYPE *lval)
{
  /* This is important!  Bison expects this */
  lval->tok = LSLScanNextToken ();
  return (ltoken_getCode (lval->tok));
}

ltoken LSLScanNextToken (void)
{
  DPRINTF (("LSLScanNextToken"));

  if (nextToken < lastToken)
    {	
      /*@-retalias@*/
      return TokenList[nextToken++];
      /*@=retalias@*/
    }
  else
    {
      lastToken = 0;		
      lineNumber++;

      line = tsource_nextLine (scanFile);       
      
      if (line != (char *) 0)
	{
	  DPRINTF (("Scan: %s", line));
	  lscanLine (line);	/* tokenize */
	  nextToken = 0;
	  return LSLScanNextToken ();	
	}
      else
	{
	  return LSLScanEofToken ();
	}
    }
}

static /*@exposed@*/ ltoken
LSLScanLookAhead (void)
{
  if (nextToken < lastToken)
    {		
      return TokenList[nextToken];
    }
  else
    {
      lastToken = 0;		
      line = tsource_nextLine (scanFile);

      if (line != (char *) 0)
	{
	  lscanLine (line);	
	  nextToken = 0;	
	  return LSLScanLookAhead ();	
	}
      else
	{
	  /* 
          ** This is a real memory leak.  Its only a few bytes
	  ** per file though, and lsl files are hardly ever used.
	  */

	  /*@-onlytrans@*/ 
	  return LSLScanEofToken ();
	  /*@=onlytrans@*/
	}
    }
}

void
LSLScanFreshToken (ltoken tok)
{
  if (lastToken < MAXLINE)
    {				
      TokenList[lastToken++] = ltoken_copy (tok);	
    }
  else
    {
      llfatalbug (message ("LSLScanFreshToken: out of range: %s", 
			   cstring_fromChars (lsymbol_toChars (ltoken_getText (tok)))));
    }
}

/*@exposed@*/ tsource *LSLScanSource (void)
{
  return scanFile;
}


void
LSLScanInit (void)
{
}

void
LSLScanReset (tsource *s)
{
  scanFile = s;
  lastToken = 0;
  nextToken = lastToken + 1;	/* force call to scanline   */
  lineNumber = 0;
}

void
LSLScanCleanup (void)
{
}
