#define Copyright         "Copyright 1995  Ed Casas"

#ifdef DEBIAN	/* Ed Casas asked for a Debian specific version string */
#define Version		  "efax v 0.7a (Debian release 07a-6)" 
#else
#define Version		  "efax v 0.7a"
#endif

/*
    Copyright (C) 1995  Ed Casas

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please contact the author if you wish to use efax or efix for
    purposes not covered by the GNU GPL.

    You may contact the author by e-mail at: edc@cce.com, by mail
    at: 2629 West 3rd Ave, Vancouver, BC, Canada, V6K 1M4, or by
    fax at: +1 604 734 5291.

*/
/*
    Modified slightly by Robert LeBlanc <rjl@renaissoft.com> for
    use with the Qfax suite of fax utilities.  The only change was
    to make efax write its terminating status code to the RESULT
    file (as specified in a "config.h" file), so that the spooler
    can access this result at a later time.
*/

const char *Usage =
  "Usage:\n"
/*"  %s [ option ]... [ -r pat | -t num file... ]\n"*/
  "  %s [ option ]... [ -r pat | -t num file... | -p num pat ]\n"
  "Options:\n"
  "  -c cap  set file format or receive capabilites to cap\n"
  "  -d dev  use modem on device dev\n"
  "  -g cmd  exec \"/bin/sh -c cmd\" for data calls\n"
  "  -i str  send modem command ATstr at start\n"
  "  -l id   set local indetification to id\n"
  "  -o opt  use protocol option opt:\n"
/*"      0     use class 2.0 instead of class 2 modem commands\n"*/
  "      1     use class 1 modem commands\n"
  "      a     if first [data mode] answer attempt fails retry as fax\n"
  "      e     ignore errors in modem initialization commands\n"
  "      r     reverse bit order on receive\n"  
  "      x     use XON instead of DC2 to trigger reception\n"
  "      z     add 100 ms to pause before each modem comand (cumulative)\n"
  "  -h hdr  use page header hdr (use %%d's for current page/total pages)\n"
  "  -f fnt  use (PBM) font file fnt for headers\n"
  "  -q ne   ask for retransmission if more than ne errors per page\n"
  "  -s      share (unlock) modem device while waiting for call\n"
  "  -v lvl  print messages of type in string lvl (ewinchamr)\n"
  "  -w      don't answer phone, wait for OK or CONNECT instead\n"
  "  -x fil  use uucp-style lock file fil\n"
  "  -z str  send modem command ATstr when done\n"
  "Commands:\n"
  "  -r      answer and receive fax into files pat.001, pat.002, ... \n"
  "  -t      send fax image files file... to telephone num\n"
  "  -p      poll num for pat.001, ... \n"
  ;

#include <ctype.h>		/* ANSI C */
#include <signal.h>    
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "efaxlib.h"		/* EFAX */
#include "efaxos.h"

#include "config.h"             /* QFAX 1.0 */

/* constants... */

#define FAXFILE "/dev/fax"  /* default fax modem device */

			    /* delays and timeouts (t/o), in deciseconds */
#define TO_RESET  26	    /* t/o for modem reset commands (per Hayes) */
#define T_CMD     1	    /* pause before each modem command */
#define TO_A      1200	    /* dial/answer (Phase A) - modem may t/o first */

#define TO_DATAF  215	    /* software adaptive answer data connect t/o */

#define T1 350		    /* T.30 T1 - waiting for DIS/DCS before Phase B */
#define T2 60		    /* T.30 T2 - waiting for frame in Phase B */
#define T3S 30		    /* T.30 response timeout (not T3) */
#define T4 30		    /* T.30 T4 - between [re]transmissions of DIS */

#define TO_DRAIN 136	    /* minimum buffer drain time (4k/300cps)(tx) */
#define TO_RTCMD 20	    /* return to command mode after DLE-ETX (rx) */
#define TO_FT    31	    /* max delay after +F[TR][MH] command */
#define TO_CHAR  51	    /* per data character (max FILL length) */
#define TO_ABRT  20	    /* max delay after sending abort sequence */

#define TO_C2B    450	    /* Class 2 DIS to CONNECT:(DCS+TCF+CFR)xretries */
#define TO_C2X    20	    /* Class 2 wait for XON: 2/5 of 5s timeout */
#define TO_C2PP   200	    /* Class 2 wait for ppr: (ppm+ppr)x3retries + 2 */
#define TO_C2R    600	    /* Class 2 receive: (TCF+FTT)x11 retrains + 5 */
#define TO_C2EOR  120	    /* Class 2 end of data rx (4 retrans x 3 s) */

#define CMDBUFSIZE 256      /* longest possible modem command or response */
#define DEFDISLEN 3	    /* length of DIS initially transmitted */
#define DEFCAP 1,3,0,2,0,0,0,0	/* default local capabilities */
#define DLE_ETX "\020\003"  /* DLE-ETX (end of data) string */
#define HDRSHFT 54	    /* shift header 6.7mm to image area */
#define HDRSPCE 20	    /* number of scan lines inserted before image */
#define HDRSTRT  4	    /* scan line where header is placed on image */
#define IDLEN 20	    /* length of T.30 identification strings */
#define LOCKPOLLDELAY 15    /* seconds between checks of lock files */
#define MAXDIS 8	    /* maximum DIS frames sent without response (T1) */
#define MAXERRPRT 32	    /* maximum number of reception errors to report */
#define MAXFIFLEN 125	    /* max FIF len = MAXFRLEN - (adx+ctl+FCF) - FCS */
#define MAXFRLEN 130        /* max frame length = 3.45s x 300 bps / 8 */
#define MAXGETTY 512        /* maximum length of ``getty'' (-g) command */
#define MAXICMD  100        /* maximum # of modem setup/reset commands */
#define MAXLKFILE 16	    /* maximum number of lock files */
#define MAXMSGBUF 8192	    /* maximum status/error message bytes held */
#define MAXNULLS 2	    /* maximum consecutive received nulls saved */
#define MAXTRAIN 2	    /* maximum training retries at lowest speed */
#define MAXRESPB 1024	    /* maximum bytes of modem responses saved */
#define MAXRETRY 3	    /* maximum retries of unacknowledged commands */
#define MINWRITE  128       /* minimum bytes before write() to modem */
#define NCAP 8              /* number of fields in a capability string */
#define NTXRETRY  3	    /* maximum re-sends per page */

const char *prompts[] = {		/* modem responses that are prompts */
  "OOK", "-CONNECT FAX", "CCONNECT", "NNO CARRIER", "EERROR",
  "NNO DIALTONE", "BBUSY", "NNO ANSWER", "E+FCERROR", 0 } ;

enum promptcodes {			/* codes for modem prompts */
   BUSY = 'B', CONNECT = 'C', OK = 'O', RING = 'R', NO = 'N',
   ERROR = 'E' } ;

			    /* signals to be caught so can hang up phone */
const int catch [] = { CATCHSIGS, 0 } ;

typedef int cap [ NCAP ] ;		/* remote/local capabilities */

                                        /* capability fields... */
enum  captype {	               VR, BR, WD, LN, DF, EC, BF, ST } ;
const int capmax [ NCAP ] = {   1,  7,  2,  2,  3,  2,  1,  7 } ;
					/* & maximum values */

					/* characters per second for br */
const int cps [ 8 ] = { 300, 600, 900, 1200, 1500, 1800, 900, 1200 } ;

					/* next br = fallback [ br ] */
const int fallback [ 8 ] = { 0, 0, 1, 2, 7, 4, 3, 6 } ;

					/* minimum scan time in ms  */
const int delay [ 8 ] = { 0 , 5, 10, 10, 20, 20, 40, 40 } ;

					/* page width in pixels */
const int pagewidth [ 3 ] = { 1728, 2048, 2432 } ;

/* Table to convert between T.30 DIS/DCS/DTC FIF and Class 2-like
   capability codes. Uses br=6, 7 for V.17 at 7200, 9600. */

typedef const struct t30tabstruct
{ 
  char *name ; 
  u_char byte, shift, mask ; 
  u_char captodis[8], distocap[16], captodcs[8], dcstocap[16] ; 
} t30tabst ;

#define X 0xff				/* invalid values */

t30tabst t30tab [ NCAP ] = {
  { "vr", 1, 1, 0x01, { 0, 1 } , { 0, 1 } , { 0, 1 } , { 0, 1 } },	 
  { "br", 1, 2, 0x0f, 
      { 0, 4, 12, 12, 13, 13 } ,
      { 0, X, X, X, 1, X, X, X, 3, X, X, X, 3, 5, 3, X } ,
      { 0, 4, 12, 8, 5, 1 } ,
      { 0, 5, 5, X, 1, 4, 4, X, 3, 7, X, X, 2, 6, X, X } } ,
  { "wd", 2, 6, 0x03, { 0, 2, 1 } , { 0, 2, 1, 2 } ,
      { 0, 2, 1 } , { 0, 2, 1, 2 } },
  { "ln", 2, 4, 0x03, { 0, 2, 1 } , { 0, 2, 1, X } ,
      { 0, 2, 1 } , { 0, 2, 1, X } },
  { "df", 1, 0, 0x01, { 0, 1 } , { 0, 1 } , { 0, 1 } , { 0, 1 } },
  { "ec", 3, 4, 0x03, { 0, 2, 2 } , { 0, X, 2, X } , 
      { 0, 3, 2 } , { 0, 0, 2, 1 } }, 
  { "bf", 5, 5, 0x01, { 0, 1 } , { 0, 1 } , { 0, 1 } , { 0, 1 } },
  { "st", 2, 1, 0x07, 
      { 7, 4, 3, 2, 6, 0, 5, 1 } , { 5, 7, 3, 2, 1, 6, 4, 0 } ,
      { 7, 4, X, 2, X, 0, X, 1 } , { 5, 7, 3, 1, X, X, X, 0 } } 
} ;

					/* values of capability fields */
const char *capvaluestr [ NCAP ] [8] = {
  { " 98lpi", "196lpi" } , 
  { " 2400bps", " 4800bps", " 7200bps", " 9600bps", "  12kbps", "14.4kbps",
    "7200V.17", "9600V.17" } ,
  { "8.5\"/215mm", " 10\"/255mm", " 12\"/303mm" } ,
  { "11\"/A4", "14\"/B4", " any  " } ,
  { "1D" , "2D" }, { "   -   ", "ECM-256", "ECM-64 " }, { " - ", "BFT" },
  { "0ms", "5ms", "10/5ms", "10ms", "20/10ms", "20ms", "40/20ms", "40ms" }
} ;

/* T.30 control frames */

enum frametype {	    
 DIS=0x01, CSI,	NSF=0x04,
 CFR=0x21, FTT,
 MCF=0x31, RTN, RTP, PIN, PIP,
 DCS=0x41, TSI,	NSS=0x44,
 CRP=0x58, DCN=0x5f,
 EOM=0x71, MPS, EOP=0x074, PRI_EOM=0x79, PRI_MPS, PRI_EOP=0x7c,
 DTC=0x81, CIG, NSC=0x84
 } ;

/* Class 1 commands to [receive=0/transmit=1] [data=0/training=1] for
   [baud rate=BR]. */

const char *c1cmd [ 2 ]  [ 2 ] [ 8 ] = { 
{ { "+FRM=24", "+FRM=48", "+FRM=72", "+FRM=96", "+FRM=122", "+FRM=146" ,
    "+FRM=74", "+FRM=98" } ,
  { "+FRM=24", "+FRM=48", "+FRM=72", "+FRM=96", "+FRM=121", "+FRM=145" ,
    "+FRM=73", "+FRM=97" } } ,
{ { "+FTM=24", "+FTM=48", "+FTM=72", "+FTM=96", "+FTM=122", "+FTM=146" ,
    "+FTM=74", "+FTM=98", } ,
  { "+FTM=24", "+FTM=48", "+FTM=72", "+FTM=96", "+FTM=121", "+FTM=145" ,
    "+FTM=73", "+FTM=97" } }
} ;

/* Modem file & features */

typedef struct modemstruct {
  TFILE *f ;
  int c1, c20 ;			/* use class 1/class 2.0 */
  int cmdpause ;		/* delay before each init command */
  u_char startchar ;		/* character to start reception */
} MODEM ;

/* Functions... */

/* Return name of frame of type 'fr'. */

const char *frname ( int fr )
{
static struct framenamestruct {  int code ;  const char *name ; } 
framenames [] = {
 {DIS,"DIS"},{CSI,"CSI"},{NSF,"NSF"},{CFR,"CFR"},{FTT,"FTT"},{MCF,"MCF"},
 {RTN,"RTN"},{RTP,"RTP"},{PIN,"PIN"},{PIP,"PIP"},{DCS,"DCS"},{TSI,"TSI"},
 {NSS,"NSS"},{CRP,"CRP"},{DCN,"DCN"},{EOM,"EOM"},{MPS,"MPS"},{EOP,"EOP"},
 {PRI_EOM,"PRI-EOM"},{PRI_MPS,"PRI-MPS"},{PRI_EOP,"PRI-EOP"},
 {DTC,"DTC"},{CIG,"CIG"},{NSC,"NSC"}, {0,0} }, *p ;

  for ( p=framenames ; p->code && p->code != fr ; p++ ) ;
  return p->code ? p->name : "UNKNOWN" ;
}


/* Send bytes to the modem, doing bit-reversal and escaping DLEs.  Flushes
   buffer if enough characters stored.  Follow by call to tputs() or
   tflush() when done to flush o/p buffer.  Returns 0 or 2 on errors. */

int sendbuf ( MODEM *m, u_char *p, int n, int t )
{
  int err=0, c=0 ;
  TFILE *f = m->f ;
  u_char *order = m->f->obitorder ;

  while ( n-- > 0 && c >= 0 ) {
    c  = order [ *p++ ] ;
    if ( c == DLE ) c = tputc ( DLE, f, t ) ;
    if ( c >= 0 ) tputc ( c, f, t ) ;
    if ( tobytes ( m->f ) >= MINWRITE ) tflush ( m->f, t ) ;
  }

  if ( c < 0 ) err = msg ( "ES2fax device write error:" ) ;

  return err ;
}


/* Get a modem response into buffer s, storing up to n bytes.  The response
   begins with most recent non-control character and ends with CR/LF.
   Returns s or null if times-out in t deciseconds or on i/o error. Trace
   messages are buffered to reduce possible timing problems. */

char *tgets( MODEM *m, char *s, int n, int t )
{
  int c=0, state=0 ;
  char *p = s ;

  while ( state < 4 ) {
    if ( ( c = tgetc ( m->f, t ) ) == EOF ) break ;
    c &= 0x7f ;
    if ( state == 0 ) msg ( "M-+ [" ) ; 
    msg ( "M-+ %s" , cname ( c ) ) ;
    switch ( state ) {
    case 0 :
    case 1 : state =               ( iscntrl ( c ) ? 1 : (p=s, 2) ) ; break ;
    case 2 : state = c == CR ? 3 : ( iscntrl ( c ) ? 1 :       2  ) ; break ;
    case 3 : state = c == LF ? 4 : ( iscntrl ( c ) ? 1 : (p=s, 2) ) ; break ;
    default: msg ( "Ecan't happen (tgets)" ) ;
    }
    if ( state == 2 && p < s+n-1 ) *p++ = c ;
  }

  *p = 0 ;
  if ( p >= s+n-1 ) msg ( "W- modem response overflow" ) ;
  if ( state ) msg ( "M- %s]" , c == EOF ? "<timeout>" : "" ) ;

  return c == EOF ? 0 : s ;
}


/* Send character or string to modem immediately (for commands).  Return
   like putc() and puts(). */

int tputcnow ( MODEM *m, char c, int t ) 
{ 
  return tputc ( c, m->f, t ) < 0 ? EOF : ( tflush ( m->f, t ) ? EOF : c ) ; 
}

int tputs ( MODEM *m, const char *s, int t )
{
  int n=0 ;
  while ( s && *s && ( n = tputcnow ( m, *s++, t ) ) != EOF ) ;
  return n ;
}


/* Range-check capability. */

int checkcap ( cap c )
{
  int err=0, i ;

  for ( i=0 ; i<NCAP ; i++ )
    if ( c[i] > capmax[i] || c[i] < 0 ) {
      err = msg ( "E3%s = %d out of range, set to 0", t30tab[i].name, c[i] ) ;
      c[i]=0 ;
    }

  return err ;
}


/* Print cap[ability] c using text values and prefix s. */

void printcap ( char *s , cap c )
{
  int i ;
  msg ( "N-+ %s" , s ) ;
  checkcap ( c ) ;
  for ( i=0 ; i<NCAP ; i++ ) 
    msg ( "N-+  %s" , capvaluestr [ i ] [ c[i] ] ) ;
  msg ( "N-" ) ;
}


/* Convert capability string to cap struct. Returns 0 or 2 on errors. */

int str2cap ( char *s, cap c )
{
  int err=0, n ;

  n = sscanf ( s, "%d,%d,%d,%d,%d,%d,%d,%d", 
	      c+0, c+1, c+2, c+3,  c+4, c+5, c+6, c+7 ) ;

  if ( n < NCAP ) msg ( "Wmissing value(s) in \"%s\"", s ) ;

  checkcap ( c ) ;

  return err ;
}


/* Convert a cap[ability] 'c' to a DIS/DCS/DTC FIF 'fif' of 'len' bytes.
   Converts into DIS format if 'isdis' is true, else into DCS/DTC
   format. */

void mkdis ( cap c , u_char *fif , int len , int isdis ) 
{
  int i, k ;
  t30tabst *p ;

  if ( len < 3 || len > 5 ) 
    msg ( "Wstrange DCS/DIS length (%d)" , len ) ;

  fif[0] = 0 ;
  fif[1] = isdis ? 0xc0 : 0x40 ;
  for ( i=2 ; i<len-1 ; i++ ) fif[i] = 0x01 ;       /* add extension bits */
  fif[i] = 0 ;

  checkcap ( c ) ;

  for ( i=0 ; p=t30tab+i, i<NCAP ; i++ ) {
    if ( ( k = ( isdis ? p->captodis : p->captodcs ) [ c [ i ] ] ) == X )
      msg ( "E3mkdis: can't happen (invalid %s)", p->name ), k=0 ;
    if ( p->byte < len ) fif [ p->byte ] |= k << p->shift ;
  }
}


/* Return length of DIS/DTC FIF (counts extension bits). */

int dislen ( u_char *fif )
{
  int n ;
  for ( n=3 ; fif [ n-1 ] & 0x01 && n < MAXFIFLEN ; n++ ) ;
  return n ;
}


/* Convert received DIS/DCS/DTC FIF to cap. Returns 0 or 3 if bad DIS/DCS
   field. */

int mkcap ( u_char *fif, cap c, int dis ) 
{
  int err=0, i, j, k, len ;
  t30tabst *p ;

  len = dislen ( fif ) ;

  for ( i=0 ; i<NCAP ; i++ ) {
    p=t30tab+i ;
    if ( p->byte >= len ) {
      c [ i ] = 0 ;
    } else {
      j = ( fif [ p->byte ] >> p->shift ) & p->mask ;
      k = ( dis ? p->distocap : p->dcstocap ) [ j ] ;
      if ( k == X ) {
	c [ i ] = 0 ;
	err = msg("E3mkcap: bad %s field (%d) set to 0", p->name, j) ;
      } else { 
	c [ i ] = k ;
      }
    }
  }
  return err ;
}


/* Compute compatible local/remote capabilities. Used by the
   sending station only and only for Class 1. Returns 0 if OK or
   3 if no compatible settings possible. */

int mincap ( cap local, cap remote, cap session )
{
  int err=0, i ;
  int msttab[2][8] = { { 0,1,3,3,5,5,7,7 } , { 0,1,1,3,3,5,5,7 } } ;

  printcap ( "local  ", local ) ;
  printcap ( "remote ", remote ) ;
  checkcap ( session ) ;

  for ( i=0 ; i<NCAP && i!=ST ; i++ )
    session[i] = remote[i] < local[i] ? remote[i] : local[i] ;

  session[ST] = msttab [ session[VR] ] [ remote[ST] ] ;

  printcap ( "session", session ) ;

  if ( local[WD] != session[WD] || local[LN] != session[LN] || 
      local[DF] != session[DF] ) 
    err = msg ("W3incompatible local and remote capabilities" ) ;

  return err ;
}


/* Scan responses since giving previous command (by cmd()) for a match to
   string 's' at start of response.  If a matching response is found and ip
   is not null, reads one integer into ip. Returns pointer to start of data
   field of response string or NULL if not found. */

char responses [ MAXRESPB ], *lresponse = 0 ;

char *sresponse ( const char *s, int *ip )
{
  char *p, *r = 0 ;
  int lens ;
  lens = strlen ( s ) ;
  for ( p=responses ; p<lresponse ; p += strlen(p) + 1 )
    if ( ! strncmp ( p, s, lens ) ) {
      r = p + lens ;
      if ( ip ) sscanf ( r, "%d", ip ) ;
    }
  return r ;
}


/* Search for a match to the string s in a null-terminated array of
   possible prefix strings pointed to by p.  The first character of each
   prefix string is skipped.  Returns pointer to the table entry or NULL if
   not found. */

char *strtabmatch ( char **p, char *s )
{
  while ( *p && strncmp ( *p+1 , s , strlen ( *p+1 ) ) ) p++ ;
  return ( ! *p || **p == '-' ) ? NULL : *p ;
}


/* Send command to modem and check responses.  Collects pending
   (unexpected) responses and then pauses for inter-command delay
   (cmdpause) if t is negative.  Writes command s to modem if s is not
   null.  Reads responses and terminates when a response is one of the
   prompts in responses[] or if times out in t deciseconds.  Repeats
   command if detects a RING response (probable collision). Returns the
   first character of the matching prefix string (e.g. 'O' for OK, 'C' for
   CONNECT, etc.)  or EOF if no such response was received within timeout
   t. */

int cmd ( MODEM *m, const char *s , int t )
{
  char buf [ CMDBUFSIZE ] , *p = "" ;
  int resplen=0 ;

  lresponse = responses ;

  retry:

  while ( s && tgets ( m, buf, CMDBUFSIZE, t<0 ? m->cmdpause : 0 ) )
    msg ( "W- unexpected response \"%s\"", buf ) ;

  msg ( s ? "C- command  \"%s\"" : "C- waiting" , s ) ;

  if ( s ) { 
    tputs ( m, "AT", t ) ; tputs ( m, s, t ) ; tputcnow ( m, CR, t ) ; 
  }

  while ( t && ( p = tgets ( m, buf, CMDBUFSIZE, t<0 ? -t : t ) ) ) {

    msg ( "C- response \"%s\"" , p ) ;

    if ( ( resplen += strlen ( buf ) + 1 ) <= MAXRESPB ) {
      strcpy ( lresponse, buf ) ;
      lresponse += strlen ( buf ) + 1 ;
    }

    if ( ( p = strtabmatch ( (char**) prompts, buf ) ) ) break ;

    if ( ! strcmp ( buf, "RING" ) ) { msleep(100) ; goto retry ; }
  }

  return p ? *p : EOF ;
}


/* Send command to modem and wait for reply after testing (and
   possibly setting) current error status via err pointer. */

void ckcmd ( MODEM *m, int *err, const char *s, int t, int r )
{
  int c ;
  if ( ( ! err || ! *err ) && ( c = cmd ( m, s, t ) ) != r ) {
    msg ( c == EOF ? 
	 "E3modem command (%s) timed out" : 
	 "E3wrong response to command (%s)", s ? s : "none" ) ;
    if ( err ) *err = 3 ;
  }
}


/* Resynchronize modem from an unknown state.  If no immediate
   response, try pulsing DTR low (needs &D{2,3,4}), and then
   cancelling data or fax data modes.  In each case, discards any
   responses for about 2 seconds and then tries command ATQ0V1 to
   enable text responses.  Returns 0 if OK or 4 if no response.
   */

int faxsync( MODEM *m )
{
  int err=0, method=0 ;

  while ( ! err ) {
    switch ( method++ ) {
    case 0 : 
      break ;
    case 1 : 
      msg ("Isync: dropping DTR") ;
      ttymode ( m->f, COMMAND ) ; msleep ( 200 ) ;
      ttymode ( m->f, DROPDTR ) ; msleep ( 200 ) ;
      ttymode ( m->f, COMMAND ) ; 
      break ;
    case 2 : 
      msg ("Isync: sending escape") ;
      tputs ( m, DLE_ETX, -1 ) ; 
      msleep ( 1500 ) ;
      tputs ( m, "+++", -1 ) ; 
      break ;
    case 3 :
      err = msg ("E4sync: modem not responding") ;
      continue ;
    }
    while ( cmd ( m, 0, method ? TO_RESET : 5 ) != EOF ) ;
    if ( cmd ( m, "Q0V1", -TO_RESET ) == OK ) break ;
  }
  return err ;
} 


/* Terminate session.  Makes sure modem is responding, sends modem reset
   commands, removes lock files. Returns 0 if OK, 3 on error.*/

int end_session ( MODEM *m, char **zcmd,  char **lkfile )
{
  int err = faxsync ( m ) ;
  for ( ; ! err && *zcmd ; zcmd++ )
    if ( cmd( m, *zcmd, -TO_RESET ) != OK )
      err = msg ("E3modem reset command (%s) failed", *zcmd ) ;
  unlockall ( lkfile ) ;
  return err ;
} 
    

/* Signal handler: end session and exit */

MODEM *sigm ; char **sigcmd, **siglkfile ;

void onsig ( int sig ) 
{ 
  msg ( "Eterminating on signal %d", sig ) ; 
  end_session ( sigm, sigcmd, siglkfile ) ;
  msg ( "Idone, returning 5" ) ;
  exit ( 5 ) ; 
} 


/* Initialize session.  Try locking and opening fax device until opened or
   get error. Then set tty modes, register signal handler, set up
   modem. Returns 0 if OK, 2 on errors, 3 if initialization failed, 4 if no
   modem response. */

int begin_session ( MODEM *m, char *fname, int reverse, char **lkfile, 
		   char **icmd, int ignerr )
{
  int i, err=0, busy=0 ;

  do {
    err = lockall ( lkfile ) ;
    if ( ! err ) err = ttyopen ( m->f, fname, reverse ) ;
    if ( err == 1 ) { 
      if ( busy++ < 1 ) 
	msg ( "Wfax device %s locked or busy. waiting...", fname ) ;
      msleep ( LOCKPOLLDELAY * 10 ) ;
    }
  } while ( err == 1 ) ;
  
  if ( ! err ) msg ( "Iopened %s", fname ) ;

  if ( ! err ) err = ttymode ( m->f, COMMAND ) ;
  
  for ( i=0 ; ! err && catch [ i ] ; i++ ) 
    if ( signal ( catch [ i ] , onsig ) == SIG_ERR ) 
      err = msg ( "ES2can't set signal %d handler:", catch [ i ] ) ;
  
  if ( !err ) err = faxsync ( m ) ;

  for ( ; ! err && *icmd ; icmd++ )
    if ( cmd ( m, *icmd, -TO_RESET ) != OK && ! ignerr )
      err = msg ( "E3modem initialization command (%s) failed", *icmd ) ;

  return err ;
}


/* Position input file to start of next page if dp is non-zero or at start
   of current page if dp is 0.  Sets ppm to MPS if there are more pages to
   be sent with the same format, EOM if there are more pages with a new
   format, EOP if no more pages. Currently assumes all files are of same
   format and sets ppm to MPS or EOP only but eventually will handle
   multi-page files with embedded format information (e.g. TIFF-F) by
   setting local[VR/WD/LN] and setting ppm to EOM if required. Returns
   0 if OK, 2 on errors.  */

int rdpage ( IFILE *f, int dp, int *ppm, cap local )
{
  int err=0 ;

  err = nextipage ( f, dp ) ? 0 : 2 ;
  *ppm = ! err && f->fname [ 1 ] ? MPS : EOP ;

  return err ;
}


/* Terminate previous page if page is non-zero and start next output page
   if page is non-negative. If page is -1 removes the most recently opened
   file. Returns 0 if OK, 2 on errors. */

int wrpage ( OFILE *f, int page )
{
  int err=0 ;

  err = nextopage ( f, page ) ;

  if ( ! err && page == -1 )
    if ( remove ( f->cfname ) )
      err = msg ( "ES2can't delete file\n%s:", f->cfname ) ; 
    else
      msg ( "Iremoved %s", f->cfname ) ; 
  
  return err ;
}


/* Send data for one page.  Figures out required padding and 198->96 lpi
   decimation based on local and session capabilitites, substitutes page
   numbers in header string and enables serial port flow control.  Inserts
   the page header before the input file data.  Converts each scan line to
   T.4 codes and adds padding (FILL) and EOL codes before writing out.
   Sends RTC when done.  Sends DLE-ETX and returns serial port to command
   mode when done. Returns 0 if OK, non-0 on errors. */

int send_data ( MODEM *m, IFILE *f, int page, int pages,
	       cap local, cap session, char *header, pbmfont *font )
{
  int done=0, err=0, noise=0, bytes=0, pad=0, nr=0, line, pixels ;
  int i, decimate, pwidth, minlen ;
  u_char buf [ MAXCODES + 2*EOLBITS/8 + 1 ], *p ;
  short runs [ MAXRUNS ] ;
  char headerbuf [ MAXLINELEN ] ;
  time_t start ;
  long dt, draintime ;
  ENCODER e ;

  newENCODER ( &e ) ;

  minlen = ( (long) cps[session[BR]] * delay[session[ST]] + 500 ) / 1000 ;
  decimate = local[VR] > session[VR] ;
  pwidth = pagewidth [ session [ WD ] ] ;

  msg ( "N- padding to %d bytes/scan line.%s", minlen, 
       decimate ? " reducing 198->96 lpi." : "" ) ;

  sprintf ( headerbuf, header, page+1, pages, page+1, pages, page+1, pages ) ;
  msg ("I- header:[%s]", headerbuf ) ;
  p = putcode ( &e, EOLCODE, EOLBITS, buf ) ;
  done = err = ttymode ( m->f, SEND ) ; 
  start = time(0) ;

  for ( line=0 ; ! done ; line++ ) {
    
    if ( line >= HDRSPCE ) {	/* read a line */
      for ( i=0 ; ! done && i <= decimate ; i++ )
	if ( ( nr = readline ( f, runs, &pixels ) ) < 0 )
	  done = 1 ;
    } else {			/* blank line */
      runs[0] = pwidth ;
      pixels = pwidth ;
      nr = 1 ;
    }

    if ( nr > 0 && line >= HDRSTRT && line < HDRSTRT + font->h ) {
      int hnr ;
      short hruns [ MAXRUNS ] ;
      hnr = texttorun ( (u_char*) headerbuf, font, line-HDRSTRT, hruns, 0 ) ;
      xshift ( hruns, hnr, HDRSHFT ) ;
      nr = runor ( runs, nr, hruns,  hnr, 0, &pixels ) ;
    } 
    
    if ( nr > 0 ) {
      if ( pixels ) {
	if ( pixels != pwidth ) nr = xpad ( runs, nr, pwidth - pixels ) ;
	p = runtocode ( &e, runs, nr, p ) ;
	while ( p - buf < minlen ) p = putcode ( &e, 0, 8, p ), pad ++ ;
	p = putcode ( &e, EOLCODE, EOLBITS, p ) ;
	sendbuf ( m, buf, p - buf, -1 ), bytes += p - buf ;
      }
      if ( tdata ( m->f, 0 ) ) noise = 1 ;
      p = buf ;
    }
  }

  for ( i=0 ; i < RTCEOL ; i++ )
    p = putcode ( &e, EOLCODE, EOLBITS, p ) ;
  p = putcode ( &e, 0, 0, p ) ;
  sendbuf ( m, buf, p - buf, -1 ), bytes += p - buf ;

  if ( noise ) msg ("W- characters received while sending" ) ;
  
  tputs ( m, DLE_ETX, -1 ) ;		/* and tflush() */

  dt = time(0) - start ;
				/* time to drain buffers + 100% + 4s */
  draintime = ( 2 * ( bytes / cps[ session[BR] ] + 1 - dt ) + 4 ) * 10 ;
  draintime = draintime < TO_DRAIN ? TO_DRAIN : draintime ;

  ckcmd ( m, 0, 0, (int) draintime, OK ) ;

  dt = time(0) - start ;

  msg ( "Isent %d lines, %d+%d bytes, %d s  %d bps" , 
       line, bytes-pad, pad, (int) dt, (bytes*8)/dt ) ;

  if ( ! err ) err = ttymode ( m->f, COMMAND ) ; 

  return err ;
}


/* Read one scan line from fax device. If pointer pels is not
   null it is used to save pixel count.  Returns number of runs
   stored, EOF on RTC, or -2 on EOF, DLE-ETX or other error. */

int readfaxruns ( MODEM *m, DECODER *d, short *runs, int *pels )
{
  int err=0, c=EOF, x, n ;
  dtab *tab, *t ;
  short shift ;
  short *p, *maxp, *q, len=0 ;

  maxp = ( p = runs ) + MAXRUNS ;

  x = d->x ; shift = d->shift ; tab = d->tab ; /* restore decoder state */

  do {
    do {
      while ( shift < 0 ) { 
	c = tgetd ( m->f, TO_CHAR ) ;
	if ( c < 0 )  {
	  x = ( x << 15 ) | 1 ; shift += 15 ;  /* EOL pad at EOF */
	} else {
	  x = ( x <<  8 ) | c ; shift +=  8 ; 
	}
      }
      t = tab + ( ( x >> shift ) & 0x1ff ) ;
      tab = t->next ;
      shift -= t->bits ;
    } while ( ! t->code ) ;
    if ( p < maxp ) *p++ = t->code ;
  } while ( t->code != -1 ) ;

  d->x = x ; d->shift = shift ; d->tab = tab ; /* save state */

  if ( p >= maxp ) msg ( "Wrun length buffer overflow" ) ;

  /* combine make-up and terminating codes and remove +1 offset
     in run lengths */

  n = p - runs - 1 ;
  for ( p = q = runs ; n-- > 0 ; )
    if ( *p > 64 && n-- > 0 ) {
      len += *q++ = p[0] + p[1] - 2 ;
      p+=2 ;
    } else {
      len += *q++ = *p++ - 1 ;
    }
  n = q - runs ;
  
  /* check for RTC and errors */

  if ( len )
    d->eolcnt = 0 ;
  else
    if ( ++(d->eolcnt) >= RTCEOL ) err = EOF ;

  if ( c < 0 ) err = - 2 ;

  if ( pels ) *pels = len ;
  
  return err ? err : n ;
}


/* Receive data. Reads scan lines from modem and writes to output file.
   Checks for errors by comparing received line width and session line
   width.  Check that the output file is still OK and if not, send one
   CANcel character and wait for protocol to complete.  */

int receive_data ( MODEM *m, OFILE *f, cap session, int *nerr )
{
  int err=0, line, nr=0, len ;
  int pwidth = pagewidth [ session [ WD ] ] ;
  short runs [ MAXRUNS ] ;
  DECODER d ;

  newDECODER ( &d ) ;
  *nerr = 0 ;

  for ( line=0 ; nr >= 0 ; line++ ) {
    nr = readfaxruns ( m, &d, runs, &len ) ;
    if ( nr >= 0 ) 
      writeline ( f, runs, nr, 1 ) ;
    if ( ( len != pwidth ) && len && line ) { 
      (*nerr)++ ;
      if ( *nerr <= MAXERRPRT ) msg ("R-+ (%d:%d)", line, len ) ;
    }

    if ( ferror ( f->f ) ) {
      err = msg ("ES2file write:") ;
      tputcnow ( m, CAN, -1 ) ;
      msg ("Wdata reception CANcelled") ;
    } 
  }
  
  if ( *nerr ) {
    if ( *nerr > MAXERRPRT ) msg ("R-+ ....." ) ;
    msg ("R-  : reception errors" ) ;
    msg ("W- %d reception errors", *nerr ) ;
  }

  if ( nr != EOF ) 
    msg ( "Wno RTC" ) ;
  else
    while ( tgetd ( m->f, TO_CHAR ) >= 0 ) ;

  msg ( "I- received %d lines, %d errors", line, *nerr ) ;

  return err ;
}


/* Send training check sequence of n zeroes.  Returns 0 or 2 on error. */

int puttrain ( MODEM *m, const char *s, int n  )
{
  int i, err=0 ;

  ckcmd ( m, &err, s, TO_FT, CONNECT ) ;
  
  if ( ! err ) {
    ttymode ( m->f, SEND ) ;

    for ( i=0 ; i<n ; i++ ) {
      tputc ( 0, m->f, -1 ) ;
      if ( tobytes ( m->f ) >= MINWRITE ) tflush ( m->f, -1 ) ;
    }
    tputs ( m, DLE_ETX, -1 ) ;	/* also does tflush() */

    ttymode ( m->f, COMMAND ) ;
    ckcmd ( m, &err, 0, TO_DRAIN, OK ) ;
    msg ( "I- sent TCF (%d bytes)", n ) ;
  }
  
  return err ;
}


/* Checks for an error-free run of at least n bytes in the received
   training check sequence. Returns 0 if check passed, 1 if error-free run
   was too short, or 3 on other errors. */

int gettrain ( MODEM *m, const char *s, int n ) 
{ 
  int err=0, c=0, i=0, maxrunl=0, runl=0 ;

  ckcmd ( m, &err, s, TO_FT, CONNECT ) ;
  
  if ( ! err ) 
    for ( i=0 ; ( c = tgetd ( m->f, TO_CHAR ) ) >= 0 ; i++ )
      if ( c ) {
	if ( runl > maxrunl ) maxrunl = runl ;
	runl = 0 ;
      } else {
	runl ++ ;
      }
  
  if ( runl > maxrunl ) maxrunl = runl ;

  ckcmd ( m, &err, 0, TO_RTCMD, NO ) ;
  
  if ( !err ) {
    err = maxrunl < n ;
    msg ( "I- received TCF (%s: run of %d correct in %d bytes)", 
	 err ? "failed" : "passed" , maxrunl, i ) ;
  }

  return err ;
}


/* Log a sent/received HDLC frame.  Display of these messages is delayed to
   avoid possible timing problems. */

void logfr ( const char *s , const char *nm , u_char *p , int n )
{
  int i=0 ;
  msg ( "I- %s %s", s, nm ) ;
  msg ( n > 10 ? "H- %s %d bytes:" : "H-+ %s %d bytes:" , s, n ) ;
  for ( i=0 ; i<n ; i++ ) {
    msg ( "H-+  %02x" , p[i] & 0xff ) ;
    if ( ( i&0xf ) == 0xf && i != n-1 ) msg ( "H-" ) ;
  }
  msg ( "H-") ;
}


/* Send HDLC control frame of type type.  Extra bits can be OR'ed
   into the frame type (FCF) to indicate that this frame follows
   a previous one (no +FTH required) and/or that more frames will
   follow.  Sets up flag, address, and fax control field bytes in
   `buf'.  Sends these plus `len` additional bytes.  Terminates
   with DLE-ETX and checks response.  Returns 0 if OK, 2 or 3 on
   error. */

#define MORE_FR  0x100 
#define SUB_FR   0x200 

int putframe ( int type, u_char *buf , int len, MODEM *m, int t )
{
  int err=0 ;

  buf [ 0 ] = 0xff ;
  buf [ 1 ] = type & MORE_FR ? 0xc0 : 0xc8 ;
  buf [ 2 ] = type & 0xff ;

  if ( ! ( type & SUB_FR ) )
    ckcmd ( m, &err, "+FTH=3" , TO_FT, CONNECT ) ;

  if ( ! err ) {
    ttymode ( m->f, SEND ) ;
    sendbuf ( m, buf, len+3, -1 ) ;
    tputs ( m, DLE_ETX, -1 ) ;	            /* and tflush() */
    ttymode ( m->f, COMMAND ) ;

    ckcmd ( m, &err, 0, TO_DRAIN, ( type & MORE_FR ) ? CONNECT : OK ) ;
    if ( ! err ) 
      logfr ( "sent", frname( type & 0x7f ), buf, len+3 ) ;
  }

  return err ;
}


/* Read HDLC frame and store it in buffer buf of size n.  Skips issuing
   +FRH command on pass==0.  Returns length of frame if OK, EOF on timeout,
   -3 if any errors as per T.30 5.4.2 (too long, FCS error) */

int getframe ( MODEM *m, int pass, u_char *buf, int n, int t )
{
  int err=0, c=0, i=0 ;

  if ( pass && ( c = cmd ( m, "+FRH=3", t ) ) != CONNECT ) 
    err = ( c == EOF ) ? -EOF : msg ( "E3get frame command failed") ;
  
  if ( err == -EOF ) { 
    tputc ( CAN, m->f, t ) ; 
    cmd ( m, "", TO_ABRT ) ; 
  }

  if ( ! err ) {
    for ( i=0 ; ( c = tgetd ( m->f, pass ? TO_CHAR : t ) ) >= 0  ; i++ )
      if ( i < n ) buf[ i ] = c ;
    if ( c == EOF ) {
      err = ( msg ( "E3timed out reading frame data"), -EOF ) ;
    } else {
      if ( i >= n ) 
	err = msg ( "E3frame too long (%d bytes)", i ) ;
      if ( ( c = cmd ( m, 0, TO_RTCMD ) ) != OK && c != CONNECT ) 
	err = msg ( "E3wrong response after frame data" ) ;
    }
  } 

  return err ? -err : ( i < n ? i : n ) ;
}


/* Reverse bit and byte order of ID strings as per T.30 5.3.6.2.4-6 */

void revcpy ( u_char *from , u_char *to )
{
  int i, j ;
  for ( i=0, j=IDLEN-1 ; i<IDLEN ; i++, j-- ) 
    to [ i ] = normalbits [ from [ j ] & 0xff ] ;
}


/* Handle procedure interrupt requests (just prints message for now).
   Returns 0. */

int proc_int( void )
{
  return msg ("W0procedure interrupt request ignored" ) ;
}


/* Class 1 send.  Each received frame elicits an appropriate reply.
   Optional or unrecognized frames are ignored.  Terminates by sending DCN
   after receiving MCF after EOP or on error.  Timeouts, bad frames or CRP
   repeat last command up to MAXRETRY times.  On training check failures
   the speed (remote capability[BR]) is reduced. The lowest speed is retried
   MAXTRAIN times.  Page transmission failures are retried NTXRETRY
   times. */


enum replies
{ NONE=0x100, TXDATA, PPM, SENDDIS } ;

int c1send ( MODEM *m, IFILE *f, int pages, 
	    cap local, char *localid, char *header, pbmfont *font ) 
{ 
  int err=0, done=0, pass=0, started=0, page=0, frame=NONE, reply=DCN ;
  int frlen, rxdislen=0, disbit=0, ppm=EOP ;
  int cmdtry=0, pagetry=0, traintry=0 ;
  cap remote = { DEFCAP }, session = { DEFCAP } ;
  char remoteid [ IDLEN + 1 ] ;
  u_char buf [ MAXFRLEN ], *fif=buf+3 ;

  for ( pass=started ; ! done ; pass++ ) {

    if ( err ) {
      frame = NONE ;
    } else {
      frlen = getframe ( m, pass, buf, MAXFRLEN, started ? T3S : T1 ) ;
      if ( frlen < 3 ) {
	frame = CRP ;
      } else {
	frame = buf [ 2 ] & 0x7f ;
	logfr ( "received" , frname(frame), buf , frlen ) ;
	cmdtry = 0 ;
      }
    }

    switch ( frame ) {
      
    case CRP:
      if ( !started || cmdtry++ >= MAXRETRY ) 
	err = msg ( "E3no response from remote" ) ;
      break ;

    case NSF:
      reply = NONE ;
      break ;

    case CSI:
      revcpy ( fif , (u_char*) remoteid ) ;
      msg ( "I- remote ID=\"%*.*s\"", IDLEN, IDLEN, remoteid ) ;
      reply = NONE ;
      break ;

    case DIS:
      started = 1 ;
      disbit = 0x80 ;
      rxdislen = dislen ( fif ) ;
      mkcap ( fif, remote, 1 ) ;
      reply = DCS ;
      break ;

    case CFR:
      reply = TXDATA ;
      break ;

    case FTT:
      if ( (remote[BR] = fallback[session[BR]]) > 0 || traintry++ < MAXTRAIN ) 
	reply = DCS ; 
      else
	err = msg ( "E1failed to train") ; 
      break ;

    case PIP: 
      proc_int() ; 
    case RTP:
    case MCF:
      page++ ;
      pagetry=0 ;
      if ( ppm == MPS && frame == MCF ) reply = TXDATA ;
      else if ( ppm == EOM ) reply = DCS ;
      else if ( ppm == EOP ) reply = DCN ;
      else err = msg ( "E2can't happen(c1send.ppm)" ) ;
      break ;

    case PIN: 
      proc_int() ;
    case RTN:
      if ( pagetry++ < NTXRETRY ) reply = DCS ;
      else err = msg( "E2too many page send retries" ) ;
      break ;

    case DCN:
      err = msg ( "E3receiver disconnect") ;
      break ;

    case NONE:
      break ;

    default:
      break ;

    } /* switch ( frame ) */

    switch ( err ? DCN : reply ) {

    case DCS:				    /* "start of Phase B" */
      revcpy ( (u_char*) localid , fif ) ;
      if ( !err ) err = 
	putframe ( TSI | MORE_FR | disbit, buf, IDLEN, m, -1 ) ;      
      mincap ( local, remote, session ) ;
      mkdis ( session, fif, rxdislen, 0 ) ;
      if ( !err ) err = 
	putframe ( DCS | SUB_FR | disbit, buf, rxdislen, m, -1 ) ;
      ckcmd ( m, &err, "+FTS=8", TO_FT, OK ) ;
      if ( !err ) err = 
	puttrain ( m, c1cmd[1][1][session[BR]], 1.65 * cps [ session[BR] ] ) ;
      reply = DCS ;
      break ;

    case TXDATA:
      err = rdpage ( f, ! ( page == 0 || pagetry > 0 ), &ppm, local ) ;
      ckcmd ( m, &err,  c1cmd [1][0][session[BR]], TO_FT, CONNECT ) ;
      if ( !err ) 
	err = send_data ( m, f, page, pages, local, session, header, font ) ;
      ckcmd ( m, &err, "+FTS=8", TO_FT, OK ) ;      
							   /* fall through */
    case PPM:
      if ( !err ) err =	putframe ( ppm | disbit, buf, 0, m, -1 ) ;
      reply = PPM ;
      break ;

    case DCN:
      putframe ( DCN | disbit, buf, 0, m, -1 ) ; 
      done = 1 ;
      break ;

    case NONE:
      break ;

    default:
      err = msg ( "E3can't happen(reply)" ) ;
      break ;

    } /* switch ( reply ) */

  } /* for ( ! done ) */

  return err ;
}


/* Class 1 receive.  Sends DIS until gets DCS/DIS or times out.  Sends ppr
  after ppm and receives data after DCS or MPS. Note: TCF (training check
  data) is received 75 +/- 20 ms after DCS so there should be no lengthy
  processing between DCS and gettrain(). */

int c1receive ( MODEM *m, OFILE *f, cap local, char *localid, 
	       int maxpgerr, int poll ) 
{ 
  int err=0, done=0, started=0, page=0, nerr=0, pass, frlen ;
  int disbit=0, good=1, nfrerr=0, frame ;
  u_char buf[MAXFRLEN], *fif=buf+3 ;
  cap session ;
  char remoteid [ IDLEN + 1 ] ;

  err = wrpage ( f, page ) ;

  for ( pass=0 ; ! err && ! done ; pass++ ) {

    if ( pass == 0 && ! poll ) {
      frame = SENDDIS ;
    } else {
      frlen = getframe ( m, pass, buf, MAXFRLEN, 
			started ? T2 : poll ? T1 : T4 ) ;
      if ( frlen >= 3 ) {
	started = 1 ;
	nfrerr = 0 ;
	frame = buf [ 2 ] ;
	if ( frame != DTC ) frame &= 0x7f ;
	logfr ( "received" , frname(frame), buf, frlen ) ;
      } else if ( ! started && ! poll && pass < MAXDIS ) {
	frame = SENDDIS ;
      } else if ( ++nfrerr < 3 ) {
	frame = NONE ;
      } else {
	err = msg ( "E3timed out" ) ;
	frame = DCN ;
      }
    }
    
    switch ( frame ) {
      
    case DIS:
      disbit = 0x80 ;
    case CRP:
    case SENDDIS:
      revcpy ( (u_char*) localid, fif ) ;
      if ( !err ) err = 
	putframe ( CSI | disbit | MORE_FR | ( poll || pass ? 0 : SUB_FR ),
		  buf, IDLEN, m, -1 ) ;
      mkdis ( local, fif, DEFDISLEN, 1 ) ;
      if ( !err ) err = 
	putframe ( DIS | disbit | SUB_FR, buf, DEFDISLEN, m, -1 ) ;
      break ;
      
    case TSI: ;
    case CSI: ;
      revcpy ( fif, (u_char*) remoteid ) ;
      msg ( "I- remote ID=\"%*.*s\"" , IDLEN, IDLEN, remoteid ) ;
      break ;
      
    case DCS: ;
      mkcap ( fif, session, 0 ) ;
      printcap ( "session", session ) ;
      if ( gettrain ( m, c1cmd [0][1][session[BR]], cps[session[BR]] ) == 0 ) {
	if ( !err ) err = putframe ( CFR | disbit, buf, 0, m, -1 ) ;
	if ( !err ) goto getdata ;
      } else {
	if ( !err ) err = putframe ( FTT | disbit, buf, 0, m, -1 ) ;
      }
      break ;

    case PRI_EOM:
    case PRI_MPS:
    case PRI_EOP:
      proc_int() ;
							   /* fall through */
    case EOM:
    case MPS:
    case EOP:
      err = wrpage ( f, ++page ) ;	
      if ( ! err )
	if ( good ) err = putframe ( MCF | disbit, buf, 0, m, -1 ) ;
	else        err = putframe ( RTN | disbit, buf, 0, m, -1 ) ;
      if ( ! err && good && ( frame == MPS || frame == PRI_MPS ) )
	goto getdata ; 
      if ( ! err && good && ( frame == EOM || frame == PRI_EOM ) ) {
	ckcmd ( m, &err, "+FTS=8", TO_FT, OK ) ;      
      }
      break ;

    getdata:
      ckcmd ( m, &err, c1cmd [0][0][session[BR]] , TO_FT, CONNECT ) ;
      if ( !err ) err = receive_data ( m, f, session, &nerr ) ;
      good = nerr < maxpgerr ;
      if ( good ) msg ( "ITreceived -> %s", f->cfname ) ;
      ckcmd ( m, &err, 0, TO_C2EOR, NO ) ;
      break ;
      
    case DTC:			/* no poll server yet */
      putframe ( DCN, buf, 0, m, -1 ) ;
      done=1 ;
      break ;

    case DCN:
      done = 1 ;
      break;

    case NONE:
      break ;

    default:
      msg ( "I%s frame ignored", frname ( frame ) ) ;
      break ;

    } /* switch */
    
  } /* do */

  wrpage ( f, -1 ) ;		/* remove last file */
  return err ;
}


void getc2dcs ( cap session )
{
  char *p ;
  if ( ( p = sresponse ( "+FCS:", 0 ) ) != 0 ) 
    str2cap ( p, session ) ;
  else if ( ( p = sresponse ( "+FDCS:", 0 ) ) != 0 )	
    str2cap ( p, session ) ;
}  

/* Class 2 fax transmission.  Retries each page up to NTXRETRY
   times.  Transmission begins after DC2 or XON is received.
   Sends the data and sends appropriate post-page message.  */

int c2send ( MODEM *m, IFILE *f, int pages, 
	    cap local, char *header, pbmfont *font )
{
  int c=0, page, err=0, done=0, try ;
  int ppr, ppm=0, hsc=-1, good, noise=0 ;
  cap session = { 0,0,0,0, 0,0,0,0 } ;
  
  for ( page=0 ; ! err && ! done ; page++ ) {

    for ( try=good=0 ; ! err && ! done && ! good && try<NTXRETRY ; try++ ) {

      done = rdpage ( f, ! ( page==0 || try > 0 ), &ppm, local ) ;
      if ( done != 1 ) err = done ;
      if ( done ) continue ;

      if ( ppm == EOM )		/* send +FDIS= */ ;

      if ( cmd ( m, "+FDT", TO_C2B ) != CONNECT ) {
	err = 2 ; 
	continue ;
      }
  
      getc2dcs ( session ) ;

      while ( ! err && ( c = tgetc ( m->f, TO_C2X ) ) != XON && c != DC2 )
	if ( c == EOF ) {
	  msg ( "Wno XON/DC2 received after CONNECT") ;
	  break ;
	} else { 
	  msg ( "W+%s", cname ( c ) ) ; 
	  noise++ ; 
	}
      
      if ( noise ) {
	msg ( "Wreceived (%d) characters while waiting to send", noise ) ;
	noise = 0 ;
      }
      
      err = send_data ( m, f, page, pages, local, session, header, font ) ;
      
      ckcmd ( m, &err, ppm == EOP ? "+FET=2" : 
	     ppm == EOM ? "+FET=1" : "+FET=0" , TO_C2PP, OK ) ;
      
      if ( sresponse ( "+FPTS:", &ppr ) || sresponse ( "+FPS:", &ppr ) ) ;
      else 
	ppr = msg ( "Wno +FP[T]S response" ), 1 ;

      good = ppr & 1 ;
      
    }
    
    if ( ppm == EOP ) done = 1 ;
    if ( try >= NTXRETRY && ! good ) {
      err = msg ( "E2too many page send retries" ) ;
      cmd ( m, m->c20 ? "+FKS" : "+FK", T3S ) ;
    }
    
  } 
  
  if ( ( sresponse ( "+FHNG:", &hsc ) || sresponse ( "+FHS:", &hsc ) )
      && hsc > 0 )
    err = msg ( "E2abnormal termination (code %d)", hsc ) ;
  
  return err ; 
}


/* Class 2 fax reception.  Send fax data receive command.  If
   response is OK, then no more pages are coming.  If it's
   CONNECT receive the data for one page.  Returns 0 or 2 for
   errors.  */

int c2receive ( MODEM *m, OFILE *f, cap local, int maxpgerr )
{
  int err=0, done=0, page, good, nerr, hsc, c=0 ;
  cap session = { 0,0,0,0, 0,0,0,0 } ;

  for ( page=0 ; ! err && ! done ; page++ ) {

    err = wrpage ( f, page ) ;

    if ( ! err && ( c = cmd ( m, "+FDR", TO_C2R ) ) == CONNECT ) {

      getc2dcs ( session ) ;

      tputcnow ( m, m->startchar, -1 ) ;

      err = receive_data ( m, f, session, &nerr ) ;

      good = nerr < maxpgerr ;

      if ( cmd ( m, 0, TO_C2EOR ) != OK )
	err = msg ( "E2no response after sending data" ) ;

      if ( good ) msg ( "ITreceived -> %s", f->cfname ) ;

      /* +FPTS= not available on Intel modems:
	 ckcmd ( m, &err, good ? "+FPTS=1" : "+FPTS=2", T3S , OK ) ; */

    } else {
      wrpage ( f, -1 ) ; 
      if ( c == OK ) done = 1 ; 
      else err = msg ( "E3receive (+FDR) command failed") ;
    }
  } 

  if ( ( sresponse ( "+FHNG:", &hsc ) || sresponse ( "+FHS:", &hsc ) )
      && hsc > 0 )
    err = msg ( "E2abnormal termination (code %d)", hsc ) ;
  
  return err ;
}


/* Dial a number and send a fax. */

int dial ( MODEM *m, char *s )
{
  int err=0, hsc=-1 ;
  char c, dsbuf [ 128 ] ;

  sprintf ( dsbuf , "D%.127s" , s ) ;
  msg ( "Idialing %s", dsbuf+1 ) ;

  c = cmd ( m, dsbuf, TO_A ) ;
  if ( c == ( m->c1 ? CONNECT : OK ) ) 
    msg ( "Iconnected" ) ; 
  else if ( c ==  BUSY ) 
    err = msg ( "W1number is busy" ) ; 
  else 
    err = msg ( "E2can't establish session" ) ;

  if ( ( sresponse ( "+FHNG:", &hsc ) || sresponse ( "+FHS:", &hsc ) )
      && hsc > 0 ) 
    err = msg ( "E2abnormal termination (code %d)", hsc ) ;

  return err ;
}


/* Answer the phone.  Open modem device and initialize it.  Remove our lock
   if sharing device with outgoing calls.  If waiting for call, wait for
   modem activity, else answer phone.  Figure out what mode we answered in
   and handle call appropriately.  Re-lock if necessary. Exec *getty if
   data call. Modems prompt as follows after answering: Class 0: CONNECT
   nnn for data; Class 1: FAX + CONNECT for fax, DATA + CONNECT nnn for
   data, just CONNECT for fax if +FAE=0; Class 2: CONNECT (data) or OK
   (fax). ("+FCON" and "CONNECT FAX" are status messages, not prompts). */

int answer ( MODEM *m, char **lkfile, 
	    int wait, int share, int softaa, char *getty )
{
  enum connectmode { NONE, DATAMODE, FAXMODE } ; 
  enum connectmode mode=NONE ;
  int datamode=0 ;			/* true if answered in data mode */
  int c=0, err=0 ;
  int hsc=-1 ;
  int crate = 19200 ;		        /* CONNECT rate */

  if ( ! err && share ) err = unlockall ( lkfile ) ;

  if ( ! err && wait ) {
    msg ( "Iwaiting for activity") ;
    tdata ( m->f, -1 ) ;
    msg ( "Iactivity detected at ") ;
  }
  
  if ( ! err && share ) {
    msleep ( 200 ) ;		/* let other programs lock port  */
    err = lockall ( lkfile ) ;
  }

  if ( ! err && softaa && *getty ) {
    if ( cmd ( m, ( wait ? 0 : "A" ), TO_DATAF ) == CONNECT ) {
      sresponse ( "CONNECT", &crate ) ;
      sresponse ( "+FHNG:", &hsc ) ;
      sresponse ( "+FHS:", &hsc ) ;
      mode = DATAMODE ;
    } else {
      int i ;	/* abort data answer mode & set fax mode to try again */
      for ( i=0 ; i<3 ; i++ ) 
	if ( cmd ( m, m->c1 ? "+FCLASS=1" : 
		  ( m->c20 ? "+FCLASS=2.0" : "+FCLASS=2" ), -TO_RESET ) == OK )
	  break ; 
    }
  }

  if ( ! err && mode == NONE ) {
    c = cmd ( m, ! wait || ( softaa && *getty ) ? "A" : 0, TO_A ) ;
    datamode = sresponse ( "DATA", 0 ) || sresponse ( "CONNECT DATA", 0 ) ;
    sresponse ( "CONNECT", &crate ) ;
    sresponse ( "+FHNG:", &hsc ) ;
    sresponse ( "+FHS:", &hsc ) ;
    if ( m->c1 )
      mode = ( c == CONNECT ) ? ( datamode ? DATAMODE : FAXMODE ) : NONE ;
    else
      mode = ( c == CONNECT ) ? DATAMODE : ( c == OK ? FAXMODE : NONE ) ;
  }
  
  if ( err || hsc >= 0 ) mode = NONE ;

  if ( ! err )
    switch ( mode ) {
    case DATAMODE : {
      char buf [ MAXGETTY ] ;
      msg ( "Idata call answered") ;
      sprintf ( buf , getty , crate, crate, crate, crate, crate, crate ) ;
      msg ( "Iexec'ing /bin/sh -c \"%s\"" , buf ) ;
      execl ( "/bin/sh" , "sh" , "-c" , buf , (void*) 0 ) ; 
      err = msg ( "ES2exec failed:" ) ;
      break ; }
    case FAXMODE :
      msg ( "Ifax call answered") ;
      break ;
    case NONE:
      err = msg ( "E3unable to answer call") ;
      break ;
    }
    
  return err  ;
}


/* Fax send/receive program for Class 1 or 2 fax modems. Returns
   0 on success, 1 if number busy or device locked, 2 for errors,
   3 for protocol errors, 4 if no modem response, 5 on fatal
   signal. */

int main( int argc, char **argv)
{
  int err=0, doneargs=0, c=0, locked, i ;

  char msgbuf [ MAXMSGBUF ] ;
  FILE *rfp;

  char *icmd[ MAXICMD ], *zcmd[ MAXICMD ], *lkfile [ MAXLKFILE+1 ] ;
  int nicmd=0, nzcmd=0, nlkfile=0 ;

  char *faxfile = FAXFILE ;
  TFILE faxdev ;
  MODEM modem = { 0, 0,0, T_CMD, DC2 }, *m = &modem ;
  int softaa=0, share=0, wait=0, reverse=0, ignerr=0 ;
  char *getty = "" ;

  char capinit [ CMDBUFSIZE ], idinit [ CMDBUFSIZE ] ;
  cap local = { DEFCAP } ;
  char localid  [ IDLEN + 1 ] = "" ;
  int maxpgerr = 10 ;

  time_t t ;
  char *header = 0, headerbuf [ MAXLINELEN ] ; 
  char *fontname = 0 ;
  pbmfont font ;

  IFILE ifile ;
  int pages = 0 ;
  OFILE ofile ;
  char *phnum="", *ansfname, fnamepat [ FILENAME_MAX ] ;

  modem.f = &faxdev ;

  cname ( 0 ) ;
  argv0 = argv[0] ;
  setvbuf ( LOGF , msgbuf , _IOFBF , MAXMSGBUF ) ;
  msg ( "I " Version "  starts ") ;
  if ( strrchr ( argv0 , '/' ) ) argv0 = strrchr ( argv0 , '/' ) + 1 ;
  msg ( "I " Copyright "  (compiled "__DATE__ " " __TIME__ ")" ) ;

  while ( ! err && ! doneargs &&
	 ( c=nextopt(argc,argv,"c:d:f:g:h:i:l:o:p:q:r:st:v:wx:z:T") ) != -1) {
    switch (c) {
    case 'c': 
      err = str2cap ( optarg , local ) ;
      sprintf ( capinit , m->c20 ? "+FCC=%.*s" : "+FDCC=%.*s" , 
	       CMDBUFSIZE-7, optarg ) ;
      if ( ! err && ! m->c1 ) { optarg = capinit ; goto addopt ; }
      break ;
    case 'l': 
      if ( strlen ( optarg ) > IDLEN ) 
	msg("Wlocal ID (%s) truncated to 20 characters", optarg ) ;
      if ( strspn ( optarg, " +0123456789" ) != strlen ( optarg ) )
	msg("Wlocal ID (%s) has non-standard characters", optarg ) ;
      sprintf ( localid, "%*.*s", IDLEN, IDLEN, optarg ) ;
      sprintf ( idinit , m->c20 ? "+FLI=\"%.*s\"" : "+FLID=\"%.*s\"" , 
	       CMDBUFSIZE-9, localid ) ;
      if ( ! m->c1 ) { optarg = idinit ; goto addopt ; }
      break ;
    case 'i': 
    addopt:
      if ( nicmd < MAXICMD ) icmd [ nicmd++ ] = optarg ;
      else err = msg ( "E2too many modem init commands"); 
      break ;
    case 'z': 
      if ( nzcmd < MAXICMD ) zcmd [ nzcmd++ ] = optarg ;
      else err = msg ( "E2too many modem reset commands"); 
      break ;
    case 'h': header = optarg ; break ;
    case 'f': fontname = optarg ; break ;
    case 'd': faxfile = optarg ; break ;
    case 'g': getty = optarg ; break ;
    case 'o': 
      for ( ; *optarg ; optarg++ ) 
	switch ( *optarg ) {
	case '0' : m->c20 = 1 ; break ;
	case '1' : m->c1 = 1 ; break ;
	case 'a' : softaa = 1 ;  break ;
	case 'e' : ignerr = 1 ;  break ;
	case 'r' : reverse = 1 ; break ;
	case 'x' : m->startchar = XON ; break ;
	case 'z' : m->cmdpause += T_CMD ; break ;
	 default : msg ( "Wunrecognized protocol option (%c)", *optarg ) ; 
	}
      break ;
    case 'q':
      if ( sscanf ( optarg , "%d", &maxpgerr ) != 1 )
	err=msg ("E2bad quality (-q) argument (%s)", optarg ) ;
      break;
    case 'p':
      phnum = optarg ;
      optarg = optind < argc ? argv [ optind ] : "" ;
					    /* fall through */
    case 'r': 
      ansfname = optarg ;
      doneargs=1 ; 
      break;
    case 's': share = 1 ; break;
    case 't': 
      if ( argv [ argc ] ) err = msg ("E2can't happen(unterminated argv)") ;
      newIFILE ( &ifile, FAX, argv + optind ) ;
      pages = argc - optind ;
      phnum = optarg ;
      doneargs=1 ; 
      break;
    case 'v': verb = optarg ; break ;
    case 'w': wait = 1 ; break ;
    case 'x': 
      if ( nlkfile < MAXLKFILE ) lkfile [ nlkfile++ ] = optarg ; 
      else err = msg ( "E2too many lock files" ) ; 
      break ;
    case 'T':			/* test: begin+end session */
      doneargs=1 ; 
      break ;
    default : fprintf ( stderr, Usage, argv0 ) ; err = 2 ; break ;
    }
  }

  for ( i=0 ; i<argc ; i++ ) 
    msg ( "Aargv[%d]=%s", i, argv[i]) ; 

  zcmd [ nzcmd ] = icmd [ nicmd ] = lkfile [ nlkfile ] = 0 ;

  readfont ( fontname, &font ) ;

  if ( ! header )
    strftime ( header = headerbuf , MAXLINELEN, "%c %%s ->  P. %%d/%%d", 
	      ( t = time(0), localtime ( &t ) ) ) ;

  sigm = m ;
  sigcmd = zcmd ;
  siglkfile = lkfile ;

  if ( ! err )
    err = begin_session ( m, faxfile, reverse, lkfile, icmd, ignerr ) ;
  locked = ( err == 1 ) ;
  
  if ( ! err )
    switch ( c ) {
    case 't':
      err = dial ( m, phnum ) ; 
      if ( ! err ) 
	if ( m->c1 )
	  err = c1send ( m, &ifile, pages, local, localid, header, &font ) ;
	else
	  err = c2send ( m, &ifile, pages, local, header, &font ) ;
      break ;
    case 'r':
      err = answer ( m, lkfile, wait, share, softaa, getty ) ;
      locked = err == 1 ;
      strftime ( fnamepat, FILENAME_MAX, *ansfname ? ansfname : "%m%d%H%M%S",
		( t = time(0), localtime ( &t ) ) ) ;
      strncat ( fnamepat, ".%03d", FILENAME_MAX - strlen ( fnamepat ) ) ;
      newOFILE ( &ofile, FAX, fnamepat, 204, 196, 1728, 2287 ) ;
      if ( ! err ) 
	if ( m->c1 )
	  err = c1receive ( m, &ofile, local, localid, maxpgerr, 0 ) ;
	else
	  err = c2receive ( m, &ofile, local, maxpgerr ) ;
      break ;
    case 'p':
      if ( ! m->c1 ) {
	msg ("E2pollling only implemented for Class 1" ) ;
      } else {
	err = dial ( m, phnum ) ; 
	if ( ! err ) 
	  err = c1receive ( m, &ofile, local, localid, maxpgerr, 1 ) ;
      }
      break ;
    case 'T':
      break ;
    }
  
  if ( ! locked && err != 4 && m->f->fd >= 0 ) 
    err = end_session ( m, zcmd, lkfile ) ;
  
  msg ( "Idone, returning %d", err ) ;

  /* The following 3 lines added for use with Qfax 1.0 */
  rfp = fopen(RESULT, "w");
  fprintf(rfp, "%d", err);
  fclose(rfp);

  return err ;

}
