/*
 * $Id: utils.c,v 1.79 1995/12/19 12:21:51 coleman Exp coleman $
 *
 * utils.c - miscellaneous utilities
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#include "zsh.h"

/* Print an error */

/**/
void
zwarnnam(char *cmd, char *fmt, char *str, int num)
{
    int waserr;

    waserr = errflag;
    zerrnam(cmd, fmt, str, num);
    errflag = waserr;
}

/**/
void
zerr(char *fmt, char *str, int num)
{
    if (errflag || noerrs)
	return;
    errflag = 1;
    trashzle();
    fprintf(stderr, "%s: ", (isset(SHINSTDIN)) ? "zsh" : argzero);
    zerrnam(NULL, fmt, str, num);
}

/**/
void
zerrnam(char *cmd, char *fmt, char *str, int num)
{
    if (cmd) {
	if (errflag || noerrs)
	    return;
	errflag = 1;
	trashzle();
	if (isset(SHINSTDIN))
	    fprintf(stderr, "%s: ", cmd);
	else
	    fprintf(stderr, "%s: %s: ", argzero, cmd);
    }
    while (*fmt)
	if (*fmt == '%') {
	    fmt++;
	    switch (*fmt++) {
	    case 's':
		while (*str)
		    niceputc(*str++, stderr);
		break;
	    case 'l':
		while (num--)
		    niceputc(*str++, stderr);
		break;
	    case 'd':
		fprintf(stderr, "%d", num);
		break;
	    case '%':
		putc('%', stderr);
		break;
	    case 'c':
		niceputc(num, stderr);
		break;
	    case 'e':
		/* print the corresponding message for this errno */
		if (num == EINTR) {
		    fputs("interrupt\n", stderr);
		    errflag = 1;
		    return;
		}
		/* If the message is not about I/O problems, it looks better *
		 * if we uncapitalize the first letter of the message        */
		if (num == EIO)
		    fputs(strerror(num), stderr);
		else {
		    char *errmsg = strerror(num);
		    fputc(tulower(errmsg[0]), stderr);
		    fputs(errmsg + 1, stderr);
		}
		break;
	    }
	} else
	    putc(*fmt++, stderr);
    if (unset(SHINSTDIN) && lineno)
	fprintf(stderr, " [%ld]\n", lineno);
    else
	putc('\n', stderr);
    fflush(stderr);
}

/* Output a single character, for the termcap routines.     *
 * This is used instead of putchar since it can be a macro. */

/**/
int
putraw(int c)
{
    putc(c, stdout);
    return 0;
}

/* Output a single character, for the termcap routines. */

/**/
int
putshout(int c)
{
    putc(c, shout);
    return 0;
}

/* Turn a character into a visible representation thereof.  The visible *
 * string is put together in a static buffer, and this function returns *
 * a pointer to it.  Printable characters stand for themselves, DEL is  *
 * represented as "^?", newline and tab are represented as "\n" and     *
 * "\t", and normal control characters are represented in "^C" form.    *
 * Characters with bit 7 set, if unprintable, are represented as "\M-"  *
 * followed by the visible representation of the character with bit 7   *
 * stripped off.  Tokens are interpreted, rather than being treated as  *
 * literal characters.                                                  */

/**/
char *
nicechar(int c)
{
    static char buf[6];
    char *s = buf;
    c &= 0xff;
    if (itok(c)) {
	if(c >= Pound && c <= Comma)
	    c = ztokens[c - Pound];
	else 
	    c = 0;
	goto done;
    }
    if(isprint(c))
	goto done;
    if(c & 0x80) {
	*s++ = '\\';
	*s++ = 'M';
	*s++ = '-';
	c &= 0x7f;
	if(isprint(c))
	    goto done;
    }
    if(c == 0x7f) {
	*s++ = '^';
	c = '?';
    } else if(c == '\n') {
	*s++ = '\\';
	c = 'n';
    } else if(c == '\t') {
	*s++ = '\\';
	c = 't';
    } else if(c < 0x20) {
	*s++ = '^';
	c += 0x40;
    }
    done:
    *s++ = c;
    *s = 0;
    return buf;
}

/**/
void
niceputc(int c, FILE *f)
{
    fputs(nicechar(c), f);
}

/**/
void
nicefputs(char *s, FILE *f)
{
    for (; *s; s++)
	fputs(nicechar(STOUC(*s)), f);
}

/**/
size_t
nicestrlen(char *s)
{
    size_t l = 0;

    for(; *s; s++)
	l += strlen(nicechar(STOUC(*s)));
    return l;
}

/* get a symlink-free pathname for s relative to PWD */

/**/
char *
findpwd(char *s)
{
    char *t;

    if (*s == '/')
	return xsymlink(s);
    s = tricat((pwd[1]) ? pwd : "", "/", s);
    t = xsymlink(s);
    zsfree(s);
    return t;
}

/* Check whether a string contains the *
 * name of the present directory.      */

/**/
int
ispwd(char *s)
{
    struct stat sbuf, tbuf;

    if (stat(s, &sbuf) == 0 && stat(".", &tbuf) == 0)
	if (sbuf.st_dev == tbuf.st_dev && sbuf.st_ino == tbuf.st_ino)
	    return 1;
    return 0;
}

static char xbuf[PATH_MAX];

/* expand symlinks in s, and remove other weird things */

/**/
char *
xsymlink(char *s)
{
    if (unset(CHASELINKS))
	return ztrdup(s);
    if (*s != '/')
	return NULL;
    strcpy(xbuf, "");
    if (xsymlinks(s + 1, 1))
	return ztrdup(s);
    if (!*xbuf)
	return ztrdup("/");
    return ztrdup(xbuf);
}

/**/
char **
slashsplit(char *s)
{
    char *t, **r, **q;
    int t0;

    if (!*s)
	return (char **) zcalloc(sizeof(char **));

    for (t = s, t0 = 0; *t; t++)
	if (*t == '/')
	    t0++;
    q = r = (char **) zalloc(sizeof(char **) * (t0 + 2));

    while ((t = strchr(s, '/'))) {
	*t = '\0';
	*q++ = ztrdup(s);
	*t = '/';
	while (*t == '/')
	    t++;
	if (!*t) {
	    *q = NULL;
	    return r;
	}
	s = t;
    }
    *q++ = ztrdup(s);
    *q = NULL;
    return r;
}

/* expands symlinks and .. or . expressions */
/* if flag = 0, only expand .. and . expressions */

/**/
int
xsymlinks(char *s, int flag)
{
    char **pp, **opp;
    char xbuf2[PATH_MAX], xbuf3[PATH_MAX];
    int t0;

    opp = pp = slashsplit(s);
    for (; *pp; pp++) {
	if (!strcmp(*pp, ".")) {
	    zsfree(*pp);
	    continue;
	}
	if (!strcmp(*pp, "..")) {
	    char *p;

	    zsfree(*pp);
	    if (!strcmp(xbuf, "/"))
		continue;
	    p = xbuf + strlen(xbuf);
	    while (*--p != '/');
	    *p = '\0';
	    continue;
	}
	if (unset(CHASELINKS)) {
	    strcat(xbuf, "/");
	    strcat(xbuf, *pp);
	    zsfree(*pp);
	    continue;
	}
	sprintf(xbuf2, "%s/%s", xbuf, *pp);
	t0 = readlink(xbuf2, xbuf3, PATH_MAX);
	if (t0 == -1 || !flag) {
	    strcat(xbuf, "/");
	    strcat(xbuf, *pp);
	    zsfree(*pp);
	} else {
	    xbuf3[t0] = '\0';	/* STUPID */
	    if (*xbuf3 == '/') {
		strcpy(xbuf, "");
		if (xsymlinks(xbuf3 + 1, flag))
		    return 1;
	    } else if (xsymlinks(xbuf3, flag))
		return 1;
	    zsfree(*pp);
	}
    }
    free(opp);
    return 0;
}

/* print a directory */

/**/
void
fprintdir(char *s, FILE *f)
{
    int t0;

    t0 = finddir(s);
    if (t0 == -1) {
	fputs(s, f);
    } else {
	putc('~', f);
	fputs(namdirs[t0].name, f);
	fputs(s + namdirs[t0].len, f);
    }
}

/* Returns the current username.  It caches the username *
 * and uid to try to avoid requerying the password files *
 * or NIS/NIS+ database.                                 */

/**/
char *
get_username(void)
{
    struct passwd *pswd;
    uid_t current_uid;
 
    current_uid = getuid();
    if (current_uid != cached_uid) {
	cached_uid = current_uid;
	zsfree(cached_username);
	if ((pswd = getpwuid(current_uid)))
	    cached_username = ztrdup(pswd->pw_name);
	else
	    cached_username = ztrdup("");
    }
    return cached_username;
}

/* Return the index of a named directory.  Returns -1 *
 * if no matching named directory is found.           */

/**/
int
findnameddir(char *s)
{
    int i;

    for (i = 0; i < userdirct; i++)
	if (!strcmp(namdirs[i].name, s))
	    return i;
    return -1;
}

/* See if a path has a named directory as its prefix. *
 * If passed a NULL argument, it will invalidate any  *
 * cached information.                                */

/**/
int
finddir(char *s)
{
    int t0, min, max;
    struct nameddir ss;
    static int last = -1;
    static char previous[PATH_MAX] = "\0";

    /* Invalidate directory cache if argument is NULL */
    if (!s) {
	*previous = '\0';
	return last = -1;
    }

    if (!strcmp(s, previous))
	return last;
    ss.name = "";
    ss.namelen = ss.homedir = 0;
    ss.dir = s;
    ss.len = strlen(s);
    if (lencmp(&ss, &namdirs[0]) < 0)
	return -1;

/* This binary search doesn't seem to make much difference but... */

    min = 0;
    max = userdirct;
    while ((t0 = (min + max) >> 1) != min)
	if (lencmp(&ss, &namdirs[t0]) < 0)
	    max = t0;
	else
	    min = t0;

/* Binary search alone doesn't work because we want the best match, *
 * i.e. the match latest in the namdirs array.                      */

    for (t0 = min; t0 >= 0; t0--)
	if (!dircmp(namdirs[t0].dir, s)) {
	    strcpy(previous, s);
	    return last = namdirs[t0].namelen<namdirs[t0].len ? t0 : -1;
	}
    return -1;
}

/* this static variable is used when addusernames() calls adduserdir() */
static nosortdirs = 0;

/* Add all the usernames in the password file/database *
 * to the named directory array                        */

/**/
void
addusernames(void)
{
#ifdef CACHE_USERNAMES
    struct passwd *pw;

    if (!usernamescached) {
	setpwent();
	nosortdirs = 1;

	/* loop through the password file/database *
	 * and add all entries returned.           */
	while ((pw = getpwent()) && !errflag)
	    adduserdir(dupstring(pw->pw_name), pw->pw_dir, 1, 1);

	endpwent();
	nosortdirs = 0;
	qsort((void *) namdirs, userdirct, sizeof *namdirs, lencmp);
	usernamescached = 1;
    }
#endif /* CACHE_USERNAMES */

    return;
}

/* add a named directory */

/**/
void
adduserdir(char *s, char *t, int ishomedir, int always)
{
    int t0 = -1, t1, t2;

    if (!interact)
	return;

    if (ishomedir) {
	if ((t0 = findnameddir(s)) != -1)
	    return;
    } else if (!t || *t != '/') {
	rmnameddir(s);
	return;
    }
    if (unset(AUTONAMEDIRS) && findnameddir(s) == -1 && always == 2)
	return;

    t2 = strlen(t);
    if (!ishomedir && t2 < PATH_MAX && (t0 = findnameddir(s)) != -1) {

    /* update value */

	zsfree(namdirs[t0].dir);
	namdirs[t0].dir = ztrdup(t);
	t1 = namdirs[t0].len;
	namdirs[t0].len = t2;
	if (!nosortdirs) {
	    if (t2 < t1)
		qsort(namdirs, t0 + 1, sizeof *namdirs, lencmp);
	    else
		qsort(namdirs + t0, userdirct - t0, sizeof *namdirs, lencmp);
	}
	finddir(NULL);		/* Invalidate directory cache */
	return;
    }
 /* add the name */

    if (userdirsz == userdirct) {
	userdirsz *= 2;
	namdirs = (Nameddir) realloc(namdirs, sizeof *namdirs * userdirsz);
	if (!namdirs)
	    return;
    }
    for (t0 = 0; t0 < userdirct; t0++)
	if (namdirs[t0].len > t2)
	    break;
    for (t1 = userdirct; t1 > t0; t1--)
	memcpy(namdirs + t1, namdirs + t1 - 1, sizeof *namdirs);
    namdirs[t0].len = t2;
    namdirs[t0].namelen = strlen(s);
    namdirs[t0].name = ztrdup(s);
    namdirs[t0].dir = ztrdup(t);
    namdirs[t0].homedir = ishomedir;
    userdirct++;
    if (t0 && namdirs[t0 - 1].len == t2 && !nosortdirs)
	qsort(namdirs, t0 + 1, sizeof *namdirs, lencmp);
    finddir(NULL);
}

/* Add a name, value pair as a named directory */

/**/
void
addnameddir(char *name, char *value)
{
    char buf[PATH_MAX], buf2[PATH_MAX];
    char *ptr;
    int dotsct;

    dotsct = fixdir(buf, value);
    if (*buf == '/') {
	adduserdir(name, buf, 0, 1);
	return;
    }
    if (dotsct) {
	strcpy(buf2, pwd);
	ptr = buf2 + strlen(buf2) - 1;
	while(dotsct--)
	    if (*--ptr == '/')
		break;
	if (ptr == buf2 || *buf)
	    ptr++;
	strcpy(ptr, buf);
    } else {
	if (*buf)
	    sprintf(buf2, "%s/%s", (!strcmp("/", pwd)) ? "" : pwd, buf);
	else
	    strcpy(buf2, pwd);
    }
    adduserdir(name, buf2, 0, 1);
}

/* get a named directory */

/**/
char *
getnameddir(char *name, int len)
{
    Param pm;
    char sav, *str, *returnval = NULL;
    struct passwd *pw;
    int i;

    if (len == 0)
	return dupstring(home);
    sav = name[len];
    name[len] = '\0';

    /* Check if it is already in the named directory table */
    if ((i = findnameddir(name)) != -1) {
	returnval = dupstring(namdirs[i].dir);
    } else if ((pm = (Param) paramtab->getnode(paramtab, name)) &&
    /* Check if there is a scalar parameter with this *
     * name whose value begins with a `/'.            */
	     (PM_TYPE(pm->flags) == PM_SCALAR) &&
	     (str = getsparam(name)) && *str == '/') {
	adduserdir(name, str, 0, 1);
	returnval = str;
    } else if ((pw = getpwnam(name))) {
    /* Else retrieve an entry from the password *
     * table/database for this user.            */
	str = xsymlink(pw->pw_dir);
	adduserdir(name, str, 1, 1);
	returnval = dupstring(str);
	zsfree(str);
    }
    name[len] = sav;
    return returnval;
}

/* Remove a named directory from the table.  *
 * Return 0 if successful.                   *
 * Return 1 if the name wasn't in the table. */

/**/
int
rmnameddir(char *name)
{
    int ind;

    if ((ind = findnameddir(name)) == -1)
	return 1;
    zsfree(namdirs[ind].name);
    zsfree(namdirs[ind].dir);
    userdirct--;
    for(; ind < userdirct; ind++)
	memcpy((void *) &namdirs[ind], (void *) &namdirs[ind + 1],
	       sizeof(*namdirs));
    finddir(NULL);   /* Invalidate directory cache */
    return 0;
}

/**/
int
dircmp(char *s, char *t)
{
    if (s) {
	for (; *s == *t; s++, t++)
	    if (!*s)
		return 0;
	if (!*s && *t == '/')
	    return 0;
    }
    return 1;
}

/**/
int
lencmp(const void *first, const void *sec)
{
#ifdef HIDE_NAMES
    /* Directories are sorted only by path length, so the name actually used
    for a directory is the last one added to the table that matches the longest
    possible part of the path. */
    return ((Nameddir) first)->len - ((Nameddir) sec)->len;
#else
# ifdef OLD_NAME_SORT
    int i;

    /* Directories are sorted first by path length, and then in reverse order
    by name length, so that the shortest available name for the longest
    possible matching path is used. */
    if ((i = ((Nameddir) first)->len - ((Nameddir) sec)->len))
	return i;
    else
	return ((Nameddir) sec)->namelen - ((Nameddir) first)->namelen;
# else
    /* Directories are sorted by the difference between the path length and the
    name length, so that directories are referred to in the shortest possible
    way. */
    return ((Nameddir) first)->len + ((Nameddir) sec)->namelen -
	((Nameddir) first)->namelen - ((Nameddir) sec)->len;
# endif
#endif
}

/* do pre-prompt stuff */

/**/
void
preprompt(void)
{
    int diff;
    List list;
    struct schedcmd *sch, *schl;

    /* If NOTIFY is not set, then check for completed *
     * jobs before we print the prompt.               */
    if (unset(NOTIFY))
	scanjobs();
    if (errflag)
	return;

    /* If a shell function named "precmd" exists, *
     * then execute it.                           */
    if ((list = getshfunc("precmd")))
	doshfunc(list, NULL, 0, 1);
    if (errflag)
	return;

    /* If 1) the parameter PERIOD exists, 2) the shell function     *
     * "periodic" exists, 3) it's been greater than PERIOD since we *
     * executed "periodic", then execute it now.                    */
    if (period && (time(NULL) > lastperiodic + period) &&
	(list = getshfunc("periodic"))) {
	doshfunc(list, NULL, 0, 1);
	lastperiodic = time(NULL);
    }
    if (errflag)
	return;

    /* If WATCH is set, then check for the *
     * specified login/logout events.      */
    if (watch) {
	diff = (int) difftime(time(NULL), lastwatch);
	if (diff > logcheck) {
	    dowatch();
	    lastwatch = time(NULL);
	}
    }
    if (errflag)
	return;

    /* Check mail */
    diff = (int) difftime(time(NULL), lastmailcheck);
    if (diff > mailcheck) {
	if (mailpath && *mailpath && **mailpath)
	    checkmailpath(mailpath);
	else if (mailfile && *mailfile) {
	    char *x[2];

	    x[0] = mailfile;
	    x[1] = NULL;
	    checkmailpath(x);
	}
	lastmailcheck = time(NULL);
    }

    /* Check scheduled commands */
    for (schl = (struct schedcmd *)&schedcmds, sch = schedcmds; sch;
	 sch = (schl = sch)->next) {
	if (sch->time < time(NULL)) {
	    execstring(sch->cmd, 0);
	    schl->next = sch->next;
	    zsfree(sch->cmd);
	    zfree(sch, sizeof(struct schedcmd));

	    sch = schl;
	}
	if (errflag)
	    return;
    }
}

/**/
void
checkmailpath(char **s)
{
    struct stat st;
    char *v, *u, c;

    while (*s) {
	for (v = *s; *v && *v != '?'; v++);
	c = *v;
	*v = '\0';
	if (c != '?')
	    u = NULL;
	else
	    u = v + 1;
	if (**s == 0) {
	    *v = c;
	    zerr("empty MAILPATH component: %s", *s, 0);
	} else if (stat(*s, &st) == -1) {
	    if (errno != ENOENT)
		zerr("%e: %s", *s, errno);
	} else if (S_ISDIR(st.st_mode)) {
	    LinkList l;
	    DIR *lock = opendir(*s);
	    char buf[PATH_MAX * 2], **arr, **ap;
	    struct dirent *de;
	    int ct = 1;

	    if (lock) {
		pushheap();
		heapalloc();
		l = newlinklist();
		readdir(lock);
		readdir(lock);
		while ((de = readdir(lock))) {
		    if (errflag)
			break;
		    if (u)
			sprintf(buf, "%s/%s?%s", *s, de->d_name, u);
		    else
			sprintf(buf, "%s/%s", *s, de->d_name);
		    addlinknode(l, dupstring(buf));
		    ct++;
		}
		closedir(lock);
		ap = arr = (char **) alloc(ct * sizeof(char *));

		while ((*ap++ = (char *)ugetnode(l)));
		checkmailpath(arr);
		popheap();
	    }
	} else {
	    if (st.st_size && st.st_atime <= st.st_mtime &&
		st.st_mtime > lastmailcheck)
		if (!u) {
		    fprintf(stderr, "You have new mail.\n");
		    fflush(stderr);
		} else {
		    char *z = u;

		    while (*z)
			if (*z == '$' && z[1] == '_') {
			    fprintf(stderr, "%s", *s);
			    z += 2;
			} else
			    fputc(*z++, stderr);
		    fputc('\n', stderr);
		    fflush(stderr);
		}
	    if (isset(MAILWARNING) && st.st_atime > st.st_mtime &&
		st.st_atime > lastmailcheck && st.st_size) {
		fprintf(stderr, "The mail in %s has been read.\n", *s);
		fflush(stderr);
	    }
	}
	*v = c;
	s++;
    }
}

/**/
void
freecompcond(void *a)
{
    Compcond cc = (Compcond) a;
    Compcond and, or, c;
    int n;

    for (c = cc; c; c = or) {
	or = c->or;
	for (; c; c = and) {
	    and = c->and;
	    if (c->type == CCT_POS ||
		c->type == CCT_NUMWORDS) {
		free(c->u.r.a);
		free(c->u.r.b);
	    } else if (c->type == CCT_CURSUF ||
		       c->type == CCT_CURPRE) {
		for (n = 0; n < c->n; n++)
		    if (c->u.s.s[n])
			zsfree(c->u.s.s[n]);
		free(c->u.s.s);
	    } else if (c->type == CCT_RANGESTR ||
		       c->type == CCT_RANGEPAT) {
		for (n = 0; n < c->n; n++)
		    if (c->u.l.a[n])
			zsfree(c->u.l.a[n]);
		free(c->u.l.a);
		for (n = 0; n < c->n; n++)
		    if (c->u.l.b[n])
			zsfree(c->u.l.b[n]);
		free(c->u.l.b);
	    } else {
		for (n = 0; n < c->n; n++)
		    if (c->u.s.s[n])
			zsfree(c->u.s.s[n]);
		free(c->u.s.p);
		free(c->u.s.s);
	    }
	    zfree(c, sizeof(struct compcond));
	}
    }
}

/**/
void
freestr(void *a)
{
    zsfree(a);
}

/**/
void
gettyinfo(struct ttyinfo *ti)
{
    if (SHTTY != -1) {
#ifdef HAVE_TERMIOS_H
# ifdef HAVE_TCGETATTR
	if (tcgetattr(SHTTY, &ti->tio) == -1)
# else
	if (ioctl(SHTTY, TCGETS, &ti->tio) == -1)
# endif
	    zerr("bad tcgets: %e", NULL, errno);
#else
# ifdef HAVE_TERMIO_H
	ioctl(SHTTY, TCGETA, &ti->tio);
# else
	ioctl(SHTTY, TIOCGETP, &ti->sgttyb);
	ioctl(SHTTY, TIOCLGET, &ti->lmodes);
	ioctl(SHTTY, TIOCGETC, &ti->tchars);
	ioctl(SHTTY, TIOCGLTC, &ti->ltchars);
# endif
#endif
#ifdef TIOCGWINSZ
/*	if (ioctl(SHTTY, TIOCGWINSZ, &ti->winsize) == -1)
	    	zerr("bad tiocgwinsz: %e",NULL,errno);*/
	ioctl(SHTTY, TIOCGWINSZ, (char *)&ti->winsize);
#endif
    }
}

/**/
void
settyinfo(struct ttyinfo *ti)
{
    if (SHTTY != -1) {
#ifdef HAVE_TERMIOS_H
# ifdef HAVE_TCGETATTR
#  ifndef TCSADRAIN
#   define TCSADRAIN 1	/* XXX Princeton's include files are screwed up */
#  endif
	tcsetattr(SHTTY, TCSADRAIN, &ti->tio);
    /* if (tcsetattr(SHTTY, TCSADRAIN, &ti->tio) == -1) */
# else
	ioctl(SHTTY, TCSETS, &ti->tio);
    /* if (ioctl(SHTTY, TCSETS, &ti->tio) == -1) */
# endif
	/*	zerr("settyinfo: %e",NULL,errno)*/ ;
#else
# ifdef HAVE_TERMIO_H
	ioctl(SHTTY, TCSETA, &ti->tio);
# else
	ioctl(SHTTY, TIOCSETN, &ti->sgttyb);
	ioctl(SHTTY, TIOCLSET, &ti->lmodes);
	ioctl(SHTTY, TIOCSETC, &ti->tchars);
	ioctl(SHTTY, TIOCSLTC, &ti->ltchars);
# endif
#endif
    }
}

#ifdef TIOCGWINSZ
extern winchanged;
#endif

/* check the size of the window and adjust if necessary */

/**/
void
adjustwinsize(void)
{
#ifdef TIOCGWINSZ
    int oldcols = columns, oldrows = lines;

    if (SHTTY == -1)
	return;

    ioctl(SHTTY, TIOCGWINSZ, (char *)&shttyinfo.winsize);
    if (shttyinfo.winsize.ws_col)
	columns = shttyinfo.winsize.ws_col;
    if (shttyinfo.winsize.ws_row)
	lines = shttyinfo.winsize.ws_row;
    if (oldcols != columns) {
	if (zleactive) {
	    resetneeded = winchanged = 1;
	    refresh();
	}
	setintenv("COLUMNS", columns);
    }
    if (oldrows != lines)
	setintenv("LINES", lines);
#endif   /*  TIOCGWINSZ */
}

/* move a fd to a place >= 10 */

/**/
int
movefd(int fd)
{
    int fe;

    if (fd == -1)
	return fd;
#ifdef F_DUPFD
    fe = fcntl(fd, F_DUPFD, 10);
#else
    if ((fe = dup(fd)) < 10)
	fe = movefd(fe);
#endif
    close(fd);
    return fe;
}

/* move fd x to y */

/**/
void
redup(int x, int y)
{
    if (x != y) {
	dup2(x, y);
	close(x);
    }
}

/* Check if a string contains a token */

/**/
int
has_token(const char *s)
{
    while(*s)
	if(itok(*s++))
	    return 1;
    return 0;
}

/* Delete a character in a string */
 
/**/
void
chuck(char *str)
{
    while ((str[0] = str[1]))
	str++;
}

/**/
int
tulower(int c)
{
    c &= 0xff;
    return (isupper(c) ? tolower(c) : c);
}

/**/
int
tuupper(int c)
{
    c &= 0xff;
    return (islower(c) ? toupper(c) : c);
}

/* copy len chars from t into s, and null terminate */

/**/
void
ztrncpy(char *s, char *t, int len)
{
    while (len--)
	*s++ = *t++;
    *s = '\0';
}

/* copy t into *s and update s */

/**/
void
strucpy(char **s, char *t)
{
    char *u = *s;

    while ((*u++ = *t++));
    *s = u - 1;
}

/**/
void
struncpy(char **s, char *t, int n)
{
    char *u = *s;

    while (n--)
	*u++ = *t++;
    *s = u;
    *u = '\0';
}

/* Return the number of elements in an array of pointers. *
 * It doesn't count the NULL pointer at the end.          */

/**/
int
arrlen(char **s)
{
    int count;

    for (count = 0; *s; s++, count++);
    return count;
}

/* Skip over a balanced pair of parenthesis. */

/**/
int
skipparens(char inpar, char outpar, char **s)
{
    int level;

    if (**s != inpar)
	return -1;

    for (level = 1; *++*s && level;)
	if (**s == inpar)
	   ++level;
	else if (**s == outpar)
	   --level;

   return level;
}

/* Convert string to long.  This function (without the z) *
 * is contained in the ANSI standard C library, but a lot *
 * of them seem to be broken.                             */

/**/
long
zstrtol(const char *s, char **t, int base)
{
    long ret = 0;
 
    if (!base)
	if (*s != '0')
	    base = 10;
	else if (*++s == 'x' || *s == 'X')
	    base = 16, s++;
	else
	    base = 8;
 
    if (base <= 10)
	for (; *s >= '0' && *s < ('0' + base); s++)
	    ret = ret * base + *s - '0';
    else
	for (; idigit(*s) || (*s >= 'a' && *s < ('a' + base - 10))
	     || (*s >= 'A' && *s < ('A' + base - 10)); s++)
	    ret = ret * base + (idigit(*s) ? (*s - '0') : (*s & 0x1f) + 9);
    if (t)
	*t = (char *)s;
    return ret;
}

/**/
int
checkrmall(char *s)
{
    fflush(stdin);
    if (*s == '/')
	fprintf(shout, "zsh: sure you want to delete all the files in %s? ",
		s);
    else
	fprintf(shout, "zsh: sure you want to delete all the files in %s/%s? ",
		(pwd[1]) ? pwd : "", s);
    fflush(shout);
    feep();
    return (getquery() == 'y');
}

/**/
int
getquery(void)
{
    char c, d;
    int isem = !strcmp(term, "emacs");

#ifdef FIONREAD
    int val;
#endif

    attachtty(mypgrp);
    if (!isem)
	setcbreak();

#ifdef FIONREAD
    ioctl(SHTTY, FIONREAD, (char *)&val);
    if (val) {
	if (!isem)
	    settyinfo(&shttyinfo);
	write(SHTTY, "n\n", 2);
	return 'n';
    }
#endif
    if (read(SHTTY, &c, 1) == 1)
	if (c == 'y' || c == 'Y' || c == '\t')
	    c = 'y';
    if (isem) {
	if (c != '\n')
	    while (read(SHTTY, &d, 1) == 1 && d != '\n');
    } else {
	settyinfo(&shttyinfo);
	if (c != '\n')
	    write(2, "\n", 1);
    }
    return (int)c;
}

static int d;
static char *guess, *best;

/**/
void
spscan(HashNode hn, int scanflags)
{
    int nd;

    nd = spdist(hn->nam, guess, (int) strlen(guess) / 4 + 1);
    if (nd <= d) {
	best = hn->nam;
	d = nd;
    }
}

/* spellcheck a word */
/* fix s and s2 ; if s2 is non-null, fix the history list too */

/**/
void
spckword(char **s, char **s2, int cmd, int ask)
{
    char *t, *u;
    int x;
    char ic = '\0';
    int ne;
    int preflen = 0;

    if (**s == '-' || **s == '%')
	return;
    if (!strcmp(*s, "in"))
	return;
    if (!(*s)[0] || !(*s)[1])
	return;
    if (shfunctab->getnode(shfunctab, *s) ||
	builtintab->getnode(builtintab, *s) ||
	cmdnamtab->getnode(cmdnamtab, *s) ||
	aliastab->getnode(aliastab, *s)  ||
	reswdtab->getnode(reswdtab, *s))
	return;
    else if (isset(HASHLISTALL)) {
	fullhash();
	if (cmdnamtab->getnode(cmdnamtab, *s))
	    return;
    }
    t = *s;
    if (*t == Tilde || *t == Equals || *t == String)
	t++;
    for (; *t; t++)
	if (itok(*t))
	    return;
    best = NULL;
    for (t = *s; *t; t++)
	if (*t == '/')
	    break;
    if (**s == Tilde && !*t)
	return;
    if (**s == String && !*t) {
	guess = *s + 1;
	if (*t || !ialpha(*guess))
	    return;
	ic = String;
	d = 100;
	scanhashtable(paramtab, 1, 0, 0, spscan, 0);
    } else if (**s == Equals) {
	if (*t)
	    return;
	if (hashcmd(guess = *s + 1, pathchecked))
	    return;
	d = 100;
	ic = Equals;
	scanhashtable(aliastab, 1, 0, 0, spscan, 0);
	scanhashtable(cmdnamtab, 1, 0, 0, spscan, 0);
    } else {
	guess = *s;
	if (*guess == Tilde || *guess == String) {
	    ic = *guess;
	    if (!*++t)
		return;
	    guess = dupstring(guess);
	    ne = noerrs;
	    noerrs = 1;
	    singsub(&guess);
	    noerrs = ne;
	    if (!guess)
		return;
	    preflen = strlen(guess) - strlen(t);
	}
	if (access(guess, F_OK) == 0)
	    return;
	if ((u = spname(guess)) != guess)
	    best = u;
	if (!*t && cmd) {
	    if (hashcmd(guess, pathchecked))
		return;
	    d = 100;
	    scanhashtable(reswdtab, 1, 0, 0, spscan, 0);
	    scanhashtable(aliastab, 1, 0, 0, spscan, 0);
	    scanhashtable(shfunctab, 1, 0, 0, spscan, 0);
	    scanhashtable(builtintab, 1, 0, 0, spscan, 0);
	    scanhashtable(cmdnamtab, 1, 0, 0, spscan, 0);
	}
    }
    if (errflag)
	return;
    if (best && (int)strlen(best) > 1 && strcmp(best, guess)) {
	if (ic) {
	    if (preflen) {
		/* do not correct the result of an expansion */
		if (strncmp(guess, best, preflen))
		    return;
		/* replace the temporarily expanded prefix with the original */
		u = (char *) ncalloc(t - *s + strlen(best + preflen) + 1);
		strncpy(u, *s, t - *s);
		strcpy(u + (t - *s), best + preflen);
	    } else {
		u = (char *) ncalloc(strlen(best) + 2);
		strcpy(u + 1, best);
	    }
	    best = u;
	    guess = *s;
	    *guess = *best = ztokens[ic - Pound];
	}
	if (ask) {
	    char *pptbuf;
	    int pptlen;
	    rstring = best;
	    Rstring = guess;
	    pptbuf = putprompt(sprompt, &pptlen, NULL, 1);
	    fwrite(pptbuf, pptlen, 1, stderr);
	    free(pptbuf);
	    fflush(stderr);
	    feep();
	    x = getquery();
	} else
	    x = 'y';
	if (x == 'y' || x == ' ') {
	    *s = dupstring(best);
	    if (s2) {
		hwrep(best);
		*s2 = dupstring(best);
	    }
	} else if (x == 'a') {
	    histdone |= HISTFLAG_NOEXEC;
	} else if (x == 'e') {
	    histdone |= HISTFLAG_NOEXEC | HISTFLAG_RECALL;
	}
	if (ic)
	    **s = ic;
    }
}

/**/
int
ztrftime(char *buf, int bufsize, char *fmt, struct tm *tm)
{
    static char *astr[] =
    {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
    static char *estr[] =
    {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
     "Aug", "Sep", "Oct", "Nov", "Dec"};
    static char *lstr[] =
    {"12", " 1", " 2", " 3", " 4", " 5", " 6", " 7", " 8", " 9",
     "10", "11"};
    char tmp[3];

#ifdef HAVE_STRFTIME
    char *origbuf = buf;
#endif

    tmp[0] = '%';
    tmp[2] = '\0';
    while (*fmt)
	if (*fmt == '%') {
	    fmt++;
	    switch (*fmt++) {
	    case 'a':
		strucpy(&buf, astr[tm->tm_wday]);
		break;
	    case 'b':
		strucpy(&buf, estr[tm->tm_mon]);
		break;
	    case 'd':
		*buf++ = '0' + tm->tm_mday / 10;
		*buf++ = '0' + tm->tm_mday % 10;
		break;
	    case 'e':
		if (tm->tm_mday > 9)
		    *buf++ = '0' + tm->tm_mday / 10;
		*buf++ = '0' + tm->tm_mday % 10;
		break;
	    case 'k':
		if (tm->tm_hour > 9)
		    *buf++ = '0' + tm->tm_hour / 10;
		*buf++ = '0' + tm->tm_hour % 10;
		break;
	    case 'l':
		strucpy(&buf, lstr[tm->tm_hour % 12]);
		break;
	    case 'm':
		*buf++ = '0' + (tm->tm_mon + 1) / 10;
		*buf++ = '0' + (tm->tm_mon + 1) % 10;
		break;
	    case 'M':
		*buf++ = '0' + tm->tm_min / 10;
		*buf++ = '0' + tm->tm_min % 10;
		break;
	    case 'p':
		*buf++ = (tm->tm_hour > 11) ? 'p' : 'a';
		*buf++ = 'm';
		break;
	    case 'S':
		*buf++ = '0' + tm->tm_sec / 10;
		*buf++ = '0' + tm->tm_sec % 10;
		break;
	    case 'y':
		*buf++ = '0' + tm->tm_year / 10;
		*buf++ = '0' + tm->tm_year % 10;
		break;
	    default:
#ifdef HAVE_STRFTIME
		*buf = '\0';
		tmp[1] = fmt[-1];
		strftime(buf, bufsize - strlen(origbuf), tmp, tm);
		buf += strlen(buf);
#else
		*buf++ = '%';
		*buf++ = fmt[-1];
#endif
		break;
	    }
	} else
	    *buf++ = *fmt++;
    *buf = '\0';
    return 0;
}

/**/
char *
join(char **arr, int delim)
{
    int len = 0;
    char **s, *ret, *ptr;
    static char *lastmem = NULL;

    for (s = arr; *s; s++)
	len += strlen(*s) + 1;
    if (!len)
	return "";
    zsfree(lastmem);
    lastmem = ptr = ret = (char *) zalloc(len);
    for (s = arr; *s; s++) {
	strucpy(&ptr, *s);
	*ptr++ = delim;
    }
    ptr[-1] = '\0';
    return ret;
}

/* Split a string containing a colon separated list *
 * of items into an array of strings.               */

/**/
char **
colonsplit(char *s)
{
    int ct;
    char *t, **ret, **ptr;

    for (t = s, ct = 0; *t; t++) /* count number of colons */
	if (*t == ':')
	    ct++;
    ptr = ret = (char **) zalloc(sizeof(char **) * (ct + 2));

    t = s;
    do {
	s = t;
        /* move t to point at next colon */
	for (; *t && *t != ':'; t++);
	*ptr = (char *) zalloc((t - s) + 1);
	ztrncpy(*ptr++, s, t - s);
    }
    while (*t++);
    *ptr = NULL;
    return ret;
}

/**/
char **
spacesplit(char *s)
{
    int ct;
    char *t, **ret, **ptr;

    for (t = s, ct = 0; *t; t++)
	if (isep(*t))
	    ct++;
    ptr = ret = (char **) ncalloc(sizeof(char **) * (ct + 2));

    t = s;
    do {
	for (s = t; *t && !isep(*t); t++);
	*ptr = (char *) ncalloc((t - s) + 1);
	ztrncpy(*ptr++, s, t - s);
    }
    while (*t++);
    *ptr = NULL;
    return ret;
}

/**/
int
findsep(char **s, char *sep)
{
    int i;
    char *t, *tt;

    if (!sep) {
	for (t = *s, i = 1; i && *t;) {
	    for (tt = ifs, i = 1; i && *tt; tt++)
		if (*tt == *t)
		    i = 0;
	    if (i)
		t++;
	}
	i = t - *s;
	*s = t;
	return i;
    }
    if (!sep[0]) {
	return **s ? ++*s, 1 : -1;
    }
    for (i = 0; **s; (*s)++, i++) {
	for (t = sep, tt = *s; *t && *tt && *t == *tt; t++, tt++);
	if (!*t)
	    return i;
    }
    return -1;
}

/**/
char *
findword(char **s, char *sep)
{
    char *r, *t, *tt;
    int f, sl;

    if (!**s)
	return NULL;

    if (sep) {
	sl = strlen(sep);
	r = *s;
	while (!(f = findsep(s, sep))) {
	    r = *s += sl;
	}
	return r;
    }
    for (t = *s, f = 1; f && *t;) {
	for (tt = ifs, f = 0; !f && *tt; tt++)
	    if (*tt == *t)
		f = 1;
	if (f)
	    t++;
    }
    *s = t;
    findsep(s, sep);
    return t;
}

/**/
int
wordcount(char *s, char *sep, int mul)
{
    int r = 1, sl, c, cc;
    char *t = s, *ti;

    if (sep) {
	sl = strlen(sep);
	for (; (c = findsep(&t, sep)) >= 0; t += sl)
	    if ((c && *(t + sl)) || mul)
		r++;
    } else {
	if (!mul)
	    for (c = 1; c && *t;) {
		for (c = 0, ti = ifs; !c && *ti; ti++)
		    if (*ti == *t)
			c = 1;
		if (c)
		    t++;
	    }
	if (!*t)
	    return 0;
	for (; *t; t++) {
	    for (c = 0, ti = ifs; !c && *ti; ti++)
		if (*ti == *t)
		    c = 1;
	    if (c && !mul) {
		for (cc = 1, t++; cc && *t;) {
		    for (cc = 0, ti = ifs; !cc && *ti; ti++)
			if (*ti == *t)
			    cc = 1;
		    if (cc)
			t++;
		}
		if (*t)
		    r++;
	    } else if (c)
		r++;
	}
    }
    return r;
}

/**/
char *
sepjoin(char **s, char *sep)
{
    char *r, *p, **t;
    int l, sl, elide = 0;
    static char *lastmem = NULL;
    char sepbuf[2];

    if (!*s)
	return "";
    if (!sep) {
	elide = 1;
	sep = sepbuf;
	sepbuf[0] = *ifs;
	sepbuf[1] = '\0';
    }
    sl = strlen(sep);
    for (t = s, l = 1 - sl; *t; l += strlen(*t) + sl, t++);
    if (l == 1)
	return "";
    zsfree(lastmem);
    lastmem = r = p = (char *) zalloc(l);
    t = s;
    if (elide)
	while (*t && !**t)
	    t++;
    for (; *t; t++) {
	strucpy(&p, *t);
	if (t[1] && (!elide || t[1][0]))
	    strucpy(&p, sep);
    }
    return r;
}

/**/
char **
sepsplit(char *s, char *sep)
{
    int n, sl, f;
    char *t, *tt, **r, **p;

    if (!sep)
	return spacesplit(s);

    sl = strlen(sep);
    n = wordcount(s, sep, 1);
    r = p = (char **) ncalloc((n + 1) * sizeof(char *));

    for (t = s; n--;) {
	tt = t;
	f = findsep(&t, sep);	/* f set not but not used? ++jhi; */
	*p = (char *) ncalloc(t - tt + 1);
	strncpy(*p, tt, t - tt);
	(*p)[t - tt] = '\0';
	p++;
	t += sl;
    }
    *p = NULL;

    return r;
}

/* Get the definition of a shell function */

/**/
List
getshfunc(char *nam)
{
    Shfunc shf;
    List l;

    if ((shf = (Shfunc) shfunctab->getnode(shfunctab, nam))) {
	/* if autoloaded and currently undefined */
	if (shf->flags & PM_UNDEFINED) {
	    if (!(l = getfpfunc(nam))) {
		zerr("function not found: %s", nam, 0);
		return NULL;
	    }
	    shf->flags &= ~PM_UNDEFINED;
	    permalloc();
	    shf->funcdef = (List) dupstruct(l);
	    lastalloc();
	}
	return shf->funcdef;
    } else {
	return NULL;
    }
}

/* allocate a tree element */

static int sizetab[N_COUNT] =
{
    sizeof(struct list),
    sizeof(struct sublist),
    sizeof(struct pline),
    sizeof(struct cmd),
    sizeof(struct redir),
    sizeof(struct cond),
    sizeof(struct forcmd),
    sizeof(struct casecmd),
    sizeof(struct ifcmd),
    sizeof(struct whilecmd),
    sizeof(struct varasg)};

static int flagtab[N_COUNT] =
{
    NT_SET(N_LIST, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_SUBLIST, 2, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_PLINE, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_CMD, 2, NT_STR | NT_LIST, NT_NODE, NT_NODE | NT_LIST, NT_NODE | NT_LIST),
    NT_SET(N_REDIR, 3, NT_STR, 0, 0, 0),
    NT_SET(N_COND, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_FOR, 1, NT_STR, NT_NODE, 0, 0),
    NT_SET(N_CASE, 0, NT_STR | NT_ARR, NT_NODE | NT_ARR, 0, 0),
    NT_SET(N_IF, 0, NT_NODE | NT_ARR, NT_NODE | NT_ARR, 0, 0),
    NT_SET(N_WHILE, 1, NT_NODE, NT_NODE, 0, 0),
    NT_SET(N_VARASG, 1, NT_STR, NT_STR, NT_STR | NT_LIST, 0)};

/**/
void *
allocnode(int type)
{
    struct node *n;

    n = (struct node *) alloc(sizetab[type]);
    memset((void *) n, 0, sizetab[type]);
    n->ntype = flagtab[type];
    if (useheap)
	n->ntype |= NT_HEAP;

    return (void *) n;
}

/**/
void *
dupstruct(void *a)
{
    struct node *n, *r;

    n = (struct node *) a;
    if (!a || ((List) a) == &dummy_list)
	return (void *) a;

    if ((n->ntype & NT_HEAP) && !useheap) {
	heapalloc();
	n = (struct node *) dupstruct2((void *) n);
	permalloc();
	n = simplifystruct(n);
    }
    r = (struct node *)dupstruct2((void *) n);

    if (!(n->ntype & NT_HEAP) && useheap)
	r = expandstruct(r, N_LIST);

    return (void *) r;
}

/**/
struct node *
simplifystruct(struct node *n)
{
    if (!n || ((List) n) == &dummy_list)
	return n;

    switch (NT_TYPE(n->ntype)) {
    case N_LIST:
	{
	    List l = (List) n;

	    l->left = (Sublist) simplifystruct((struct node *)l->left);
	    if ((l->type & Z_SYNC) && !l->right)
		return (struct node *)l->left;
	}
	break;
    case N_SUBLIST:
	{
	    Sublist sl = (Sublist) n;

	    sl->left = (Pline) simplifystruct((struct node *)sl->left);
	    if (sl->type == END && !sl->flags && !sl->right)
		return (struct node *)sl->left;
	}
	break;
    case N_PLINE:
	{
	    Pline pl = (Pline) n;

	    pl->left = (Cmd) simplifystruct((struct node *)pl->left);
	    if (pl->type == END && !pl->right)
		return (struct node *)pl->left;
	}
	break;
    case N_CMD:
	{
	    Cmd c = (Cmd) n;
	    int i = 0;

	    if (empty(c->args))
		c->args = NULL, i++;
	    if (empty(c->redir))
		c->redir = NULL, i++;
	    if (empty(c->vars))
		c->vars = NULL, i++;

	    c->u.list = (List) simplifystruct((struct node *)c->u.list);
	    if (i == 3 && !c->flags &&
		(c->type == CWHILE || c->type == CIF ||
		 c->type == COND))
		return (struct node *)c->u.list;
	}
	break;
    case N_FOR:
	{
	    Forcmd f = (Forcmd) n;

	    f->list = (List) simplifystruct((struct node *)f->list);
	}
	break;
    case N_CASE:
	{
	    struct casecmd *c = (struct casecmd *)n;
	    List *l;

	    for (l = c->lists; *l; l++)
		*l = (List) simplifystruct((struct node *)*l);
	}
	break;
    case N_IF:
	{
	    struct ifcmd *i = (struct ifcmd *)n;
	    List *l;

	    for (l = i->ifls; *l; l++)
		*l = (List) simplifystruct((struct node *)*l);
	    for (l = i->thenls; *l; l++)
		*l = (List) simplifystruct((struct node *)*l);
	}
	break;
    case N_WHILE:
	{
	    struct whilecmd *w = (struct whilecmd *)n;

	    w->cont = (List) simplifystruct((struct node *)w->cont);
	    w->loop = (List) simplifystruct((struct node *)w->loop);
	}
    }

    return n;
}

/**/
struct node *
expandstruct(struct node *n, int exp)
{
    struct node *m;

    if (!n || ((List) n) == &dummy_list)
	return n;

    if (exp != N_COUNT && exp != NT_TYPE(n->ntype)) {
	switch (exp) {
	case N_LIST:
	    {
		List l;

		m = (struct node *) allocnode(N_LIST);
		l = (List) m;
		l->type = Z_SYNC;
		l->left = (Sublist) expandstruct(n, N_SUBLIST);

		return (struct node *)l;
	    }
	case N_SUBLIST:
	    {
		Sublist sl;

		m = (struct node *) allocnode(N_SUBLIST);
		sl = (Sublist) m;
		sl->type = END;
		sl->left = (Pline) expandstruct(n, N_PLINE);

		return (struct node *)sl;
	    }
	case N_PLINE:
	    {
		Pline pl;

		m = (struct node *) allocnode(N_PLINE);
		pl = (Pline) m;
		pl->type = END;
		pl->left = (Cmd) expandstruct(n, N_CMD);

		return (struct node *)pl;
	    }
	case N_CMD:
	    {
		Cmd c;

		m = (struct node *) allocnode(N_CMD);
		c = (Cmd) m;
		switch (NT_TYPE(n->ntype)) {
		case N_WHILE:
		    c->type = CWHILE;
		    break;
		case N_IF:
		    c->type = CIF;
		    break;
		case N_COND:
		    c->type = COND;
		}
		c->u.list = (List) expandstruct(n, NT_TYPE(n->ntype));
		c->args = newlinklist();
		c->vars = newlinklist();
		c->redir = newlinklist();

		return (struct node *)c;
	    }
	}
    } else
	switch (NT_TYPE(n->ntype)) {
	case N_LIST:
	    {
		List l = (List) n;

		l->left = (Sublist) expandstruct((struct node *)l->left,
						 N_SUBLIST);
		l->right = (List) expandstruct((struct node *)l->right,
					       N_LIST);
	    }
	    break;
	case N_SUBLIST:
	    {
		Sublist sl = (Sublist) n;

		sl->left = (Pline) expandstruct((struct node *)sl->left,
						N_PLINE);
		sl->right = (Sublist) expandstruct((struct node *)sl->right,
						   N_SUBLIST);
	    }
	    break;
	case N_PLINE:
	    {
		Pline pl = (Pline) n;

		pl->left = (Cmd) expandstruct((struct node *)pl->left,
					      N_CMD);
		pl->right = (Pline) expandstruct((struct node *)pl->right,
						 N_PLINE);
	    }
	    break;
	case N_CMD:
	    {
		Cmd c = (Cmd) n;

		if (!c->args)
		    c->args = newlinklist();
		if (!c->vars)
		    c->vars = newlinklist();
		if (!c->redir)
		    c->redir = newlinklist();

		switch (c->type) {
		case CFOR:
		case CSELECT:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_FOR);
		    break;
		case CWHILE:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_WHILE);
		    break;
		case CIF:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_IF);
		    break;
		case CCASE:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_CASE);
		    break;
		case COND:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_COND);
		    break;
		case ZCTIME:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_SUBLIST);
		    break;
		default:
		    c->u.list = (List) expandstruct((struct node *)c->u.list,
						    N_LIST);
		}
	    }
	    break;
	case N_FOR:
	    {
		Forcmd f = (Forcmd) n;

		f->list = (List) expandstruct((struct node *)f->list,
					      N_LIST);
	    }
	    break;
	case N_CASE:
	    {
		struct casecmd *c = (struct casecmd *)n;
		List *l;

		for (l = c->lists; *l; l++)
		    *l = (List) expandstruct((struct node *)*l, N_LIST);
	    }
	    break;
	case N_IF:
	    {
		struct ifcmd *i = (struct ifcmd *)n;
		List *l;

		for (l = i->ifls; *l; l++)
		    *l = (List) expandstruct((struct node *)*l, N_LIST);
		for (l = i->thenls; *l; l++)
		    *l = (List) expandstruct((struct node *)*l, N_LIST);
	    }
	    break;
	case N_WHILE:
	    {
		struct whilecmd *w = (struct whilecmd *)n;

		w->cont = (List) expandstruct((struct node *)w->cont,
					      N_LIST);
		w->loop = (List) expandstruct((struct node *)w->loop,
					      N_LIST);
	    }
	}

    return n;
}

/* duplicate a syntax tree node of given type, argument number */

/**/
void *
dupnode(int type, void *a, int argnum)
{
    if (!a)
	return NULL;
    switch (NT_N(type, argnum)) {
    case NT_NODE:
	return (void *) dupstruct2(a);
    case NT_STR:
	return (useheap) ? ((void *) dupstring(a)) :
	    ((void *) ztrdup(a));
    case NT_LIST | NT_NODE:
	if (type & NT_HEAP)
	    if (useheap)
		return (void *) duplist(a, (VFunc) dupstruct2);
	    else
		return (void *) list2arr(a, (VFunc) dupstruct2);
	else if (useheap)
	    return (void *) arr2list(a, (VFunc) dupstruct2);
	else
	    return (void *) duparray(a, (VFunc) dupstruct2);
    case NT_LIST | NT_STR:
	if (type & NT_HEAP)
	    if (useheap)
		return (void *) duplist(a, (VFunc) dupstring);
	    else
		return (void *) list2arr(a, (VFunc) ztrdup);
	else if (useheap)
	    return (void *) arr2list(a, (VFunc) dupstring);
	else
	    return (void *) duparray(a, (VFunc) ztrdup);
    case NT_NODE | NT_ARR:
	return (void *) duparray(a, (VFunc) dupstruct2);
    case NT_STR | NT_ARR:
	return (void *) duparray(a, (VFunc) (useheap ? dupstring : ztrdup));
    default:
	abort();
    }
}

/* Free a syntax tree node of given type, argument number */

/**/
void
freetreenode(int type, void *a, int argnum)
{
    if (!a)
	return;
    switch (NT_N(type, argnum)) {
    case NT_NODE:
	freestruct(a);
	break;
    case NT_STR:
	zsfree(a);
	break;
    case NT_LIST | NT_NODE:
    case NT_NODE | NT_ARR:
	{
	    char **p = (char **)a;

	    while (*p)
		freestruct(*p++);
	    free(a);
	}
	break;
    case NT_LIST | NT_STR:
    case NT_STR | NT_ARR:
	freearray(a);
	break;
    default:
	abort();
    }
}

/* duplicate a syntax tree */

/**/
void **
dupstruct2(void *a)
{
    struct node *n = (struct node *)a, *m;
    int type;

    if (!n || ((List) n) == &dummy_list)
	return a;
    type = n->ntype;
    m = (struct node *) alloc(sizetab[NT_TYPE(type)]);
    m->ntype = (type & ~NT_HEAP);
    if (useheap)
	m->ntype |= NT_HEAP;
    switch (NT_TYPE(type)) {
    case N_LIST:
	{
	    List nl = (List) n;
	    List ml = (List) m;

	    ml->type = nl->type;
	    ml->left = (Sublist) dupnode(type, nl->left, 0);
	    ml->right = (List) dupnode(type, nl->right, 1);
	}
	break;
    case N_SUBLIST:
	{
	    Sublist nsl = (Sublist) n;
	    Sublist msl = (Sublist) m;

	    msl->type = nsl->type;
	    msl->flags = nsl->flags;
	    msl->left = (Pline) dupnode(type, nsl->left, 0);
	    msl->right = (Sublist) dupnode(type, nsl->right, 1);
	}
	break;
    case N_PLINE:
	{
	    Pline npl = (Pline) n;
	    Pline mpl = (Pline) m;

	    mpl->type = npl->type;
	    mpl->left = (Cmd) dupnode(type, npl->left, 0);
	    mpl->right = (Pline) dupnode(type, npl->right, 1);
	}
	break;
    case N_CMD:
	{
	    Cmd nc = (Cmd) n;
	    Cmd mc = (Cmd) m;

	    mc->type = nc->type;
	    mc->flags = nc->flags;
	    mc->lineno = nc->lineno;
	    mc->args = (LinkList) dupnode(type, nc->args, 0);
	    mc->u.generic = (void *) dupnode(type, nc->u.generic, 1);
	    mc->redir = (LinkList) dupnode(type, nc->redir, 2);
	    mc->vars = (LinkList) dupnode(type, nc->vars, 3);
	}
	break;
    case N_REDIR:
	{
	    Redir nr = (Redir) n;
	    Redir mr = (Redir) m;

	    mr->type = nr->type;
	    mr->fd1 = nr->fd1;
	    mr->fd2 = nr->fd2;
	    mr->name = (char *)dupnode(type, nr->name, 0);
	}
	break;
    case N_COND:
	{
	    Cond nco = (Cond) n;
	    Cond mco = (Cond) m;

	    mco->type = nco->type;
	    mco->left = (void *) dupnode(type, nco->left, 0);
	    mco->right = (void *) dupnode(type, nco->right, 1);
	}
	break;
    case N_FOR:
	{
	    Forcmd nf = (Forcmd) n;
	    Forcmd mf = (Forcmd) m;

	    mf->inflag = nf->inflag;
	    mf->name = (char *)dupnode(type, nf->name, 0);
	    mf->list = (List) dupnode(type, nf->list, 1);
	}
	break;
    case N_CASE:
	{
	    struct casecmd *ncc = (struct casecmd *)n;
	    struct casecmd *mcc = (struct casecmd *)m;

	    mcc->pats = (char **)dupnode(type, ncc->pats, 0);
	    mcc->lists = (List *) dupnode(type, ncc->lists, 1);
	}
	break;
    case N_IF:
	{
	    struct ifcmd *nic = (struct ifcmd *)n;
	    struct ifcmd *mic = (struct ifcmd *)m;

	    mic->ifls = (List *) dupnode(type, nic->ifls, 0);
	    mic->thenls = (List *) dupnode(type, nic->thenls, 1);

	}
	break;
    case N_WHILE:
	{
	    struct whilecmd *nwc = (struct whilecmd *)n;
	    struct whilecmd *mwc = (struct whilecmd *)m;

	    mwc->cond = nwc->cond;
	    mwc->cont = (List) dupnode(type, nwc->cont, 0);
	    mwc->loop = (List) dupnode(type, nwc->loop, 1);
	}
	break;
    case N_VARASG:
	{
	    Varasg nva = (Varasg) n;
	    Varasg mva = (Varasg) m;

	    mva->type = nva->type;
	    mva->name = (char *)dupnode(type, nva->name, 0);
	    mva->str = (char *)dupnode(type, nva->str, 1);
	    mva->arr = (LinkList) dupnode(type, nva->arr, 2);
	}
	break;
    }
    return (void **) m;
}

/* free a syntax tree */

/**/
void
freestruct(void *a)
{
    struct node *n = (struct node *)a;
    int type;

    if (!n || ((List) n) == &dummy_list)
	return;

    type = n->ntype;
    switch (NT_TYPE(type)) {
    case N_LIST:
	{
	    List nl = (List) n;

	    freetreenode(type, nl->left, 0);
	    freetreenode(type, nl->right, 1);
	}
	break;
    case N_SUBLIST:
	{
	    Sublist nsl = (Sublist) n;

	    freetreenode(type, nsl->left, 0);
	    freetreenode(type, nsl->right, 1);
	}
	break;
    case N_PLINE:
	{
	    Pline npl = (Pline) n;

	    freetreenode(type, npl->left, 0);
	    freetreenode(type, npl->right, 1);
	}
	break;
    case N_CMD:
	{
	    Cmd nc = (Cmd) n;

	    freetreenode(type, nc->args, 0);
	    freetreenode(type, nc->u.generic, 1);
	    freetreenode(type, nc->redir, 2);
	    freetreenode(type, nc->vars, 3);
	}
	break;
    case N_REDIR:
	{
	    Redir nr = (Redir) n;

	    freetreenode(type, nr->name, 0);
	}
	break;
    case N_COND:
	{
	    Cond nco = (Cond) n;

	    freetreenode(type, nco->left, 0);
	    freetreenode(type, nco->right, 1);
	}
	break;
    case N_FOR:
	{
	    Forcmd nf = (Forcmd) n;

	    freetreenode(type, nf->name, 0);
	    freetreenode(type, nf->list, 1);
	}
	break;
    case N_CASE:
	{
	    struct casecmd *ncc = (struct casecmd *)n;

	    freetreenode(type, ncc->pats, 0);
	    freetreenode(type, ncc->lists, 1);
	}
	break;
    case N_IF:
	{
	    struct ifcmd *nic = (struct ifcmd *)n;

	    freetreenode(type, nic->ifls, 0);
	    freetreenode(type, nic->thenls, 1);

	}
	break;
    case N_WHILE:
	{
	    struct whilecmd *nwc = (struct whilecmd *)n;

	    freetreenode(type, nwc->cont, 0);
	    freetreenode(type, nwc->loop, 1);
	}
	break;
    case N_VARASG:
	{
	    Varasg nva = (Varasg) n;

	    freetreenode(type, nva->name, 0);
	    freetreenode(type, nva->str, 1);
	    freetreenode(type, nva->arr, 2);
	}
	break;
    }
    zfree(n, sizetab[NT_TYPE(type)]);
}

/**/
LinkList
duplist(LinkList l, VFunc func)
{
    LinkList ret;
    LinkNode node;

    ret = newlinklist();
    for (node = firstnode(l); node; incnode(node))
	addlinknode(ret, func(getdata(node)));
    return ret;
}

/**/
char **
duparray(char **arr, VFunc func)
{
    char **ret, **rr;

    ret = (char **) alloc((arrlen(arr) + 1) * sizeof(char *));
    for (rr = ret; *arr;)
	*rr++ = (char *)func(*arr++);
    *rr = NULL;

    return ret;
}

/**/
char **
list2arr(LinkList l, VFunc func)
{
    char **arr, **r;
    LinkNode n;

    arr = r = (char **) alloc((countlinknodes(l) + 1) * sizeof(char *));

    for (n = firstnode(l); n; incnode(n))
	*r++ = (char *)func(getdata(n));
    *r = NULL;

    return arr;
}

/**/
LinkList
arr2list(char **arr, VFunc func)
{
    LinkList l = newlinklist();

    while (*arr)
	addlinknode(l, func(*arr++));

    return l;
}

/**/
char **
mkarray(char *s)
{
    char **t = (char **) zalloc((s) ? (2 * sizeof s) : (sizeof s));

    if ((*t = s))
	t[1] = NULL;
    return t;
}

/**/
void
feep(void)
{
    if (unset(NOBEEP))
	write(2, "\07", 1);
}

/**/
void
freearray(char **s)
{
    char **t = s;

    while (*s)
	zsfree(*s++);
    free(t);
}

/**/
int
equalsplit(char *s, char **t)
{
    for (; *s && *s != '='; s++);
    if (*s == '=') {
	*s++ = '\0';
	*t = s;
	return 1;
    }
    return 0;
}

/* see if the right side of a list is trivial */

/**/
void
simplifyright(List l)
{
    Cmd c;

    if (l == &dummy_list || !l->right)
	return;
    if (l->right->right || l->right->left->right ||
	l->right->left->flags || l->right->left->left->right ||
	l->left->flags)
	return;
    c = l->left->left->left;
    if (c->type != SIMPLE || nonempty(c->args) || nonempty(c->redir)
	|| nonempty(c->vars))
	return;
    l->right = NULL;
    return;
}

/* initialize the ztypes table */

/**/
void
inittyptab(void)
{
    int t0;
    char *s;

    for (t0 = 0; t0 != 256; t0++)
	typtab[t0] = 0;
    for (t0 = 0; t0 != 32; t0++)
	typtab[t0] = typtab[t0 + 128] = ICNTRL;
    typtab[127] = ICNTRL;
    for (t0 = '0'; t0 <= '9'; t0++)
	typtab[t0] = IDIGIT | IALNUM | IWORD | IIDENT | IUSER;
    for (t0 = 'a'; t0 <= 'z'; t0++)
	typtab[t0] = typtab[t0 - 'a' + 'A'] = IALPHA | IALNUM | IIDENT | IUSER | IWORD;
    for (t0 = 0240; t0 != 0400; t0++)
	typtab[t0] = IALPHA | IALNUM | IIDENT | IUSER | IWORD;
    typtab['_'] = IIDENT | IUSER;
    typtab['-'] = IUSER;
    typtab[' '] |= IBLANK | INBLANK;
    typtab['\t'] |= IBLANK | INBLANK;
    typtab['\n'] |= INBLANK;
    for (t0 = (int)STOUC(Pound); t0 <= (int)STOUC(Nularg); t0++)
	typtab[t0] |= ITOK;
    for (s = ifs; *s; s++)
	typtab[(int)(unsigned char)*s] |= ISEP;
    for (s = wordchars; *s; s++)
	typtab[(int)(unsigned char)*s] |= IWORD;
    for (s = SPECCHARS; *s; s++)
	typtab[(int)(unsigned char)*s] |= ISPECIAL;
}

/**/
char **
arrdup(char **s)
{
    char **x, **y;

    y = x = (char **) ncalloc(sizeof(char *) * (arrlen(s) + 1));

    while ((*x++ = dupstring(*s++)));
    return y;
}

/**/
char *
spname(char *oldname)
{
    char *p, spnameguess[PATH_MAX + 1], spnamebest[PATH_MAX + 1];
    static char newname[PATH_MAX + 1];
    char *new = newname, *old;

    old = oldname;
    for (;;) {
	while (*old == '/')
	    *new++ = *old++;
	*new = '\0';
	if (*old == '\0')
	    return newname;
	p = spnameguess;
	for (; *old != '/' && *old != '\0'; old++)
	    if (p < spnameguess + PATH_MAX)
		*p++ = *old;
	*p = '\0';
	if (mindist(newname, spnameguess, spnamebest) >= 3)
	    return NULL;
	for (p = spnamebest; (*new = *p++);)
	    new++;
    }
}

/**/
int
mindist(char *dir, char *mindistguess, char *mindistbest)
{
    int mindistd, nd;
    DIR *dd;
    struct dirent *de;
    char buf[PATH_MAX];

    if (dir[0] == '\0')
	dir = ".";
    mindistd = 100;
    sprintf(buf, "%s/%s", dir, mindistguess);
    if (access(buf, F_OK) == 0) {
	strcpy(mindistbest, mindistguess);
	return 0;
    }
    if (!(dd = opendir(dir)))
	return mindistd;
    while ((de = readdir(dd))) {
	nd = spdist(de->d_name, mindistguess,
		    (int)strlen(mindistguess) / 4 + 1);
	if (nd <= mindistd) {
	    strcpy(mindistbest, de->d_name);
	    mindistd = nd;
	    if (mindistd == 0)
		break;
	}
    }
    closedir(dd);
    return mindistd;
}

/**/
int
spdist(char *s, char *t, int thresh)
{
    char *p, *q;
    char *keymap =
    "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\
\t1234567890-=\t\
\tqwertyuiop[]\t\
\tasdfghjkl;'\n\t\
\tzxcvbnm,./\t\t\t\
\n\n\n\n\n\n\n\n\n\n\n\n\n\n\
\t!@#$%^&*()_+\t\
\tQWERTYUIOP{}\t\
\tASDFGHJKL:\"\n\t\
\tZXCVBNM<>?\n\n\t\
\n\n\n\n\n\n\n\n\n\n\n\n\n\n";

    if (!strcmp(s, t))
	return 0;
/* any number of upper/lower mistakes allowed (dist = 1) */
    for (p = s, q = t; *p && tulower(*p) == tulower(*q); p++, q++);
    if (!*p && !*q)
	return 1;
    if (!thresh)
	return 200;
    for (p = s, q = t; *p && *q; p++, q++)
	if (*p == *q)
	    continue;		/* don't consider "aa" transposed, ash */
	else if (p[1] == q[0] && q[1] == p[0])	/* transpositions */
	    return spdist(p + 2, q + 2, thresh - 1) + 1;
	else if (p[1] == q[0])	/* missing letter */
	    return spdist(p + 1, q + 0, thresh - 1) + 2;
	else if (p[0] == q[1])	/* missing letter */
	    return spdist(p + 0, q + 1, thresh - 1) + 2;
	else if (*p != *q)
	    break;
    if ((!*p && strlen(q) == 1) || (!*q && strlen(p) == 1))
	return 2;
    for (p = s, q = t; *p && *q; p++, q++)
	if (p[0] != q[0] && p[1] == q[1]) {
	    int t0;
	    char *z;

	/* mistyped letter */

	    if (!(z = strchr(keymap, p[0])) || *z == '\n' || *z == '\t')
		return spdist(p + 1, q + 1, thresh - 1) + 1;
	    t0 = z - keymap;
	    if (*q == keymap[t0 - 15] || *q == keymap[t0 - 14] ||
		*q == keymap[t0 - 13] ||
		*q == keymap[t0 - 1] || *q == keymap[t0 + 1] ||
		*q == keymap[t0 + 13] || *q == keymap[t0 + 14] ||
		*q == keymap[t0 + 15])
		return spdist(p + 1, q + 1, thresh - 1) + 2;
	    return 200;
	} else if (*p != *q)
	    break;
    return 200;
}

/* set cbreak mode, or the equivalent */

/**/
void
setcbreak(void)
{
    struct ttyinfo ti;

    ti = shttyinfo;
#ifdef HAS_TIO
    ti.tio.c_lflag &= ~ICANON;
    ti.tio.c_cc[VMIN] = 1;
    ti.tio.c_cc[VTIME] = 0;
#else
    ti.sgttyb.sg_flags |= CBREAK;
#endif
    settyinfo(&ti);
}

/* give the tty to some process */

/**/
void
attachtty(pid_t pgrp)
{
    static int ep = 0;

    if (jobbing) {
#ifdef HAVE_TCSETPGRP
	if (SHTTY != -1 && tcsetpgrp(SHTTY, pgrp) == -1 && !ep)
#else
# if ardent
	if (SHTTY != -1 && setpgrp() == -1 && !ep)
# else
	int arg = pgrp;

	if (SHTTY != -1 && ioctl(SHTTY, TIOCSPGRP, &arg) == -1 && !ep)
# endif
#endif
	{
	    if (kill(pgrp, 0) == -1)
		attachtty(mypgrp);
	    else {
		zerr("can't set tty pgrp: %e", NULL, errno);
		fflush(stderr);
		opts[MONITOR] = OPT_UNSET;
		ep = 1;
		errflag = 0;
	    }
	}
    }
}

/* get the process group associated with the tty */

/**/
pid_t
gettygrp(void)
{
    pid_t arg;

    if (SHTTY == -1)
	return -1;

#ifdef HAVE_TCSETPGRP
    arg = tcgetpgrp(SHTTY);
#else
    ioctl(SHTTY, TIOCGPGRP, &arg);
#endif

    return arg;
}

/* Return the output baudrate */

/**/
long
getbaudrate(struct ttyinfo *shttyinfo)
{
    int speedcode;
#ifdef HAS_TIO
    long tempbaud;

# if defined(HAVE_TCGETATTR) && defined(HAVE_TERMIOS_H)
    tempbaud = cfgetospeed(&shttyinfo->tio);
    if (tempbaud >= 100)
        return tempbaud;
    else
        speedcode = (int) tempbaud;
# else
    speedcode = shttyinfo->tio.c_cflag & CBAUD;
# endif
#else
    speedcode = shttyinfo->sgttyb.sg_ospeed;
#endif

    switch (speedcode) {
    case B0:
	return (0L);
    case B50:
	return (50L);
    case B75:
	return (75L);
    case B110:
	return (110L);
    case B134:
	return (134L);
    case B150:
	return (150L);
    case B200:
	return (200L);
    case B300:
	return (300L);
    case B600:
	return (600L);
#ifdef _B900
    case _B900:
	return (900L);
#endif
    case B1200:
	return (1200L);
    case B1800:
	return (1800L);
    case B2400:
	return (2400L);
#ifdef _B3600
    case _B3600:
	return (3600L);
#endif
    case B4800:
	return (4800L);
#ifdef _B7200
    case _B7200:
	return (7200L);
#endif
    case B9600:
	return (9600L);
#ifdef B19200
    case B19200:
	return (19200L);
#else
# ifdef EXTA
    case EXTA:
	return (19200L);
# endif
#endif
#ifdef B38400
    case B38400:
	return (38400L);
#else
# ifdef EXTB
    case EXTB:
	return (38400L);
# endif
#endif
    default:
	break;
    }
    return (0L);
}
