/* This file is part of
 * ======================================================
 * 
 *           LyX, the High Level Word Processor 	 
 *	    Copyright (C) 1995 Matthias Ettrich
 *
 *           This file is Copyleft (C) 1996
 *           Lars Gullik Bjnnes
 *
 *======================================================
 */

#include "config.h"
#include "definitions.h"
#include "buffer.h"
#include "file.h"
#include "lyx_cb.h"
#include "paths.h"
#include "minibuffer.h"

extern void MenuWrite(Buffer*);
extern void CloseBuffer();

// this is the global bufferlist object
BufferList bufferlist;


BufferList::BufferList()
{
#ifdef buffer_as_list
	List = new node; // header node
#else
	for (int i=NUMBER_OF_BUFFERS-1; i >= 0; i--) {
		buffer[i] = new Buffer();
	}
	current_buffer = 0;
	old_current_buffer = 0;
#endif
}


BufferList::~BufferList()
{
	// I am sure something should be done here too.
}

bool BufferList::isEmpty()
{
#ifdef buffer_as_list
	if (List->next == NULL)
		return true;
	return false;
#else
	return false;
#endif
}

Buffer* BufferList::current()
{
#ifdef buffer_as_list
	if (List->next != NULL)
		return List->next->element;
	return NULL;  // no buffers loaded
#else
	return buffer[current_buffer];
#endif
}


void BufferList::QwriteAll()
{
#ifdef buffer_as_list
	node *tmpn;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		if (tmpn->element->parameters.paragraph
		    && !tmpn->element->lyx_current) {
			if (fl_show_question("Changes in document:",
					     tmpn->element->filename,
					     "Save Document?"))
				MenuWrite(tmpn->element);
		}
	}
#else
	for (int i=0; i < NUMBER_OF_BUFFERS; i++) {
		if (buffer[i]->parameters.paragraph
		    && !buffer[i]->lyx_current) {
			if (fl_show_question("Changes in document:",
					     buffer[i]->filename,
					     "Save document?"))
				MenuWrite(buffer[i]);
		}
	}
#endif
}


void BufferList::write(Buffer *buf)
{
	// int n, i; // unused
	if (buf->parameters.paragraph) {
		minibuffer.Set("Saving document", buf->filename,"...");
		/* revise this: dabn100 */
		/* make a backup */ 
		char * s = AddExtension (NULL, buf->filename, "~");
		rename(buf->filename, s);
		delete s;
		
		if (buf->text)
			buf->parameters.cursor = buf->text->cursor;
		
// 		WriteFile(buf->filename, 
// 			  buf->parameters);
		buf->parameters.writeFile(buf->filename,0);
		buf->lyx_current = true;
		minibuffer.Set("Document saved as", buf->filename);
		
	}
}


void BufferList::closeAll()
{
#ifdef buffer_as_list
	// this will not work yet.
	node *tmpn;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		// a rewrite of CloseBuffer is needed for this to work
		tmpn->element->close();
		// or even more nice
		// tmp
	}
       
#else
	for (int i = 0; i < NUMBER_OF_BUFFERS; ++i) if (buffer[i]->text) {
		current_buffer = i;
		CloseBuffer();
	}
#endif
}


void BufferList::resize()
{
#ifdef buffer_as_list
	node *tmpn;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		if (tmpn->element->screen)
			delete tmpn->element->screen;
		tmpn->element->screen = NULL
	}
#else
	for (int i =0; i < NUMBER_OF_BUFFERS; i++) {
		if (buffer[i]->screen)
			delete buffer[i]->screen;
		buffer[i]->screen = NULL;
	}
#endif
}


void BufferList::close()
{
#ifdef buffer_as_list
	fprintf(stderr,"Sorry! Cannot close document. Not implemented!\n");
#else
	int i;
	
	/* This doesn't make sense if there is no document */
	if (!bufferlist.current()->parameters.paragraph)
		return;
	
	/* doit save buffer etc */
	i = current_buffer;
	if (buffer[i]->parameters.paragraph && !buffer[i]->lyx_current
	    && !quitting) {
		ProhibitInput();
		if (fl_show_question("Changes in document:",
				     buffer[i]->filename,
				     "Save document?"))
			MenuWrite(buffer[i]);
		AllowInput();
	}
	
	if (bufferlist.current()->filename) {
		delete bufferlist.current()->filename;
		bufferlist.current()->filename = NULL;
	}
	if (bufferlist.current()->tmppath) {
		DestroyBufferTmpDir (bufferlist.current()->tmppath);
		delete bufferlist.current()->tmppath;
		bufferlist.current()->tmppath = NULL;
	}
	LyXParagraph *par = bufferlist.current()->parameters.paragraph;
	LyXParagraph *tmppar = par;
	while(par){
		tmppar = par->next;
		delete par;
		par = tmppar;
	}
	bufferlist.current()->parameters.paragraph = NULL;

	// clear the undo stacks
	bufferlist.current()->parameters.undostack.Clear();
	bufferlist.current()->parameters.redostack.Clear();;
	
	if (bufferlist.current()->text) {
		delete bufferlist.current()->text;
		bufferlist.current()->text = NULL;
	}
	
	if (bufferlist.current()->screen) {
		delete bufferlist.current()->screen;
		bufferlist.current()->screen = NULL;
	}
	
	bufferlist.current()->lyx_current = false;
	bufferlist.current()->dvi_current = false;
	bufferlist.current()->bak_current = false;
	
	bufferlist.current()->parameters.Copy(lyxrc.parameters);
	
	// current buffer must be reset here
	i=0;
	while(i<NUMBER_OF_BUFFERS && !buffer[i]->text) i++;
	current_buffer = i;
	if (i==NUMBER_OF_BUFFERS)
		current_buffer = 0;
#endif
}


void BufferList::makePup(int pup)
{
#ifdef buffer_as_list
	node *tmpn;
	int ant = 0;
	char *relbuf;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		if (tmpn->element->parameters.paragraph) {
			relbuf = MakeDisplayPath (NULL,
						  tmpn->element->filename);
			fl_addtopup(pup, relbuf);
			ant++;
			delete relbuf;
		}
	}
	if (ant == 0) fl_addtopup(pup,"No Documents Open!%t");
#else
	int ant=0;
	char * relbuf;
	for(int n=0; n<NUMBER_OF_BUFFERS; n++){
		if(buffer[n]->parameters.paragraph){
			relbuf = MakeDisplayPath (NULL, buffer[n]->filename);
			fl_addtopup(pup, relbuf);
			ant++;
			delete relbuf;
		}
	}
	if (ant == 0) fl_addtopup(pup,"No Documents Open!%t");
#endif
}


void BufferList::switchBuffer(int choice)
{
#ifdef buffer_as_list
	fprintf(stderr,"Sorry! Cannot switch document. Not implemented!\n");
#else
	// should perhaps move this func to menus.C   
	int i=0;
	while (choice--) {
		while (!buffer[i++]->parameters.paragraph);
	}
	i--; // this is ok since choice never == 0
	
	if (i>=NUMBER_OF_BUFFERS) return;
	
	old_current_buffer = current_buffer;
	current_buffer = i;
#endif
}


void BufferList::prev()
{
#ifdef buffer_as_list
	fprintf(stderr,"Sorry! Cannot switch to previous document."
		" Not implemented!\n");
#else
	int tmp;
	tmp = current_buffer;
	current_buffer = old_current_buffer;
	old_current_buffer = tmp;
#endif
}


void BufferList::updateInset(Inset *inset)
{
#ifdef buffer_as_list
	node *tmpn;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		// this should be done in the buffer.
		if (tmpn->element->text) {
			if (tmpn->element->text->UpdataInset(inset)) {
				tmpn->element->lyx_current = false;
				tmpn->element->bak_current = false;
				tmpn->element->dvi_current = false;
				return;
			}
		}
	}
#else
	int i=0;
	while (i < NUMBER_OF_BUFFERS - 1){
		if (buffer[i]->text){
			if (buffer[i]->text->UpdateInset(inset)){
				buffer[i]->lyx_current = false;
				buffer[i]->bak_current = false;
				buffer[i]->dvi_current = false;
				return;
			}
		}
		i++;
	}
#endif
}


void BufferList::emergencyWriteAll()
{
#ifdef buffer_as_list
	node *tmpn;
	int a, i,n;
	char *s;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		if (tmpn->element->parameters.paragraph
		    && !tmpn->element->lyx_current) {

			for (n=0; tmpn->element->filename[n]; n++);
			s = new char[n+10];
			for (i=0; i<n; i++) 
				s[i] = tmpn->element->filename[i];
			s[i] = '#';
			s[i+1] = '\0';
			for (a=i+2; a && s[a-1]!='/'; a--) {
				s[a] = s[a-1];
			}
			s[a]='#';
			
			fprintf(stderr,
				"lyx: attempting to save"
				" document %s in %s...\n", 
				tmpn->element->filename, s);
			
			if (tmpn->element->text)
				tmpn->element->parameters.cursor =
					tmpn->element->text->cursor;
			
			WriteFile(s, 
				  tmpn->element->parameters);
			tmpn->element->lyx_current = true;
			
			delete s;
		}
	}
#else
	int a, buf, i,n;
	char *s;
	for (buf=0; buf < NUMBER_OF_BUFFERS; buf++) {
		if (buffer[buf]->parameters.paragraph &&
		    !buffer[buf]->lyx_current) {
			
			for (n=0; buffer[buf]->filename[n]; n++);
			s = new char[n+10];
			for (i=0; i<n; i++) 
				s[i] = buffer[buf]->filename[i];
			s[i] = '#';
			s[i+1] = '\0';
			for (a=i+2; a && s[a-1]!='/'; a--) {
				s[a] = s[a-1];
			}
			s[a]='#';
			
			fprintf(stderr,
				"lyx: attempting to save"
				" document %s in %s...\n", 
				buffer[buf]->filename, s);
			
			if (buffer[buf]->text)
				buffer[buf]->parameters.cursor =
					buffer[buf]->text->cursor;
			
			buffer[buf]->parameters.writeFile(s,1);
			buffer[buf]->lyx_current = true;
			
			delete s;
		}
	}
#endif
}


void BufferList::readFile(const char* s, bool ronly)
{
#ifdef buffer_as_list
	Buffer *new_buffer;
	
	// make a new buffer
	new_buffer = new Buffer();
	new_buffer->init();

	new_buffer->filename = StringCopy(s);

	if (ronly || lyxrc.use_tempdir) {
		new_buffer->tmppath = CreateBufferTmpDir (system_tempdir);
	} else {
		new_buffer->tmppath = NULL;
	}

	ReadFile(new_buffer->parameters);
	new_buffer->lyx_current = true;
	new_buffer->bak_current = true;
	new_buffer->dvi_current = false;
	new_buffer->read_only = ronly;
	
	// insert the new buffer at top of List.
	node *tmpn;
	tmpn = new node;
	tmpn->next = List->next; // we have a header node.
	tmpn->element = new_buffer;
	List->next = tmpn;
	
#else
	int i=0;
	while (i < NUMBER_OF_BUFFERS - 1
	       && buffer[i]->parameters.paragraph) i++;

	if (lyx_debug_level)
	fprintf(stderr,"assigning to buffer %d\n",i);
	
	// here we should asure that the parameters are correct.
	buffer[i]->init();
	
	buffer[i]->filename = StringCopy(s);
	if (ronly || lyxrc.use_tempdir) {
	   buffer[i]->tmppath = CreateBufferTmpDir (system_tempdir);
	} else buffer [i]->tmppath = NULL;

	buffer[i]->parameters.readFile();
	
	buffer[i]->lyx_current = true;
	buffer[i]->bak_current = true;
	buffer[i]->dvi_current = false;
	buffer[i]->read_only = ronly;
	old_current_buffer = current_buffer;
	current_buffer = i;
#endif
}


bool BufferList::exists(const char* s)
{
#ifdef buffer_as_list
	node *tmpn;
	tmpn = List;
	while (tmpn->next != NULL) {
		tmpn = tmpn->next;
		if (StringEqual(tmpn->element->filename, s))
			return true;
	}
	return false;
#else
	int i;
	
	for (i=0; i < NUMBER_OF_BUFFERS; i++)
		     if (StringEqual(buffer[i]->filename,s))
			     return true;
	
	return false;
#endif
}


void BufferList::newFile(const char* s,const char* fname)
{
#ifdef buffer_as_list
	Buffer *new_buffer;
	new_buffer = new Buffer;

	//_must_ be done before reading template
	//otherwise figinsets flage
	new_buffer->filename = StringCopy(s);
	if (lyxrc.use_tempdir) {
		new_buffer->tmppath = CreateBufferTmpDir (system_tempdir);
	} else {
		new_buffer->tmppath = NULL;
	}
	// read template
	if (fname && OpenFile(fname)) {
		new_buffer->init();
		ReadFile(new_buffer->parameters);
		CloseFile;
	} else { // start with empty buffer
		new_buffer->parameters.paragraph = new LyXParagraph();
	}
	new_buffer->lyx_current = false;
	new_buffer->bak_current = false;
	new_buffer->dvi_current = false;
#else
	/* find a free buffer */ 
	int i=0;
	while (i < NUMBER_OF_BUFFERS - 1
	       && buffer[i]->parameters.paragraph)
		i++;
	//_must_ be done before reading template
	//otherwise figinsets flake
	buffer[i]->filename = StringCopy(s);
	if (lyxrc.use_tempdir) {
	   buffer[i]->tmppath = CreateBufferTmpDir (system_tempdir);
	} else buffer [i]->tmppath = NULL;
	// setdefaults
	buffer[i]->parameters.Copy(lyxrc.parameters); 
	// read template
	if (fname && OpenFile(fname)){
	        buffer[i]->init();
		buffer[i]->parameters.readFile();
		CloseFile();
	}
	else {  // start with empty buffer
		buffer[i]->parameters.paragraph = new LyXParagraph();
	}
	buffer[i]->lyx_current = false;
	buffer[i]->bak_current = false;
	buffer[i]->dvi_current = false;
	
	old_current_buffer = current_buffer;
	current_buffer = i;
#endif
}




