/* 
 * Convert ASCII to PostScript.
 * Copyright (c) 1995 Markku Rossi.
 *
 * Author: Markku Rossi <mtr@iki.fi>
 */

/*
 * This file is part of genscript.
 * 
 * Genscript is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Genscript is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with genscript; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gsint.h"

/* 
 * Types and definitions.
 */

/* Values for token flags. */

/* EPSF. */
#define F_EPSF_CENTER			0x01
#define F_EPSF_RIGHT			0x02
#define M_EPSF_JUSTIFICATION		0x03

#define F_EPSF_NO_CPOINT_UPDATE_X	0x04
#define F_EPSF_NO_CPOINT_UPDATE_Y	0x08

#define F_EPSF_ABSOLUTE_X		0x10
#define F_EPSF_ABSOLUTE_Y		0x20

#define F_EPSF_SCALE_X			0x40
#define F_EPSF_SCALE_Y			0x80


/* Predicates for the current body font. */

/* Return the width of the character <ch> */
#define CHAR_WIDTH(ch) (font_widths[(unsigned char) (ch)])

/* Is character <ch> printable. */
#define ISPRINT(ch) (font_ctype[(unsigned char) (ch)] != ' ')

/* Does character <ch> exist in current body font? */
#define EXISTS(ch) (font_ctype[(unsigned char) (ch)] == '*')


/* Current point y movement from line to line. */
#define LINESKIP (Fpt + baselineskip)

#define RESOURCE_LINE_WIDTH 75

/* Token types. */
typedef enum
{
  tNONE,
  tEOF,
  tSTRING,
  tLINEFEED,
  tNEWLINE,
  tWRAPPED_NEWLINE,
  tEPSF,
  tFONT
} TokenType;

/* Token structure. */
struct gs_token_st
{
  TokenType type;
  unsigned int flags;
  double new_x;			/* Current point x after this token. */
  double new_y;			/* Current point y after this token. */
  int new_col;			/* Line column after this token. */

  union
    {
      char *str;
      struct
	{
	  double x;		/* x-offset */
	  double y;		/* y-offset */
	  double w;		/* width */
	  double h;		/* height */
	  double xscale;
	  double yscale;
	  int llx, lly, urx, ury; /* Bounding box. */
	  char filename[512];
	  char *skipbuf;
	  unsigned int skipbuf_len;
	  unsigned int skipbuf_pos;
	  FILE *fp;		/* File from which eps image is read. */
	  int pipe;		/* Is <fp> opened to pipe?  */
	} epsf;
      struct
	{
	  char name[512];
	  double size;
	} font;
    } u;
};

typedef struct gs_token_st Token;


/*
 * Prototypes for static functions.
 */

static int recognize_eps_file (Token *token);
static void paste_epsf (Token *token);

/*
 * Check if InputStream <is> contains a PostScript file that can be 
 * passed through without any modifications.  Returns 1 if file was
 * passed or 0 otherwise.
 */
static int do_pass_through_ps_file (char *fname, InputStream *is);

/*
 * Read one float dimension from InputStream <is>.  If <units> is
 * true, then number can be followed by an optional unit specifier.
 * If horizontal is true, then dimension is horizontal, otherwise it
 * is vertical (this is used to find out how big 'line' units are).
 */
static double read_float (InputStream *is, int units, int horizontal);

/*
 * Print linenumber <linenum> to the beginning of the current line.
 * Current line start is specified by point (x, y).
 */
static void print_line_number (double x, double y, double space,
			       double margin, unsigned int linenum);


/*
 * Global functions.
 */

void
dump_ps_header ()
{
  char *cp, *cp2;
  int i, j, got;
  /* 
   * Header.
   */

  fprintf (ofp, "%%!PS-Adobe-3.0\n");
  fprintf (ofp, "%%%%BoundingBox: %d %d %d %d\n", media->llx, media->lly,
	   media->urx, media->ury);
  fprintf (ofp, "%%%%Title: %s\n", title);
  fprintf (ofp, "%%%%For: %s\n", passwd->pw_gecos);
  fprintf (ofp, "%%%%Creator: %s\n", version_string);
  fprintf (ofp, "%%%%CreationDate: %s\n", date_string);
  fprintf (ofp, "%%%%Orientation: %s\n", landscape ? "Landscape" : "Portrait");
  fprintf (ofp, "%%%%Pages: (atend)\n");
  fprintf (ofp, "%%%%DocumentMedia: %s %d %d 0 () ()\n",
	   media->name, media->w, media->h);
  fprintf (ofp, "%%%%DocumentNeededResources: (atend)\n");

  if (count_key_value_set (pagedevice) > 0)
    fprintf (ofp, "%%%%LanguageLevel: 2\n");

  fprintf (ofp, "%%%%EndComments\n");


  /*
   * Procedure Definitions.
   */

  fprintf (ofp, "%%%%BeginProlog\n");

  /* Prolog. */
  fprintf (ofp, "%%%%BeginResource: procset Genscript-Prolog %d.%d 0\n",
	   MAJOR_VERSION, MINOR_VERSION);
  if (!paste_file ("genscript", ".pro"))
    fatal ("couldn't find prolog \"%s\": %s\n", "genscript.pro",
	   strerror (errno));
  fprintf (ofp, "%%%%EndResource\n");

  /* Encoding vector. */
  fprintf (ofp, "%%%%BeginResource: procset Genscript-Encoding-%s %d.%d 0\n",
	   encoding_name, MAJOR_VERSION, MINOR_VERSION);
  if (!paste_file (encoding_name, ".enc"))
    fatal ("couldn't find encoding file \"%s.enc\": %s\n", encoding_name,
	   strerror (errno));
  fprintf (ofp, "%%%%EndResource\n");

  /* EPSF import. */
  fprintf (ofp, "%%%%BeginResource: procset Genscript-EPSF-Import %d.%d 0\n",
	   MAJOR_VERSION, MINOR_VERSION);
  if (!paste_file ("epsf-import", ".ps"))
    fatal ("couldn't find resource \"epsf-import.ps\": %s", strerror (errno));
  fprintf (ofp, "%%%%EndResource\n");

  fprintf (ofp, "%%%%EndProlog\n");


  /* 
   * Document Setup.
   */

  fprintf (ofp, "%%%%BeginSetup\n");

  /* Download fonts. */
  for (got = strhash_get_first (download_fonts, &cp, &j, (void **) &cp2); got;
       got = strhash_get_next (download_fonts, &cp, &j, (void **) &cp2))
    download_font (cp);

  /* For each required font, emit %%IncludeResouce comment. */
  for (got = strhash_get_first (res_fonts, &cp, &j, (void **) &cp2); got;
       got = strhash_get_next (res_fonts, &cp, &j, (void **) &cp2))
    fprintf (ofp, "%%%%IncludeResource: font %s\n", cp);

  /* Dump our current dynamic state. */
  fprintf (ofp, "/d_page_w %d def\n", d_page_w);
  fprintf (ofp, "/d_page_h %d def\n", d_page_h);
  fprintf (ofp, "/d_header_x %d def\n", 0);
  fprintf (ofp, "/d_header_y %d def\n", d_output_h);
  fprintf (ofp, "/d_header_w %d def\n", d_header_w);
  fprintf (ofp, "/d_header_h %d def\n", d_header_h);
  fprintf (ofp, "/d_output_w %d def\n", d_output_w);
  fprintf (ofp, "/d_output_h %d def\n", d_output_h);

  fprintf (ofp, "/cols %d def\n", num_columns);

#define ZEROPAD(num) (num < 10 ? "0" : "")
  fprintf (ofp, "/usadatestr (%s%d/%s%d/%s%d) def\n",
	   ZEROPAD (run_tm.tm_mon + 1),	run_tm.tm_mon + 1,
	   ZEROPAD (run_tm.tm_mday), 	run_tm.tm_mday,
	   ZEROPAD (run_tm.tm_year), 	run_tm.tm_year);
  fprintf (ofp, "/eurdatestr (%s%d/%s%d/%s%d) def\n",
	   ZEROPAD (run_tm.tm_year), 	run_tm.tm_year,
	   ZEROPAD (run_tm.tm_mon + 1),	run_tm.tm_mon + 1,
	   ZEROPAD (run_tm.tm_mday), 	run_tm.tm_mday);
  fprintf (ofp, "/findatestr (%d.%d.%d) def\n",
 	   run_tm.tm_mday,
 	   run_tm.tm_mon + 1,
 	   run_tm.tm_year+1900);
  fprintf (ofp, "/timestr (%s%d:%s%d:%s%d) def\n",
	   ZEROPAD (run_tm.tm_hour), 	run_tm.tm_hour,
	   ZEROPAD (run_tm.tm_min),  	run_tm.tm_min,
	   ZEROPAD (run_tm.tm_sec),  	run_tm.tm_sec);

  /* User supplied header? */
  if (page_header)
    {
      fprintf (ofp, "/user_header_p true def\n");
      fprintf (ofp, "/user_header_str (%s) def\n", page_header);
    }
  else
    fprintf (ofp, "/user_header_p false def\n");

  fprintf (ofp, "/HFpt %g def\n", HFpt);

  /* Underlay string. */
  if (underlay)
    fprintf (ofp, "/ul_str (%s) def\n", underlay);


  /* Select our fonts. */

  /* Header font HF. */
  fprintf (ofp, "/%s /HF-gs-font MF\n", HFname);
  fprintf (ofp, "/HF /HF-gs-font findfont HFpt scalefont def\n");

  /* Our default typing font F. */
  fprintf (ofp, "/%s /F-gs-font MF\n", Fname);
  fprintf (ofp, "/F-gs-font findfont %g scalefont setfont\n", Fpt);

  /* Underlay. */
  if (underlay != NULL)
    {
      fprintf (ofp, "/ul_ptsize %g def\n", ul_ptsize);
      fprintf (ofp, "/ul_gray %g def\n", ul_gray);
      fprintf (ofp, "/%s /F-ul-font MF\n", ul_font);
      fprintf (ofp, "/ul_font /F-ul-font findfont ul_ptsize scalefont def\n");
    }

  /* Number of copies. */
  fprintf (ofp, "/#copies %d def\n", num_copies);

  /* Statusdict definitions. */
  if (count_key_value_set (statusdict) > 0)
    {
      fprintf (ofp, "%% Statustdict definitions:\nstatusdict begin\n  ");
      i = 2;
      for (got = strhash_get_first (statusdict, &cp, &j, (void **) &cp2); got;
	   got = strhash_get_next (statusdict, &cp, &j, (void **) &cp2))
	{
	  j = strlen (cp) + 1 + strlen (cp2) + 1;
	  if (i + j > RESOURCE_LINE_WIDTH)
	    {
	      fprintf (ofp, "\n  ");
	      i = 2;
	    }
	  fprintf (ofp, "%s %s ", cp2, cp);
	  i += j;
	}
      fprintf (ofp, "\nend\n");
    }

  /* Page device definitions. */
  if (count_key_value_set (pagedevice) > 0)
    {
      fprintf (ofp, "%% Pagedevice definitions:\n");
      fprintf (ofp, "gs_languagelevel 1 gt {\n  <<\n    ");
      i = 4;
      for (got = strhash_get_first (pagedevice, &cp, &j, (void **) &cp2); got;
	   got = strhash_get_next (pagedevice, &cp, &j, (void **) &cp2))
	{
	  j = strlen (cp2) + 1 + strlen (cp) + 2;
	  if (i + j > RESOURCE_LINE_WIDTH)
	    {
	      fprintf (ofp, "\n    ");
	      i = 4;
	    }
	  fprintf (ofp, "/%s %s ", cp, cp2);
	  i += j;
	}
      fprintf (ofp, "\n  >> setpagedevice\n} if\n");
    }

  /*
   * And finally, dump header procset.  Header must come after all
   * font inclusions and genscript's dynamic state definition.  So
   * dump it here at the very last place; everything else is already
   * done.
   */
  if (header != HDR_NONE)
    {
      char *hdr;
      if (header == HDR_SIMPLE)
	hdr = "simple";
      else
	hdr = fancy_header_name;
      
      fprintf (ofp, "%%%%BeginResource: procset Genscript-Header-%s %d.%d 0\n",
	       hdr, MAJOR_VERSION, MINOR_VERSION);
      if (!paste_file (hdr, ".hdr"))
	fatal ("couldn't find header definition file \"%s.hdr\": %s\n", hdr,
	       strerror (errno));
      fprintf (ofp, "%%%%EndResource\n");
    }

  fprintf (ofp, "%%%%EndSetup\n");
}


void
dump_ps_trailer ()
{
  int i, j, got;
  char *cp;
  void *value;

  /* Trailer. */
  fprintf (ofp, "%%%%Trailer\n");
  fprintf (ofp, "%%%%Pages: %d\n", total_pages);

  /* Document needed resources. */

  /* fonts. */
  fprintf (ofp, "%%%%DocumentNeededResources: font ");
  i = 32;			/* length of the previous string. */
  for (got = strhash_get_first (res_fonts, &cp, &j, &value); got;
       got = strhash_get_next (res_fonts, &cp, &j, &value))
    {
      if (i + strlen (cp) + 1 > RESOURCE_LINE_WIDTH)
	{
	  fprintf (ofp, "\n%%%%+ font ");
	  i = 9;		/* length of the previous string. */
	}
      fprintf (ofp, "%s ", cp);
      i += strlen (cp) + 1;
    }
  fprintf (ofp, "\n");

  fprintf (ofp, "%%%%EOF\n");
}


void
dump_ps_page_header (char *fname, int current_pagenum)
{
  char buf[512];
  char *ftail;

  /* Create fdir and ftail. */
  ftail = strrchr (fname, '/');
  if (ftail == NULL)
    {
      buf[0] = '\0';
      ftail = fname;
    }
  else
    {
      ftail++;
      strncpy (buf, fname, ftail - fname);
      buf[ftail - fname] = '\0';
    }

  /* Page start comment. */
  switch (page_label)
    {
    case LABEL_SHORT:
      fprintf (ofp, "%%%%Page: (%d) %d\n", total_pages, total_pages);
      break;

    case LABEL_LONG:
      fprintf (ofp, "%%%%Page: (%s:%3d) %d\n", ftail, current_pagenum,
	       total_pages);
      break;
    }

  /* 
   * Page Setup.
   */

  fprintf (ofp, "%%%%BeginPageSetup\n");

  fprintf (ofp, "_S\n");

  if (landscape)
    {
      fprintf (ofp, "90 rotate\n");
      fprintf (ofp, "%d %d translate\n", media->lly, media->llx - media->w);
    }
  else
    fprintf (ofp, "%d %d translate\n", media->llx, media->lly);

  fprintf (ofp, "/pagenum %d def\n", current_pagenum);
  fprintf (ofp, "/fname (%s) def\n", fname);
  fprintf (ofp, "/fdir (%s) def\n", buf);
  fprintf (ofp, "/ftail (%s) def\n", ftail);

  /* File's last modification time. */
  strftime (buf, sizeof (buf), "%a %b %e %T %Y", &mod_tm);
  fprintf (ofp, "/fmodstr (%s) def\n", buf);

  fprintf (ofp, "%%%%EndPageSetup\n");


  /* 
   * Mark standard page decorations.
   */

  /* Underlay. */
  if (underlay != NULL)
    fprintf (ofp, "underlay\n");

  /* Header and column lines. */
  switch (header)
    {
    case HDR_NONE:
      break;

    case HDR_SIMPLE:
      fprintf (ofp, "do_header\n");
      break;

    case HDR_FANCY:
      if (num_columns > 1)
	fprintf (ofp, "column_lines\n");

      fprintf (ofp, "do_header\n");
      break;
    }
}


void
dump_ps_page_trailer ()
{
  fprintf (ofp, "_R\n");
  fprintf (ofp, "S\n");
}


/* 
 * Get next token from input file <fp>.
 */

/* Help macros. */

/* Check if character <ch> fits to current line. */
#define FITS_ON_LINE(ch) ((linepos + CHAR_WIDTH (ch) < linew) || col == 0)

/* Is line buffer empty? */
#define BUFFER_EMPTY() (bufpos == 0)

/* Unconditionally append character <ch> to the line buffer. */
#define APPEND_CHAR(ch) 				\
  do {							\
    if (bufpos >= buflen)				\
      {							\
	buflen += 4096;					\
	buffer = xrealloc (buffer, buflen);		\
      }							\
    buffer[bufpos++] = ch;				\
  } while (0)

/* 
 * Copy character <ch> (it fits to this line) to output buffer and
 * update current point counters.
 */
#define EMIT(ch) 		\
  do {				\
    APPEND_CHAR (ch);		\
    linepos += CHAR_WIDTH (ch);	\
    col++;			\
  } while (0)


/* Read one special escape from input <fp>. */
static void
read_special_escape (InputStream *is, Token *token)
{
  char escname[256];
  int i;
  int ch;

  /* Get escape name. */
  for (i = 0; i < sizeof (escname) - 1 && (ch = is_getc (is)) != EOF; i++)
    {
      if (!isalnum (ch))
	{
	  is_ungetc (ch, is);
	  break;
	}
      else
	escname[i] = ch;
    }
  escname[i] = '\0';

  if (MATCH (escname, "epsf"))
    {
      int i;
      int pw, ph;
      double scale;

      token->flags = 0;
      token->u.epsf.x = 0.0;
      token->u.epsf.y = 0.0;
      token->u.epsf.h = 0.0;

      ch = is_getc (is);
      if (ch == '[')
	{
	  /* Read options. */
	  while ((ch = is_getc (is)) != EOF && ch != ']')
	    {
	      switch (ch)
		{
		case 'c':	/* center justification */
		  token->flags &= ~M_EPSF_JUSTIFICATION;
		  token->flags |= F_EPSF_CENTER;
		  break;

		case 'n':	/* no current point update */
		  /* Check the next character. */
		  ch = is_getc (is);
		  switch (ch)
		    {
		    case 'x':
		      token->flags |= F_EPSF_NO_CPOINT_UPDATE_X;
		      break;

		    case 'y':
		      token->flags |= F_EPSF_NO_CPOINT_UPDATE_Y;
		      break;

		    default:
		      is_ungetc (ch, is);
		      token->flags |= F_EPSF_NO_CPOINT_UPDATE_X;
		      token->flags |= F_EPSF_NO_CPOINT_UPDATE_Y;
		      break;
		    }
		  break;

		case 'r':	/* right justification */
		  token->flags &= ~M_EPSF_JUSTIFICATION;
		  token->flags |= F_EPSF_RIGHT;
		  break;


		case 's':	/* scale */
		  /* Check the next character. */
		  ch = is_getc (is);
		  switch (ch)
		    {
		    case 'x':
		      token->flags |= F_EPSF_SCALE_X;
		      token->u.epsf.xscale = read_float (is, 0, 1);
		      break;

		    case 'y':
		      token->flags |= F_EPSF_SCALE_Y;
		      token->u.epsf.yscale = read_float (is, 0, 0);
		      break;

		    default:
		      is_ungetc (ch, is);
		      token->flags |= F_EPSF_SCALE_X;
		      token->flags |= F_EPSF_SCALE_Y;
		      token->u.epsf.xscale = token->u.epsf.yscale
			= read_float (is, 0, 1);
		      break;
		    }
		  break;

		case 'x':	/* x-position */
		  token->u.epsf.x = read_float (is, 1, 1);

		  /* Check the next character. */
		  ch = is_getc (is);
		  switch (ch)
		    {
		    case 'a':
		      token->flags |= F_EPSF_ABSOLUTE_X;
		      break;

		    default:
		      is_ungetc (ch, is);
		      break;
		    }
		  break;

		case 'y':	/* y-position */
		  token->u.epsf.y = - read_float (is, 1, 0);

		  /* Check the next character. */
		  ch = is_getc (is);
		  switch (ch)
		    {
		    case 'a':
		      token->flags |= F_EPSF_ABSOLUTE_Y;
		      break;

		    default:
		      is_ungetc (ch, is);
		      break;
		    }
		  break;

		case 'h':	/* height */
		  token->u.epsf.h = read_float (is, 1, 0);
		  break;

		case ' ':
		case '\t':
		  break;

		default:
		  fatal ("illegal option %c for ^@epsf escape", ch);
		}
	    }	      
	  if (ch != ']')
	    fatal ("malformed ^@epsf escape: no ']' after options");

	  ch = is_getc (is);
	}
      if (ch == '{')
	{
	  /* Read filename. */
	  for (i = 0; (ch = is_getc (is)) != EOF && ch != '}'; i++)
	    {
	      token->u.epsf.filename[i] = ch;
	      if (i >= sizeof (token->u.epsf.filename))
		fatal ("too long file name for ^@epsf escape:\n%.*s",
		       i, token->u.epsf.filename);
	    }
	  if (ch == EOF)
	    fatal ("unexpected EOF while scanning ^@epsf escape");

	  token->u.epsf.filename[i] = '\0';
	  token->type = tEPSF;
	}
      else
	fatal ("malformed ^@epsf escape: no '{' found");

      /* 
       * Now we have a valid epsf-token in <token>.  Let's read BoundingBox
       * and do some calculations.
       */
      if (!recognize_eps_file (token))
	/* Recognize eps has already printed error message so we are done. */
	token->type = tNONE;
      else
	{
	  /* Some fixups for x and y dimensions. */
	  token->u.epsf.y += LINESKIP - 1;
	  if (token->u.epsf.h != 0.0)
	    token->u.epsf.h -= 1.0;

	  /* Count picture's width and height. */

	  pw = token->u.epsf.urx - token->u.epsf.llx;
	  ph = token->u.epsf.ury - token->u.epsf.lly;

	  /* The default scale. */
	  if (token->u.epsf.h == 0.0)
	    scale = 1.0;
	  else
	    scale = token->u.epsf.h / ph;

	  if ((token->flags & F_EPSF_SCALE_X) == 0)
	    token->u.epsf.xscale = scale;
	  if ((token->flags & F_EPSF_SCALE_Y) == 0)
	    token->u.epsf.yscale = scale;

	  pw *= token->u.epsf.xscale;
	  ph *= token->u.epsf.yscale;

	  token->u.epsf.w = pw;
	  token->u.epsf.h = ph;
	}
    }
  else if (MATCH (escname, "font"))
    {
      ch = is_getc (is);
      if (ch == '{')
	{
	  char *cp;

	  /* Read font name. */
	  for (i = 0; (ch = is_getc (is)) != EOF && ch != '}'; i++)
	    {
	      token->u.font.name[i] = ch;
	      if (i >= sizeof (token->u.font.name))
		fatal ("too long font name for ^@font escape:\n%.*s",
		       i, token->u.font.name);
	    }
	  token->u.font.name[i] = '\0';

	  /* Check for the default font. */
	  if (strcmp (token->u.font.name, "default") == 0)
	    token->u.font.name[0] = '\0';
	  else
	    {
	      if (!parse_font_spec (token->u.font.name, &cp,
				    &token->u.font.size))
		fatal ("malformed font spec for ^@font escape: %s",
		       token->u.font.name);
	      
	      strcpy (token->u.font.name, cp);
	      xfree (cp);
	    }
	  token->type = tFONT;
	}
      else
	fatal ("malformed ^@font escape: no '{' found");
    }
  else if (MATCH (escname, "comment"))
    {
      /* Comment the rest of this line. */
      while ((ch = is_getc (is)) != EOF && ch != nl)
	;
      token->type = tNONE;
    }
  else
    fatal ("unknown special escape: %s", escname);
}


static void
get_next_token (InputStream *is, double linepos, unsigned int col,
		unsigned int linew, Token *token)
{
  static unsigned char *buffer = NULL; /* output buffer */
  static unsigned int buflen = 0; /* output buffer's length */
  unsigned int bufpos = 0;	/* current position in output buffer */
  int ch = 0;
  int done = 0;
  int i;
  static int pending_token = tNONE;

  if (pending_token != tNONE)
    {
      token->type = pending_token;
      pending_token = tNONE;
      return;
    }

#define DONE_DONE 1
#define DONE_WRAP 2

  while (!done)
    {
      ch = is_getc (is);
      switch (ch)
	{
	case EOF:
	  if (BUFFER_EMPTY ())
	    {
	      token->type = tEOF;
	      return;
	    }

	  done = DONE_DONE;
	  break;

	case '(':
	case ')':
	case '\\':
	  if (FITS_ON_LINE (ch))
	    {
	      APPEND_CHAR ('\\');
	      EMIT (ch);
	    }
	  else
	    {
	      is_ungetc (ch, is);
	      done = DONE_WRAP;
	    }
	  break;

	case '\r':
	case '\n':
	  /*
	   * One of these is the newline character and the other one
	   * is just a whitespace.
	   */
	  if (ch == nl)
	    {
	      /* The newline character. */
	      if (BUFFER_EMPTY ())
		{
		  token->type = tNEWLINE;
		  return;
		}
	      else
		{
		  is_ungetc (ch, is);
		  done = DONE_DONE;
		}
	    }
	  else
	    {
	      /* Whitespace. */
	      if (FITS_ON_LINE (' '))
		EMIT (' ');
	      else
		{
		  is_ungetc (ch, is);
		  done = DONE_WRAP;
		}
	    }
	  break;

	case '\t':
	  if (font_is_fixed)
	    {
	      i = tab_size - (col % tab_size);
	      for (; i > 0; i--)
		{
		  if (FITS_ON_LINE (' '))
		    EMIT (' ');
		  else
		    {
		      done = DONE_WRAP;
		      break;
		    }
		}
	    }
	  else
	    {
	      /* Proportional font. */

	      double grid = tab_size * CHAR_WIDTH ('m');
	      col++;

	      /* Move linepos to the next multiple of <grid>. */
	      linepos = ((int) (linepos / grid) + 1) * grid;
	      if (linepos >= linew)
		done = DONE_WRAP;
	      else
		done = DONE_DONE;
	    }
	  break;

	case '\f':
	  if (BUFFER_EMPTY ())
	    {
	      token->type = tLINEFEED;
	      return;
	    }
	  else
	    {
	      is_ungetc (ch, is);
	      done = DONE_DONE;
	    }
	  break;

	case 0:
	  if (special_escapes)
	    {
	      if (BUFFER_EMPTY ())
		{
		  /* Interpret special escapes. */
		  read_special_escape (is, token);
		  if (token->type != tNONE)
		    return;

		  /*
		   * Got tNONE special escape => read_special_escape()
		   * has already done what was needed.  Just read more.
		   */
		  break;
		}
	      else
		{
		  is_ungetc (ch, is);
		  done = DONE_DONE;
		  break;
		}
	    }
	  /* FALLTHROUGH */
	  
	default:
	  /* Handle backspace character. */
	  if (ch == bs)
	    {
	      if (BUFFER_EMPTY () || !EXISTS (buffer[bufpos - 1]))
		linepos -= CHAR_WIDTH ('m');
	      else
		linepos -= CHAR_WIDTH (buffer[bufpos - 1]);

	      done = DONE_DONE;
	      break;
	    }

	  /* Check normal characters. */
	  if (EXISTS (ch))
	    {
	      if (FITS_ON_LINE (ch))
		{
		  /* 
		   * Print control characters in escaped form so 
		   * PostScript interpreter will not hang on them.
		   */
		  if (ch < 040)
		    {
		      char buf[10];

		      sprintf (buf, "\\%o", ch);
		      for (i = 0; buf[i]; i++)
			APPEND_CHAR (buf[i]);
		      
		      /* Update current point counters manually. */
		      linepos += CHAR_WIDTH (ch);
		      col++;
		    }
		  else
		    EMIT (ch);
		}
	      else
		{
		  is_ungetc (ch, is);
		  done = DONE_WRAP;
		}
	    }
	  else if (ISPRINT (ch))
	    {
	      /* Printable, but do not exists in this font. */
	      if (FITS_ON_LINE ('?'))
		{
		  EMIT ('?');
		  if (missing_chars[ch]++ == 0)
		    num_missing_chars++;
		}
	      else
		{
		  is_ungetc (ch, is);
		  done = DONE_WRAP;
		}
	    }
	  else
	    {
	      char buf[20];
	      double len = 0.0;

	      /*
	       * Non-printable and does not exist in current font, print
	       * it in octal.
	       */

	      sprintf (buf, "\\%3o", ch);
	      for (i = 0; buf[i]; i++)
		if (buf[i] == ' ')
		  buf[i] = '0';

	      /* Count length. */
	      for (i = 0; buf[i]; i++)
		len += CHAR_WIDTH (buf[i]);

	      if (linepos + len < linew || col == 0)
		{
		  /* Print it. */
		  APPEND_CHAR ('\\'); /* Escape the first '\\' character. */
		  for (i = 0; buf[i]; i++)
		    EMIT (buf[i]);
		}
	      else
		{
		  is_ungetc (ch, is);
		  done = DONE_WRAP;
		}
	    }
	  break;
	}
    }

  /* Got a string. */

  /* Check for wrapped line. */
  if (done == DONE_WRAP)
    {
      /* This line is too long. */
      num_truncated_lines++;

      ch = nl;
      if (truncate_lines)
	{
	  /* Truncate this line. */
	  while ((ch = is_getc (is)) != EOF && ch != nl)
	    ;
	}
      if (ch == nl)
	pending_token = tWRAPPED_NEWLINE;
      else
	pending_token = tEOF;
    }

  APPEND_CHAR ('\0');
  token->type = tSTRING;
  token->u.str = (char *) buffer;
  token->new_x = linepos;
  token->new_col = col;
}


void
process_file (char *fname, InputStream *is)
{
  int col;
  double x, y;
  double lx, ly;
  double linewidth;		/* Line width in points. */
  double lineend;
  int done = 0;
  int page_clear;
  unsigned int line_column;
  unsigned int current_pagenum = 0;
  unsigned int current_linenum;
  unsigned int current_file_linenum = 1;
  double linenumber_space;
  double linenumber_margin;
  Token token;
  int reuse_last_token = 0;

  if (pass_through_ps_files && do_pass_through_ps_file (fname, is))
    return;

  linewidth = d_output_w / num_columns - 2 * d_output_x_margin
    - line_indent * CHAR_WIDTH ('m');

  while (!done)
    {
      /* Start a new page. */
      page_clear = 1;
      total_pages++;
      current_pagenum++;

      for (col = 0; !done && col < num_columns; col++)
	{
	  /* Move to the beginning of the column <col>. */
	  lx = x = col * d_output_w / (float) num_columns + d_output_x_margin
	    + line_indent * CHAR_WIDTH ('m');
	  lineend = lx + linewidth;

	  ly = y = d_output_h - d_output_y_margin - LINESKIP;
	  current_linenum = 0;
	  line_column = 0;

	  while (1)
	    {
	      if (line_numbers && line_column == 0)
		{
		  /* 
		   * Count the space needed by linenumbers.  This should
		   * be enought for 99999 lines.
		   */

		  linenumber_space = CHAR_WIDTH ('0') * 5 + 1.0;
		  linenumber_margin = CHAR_WIDTH (':') + CHAR_WIDTH ('x');

		  /* Forward x by the amount needed by our line numbers. */
		  x += linenumber_space + linenumber_margin;
		}

	      /* Get token. */
	      if (!reuse_last_token)
		get_next_token (is, x, line_column, lineend, &token);
	      reuse_last_token = 0;

	      /*
	       * Page header printing is delayed to this point because
	       * we want to handle files ending with a newline character
	       * with care.  If the last newline would cause a pagebreak,
	       * otherwise we would print page header to the non-existent
	       * next page and that would be ugly ;)
	       */
	      
	      if (token.type == tEOF)
		{
		  done = 1;
		  goto end_of_page;
		}

	      /*
	       * Now we know that we are going to make marks to this page
	       * => print page header.
	       */

	      if (page_clear)
		{
		  dump_ps_page_header (fname, current_pagenum);
		  page_clear = 0;
		}

	      /* Print line numbers if needed. */
	      if (line_numbers && line_column == 0 && token.type != tLINEFEED)
		print_line_number (lx, y, linenumber_space, linenumber_margin,
				   current_file_linenum);

	      /* Check rest of tokens. */
	      switch (token.type)
		{
		case tLINEFEED:
		  goto end_of_column;
		  break;

		case tSTRING:
		  fprintf (ofp, "%g %g M\n", x, y);
		  fprintf (ofp, "(%s) s\n", token.u.str);
		  x = token.new_x;
		  line_column = token.new_col;
		  break;
		  
		case tNEWLINE:
		  current_file_linenum++;
		  /* FALLTHROUGH */

		case tWRAPPED_NEWLINE:
		  current_linenum++;
		  y -= LINESKIP;
		  if (current_linenum >= lines_per_page || y <= 0)
		    goto end_of_column;

		  x = col * d_output_w / (float) num_columns
		    + d_output_x_margin + line_indent * CHAR_WIDTH ('m');
		  line_column = 0;
		  break;

		case tEPSF:
		  /* Count current point movement. */

		  if (token.flags & F_EPSF_ABSOLUTE_Y)
		    token.new_y = ly;
		  else
		    token.new_y = y;
		  token.new_y += token.u.epsf.y - token.u.epsf.h;

		  if (token.flags & F_EPSF_ABSOLUTE_X)
		    token.new_x = lx;
		  else
		    token.new_x = x;
		  token.new_x += token.u.epsf.x;

		  /* Check flags. */
		  
		  /* Justification flags overwrite <x_ofs>. */
		  if (token.flags & F_EPSF_CENTER)
		    token.new_x = lx + (linewidth - token.u.epsf.w) / 2;
		  if (token.flags & F_EPSF_RIGHT)
		    token.new_x = lx + (linewidth - token.u.epsf.w);

		  /* Check if eps file does not fit to this column. */
		  if ((token.flags & F_EPSF_NO_CPOINT_UPDATE_Y) == 0
		      && token.new_y <= 0)
		    {
		      if (current_linenum == 0)
			{
			  /*
			   * At the beginning of the column, warn user
			   * and print image.
			   */
			  message (0, "EPS file \"%s\" is too \
large for page\n",
				   token.u.epsf.filename);
			}
		      else
			{
			  /* Must start a new column. */
			  reuse_last_token = 1;
			  goto end_of_column;
			}
		    }

		  /* Do paste. */
		  paste_epsf (&token);

		  /* Update current point? */
		  if (!(token.flags & F_EPSF_NO_CPOINT_UPDATE_Y))
		    y = token.new_y;
		  if (!(token.flags & F_EPSF_NO_CPOINT_UPDATE_X))
		    x = token.new_x + token.u.epsf.w;
		  
		  if (y <= 0)
		    goto end_of_column;
		  break;

		case tFONT:
		  /* Select a new current font. */
		  message (1, "^@font=");
		  if (token.u.font.name[0] == '\0')
		    {
		      /* Select the default font. */
		      Fpt = default_Fpt;
		      Fname = default_Fname;
		      fprintf (ofp,
			       "/F-gs-font findfont %g scalefont setfont\n",
			       Fpt);
		    }
		  else
		    {
		      strhash_put (res_fonts, token.u.font.name,
				   strlen (token.u.font.name) + 1,
				   NULL, NULL);
		      fprintf (ofp, "/%s /F-gs-user-font MF\n",
			       token.u.font.name);
		      fprintf (ofp, "/F-gs-user-font findfont %g scalefont \
setfont\n",
			       token.u.font.size);
		      Fpt = token.u.font.size;
		      Fname = token.u.font.name;
		      
		    }
		  message (1, "%s %gpt\n", Fname, Fpt);
		  read_font_info ();
		  break;

		case tNONE:
		default:
		  fatal("process_file(): got illegal token %d", token.type);
		  break;
		}
	    }
	end_of_column:
	  ;			/* ULTRIX's cc needs this line. */
	}

    end_of_page:
      if (!page_clear)
	dump_ps_page_trailer ();
    }
  if (page_clear)
    total_pages--;
}


/*
 * Static functions.
 */

static int
recognize_eps_file (Token *token)
{
  int i;
  char filename[512];
  char buf[4096];
  int line;
  int valid_epsf;
  float llx, lly, urx, ury;

  message (1, "^@epsf=\"%s\"\n", token->u.epsf.filename);

  i = strlen (token->u.epsf.filename);
  if (i > 0 && token->u.epsf.filename[i - 1] == '|')
    {
      /* Read EPS data from pipe. */
      token->u.epsf.pipe = 1;
      token->u.epsf.filename[i - 1] = '\0';
      token->u.epsf.fp = popen (token->u.epsf.filename, "r");
      if (token->u.epsf.fp == NULL)
	{
	  message (0, "epsf: couldn't open pipe to command \"%s\": %s\n",
		   token->u.epsf.filename, strerror (errno));
	  return 0;
	}
    }
  else
    {
      /* Read EPS data from file. */
      tilde_subst (token->u.epsf.filename, filename);

      token->u.epsf.fp = fopen (filename, "r");
      if (token->u.epsf.fp == NULL)
	{
	  if (token->u.epsf.filename[0] != '/')
	    {
	      /* Name is not absolute, let's lookup path. */
	      FileLookupCtx ctx;

	      strcpy (ctx.name, token->u.epsf.filename);
	      strcpy (ctx.suffix, "");

	      if (pathwalk (libpath, file_lookup, &ctx))
		token->u.epsf.fp = fopen (ctx.fullname, "r");
	    }
	  if (token->u.epsf.fp == NULL)
	    {
	      message (0, "couldn't open EPS file \"%s\": %s\n",
		       token->u.epsf.filename, strerror (errno));
	      return 0;
	    }
	}
    }

  /* Find BoundingBox DSC comment. */

  line = 0;
  valid_epsf = 0;
  token->u.epsf.skipbuf = NULL;
  token->u.epsf.skipbuf_len = 0;
  token->u.epsf.skipbuf_pos = 0;

  while (fgets (buf, sizeof (buf), token->u.epsf.fp))
    {
      line++;

      /* Append data to the skip buffer. */
      i = strlen (buf);
      if (i + token->u.epsf.skipbuf_pos >= token->u.epsf.skipbuf_len)
	{
	  token->u.epsf.skipbuf_len += 8192;
	  token->u.epsf.skipbuf = xrealloc (token->u.epsf.skipbuf,
					    token->u.epsf.skipbuf_len);
	}
      memcpy (token->u.epsf.skipbuf + token->u.epsf.skipbuf_pos, buf, i);
      token->u.epsf.skipbuf_pos += i;

      /* Check the "%!" magic cookie. */
      if (line == 1)
	{
	  if (buf[0] != '%' || buf[1] != '!')
	    {
	      message (0,
		       "EPS file \"%s\" does not start with \"%%!\" magic\n",
		       token->u.epsf.filename);
	      break;
	    }
	}

#define BB_DSC "%%BoundingBox:"

      if (strncmp (buf, BB_DSC, strlen (BB_DSC)) == 0)
	{
	  i = sscanf (buf + strlen (BB_DSC), "%g %g %g %g",
		      &llx, &lly, &urx, &ury);
	  if (i != 4)
	    {
	      /* (atend) ? */

	      /* Skip possible whitespace. */
	      for (i = strlen (BB_DSC);
		   buf[i] && (buf[i] == ' ' || buf[i] == '\t');
		   i++)
		;
#define BB_DSC_ATEND "(atend)"
	      if (strncmp (buf + i, BB_DSC_ATEND, strlen (BB_DSC_ATEND)) != 0)
		{
		  /* No, this BoundingBox comment is corrupted. */
		  message (0, "EPS file \"%s\" contains malformed \
%%%%BoundingBox row:\n\"%.*s\"\n",
			   token->u.epsf.filename, strlen (buf) - 1, buf);
		  break;
		}
	    }
	  else
	    {
	      /* It was a valid EPS file. */

	      /* We store bounding box in int format. */
	      token->u.epsf.llx = llx;
	      token->u.epsf.lly = lly;
	      token->u.epsf.urx = urx;
	      token->u.epsf.ury = ury;

	      valid_epsf = 1;
	      break;
	    }
	}
    }

  /* Check that we found the BoundingBox comment. */
  if (!valid_epsf)
    {
      message (0, "EPS file \"%s\" is not a valid EPS file\n",
	       token->u.epsf.filename);
      if (token->u.epsf.pipe)
	pclose (token->u.epsf.fp);
      else
	fclose (token->u.epsf.fp);
      xfree (token->u.epsf.skipbuf);
      return 0;
    }

  message (2, "BoundingBox: %d %d %d %d\n",
	   token->u.epsf.llx, token->u.epsf.lly,
	   token->u.epsf.urx, token->u.epsf.ury);

  return 1;
}


static void
paste_epsf (Token *token)
{
  char buf[4096];
  int i;
  
  /* EPSF import header. */
  fprintf (ofp, "BeginEPSF\n");
  fprintf (ofp, "%g %g translate\n", token->new_x, token->new_y);
  fprintf (ofp, "%g %g scale\n", token->u.epsf.xscale, token->u.epsf.yscale);
  fprintf (ofp, "%d %d translate\n", -token->u.epsf.llx,
	   -token->u.epsf.lly);
  fprintf (ofp, "%d %d %d %d Box clip\n",
	   token->u.epsf.llx - 1,
	   token->u.epsf.lly - 1,
	   token->u.epsf.urx - token->u.epsf.llx + 2,
	   token->u.epsf.ury - token->u.epsf.lly + 2);
  fprintf (ofp, "%%%%BeginDocument: %s%s\n", token->u.epsf.filename,
	   token->u.epsf.pipe ? "|" : "");

  /* Dump skip buffer. */
  fwrite (token->u.epsf.skipbuf, 1, token->u.epsf.skipbuf_pos, ofp);

  /* Dump file. */
  while ((i = fread (buf, 1, sizeof (buf), token->u.epsf.fp)) != 0)
    fwrite (buf, 1, i, ofp);

  if (token->u.epsf.pipe)
    pclose (token->u.epsf.fp);
  else
    fclose (token->u.epsf.fp);
  fprintf (ofp, "\n");		/* Add a newline to keep comments correct */

  xfree (token->u.epsf.skipbuf);

  /* EPSF import trailer. */
  fprintf (ofp, "%%%%EndDocument\n");
  fprintf (ofp, "EndEPSF\n");
}


static double
read_float (InputStream *is, int units, int horizontal)
{
  char buf[256];
  int i, ch;
  double val;

  for (i = 0; (i < sizeof (buf) - 1
	       && (ch = is_getc (is)) != EOF
	       && ISNUMBERDIGIT (ch)); 
       i++)
    buf[i] = ch;
  buf[i] = '\0';
  if (ch != EOF)
    is_ungetc (ch, is);

  val = atof (buf);

  if (units)
    {
      /* Get unit. */
      ch = is_getc (is);
      switch (ch)
	{
	case 'c':
	  val *= 72 / 2.54;
	  break;

	case 'p':
	  break;

	case 'i':
	  val *= 72;
	  break;

	default:
	  is_ungetc (ch, is);
	  /* FALLTHROUGH */

	case 'l':
	  if (horizontal)
	    val *= CHAR_WIDTH ('m');
	  else
	    val *= LINESKIP;
	  break;
	}
    }

  return val;
}


static int
do_pass_through_ps_file (char *fname, InputStream *is)
{
  int ch;
  unsigned long saved_pos = is->bufpos;

  /* Try to recognize PostScript magic `%!' (or Windoze damaged '^D%!'). */
  ch = is_getc (is);
  if (ch == 4)
    /* Windoze damaged. */
    ch = is_getc (is);

  if (ch != '%')
    goto fail;
  ch = is_getc (is);
  if (ch != '!')
    goto fail;

  /* Yes, it really is a PostScript file.  Now do the pass through. */

  message (1, "passing through PostScript file \"%s\"\n", fname);
  assert (is->bufpos >= 2);
  is->bufpos -= 2;
  do
    {
      fwrite (is->buf + is->bufpos, 1, is->data_in_buf - is->bufpos, ofp);
      is->bufpos = is->data_in_buf;

      /* Read more data to the input buffer. */
      ch = is_getc (is);
      is->bufpos = 0;
    }
  while (ch != EOF);

  return 1;

 fail:

  /* Seek stream to its start. */
  is->bufpos = saved_pos;

  return 0;
}


static void
print_line_number (double x, double y, double space, double margin,
		   unsigned int linenum)
{
  double len = 0.0;
  char buf[20];
  int i;

  /* Count linenumber string length. */
  sprintf (buf, "%d", linenum);
  for (i = 0; buf[i]; i++)
    len += CHAR_WIDTH (buf[i]);

  fprintf (ofp, "%g %g M (%s:) s\n", x + space - len, y, buf);
}
