/*
Copyright (c) 1996-1997 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: w3mux.c,v 1.35 1997/11/13 23:13:41 spreitze Exp $ */
/* Last edited by Mike Spreitzer November 13, 1997 2:49 pm PST */

/***********************************************************************/
/***********************************************************************

This file contains an implementation of the W3C MUX protocol, adapted
for ILU.  Basically it is the protocol described in

  http://www.w3.org/Protocols/MUX/WD-mux-961023.html

modified as described in

  http://lists.w3.org/Archives/Member/w3c-mux/msg00039.html

************************************************************************/
/***********************************************************************/


#include <iluntrnl.h>

#define ILU_HASH_FINDINTABLE(tb,key)	ilu_hash_FindInTable((tb),(key))
#define ILU_HASH_BEGINENUMERATION(tb,ptr) ilu_hash_BeginEnumeration((tb),(ptr))
#define ILU_HASH_NEXT(ptr,keyptr,dataptr) ilu_hash_Next((ptr),(keyptr),(dataptr))
#define ILU_HASH_REMOVEFROMTABLE(tb,key)  ilu_hash_RemoveFromTable((tb),(key))
#define ILU_HASH_ADDTOTABLE(tb,key,data)  ilu_hash_AddToTable((tb),(key),(data))
#define ILU_HASH_MAKENEWTABLE(size,hashfn,compfn) ilu_hash_MakeNewTable((size),(hashfn),(compfn))
#define ILU_HASH_HASHPOINTER ilu_hash_HashPointer
#define ILU_HASH_POINTERCOMPARE ilu_hash_PointerCompare

#include <transprt.h>
#include <mooring.h>

#ifndef WIN32
#include <sys/types.h>	/* netinet/in.h requires this on Solaris 1 */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>	/* for gethostbyname(), etc. */
#else
#include <winsock.h>
#endif /* WIN32 */
  
/***********************************************************************

  In this transport, we mux messages for different ilu_Servers at the
  same address space, or for different connections to the same ilu_Server
  at the same address space, over a single tcp/ip connection (or other
  similar underlying connection technology, which must be reliable but
  not necessarily boundaried).  Instances of w3mux ilu_Transport are
  grouped into equivalence classes by the tcp/ip stream that they use.
  Inside each equivalence class (EC), instances are distinguished by the
  channel identifier they use.  On the server side, channel ids are
  assigned when a new mooring in a particular equivalence class is
  created; if all available ids are in use, the mooring cannot be
  created.  On the client side, channel ids are somewhat insignificant;
  they are used to determine the setup class ID used in the channel
  init message.

  Moorings:

  When we create a new mux mooring (which I'll call a channel), we
  check to see if a tcp mooring for th EC of the channel already
  exists.  If not, we create it with "GetMooring", and register a
  callback handler "mHandler" on it.  "mHandler" will accept a new
  incoming connection, and register a different callback handler,
  "tHandler", on the new fd.  In the threaded case, a new thread is
  forked when the new tcp mooring is created, running the function
  "handleMooring".  "handleMooring" waits for input on the underlying
  tcp socket, then calls "mHandler" to process the input.

  In general, mux moorings accept clients by looking at a list of
  incoming client accept requests (built by "tHandler") which hangs
  off "w3mux_ConnReqs".  Each mooring looks at the list when its
  mo_accept_client routine is called, and if it finds a request to
  connect to its channel, it will accept the request and create a
  new mux transport (which I'll call a session).  In the threaded case,
  the mooring will block on "w3mux_ConnReqsCond" waiting for something
  to be added to the list of connection requests; in the non-threaded
  case, the "mo_accept_client" call will be called as a result of
  "tHandler" calling the request handler for the mooring.

  Transports:

  Each EC of mux transports works on top of a single tcp/ip transport.
  In the single-threaded case, this tcp/ip transport is handled by
  an input handler called "tHandler".  In the multi-threaded case,
  a work thread running "handleTransport" calls "tHandler" when input
  is available on the underlying tcp/ip transport.

  Outgoing mux transports (sessions) are created by invoking the
  tcr_createTransport method of the mux transport creator.  This
  routine checks to see if an underlying tcp/ip transport for the EC
  exists; if not, it makes one using "GetTransport", and either forks
  the work thread for it (running "handleTransport") or registers
  "tHandler" as an input handler for it.  muxtcr_createTransport may
  fail if the max number of sessions is already present on the
  underlying connection.  Otherwise, it adds a new session to the list
  of sessions associated with the underlying tcp/ip transport.
  Incoming sessions are created by "tHandler" when a "NewSession"
  header is received in the tcp/ip transports data stream.

  "tHandler" works by reading messages from the underlying transport,
  decoding the header, and dispatching each message in the proper way.
  There are three classes of message:  control messages, which may
  affect the state of the underlying transport or any of the associated
  sessions; new session messages, which may also contain data; and
  data messages, which contain data intended for a particular session.
  Access to the state maintained by the transport is protected by
  "state_lock".  Access to the transport itself is protected by
  "io_lock".  Access to the state of each superimposed session is
  protected by the "lock" field of that session's data.

  Control messages are handled wholely by "tHandler", though it might
  notify a condition variable (or, in the single-threaded case, run an
  input handler) to allow channels or sessions to handle changes in
  their state.  New session messages are also handled by "tHandler" by
  adding a new session request to "w3mux_ConnReqs", then notifying the
  channel.  Data for a session is handled by inserting the data into
  the session's input buffer, prefixed with a 4-byte Sun RPC RM style
  header indicating the number of bytes in that message fragment, and
  whether it is the last chunk of the current message.  End-of-file on
  the transport, or receipt of a close-session control message for a
  session, will set the "eofRcvd" flag of the session's data.

  Flow control is maintained by the explicit flow control construct of
  the MUX protocol.  Each session keeps track of how much output
  capability it has been granted by the other side, and always outputs
  less than that amount.  On input, when a session has finished
  consuming all of a particular message fragment, credit is sent back
  to the other side with a SendCredit control message.  There is
  currently no way for an application-specified buffer to be used, so
  all messages will be transported in relatively small fragments.

***********************************************************************/

/**********************************************************************/
/**********************************************************************/
/*								      */
/*			Data Types				      */
/*								      */
/**********************************************************************/
/**********************************************************************/

typedef struct MooringList_s {
  ilu_Mooring		mooring;
  struct MooringList_s *next;
} *MooringList;

typedef struct TransportList_s {
  ilu_Transport		transport;
  struct TransportList_s *next;
} *TransportList;

typedef struct Endpoint_s {
  ilu_cardinal			ip_addr;
  ilu_shortcardinal		port;
} Endpoint_s, *Endpoint;

typedef unsigned int flagbit;
typedef struct w3mux_hdr {
  union {
    struct {
      flagbit long_length : 1;
      flagbit control : 1;
      flagbit syn : 1;
      flagbit fin : 1;
      flagbit rst : 1;
      flagbit push : 1;
      unsigned int session_id : 8;
      unsigned int session_fragment_size : 18;    /* used for other purposes if long_length is set */
    } data_hdr;
    struct {
      flagbit long_length : 1;
      flagbit control : 1;
      unsigned int control_code : 4;
      unsigned int session_id : 8;
      unsigned int session_fragment_size : 18;    /* used for other purposes if long_length is set */
    } control_message;
    ilu_cardinal word;
  } contents;
} MuxHeader;

typedef enum TransportDataState_e { Accepting_1, Accepting_2, Normal } TransportDataState;

typedef struct TransportData_s {
  
  ilu_Mutex		state_lock;	/* mutex protecting the underlying transport's input buffer */
  ilu_Mutex		io_lock;	/* mutex protecting the underlying transport (xmu = ymu = lock) */
  ilu_Transport		transport;	/* TCP/IP transport that underlies this whole thing */
  ilu_boolean		active;		/* ilu_TRUE if input handler or worker thread has been registered/started */
  ilu_Passport		initiator;	/* NIL for initiator side, contains connection pp for acceptor side */
  ilu_TransportInfo	tinfo;		/* if initiator side, contains ilu_TransportInfo */
  ilu_string		peerinfo;	/* if acceptor side, contains string describing other end */
  ilu_boolean		closing;	/* TRUE if being shut down -- no additional data can be received */
  TransportDataState	state;		/* current state */
  MuxHeader		accepting;	/* if in an accepting, the current header */

  TransportList		sessions;	/* list of w3mux transports participating */

  ilu_Condition		input_cond;	/* condition on which threaded impls wait for input */
  
  ilu_TIH_s		td_loClo;	/* Input handler to give to underlying transport */

} *TransportData;

typedef struct MooringData_s {

  ilu_Mooring		mooring;	/* TCP/IP mooring that underlies this whole thing */
  ilu_boolean		active;
  ilu_TransportInfo	tinfo_in;	/* tinfo for actual mooring */
  ilu_TransportInfo	tinfo_out;	/* tinfo returned during creation of mooring */
  ilu_cardinal		ip_addr;	/* only valid for TCP/IP moorings */
  ilu_shortcardinal	port;		/* only valid for TCP/IP moorings */
  struct {
    ilu_Mutex		lock;
    MooringList		list;
  } channels;				/* channels which refer explicitly to this mooring */

} *MooringData;

typedef struct MooringDataList_s {
  MooringData			md;
  struct MooringDataList_s *	next;
} *MooringDataList;

typedef struct TransportDataList_s {
  TransportData			td;
  struct TransportDataList_s *	next;
} *TransportDataList;

typedef struct {
  /* L1, L2 unconstrained */

  ilu_Mutex	  lock;

  /* while holding self->lock */
  ilu_bytes       t_inBuff;
  ilu_cardinal    t_inNext, t_inLimit;

  /* L2 >= {ymu} */
  /* For input: */

  ilu_boolean     inWedged;	/* further input msgs forbidden? */
  ilu_boolean     lastIn;	/* current chunk is last? */
  ilu_boolean     firstIn;	/* next chunk is first? */
  ilu_cardinal    inLength;	/* of lower-level data in buffer */
  ilu_cardinal    inSize;	/* size of input buffer */
  ilu_cardinal    accumCredit;	/* credit to be sent to other side
				   when convenient */
  ilu_cardinal    prevChunkSize; /* size in bytes of prev chunk */

  /*for read:  L2 includes xmu or ymu;
    for write: L2    >=   {xmu,   ymu}*/
  ilu_boolean     busyIn;	/* currently processing input msg? */
  ilu_boolean     skipping;	/* prev msg not yet finished? */
  ilu_boolean	  eofRcvd;	/* received session close msg? */
  /*
   * ymu's invariant includes all the following.  Unconsumed
   * lower-level input bytes are at indices [t_inNext, inLength) in
   * t_inBuff.  "next" in def'n of (firstIn) means first chunk
   * whose header hasn't been entirely processed by this layer.
   * (skipping) means we're between messages as far as our caller is
   * concerned, but we haven't finished skipping over all the input
   * from the previous message.  When busyIn||skipping: firstIn and
   * lastIn are meaningful, and the unconsumed bytes of the current
   * chunk are at indices [t_inNext, t_inLimit).  skipping =>
   * t_inNext == t_inLimit.  t_inNext==t_inLimit when
   * !busyIn && !skipping && !inWedged.  inSize is the allocated
   * size of t_inBuff.  busyIn, skipping, and inWedged are mutually
   * exclusive.  When inWedged, no more messages may be input (we've
   * lost synch with lower layer).
   */

  /* L2 >= {xmu} */
  /* For output: */

  ilu_boolean	  firstOut;	/* next chunk is first ever sent? */
  ilu_boolean     busyOut;	/* currently processing output msg? */
  ilu_boolean	  usesCredit;	/* TRUE if user-level flow control */
  ilu_integer	  disabled;	/* keep track of enable/disable */
  ilu_cardinal    outStart;	/* loc o 1st payload byte of cur msg */
  ilu_cardinal    outSize;	/* size of output buffer */
  ilu_cardinal	  maxFragSize;	/* max size for a msg fragment */
  ilu_cardinal	  outCredit;	/* how many bytes may be sent */
  /*
   * Unwritten lower-level data are at indices [0, t_outNext) of
   * t_outBuff.  When !busyOut, t_outLimit==t_outNext.  When
   * busyOut, payload so far of current message is at indices
   * [outStart, t_outNext); header needs to be written at
   * [outStart-4, outStart).  The allocated size of t_outBuff is
   * outSize, which is always >= t_outLimit + (busyOut?4:8).
   */

  /* For general use: */
  
  ilu_boolean	dirIsIn;
  /*
   * xmu's invariant includes this: at most one of (busyIn),
   * (busyOut) is true; when one is, (dirIsIn) indicates which one.
   */

  ilu_Condition	cond;		/* used to wait for I/O */

  /* L2 unconstrained */

  ilu_TIH			tih;
  unsigned int			session_id; /* integer designator for this transport */
  unsigned int			channel_id;
  TransportData			group;
  int *				loophandle; /* for single threaded tc_wait_for_input */
  ilu_boolean			interrupted;/* TRUE if has been interrupted */
}		*TransportParms;

typedef struct {
  ilu_Mutex			lock;
  MuxHeader			header;
  ilu_TIH			tih;
  ilu_Mooring			self;		/* back pointer to self */
  ilu_cardinal			channel_id;
  MooringData			group;
  ilu_integer			disabled;
}		*MooringParms;

typedef struct {
  ilu_cardinal			channel_id;
  ilu_TransportInfo		lower_tinfo;
  ilu_TransportCreator		lower;
}		*CreatorParms;

typedef struct ConnReqList_s {
  MuxHeader			header;
  TransportData			transport;
  struct ConnReqList_s *	next;
} *ConnReqList;

/**********************************************************************/
/**********************************************************************/
/*								      */
/*			Macro definitions			      */
/*								      */
/**********************************************************************/
/**********************************************************************/

#define MUX_LONG_LENGTH         0x80000000
#define MUX_CONTROL             0x40000000
#define MUX_SYN                 0x20000000
#define MUX_FIN                 0x10000000
#define MUX_RST                 0x08000000
#define MUX_PUSH                0x04000000
#define MUX_SESSION             0x03FC0000
#define MUX_LENGTH              0x0003FFFF

#define SHORT_HEADER_SIZE	0x0003FFFF

#define MIN_DYNAMIC_CHANNEL_ID	0x0003F000
#define MAX_DYNAMIC_CHANNEL_ID	0x0003FFFF

/* control messages */
#define MUX_DefineString	0
#define MUX_DefineStack		1
#define MUX_MuxControl		2
#define MUX_SendCredit		3
#define MUX_DefineEndpoint	4

#define MUX_INITIAL_CREDIT	4096

#ifdef WORDS_BIGENDIAN
#define FORMAT_HEADER(where,size,eom,fo,channel,session)\
(						\
 (((fo) || ((size) > SHORT_HEADER_SIZE)) ?	\
  ((where)[0] = ((fo) ? ((channel) & 0xFF) : 0),\
   (where)[1] = ((fo) ? (((channel) >> 8) & 0xFF) : 0),	\
   (where)[2] = ((((session) << 2) & 0xFC) | ((fo) ? (((channel) >> 16) & 0x3) : 0)),	\
   (where)[3] = ((((session) >> 6) & 0x3) | ((fo) ? 0x20 : 0) | ((eom) ? 0x04 : 0) | 0x80), \
   (where)[4] = ((size) & 0xFF),		\
   (where)[5] = (((size) >> 8) & 0xFF),		\
   (where)[6] = (((size) >> 16) & 0xFF),	\
   (where)[7] = (((size) >> 24) & 0xFF)) :	\
  ((where)[0] = ((size) & 0xFF),		\
   (where)[1] = (((size) >> 8) & 0xFF),		\
   (where)[2] = ((((size) >> 16) & 0x3) | (((session) << 2) & 0xFC)), \
   (where)[3] = ((((session) >> 6) & 0x3F) | ((eom) ? 0x04 : 0)))), \
 (fo) = ilu_FALSE				\
)
#else
#define FORMAT_HEADER(where,size,eom,fo,channel,session)\
(						\
 (((fo) || ((size) > SHORT_HEADER_SIZE)) ?	\
  ((where)[3] = ((fo) ? ((channel) & 0xFF) : 0),\
   (where)[2] = ((fo) ? (((channel) >> 8) & 0xFF) : 0),	\
   (where)[1] = ((((session) << 2) & 0xFC) | ((fo) ? (((channel) >> 16) & 0x3) : 0)),	\
   (where)[0] = ((((session) >> 6) & 0x3) | ((fo) ? 0x20 : 0) | ((eom) ? 0x04 : 0) | 0x80), \
   (where)[7] = ((size) & 0xFF),		\
   (where)[6] = (((size) >> 8) & 0xFF),		\
   (where)[5] = (((size) >> 16) & 0xFF),	\
   (where)[4] = (((size) >> 24) & 0xFF)) :	\
  ((where)[3] = ((size) & 0xFF),		\
   (where)[2] = (((size) >> 8) & 0xFF),		\
   (where)[1] = ((((size) >> 16) & 0x3) | (((session) << 2) & 0xFC)), \
   (where)[0] = ((((session) >> 6) & 0x3) | ((eom) ? 0x04 : 0)))), \
 (fo) = ilu_FALSE				\
)
#endif

#ifdef WORDS_BIGENDIAN
#define PARSE_HEADER(ptr,last,size)		\
	((last) = (((ptr)[0] & 0x80) != 0),	\
	 (size) = ((((ptr)[0] & 0x7F) << 24) |	\
		   ((ptr)[1] << 16) |		\
		   ((ptr)[2] << 8) |		\
		   ((ptr)[3])))
#else
#define PARSE_HEADER(ptr,last,size)		\
	((last) = (((ptr)[3] & 0x80) != 0),	\
	 (size) = ((((ptr)[3] & 0x7F) << 24) |	\
		   ((ptr)[2] << 16) |		\
		   ((ptr)[1] << 8) |		\
		   ((ptr)[0])))
#endif

#define BUFFERSIZE		(8192 + 8)
#define DIRECT_THRESHOLD	(BUFFERSIZE - 8)
#define SOFTSIZE(p)		((p)->inSize - 8)
#define HARDSIZE(p)		((p)->inSize)
#define HEADERSIZE(h)		(sizeof(MuxHeader) + ((((MuxHeader *)(h))->contents.data_hdr.long_length) ? 4 : 0))
#define HEADERLENGTH(size,fo)	(((fo) || ((size) > SHORT_HEADER_SIZE)) ? 8 : 4)

#define CONNECTION_TABLE_BUCKETS	23

#define BYTE_SWAP_WORD(a) ( ((a) << 24) | \
			   (((a) << 8) & 0x00ff0000) | \
			   (((a) >> 8) & 0x0000ff00) | \
			   ((ilu_cardinal)(a) >>24) )

#ifdef WORDS_BIGENDIAN
#define LITTLE_ENDIAN_TO_NATIVE(a)	BYTE_SWAP_WORD(a)
#define NATIVE_TO_LITTLE_ENDIAN(a)	BYTE_SWAP_WORD(a)
#define BIG_ENDIAN_TO_NATIVE(a)		(a)
#define NATIVE_TO_BIG_ENDIAN(a)		(a)
#else
#define LITTLE_ENDIAN_TO_NATIVE(a)	(a)
#define NATIVE_TO_LITTLE_ENDIAN(a)	(a)
#define BIG_ENDIAN_TO_NATIVE(a)		BYTE_SWAP_WORD(a)
#define NATIVE_TO_BIG_ENDIAN(a)		BYTE_SWAP_WORD(a)
#endif

#define ROUND_UP(val,pt)	((((val)%(pt))==0)?(val):((val)+((pt)-((val)%(pt)))))

#define W3MUX_MOORINGPARMS(m)	((MooringParms)((m)->mo_data))
#define W3MUX_MOORINGDATA(m)	(((MooringParms)((m)->mo_data))->group)
#define W3MUX_TRANSPORTPARMS(t) ((TransportParms)((t)->tr_data))
#define W3MUX_TRANSPORTDATA(t)	(((TransportParms)((t)->tr_data))->group)

#define TRANSPORT_DATA_NAME(td)	(((td)->initiator == NIL) ? (td)->tinfo[0] : (td)->peerinfo)
#define TRANSPORT_DATA_SESSION(td)	((td)->input_session)
#define SESSION_NAME(t)			TRANSPORT_DATA_NAME(W3MUX_TRANSPORTDATA(t)), W3MUX_TRANSPORTPARMS(t)->session_id, W3MUX_TRANSPORTPARMS(t)->channel_id

#define transport_has_exposed_input(t)  ((t)->tr_inBuff && ((t)->tr_inNext < (t)->tr_inLimit))

/**********************************************************************/
/**********************************************************************/
/*								      */
/*			Global variables			      */
/*								      */
/**********************************************************************/
/**********************************************************************/

static HashTable	w3mux_Channels = NIL;
/* This is a hash table which maps an 18-bit channel ID to the
 * ilu_Mooring which handles requests for that channel.
 */
static MooringDataList	w3mux_Moorings = NIL;
/* List of all lower-level, putatively tcp, moorings that are
 * used by channels.  Often this will be only a single entry.
 */
static ilu_Mutex	w3mux_ChannelsLock = NIL;
/* Mutex to protect w3mux_Channels and w3mux_Moorings */

static TransportDataList w3mux_Transports = NIL;
/* List of all lower-level, putatively tcp, transports that are
 * used by sessions.
 */
static HashTable	w3mux_Endpoints = NIL;
/* This is a hash table which maps tinfo strings of the mooring
 * layers under the w3mux layer (typically just a TCP/IP layer)
 * into w3mux mooring equivalence classes.  The key is a tinfo
 * element, typically "tcp_*_*", and the data is a list of
 * TransportData.  We take special care with tinfo of the form
 * [ "tcp_*_*" ], that is, one item beginning with TCP.  The tinfo
 * fields representing ip addr and port are canonicalized so that
 * we can participate in the DefineEndpoint messages properly.
 */
static ilu_Mutex	w3mux_EndpointsLock = NIL;
/* Mutex to protect both w3mux_Endpoints and w3mux_Transports */

static ConnReqList	w3mux_ConnReqs = NIL;
static ilu_Mutex	w3mux_ConnReqsLock = NIL;
static ilu_Condition	w3mux_ConnReqsCond = NIL;
/* List of incoming connection requests to be processed, and mutex to
 * protect them.  In the threaded case, all moorings wait on the condition
 * for a new accept request to come it.
 */

static MuxHeader	w3mux_ZeroHeader = { 0 };
/* Zero'ed out header used to initialized malloc'ed vars of type MuxHeader */

/**********************************************************************/
/**********************************************************************/
/*								      */
/*			Transport functions			      */
/*								      */
/**********************************************************************/
/**********************************************************************/

/* ...while holding td->lock... */
static void
  AddSession (TransportData md,
	      ilu_Transport session,
	      ilu_Error *err)
{
  TransportList	ml;

  ml = ilu_MallocE(sizeof(*ml), err);
  if (ILU_ERRNOK(*err)) return;
  ml->next = md->sessions;
  ml->transport = session;
  md->sessions = ml;
}

/* ...while holding td->lock... */
static void
  RemoveSession (TransportData md,
		 ilu_Transport session,
		 ilu_Error *err)
{
  TransportList p, q;

  for (p = md->sessions, q = NIL;  p != NIL;  q = p, p = p->next) {
    if (p->transport == session) {
      if (q == NIL)
	md->sessions = p->next;
      else
	q->next = p->next;
      ilu_free(p);
      break;
    }
  }
  ILU_CLER(*err);
}

/* ...while holding td->lock... */
static ilu_Transport
  FindSession (TransportData td, unsigned int id)
{
  TransportList p;

  for (p = td->sessions;  p != NIL;  p = p->next) {
    if (W3MUX_TRANSPORTPARMS(p->transport)->session_id == id) {
      return (p->transport);
    }
  };
  return NIL;
}

/* ...while holding some lock... */
static void
  RemoveTransportFromList (TransportDataList *l,
			   TransportData td)
{
  TransportDataList p, q;

  for (p = *l, q = NIL;  p != NIL;  q = p, p = p->next) {
    if (p->td == td) {
      if (q == NIL)
	*l = p->next;
      else
	q->next = p->next;
      ilu_free(p);
      break;
    }
  }
}

/* L1.sup < trmu; L2 >= {xmu, ymu} */
static ilu_integer
  _w3mux_CloseDFd (ilu_Transport self)
/*
 * Estimates how many FDs would be freed by closing this
 * transport.  May err high, but not low.
 */
{
  return 1;
}

static void tHandler (void *);

/* L1.sup < trmu; L2 >= {xmu, ymu} */
static ilu_boolean
  _w3mux_SetInputHandler (ilu_Transport self,
			  ilu_TIH tih,
			  ILU_ERRS((no_memory, internal, no_resources)) * err)
{
  TransportParms	p = W3MUX_TRANSPORTPARMS(self);
  TransportData		td = W3MUX_TRANSPORTDATA(self);
  ilu_boolean ans;

  ilu_AcquireMutex(p->lock);
  if (tih &&
      (p->inWedged || (p->t_inNext < p->inLength)
       || p->busyIn && p->lastIn)) {
    ans = ilu_FALSE;
  } else {
    ilu_AcquireMutex(td->state_lock);
    if (tih && (!td->active)) {
      ans = _ilu_SetTransportInputHandler(td->transport, &td->td_loClo, err);
      if (ILU_ERRNOK(*err)) {
	ilu_ReleaseMutex(td->state_lock);
	ilu_ReleaseMutex(p->lock);
	return ilu_FALSE;
      } else if (ans) {		/* something already here */
	td->active = ilu_TRUE;
      };
    } else {
      ans = ilu_TRUE;
    };
    ilu_ReleaseMutex(td->state_lock);
    p->tih = tih;
  }
  ilu_ReleaseMutex(p->lock);
  ILU_CLER(*err);
  return ans;
}

/* Main Invariant holds; L2 >= {ymu} */
static ilu_boolean
  _w3mux_WaitForInput(ilu_Transport self,
		      int * disabled_p,
		      ilu_FineTime * limit,
		      ILU_ERRS((interrupted)) * err)
/*
 * Returns ILU_ERROK(*err) after either: (1) there is a decent
 * reason to suspect that, after a tc_begin_message if boundaried
 * and not already reading a message, "input progress" can be
 * made, or an error or EOM or EOF can be detected, without
 * blocking, (2) *limit passed (limit==NIL means *limit ==
 * +infinity), or (3) interrupt requested in a multi-threaded
 * runtime (in which case interrupted is raised), (4)
 * tc_interruptST(self, ..) was called in a single-threaded
 * runtime (in which case interrupted is *not* raised), (5) the
 * runtime is multi-threaded and waiting is disabled for (self)'s
 * wait-cohort (see tc_disableWait).  Sets (*disabled), to a
 * non-zero value when only (5) applies; this is only a hint, and
 * can be wrong (with the only adverse consequence being a small
 * performance hit) --- as long as false positives are not
 * frequent.  The exposed input buffer holds no more input (else
 * internal/tcInputSkipsBuff is raised).  See
 * transport_wait_for_input for a convenient wrapper.  "Input
 * progress" means any advancement in the state of any stage in
 * the input pipeline --- regardless of whether any bytes dribble
 * out this end.  Used in single-threaded and multi-threaded
 * runtimes.  In S-T R/T, blocks by running main loop; in M-T,
 * blocks only the calling thread.
 */
{
  int loophandle;
  int *old_loophandle_p;
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  TransportData td = W3MUX_TRANSPORTDATA(self);
  ilu_boolean interrupted = ilu_FALSE, result;

  ilu_AcquireMutex(p->lock);
  if (p->inWedged ||
      p->eofRcvd || (p->t_inNext < p->inLength)
      || p->busyIn && p->lastIn) {
    ilu_ReleaseMutex(p->lock);
    return (*disabled_p = 0, ILU_CLER(*err));
  } else {
    ilu_AcquireMutex(td->state_lock);
    if (((td->transport)->tr_inBuff) &&
	(((td->transport)->tr_inNext < (td->transport)->tr_inLimit))) {
      ilu_ReleaseMutex(td->state_lock);
      ilu_ReleaseMutex(p->lock);
      return (*disabled_p = 0, ILU_CLER(*err));
    };      
    ilu_ReleaseMutex(td->state_lock);
    if (p->cond != NIL) {
      ilu_CMWait1(p->cond, p->lock, err);
      if (ILU_ERRNOK(*err)) return ilu_FALSE;
      *disabled_p = (p->disabled > 0);
      interrupted = p->interrupted;
      p->interrupted = ilu_FALSE;
    } else {
      old_loophandle_p = p->loophandle;
      loophandle = 0;
      p->loophandle = &loophandle;
      ilu_AcquireMutex(td->state_lock);
      if (!td->active) {
	result = _ilu_SetTransportInputHandler(td->transport, &td->td_loClo, err);
	if (ILU_ERRNOK(*err)) {
	  ilu_ReleaseMutex(td->state_lock);
	  ilu_ReleaseMutex(p->lock);
	  p->loophandle = old_loophandle_p;
	  return ilu_FALSE;
	} else if (!result) {	/* something already here */
	  *disabled_p = ilu_FALSE;
	  interrupted = ilu_FALSE;
	  ilu_ReleaseMutex(td->state_lock);
	  p->loophandle = old_loophandle_p;
	  goto ret1;
	} else
	  td->active = ilu_TRUE;
      }
      ilu_ReleaseMutex(td->state_lock);
      ILU_NOTE(W3MUX_DEBUG,
	    ("_w3mux_WaitForInput:  %s(%u=>%u), waiting for input with loophandle %p\n",
	     SESSION_NAME(self), p->loophandle));
      ilu_ReleaseMutex(p->lock);
      ilu_RunMainLoop (&loophandle);
      ilu_AcquireMutex(p->lock);
      *disabled_p = (p->disabled > 0);
      p->loophandle = old_loophandle_p;
      interrupted = p->interrupted;
      p->interrupted = ilu_FALSE;
    }
  }

 ret1:
  ilu_ReleaseMutex(p->lock);
  ILU_NOTE(W3MUX_DEBUG,
	("_w3mux_WaitForInput (%s(%u=>%u)) : interrupted=%s, disabled=%s\n",
	 SESSION_NAME(self), interrupted ? "True" : "False", *disabled_p ? "True" : "False"));
  if (interrupted)
    return ILU_ERR_CONS0(interrupted, err, ilu_FALSE);
  else
    return ILU_CLER(*err);
}

/* L1.sup < trmu; L2 >= {xmu} */
static ilu_boolean
  _w3mux_Interrupt (ilu_Transport self,
		    ILU_ERRS((bad_param)) * err)
/*
 * Applicable only in single-threaded runtime.  Causes current
 * calls on tc_wait_for_input, tc_begin_message, tc_end_message,
 * and tc_write_bytes to return in the appropriate way.
 */
{
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  ilu_AcquireMutex(p->lock);
  ILU_NOTE(W3MUX_DEBUG, ("_w3mux_Interrupt:  %s(%u=>%u)\n", SESSION_NAME(self)));
  if (p->loophandle != NIL) {
    p->interrupted = TRUE;
    ilu_ExitMainLoop(p->loophandle);
  };
  ilu_ReleaseMutex(p->lock);
  return ILU_CLER(*err);
}

/* L1 >= {cmu}; L1.sup < trmu; L2 >= {xmu} */
static ilu_boolean
  _w3mux_DisableWait (ilu_Transport self,
		      ILU_ERRS((broken_locks,
				bad_param,
				internal)) *err)
/*
 * Applicable only in multi-threaded runtime.  Caller ensures that
 * for each T, at every moment, the number of tc_disableWait(T,..)
 * calls that have been made so far is always at least as great as
 * the number of tc_enableWait(T,..) calls, regardless of thread
 * scheduling.  Moorings/Transports are partitioned into disjoint
 * sets called `wait-cohorts'.  Each mooring and each transport is
 * in charge of identifying which set it belongs to; these sets
 * are not explicitly reified --- they only appear in the
 * following definition.  While there have been more
 * {mo,tc}_disableWait calls on members of (self)'s wait-cohort
 * than {mo,tc}_enableWait calls, we say waiting is disabled for
 * (self)'s wait-cohort.  This condition affecte the behavior of
 * tc_wait_for_input, above.
 */
{
  TransportParms tp = W3MUX_TRANSPORTPARMS(self);
  ilu_Condition cond;
  ilu_boolean disabled;

  ilu_AcquireMutex(tp->lock);
  tp->disabled += 1;
  disabled = (tp->disabled > 0);
  cond = tp->cond;
  ilu_ReleaseMutex(tp->lock);
  if (disabled)
    ilu_NotifyCondition(cond);
  return ILU_CLER(*err);
}

/* L1 >= {cmu}; L1.sup < trmu; L2 >= {xmu} */
static ilu_boolean
  _w3mux_EnableWait (ilu_Transport self, 
		     ILU_ERRS((broken_locks,
			       bad_param,
			       internal)) *err)
/*
 * [see comment in _w3mux_DisableWait]
 */
{
  TransportParms tp = W3MUX_TRANSPORTPARMS(self);
  ilu_Condition cond;

  ilu_AcquireMutex(tp->lock);
  tp->disabled -= 1;
  ilu_ReleaseMutex(tp->lock);
  return ILU_CLER(*err);
}

static ilu_cardinal
  figure_size (ilu_Transport	self,
	       TransportParms	p,
	       ilu_cardinal	dsize)
{
  ilu_integer  amount;
  ilu_cardinal headersize;
  ilu_cardinal size;

  /* returns max number of bytes which can be currently output on self */
  if (p->usesCredit) {
    amount = MIN(dsize, p->outCredit - 4);
    while (amount > 0) {
      headersize = (amount > SHORT_HEADER_SIZE) ? 8 : 4;
      size = ROUND_UP(amount + headersize, headersize);
      if (size <= p->outCredit)
	return amount;
      else
	amount -= 4;
    }
    return 0;
  } else {
    return dsize;
  }
}

static void
  WaitForCredit (ilu_Transport self,
		 TransportParms p,
		 ilu_Error *err)
{
  ilu_integer loophandle;

  while (p->outCredit < 5) {
    if (p->cond != NIL) {
      ilu_CMWait1(p->cond, p->lock, err);
      if (ILU_ERRNOK(*err)) return;
    } else {
      loophandle = 0;
      p->loophandle = &loophandle;
      ilu_ReleaseMutex(p->lock);
      ilu_RunMainLoop(p->loophandle);
      ilu_AcquireMutex(p->lock);
      p->loophandle = NIL;
    }
  }      
}

/* while holding p->lock, !holding td->io_lock */
static ilu_boolean
  DoWriteWork (ilu_Transport	self,
	       TransportParms	p,
	       TransportData	td,
	       ilu_bytes	buffer,		/* == NIL means self->tr_outBuff */
	       ilu_cardinal	totalsize,	/* == 0 means empty self->tr_outBuff */
	       ilu_boolean	lastChunk,
	       ilu_boolean	flush,
	       ilu_Error *	err)
{
  /* Note:  We can only write chunks that are less than the output credit
   * for "self", including the padding and header.
   */
  ilu_cardinal amount, headerlength;
  ilu_cardinal outstart, size;
  ilu_byte padding[8];
  ilu_boolean ans, lflush, lastflag;

  /* we are writing from our internal buffer, and we
   * need to empty it.  We adjust our writes to the output credit that is
   * available.
   */
  while (p->outStart < self->tr_outNext) {
    WaitForCredit (self, p, err);
    if (ILU_ERRNOK(*err)) return ilu_FALSE;
    amount = figure_size (self, p, self->tr_outNext - p->outStart);
    if (amount > 0) {
      headerlength = HEADERLENGTH(amount,p->firstOut);
      outstart = p->outStart - headerlength;
      lastflag = lastChunk && (buffer == NIL) && (amount == (self->tr_outNext - p->outStart));
      FORMAT_HEADER(self->tr_outBuff + outstart, amount, lastflag,
		    p->firstOut, p->channel_id, p->session_id);
      lflush = ((amount % headerlength) == 0) && (flush || (amount < (self->tr_outNext - p->outStart)));
      ilu_AcquireMutex(td->io_lock);
      ans = (*td->transport->tr_class->tc_write_bytes)
	(td->transport, self->tr_outBuff + outstart, headerlength + amount, lflush, err);
      if (ans && ((amount % headerlength) != 0)) {
	lflush = (flush || (amount < (self->tr_outNext - p->outStart)));
	ans = (*td->transport->tr_class->tc_write_bytes)
	  (td->transport, padding, headerlength - (amount % headerlength), lflush, err);
      }
      ilu_ReleaseMutex(td->io_lock);
      if (!ans) return ilu_FALSE;
      p->outStart += amount;
      p->outCredit -= (headerlength + amount + (((amount % headerlength) != 0) ? (headerlength - (amount % headerlength)) : 0));
    }
  }
  self->tr_outNext = 8;
  p->outStart = 8;

  if (buffer != NIL) {
    /* Empty this buffer, too, but we need to write the headers separately */
    size = 0;
    while (size < totalsize) {
      WaitForCredit (self, p, err);
      if (ILU_ERRNOK(*err)) return ilu_FALSE;
      amount = figure_size (self, p, totalsize - size);
      if (amount > 0) {
	headerlength = HEADERLENGTH(amount, p->firstOut);
	lastflag = lastChunk && (amount == (totalsize - size));
	FORMAT_HEADER(padding, amount, lastflag, p->firstOut, p->channel_id, p->session_id);
	lflush = ((amount % headerlength) == 0) && (flush || (amount < (totalsize - size)));
	ilu_AcquireMutex(td->io_lock);
	ans = (*td->transport->tr_class->tc_write_bytes)
	  (td->transport, padding, headerlength, ilu_FALSE, err);
	if (ans)
	  ans = (*td->transport->tr_class->tc_write_bytes)
	    (td->transport, buffer + size, amount, lflush, err);
	if (ans && ((amount % headerlength) != 0)) {
	  lflush = (flush || (amount < (totalsize - size)));
	  ans = (*td->transport->tr_class->tc_write_bytes)
	    (td->transport, padding, headerlength - (amount % headerlength), lflush, err);
	}
	ilu_ReleaseMutex(td->io_lock);
	if (!ans) return ilu_FALSE;
	p->outCredit -= (headerlength + amount + (((amount % headerlength) != 0) ? (headerlength - (amount % headerlength)) : 0));
	size += amount;
      }
    }
  }
  ILU_CLER(*err);
  return ilu_TRUE;
}

/* while holding p->lock, !holding td->io_lock */
static ilu_boolean
  SendCredit (ilu_Transport self,
	      TransportParms p,
	      TransportData td,
	      ilu_Error *err)
{
  ilu_boolean immediate = ilu_TRUE;
  struct {
    MuxHeader		hdr;
    ilu_cardinal	amount;
  } msg;
  ILU_NOTE(W3MUX_DEBUG,
	   ("(w3mux) SendCredit:  %s(%u=>%u) sending credit of %u bytes%s...\n",
	    SESSION_NAME(self), p->accumCredit, immediate ? " (immediately)" : ""));
  ilu_AcquireMutex(td->io_lock);
  msg.hdr.contents.control_message.long_length = 1;
  msg.hdr.contents.control_message.control = 1;
  msg.hdr.contents.control_message.control_code = MUX_SendCredit;
  msg.hdr.contents.control_message.session_id = p->session_id;
  msg.hdr.contents.control_message.session_fragment_size = 0;
  msg.hdr.contents.word = NATIVE_TO_LITTLE_ENDIAN(msg.hdr.contents.word);
  msg.amount = NATIVE_TO_LITTLE_ENDIAN(p->accumCredit);
  (*td->transport->tr_class->tc_write_bytes)(td->transport, (ilu_bytes) &msg,
					     sizeof(msg), immediate, err);
  ilu_ReleaseMutex(td->io_lock);
  p->accumCredit = 0;
  return (ILU_ERROK(*err));
}

/* while holding p->lock, !holding td->io_lock */
static ilu_cardinal
  ReadWork(ilu_Transport self,
	   ilu_bytes buffer,
	   ilu_cardinal len,
	   ilu_TransportReport * rpt,
	   int *progress,
	   ILU_ERRS((IoErrs)) * err)
{
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  TransportData td = W3MUX_TRANSPORTDATA(self);
  ilu_Transport td_session;
  ilu_cardinal rem = p->t_inLimit - p->t_inNext;

  rpt->tr_eom = rpt->tr_eof = *progress = 0;
  if (!p->lastIn) {
    /* process another chunk. */
    ilu_cardinal    csize, cbsize;
    if ((p->t_inNext + 4) <= p->inLength) {
      /* Chunk's header already in buffer. */
      *progress = 1;
    } else if (p->eofRcvd) {
      rpt->tr_eof = ilu_TRUE;
      ILU_CLER(*err);
      return 0;
    } else {
      /* This should never happen.  tHandler should always put another header in the
       * buffer before allowing any routine that calls ReadWork to be called.
       * However, it may happen at EOF.
       */
      ILU_NOTE(W3MUX_DEBUG, ("(w3mux) ReadWork(%s(%u=>%u)) -- no bytes\n", SESSION_NAME(self)));
      ILU_CLER(*err);
      return 0;
    }
    p->accumCredit += p->prevChunkSize;
    PARSE_HEADER(p->t_inBuff + p->t_inNext, p->lastIn, csize);
    p->firstIn = FALSE;
    p->t_inNext += 4;
    p->accumCredit += 4;
    p->prevChunkSize += csize;
    cbsize = MIN(p->inLength - p->t_inNext, csize);
    p->t_inLimit = p->t_inNext + cbsize;
    SendCredit(self, p, td, err);
    if (err)
      return 0;
    ILU_NOTE(W3MUX_DEBUG,
	     ("(w3mux) ReadWork(%s(%u=>%u)): Parse header of length %lu chunk (last=%d), %lu in buffer.\n",
	      SESSION_NAME(self), csize, p->lastIn != 0, cbsize));
  }
  /* Now use what's in t_inBuff */
  rem = p->t_inLimit - p->t_inNext;
  if (rem == 0 && p->lastIn) {
    rpt->tr_eom = TRUE;
    return 0;
  }
  if (buffer == NIL)
    return rem;
  if (rem > 0) {
    ilu_cardinal    l1 = MIN(rem, len);
    memcpy((void *) buffer,
	   (void *) (p->t_inBuff + p->t_inNext),
	   l1);
    p->t_inNext += l1;
    return l1;
  }
  return 0;
}

/* Main Invariant holds; L2 >= {xmu}; input => L2 >= {ymu} */
static ilu_ReadHeaderResultCode
  _w3mux_BeginMessage (ilu_Transport self,
		       ilu_boolean input_p,
		       ILU_ERRS((IoErrs)) * err)
/*
 * Begin a new message in the given direction (with caveats, in
 * the input case).  Raises internal/tcNotBoundaried if transport
 * is not boundaried.  Raises internal/beginMessage if not done
 * with previous message.  For output, returns either ilu_rhrc_ok
 * or _error.  May return any of the following four codes for
 * input.  Returns _ok if input message successfully started.
 * Returns _eof if EOF instead of a message was waiting; caller
 * will close the ilu_Transport.  May return _nothing to indicate
 * EOF has not been detected and message has not been started (due
 * to lack of input).  Must not block before start of EOF or next
 * message is detected; otherwise, might block.  When
 * single-threaded, will raise interrupted (with meaningless
 * ilu_interruptSet) if tc_interruptST invoked while blocked; when
 * MT, will raise interrupted (with meaningful ilu_interruptSet)
 * when the alert-this-thread signal is detected.  Returns _error
 * iff raising an error, in which case the message has not been
 * started.  Never returns _handled.
 */
{
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  int progress = 1;

  ILU_NOTE(W3MUX_DEBUG,
	("_w3mux_BeginMessage:  %s(%u=>%u), %s\n",
	 SESSION_NAME(self), input_p ? "input" : "output"));
  if (p->busyIn || p->busyOut)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_beginMessage,
			 ilu_rhrc_error);
  ilu_AcquireMutex(p->lock);
/*
  if (input_p && p->eofRcvd) {
    ilu_ReleaseMutex(p->lock);
    ILU_CLER(*err);
    return ilu_rhrc_eof; };
*/
  if (!input_p && p->eofRcvd) {
    ilu_ReleaseMutex(p->lock);
    ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_conn_lost, 0);
    return ilu_rhrc_error;
  }
  p->dirIsIn = input_p;
  if (input_p) {
    ilu_cardinal    csize, cbsize;
    ilu_TransportReport rpt = {FALSE, FALSE};
    if (p->inWedged) {
      ilu_ReleaseMutex(p->lock);
      return ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_conn_lost,
			   ilu_rhrc_error);
    }
    if (p->skipping) {
      while (progress && !rpt.tr_eom) {
	(void) ReadWork(self, NIL, 0, &rpt, &progress, err);
	p->t_inNext = p->t_inLimit;
	if (ILU_ERRNOK(*err)) {
	  p->inWedged = TRUE; p->skipping = FALSE;
	  ilu_ReleaseMutex(p->lock);
	  return ilu_rhrc_error;
	}
      }
      if (!progress) {
	ilu_ReleaseMutex(p->lock);
	return ilu_rhrc_nothing;
      }
      p->skipping = FALSE;
    }
    p->busyIn = p->firstIn = TRUE;
    p->lastIn = FALSE;
    p->prevChunkSize = 0;
    if (p->t_inNext == p->inLength) {
      (void) ReadWork(self, NIL, 0, &rpt, &progress, err);
      if (ILU_ERRNOK(*err)) {
	p->inWedged = TRUE; p->busyIn = FALSE;
	ilu_ReleaseMutex(p->lock);
	return ilu_rhrc_error;
      }
      if (p->t_inNext >= p->inLength) {
	p->busyIn = FALSE;
	ilu_ReleaseMutex(p->lock);
	return (rpt.tr_eof ? ilu_rhrc_eof : ilu_rhrc_nothing);
      }
    } else if ((p->t_inNext + 4) <= p->inLength) {
      PARSE_HEADER(p->t_inBuff + p->t_inNext, p->lastIn, csize);
      p->firstIn = FALSE;
      p->t_inNext += 4;
      p->accumCredit += 4;
      p->prevChunkSize = csize;
      cbsize = MIN(p->inLength - p->t_inNext, csize);
      p->t_inLimit = p->t_inNext + cbsize;
      ILU_NOTE(W3MUX_DEBUG,
	    ("_w3mux_BeginMessage:  %s(%u=>%u), parse header of length %lu chunk (last=%d), %lu in buffer.\n",
	     SESSION_NAME(self), csize, p->lastIn != 0, cbsize));
    }
  } else {
    p->busyOut = TRUE;
    p->outStart = self->tr_outNext += 8;
    self->tr_outLimit = p->outSize - 8;
  }
  ilu_ReleaseMutex(p->lock);
  ILU_CLER(*err);
  return ilu_rhrc_ok;
}

/* Main Invariant holds; L2 >= {xmu}; input => L2 >= {ymu} */
static ilu_boolean
  _w3mux_EndMessage(ilu_Transport self,
		    ilu_boolean flush,
		    ilu_Message * msgh,
		    ILU_ERRS((IoErrs)) * err)
/*
 * Finish the current message in the current direction.  If output
 * and flush, be sure to start it on its way; otherwise, this call
 * might do blocking I/O, and/or it might schedule I/O to be done
 * later.  If single-threaded, blocking is done via the main loop;
 * if multi-threaded, blocking affects only the calling thread. If
 * unreliable and output, return a copy (including ownership) of
 * the whole message (i.e., all the bytes given to tc_write_bytes)
 * to the caller through OUT parameter (msg), so that it may later
 * be retransmitted with tc_send_whole_message; otherwise ignore
 * (msg).  Raises interrupted as for tc_begin_message.  Raises
 * internal/endMessage if not currently I/Oing a message.  Will
 * raise internal/tcBytesDropped if direction is input and the
 * whole message has not yet been read, unless the inability to
 * block prevents accurate determination of whether that condition
 * holds.  Raises internal/tcNotBoundaried if transport is not
 * boundaried.  Raises internal/tcNoMsgHandle if unreliable,
 * output, and msg == NIL.
 */
{
  register TransportParms p = W3MUX_TRANSPORTPARMS(self);
  TransportData td = W3MUX_TRANSPORTDATA(self);
  ilu_boolean ans;

  ilu_AcquireMutex(p->lock);
  if (p->busyOut) {
    ilu_cardinal	n1 = self->tr_outNext - p->outStart;
	
    ILU_NOTE(W3MUX_DEBUG,
	  ("_w3mux_EndMessage:  %s(%u=>%u), output, flush=%d, last chunkSize=%lu)\n",
	   SESSION_NAME(self), (flush != 0), (long unsigned) (n1)));

    ans = DoWriteWork(self, p, td, NIL, 0, ilu_TRUE, ilu_TRUE, err);
    p->busyOut = ilu_FALSE;
    self->tr_outLimit = self->tr_outNext;
    ilu_ReleaseMutex(p->lock);
    return ans;
  } else if (p->busyIn) {
    ilu_boolean     first = TRUE, drops = FALSE;
    ilu_TransportReport rpt = {FALSE, FALSE};
    ILU_NOTE(W3MUX_DEBUG,
	  ("_w3mux_EndMessage:  %s(%u=>%u), input\n",
	   SESSION_NAME(self)));
    if (!p->lastIn) {
      ilu_cardinal    lread;
      int             progress;
      drops = (p->t_inLimit != p->t_inNext);
      p->t_inNext = p->t_inLimit;
      lread = ReadWork(self, NIL, 0, &rpt, &progress, err);
      if (ILU_ERRNOK(*err))
	goto wedge;
    } else {
      p->accumCredit += p->prevChunkSize;
      SendCredit (self, p, td, err);
      if (ILU_ERRNOK(*err))
	goto wedge;
    }
    drops = (drops ||
	     p->t_inLimit != p->t_inNext);
    p->t_inNext = p->t_inLimit;
    p->busyIn = FALSE;
    if (p->lastIn)
      p->skipping = FALSE;
    else
      p->skipping = TRUE;
    ilu_ReleaseMutex(p->lock);
    if (drops)
      return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcBytesDropped,
			   FALSE);
    return ILU_CLER(*err);
wedge:
    p->inWedged = TRUE;
    p->busyIn = FALSE;
    ilu_ReleaseMutex(p->lock);
    return (FALSE);
  } else {
    ilu_ReleaseMutex(p->lock);
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_endMessage, FALSE);
  }
}

/*Main Invariant holds; L2 >= {xmu}*/
static ilu_boolean
  _w3mux_SendWholeMessage(ilu_Transport self,
			  ilu_Message * msgh,
			  ILU_ERRS((IoErrs)) * err)
/*
 * Applicable only to unreliable transports; others may raise
 * internal/tcReliable.  Like {begin_message; write_bytes;
 * end_message}, except that the message is not returned again at
 * the end.  Caller retains ownership of *msg and *msg->msg_base.
 */
{
  return ILU_ERR_CONS1(internal, err, minor, ilu_im_tcReliable, FALSE);
}

/*Main Invariant holds; L2 >= {xmu}*/
static ilu_boolean
  _w3mux_WriteBytes (ilu_Transport self,
		     ilu_bytes b,
		     ilu_cardinal bufferSize,
		     ilu_boolean flush,
		     ILU_ERRS((IoErrs)) * err)
/*
 * Write the contents of the exposed buffer (if any), followed by
 * the given buffer (if buf != NIL).  If flush and not boundaried,
 * be sure to start these bytes on their way; flush is not
 * significant when boundaried.  Prepare the exposed buffer to
 * receive at least 16 more bytes of output.  Raises interrupted
 * as for tc_begin_message.  Raises internal/bytesWithoutMsg if
 * boundaried and not currently outputting a message.  Caller
 * retains ownership of buf.
 */
{
  TransportParms	p = W3MUX_TRANSPORTPARMS(self);
  TransportData		td = W3MUX_TRANSPORTDATA(self);
  ilu_boolean		ans;

  ilu_AcquireMutex(p->lock);
  if (!p->busyOut) {
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_bytesWithoutMsg, FALSE);
  }
  ans = DoWriteWork(self, p, td, b, bufferSize, FALSE, flush, err);
  if (!ilu_Check((self->tr_outLimit - self->tr_outNext) >= 16, err))
    ans = ilu_FALSE;
  ilu_ReleaseMutex(p->lock);
  return ans;
}

/*Main Invariant holds; L2 >= {xmu, ymu}*/
static ilu_cardinal
  _w3mux_ReadBytes (ilu_Transport self,
		    ilu_bytes buffer,
		    ilu_cardinal len,
		    ilu_TransportReport * rpt,
		    ILU_ERRS((IoErrs)) * err)
/*
 * Read some bytes into the given buffer (buf==NIL means the
 * exposed buffer is being given and bufLen is not meaningful).
 * The exposed input buffer holds no more input (else
 * internal/tcInputSkipsBuff is raised).  When buf==NIL, result is
 * final inLimit - inNext.  Note that buf==NIL grants freedom to
 * transfer headers and trailers along with payload.  Does not
 * read past message boundary (if boundaried), EOF, or end of
 * given buffer.  Does not block (so any loop involving this
 * procedure must also involve tc_wait_for_input).  Makes some
 * input progress, if permitted by the above restrictions.  Sets
 * EOM and EOF bits in *rpt; may falsely set them to FALSE if at
 * least 1 byte was delivered.  An unboundaried transport always
 * sets EOM to FALSE.  EOF (between messages for boundaried
 * transports, anywhere at all for others) is not a reason to
 * raise an error.  Returns number of bytes transferred into given
 * buffer, even when an error is raised.  Raises
 * internal/bytesWithoutMsg if boundaried and not currently
 * inputting a message.  Caller retains ownership of *buf.
 */
{
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  int progress;
  ilu_cardinal ans;

  if (!p->busyIn)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_bytesWithoutMsg, 0);
  if (buffer == NIL)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_broken, 0);
  ilu_AcquireMutex(p->lock);

  if (p->t_inBuff != NIL &&
      (p->t_inNext < p->t_inLimit)) {
    ans = MIN(len, p->t_inLimit - p->t_inNext);
    memcpy((void *) buffer, (void *) (p->t_inBuff + p->t_inNext), ans);
    p->t_inNext += ans;
    ILU_CLER(*err);
  } else {
    ans = ReadWork(self, buffer, len, rpt, &progress, err);
  }
  ilu_ReleaseMutex(p->lock);
  return ans;
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static ilu_boolean
  _w3mux_Close (ilu_Transport self,
		ilu_integer * dfd,
		ILU_ERRS((internal)) * err)
{
  MuxHeader header;
  TransportData td = W3MUX_TRANSPORTDATA(self);
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  ilu_Error lerr;

  ILU_NOTE(W3MUX_DEBUG,
	("_w3mux_Close:  %s(%u=>%u), eofRcvd=%s\n",
	 SESSION_NAME(self), p->eofRcvd ? "TRUE" : "FALSE"));

  ilu_AcquireMutex(p->lock);
  ilu_AcquireMutex(td->io_lock);
  *dfd = 0;
  if (!p->eofRcvd) {
    /* Send session shutdown message */
    header.contents.data_hdr.long_length = 0;
    header.contents.data_hdr.control = 0;
    header.contents.data_hdr.syn = 0;
    header.contents.data_hdr.fin = 1;
    header.contents.data_hdr.rst = 0;
    header.contents.data_hdr.push = 1;
    header.contents.data_hdr.session_id = p->session_id;
    header.contents.data_hdr.session_fragment_size = 0;
    header.contents.word = NATIVE_TO_LITTLE_ENDIAN(header.contents.word);
    ILU_NOTE(W3MUX_DEBUG,
	     ("_w3mux_Close:  sending close message for %s(%u=>%u)...\n",
	      SESSION_NAME(self)));
    (*td->transport->tr_class->tc_write_bytes)(td->transport, (ilu_bytes) &header, sizeof(header), ilu_TRUE, err);
    if (ILU_ERRNOK(*err)) return ilu_FALSE;
  };
  RemoveSession(td, self, err);
  ilu_ReleaseMutex(td->io_lock);
  if (ILU_ERRNOK(*err)) return ilu_FALSE;
  if (p->tih) {
    ilu_Closure     c = _ilu_ClosureFromTIH(p->tih);
    ILU_ERRS((internal, bad_param, bad_locks, broken_locks)) lerr;
    if (!ilu_Check(ilu_DoSoon(c, &lerr), err)) {
      ILU_HANDLED(lerr);
      return FALSE;
    }
    p->tih = NIL;
  }
  ilu_free(p->t_inBuff);
  ilu_free(self->tr_outBuff);
  ilu_ReleaseMutex(p->lock);
  ilu_DestroyMutex(p->lock, &lerr);
  ILU_HANDLED(lerr);
  if (p->cond != NIL)
    ilu_DestroyCondition(p->cond);
  ilu_free(p);
  ilu_free(self);
  return ILU_CLER(*err);
}

static struct _ilu_TransportClass_s w3mux_TransportClass = {
  ilu_TRUE,			/* boundaried */
  ilu_TRUE,			/* reliable */
  _w3mux_CloseDFd,
  _w3mux_SetInputHandler,
  _w3mux_WaitForInput,
  _w3mux_Interrupt,
  _w3mux_DisableWait,
  _w3mux_EnableWait,
  _w3mux_BeginMessage,
  _w3mux_EndMessage,
  _w3mux_SendWholeMessage,
  _w3mux_WriteBytes,
  _w3mux_ReadBytes,
  _w3mux_Close
};

/**********************************************************************/
/**********************************************************************/
/*								      */
/*			Mooring Functions			      */
/*								      */
/**********************************************************************/
/**********************************************************************/

static ilu_Mooring
  FindMooring (ilu_cardinal id)
{
  ilu_Mooring m = NIL;

  ilu_AcquireMutex(w3mux_ChannelsLock);
  m = ILU_HASH_FINDINTABLE(w3mux_Channels, (ilu_refany) id);
  ilu_ReleaseMutex(w3mux_ChannelsLock);
  return m;
}

/* Main Invariant holds; L2 >= {xmu, ymu} */
static ilu_Transport
  NewTrans (TransportData td, unsigned int session_id, unsigned int channel_id, ilu_boolean firstOut, ilu_Error *err)
{
  TransportParms tp;
  ilu_Transport t;
  char buf[1024];
  ilu_Error lerr;

  t = (ilu_Transport) ilu_MallocE(sizeof(*t), err);
  t->tr_inBuff = NIL;
  t->tr_outBuff = NIL;
  if (ILU_ERRNOK(*err)) goto err1;
  tp = (TransportParms) ilu_MallocE(sizeof(*tp), err);
  if (ILU_ERRNOK(*err)) goto err2;
  tp->t_inBuff = ilu_MallocE(BUFFERSIZE, err);
  if (ILU_ERRNOK(*err)) goto err3;
  t->tr_outBuff = ilu_MallocE(BUFFERSIZE, err);
  if (ILU_ERRNOK(*err)) goto err4;
  sprintf (buf, "mux (%u=>%u) over", session_id, channel_id);
  tp->lock = ilu_CreateMutex(buf, TRANSPORT_DATA_NAME(td));
  if (tp->lock == NIL) goto err5;
  if (ilu_CanCondition()) {
    tp->cond = ilu_CreateCondition(buf, TRANSPORT_DATA_NAME(td), err);
    if (ILU_ERRNOK(*err)) goto err6;
  } else
    tp->cond = NIL;
  tp->busyIn = tp->busyOut = FALSE;
  tp->inWedged = tp->skipping = FALSE;
  tp->inLength = 0;
  tp->inSize = tp->outSize = BUFFERSIZE;
  tp->maxFragSize = 0;
  tp->outCredit = MUX_INITIAL_CREDIT;
  tp->eofRcvd = ilu_FALSE;
  tp->firstOut = firstOut;
  tp->group = td;
  tp->session_id = session_id;
  tp->channel_id = channel_id;
  tp->accumCredit = 0;
  tp->prevChunkSize = 0;
  tp->tih = NIL;
  tp->loophandle = NIL;
  tp->t_inNext = tp->t_inLimit = 0;
  tp->disabled = ilu_FALSE;
  tp->interrupted = ilu_FALSE;
  t->tr_outNext = t->tr_outLimit = 0;
  t->tr_inBuff = NIL;
  t->tr_inNext = t->tr_inLimit = 0;
  t->tr_class = &w3mux_TransportClass;
  t->tr_data = tp;
  ILU_CLER(*err);
  return t;

 err6:
  ilu_DestroyMutex(tp->lock, &lerr);
  ILU_HANDLED(lerr);
 err5:
  ilu_free(t->tr_outBuff);
 err4:
  ilu_free(tp->t_inBuff);
 err3:
  ilu_free(tp);
 err2:
  ilu_free(t);
 err1:
  return NIL;
}
  
static void
  AddChannel (MooringData md,
	      ilu_Mooring channel,
	      ilu_Error *err)
{
  MooringList	ml;

  ml = ilu_MallocE(sizeof(*ml), err);
  if (ILU_ERRNOK(*err)) return;
  ilu_AcquireMutex(md->channels.lock);
  ml->next = md->channels.list;
  ml->mooring = channel;
  md->channels.list = ml;
  ilu_ReleaseMutex(md->channels.lock);
}

static void
  RemoveChannel (MooringData md,
		 ilu_Mooring channel,
		 ilu_Error *err)
{
  MooringList p, q;

  for (p = md->channels.list, q = NIL;  p != NIL;  q = p, p = p->next) {
    if (p->mooring == channel) {
      if (q == NIL)
	md->channels.list = p->next;
      else
	q->next = p->next;
      ilu_free(p);
      break;
    }
  }
  ILU_CLER(*err);
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static ilu_integer
  _w3mux_MooringDFd (ilu_Mooring self,
		     ilu_boolean add)
/*
 * Estimates how many FDs will be freed by closing this mooring
 * (if !add) or consumed by accepting a connection (if add).  In
 * either case, may err high but not low.
 */
{
  MooringData md = W3MUX_MOORINGDATA(self);
  ilu_integer dfd2 = 0;

  /* in general, if !add, the answer is 0 unless this is the only mooring
     for the underlying tcp mooring, in which case the answer is whatever
     it yields.
     If add, the answer is 0, if the underlying tcp mooring already exists;
     otherwise it is whatever that layer returns.
   */

  ilu_AcquireMutex(md->channels.lock);
  if (add && (!md->active)) {
    /* we'd have to create an underlying mooring, so ask it */
    dfd2 = mooring_dfd(md->mooring, add);
  } else if ((!add) && (md->channels.list->mooring == self)
	     && (md->channels.list->next == NIL)) {
    /* only channel on this mooring, so see what effect closing it would have */
    dfd2 = mooring_dfd(md->mooring, add);
  } else dfd2 = 0;	/* no real change, just adding or removing a channel */
  ilu_ReleaseMutex(md->channels.lock);
  return dfd2;
}

static void
  FreeTransportData (TransportData td)
{
  ilu_Error lerr;
  ilu_integer ldfd;

  transport_close(td->transport, &ldfd, &lerr);
  ilu_AcquireMutex(ilu_cmu);
  ilu_fdstaken -= 1;
  ilu_ReleaseMutex(ilu_cmu);
  ILU_HANDLED(lerr);
  /* Remove connection from endpoints table */
  {
    HashEnumerator he;
    ilu_TransportInfo tinfo;
    TransportData td2;
    
    ilu_AcquireMutex(w3mux_EndpointsLock);
    ILU_HASH_BEGINENUMERATION(w3mux_Endpoints, &he);
    while (ILU_HASH_NEXT(&he, (ilu_refany *) &tinfo, (ilu_refany *) &td2)) {
      if (td2 == td) {
	ILU_HASH_REMOVEFROMTABLE(w3mux_Endpoints, tinfo);
	ilu_free(tinfo);
      }
    }
    RemoveTransportFromList(&w3mux_Transports, td);
    ilu_ReleaseMutex(w3mux_EndpointsLock);
  }
  if (td->initiator != NIL) {
    ilu_DestroyPassport(td->initiator, &lerr);
    ILU_HANDLED(lerr);
    ilu_free(td->peerinfo);
  } else {
    ilu_free(td->tinfo);
  }
  if (td->input_cond != NIL) {
    lerr = ilu_DestroyCondition(td->input_cond);
    ILU_HANDLED(lerr);
  };
  ilu_ReleaseMutex(td->io_lock);
  ilu_DestroyMutex(td->io_lock, &lerr);
  ILU_HANDLED(lerr);
  ilu_ReleaseMutex(td->state_lock);
  ilu_DestroyMutex(td->state_lock, &lerr);
  ILU_HANDLED(lerr);
  ilu_free(td);
}

/* holding td->state_lock... */
static ilu_boolean
  HandleClosedConnection (TransportData td, ilu_Error *err)
{
  TransportList tl;
  TransportParms session_parms;
  ilu_Transport session;
  ilu_integer ldfd;
  TransportList this_link;
  ilu_Condition cond;

  ILU_NOTE(W3MUX_DEBUG,
	("(w3mux) HandleClosedConnection:  handling closed connection '%s'\n",
	 TRANSPORT_DATA_NAME(td)));
  if (td->sessions == NIL) {
    ilu_AcquireMutex(td->io_lock);
    FreeTransportData(td);
    ILU_CLER(*err);
    return ilu_FALSE;
  } else {
    this_link = td->sessions;
    td->sessions = td->sessions->next;
    session = this_link->transport;
    ilu_free(this_link);
    session_parms = W3MUX_TRANSPORTPARMS(session);
    ilu_AcquireMutex(session_parms->lock);
    cond = session_parms->cond;
    session_parms->eofRcvd = ilu_TRUE;
    ilu_ReleaseMutex(session_parms->lock);
    /* now give a session a chance to handle that EOF */
    ilu_ReleaseMutex(td->state_lock);
    if (_ilu_HasForkProc()) {
      ilu_CondNotify (cond, err);
      if (ILU_ERRNOK(*err)) {
	ILU_NOTE(W3MUX_DEBUG,
	      ("(w3mux) HandleClosedConnection:  %s(%u=>%u), error notifying condition %p\n",
	       SESSION_NAME(session), td->input_cond));
	ILU_HANDLED(*err);
      }
    } else if (session_parms->tih) {
      (*session_parms->tih->proc)(session_parms->tih->rock);
    } else if (session_parms->loophandle != 0) {
      ilu_ExitMainLoop(session_parms->loophandle);
    } else {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) HandleClosedConnection:  %s(%u=>%u), neither loophandle nor tih nor input_cond established!\n",
	     SESSION_NAME(session)));
    }
    ilu_AcquireMutex(td->state_lock);
    return ilu_TRUE;
  }
}

typedef enum GetHeaderResults { HeaderGood, HeaderEOF, HeaderErr, HeaderNoBytes } GetHeaderResults;

/* ...holding td->state_lock... */
static GetHeaderResults
  GetNBytes (TransportData td,
	     ilu_byte *where,
	     ilu_cardinal how_much,
	     ilu_string debugging_label,
	     ilu_Error *err)
{
  ilu_integer size, to_read = how_much;
  ilu_TransportReport report = { ilu_FALSE, ilu_FALSE };

  while (to_read > 0) {
    size = transport_read_upto_bytes(td->transport, where + (how_much - to_read), to_read, &report, err);
    if (ILU_ERRNOK(*err)) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) tHandler('%s'):  couldn't read %s, err '%s'\n",
	     TRANSPORT_DATA_NAME(td), debugging_label, ILU_ERR_NAME(*err)));
      return HeaderErr;
    } else if (report.tr_eof) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) tHandler('%s'):  other side closed connection\n",
	     TRANSPORT_DATA_NAME(td)));
      return HeaderEOF;
    } else if ((size == 0) || report.tr_eom) {
      /* false alarm */
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) tHandler('%s'):  false alarm, no actual bytes\n",
	     TRANSPORT_DATA_NAME(td)));
      return HeaderNoBytes;
    } else {
      to_read -= size;
    }
  }
  ILU_CLER(*err);
  return HeaderGood;
}

static void
  AddConnReq (TransportData td, MuxHeader *h)
{
  ConnReqList	p;
  p = ilu_must_malloc(sizeof(*p));
  p->header = *h;
  p->transport = td;
  p->next = w3mux_ConnReqs;
  w3mux_ConnReqs = p;
}

/* ...holding td->state_lock... */
static ilu_Transport
  HandleNewSession (TransportData td,
		    MuxHeader *header,
		    ilu_Error *err)
{
  ilu_Mooring		m;
  MooringParms		mp;
  MooringData		md;
  ilu_Transport		session = NIL;

  /* figure out the appropriate mooring, by looking at channel id in the
   * session_fragment_size header, and call its req handler.
   */
  m = FindMooring (header->contents.data_hdr.session_fragment_size);
  if (m == NIL) {
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) HandleNewSession:  can't find channel %u for session %u in w3mux_Moorings\n",
	   header->contents.data_hdr.session_fragment_size,
	   header->contents.data_hdr.session_id));
    ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_tinfo, NIL);
  } else {
    /* add the header to the list of headers */
    ilu_AcquireMutex(w3mux_ConnReqsLock);
    AddConnReq (td, header);
    ilu_ReleaseMutex(w3mux_ConnReqsLock);
    /* determine appropriate state for this transport data */
    if (header->contents.data_hdr.long_length == 1) {
    /* associated data with this one; block this transport thread until
       the connection is accepted and the data can be read */
      td->accepting = *header;
      td->state = Accepting_1;
    };
    /* call the tih of the mooring */
    mp = W3MUX_MOORINGPARMS(m);
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) HandleNewSession:  %s(%u=>%u)...\n",
	   TRANSPORT_DATA_NAME(td), header->contents.data_hdr.session_id,
	   header->contents.data_hdr.session_fragment_size));
    ilu_ReleaseMutex(td->state_lock);	/* so that tih can use it... */
    if (_ilu_HasForkProc()) {
      ilu_NotifyCondition (w3mux_ConnReqsCond);
    } else if (mp->tih) {
      (*mp->tih->proc)(mp->tih->rock);
    } else {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) HandleNewSession:  NIL tih for channel %s:%u\n",
	     TRANSPORT_DATA_NAME(td), header->contents.data_hdr.session_fragment_size));
      ILU_ERR_CONS1(internal, err, minor, ilu_im_broken, NIL);
    }
    ilu_AcquireMutex(td->state_lock);
  }
  if (ILU_ERROK(*err)) {
    session = FindSession(td, header->contents.data_hdr.session_id);
  };
  return session;
}

/* holding td->state_lock... */
static GetHeaderResults
  HandleControlMessage (TransportData td,
			MuxHeader *header,
			ilu_Error *err)
{
  GetHeaderResults result;

  if (header->contents.control_message.control_code == MUX_DefineEndpoint) {
    Endpoint_s endpoint;
    char *tmp_tinfo[2];
    char tmp_buf[200];
    char *tempname;
    static struct in_addr myaddr;
    ilu_TransportInfo tinfo;
    TransportData td2;

    endpoint.port = header->contents.control_message.session_fragment_size;
    /* Note IP addr is big-endian */
    result = GetNBytes (td, (ilu_bytes) &endpoint.ip_addr, 4, "endpoint data", err);
    if (result != HeaderGood)
      return result;
    /* Register this transport as being able to reach that endpoint */
    ilu_AcquireMutex(w3mux_EndpointsLock);
    myaddr.s_addr = endpoint.ip_addr;
    tempname = (ilu_string) inet_ntoa(myaddr);
    sprintf(tmp_buf, "tcp_%s_%u", tempname, endpoint.port);
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) HandleControlMessage (%s):  DefineEndpoint <%s>\n",
	   TRANSPORT_DATA_NAME(td), tmp_buf));
    tmp_tinfo[0] = tmp_buf;
    tmp_tinfo[1] = NIL;
    td2 = ILU_HASH_FINDINTABLE(w3mux_Endpoints, tmp_tinfo);
    if (td2 == NIL) {
      tinfo = _ilu_CopyTinfo(tmp_tinfo, err);
      if (ILU_ERRNOK(*err)) return HeaderErr;
      ILU_HASH_ADDTOTABLE (w3mux_Endpoints, tinfo, td);
    };
    ilu_ReleaseMutex(w3mux_EndpointsLock);
    if (ILU_ERRNOK(*err)) return HeaderErr;
    else return result;

  } else if (header->contents.control_message.control_code == MUX_DefineString) {
    ilu_cardinal string_size, buf_size;
    char *tmp_buf;

    result = GetNBytes (td, (ilu_bytes) &string_size, 4, "defined string", err);
    if (result != HeaderGood) return result;
    string_size = LITTLE_ENDIAN_TO_NATIVE(string_size);
    buf_size = ROUND_UP(string_size, 8);
    tmp_buf = ilu_MallocE(buf_size + 1, err);
    if (ILU_ERRNOK(*err)) return HeaderErr;
    result = GetNBytes (td, (ilu_bytes) tmp_buf, buf_size, "defined_string", err);
    if (result != HeaderGood) return result;
    tmp_buf[string_size] = '\0';
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) HandleControlMessage (%s):  DefineString <%u> = <%s> (ignored)\n",
	   TRANSPORT_DATA_NAME(td), header->contents.control_message.session_fragment_size,
	   tmp_buf));
    ilu_free(tmp_buf);    

  } else if (header->contents.control_message.control_code == MUX_DefineStack) {
    ilu_cardinal stack_size, buf_size;
    ilu_cardinal *tmp_buf;

    result = GetNBytes (td, (ilu_bytes) &stack_size, 4, "defined stack", err);
    if (result != HeaderGood) return result;
    stack_size = LITTLE_ENDIAN_TO_NATIVE(stack_size);
    buf_size = ROUND_UP(stack_size, 8);
    tmp_buf = (ilu_cardinal *) ilu_MallocE(buf_size, err);
    if (ILU_ERRNOK(*err)) return HeaderErr;
    result = GetNBytes (td, (ilu_bytes) tmp_buf, buf_size, "defined_string", err);
    if (result != HeaderGood) return result;
#ifdef ENABLE_DEBUGGING
    if ((ilu_DebugLevel & W3MUX_DEBUG) != 0) {
      unsigned int i, limit;
      ilu_DebugPrintf("(w3mux) HandleControlMessage (%s):  DefineStack <%u> = <",
		      TRANSPORT_DATA_NAME(td),
		      header->contents.control_message.session_fragment_size);
      for (i = 0, limit = stack_size / 4;  i < limit;  i++) {
	ilu_DebugPrintf("%s%u", (i != 0) ? " " : "", tmp_buf[i]);
      };
      ilu_DebugPrintf("> (ignored)\n");
    }
#endif /* ENABLE_DEBUGGING */
    ilu_free(tmp_buf);    

  } else if (header->contents.control_message.control_code == MUX_MuxControl) {
    ilu_Transport session;
    ilu_cardinal max_fragment_size;

    result = GetNBytes (td, (ilu_bytes) &max_fragment_size, 4, "mux_control", err);
    if (result != HeaderGood) return result;
    max_fragment_size = LITTLE_ENDIAN_TO_NATIVE(max_fragment_size);
    session = FindSession(td, header->contents.control_message.session_id);
    if (session != NIL) {
      W3MUX_TRANSPORTPARMS(session)->maxFragSize = max_fragment_size;
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) HandleControlMessage (%s):  MuxControl <%u> = %u\n",
	     TRANSPORT_DATA_NAME(td), W3MUX_TRANSPORTPARMS(session)->session_id,
	     header->contents.control_message.session_fragment_size));
    } else {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) HandleControlMessage (%s):  MuxControl for bad session %u!  Ignored.\n",
	     TRANSPORT_DATA_NAME(td), W3MUX_TRANSPORTPARMS(session)->session_id));
      return HeaderGood;
    }

  } else if (header->contents.control_message.control_code == MUX_SendCredit) {
    ilu_Transport session;
    ilu_cardinal credit_amount;

    result = GetNBytes (td, (ilu_bytes) &credit_amount, 4, "credit amount", err);
    if (result != HeaderGood) return result;
    credit_amount = LITTLE_ENDIAN_TO_NATIVE(credit_amount);
    session = FindSession(td, header->contents.control_message.session_id);
    if (session != NIL) {
      if (credit_amount == 0)
	W3MUX_TRANSPORTPARMS(session)->outCredit = 0;
      else
	W3MUX_TRANSPORTPARMS(session)->outCredit += credit_amount;
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) HandleControlMessage (%s):  SendCredit <%u> += %u => %u\n",
	     TRANSPORT_DATA_NAME(td), W3MUX_TRANSPORTPARMS(session)->session_id,
	     credit_amount, W3MUX_TRANSPORTPARMS(session)->outCredit));
    } else {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) HandleControlMessage (%s):  SendCredit for bad session %u!  Ignored.\n",
	     TRANSPORT_DATA_NAME(td), header->contents.control_message.session_id));
      return HeaderGood;
    }
    
  } else {
    return ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_protocol_sync_lost, HeaderErr);
  }
}

/* holding td->state_lock and p->lock... */
static void
  StuffBytes (ilu_Transport self, TransportData td, ilu_cardinal how_many, ilu_cardinal headersize, ilu_Error *err)
{
  TransportParms p = W3MUX_TRANSPORTPARMS(self);
  GetHeaderResults result;

  ILU_NOTE(W3MUX_DEBUG,
	("(w3mux) StuffBytes:  %s(%u=>%u), %u bytes\n",
	 SESSION_NAME(self), how_many));

  if ((p->inSize - (p->inLength - p->t_inNext)) < how_many) {
    ILU_ERR_CONS1(no_resources, err, minor, ilu_nrm_ENOBUFS, 0);
    return;
  };
  /* may need to shift bytes currently in the buffer to the front to make room for new bytes */
  if ((p->inSize - p->inLength) < how_many) {
    memmove(p->t_inBuff, p->t_inBuff + p->t_inNext, (p->inLength - p->t_inNext));
    p->t_inLimit -= p->t_inNext;
    p->inLength -= p->t_inNext;
    p->t_inNext = 0;
  };
  result = GetNBytes(td, p->t_inBuff + p->inLength, how_many, "stuffing bytes", err);
  switch (result) {
  case HeaderErr:
    return;
  case HeaderEOF:
    ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_eof, 0);
    return;
  case HeaderGood:
    p->inLength += how_many;
    {	/* now read any padding bytes */
      ilu_cardinal to_read = ROUND_UP(how_many, headersize) - how_many;
      ilu_byte pad[8];
      result = GetNBytes(td, pad, to_read, "padding bytes", err);
      switch (result) {
      case HeaderErr:
	return;
      case HeaderEOF:
	ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_eof, 0);
	return;
      case HeaderGood:
	p->accumCredit += to_read;
	return;
      case HeaderNoBytes:
	ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_conn_lost, 0);
	return;
      }
    }
  case HeaderNoBytes:
    ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_conn_lost, 0);
    return;
  }
}

static ilu_boolean SkipFragment (TransportData td,
				 ilu_cardinal fsize,
				 ilu_cardinal round_to)
{
  ilu_cardinal util;
  ilu_byte tmp_buf[1000];
  ilu_Error lerr;
  ilu_cardinal size = fsize;
  GetHeaderResults result;

  if (size > 0) {
    util = ROUND_UP(size, round_to);
    while (util > 0) {
      size = MIN(sizeof(tmp_buf), util);
      result = GetNBytes(td, tmp_buf, size, "skipped bytes", &lerr);
      if (result == HeaderErr || result == HeaderEOF)
	return ilu_FALSE;
      if (result == HeaderGood)
	util -= size;
    }
  }
  return ilu_TRUE;
}

static void
  tHandler (ilu_refany rock)
{
  TransportData td = (TransportData) rock;
  int dfd = 0;
  ilu_Error lerr;
  MuxHeader header;
  ilu_cardinal long_length;
  ilu_Transport session = NIL;
  TransportParms session_parms;
  ilu_cardinal size = 0, round_to = 0;
  GetHeaderResults result;
  ilu_cardinal util;
  ilu_byte tmp_buf[1024];
  ilu_Transport Sessions[10];
  typedef struct Session_s {
    ilu_Transport session;
    struct Session_s *next;
  } Session;
  Session *firstSession = NIL;
  Session *lastSession = NIL;
  ilu_cardinal nSessions = 0;

  ILU_NOTE(W3MUX_DEBUG,
	("(w3mux) tHandler('%s', %s):  input detected\n",
	 TRANSPORT_DATA_NAME(td),
	 ((td->state == Accepting_1) ? "Accepting_1" :
	  ((td->state == Accepting_2) ? "Accepting_2" :
	   ((td->state == Normal) ? "Normal" :
	    (sprintf((char *) tmp_buf, "Bad state: %d", (int) td->state), (char *) tmp_buf))))));

  ilu_AcquireMutex(td->state_lock);

  if (!_ilu_HasForkProc()) {
    /* first, disable other requests on this transport */
    transport_set_input_handler(td->transport, NIL, &lerr);
    td->active = ilu_FALSE;
  }

  /* We can be in three states:  accepting-1, in which case there is data waiting
     in the lower buffer, and td->accepting.contents.word is non-zero, and we
     should do nothing;  accepting-2, in which case we have opened a new connection,
     td->accepting contains the header to parse, and we should begin execution at
     the label "accepting";  normal, in which case there is input to read from the
     lower transport, so get to it.
     */

 top:

  if (td->state == Accepting_1) {
    td->state = Accepting_2;
    goto reestablish;
  } else if (td->state == Accepting_2) {
    header = td->accepting;
    td->state = Normal;
    goto accepting;
  } else if (td->state != Normal) {
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) tHandler:  bad state %d in (%s)->state\n",
	   (int) td->state, TRANSPORT_DATA_NAME(td)));
    goto shutdown;
  };
    
  if (!td->closing) {
    /* read the header of the next chunk and dispatch it appropriately.
     */
    result = GetNBytes(td, (ilu_bytes) &header.contents.word, 4, "header", &lerr);
    if (result != HeaderGood && result != HeaderEOF)
      goto reestablish;
  }

  /* did we get EOF at some point? */
  if (td->closing || result == HeaderEOF) {
    td->closing = ilu_TRUE;
    if (!HandleClosedConnection (td, &lerr)) {
      /* underlying connection (and td) no longer exists!  Exit gracefully. */
      return;	/* without re-establishing handler */
    } else goto reestablish;
  };

  /* Alright, now examine the header, and see what kind of message we have here.
   * There are three possibilities:
   *   1)  a control message for no particular session, and
   *   2)  a setup message for a new session (which may also include data for
   *       the new session,
   *   3)  a data message for some particular session.
   */
  header.contents.word = LITTLE_ENDIAN_TO_NATIVE(header.contents.word);

  if (header.contents.data_hdr.control == 1) {
    HandleControlMessage (td, &header, &lerr);
  } else {
    if ((header.contents.data_hdr.long_length == 1) && (header.contents.data_hdr.syn == 1)) {
      /* request for new session */
      session = HandleNewSession (td, &header, &lerr);
      if (ILU_ERRNOK(lerr))
	goto shutdown;
      else if (td->state == Normal)
	goto reestablish;
      else {
	ilu_ReleaseMutex(td->state_lock);
	return;
      }
    } else {
    accepting:
      /* message fragment for existing session -- figure out which one */
      if ((session = FindSession (td, header.contents.data_hdr.session_id)) == NIL) {
	ILU_NOTE(W3MUX_DEBUG,
	      ("(w3mux) tHandler(%s):  non-existent session id %u!  Skipping fragment.\n",
	       TRANSPORT_DATA_NAME(td), header.contents.data_hdr.session_id));
      }
    };
    size = header.contents.data_hdr.session_fragment_size;
    if (header.contents.data_hdr.long_length == 1) {
      if (GetNBytes(td, (ilu_bytes) &size, 4, "long size", &lerr) != HeaderGood)
	goto shutdown;	/* should interpret error */
      size = LITTLE_ENDIAN_TO_NATIVE(size);
      round_to = 8;
    } else
      round_to = 4;
    if (session == NIL)	{/* odd packet */
      if (!SkipFragment(td, size, round_to))
	goto shutdown;
    } else {
      session_parms = W3MUX_TRANSPORTPARMS(session);
      ilu_AcquireMutex(session_parms->lock);
      if (size > 0) {
	/* stuff the data into the session's input buffer */
	if ((HARDSIZE(session_parms) - session_parms->inLength) < 4) {
	  ILU_NOTE(W3MUX_DEBUG,
		   ("(w3mux) tHandler:  %s(%u=>%u), no room in internal buffer for header!\n",
		    SESSION_NAME(session)));
	  if (!SkipFragment (td, size, round_to))
	    goto shutdown;
	} else {
	  util = (size & 0x7FFFFFFF) | ((header.contents.data_hdr.push == 1) ? 0x80000000 : 0);
	  memcpy((void *) (session_parms->t_inBuff
			   + session_parms->inLength),
		 (void *) &util, 4);
	  session_parms->inLength += 4;
	  session_parms->accumCredit += (round_to - 4); /* we may have reduced the header size by 4 bytes
							   which should be credited... */
	  StuffBytes(session, td, size, round_to, &lerr);
	  if (ILU_ERRNOK(lerr)) {
	    ILU_NOTE(W3MUX_DEBUG,
		     ("(w3mux) tHandler:  %s(%u=>%u), error <%s> trying to clear fragment of %u bytes, closing conn\n",
		      SESSION_NAME(session), ILU_ERR_NAME(lerr), size));
	    goto shutdown;
	  };
	} /* if room in buffer */
      } /* if (size > 0) */
      if (header.contents.data_hdr.fin == 1)
	session_parms->eofRcvd = ilu_TRUE;
      ilu_ReleaseMutex(session_parms->lock);
    } /* session != NIL */
  } /* if data message */

  /* We need to keep processing stuff until the underlying buffer is empty.
   * On the other hand, we need to keep track of what sessions we've affected,
   * so that we can notify them of changes.  So we push the sessions onto a list
   * if there's still work to be done, and unwind the list later.
   */
  if (session != NIL) {
    if (nSessions > (sizeof(Sessions)/sizeof(ilu_Transport))) {
      Session *n = (Session *) ilu_must_malloc(sizeof(Session));
      n->session = session;
      n->next = NIL;
      if (lastSession == NIL)
	{ lastSession = firstSession = n; }
      else
	{ lastSession->next = n ; lastSession = n; };
    } else {
      Sessions[nSessions++] = session;
    }
    /* if there's still more to do, then go back and do some more work */
    if (transport_has_exposed_input(td->transport))
      goto top;
  };

  /* OK, we've processed all the data in the underlying buffer, and now we have to
   * animate the various sessions.  For each session, we have three cases:
   * we could be waiting for input on this session by running the main loop,
   * or we could be waiting for our input handler function to be called, or we could be waiting
   * on a condition.  Do the right thing to wake up that session.
   */
  util = 0;
  lastSession = firstSession;
  while (util < nSessions || lastSession != NIL) {
    if (util < nSessions)
      session = Sessions[util++];
    else {
      Session *n;
      n = lastSession;
      session = n->session;
      lastSession = n->next;
      ilu_free(n);
    }
    ILU_NOTE(W3MUX_DEBUG,
	     ("(w3mux) tHandler:  dispatching session %s(%u=>%u)...\n",
	      SESSION_NAME(session)));
    session_parms = W3MUX_TRANSPORTPARMS(session);
    ilu_ReleaseMutex(td->state_lock);
    if (_ilu_HasForkProc()) {
      ilu_CondNotify (session_parms->cond, &lerr);
      if (ILU_ERRNOK(lerr)) {
	ILU_NOTE(W3MUX_DEBUG,
	      ("(w3mux) tHandler:  %s(%u=>%u), error notifying condition %p\n",
	       SESSION_NAME(session), session_parms->cond));
	ILU_HANDLED(lerr);
      }
    } else if (session_parms->loophandle != NIL) {
      ilu_ExitMainLoop(session_parms->loophandle);
    } else if (session_parms->tih) {
      (*session_parms->tih->proc)(session_parms->tih->rock);
    }
    ilu_AcquireMutex(td->state_lock);
  }

 reestablish:	/* reset the input handler, release the lock, and continue */
  ILU_HANDLED(lerr);
  if (!_ilu_HasForkProc()) {
    if (!_ilu_SetTransportInputHandler(td->transport, &td->td_loClo, &lerr)) {
      if (ILU_ERROK(lerr)) goto top;
    } else
      td->active = ilu_TRUE;
  }
  ILU_HANDLED(lerr);
  ilu_ReleaseMutex(td->state_lock);
  return; 

 shutdown:	/* close the underlying connection, notify all sessions */
  ILU_HANDLED(lerr);
  td->closing = ilu_TRUE;
  goto reestablish;	/* repeat until done */
}

void handleTransport (void *rock)
{
  TransportData td = (TransportData) rock;
  ilu_Error lerr;
  ilu_boolean ans;
  int disabled;

  td->active = ilu_TRUE;
  while (td->active && !td->closing) {
    /* wait for input */
    ans = transport_wait_for_input(td->transport, &disabled, NIL, &lerr);
    ILU_ERR_SWITCH(lerr) {
      ILU_SUCCESS_CASE {
	if (disabled)
	  continue;	/* XXX what's the right thing to do here? */
	else {
	  tHandler ((void *) td);
	  ilu_AcquireMutex(td->state_lock);
	  while (td->state == Accepting_1) {
	    ilu_CMWait1(td->input_cond, td->state_lock, &lerr);
	    if (ILU_ERRNOK(lerr)) {
	      ILU_NOTE(W3MUX_DEBUG,
		    ("(w3mux) handleTransport:  err <%s> waiting for accept on %s.  Exiting thread.\n",
		     ILU_ERR_NAME(lerr), TRANSPORT_DATA_NAME(td)));
	      ILU_HANDLED(lerr);
	      return;
	    }
	  }
	  ilu_ReleaseMutex(td->state_lock);
	}
      }
      ILU_ERR_CASE(interrupted, e) {
	/* XXX what to do here */
      }
      ILU_ERR_ELSE {
	ILU_NOTE(W3MUX_DEBUG,
	      ("(w3mux) handleTransport:  err <%s> waiting for input on %s.  Exiting thread.\n",
	       ILU_ERR_NAME(lerr), TRANSPORT_DATA_NAME(td)));
	ILU_HANDLED(lerr);
	return;
      }
    } ILU_ERR_ENDSWITCH;
  }
}

/* ...while holding w3mux_EndpointsLock... */
static TransportData
  NewTransportData (ilu_Transport lower,
		    ilu_Passport pp,
		    ilu_TransportInfo lower_tinfo,
		    ilu_string lower_peerinfo,
		    ilu_Mooring m,
		    ilu_Error *err)
{
  ilu_TransportInfo tinfo = NIL, key = NIL;
  ilu_Condition cond = NIL;
  TransportData ld;
  ilu_string peerinfo;
  TransportDataList tdl;
  MooringDataList mdl;
  struct {
    MuxHeader hdr;
    ilu_cardinal ip_addr;
  } msg;
  ilu_Error lerr;
  
  if (lower_tinfo != NIL)
    tinfo = _ilu_CopyTinfo(lower_tinfo, err);
  else
    peerinfo = ilu_StrdupE(lower_peerinfo, err);
  if (ILU_ERRNOK(*err)) goto free2;
  if (ilu_CanCondition())
    cond = ilu_CreateCondition("w3muxed-transport", (lower_tinfo != NIL) ? lower_tinfo[0] : lower_peerinfo, err);
  if (ILU_ERRNOK(*err)) goto free3;
  ld = ilu_MallocE(sizeof(*ld), err);
  if (ILU_ERRNOK(*err)) goto free4;
  if (lower_tinfo != NIL)
    key = _ilu_CopyTinfo(lower_tinfo, err);
  if (ILU_ERRNOK(*err)) goto free5;
  ld->io_lock = ilu_CreateMutex("w3mux transport I/O", (lower_tinfo != NIL) ? lower_tinfo[0] : lower_peerinfo);
  if (ld->io_lock == NIL) goto free6;
  ld->state_lock = ilu_CreateMutex("w3mux transport State", (lower_tinfo != NIL) ? lower_tinfo[0] : lower_peerinfo);
  if (ld->state_lock == NIL) goto free7;
  ld->transport = lower;
  ld->tinfo = tinfo;
  ld->active = ilu_FALSE;
  ld->initiator = pp;
  ld->peerinfo = peerinfo;
  ld->closing = ilu_FALSE;
  ld->sessions = NIL;
  ld->input_cond = cond;
  ld->state = Normal;
  ld->accepting.contents.word = 0;
  ld->td_loClo.proc = tHandler;
  ld->td_loClo.rock = ld;
  if (key != NIL)
    ILU_HASH_ADDTOTABLE(w3mux_Endpoints, key, ld);
  tdl = ilu_must_malloc(sizeof(*tdl));
  tdl->td = ld;
  tdl->next = w3mux_Transports;
  w3mux_Transports = tdl;
  if (_ilu_HasForkProc()) {
    (void) _ilu_ForkNewThread (handleTransport, (void *) ld, &lerr);
  } else {
    transport_set_input_handler(ld->transport, &ld->td_loClo, &lerr);
  }
  if (ILU_ERRNOK(lerr)) {
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) NewTransportData:  Can't establish transport handler for tcp connection from %s, err is %s.\n",
	   peerinfo, ILU_ERR_NAME(lerr)));
    ILU_HANDLED(lerr);
    goto free8;
  } else
    ld->active = ilu_TRUE;

  ILU_NOTE(W3MUX_DEBUG,
	   ("(w3mux) NewTransportData:  new transport %p to <%s> created\n",
	    ld->transport, TRANSPORT_DATA_NAME(ld)));

  ilu_AcquireMutex(w3mux_ChannelsLock);
  for (mdl = w3mux_Moorings;  mdl != NIL;  mdl = mdl->next) {
    if (mdl->md->port != 0 && ((m == NIL) || (mdl->md->mooring != m))) {
      msg.hdr.contents.word = 0;
      msg.hdr.contents.control_message.long_length = 1;
      msg.hdr.contents.control_message.control = 1;
      msg.hdr.contents.control_message.control_code = MUX_DefineEndpoint;
      msg.hdr.contents.control_message.session_fragment_size = mdl->md->port;
      msg.hdr.contents.word = NATIVE_TO_LITTLE_ENDIAN(msg.hdr.contents.word);
      msg.ip_addr = mdl->md->ip_addr;		/* always big-endian */
      ILU_NOTE(W3MUX_DEBUG,
	       ("(w3mux) NewTransportData:  sending endpoint data for mooring endpoint %s:%u\n",
		inet_ntoa(*((struct in_addr *)(&mdl->md->ip_addr))), mdl->md->port));
      (*lower->tr_class->tc_write_bytes)(lower, (ilu_bytes) &msg, sizeof(msg), ilu_TRUE, err);
      if (ILU_ERRNOK(*err)) break;
    }
  }
  ILU_HANDLED(*err);
  ilu_ReleaseMutex(w3mux_ChannelsLock);
  ilu_AcquireMutex(ld->state_lock);
  ilu_AcquireMutex(ld->io_lock);
  ILU_CLER(*err);
  return ld;

 free8:
  ilu_DestroyMutex(ld->state_lock, &lerr);
  ILU_HANDLED(lerr);
 free7:
  ilu_DestroyMutex(ld->io_lock, &lerr);
  ILU_HANDLED(lerr);
 free6:
  ilu_free(key);
 free5:
  ilu_free(ld);
 free4:
  if (cond != NIL)
    ilu_DestroyCondition(cond);
 free3:
  if (lower_tinfo == NIL)
    ilu_free(peerinfo);
  else
    ilu_free(tinfo);
 free2:
  return NIL;
}

static void
  mHandler (ilu_refany rock)
{
  MooringData ec = (MooringData) rock;
  int dfd = 0;
  ilu_Error lerr;
  ilu_Transport t = NIL;
  ilu_Passport pp = NIL;
  ilu_string peerinfo = NIL;
  TransportData td = NIL;
  struct {
    MuxHeader hdr;
    ilu_cardinal ip_addr;
  } msg;
  MooringDataList mdl;

  ILU_NOTE(W3MUX_DEBUG,
	("(w3mux) mHandler:  connection request on socket '%s'\n",
	 ec->tinfo_out[0]));

  /* Check to see whether we have the fd budget to accept a new connection */
  ilu_AcquireMutex(ilu_cmu);
  dfd = mooring_dfd(ec->mooring, ilu_TRUE);
  if (ilu_fdbudget < ilu_fdstaken + dfd) {
    lerr = _ilu_ReduceFdsTo(ilu_fdbudget - dfd);
    if (ILU_ERRNOK(lerr)) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) mHandler:  err <%s> while attempting to reduce fds from %s to %s\n",
	     ILU_ERR_NAME(lerr), ilu_fdstaken, ilu_fdstaken - dfd));
      ILU_HANDLED(lerr);
      goto done1;
    };
    if ((ilu_fdbudget < ilu_fdstaken + dfd) && (dfd > 0)) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("(w3mux) mHandler:  FD budget exhausted.\n"));
      /* XXX - Shouldn't we do something here, like accepting the connection,
	 then closing it immediately, to clear the tcp mooring? */
      goto done1;
    };
  } else {
    ilu_ReleaseMutex(ilu_cmu);
  };

  /* OK, we have the fds, now create a passport to hold the incoming caller ident */
  if ((pp = ilu_CreatePassport(NIL, &lerr)) == NIL) {
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) mHandler:  err <%s> while attempting to create passport.\n",
	   ILU_ERR_NAME(lerr)));
    ILU_HANDLED(lerr);
    goto done0;
  };

  /* now accept the connection and get back an ilu_Transport */
  t = mooring_accept_connection(ec->mooring, &peerinfo, &dfd, pp, &lerr);
  ilu_AcquireMutex(ilu_cmu);
  ilu_DeltaFD(dfd);
  ilu_ReleaseMutex(ilu_cmu);
  if (ILU_ERRNOK(lerr)) {
    ILU_NOTE(W3MUX_DEBUG,
	  ("(w3mux) mHandler:  err <%s> while attemping to accept connection on %s.\n",
	   ILU_ERR_NAME(lerr), ec->tinfo_out[0]));
    ILU_HANDLED(lerr);
    goto done2;
  };

  td = NewTransportData (t, pp, NIL, peerinfo, ec->mooring, &lerr);
  if (ILU_ERRNOK(lerr)) goto done3;

  ILU_NOTE(W3MUX_DEBUG,
	("(w3mux) mHandler:  accepted new conn %p from '%s' on socket '%s'\n",
	 t, peerinfo, ec->tinfo_out[0]));

  /* successful completion, return */
  ilu_ReleaseMutex(td->io_lock);
  ilu_ReleaseMutex(td->state_lock);
  goto done0;

  /* handle various return cleanup cases */
 done4:
  FreeTransportData(td);
 done3:
  transport_close(t,&dfd,&lerr);
  ilu_AcquireMutex(ilu_cmu);
  ilu_DeltaFD(dfd);
  ilu_ReleaseMutex(ilu_cmu);
  if (ILU_ERRNOK(lerr)) ILU_HANDLED(lerr);
  ilu_free(peerinfo);
 done2:
  ilu_DestroyPassport(pp, &lerr);
  ILU_HANDLED(lerr);
  goto done0;
 done1:
  ilu_ReleaseMutex(ilu_cmu);
 done0:
  return;
}

void handleMooring (void *rock)
{
  MooringData md = (MooringData) rock;
  ilu_Error lerr;
  ilu_boolean ans;
  int disabled;

  md->active = ilu_TRUE;
  while (md->active) {
    /* wait for input */
    ans = (*md->mooring->mo_wait_for_req) (md->mooring, &disabled, &lerr);
    ILU_ERR_SWITCH(lerr) {
      ILU_SUCCESS_CASE {
	if (disabled)
	  continue;	/* XXX what's the right thing to do here? */
	else
	  mHandler ((void *) md);
      }
      ILU_ERR_CASE(interrupted, e) {
	/* XXX what to do here */
      }
      ILU_ERR_ELSE {
	ILU_NOTE(W3MUX_DEBUG,
	      ("(w3mux) handleMooring:  err <%s> waiting for a connection request on %s.  Exiting thread.\n",
	       ILU_ERR_NAME(lerr), md->tinfo_out[0]));
	return;
      }
    } ILU_ERR_ENDSWITCH;
  }
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static ilu_boolean
  _w3mux_MooringSetReqHandler(ilu_Mooring self,
			      ilu_TIH tih,
			      ILU_ERRS((no_memory, imp_limit, no_resources,
					broken_locks, internal)) * err)
{
  MooringParms mp = W3MUX_MOORINGPARMS(self);

  /* The actual work here is fairly small.  We just tuck away the tih and rock
     so that they can be called by mHandler when necessary */
  mp->tih = tih;
  return ILU_CLER(*err);
}

/*Main Invariant holds; L2 >= {ymu}*/
static ilu_boolean
  _w3mux_MooringWaitForReq (ilu_Mooring self,
			    int * disabled_p,
			    ILU_ERRS((broken_locks)) * err)
/*
 * In a multi-threaded runtime, a thread calls this to wait for
 * the next connection request.  Returns ILU_ERROK(*err) after
 * either: (1) there is a decent reason to suspect that a
 * connection request is present, or an error or EOM or EOF can be
 * detected without blocking, (2) the thread system's generic
 * interrupt-that-thread function has been invoked on this thread
 * (in which case interrupted is raised), or (3) waiting is
 * disabled for (self)'s wait-cohort (see mo_disableWait).  Sets
 * (*disabled), to a non-zero value when only (3) applies; this is
 * only a hint, and can be wrong (with the only adverse
 * consequence being a small performance hit) --- as long as false
 * positives are not frequent.  Caller guarantees that (self) is
 * not closed at start of call (of course); callee guarantees that
 * (self) is not closed at time of return.
 */
{
  MooringParms mp = W3MUX_MOORINGPARMS(self);
  MooringData md = W3MUX_MOORINGDATA(self);
  MuxHeader *h;
  ilu_Transport t = NIL;
  ConnReqList p, q;

  ilu_AcquireMutex(w3mux_ConnReqsLock);
  while (!mp->disabled) {
    if (w3mux_ConnReqs != NIL) {
      for (p = w3mux_ConnReqs, q = NIL;  p != NIL;  q = p, p = p->next) {
	h = &p->header;
	/* check contents of header for appropriate codes */
	if ((h->contents.data_hdr.syn == 1) ||
	    (h->contents.data_hdr.session_fragment_size == mp->channel_id)) {
	  ilu_ReleaseMutex(w3mux_ConnReqsLock);
	  *disabled_p = ilu_FALSE;
	  return ILU_CLER(*err);
	}
      }
    }
    ilu_CMWait1(w3mux_ConnReqsCond, w3mux_ConnReqsLock, err);
    if (ILU_ERRNOK(*err)) return ilu_FALSE;
  }
  ilu_ReleaseMutex(w3mux_ConnReqsLock);
  *disabled_p = ilu_TRUE;
  return ILU_CLER(*err);
}

/* L1 >= {cmu}; L1.sup < trmu; L2 >= {xmu} */
static ilu_boolean
  _w3mux_MooringDisableWait (ilu_Mooring self,
			     ILU_ERRS((broken_locks,
				       bad_param,
				       internal)) *err)
/*
 * Applicable only in multi-threaded runtime.  Caller ensures that
 * for each M, at every moment, the number of mo_disableWait(M,..)
 * calls that have been made so far is always at least as great as
 * the number of mo_enableWait(M,..) calls, regardless of thread
 * scheduling.  Moorings/Transports are partitioned into disjoint
 * sets called `wait-cohorts'.  Each mooring and each transport is
 * in charge of identifying which set it belongs to; these sets
 * are not explicitly reified --- they only appear in the
 * following definition.  While there have been more
 * {mo,tc}_disableWait calls on members of (self)'s wait-cohort
 * than {mo,tc}_enableWait calls, we say waiting is disabled for
 * (self)'s wait-cohort.  This condition affecte the behavior of
 * mo_wait_for_req, above.
 */
{
  MooringParms mp = W3MUX_MOORINGPARMS(self);
  ilu_Condition cond;
  ilu_boolean disabled;

  ilu_AcquireMutex(mp->lock);
  mp->disabled += 1;
  disabled = (mp->disabled > 0);
  ilu_ReleaseMutex(mp->lock);
  if (disabled)
    ilu_NotifyCondition(w3mux_ConnReqsCond);
  return ILU_CLER(*err);
}

/* L1 >= {cmu}; L1.sup < trmu; L2 >= {xmu} */
static ilu_boolean
  _w3mux_MooringEnableWait (ilu_Mooring self,
			    ILU_ERRS((broken_locks,
				      bad_param,
				      internal)) *err)
/*
 * [see comment on _w3mux_MooringDisableWait]
 */
{
  MooringParms mp = W3MUX_MOORINGPARMS(self);
  ilu_Condition cond;

  ilu_AcquireMutex(mp->lock);
  mp->disabled -= 1;
  ilu_ReleaseMutex(mp->lock);
  return ILU_CLER(*err);
}

/* Main Invariant holds; L2 >= {xmu, ymu} */
static ilu_Transport
  _w3mux_MooringAcceptClient(ilu_Mooring self,
			     ilu_string * tinfo_out,
			     ilu_integer *dfd,
			     ilu_Passport pp,
			     ILU_ERRS((IoErrs)) * err)
/*
 * Create a new transport in response to the waiting connection
 * request; will return NIL if there wasn't really a connection
 * request waiting.  Also returns (with ownership) through
 * peerinfo (if not NIL) a string (not necessarily a real "tinfo")
 * identifying the peer.  Stores at *dfd the number of FDs
 * consumed (regardless of whether error is raised).
 *
 * It works by examining the accept.header field in the associated
 * MooringData.  Typically, this will name a particular session ID
 * which is to identify messages in the new ilu_Transport.  We create
 * a new mux transport, fill in the session ID, and return.
 */
{
  MooringParms mp = W3MUX_MOORINGPARMS(self);
  MuxHeader h;
  TransportData td;
  ilu_Transport t = NIL;
  ConnReqList p, q;

  *dfd = 0;

  ilu_AcquireMutex(w3mux_ConnReqsLock);

  for (p = w3mux_ConnReqs, q = NIL;  p != NIL;  q = p, p = p->next) {
    /* check contents of header for appropriate codes */
    if ((p->header.contents.data_hdr.syn == 1) &&
	(p->header.contents.data_hdr.session_fragment_size == mp->channel_id)) {
      /* this one's for us */
      break;
    };
  }
  if (p == NIL) {
    ILU_CLER(*err);
    goto ret;
  } else {
    h = p->header;
    td = p->transport;
    if (q == NIL)
      w3mux_ConnReqs = p->next;
    else
      q->next = p->next;
    ilu_free(p);
  }

  /* do tinfo_out at this stage to avoid free hassles if it fails */
  if (tinfo_out != NIL) {
    static char *format_string = "w3mux ch %u s %u over %s";
    char *buf = ilu_MallocE(strlen(format_string) + 5 + strlen(TRANSPORT_DATA_NAME(td)) + 2, err);
    if (ILU_ERRNOK(*err)) goto ret;
    sprintf(buf, format_string, mp->channel_id,
	    h.contents.control_message.session_id, TRANSPORT_DATA_NAME(td));
    *tinfo_out = buf;
  };

  ilu_AcquireMutex(td->state_lock);
  /* check TransportData to see if session ID is taken */
  if (FindSession(td, h.contents.control_message.session_id) != ILU_NIL) {
    ILU_NOTE(W3MUX_DEBUG,
	  ("_w3mux_MooringAcceptClient:  session %u over <%s> already in use!\n",
	   h.contents.control_message.session_id, TRANSPORT_DATA_NAME(td)));
    ILU_ERR_CONS1(comm_failure, err, minor, ilu_cfm_bad_address, 0);
    goto err1;
  };

  /* No; create new transport data structure and fill it in */
  t = NewTrans(td, h.contents.data_hdr.session_id, mp->channel_id, ilu_FALSE, err);
  if (ILU_ERRNOK(*err)) goto err1;
  AddSession(td, t, err);
  if (ILU_ERRNOK(*err)) { t = NIL; goto err1; };	/* XXX leaks transport here? */
  td->state = Accepting_2;
 err1:
  ilu_ReleaseMutex(td->state_lock);
 ret:
  ilu_ReleaseMutex(w3mux_ConnReqsLock);
  if (t != NIL) {
    if (_ilu_HasForkProc())
      ilu_NotifyCondition(td->input_cond);
    else
      tHandler((void *) td);
  } else if (tinfo_out != NIL)
    ilu_free (*tinfo_out);
  return t;
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
static          ilu_boolean
_w3mux_CloseMooring(ilu_Mooring self,
		    ilu_integer * dfd,
		    ILU_ERRS((bad_locks, broken_locks,
			      internal)) * err)
/*
 * Even frees self.  Stores at *dfd the number of FDs freed (even
 * if error raised).
 */
{
  /* we have to look at the MooringData.  If we are the only channel,
   * we will also close the underlying mooring.
   */
  MooringData md = W3MUX_MOORINGDATA(self);
  MooringParms mp = W3MUX_MOORINGPARMS(self);
  ilu_integer dfd2 = 0;

  *dfd = 0;

  ilu_AcquireMutex(w3mux_ChannelsLock);
  ilu_AcquireMutex(md->channels.lock);
  if ((md->channels.list->mooring == self) &&
      (md->channels.list->next == NIL)) {
    /* only channel in this EC, so close the underlying mooring, too */
    if (!(mooring_close(md->mooring, &dfd2, err))) {
      *dfd = dfd2;
      ilu_ReleaseMutex(md->channels.lock);
      ilu_ReleaseMutex(w3mux_ChannelsLock);
      ILU_NOTE(W3MUX_DEBUG,
	    ("_w3mux_CloseMooring:  Can't release underlying mooring %s.  Error %s.\n",
	     md->tinfo_in[0], ILU_ERR_NAME(*err)));
      return ilu_FALSE;
    }
    md->mooring = NIL;
    md->active = ilu_FALSE;
  }
  *dfd = dfd2;

  /* Take mooring off list in MooringData */
  RemoveChannel (md, self, err);
  if (ILU_ERRNOK(*err)) {
    ilu_ReleaseMutex(md->channels.lock);
    ilu_ReleaseMutex(w3mux_ChannelsLock);
    ILU_NOTE(W3MUX_DEBUG,
	  ("_w3mux_CloseMooring:  Can't remove channel %u.  Error %s.\n",
	   mp->channel_id, ILU_ERR_NAME(*err)));
    return ilu_FALSE;
  };
  ILU_HASH_REMOVEFROMTABLE(w3mux_Channels, (ilu_refany) mp->channel_id);
  
  if (mp->tih) {
    ilu_Closure     c = _ilu_ClosureFromTIH(mp->tih);
    ILU_ERRS((internal, bad_param, bad_locks, broken_locks)) lerr;
    if (!ilu_Check(ilu_DoSoon(c, &lerr), err)) {
      ILU_HANDLED(lerr);
      return FALSE;
    }
    mp->tih = NIL;
  }
  
  /* Free up the mooring parms */
  ilu_free(mp);

  /* if the MooringData is no longer needed, free it */
  if (md->channels.list == NIL && md->mooring == NIL) {
    ilu_Error lerr;
    MooringDataList p, q;
    ilu_DestroyMutex(md->channels.lock, &lerr);
    ILU_HANDLED(lerr);
    for (p = w3mux_Moorings, q = NIL;  p != NIL;  q = p, p = p->next) {
      if (p->md == md) {
	if (q == NIL)
	  w3mux_Moorings = p->next;
	else
	  q->next = p->next;
	ilu_free(p);
	break;
      }
    }
    ilu_free(md->tinfo_out);
    ilu_free(md->tinfo_in);
    ilu_free(md);
  }

  /* finally, free self */
  ilu_free(self);
  ilu_ReleaseMutex(w3mux_ChannelsLock);
  return ILU_CLER(*err);
}

/*L1, L2 unconstrained*/
static struct _ilu_Mooring_s mooringProto = {
  _w3mux_MooringDFd,
  _w3mux_MooringSetReqHandler,
  _w3mux_MooringWaitForReq,
  _w3mux_MooringDisableWait,
  _w3mux_MooringEnableWait,
  _w3mux_MooringAcceptClient,
  _w3mux_CloseMooring,
  NIL				/* data */
};

/**********************************************************************/
/**********************************************************************/
/*								      */
/*			TransportCreator Functions		      */
/*								      */
/**********************************************************************/
/**********************************************************************/

/* after:  if success, w3mux_EndpointsLock is held */
static TransportData
  GetTransport(CreatorParms cp,
	       ilu_boolean buffer,
	       ilu_integer *dfd,
	       ILU_ERRS((no_memory)) * err)
{
  TransportData	ld;
  ilu_Error lerr;
  ilu_Transport lower;
  ilu_Condition cond = NIL;

  ilu_AcquireMutex(w3mux_EndpointsLock);
  /* First, find out the subchannel info */
  ld = (TransportData) ILU_HASH_FINDINTABLE(w3mux_Endpoints, cp->lower_tinfo);
  if (ld == NIL) {
    /* create new tcp transport, and new mux channel on top */
    lower = ((*cp->lower->tcr_createTransport)
	     (cp->lower, TRUE, dfd, NIL, err));
    ld = NewTransportData (lower, NIL, cp->lower_tinfo, NIL, NIL, err);
    if (ILU_ERRNOK(*err)) goto errout;
  } else {
    ilu_AcquireMutex(ld->state_lock);
    ilu_AcquireMutex(ld->io_lock);
    *dfd = 0;
  };  
  ilu_ReleaseMutex(ld->io_lock);
  ILU_CLER(*err);
  return (ld);

 errout:
  ilu_ReleaseMutex(w3mux_EndpointsLock);
  return NIL;
}

/* ...holding w3mux_EndpointsLock... */
static ilu_byte
  AllocateSessionID (TransportData td, ilu_Error *err)
{
  ilu_cardinal id = (td->initiator == NIL) ? 2 : 3;
  for (; id <= 255; id += 2) {
    if (FindSession(td, id) == NIL)
      return (id);
  };
  return ILU_ERR_CONS1(no_resources, err, minor, ilu_nrm_mux_sessions, 0);
}

/* Main Invariant holds */
static ilu_Transport
  _w3mux_CreateTransport(ilu_TransportCreator self,
			 ilu_boolean buffer,
			 ilu_integer *dfd,
			 ilu_Passport pp,	/* unused by this transport */
			 ILU_ERRS((IoErrs)) * err)
/*
 * The outgoing Transport instance creation procedure.  Caller
 * promises exposed buffers won't be used if !buffer (in which case
 * callee is free to not allocate any).  The result is owned by the
 * caller (who will eventually close it).  Always stores at *dfd
 * the number of FDs actually consumed, even when erring.
 */
{
  CreatorParms		cp = (CreatorParms) self->tcr_data;
  TransportData		td;
  ilu_byte		id;
  ilu_integer		ldfd;
  struct {
    MuxHeader		header;
    ilu_cardinal	long_size;
  } hdr;
  ilu_Transport		t;
  ilu_boolean		result;
  TransportParms	p;

  td = GetTransport (cp, buffer, dfd, err);
  if (ILU_ERRNOK(*err)) goto err0;
  /* now find an unused session ID in the transport group */
  id = AllocateSessionID(td, err);
  if (ILU_ERRNOK(*err)) goto err1;
  /* finally, create a new ilu_Transport and return it */
  t = NewTrans (td, id, cp->channel_id, ilu_TRUE, err);
  if (ILU_ERRNOK(*err)) goto err1;
  AddSession(td, t, err);
  if (ILU_ERRNOK(*err)) goto err1;	/* XXX - leak "t" here */
  ILU_NOTE(W3MUX_DEBUG,
	("_w3mux_CreateTransport:  created outgoing mux session %u to channel %u, tcp '%s'\n",
	 id, cp->channel_id, cp->lower_tinfo[0]));
  ilu_ReleaseMutex(td->state_lock);
  ilu_ReleaseMutex(w3mux_EndpointsLock);
  if (!result)
    tHandler((ilu_refany) td);
  return t;

 err1:
  ilu_ReleaseMutex(td->state_lock);
  ilu_ReleaseMutex(w3mux_EndpointsLock);
 err0:
  ILU_NOTE(W3MUX_DEBUG,
	("_w3mux_CreateTransport:  error %s in creating new session to channel %u, tcp '%s'\n",
	 ILU_ERR_NAME(*err), cp->channel_id, cp->lower_tinfo[0]));
  return NIL;
}

/* ...holding w3mux_ChannelsLock... */
static unsigned long
  UseOrAssignChannelID (unsigned long id)
{
  unsigned long retval;
  ilu_refany md;
  ilu_refany p;

  if (id != 0) {
    MooringData md;
    /* check to see it's not in use */
    md = ILU_HASH_FINDINTABLE(w3mux_Channels, (ilu_refany) id);
    retval = (md != NIL) ? 0 : id;
  } else {
    for (retval = MIN_DYNAMIC_CHANNEL_ID, md = (ilu_refany) 1;
	 md != NIL && retval <= MAX_DYNAMIC_CHANNEL_ID;
	 retval += 1) {
      md = ILU_HASH_FINDINTABLE(w3mux_Channels, (ilu_refany) retval);
    }
    if (retval > MAX_DYNAMIC_CHANNEL_ID)
      retval = 0;
  }
  return retval;
}

static MooringData
  FigureMooringData (unsigned long channel_id,
		     ilu_TransportInfo lower)
{
  MooringDataList p;

  for (p = w3mux_Moorings;  p != NIL;  p = p->next) {
    if (strcmp(lower[0], p->md->tinfo_in[0]) == 0) {
      return p->md;
    };
  };
  return NIL;
}

/* before:  ML = {}
   after, no err:  ML = {ListenSockets, lower}
   after, err:     ML = {} */
static MooringData
  GetMooring(CreatorParms cp,
	     ilu_TransportInfo * tinfo_out,
	     unsigned long *id,
	     ilu_integer *dfd,
	     ilu_Passport pp,	/* unused in this transport */
	     ILU_ERRS((no_memory)) * err)
{
  MooringData	ld;
  ilu_Mooring	lower;
  ilu_Error lerr;
  ilu_TransportInfo tinfoOut = NIL;
  ilu_TransportInfo tinfoIn = NIL;	/* doubles as flag for new/old */
  ilu_Condition cond;
  MooringDataList mdl;
  char buf[200];
  unsigned long port;
  ilu_TIH mooring_handler;

  ilu_AcquireMutex(w3mux_ChannelsLock);
  /* first, figure out what id to use */
  *id = UseOrAssignChannelID(cp->channel_id);
  if (*id == 0) {
    ILU_ERR_CONS1(no_resources, err, minor, ilu_nrm_mux_channels, err);
    goto free1;
  };
  ld = FigureMooringData(*id, cp->lower_tinfo);
  if (ld == NIL) {
    /* create new tcp mooring, and new mux channel on top */
    lower = ((*cp->lower->tcr_createMooring)
	     (cp->lower, &tinfoOut, TRUE, dfd, pp, err));
    if (ILU_ERRNOK(*err)) goto free1;
    tinfoIn = _ilu_CopyTinfo(cp->lower_tinfo, err);
    if (ILU_ERRNOK(*err)) goto free2;
    if (ilu_CanCondition())
      cond = ilu_CreateCondition("w3muxed-transport ", cp->lower_tinfo[0], err);
    if (ILU_ERRNOK(*err)) goto free3;
    ld = ilu_MallocE(sizeof(*ld), err);
    if (ILU_ERRNOK(*err)) goto free4;
    ld->mooring = lower;
    ld->tinfo_in = tinfoIn;
    ld->tinfo_out = tinfoOut;
    ld->active = ilu_FALSE;
    if (tinfoOut[1] == NIL && (sscanf(tinfoOut[0], "tcp_%199[^_]_%lu", &buf, &port) == 2)) {
      ld->ip_addr = inet_addr(buf);
      ld->port = port;
    } else {
      ld->ip_addr = 0;
      ld->port = 0;
    };
    ld->channels.lock = ilu_CreateMutex("w3mux members over ", cp->lower_tinfo[0]);
    ld->channels.list = NIL;
    mdl = (MooringDataList) ilu_MallocE(sizeof(*mdl), err);
    if (ILU_ERRNOK(*err)) goto free6;
    mdl->md = ld;
    mdl->next = w3mux_Moorings;
    w3mux_Moorings = mdl;
    /* activate it by registering a callback handler, or forking a work thread */
    if (_ilu_HasForkProc()) {
      (void) _ilu_ForkNewThread (handleMooring, (void *) ld, err);
    } else {
      mooring_handler = ilu_MallocE(sizeof(*mooring_handler), err);
      if (ILU_ERROK(*err)) {
	mooring_handler->next = NIL;
	mooring_handler->proc = mHandler;
	mooring_handler->rock = ld;
	mooring_set_req_handler(ld->mooring, mooring_handler, err);
	ld->active = ilu_TRUE;
      };
    }
    if (ILU_ERRNOK(*err)) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("_w3mux_CreateMooring:  Can't establish mooring handler for tcp socket %s, err is %s.\n",
	     ld->tinfo_in[0], ILU_ERR_NAME(*err)));
    };
  } else {
    *dfd = 0;
  };
  if (tinfo_out != NIL) {
    *tinfo_out = _ilu_CopyTinfo(ld->tinfo_out, err);
    if (ILU_ERRNOK(*err)) {
      if (tinfoIn == NIL)
	goto free1;
      else
	goto free6;
    }
  };
  ILU_CLER(*err);
  return (ld);

 free6:
  ilu_DestroyMutex(ld->channels.lock, &lerr);
  ILU_HANDLED(lerr);
 free5:
  ilu_free(ld);
 free4:
  if (cond != NIL)
    ilu_DestroyCondition(cond);
 free3:
  ilu_free(tinfoIn);
 free2:
  ilu_free(tinfoOut);
  (*lower->mo_close)(lower, dfd, &lerr);
  ILU_HANDLED(lerr);
 free1:
  ilu_ReleaseMutex(w3mux_ChannelsLock);
  return NIL;
}

/* L1.sup < trmu; L2 unconstrained */
static ilu_Mooring
  _w3mux_CreateMooring (ilu_TransportCreator self,
			ilu_TransportInfo * tinfo_out,
			ilu_boolean buffer,
			ilu_integer *dfd,
			ilu_Passport pp,	/* unused in this transport */
			ILU_ERRS((no_memory)) * err)
/*
 * The Mooring instance creation procedure.  Caller promises
 * exposed buffers of results of mo_accept_connection won't be
 * used if !buffer (in which case mo_accept_connection is free to
 * not allocate any).  If tinfo_out != NIL, store through it a
 * fully-detailed transport info string, which will be passed to
 * ilu_GetTransportCreator in a peer.  Always store at *dfd the
 * number of FDs actually consumed.  The results are owned by the
 * caller (who will eventually close the mooring and ilu_free the
 * string).
 */
{
  CreatorParms		cp = (CreatorParms) self->tcr_data;
  MooringParms		mp;
  ilu_Mooring		ans;
  MooringData		md;
  unsigned long		id;
  ilu_TransportInfo	subtinfo = NIL;

  md = GetMooring (cp, (tinfo_out != NIL) ? &subtinfo : NIL, &id, dfd, pp, err);
  if (ILU_ERRNOK(*err)) return NIL;
  /* now holding lock on "lower" */
  mp = (MooringParms) ilu_MallocE(sizeof(*mp), err);
  if (ILU_ERRNOK(*err)) goto err1;
  ans = (ilu_Mooring) ilu_MallocE(sizeof(*ans), err);
  if (ILU_ERRNOK(*err)) goto err2;
  if (tinfo_out != NIL) {
    char buf[20];
    sprintf (buf, "w3mux_%u", id);
    *tinfo_out = _ilu_ConcatTinfo (buf, subtinfo, err);
    if (ILU_ERRNOK(*err)) goto err3;
    ilu_free(subtinfo);
  }
  ILU_NOTE(W3MUX_DEBUG,
	("_w3mux_CreateMooring:  channel %u over %s created:  %p\n",
	 id, md->tinfo_out[0], ans));
  *ans = mooringProto;
  ans->mo_data = mp;
  mp->self = ans;
  mp->group = md;
  mp->channel_id = id;
  mp->disabled = ilu_FALSE;
  AddChannel (md, ans, err);
  ILU_HASH_ADDTOTABLE (w3mux_Channels, (ilu_refany) id, ans);
  ilu_ReleaseMutex(w3mux_ChannelsLock);
  return (ans);

 err3:
  ilu_free(ans);
 err2:
  ilu_free(mp);
 err1:
  if (tinfo_out != NIL)
    ilu_free(subtinfo);
  ilu_ReleaseMutex(w3mux_ChannelsLock);
  return NIL;
}

/*L1, L2 unconstrained*/
static void
  _w3mux_CloseCreator (ilu_TransportCreator self)
/* Even frees self. */
{
  CreatorParms    cp = (CreatorParms) self->tcr_data;
  (*cp->lower->tcr_close) (cp->lower);
  if (cp->lower_tinfo != NIL)
    ilu_free(cp->lower_tinfo);
  ilu_free(cp);
  ilu_free(self);
  return;
}

/* L1.sup < trmu; L2 unconstrained */
static ilu_integer
  _w3mux_FdUsage (ilu_TransportCreator self, ilu_boolean mooring)
/*
 * Estimates how many FDs will be consumed when creating a mooring
 * (if mooring) or outgoing transport (if !mooring).  Estimate may
 * err high, but not low.
 */
{
  /* We will use 0 file descriptions in this layer, so just
   * pass along whatever's necessary in any underlying layer.
   * It may be high, but won't be low.
   */
  CreatorParms    cp = (CreatorParms) self->tcr_data;
  return (*cp->lower->tcr_dfd) (cp->lower, mooring);
}

/*L1, L2 unconstrained*/
static ilu_cardinal
  hash_tinfo (void *key, ilu_cardinal modulus)
{
  ilu_TransportInfo tinfo = (ilu_TransportInfo) key;
  int i;
  ilu_cardinal hashval = 0xFFFFFFFF;

  for (i = 0;  tinfo[i] != NIL;  i++) {
    hashval = ilu_CRC32WithAccum((ilu_bytes) tinfo[i], strlen(tinfo[i]), hashval);
  };
  return (hashval % modulus);
}

/*L1, L2 unconstrained*/
static ilu_boolean
  compare_tinfo (void *key1, void *key2)
{
  ilu_TransportInfo tinfo1 = (ilu_TransportInfo) key1;
  ilu_TransportInfo tinfo2 = (ilu_TransportInfo) key2;
  int i;

  for (i = 0;  tinfo1[i] != NIL;  i++) {
    if (tinfo2[i] == NIL ||
	strcmp(tinfo1[i], tinfo2[i]) != 0)
      return ilu_FALSE;
  }
  return (tinfo2[i] == NIL);
}

static struct _ilu_TransportCreator_s myCreatorProto = {
  TRUE,				/* boundaried */
  TRUE,				/* reliable */
  0,				/* tcr_holds */
  FALSE,			/* tcr_wantClose */
  _w3mux_FdUsage,
  _w3mux_CreateTransport,
  _w3mux_CreateMooring,
  _w3mux_CloseCreator,
  NIL				/* data */
};

/*L1_sup < trmu; L2 unconstrained*/
static CreatorParms
  InterpretInfo(ilu_TransportInfo info,
		ILU_ERRS((no_memory, inv_objref)) * err)
{
  CreatorParms    cp;
  ilu_TransportCreator lower;
  unsigned long id;
  ilu_TransportInfo lower_tinfo;
  char hostname[200];
  ilu_cardinal port;

  if ((strcmp(info[0], "w3mux") == 0) && (info[1] != NIL))
    id = 0;
  else if (sscanf(info[0], "w3mux_%lu", &id) != 1 ||
	   info[1] == NIL)
    return ILU_ERR_CONS1(inv_objref, err, minor, ilu_iom_ts, NIL);
  lower = _ilu_GetTransportCreator(info + 1, err);
  if (ILU_ERRNOK(*err))
    return NIL;
  if (lower->tcr_boundaried || !lower->tcr_reliable)
    return ILU_ERR_CONS1(inv_objref, err, minor, ilu_iom_ts, NIL);
  if (info[2] == NIL && (sscanf(info[1], "tcp_%199[^_]_%lu", hostname, &port) == 2)) {
    ilu_string tempname;
    struct in_addr myaddr;
    struct hostent *he;
    struct in_addr *hea;
    char *tmp_tinfo[2];
    he = gethostbyname(hostname);
    if (he == NIL || he->h_addr_list == NIL || he->h_addr_list[0] == NIL)
      lower_tinfo = _ilu_CopyTinfo(info + 1, err);
    else {
      hea = (struct in_addr *) (he->h_addr_list[0]);
      tempname = (ilu_string) inet_ntoa(myaddr = *hea);
      sprintf(hostname, "tcp_%s_%u", tempname, port);
      tmp_tinfo[0] = hostname;
      tmp_tinfo[1] = NIL;
      lower_tinfo = _ilu_CopyTinfo(tmp_tinfo, err);
    };
  } else
    lower_tinfo = _ilu_CopyTinfo(info + 1, err);
  if (ILU_ERRNOK(*err))
    return NIL;
  cp = (CreatorParms) ilu_malloc(sizeof(*cp));
  if (cp == NIL)
    return ILU_ERR_CONS1(no_memory, err, nbytes, sizeof(*cp), NIL);
  cp->lower = lower;
  cp->channel_id = id;
  cp->lower_tinfo = lower_tinfo;
  return cp;
}

static ilu_boolean _w3mux_InitializeDataStructures()
{
  ilu_Error lerr;

  if (w3mux_Channels == NIL) {
    /* initialize hash table of ListenSockets */
    w3mux_Channels = ILU_HASH_MAKENEWTABLE(CONNECTION_TABLE_BUCKETS,
					   ILU_HASH_HASHPOINTER,
					   ILU_HASH_POINTERCOMPARE);
    if (w3mux_Channels == NIL) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("_w3mux_InitializeDataStructures:  "
	     "Can't allocate hash table for underlying ListenSockets!\n"));
      return ilu_FALSE;
    };
    w3mux_ChannelsLock = ilu_CreateMutex("w3mux", "ChannelsLock");

    /* initialize hash table of Transports */
    w3mux_Endpoints = ILU_HASH_MAKENEWTABLE(CONNECTION_TABLE_BUCKETS,
					      hash_tinfo, compare_tinfo);
    if (w3mux_Endpoints == NIL) {
      ILU_NOTE(W3MUX_DEBUG,
	    ("w3mux_Initialize:  "
	     "Can't allocate hash table for underlying Transports!\n"));
      return ilu_FALSE;
    };
    w3mux_EndpointsLock = ilu_CreateMutex("w3mux", "TransportsLock");

    /* create a lock to use for channel acceptance */
    w3mux_ConnReqsLock = ilu_CreateMutex("w3mux", "ConnReqsLock");

    /* create a condition to use in waiting for a new session req */
    if (_ilu_HasForkProc()) {
      w3mux_ConnReqsCond = ilu_CreateCondition("w3mux", "ConnReqsCond", &lerr);
      if (ILU_ERRNOK(lerr)) {
	ILU_NOTE(W3MUX_DEBUG,
	      ("w3mux_Initialize:  "
	       "Can't create w3mux_ConnReqsCond, err <%s>\n", ILU_ERR_NAME(lerr)));
	ILU_HANDLED(lerr);
	return ilu_FALSE;
      }
    }
  }
}

/*L1_sup < trmu*/
ilu_TransportCreator
  _ilu_w3mux_TransportCreator(ilu_TransportInfo tinfo,
			      ILU_ERRS((no_memory,
					inv_objref)) * err)
/*
 * Something that parses "transport info" and returns corresponding
 * a ilu_TransportCreator.
 */
{
  ilu_TransportCreator ans;
  CreatorParms    cp;

  _w3mux_InitializeDataStructures();

  cp = InterpretInfo(tinfo, err);
  if (ILU_ERRNOK(*err))
    return NIL;
  ans = (ilu_TransportCreator) ilu_malloc(sizeof(*ans));
  if (ans == NIL)
    return ILU_ERR_CONS1(no_memory, err, nbytes, sizeof(*ans), NIL);
  *ans = myCreatorProto;
  ans->tcr_data = cp;
  ILU_CLER(*err);
  return ans;
}
