/*	$Id: line.c,v 1.16 1997/10/31 20:35:50 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * All the common editing function are declared in this file.
 * This includes the backspace and delete functions, the kill ring
 * and the registers.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "zile.h"
#include "extern.h"

/*
 * This function creates a new line structure for holding a line
 * of buffer text.  If called with `maxsize == 0', it does not allocate
 * a text space at all (this trick is used for allocating the
 * buffer limit marker `bp->limitp').
 *
 * The '\n' is not stored in the text buffer since it is implicit.
 *
 * The tail of the buffer structure is used for storing the text buffer.
 */
linep
new_line(int maxsize)
{
	linep lp;

	lp = (linep)xmalloc(sizeof *lp + maxsize - sizeof lp->text);
	memset(lp, 0, sizeof *lp);

	lp->size = 0;
	lp->maxsize = maxsize;

	return lp;
}

/*
 * This function resizes the text buffer of the line by reallocating
 * the whole line structure.  If the pointer returned by realloc()
 * is different from the original one, scan all the windows looking
 * for the references to the old pointer and update their value.
 */
linep
resize_line(windowp wp, linep lp, int maxsize)
{
	linep newlp;
	windowp wp1;

	newlp = (linep)xrealloc(lp, sizeof *lp + maxsize - sizeof lp->text);
	newlp->maxsize = maxsize;

	/*
	 * Little optimization: if the reallocated pointer
	 * is equal to the original, perform no other operations.
	 */
	if (newlp == lp)
		return newlp;

	newlp->prev->next = newlp;
	newlp->next->prev = newlp;

	/*
	 * Scan all the windows searching for points and marks
	 * pointing at the reallocated line.
	 */
	for (wp1 = head_wp; wp1 != NULL; wp1 = wp1->next)
		if (wp1->pointp == lp)
			wp1->pointp = newlp;

	if (wp->bp->markp == lp)
		wp->bp->markp = newlp;

	return newlp;
}

/*
 * Free the line contents.
 */
void
free_line(linep lp)
{
	if (lp->anchors != NULL)
		free(lp->anchors);
	free(lp);
}

/*
 * Insert the character `c' at the current point position
 * into the current buffer.
 */
int
insert_char(int c)
{
	windowp wp;
	int pointo;

	if (warn_if_readonly_buffer())
		return FALSE;

	if (cur_bp->flags & BFLAG_OVERWRITE) {
		if (cur_wp->pointo < cur_wp->pointp->size) {
			/*
			 * XXX Emacs behaviour:
			 * "Before a tab, such characters insert until
			 * the tab is filled in."
			 */
			undo_save(UNDO_REPLACE_CHAR,
				  cur_wp->pointn, cur_wp->pointo,
				  cur_wp->pointp->text[cur_wp->pointo]);
			cur_wp->pointp->text[cur_wp->pointo] = c;
			++cur_wp->pointo;

			cur_bp->flags |= BFLAG_MODIFIED;

			if (cur_bp->flags & BFLAG_FONTLOCK)
				font_lock_reset_anchors(cur_bp, cur_wp->pointp);

			return TRUE;
		}
		/*
		 * Fall through the "insertion" mode of a character
		 * at the end of the line, since it is totally
		 * equivalent also in "overwrite" mode.
		 */
	}

	/*
	 * Resize the line if required.
	 */
	if (cur_wp->pointp->size + 1 >= cur_wp->pointp->maxsize)
		resize_line(cur_wp, cur_wp->pointp,
			    cur_wp->pointp->maxsize + 10);

	/*
	 * Move the line text one position forward after the
	 * point if required.
	 * This code assumes that memmove(d, s, 0) does nothing.
	 */
	memmove(cur_wp->pointp->text + cur_wp->pointo + 1,
		cur_wp->pointp->text + cur_wp->pointo,
		cur_wp->pointp->size - cur_wp->pointo);

	undo_save(UNDO_REMOVE_CHAR, cur_wp->pointn, cur_wp->pointo, 0);
	cur_wp->pointp->text[cur_wp->pointo] = c;

	pointo = cur_wp->pointo;

	/*
	 * Scan all the windows searching for points
	 * pointing at the modified line.
	 */
	for (wp = head_wp; wp != NULL; wp = wp->next)
		if (wp->pointp == cur_wp->pointp && wp->pointo >= pointo)
			++wp->pointo;

	if (cur_bp->markp == cur_wp->pointp && cur_bp->marko >= pointo)
		++cur_bp->marko;

	++cur_wp->pointp->size;

	cur_bp->flags |= BFLAG_MODIFIED;

	if (cur_bp->flags & BFLAG_FONTLOCK)
		font_lock_reset_anchors(cur_bp, cur_wp->pointp);

	return TRUE;
}

static void
insert_expanded_tab(void)
{
	int col = 0, t = cur_bp->tab_width;
	char *sp = cur_wp->pointp->text, *p = sp;

	while (p < sp + cur_wp->pointo) {
		if (*p == '\t')
			col |= t - 1;
		++col, ++p;
	}

	for (col = t - col % t; col > 0; --col)
		insert_char(' ');
}

/*
 * Insert a tabulation at the current point position into
 * the current buffer.  Convert the tabulation into spaces
 * if the `expand-tabs' variable is bound and set to true.
 */
int
insert_tab(void)
{
	if (warn_if_readonly_buffer())
		return FALSE;

	if (!lookup_bool_variable("expand-tabs"))
		insert_char('\t');
	else
		insert_expanded_tab();

	return TRUE;
}

/*
 * Insert a newline at the current point position into
 * the current buffer.
 */
int
insert_newline(void)
{
	linep lp1, lp2;
	int lp1len, lp2len;
	windowp wp;

	if (warn_if_readonly_buffer())
		return FALSE;

	undo_save(UNDO_REMOVE_CHAR, cur_wp->pointn, cur_wp->pointo, 0);

	/*
	 * Calculate the two line lengths.
	 */
	lp1len = cur_wp->pointo;
	lp2len = cur_wp->pointp->size - lp1len;

	lp1 = cur_wp->pointp;
	lp2 = new_line(lp2len ? lp2len : 1);

	/*
	 * Copy the text after the point into the new line.
	 * This code assumes that memcpy(d, s, 0) does nothing.
	 */
	memcpy(lp2->text, lp1->text + lp1len, lp2len);

	lp1->size -= lp2len;
	lp2->size = lp2len;

	/*
	 * Update line linked list.
	 */
	lp2->next = lp1->next;
	lp2->next->prev = lp2;
	lp2->prev = lp1;
	lp1->next = lp2;
	++cur_bp->num_lines;

	/*
	 * Scan all the windows searching for points
	 * pointing at the changed line.
	 */
	for (wp = head_wp; wp != NULL; wp = wp->next) {
		if (wp->bp != cur_bp)
			continue;
		if (wp->pointp == lp1 && wp->pointo >= lp1len) {
			wp->pointp = lp2;
			wp->pointo -= lp1len;
			++wp->pointn;
		} else if (wp->pointn > cur_wp->pointn)
			++wp->pointn;
	}

	if (cur_bp->markp == lp1 && cur_bp->marko >= lp1len) {
		cur_bp->markp = lp2;
		cur_bp->marko -= lp1len;
	}

	cur_bp->flags |= BFLAG_MODIFIED;

	if (cur_bp->flags & BFLAG_FONTLOCK) {
		font_lock_reset_anchors(cur_bp, cur_wp->pointp->prev);
		font_lock_reset_anchors(cur_bp, cur_wp->pointp);
	}

	thisflag |= FLAG_NEED_RESYNC;

	return TRUE;
}

void
insert_string(char *s)
{
	for (; *s != '\0'; ++s)
		if (*s == '\n')
			insert_newline();
		else
			insert_char(*s);
}

void
insert_nstring(char *s, size_t size)
{
	for (; 0 < size--; ++s)
		if (*s == '\n')
			insert_newline();
		else
			insert_char(*s);
}

int
self_insert_command(int c)
{
	if (c == KBD_RET)
		insert_newline();
	else if (c == KBD_TAB)
		insert_tab();
	else if (c <= 255)
		insert_char(c);
	else {
		ding();
		return FALSE;
	}

	return TRUE;
}

DEFUN("self-insert-command", self_insert_command)
/*+
Insert the character you type.
+*/
{
	int uni, c;

	c = cur_tp->getkey();

	for (uni = 0; uni < uniarg; ++uni)
		if (!self_insert_command(c))
			return FALSE;

	return TRUE;
}

void
bprintf(const char *fmt, ...)
{
	va_list ap;
	char buf[1024];

	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	va_end(ap);

	insert_string(buf);
}

int
delete_char(void)
{
	windowp wp;

	if (cur_wp->pointo < cur_wp->pointp->size) {
		if (warn_if_readonly_buffer())
			return FALSE;

		undo_save(UNDO_INSERT_CHAR, cur_wp->pointn, cur_wp->pointo,
			  cur_wp->pointp->text[cur_wp->pointo]);

		/*
		 * Move the text one position backward after the point,
		 * if required.
		 * This code assumes that memcpy(d, s, 0) does nothing.
		 */
		memcpy(cur_wp->pointp->text + cur_wp->pointo,
		       cur_wp->pointp->text + cur_wp->pointo + 1,
		       cur_wp->pointp->size - cur_wp->pointo - 1);
		--cur_wp->pointp->size;

		/*
		 * Scan all the windows searching for points
		 * pointing at the modified line.
		 */
		for (wp = head_wp; wp != NULL; wp = wp->next)
			if (wp->pointp == cur_wp->pointp
			    && wp->pointo > cur_wp->pointo)
				--wp->pointo;

		if (cur_bp->markp == cur_wp->pointp
		    && cur_bp->marko > cur_wp->pointo)
			--cur_bp->marko;

		if (cur_bp->flags & BFLAG_FONTLOCK)
			font_lock_reset_anchors(cur_bp, cur_wp->pointp);

		cur_bp->flags |= BFLAG_MODIFIED;

		return TRUE;
	} else if (cur_wp->pointp->next != cur_bp->limitp) {
		linep lp1, lp2;
		int lp1len, lp2len;

		if (warn_if_readonly_buffer())
			return FALSE;

		undo_save(UNDO_INSERT_CHAR, cur_wp->pointn, cur_wp->pointo, '\n');

		lp1 = cur_wp->pointp;
		lp2 = cur_wp->pointp->next;
		lp1len = lp1->size;
		lp2len = lp2->size;

		/*
		 * Resize line if required.
		 */
		if (lp1len + lp2len + 1 >= lp1->maxsize)
			lp1 = resize_line(cur_wp, lp1, lp1len + lp2len + 1);

		if (lp2len > 0) {
			/*
			 * Move the next line text into the current line.
			 */
			memcpy(lp1->text + lp1len, lp2->text, lp2len);
			lp1->size += lp2len;
		}

		/*
		 * Update line linked list.
		 */
		lp1->next = lp2->next;
		lp1->next->prev = lp1;
		--cur_bp->num_lines;

		/*
		 * Free the unlinked line.
		 */
		free_line(lp2);

		/*
		 * Scan all the windows searching for points
		 * pointing at the deallocated line.
		 */
		for (wp = head_wp; wp != NULL; wp = wp->next) {
			if (wp->bp != cur_bp)
				continue;
			if (wp->pointp == lp2) {
				wp->pointp = lp1;
				wp->pointo += lp1len;
				--wp->pointn;
			} else if (wp->pointn > cur_wp->pointn)
				--wp->pointn;
		}

		if (cur_bp->markp == lp2) {
			cur_bp->markp = lp1;
			cur_bp->marko += lp1len;
		}

		cur_bp->flags |= BFLAG_MODIFIED;

		if (cur_bp->flags & BFLAG_FONTLOCK)
			font_lock_reset_anchors(cur_bp, cur_wp->pointp);

		thisflag |= FLAG_NEED_RESYNC;

		return TRUE;
	}

	minibuf_error("%FCEnd of buffer%E");

	return FALSE;
}

DEFUN("delete-char", delete_char)
/*+
Delete the following character.
Join lines if the character is a newline.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_backward_delete_char(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!delete_char())
			return FALSE;

	return TRUE;
}

int
backward_delete_char(void)
{
	windowp wp;

	if (cur_wp->pointo > 0) {
		if (warn_if_readonly_buffer())
			return FALSE;

		undo_save(UNDO_INSERT_CHAR, cur_wp->pointn, cur_wp->pointo-1,
			  cur_wp->pointp->text[cur_wp->pointo - 1]);

		/*
		 * Move the text one position backward before the point,
		 * if required.
		 * This code assumes that memcpy(d, s, 0) does nothing.
		 */
		memcpy(cur_wp->pointp->text + cur_wp->pointo - 1,
		       cur_wp->pointp->text + cur_wp->pointo,
		       cur_wp->pointp->size - cur_wp->pointo);
		--cur_wp->pointp->size;

		/*
		 * Scan all the windows searching for points
		 * pointing at the modified line.
		 */
		for (wp = head_wp; wp != NULL; wp = wp->next)
			if (wp->pointp == cur_wp->pointp
			    && wp->pointo >= cur_wp->pointo)
				--wp->pointo;

		if (cur_bp->markp == cur_wp->pointp
		    && cur_bp->marko >= cur_wp->pointo)
			--cur_bp->marko;

		cur_bp->flags |= BFLAG_MODIFIED;

		if (cur_bp->flags & BFLAG_FONTLOCK)
			font_lock_reset_anchors(cur_bp, cur_wp->pointp);

		return TRUE;
	} else if (cur_wp->pointp->prev != cur_bp->limitp) {
		linep lp1, lp2;
		int lp1len, lp2len;

		if (warn_if_readonly_buffer())
			return FALSE;

		undo_save(UNDO_INSERT_CHAR, cur_wp->pointn - 1,
			  cur_wp->pointp->prev->size, '\n');

		lp1 = cur_wp->pointp->prev;
		lp2 = cur_wp->pointp;
		lp1len = lp1->size;
		lp2len = lp2->size;

		/*
		 * Resize line if required.
		 */
		if (lp1len + lp2len + 1 >= lp1->maxsize)
			lp1 = resize_line(cur_wp, lp1, lp1len + lp2len + 1);

		if (lp2len > 0) {
			/*
			 * Move the current line text into the previous line.
			 */
			memcpy(lp1->text + lp1len, lp2->text, lp2len);
			lp1->size += lp2len;
		}

		/*
		 * Update line linked list.
		 */
		lp1->next = lp2->next;
		lp1->next->prev = lp1;
		--cur_bp->num_lines;
		assert(cur_bp->limitp->next != lp2);
		assert(cur_bp->limitp->prev != lp2);

		/*
		 * Free the unlinked line.
		 */
		free_line(lp2);

		cur_wp->pointp = lp1;
		cur_wp->pointo += lp1len;
		--cur_wp->pointn;

		/*
		 * Scan all the windows searching for points
		 * pointing at the deallocated line.
		 */
		for (wp = head_wp; wp != NULL; wp = wp->next) {
			if (wp->bp != cur_bp)
				continue;
			if (wp->pointp == lp2) {
				wp->pointp = lp1;
				wp->pointo += lp1len;
				--wp->pointn;
			} else if (wp->pointn > cur_wp->pointn)
				--wp->pointn;
		}

		if (cur_bp->markp == lp2) {
			cur_bp->markp = lp1;
			cur_bp->marko += lp1len;
		}

		cur_bp->flags |= BFLAG_MODIFIED;

		if (cur_bp->flags & BFLAG_FONTLOCK)
			font_lock_reset_anchors(cur_bp, cur_wp->pointp);

		thisflag |= FLAG_NEED_RESYNC;

		return TRUE;
	}

	minibuf_error("%FCBeginning of buffer%E");

	return FALSE;
}

DEFUN("backward-delete-char", backward_delete_char)
/*+
Delete the previous character.
Join lines if the character is a newline.
+*/
{
	int uni;

	if (uniarg < 0)
		return F_delete_char(-uniarg);

	for (uni = 0; uni < uniarg; ++uni)
		if (!backward_delete_char())
			return FALSE;

	return TRUE;
}

/*--------------------------------------------------------------------------
 * Kill ring.
 *--------------------------------------------------------------------------*/

static char *kill_ring_text;
static int kill_ring_size;
static int kill_ring_maxsize;

static void
flush_kill_ring(void)
{
	kill_ring_size = 0;
	if (kill_ring_maxsize > 1024) {
		/*
		 * Deallocate memory area, since is big enough.
		 */
		kill_ring_maxsize = 0;
		free(kill_ring_text);
		kill_ring_text = NULL;
	}
}

static void
kill_ring_push(char c)
{
	if (kill_ring_text == NULL) {
		kill_ring_maxsize = 16;
		kill_ring_text = (char *)xmalloc(kill_ring_maxsize);
	} else if (kill_ring_size + 1 >= kill_ring_maxsize) {
		kill_ring_maxsize += 16;
		kill_ring_text = (char *)xrealloc(kill_ring_text, kill_ring_maxsize);
	}
	kill_ring_text[kill_ring_size++] = c;
}

static void
kill_ring_push_nstring(char *s, size_t size)
{
	if (kill_ring_text == NULL) {
		kill_ring_maxsize = size;
		kill_ring_text = (char *)xmalloc(size);
	} else if (kill_ring_size + size >= kill_ring_maxsize) {
		kill_ring_maxsize += size;
		kill_ring_text = (char *)xrealloc(kill_ring_text, kill_ring_maxsize);
	}
	memcpy(kill_ring_text + kill_ring_size, s, size);
	kill_ring_size += size;
}

DEFUN("kill-line", kill_line)
/*+
Kill the rest of the current line; if no nonblanks there, kill thru newline.
+*/
{
	if (!(lastflag & FLAG_DONE_KILL))
		flush_kill_ring();

	if (cur_wp->pointo < cur_wp->pointp->size) {
		if (warn_if_readonly_buffer())
			return FALSE;

		undo_save(UNDO_INSERT_BLOCK, cur_wp->pointn, cur_wp->pointo,
			  cur_wp->pointp->size - cur_wp->pointo);
		undo_nosave = 1;
		while (cur_wp->pointo < cur_wp->pointp->size) {
			kill_ring_push(cur_wp->pointp->text[cur_wp->pointo]);
			F_delete_char(1);
		}
		undo_nosave = 0;

		thisflag |= FLAG_DONE_KILL;

		return TRUE;
	} else if (cur_wp->pointp->next != cur_bp->limitp) {
		if (!F_delete_char(1))
			return FALSE;

		kill_ring_push('\n');

		thisflag |= FLAG_DONE_KILL;

		return TRUE;
	}

	minibuf_error("%FCEnd of buffer%E");

	return FALSE;
}

DEFUN("kill-region", kill_region)
/*+
Kill between point and mark.
The text is deleted but saved in the kill ring.
The command C-y (yank) can retrieve it from there.
If the buffer is read-only, Zile will beep and refrain from deleting
the text, but put the text in the kill ring anyway.  This means that
you can use the killing commands to copy text from a read-only buffer.
If the previous command was also a kill command,
the text killed this time appends to the text killed last time
to make one entry in the kill ring.
+*/
{
	struct region r;

	if (!(lastflag & FLAG_DONE_KILL))
		flush_kill_ring();

	if (warn_if_no_mark())
		return FALSE;

	calculate_region(&r);

	if (cur_bp->flags & BFLAG_READONLY) {
		/*
		 * The buffer is readonly; save only in the kill buffer
		 * and complain.
		 */
		char *p;

		p = copy_text_block(r.startn, r.starto, r.size);
		kill_ring_push_nstring(p, r.size);
		free(p);

		warn_if_readonly_buffer();
	} else {
		int size = r.size;

		if (cur_wp->pointp != r.startp || r.starto != cur_wp->pointo)
			F_exchange_point_and_mark(1);
		undo_save(UNDO_INSERT_BLOCK, cur_wp->pointn, cur_wp->pointo, size);
		undo_nosave = 1;
		while (size--) {
			if (cur_wp->pointo < cur_wp->pointp->size)
				kill_ring_push(cur_wp->pointp->text[cur_wp->pointo]);
			else
				kill_ring_push('\n');
			F_delete_char(1);
		}
		undo_nosave = 0;
	}

	thisflag |= FLAG_DONE_KILL;

	return TRUE;
}

DEFUN("yank", yank)
/*+
Reinsert the last stretch of killed text.
More precisely, reinsert the stretch of killed text most recently
killed OR yanked.  Put point at end, and set mark at beginning.
+*/
{
	if (kill_ring_size == 0) {
		minibuf_error("%FCKill ring is empty%E");
		return FALSE;
	}

	if (warn_if_readonly_buffer())
		return FALSE;

	F_set_mark_command(1);

	undo_save(UNDO_REMOVE_BLOCK, cur_wp->pointn, cur_wp->pointo, kill_ring_size);
	undo_nosave = 1;
	insert_nstring(kill_ring_text, kill_ring_size);
	undo_nosave = 0;

	return TRUE;
}

/*--------------------------------------------------------------------------
 * Registers.
 *--------------------------------------------------------------------------*/

#define NUM_REGISTERS	256

static struct {
	char	*text;
	size_t	size;
} regs[NUM_REGISTERS];

DEFUN("copy-to-register", copy_to_register)
/*+
Copy region into the user specified register.
+*/
{
	struct region r;
	char *p;
	int reg;

	minibuf_write("Copy to register: ");
	if ((reg = cur_tp->getkey()) == KBD_CANCEL)
		return cancel();
	minibuf_clear();
	reg %= NUM_REGISTERS;

	if (warn_if_no_mark())
		return FALSE;

	calculate_region(&r);

	p = copy_text_block(r.startn, r.starto, r.size);
	if (regs[reg].text != NULL)
		free(regs[reg].text);
	regs[reg].text = p;
	regs[reg].size = r.size;

	return TRUE;
}

DEFUN("insert-register", insert_register)
/*+
Insert contents of the user specified register.
Puts point before and mark after the inserted text.
+*/
{
	int reg;

	if (warn_if_readonly_buffer())
		return FALSE;

	minibuf_write("Insert register: ");
	if ((reg = cur_tp->getkey()) == KBD_CANCEL)
		return cancel();
	minibuf_clear();
	reg %= NUM_REGISTERS;

	if (regs[reg].text == NULL) {
		minibuf_error("%FCRegister does not contain text%E");
		return FALSE;
	}

	undo_save(UNDO_REMOVE_BLOCK, cur_wp->pointn, cur_wp->pointo, regs[reg].size);
	undo_nosave = 1;
	insert_nstring(regs[reg].text, regs[reg].size);
	undo_nosave = 0;

	F_set_mark_command(1);

	return TRUE;
}
