/********************************************************************** 
 Freeciv - Copyright (C) 1996 - A Kjeldberg, L Gregersen, P Unold
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <player.h>
#include <unithand.h>
#include <packets.h>
#include <civserver.h>
#include <map.h>
#include <maphand.h>
#include <cityhand.h>
#include <unit.h>
#include <plrhand.h>
#include <city.h>
#include <log.h>
#include <mapgen.h>
#include <events.h>
#include <shared.h>
#include <aiunit.h>

extern struct advance advances[];

void update_unit_activity(struct player *pplayer, struct unit *punit);
void handle_unit_enter_city(struct player *pplayer, struct city *pcity);
void unit_restore_hitpoints(struct player *pplayer, struct unit *punit);
void unit_restore_movepoints(struct player *pplayer, struct unit *punit);
int hp_gain_coord(struct unit *punit);
void do_unit_goto(struct player *pplayer, struct unit *punit);
int unit_ignores_citywalls(struct unit *punit);
int is_ok_city_spot(int x, int y);

int ai_calc_pollution(struct unit *punit, struct player *pplayer, int x, int y);
int ai_calc_mine(struct unit *punit, struct player *pplayer, int x, int y);
int ai_calc_road(struct unit *punit, struct player *pplayer, int x, int y);
int diplomat_on_tile(int x, int y);
int in_city_radius(struct player *pplayer, int x, int y);
int make_dx(int x1, int x2);
int make_dy(int y1, int y2);


/* May wish to make this global */
static
char* n_if_vowel(char ch)
{
	if (strchr("AEIOUaeiou",ch))
		return "n";
	else
		return "";
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_goto_tile(struct player *pplayer, 
			   struct packet_unit_request *req)
{
  struct unit *punit;
  
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    punit->goto_dest_x=req->x;
    punit->goto_dest_y=req->y;

    punit->activity=ACTIVITY_GOTO;
    punit->activity_count=0;

    send_unit_info(0, punit, 0);
      
    do_unit_goto(pplayer, punit);  
  }
}

/**************************************************************************
...
**************************************************************************/
void handle_upgrade_unit_request(struct player *pplayer, 
			    struct packet_unittype_info *packet)
{
  struct genlist_iterator myiter;
  struct unit *punit;
  int cost;
  int to_unit;
  int upgraded = 0;
  if ((to_unit =can_upgrade_unittype(pplayer, packet->type)) == -1) {
    notify_player(pplayer, "Game: illegal package, can't upgrade %s (yet).", 
		  unit_types[packet->type].name);
    return;
  }
  cost = unit_upgrade_price(pplayer, packet->type, to_unit);

  genlist_iterator_init(&myiter, &pplayer->units.list, 0);
  for(; ITERATOR_PTR(myiter);) {
    punit=(struct unit *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    if (cost > pplayer->economic.gold)
      break;
    if (punit->type == packet->type && map_get_city(punit->x, punit->y)) {
      pplayer->economic.gold -= cost;
      if (punit->hp==get_unit_type(punit->type)->hp) {
	punit->hp=get_unit_type(to_unit)->hp;
      }
      punit->type = to_unit;
      send_unit_info(0, punit, 0);
      upgraded++;
    }
  }
  if (upgraded > 0) {
    notify_player(pplayer, "Game: %d %s upgraded to %s for %d credits", 
		  upgraded, unit_types[packet->type].name, 
		  unit_types[to_unit].name, cost * upgraded);
    send_player_info(pplayer, pplayer);
  } else {
    notify_player(pplayer, "Game: No units could be upgraded.");
  }
}

/**************************************************************************
... 
**************************************************************************/

int is_sailing_unit_tile(int x, int y)
{
  struct genlist_iterator myiter;
  genlist_iterator_init(&myiter, &map_get_tile(x,y)->units.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    struct unit *punit=(struct unit *)ITERATOR_PTR(myiter);
    if (is_sailing_unit(punit)) 
      return 1;
  }
  return 0;
}

/**************************************************************************
... 
**************************************************************************/

int is_my_zoc(struct unit *myunit, int x0, int y0)
{
 
  struct unit_list *punit_list;
  struct unit *punit;
  int x,y;
  int owner=myunit->owner;
  for (x=x0-1;x<x0+2;x++)
    for (y=y0-1;y<y0+2;y++) {
      punit_list=&map_get_tile(x, y)->units;
      if((punit=unit_list_get(punit_list, 0)) && punit->owner!=owner) {
	if (is_sailing_unit(myunit)) {
	  if (is_sailing_unit_tile(x,y)) 
	    return 0;
	} else 
	  return 0;
      }
    }
  return 1;
}

/**************************************************************************
... 
**************************************************************************/

int zoc_ok_move(struct unit *punit,int x, int y)
{
  struct unit_list *punit_list;

  punit_list=&map_get_tile(x, y)->units;
    if((unit_list_get(punit_list, 0))) 
      return 1;
    if (!is_ground_unit(punit) || unit_flag(punit->type, F_IGZOC))
    return 1;
  if (map_get_city(x,y) || map_get_city(punit->x, punit->y))
      return 1;
  return (is_my_zoc(punit, punit->x, punit->y) || 
      is_my_zoc(punit, x, y)); 
}


/**************************************************************************
... 
**************************************************************************/
int can_unit_move_to_tile(struct unit *punit, int x, int y)
{
  struct tile *ptile,*ptile2;
  struct unit_list *punit_list;
  struct unit *punit2;
  int zoc;
  if(punit->activity!=ACTIVITY_IDLE && punit->activity!=ACTIVITY_GOTO)
    return 0;
  
  if(x<0 || x>=map.xsize || y<0 || y>=map.ysize)
    return 0;
  
  if(!is_tiles_adjacent(punit->x, punit->y, x, y))
    return 0;

  ptile=map_get_tile(x, y);
  ptile2=map_get_tile(punit->x, punit->y);
  if(is_ground_unit(punit)) {
    if(ptile->terrain==T_OCEAN)  {
      if(!is_transporter_with_free_space(&game.players[punit->owner], x, y))
	return 0;
    }
    if (ptile2->terrain==T_OCEAN) {
      if  (!unit_flag(punit->type, F_MARINES) && map_get_city(x,y) && map_get_city(x,y)->owner!=punit->owner) /* no marines */
	return 0;
      punit_list=&map_get_tile(x, y)->units;
      punit2 = unit_list_get(punit_list, 0);
      if (!punit2 || punit2->owner == punit->owner)
	return 1;
    }
  } 
  else if(is_sailing_unit(punit)) {
    if(ptile->terrain!=T_OCEAN && ptile->terrain!=T_UNKNOWN)
      if(!map_get_city(x, y) || map_get_city(x, y)->owner!=punit->owner)
	return 0;
  } 
  zoc = zoc_ok_move(punit, x, y);
  if (!zoc) 
    notify_player_ex(&game.players[punit->owner], punit->x, punit->y, E_NOEVENT, "Game: Can only move into your own zone of control.");
  return zoc;
}

/**************************************************************************
...
**************************************************************************/
int unit_bribe_cost(struct unit *punit)
{  
  int cost;
  struct city *capital;
  int dist;

  cost = (&game.players[punit->owner])->economic.gold+750;
  capital=find_palace(&game.players[punit->owner]);
  if (capital)
    dist=min(32, map_distance(capital->x, capital->y, punit->x, punit->y)); 
  else
    dist=32;
    if (get_government(punit->owner)==G_COMMUNISM)
      dist = min(10, dist);
  cost=(cost/(dist+2))*(get_unit_type(punit->type)->build_cost/10);
  if (unit_flag(punit->type, F_SETTLERS)) 
    cost/=2;
  return cost;
}

/***************************************************************
...
***************************************************************/
void diplomat_bribe(struct player *pplayer, struct unit *pdiplomat, struct unit *pvictim)
{
  if(pplayer->economic.gold>=pvictim->bribe_cost) {
    if(game.players[pvictim->owner].government==G_DEMOCRACY)
      notify_player_ex(pplayer, pdiplomat->x, pdiplomat->y, E_NOEVENT, 
	"Game: You can't bribe a unit from a democratic nation.");
    else {
      pplayer->economic.gold-=pvictim->bribe_cost;
      notify_player_ex(&game.players[pvictim->owner], 
		    pvictim->x, pvictim->y, E_DIPLOMATED, 
		    "Game: One of your units was bribed!");
      notify_player_ex(pplayer, pvictim->x, pvictim->y, E_NOEVENT, 
		    "Game: Succeeded in bribing the enemy unit.");
      
      create_unit(pplayer, pvictim->x, pvictim->y,
		  pvictim->type, pvictim->veteran, pdiplomat->homecity);
      wipe_unit(0, pvictim);
      pdiplomat->moves_left=0;
      send_unit_info(pplayer, pdiplomat, 0);
      send_player_info(pplayer, pplayer);
    }
  }
}

void diplomat_get_tech(struct player *pplayer, struct unit *pdiplomat, struct city  *city)
{
  int tec;
  int i;
  int j=0;
  struct player *target=&game.players[city->owner];
  if (pplayer==target)
    return;

  if (diplomat_on_tile(city->x, city->y)) {
    notify_player_ex(pplayer, city->x, city->y, E_NOEVENT, 
		     "Game: Your spy has been eliminated by a defending spy in %s.", city->name);
    notify_player_ex(target, city->x, city->y, E_DIPLOMATED,
		     "Game: A%s spy has been eliminated in %s..", n_if_vowel(get_race_name(target->race)[0]), city->name);
    wipe_unit(0, pdiplomat);
    return;
  }
  
  for (i=1;i<A_LAST;i++) {
    if (get_invention(pplayer, i)!=TECH_KNOWN && get_invention(target, i)== TECH_KNOWN) {
      j++;
    }
  }
  if (!j) 
    return;
  if (city->steal) {
    notify_player_ex(pplayer, city->x, city->y, E_NOEVENT,
		       "Game: Your diplomat was caught in the attempt of stealing technology in %s.", city->name);
    notify_player_ex(target, city->x, city->y, E_DIPLOMATED,
		     "Game: %s diplomat failed in stealing technology in %s", pplayer->name, city->name);
    return;
  }

  j=myrand(j)+1;
  for (i=1;i<A_LAST;i++) {
    if (get_invention(pplayer, i)!=TECH_KNOWN && 
	get_invention(target, i)== TECH_KNOWN) 
      j--;
    if (!j) break;
  }
  if (i==A_LAST) {
    printf("Bug in diplomat_a_tech\n");
    return;
  }
  city->steal=1;
  set_invention(pplayer, i, TECH_KNOWN);
  update_research(pplayer);
  do_tech_cost(pplayer);
  pplayer->research.researchpoints++;
  notify_player_ex(pplayer, city->x, city->y, E_NOEVENT,
		   "Game: Your diplomat stole %s from %s",
		   advances[i].name, target->name); 
  notify_player_ex(target, city->x, city->y, E_DIPLOMATED,
		   "Game: %s diplomat stole %s in %s.", 
		   pplayer->name, advances[i].name, city->name); 
  if (pplayer->research.researching==i) {
    tec=pplayer->research.researched;
    if (!choose_goal_tech(pplayer))
      choose_random_tech(pplayer);
    pplayer->research.researched=tec;
  }

  wipe_unit(0, pdiplomat);
}

int diplomat_on_tile(int x, int y)
{
  struct genlist_iterator myiter;
  genlist_iterator_init(&myiter, &map_get_tile(x, y)->units.list,0);
  for (;ITERATOR_PTR(myiter);ITERATOR_NEXT(myiter)) {
    if (unit_flag(((struct unit*)ITERATOR_PTR(myiter))->type, F_DIPLOMAT))
      return 1;
  }
  return 0;
}

void diplomat_incite(struct player *pplayer, struct unit *pdiplomat, struct city *pcity)
{
  struct player *cplayer;
  struct city *pnewcity, *pc2;
  int i;
  if (!pcity)
    return;
 
  cplayer=city_owner(pcity);
  if (cplayer==pplayer || cplayer==NULL) 
    return;
  if(game.players[cplayer->player_no].government==G_DEMOCRACY) {
      notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		       "Game: You can't subvert a city from a democratic nation.");
      return;
  }
  if (pplayer->economic.gold<pcity->incite_revolt_cost) { 
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		     "Game: You don't have enough credits to subvert %s.", pcity->name);
    
    return;
  }
  if (diplomat_on_tile(pcity->x, pcity->y)) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		     "Game: Your spy has been eliminated by a defending spy in %s.", pcity->name);
    notify_player_ex(cplayer, pcity->x, pcity->y, E_DIPLOMATED,
		     "Game: A%s spy has been eliminated in %s..", n_if_vowel(get_race_name(cplayer->race)[0]), pcity->name);
    wipe_unit(0, pdiplomat);
    return;
  }

  pplayer->economic.gold-=pcity->incite_revolt_cost;
  if (pcity->size >1) {
    pcity->size--;
    city_auto_remove_worker(pcity);
  }
  notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		   "Game: Revolt incited in %s, you now rule the city!", 
		   pcity->name);
  notify_player_ex(cplayer, pcity->x, pcity->y, E_DIPLOMATED, 
		   "Game: %s has revolted, %s influence suspected", pcity->name, get_race_name(pplayer->race));
  pnewcity=(struct city *)malloc(sizeof(struct city));
  *pnewcity=*pcity;
  remove_city(pcity);
  for (i=0;i<4;i++) {
    pc2=find_city_by_id(pnewcity->trade[i]);
    if (can_establish_trade_route(pnewcity, pc2))    
      establish_trade_route(pnewcity, pc2);
  }
  pnewcity->id=get_next_id_number();
  pnewcity->owner=pplayer->player_no;
  unit_list_init(&pnewcity->units_supported);
  city_list_insert(&pplayer->cities, pnewcity);
  map_set_city(pnewcity->x, pnewcity->y, pnewcity);
  raze_city(pcity);
  city_refresh(pnewcity);
  send_city_info(0, pnewcity, 0);
  send_player_info(pplayer, pplayer);
  wipe_unit(0, pdiplomat);
}

void diplomat_sabotage(struct player *pplayer, struct unit *pdiplomat, struct city *pcity)
{
  struct player *cplayer;
  char *prod;
  if (!pcity)
    return;
  cplayer=city_owner(pcity);
  if (cplayer==pplayer ||cplayer==NULL) return;

  if (diplomat_on_tile(pcity->x, pcity->y)) {
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		     "Game: Your spy has been eliminated by a defending spy in %s.", pcity->name);
    notify_player_ex(cplayer, pcity->x, pcity->y, E_DIPLOMATED,
		     "Game: A%s spy has been eliminated in %s.", n_if_vowel(get_race_name(cplayer->race)[0]), pcity->name);
    wipe_unit(0, pdiplomat);
    return;
  }

  switch (myrand(2)) {
  case 0:
    pcity->shield_stock=0;
    if (pcity->is_building_unit) 
      prod=unit_name(pcity->currently_building);
    else
      prod=building_name(pcity->currently_building);
    notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
    "Game: Your Diplomat succeeded destroying the production of %s in %s", 
    prod, pcity->name);
    notify_player_ex(cplayer, pcity->x, pcity->y, E_DIPLOMATED, 
		     "Game: The production of %s was destroyed in %s, %s are suspected for the sabotage.", 
		     prod, pcity->name, get_race_name_plural(cplayer->race));

    break;
  case 1:
    {
      int building;
      int i;
      for (i=0;i<10;i++) {
	building=myrand(B_LAST);
	if (city_got_building(pcity, building) 
	    && !is_wonder(building) && building!=B_PALACE) {
	  pcity->improvements[building]=0;
	  break;
	}
      }
      if (i<10) {
	notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
			 "Game: Your Diplomat destroyed %s in %s.", 
			 building_name(building), pcity->name);
	notify_player_ex(cplayer, pcity->x, pcity->y, E_DIPLOMATED,
			 "Game: The %s destroyed %s in %s.", 
			 get_race_name(cplayer->race), building_name(building), pcity->name);
      } else {
	notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
		      "Game: Your Diplomat was caught in the attempt of industrial sabotage!");
	notify_player_ex(cplayer, pcity->x, pcity->y, E_DIPLOMATED,
			 "Game: You caught a%s %s diplomat in industrial sabotage!",
		n_if_vowel(get_race_name(cplayer->race)[0]),
		get_race_name(cplayer->race));
      }
    }
    break;
  }
  send_city_info(cplayer, pcity, 1);

  wipe_unit(0, pdiplomat);
}

/***************************************************************
...
***************************************************************/
void handle_diplomat_action(struct player *pplayer, 
			    struct packet_diplomat_action *packet)
{
  struct unit *pdiplomat=unit_list_find(&pplayer->units, packet->diplomat_id);
  struct unit *pvictim=find_unit_by_id(packet->target_id);
  struct city *pcity=find_city_by_id(packet->target_id);
  if (!unit_flag(pdiplomat->type, F_DIPLOMAT)) 
    return;
  if(pdiplomat && pdiplomat->moves_left>0) {
    pdiplomat->moves_left=0;
    send_unit_info(pplayer, pdiplomat, 0);
    switch(packet->action_type) {
     case DIPLOMAT_BRIBE:
       if(pvictim && diplomat_can_do_action(pdiplomat, DIPLOMAT_BRIBE,
					  pvictim->x, pvictim->y))
	 diplomat_bribe(pplayer, pdiplomat, pvictim);
      break;
     case DIPLOMAT_SABOTAGE:
       if(pcity && diplomat_can_do_action(pdiplomat, DIPLOMAT_SABOTAGE, 
					  pcity->x, pcity->y))
	 diplomat_sabotage(pplayer, pdiplomat, pcity);
       break;
     case DIPLOMAT_EMBASSY:
      if(pcity && diplomat_can_do_action(pdiplomat, DIPLOMAT_EMBASSY, 
					 pcity->x, pcity->y)) {
	pplayer->embassy|=(1<<pcity->owner);
	send_player_info(pplayer, pplayer);
	notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
			 "Game: You have established an embassy in %s",
			 pcity->name);
	
	notify_player_ex(&game.players[pcity->owner], pcity->x, pcity->y, E_DIPLOMATED, "Game: The %s has established an embassy in %s",
		      get_race_name_plural(pplayer->race),
		      pcity->name);
      }
      wipe_unit(0, pdiplomat);
      break;
     case DIPLOMAT_INCITE:
       if(pcity && diplomat_can_do_action(pdiplomat, DIPLOMAT_INCITE, 
					  pcity->x, pcity->y))
	 diplomat_incite(pplayer, pdiplomat, pcity);
      break;
     case DIPLOMAT_STEAL:
       if(pcity && diplomat_can_do_action(pdiplomat, DIPLOMAT_STEAL, 
					  pcity->x, pcity->y))
	 diplomat_get_tech(pplayer, pdiplomat, pcity);
      break;
    }
  }
}

/***************************************************************
...
***************************************************************/
void player_restore_units(struct player *pplayer)
{
  struct genlist_iterator myiter, myiter2;
  struct unit *punit;
  struct city *pcity;
  int leonardo=0;
  genlist_iterator_init(&myiter, &pplayer->units.list, 0);

  pcity=city_list_find_id(&pplayer->cities, 
			  game.global_wonders[B_LEONARDO]);
  if ((pcity && !wonder_is_obsolete(B_LEONARDO))) 
    leonardo=1;

  for(; ITERATOR_PTR(myiter);) {
    punit=(struct unit *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    if (leonardo && can_upgrade_unittype(pplayer, punit->type)!=-1) {
      if (punit->hp==get_unit_type(punit->type)->hp) 
	punit->hp = get_unit_type(can_upgrade_unittype(pplayer, punit->type))->hp;
      notify_player(pplayer, "Game: Leonardo's workshop has upgraded %s to %s ",
		    unit_types[punit->type].name, 
		    unit_types[can_upgrade_unittype(pplayer, punit->type)].name);
      punit->type = can_upgrade_unittype(pplayer, punit->type);
      punit->veteran = 0;
      leonardo = 0;
    }

    unit_restore_hitpoints(pplayer, punit);
    unit_restore_movepoints(pplayer, punit);
    
    if(is_air_unit(punit)) {
      punit->fuel--;
      if(map_get_city(punit->x, punit->y))
	punit->fuel=get_unit_type(punit->type)->fuel;
      else {
	genlist_iterator_init(&myiter2, 
			      &map_get_tile(punit->x, punit->y)->units.list,0);
	for (;ITERATOR_PTR(myiter2);ITERATOR_NEXT(myiter2)) {
	  if (unit_flag(((struct unit*)ITERATOR_PTR(myiter2))->type,F_CARRIER))
	    punit->fuel=get_unit_type(punit->type)->fuel;
	  else if (unit_flag(punit->type, F_MISSILE) && unit_flag(((struct unit*)ITERATOR_PTR(myiter2))->type, F_SUBMARINE))
	    punit->fuel=get_unit_type(punit->type)->fuel;
	}
      }
      if(punit->fuel<=0) {
	send_remove_unit(0, punit->id);
	notify_player_ex(pplayer, punit->x, punit->y, E_UNIT_LOST, 
			 "Game: Your %s has run out of fuel",
			 unit_name(punit->type));
	wipe_unit(0, punit);
      }
    } else if (punit->type==U_TRIREME) {
      struct city *pcity;
      pcity=city_list_find_id(&pplayer->cities, 
			    game.global_wonders[B_LIGHTHOUSE]);
      if (!(pcity && !wonder_is_obsolete(B_LIGHTHOUSE))) { 
	if (!is_coastline(punit->x, punit->y) && (myrand(100) > 50)) {
	  notify_player_ex(pplayer, punit->x, punit->y, E_UNIT_LOST, 
			   "Game: Your Trireme has been lost on the high seas");
	  wipe_unit(pplayer, punit);
	}
      }
    }
  }
}


/***************************************************************
...
***************************************************************/
void unit_restore_hitpoints(struct player *pplayer, struct unit *punit)
{
  struct city *pcity;
  
  punit->hp+=hp_gain_coord(punit);
  
  pcity=city_list_find_id(&pplayer->cities, game.global_wonders[B_UNITED]);

  if(pcity && !wonder_is_obsolete(B_UNITED))
    punit->hp+=2;
    
  if(punit->hp>get_unit_type(punit->type)->hp)
    punit->hp=get_unit_type(punit->type)->hp;
}
  

/***************************************************************
...
***************************************************************/
void unit_restore_movepoints(struct player *pplayer, struct unit *punit)
{
  punit->moves_left=unit_move_rate(punit);
}


/***************************************************************
...
***************************************************************/
int hp_gain_coord(struct unit *punit)
{
  int hp=1;
  struct city *pcity;
  if (unit_on_fortress(punit))
    hp=get_unit_type(punit->type)->hp/4;
  if((pcity=game_find_city_by_coor(punit->x,punit->y))) {
    if ((city_got_barracks(pcity) && is_ground_unit(punit)) ||
	(city_got_building(pcity, B_AIRPORT) && is_air_unit(punit)) || 
	(city_got_building(pcity, B_AIRPORT) && is_heli_unit(punit)) || 
	(city_got_building(pcity, B_PORT) && is_sailing_unit(punit))) {
      hp=get_unit_type(punit->type)->hp;
    }
    else if (is_ground_unit(punit)) 
      hp=get_unit_type(punit->type)->hp/3;
    else if (is_heli_unit(punit)) {
      if (hp>1) 
	hp--;
    }
    else
      hp++; 
  }
  if(punit->activity==ACTIVITY_FORTIFY)
    hp++;
  
  return hp;
}



/**************************************************************************
... this is a crude function!!!! 
**************************************************************************/
int rate_unit(struct unit *punit, struct unit *against)
{
  int val;
  struct city *pcity=map_get_city(punit->x, punit->y);
  if(punit)
    val=get_defense_power(punit);
  else
    return 0;
  if (pcity && !unit_ignores_citywalls(against))
    val*=3;
  else if (pcity || unit_on_fortress(punit))
    val*=2;
  else if (punit->activity==ACTIVITY_FORTIFY)
    val*=1.5;
  return val*100+punit->hp;
}

/**************************************************************************
get best defending unit which is NOT owned by pplayer
**************************************************************************/
struct unit *get_defender(struct player *pplayer, struct unit *aunit, int x, int y)
{
  struct unit_list *punit_list;
  struct unit *punit;
  struct unit *bestdef;
  int bestvalue=-1;
  struct genlist_iterator myiter;

  punit_list=&map_get_tile(x, y)->units;
  if (!(punit=unit_list_get(punit_list, 0))) 
    return 0;
  if (pplayer->player_no==punit->owner)
    return 0;
  genlist_iterator_init(&myiter, &punit_list->list, 0);
  bestdef=0;
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    punit=(struct unit *)ITERATOR_PTR(myiter);
    if(unit_can_defend_here(punit) && rate_unit(punit, aunit)>bestvalue) {
      bestvalue=rate_unit(punit, aunit);
      bestdef=punit;
    }
  }
    return bestdef;
}

/**************************************************************************
...
**************************************************************************/
int get_attack_power(struct unit *punit)
{
  int power;
  power=get_unit_type(punit->type)->attack_strength*10;
  if (punit->veteran)
    power*=1.5;
  if (punit->moves_left==1)
    return power/3;
  if (punit->moves_left==2)
    return (power*2)/3;
  return power;
}

/**************************************************************************
...
**************************************************************************/
int get_defense_power(struct unit *punit)
{
  int power;
  int terra;
  if (!punit || punit->type<0 || punit->type>=U_LAST)
    abort();
  power=get_unit_type(punit->type)->defense_strength*10;
  if (punit->veteran)
    power*=1.5;
  
  terra=map_get_terrain(punit->x, punit->y);
  power=(power*get_tile_type(terra)->defense_bonus)/10;
  return power;
}

/**************************************************************************
...
**************************************************************************/
int unit_ignores_citywalls(struct unit *punit)
{
  return (unit_flag(punit->type, F_IGWALL));
}

/**************************************************************************
...
**************************************************************************/
int unit_behind_walls(struct unit *punit)
{
  struct city *pcity;
  
  if((pcity=game_find_city_by_coor(punit->x,punit->y)))
    return city_got_citywalls(pcity);
  
  return 0;
}

/**************************************************************************
...
**************************************************************************/
int unit_on_fortress(struct unit *punit)
{
  return (map_get_special(punit->x, punit->y)&S_FORTRESS);
}

int unit_behind_coastal(struct unit *punit)
{
  struct city *pcity;
  return ((pcity=game_find_city_by_coor(punit->x, punit->y)) && city_got_building(pcity, B_COASTAL));
}

int unit_behind_sam(struct unit *punit)
{
  struct city *pcity;
  return ((pcity=game_find_city_by_coor(punit->x, punit->y)) && city_got_building(pcity, B_SAM));
}

int unit_behind_sdi(struct unit *punit)
{
  struct city *pcity;
  return ((pcity=game_find_city_by_coor(punit->x, punit->y)) && city_got_building(pcity, B_SDI));
}

/**************************************************************************
...
**************************************************************************/
void maybe_make_veteran(struct unit *punit)
{
    struct city *pcity;
    if (punit->veteran) 
      return;
    pcity=city_list_find_id(&game.players[punit->owner].cities, 
			    game.global_wonders[B_SUNTZU]);
    if(pcity && !wonder_is_obsolete(B_SUNTZU)) 
      punit->veteran = 1;
    else
      punit->veteran=myrand(2);
}

/**************************************************************************
...
**************************************************************************/
void unit_versus_unit(struct unit *attacker, struct unit *defender)
{
  int attackpower=get_attack_power(attacker);
  int defensepower=get_defense_power(defender);
  if (unit_flag(defender->type, F_PIKEMEN) && unit_flag(attacker->type, F_HORSE)) 
    defensepower*=2;
  if (unit_flag(defender->type, F_AEGIS) && (is_air_unit(attacker) || is_heli_unit(attacker)))
    defensepower*=2;
  if (is_air_unit(attacker)) {
    if (unit_behind_sam(defender))
      defensepower*=2;
    if (unit_behind_sdi(defender) && unit_flag(attacker->type, F_MISSILE))
      defensepower*=2;
  } else if (is_sailing_unit(attacker)) {
    if (unit_behind_coastal(defender))
      defensepower*=2;
  } else if (!unit_ignores_citywalls(attacker) && 
	     (is_heli_unit(attacker) || is_ground_unit(attacker)) && 
	     unit_behind_walls(defender)) 
    defensepower*=3;
  if (unit_on_fortress(defender) && 
      !map_get_city(defender->x, defender->y)) 
    defensepower*=2;
  if ((defender->activity == ACTIVITY_FORTIFY || 
       map_get_city(defender->x, defender->y)) && 
      is_ground_unit(defender))
    defensepower*=1.5;
  log(LOG_DEBUG, "attack:%d, defense:%d\n", attackpower, defensepower);
  while (attacker->hp>0 && defender->hp>0) {
    if (!attackpower) {
      attacker->hp= 0; 
    } else if (!defensepower) {
      defender->hp= 0;
    } else if (myrand(attackpower+defensepower)>= defensepower) {
      defender->hp=defender->hp-get_unit_type(attacker->type)->firepower;
    } else {
      if (is_sailing_unit(defender) && map_get_city(defender->x, defender->y))
	attacker->hp=attacker->hp-1;              /* pearl harbour */
      else
	attacker->hp=attacker->hp-get_unit_type(defender->type)->firepower;

    }
  }
  if (attacker->hp<0) attacker->hp=0;
  if (defender->hp<0) defender->hp=0;

  if (attacker->hp)
    maybe_make_veteran(attacker); 
  else if (defender->hp)
    maybe_make_veteran(defender);
}

/**************************************************************************
...
**************************************************************************/
void create_unit(struct player *pplayer, int x, int y, enum unit_type_id type,
		 int make_veteran, int homecity_id)
{
  struct unit *punit;
  struct city *pcity;
  punit=(struct unit *)malloc(sizeof(struct unit));
  punit->type=type;
  punit->id=get_next_id_number();
  punit->owner=pplayer->player_no;
  punit->x=x;
  punit->y=y;
  punit->goto_dest_x=0;
  punit->goto_dest_y=0;
  
  pcity=game_find_city_by_id(homecity_id);
  punit->veteran=make_veteran;
  punit->homecity=homecity_id;
  punit->hp=get_unit_type(punit->type)->hp;
  punit->activity=ACTIVITY_IDLE;
  punit->activity_count=0;
  punit->upkeep=0;
  punit->unhappiness=0;
  punit->fuel=get_unit_type(punit->type)->fuel;
  punit->ai.control=0;
  punit->ai.ai_role = AIUNIT_NONE;
  unit_list_insert(&pplayer->units, punit);
  unit_list_insert(&map_get_tile(x, y)->units, punit);
  if (pcity)
    unit_list_insert(&pcity->units_supported, punit);
  punit->bribe_cost=unit_bribe_cost(punit);
  punit->moves_left=unit_move_rate(punit);
  send_unit_info(0, punit, 0);
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_change_homecity(struct player *pplayer, 
				 struct packet_unit_request *req)
{
  struct unit *punit;
  
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    struct city *pcity;
    if((pcity=city_list_find_id(&pplayer->cities, req->city_id))) {
      unit_list_insert(&pcity->units_supported, punit);
      
      if((pcity=city_list_find_id(&pplayer->cities, punit->homecity)))
	unit_list_unlink(&pcity->units_supported, punit);
      
      punit->homecity=req->city_id;
      send_unit_info(pplayer, punit, 0);
    }
  }
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_disband(struct player *pplayer, 
			 struct packet_unit_request *req)
{
  struct unit *punit;
  struct city *pcity;
  /* give 1/2 of the worth of the unit, to the currently builded thing 
     have to be on the location of the city_square
   */
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    if ((pcity=map_get_city(punit->x, punit->y))) {
      pcity->shield_stock+=(get_unit_type(punit->type)->build_cost/2);
      send_city_info(pplayer, pcity, 0);
    }
    wipe_unit(pplayer, punit);
  }
}


/**************************************************************************
...
**************************************************************************/
void handle_unit_build_city(struct player *pplayer, 
			    struct packet_unit_request *req)
{
  struct unit *punit;
  char *name;
  struct city *pcity;
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    if (!unit_flag(punit->type, F_SETTLERS)) {
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		    "Game: You need a settler to build a city.");
      return;
    }  
    
    if ((pcity=map_get_city(punit->x, punit->y))) {
      if (pcity->size>8) {
	notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT, 
		    "Game: Your settlers doesn't feel comfortable here.");
	return;
      }
      else {
	pcity->size++;
	  if (!add_adjust_workers(pcity))
	    auto_arrange_workers(pcity);
        wipe_unit(0, punit);
	send_city_info(pplayer, pcity, 0);
	notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		      "Game: Settlers added to aid %s in growing", 
		      pcity->name);
	return;
      }
    }
    
    if(can_unit_build_city(punit)) {
      if(!(name=get_sane_name(req->name))) {
	notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT, 
		      "Game: Let's not build a city with such a stupid name.");
	return;
      }

      send_remove_unit(0, req->unit_id);
      map_set_special(punit->x, punit->y, S_ROAD);
      send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
      create_city(pplayer, punit->x, punit->y, name);
      game_remove_unit(req->unit_id);
    } else
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		    "Game: Can't place city here.");
  }
}



/**************************************************************************
...
**************************************************************************/
void send_remove_unit(struct player *pplayer, int unit_id)
{
  int o;
  
  struct packet_generic_integer packet;

  packet.value=unit_id;

  for(o=0; o<game.nplayers; o++)           /* dests */
    if(!pplayer || &game.players[o]==pplayer)
      send_packet_generic_integer(game.players[o].conn, PACKET_REMOVE_UNIT,
				  &packet);
}

/**************************************************************************
...
**************************************************************************/
void update_unit_activities(struct player *pplayer)
{
  struct genlist_iterator myiter;
  struct unit *punit;

  genlist_iterator_init(&myiter, &pplayer->units.list, 0);
  for(; ITERATOR_PTR(myiter);) {
    punit=(struct unit *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    update_unit_activity(pplayer, punit);
  }
}


/**************************************************************************
...
**************************************************************************/
void set_unit_activity(struct unit *punit, enum unit_activity new_activity)
{
  punit->activity=new_activity;
  punit->activity_count=0;
}

/**************************************************************************
...
**************************************************************************/
void update_unit_activity(struct player *pplayer, struct unit *punit)
{
  punit->activity_count+= (get_unit_type(punit->type)->move_rate)/3;
  
   if(punit->activity==ACTIVITY_PILLAGE && punit->activity_count>=1) {
      if(map_get_special(punit->x, punit->y)&S_IRRIGATION)
	map_clear_special(punit->x, punit->y, S_IRRIGATION);
      else if(map_get_special(punit->x, punit->y)&S_RAILROAD)
	map_clear_special(punit->x, punit->y, S_RAILROAD);
      else 
	map_clear_special(punit->x, punit->y, S_ROAD);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    set_unit_activity(punit, ACTIVITY_IDLE);
   }

  if(punit->activity==ACTIVITY_POLLUTION && punit->activity_count>=3) {
    map_clear_special(punit->x, punit->y, S_POLLUTION);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    set_unit_activity(punit, ACTIVITY_IDLE);
  }

  if(punit->activity==ACTIVITY_FORTRESS && punit->activity_count>=3) {
    map_set_special(punit->x, punit->y, S_FORTRESS);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    set_unit_activity(punit, ACTIVITY_IDLE);
  }
  
  if(punit->activity==ACTIVITY_IRRIGATE && 
     punit->activity_count>=map_build_irrigation_time(punit->x, punit->y)) {
    map_irrigate_tile(punit->x, punit->y);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    punit->activity=ACTIVITY_IDLE;
    set_unit_activity(punit, ACTIVITY_IDLE);
  }

  if(punit->activity==ACTIVITY_ROAD && 
     punit->activity_count>map_build_road_time(punit->x, punit->y)) {
    map_set_special(punit->x, punit->y, S_ROAD);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    punit->activity=ACTIVITY_IDLE;
    set_unit_activity(punit, ACTIVITY_IDLE);
  }

  if(punit->activity==ACTIVITY_RAILROAD && punit->activity_count>=3) {
    map_set_special(punit->x, punit->y, S_RAILROAD);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    punit->activity=ACTIVITY_IDLE;
    handle_unit_activity_request(pplayer, punit, ACTIVITY_IDLE);
    set_unit_activity(punit, ACTIVITY_IDLE);
  }
  
  if(punit->activity==ACTIVITY_MINE && 
     punit->activity_count>=map_build_mine_time(punit->x, punit->y)) {
    map_mine_tile(punit->x, punit->y);
    send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
    set_unit_activity(punit, ACTIVITY_IDLE);
  }

  if(punit->activity==ACTIVITY_GOTO) {
    do_unit_goto(pplayer, punit);
    return;
  }
  
  if(punit->activity==ACTIVITY_IDLE && 
     map_get_terrain(punit->x, punit->y)==T_OCEAN &&
     is_ground_unit(punit))
    set_unit_activity(punit, ACTIVITY_SENTRY);

  
  
  send_unit_info(0, punit, 0);
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_info(struct player *pplayer, struct packet_unit_info *pinfo)
{
  struct unit *punit;

  punit=unit_list_find(&pplayer->units, pinfo->id);

  if(punit) {
    if(punit->ai.control) {
      punit->ai.control=0;
    }
    if(punit->activity!=pinfo->activity)
      handle_unit_activity_request(pplayer, punit, pinfo->activity);
    else if (!same_pos(punit->x, punit->y, pinfo->x, pinfo->y))
      handle_unit_move_request(pplayer, punit, pinfo->x, pinfo->y);
  }
}

/**************************************************************************
...
**************************************************************************/
void do_nuke_tile(int x, int y)
{
  struct unit_list *punit_list;
  struct city *pcity;
  punit_list=&map_get_tile(x, y)->units;
  
  while(unit_list_size(punit_list)) {
    struct unit *punit=unit_list_get(punit_list, 0);
    wipe_unit(0, punit);
  }

  if((pcity=game_find_city_by_coor(x,y))) {
    pcity->size/=2;
    auto_arrange_workers(pcity);
    send_city_info(0, pcity, 0);
  }
  else if ((map_get_terrain(x,y)!=T_OCEAN && map_get_terrain(x,y)<=T_TUNDRA) &&
           (!(map_get_special(x,y)&S_POLLUTION)) && myrand(2)) { 
    map_set_special(x,y, S_POLLUTION);
    send_tile_info(0, x, y, TILE_KNOWN);
  }
}

/**************************************************************************
...
**************************************************************************/
void do_nuclear_explosion(int x, int y)
{
  int i,j;
  for (i=0;i<3;i++)
    for (j=0;j<3;j++)
      do_nuke_tile(x+i-1,y+j-1);
}

struct city *sdi_defense_close(int owner, int x, int y)
{
  struct city *pcity;
  int lx, ly;
  for (lx=x-2;lx<x+3;lx++)
    for (ly=y-2;ly<y+3;ly++) {
      pcity=game_find_city_by_coor(lx,ly);
      if (pcity && (pcity->owner!=owner) && city_got_building(pcity, B_SDI))
	return pcity;
    }
  return NULL;

}

/**************************************************************************
...
**************************************************************************/
void handle_unit_attack_request(struct player *pplayer, struct unit *punit,
				struct unit *pdefender)
{
  int o;
  struct packet_unit_combat combat;
  struct unit *plooser, *pwinner;
  struct city *pcity;
  punit->moves_left-=3;
    
  if(punit->moves_left<0)
    punit->moves_left=0;
  

  if(punit->type==U_NUCLEAR) {
    struct packet_nuke_tile packet;
    
    packet.x=pdefender->x;
    packet.y=pdefender->y;
    if ((pcity=sdi_defense_close(punit->owner, pdefender->x, pdefender->y))) {
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT, "Game: Your Nuclear missile has been shot down by SDI, what a waste.");
      notify_player_ex(&game.players[pcity->owner], 
		       pdefender->x, pdefender->y, E_NOEVENT, 
             "Game: The nuclear attack on %s was avoided by your SDI defense",
		       pcity->name); 
      wipe_unit(0, punit);
      return;
    } 

    for(o=0; o<game.nplayers; o++)
      send_packet_nuke_tile(game.players[o].conn, &packet);
    
    do_nuclear_explosion(pdefender->x, pdefender->y);
    return;
  }
  
  unit_versus_unit(punit, pdefender);

  if (punit->hp && (pcity=map_get_city(pdefender->x, pdefender->y)) && pcity->size>1 && !city_got_citywalls(pcity) && is_ground_unit(punit)) {
    pcity->size--;
    city_auto_remove_worker(pcity);
    city_refresh(pcity);
    send_city_info(0, pcity, 0);
  }
  if (unit_flag(punit->type, F_ONEATTACK)) 
    punit->moves_left = 0;
  pwinner=(punit->hp) ?     punit : pdefender;
  plooser=(pdefender->hp) ? punit : pdefender;
    
  combat.attacker_unit_id=punit->id;
  combat.defender_unit_id=pdefender->id;
  combat.attacker_hp=punit->hp;
  combat.defender_hp=pdefender->hp;
  combat.make_winner_veteran=pwinner->veteran;
  
  for(o=0; o<game.nplayers; o++)
    if(map_get_known(punit->x, punit->y, &game.players[o]) ||
       map_get_known(pdefender->x, pdefender->y, &game.players[o]))
      send_packet_unit_combat(game.players[o].conn, &combat);
  
  if(punit==plooser) {
    notify_player_ex(&game.players[pwinner->owner], 
		     pwinner->x, pwinner->y, E_UNIT_WIN, 
		  "Game: You survived the pathetic attack from a%s %s of the %s.", 
		  n_if_vowel(*unit_name(punit->type)), 
		  unit_name(punit->type),  
		  races[game.players[plooser->owner].race].name);
    notify_player_ex(&game.players[plooser->owner], 
		     pdefender->x, pdefender->y, E_NOEVENT, 
		     "Game: Your attack failed!");
    wipe_unit(pplayer, plooser);
  }
  else {
    kill_unit(pplayer, plooser);
    notify_player_ex(&game.players[pwinner->owner], 
		     punit->x, punit->y, E_NOEVENT, 
		     "Game: Your attack was succesful!");
  }
  if (pwinner == punit && unit_flag(punit->type, F_MISSILE)) {
    wipe_unit(pplayer, pwinner);
    return;
  }
  send_unit_info(0, pwinner, 0);
}

/**************************************************************************
...
**************************************************************************/
int find_a_unit_type()
{
  int num;
  
  num = 2;
  if (game.global_advances[A_CHIVALRY])
    num = 3;
  if (game.global_advances[A_GUNPOWDER]) 
    num = 4;

  switch (myrand(num)) {
  case 0:
    return U_HORSEMEN;
  case 1:
    return U_LEGION;
  case 2:
    return U_CHARIOT;
  case 3:
    return U_KNIGHTS;
  case 4:
    return U_MUSKETEERS;
  default:
    return U_HORSEMEN;
  }
}

/**************************************************************************
...
**************************************************************************/
int can_unit_attack_tile(struct unit *punit, int dest_x, int dest_y)
{
  struct unit *pdefender;
  int fromtile=map_get_terrain(punit->x, punit->y);
  int totile=map_get_terrain(dest_x, dest_y);

  if(!is_military_unit(punit))
    return 0;
  
  pdefender=get_defender(&game.players[punit->owner], punit, dest_x, dest_y);
    /*only fighters can attack planes, except for city attacks */
  if (!unit_flag(punit->type, F_FIGHTER) && is_air_unit(pdefender) && !map_get_city(dest_x, dest_y)) {
    return 0;
  }
  /* can't attack with ground unit from ocean */
  if(fromtile==T_OCEAN && is_ground_unit(punit) && !unit_flag(punit->type, F_MARINES)) {
    return 0;
  }

  if(fromtile!=T_OCEAN && totile==T_OCEAN && is_ground_unit(punit)) {
    return 0;
  }
  
  /* Shore bombardement */
  else if (fromtile==T_OCEAN && is_sailing_unit(punit) && totile!=T_OCEAN)
    return (get_attack_power(punit)>0);
  return 1;
}


/**************************************************************************
...
**************************************************************************/
void handle_unit_enter_hut(struct unit *punit)
{
  struct player *pplayer=&game.players[punit->owner];
  if (is_air_unit(punit))
    return;
  map_get_tile(punit->x, punit->y)->special^=S_HUT;
  
  send_tile_info(0, punit->x, punit->y, TILE_KNOWN);
  
  switch (myrand(12)) {
  case 0:
    notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT, 
		     "Game: You found 25 credits.");
    pplayer->economic.gold+=25;
    break;
  case 1:
  case 2:
  case 3:
    notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		  "Game: You found 50 credits.");
    pplayer->economic.gold+=50;
    break;
  case 4:
    notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		     "Game: You found 100 credits"); 
    pplayer->economic.gold+=100;
    break;
  case 5:
  case 6:
  case 7:
/*this function is hmmm a hack */
    notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		  "Game: You found ancient scrolls of wisdom."); 
    {
      int res=pplayer->research.researched;
      int wasres=pplayer->research.researching;

      choose_random_tech(pplayer);
 
      pplayer->research.researchpoints++;
      set_invention(pplayer, pplayer->research.researching, TECH_KNOWN);
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		       "Game: You gain knowledge about %s.", 
		       advances[pplayer->research.researching].name); 
      remove_obsolete_buildings(pplayer);
      
      if (get_invention(pplayer,wasres)==TECH_KNOWN) {
	if (!choose_goal_tech(pplayer))
	  choose_random_tech(pplayer);
	pplayer->research.researched=res;
      }  else {
	pplayer->research.researched=res;
	pplayer->research.researching=wasres;
      }
     do_tech_cost(pplayer);
    }
    break;
  case 8:
  case 9:
    notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		     "Game: A band of friendly mercenaries joins your cause.");
    create_unit(pplayer, punit->x, punit->y, find_a_unit_type(), 0, punit->homecity);
    break;
  case 10:
    if (in_city_radius(pplayer,punit->x, punit->y))
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		       "Game: An abandoned village is here.");
    else {
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		       "Game: Your unit has been cowardly slaughtered by a band of barbarians");
      wipe_unit(pplayer, punit);
    }
    break;
  case 11:
    if (is_ok_city_spot(punit->x, punit->y)) {
      map_set_special(punit->x, punit->y, S_ROAD);
      send_tile_info(0, punit->x, punit->y, TILE_KNOWN);

      create_city(pplayer, punit->x, punit->y, get_city_name_suggestion(pplayer));
    } else {
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		   "Game: Friendly nomads are impressed by you, and join you");
      create_unit(pplayer, punit->x, punit->y, U_SETTLERS, 0, punit->homecity);
    }
    break;
  }
  send_player_info(pplayer, pplayer);
}

/**************************************************************************
...
**************************************************************************/
int try_move_unit(struct unit *punit, int dest_x, int dest_y) 
{
  if (myrand(1+map_move_cost(punit, dest_x, dest_y))>punit->moves_left && punit->moves_left<unit_move_rate(punit)) {
    punit->moves_left=0;
    send_unit_info(&game.players[punit->owner], punit, 0);
  }
  return punit->moves_left;
}
/**************************************************************************
...
**************************************************************************/
int do_airline(struct unit *punit, int x, int y)
{
  struct city *city1, *city2;
  
  if (!punit->moves_left)
    return 0;
  if (!(city1=map_get_city(punit->x, punit->y))) 
    return 0;
  if (!(city2=map_get_city(x,y)))
    return 0;
  if (city1->owner != city2->owner) 
    return 0;
  if (city1->airlift + city2->airlift < 2) 
    return 0;
  city1->airlift=0;
  city2->airlift=0;
  punit->moves_left = 0;
  punit->x = x;
  punit->y = y;
  send_unit_info(&game.players[punit->owner], punit, 0);
  notify_player_ex(&game.players[punit->owner], punit->x, punit->y, E_NOEVENT,
		   "Game: unit transported succesfully.");
  return 1;
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_move_request(struct player *pplayer, struct unit *punit,
			      int dest_x, int dest_y)
{
  int unit_id;
  struct unit *pdefender;
  struct unit_list cargolist;
  
  unit_id=punit->id;
  if (do_airline(punit, dest_x, dest_y))
    return;
  pdefender=get_defender(pplayer, punit, dest_x, dest_y);

  if(pdefender && pdefender->owner!=punit->owner) {
    if(can_unit_attack_tile(punit,dest_x , dest_y)) {
      if(punit->moves_left<=0)
	notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
			 "Game: This unit has no moves left.");
      else
	  handle_unit_attack_request(pplayer, punit, pdefender);
    } else
      notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
		       "Game: You can't attack there.");
  }
  else if(can_unit_move_to_tile(punit, dest_x, dest_y) && try_move_unit(punit, dest_x, dest_y)) {
    int src_x, src_y;
    struct city *pcity;
    
    if((pcity=map_get_city(dest_x, dest_y))) {
      if ((pcity->owner!=punit->owner && (is_air_unit(punit) || 
					  !is_military_unit(punit)))) {
	notify_player_ex(pplayer, punit->x, punit->y, E_NOEVENT,
			 "Game: Only ground troops can take over a city.");
	return;
      }
    }

    if(!unit_list_find(&pplayer->units, unit_id))
      return; /* diplomat or caravan action killed unit */

    
    /* light the squares the unit is entering */
    light_square(pplayer, dest_x, dest_y, 
		 get_unit_type(punit->type)->vision_range);
    
    /* ok now move the unit */

    src_x=punit->x;
    src_y=punit->y;

    unit_list_unlink(&map_get_tile(src_x, src_y)->units, punit);

    if(get_transporter_capacity(punit)) {
      struct genlist_iterator myiter;
      transporter_cargo_to_unitlist(punit, &cargolist);
      genlist_iterator_init(&myiter, &cargolist.list, 0);
      for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
	struct unit *pcarried=(struct unit *)ITERATOR_PTR(myiter);
	pcarried->x=dest_x;
	pcarried->y=dest_y;
	send_unit_info(0, pcarried, 1);
      }
    }
      
    punit->x=dest_x;
    punit->y=dest_y;
    
    if((punit->moves_left-=map_move_cost(punit, dest_x, dest_y))<0)
      punit->moves_left=0;
    send_unit_info(0, punit, 1);
    
    unit_list_insert(&map_get_tile(dest_x, dest_y)->units, punit);

    if(get_transporter_capacity(punit)) {
      move_unit_list_to_tile(&cargolist, punit->x, punit->y);
      genlist_unlink_all(&cargolist.list); 
    }
      
    /* ok entered new tile */
    
    if(pcity)
      handle_unit_enter_city(pplayer, pcity);

    if((map_get_tile(dest_x, dest_y)->special&S_HUT))
      handle_unit_enter_hut(punit);
  }
}

/**************************************************************************
...
**************************************************************************/
void raze_city(struct city *pcity)
{
  int i;
  pcity->improvements[B_PALACE]=0;
  for (i=0;i<B_LAST;i++) {
    if (city_got_building(pcity, i) && !is_wonder(i) 
	&& (myrand(100) < game.razechance)) {
      pcity->improvements[i]=0;
    }
  }
  if (pcity->shield_stock > 0)
    pcity->shield_stock=0;
  /*  advisor_choose_build(pcity);  we add the civ bug here :)*/
}

/**************************************************************************
...
**************************************************************************/

void get_a_tech(struct player *pplayer, struct player *target)
{
  int tec;
  int i;
  int j=0;
  for (i=0;i<A_LAST;i++) {
    if (get_invention(pplayer, i)!=TECH_KNOWN && 
	get_invention(target, i)== TECH_KNOWN) {
      j++;
    }
  }
  if (!j) 
    return;
  j=myrand(j)+1;
  for (i=0;i<A_LAST;i++) {
    if (get_invention(pplayer, i)!=TECH_KNOWN && 
	get_invention(target, i)== TECH_KNOWN) 
      j--;
    if (!j) break;
  }
  if (i==A_LAST) 
    printf("Bug in get_a_tech\n");
  set_invention(pplayer, i, TECH_KNOWN);
  update_research(pplayer);
  do_tech_cost(pplayer);
  pplayer->research.researchpoints++;
  notify_player(pplayer, "Game: You acquired %s from %s",
		advances[i].name, target->name); 
  notify_player(target, "Game: %s discovered %s in the city.", pplayer->name, 
		advances[i].name); 
  if (pplayer->research.researching==i) {
    tec=pplayer->research.researched;
    if (!choose_goal_tech(pplayer))
      choose_random_tech(pplayer);
    pplayer->research.researched=tec;
  }
}

int build_points_left(struct city *pcity)
{
 return (building_value(pcity->currently_building) - pcity->shield_stock);
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_help_build_wonder(struct player *pplayer, 
				   struct packet_unit_request *req)
{
  struct unit *punit;
  
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    struct city *pcity_dest;
    
    pcity_dest=find_city_by_id(req->city_id);
    
    if(unit_flag(punit->type, F_CARAVAN) && pcity_dest &&
       unit_can_help_build_wonder(punit, pcity_dest)) {
      pcity_dest->shield_stock+=50;
      if (build_points_left(pcity_dest) < 0) {
	pcity_dest->shield_stock = building_value(pcity_dest->currently_building);
      }

      notify_player_ex(pplayer, pcity_dest->x, pcity_dest->y, E_NOEVENT,
		  "Game: Your %s help building the %s in %s. (%d remaining)", 
		       unit_name(punit->type),
		       get_improvement_type(pcity_dest->currently_building)->name,
		       pcity_dest->name, 
		       build_points_left(pcity_dest)
		       );

      wipe_unit(0, punit);
      send_player_info(pplayer, pplayer);
      send_city_info(pplayer, pcity_dest, 0);
    }
  }
}

/**************************************************************************
...
**************************************************************************/

void handle_unit_establish_trade(struct player *pplayer, 
				 struct packet_unit_request *req)
{
  struct unit *punit;
  
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    
    struct city *pcity_homecity, *pcity_dest;
    
    pcity_homecity=city_list_find_id(&pplayer->cities, punit->homecity);
    pcity_dest=find_city_by_id(req->city_id);
    
    if(unit_flag(punit->type, F_CARAVAN) && pcity_homecity && pcity_dest && 
       is_tiles_adjacent(punit->x, punit->y, pcity_dest->x, pcity_dest->y) &&
       can_establish_trade_route(pcity_homecity, pcity_dest)) {
      int revenue;

      revenue=establish_trade_route(pcity_homecity, pcity_dest);
      notify_player_ex(pplayer, pcity_dest->x, pcity_dest->y, E_NOEVENT,
		       "Game: Your %s has arrived in %s, and revenues account to %d in gold.", 
		       unit_name(punit->type), pcity_dest->name, revenue);
      wipe_unit(0, punit);
      pplayer->economic.gold+=revenue;
      send_player_info(pplayer, pplayer);
      send_city_info(pplayer, pcity_homecity, 0);
      send_city_info(pplayer, pcity_dest, 0);
    }
  }
}

int can_place_partisan(int x, int y) 
{
  struct tile *ptile;
  ptile = map_get_tile(x, y);
  return (!map_get_city(x, y) && !unit_list_size(&ptile->units) && map_get_terrain(x,y) != T_OCEAN && map_get_terrain(x, y) < T_UNKNOWN); 
}

int place_partisan(struct city *pcity)
{
  int x,y;
  int count;

  count = 0;
  for (x = pcity->x -2; x < pcity->x + 3; x++)
    for (y = pcity->y - 2; y < pcity->y + 3; y++) {
      if (can_place_partisan(x, y)) 
	count++;
    }
  if (count) {
    count = myrand(count) + 1;
    for (x = pcity->x - 2; x < pcity->x + 3; x++)
      for (y = pcity->y - 2; y < pcity->y + 3; y++) {
	if (can_place_partisan(x, y))
	  count--;
	if (!count) {
	  create_unit(&game.players[pcity->owner], x, y, U_PARTISAN, 0, 0);
	  return 1;
	}
      }
    printf("Bug in place partisan");
    return 0;
  } else 
    return 0;
}

void make_partisans(struct city *pcity)
{
  int partisans;
  if (!game.global_advances[A_GUERILLA] || pcity->original != pcity->owner)
    return;
  if (get_invention(city_owner(pcity), A_COMMUNISM) != TECH_KNOWN 
      && get_invention(city_owner(pcity), A_GUNPOWDER) != TECH_KNOWN)
    return;
  if (get_government(pcity->owner)!=G_DEMOCRACY
      && get_government(pcity->owner)!=G_COMMUNISM) 
    return;
  
  partisans = myrand(1 + pcity->size/2) + 1;
  if (partisans > 8) 
    partisans = 8;
  
  while (partisans && place_partisan(pcity)) partisans--;
 
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_enter_city(struct player *pplayer, struct city *pcity)
{
  int i;
  int coins;
  struct city *pc2;
  struct player *cplayer;
  if(pplayer->player_no!=pcity->owner) {
    struct city *pnewcity;
    cplayer=&game.players[pcity->owner];
    pcity->size--;
    if (pcity->size<1) {
      notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT,
		       "Game: You destroy %s completely.", pcity->name);
      notify_player_ex(cplayer, pcity->x, pcity->y, E_CITY_LOST, 
		    "Game: %s has been destroyed by %s", 
		    pcity->name, pplayer->name);
      remove_city(pcity);
      return;
    }
    city_auto_remove_worker(pcity);
    get_a_tech(pplayer, cplayer);
    coins=cplayer->economic.gold;
    coins=myrand((coins/20)+1)+(coins*(pcity->size))/200;
    pplayer->economic.gold+=coins;
    cplayer->economic.gold-=coins;
    send_player_info(cplayer, cplayer);
    if (pcity->owner != pplayer->player_no) {
      notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		       "Game: You conquer %s, your lootings accumulate to %d gold", 
		       pcity->name, coins);
      notify_player_ex(cplayer, pcity->x, pcity->y, E_CITY_LOST, 
		       "Game: %s conquered %s and looted %d gold from the city.",
		       pplayer->name, pcity->name, coins);
    } else {
      notify_player_ex(pplayer, pcity->x, pcity->y, E_NOEVENT, 
		       "Game: You have liberated %s!! lootings accumulate to %d gold", 		       pcity->name, coins);
      
      notify_player_ex(cplayer, pcity->x, pcity->y, E_CITY_LOST, 
		       "Game: %s liberated %s and looted %d gold from the city.",		       pplayer->name, pcity->name, coins);
    }

    pnewcity=(struct city *)malloc(sizeof(struct city));
    make_partisans(pcity);
    *pnewcity=*pcity;
    remove_city(pcity);
    for (i=0;i<4;i++) {
      pc2=find_city_by_id(pnewcity->trade[i]);
      if (can_establish_trade_route(pnewcity, pc2))    
	establish_trade_route(pnewcity, pc2);
    }
    /* now set things up for the new owner */
    
    pnewcity->id=get_next_id_number();
    pnewcity->owner=pplayer->player_no;

    unit_list_init(&pnewcity->units_supported);
    city_list_insert(&pplayer->cities, pnewcity);
    
    map_set_city(pnewcity->x, pnewcity->y, pnewcity);
    raze_city(pnewcity);
    city_refresh(pnewcity);
    send_city_info(0, pnewcity, 0);
    send_player_info(pplayer, pplayer);
  }
}


/**************************************************************************
...
**************************************************************************/
void handle_unit_auto_request(struct player *pplayer, 
			      struct packet_unit_request *req)
{
  struct unit *punit;
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    punit->ai.control=1;
    send_unit_info(pplayer, punit, 0);
  }
}
 
/**************************************************************************
...
**************************************************************************/
void handle_unit_activity_request(struct player *pplayer, struct unit *punit, 
				  enum unit_activity new_activity)
{
  if((punit->moves_left>0 || punit->activity==ACTIVITY_GOTO) && 
     can_unit_do_activity(punit, new_activity)) {
    punit->activity=new_activity;
    punit->activity_count=0;

     send_unit_info(0, punit, 0);
  }
}

/**************************************************************************
...
**************************************************************************/
void handle_unit_unload_request(struct player *pplayer, 
				struct packet_unit_request *req)
{
  struct unit *punit;
  if((punit=unit_list_find(&pplayer->units, req->unit_id))) {
    struct genlist_iterator myiter;
    genlist_iterator_init(&myiter, &map_get_tile(punit->x, punit->y)->units.list, 0);
    for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
      struct unit *punit2=(struct unit *)ITERATOR_PTR(myiter);
      if(punit2->activity==ACTIVITY_SENTRY) {
	set_unit_activity(punit2, ACTIVITY_IDLE);
	send_unit_info(0, punit2, 0);
      }
    }
  }

}



/**************************************************************************
this is a highlevel routine
**************************************************************************/
void wipe_unit(struct player *dest, struct unit *punit)
{
  if(get_transporter_capacity(punit) && 
     map_get_terrain(punit->x, punit->y)==T_OCEAN) {
    struct genlist_iterator myiter;
    struct unit_list list;
    
    transporter_cargo_to_unitlist(punit, &list);
    genlist_iterator_init(&myiter, &list.list, 0);
    for(; ITERATOR_PTR(myiter);) {
      struct unit *punit2=(struct unit *)ITERATOR_PTR(myiter);
      ITERATOR_NEXT(myiter);
	send_remove_unit(0, punit2->id);
	game_remove_unit(punit2->id);
    }
  }
  
  send_remove_unit(0, punit->id);
  game_remove_unit(punit->id);
}


/**************************************************************************
this is a highlevel routine
the unit has been killed in combat => all other units on the
tile dies unless ...
**************************************************************************/
void kill_unit(struct player *dest, struct unit *punit)
{
  int klaf;
  struct city *pcity = map_get_city(punit->x, punit->y);
  klaf=unit_list_size(&(map_get_tile(punit->x, punit->y)->units));
  if( (pcity) || 
      (map_get_special(punit->x, punit->y)&S_FORTRESS) || 
      (klaf == 1)) {
    if (pcity) 
      notify_player_ex(&game.players[punit->owner], 
		       punit->x, punit->y, E_UNIT_LOST,
		       "Game: You lost a%s %s under an attack from %s, in %s",
		       n_if_vowel(get_unit_type(punit->type)->name[0]),
		       get_unit_type(punit->type)->name, dest->name, pcity->name);
    else
      notify_player_ex(&game.players[punit->owner], 
		       punit->x, punit->y, E_UNIT_LOST,
		       "Game: You lost a%s %s under an attack from %s",
		       n_if_vowel(get_unit_type(punit->type)->name[0]),
		       get_unit_type(punit->type)->name, dest->name);
    send_remove_unit(0, punit->id);
    game_remove_unit(punit->id);
  }  else {
    struct genlist_iterator myiter;
      notify_player_ex(&game.players[punit->owner], 
		       punit->x, punit->y, E_UNIT_LOST, 
		       "Game: You lost %d units under an attack from %s",
		       klaf, dest->name);
      genlist_iterator_init(&myiter, 
			  &map_get_tile(punit->x, punit->y)->units.list, 0);
    
      for(; ITERATOR_PTR(myiter); ) {
	struct unit *punit2=(struct unit *)ITERATOR_PTR(myiter);
	ITERATOR_NEXT(myiter);
	notify_player_ex(&game.players[punit2->owner], 
			 punit->x, punit->y, E_UNIT_LOST,
			 "Game: You lost a%s %s under an attack from %s",
			 n_if_vowel(get_unit_type(punit2->type)->name[0]),
			 get_unit_type(punit2->type)->name, dest->name);
	send_remove_unit(0, punit2->id);
	game_remove_unit(punit2->id);
      }
  } 
}

/**************************************************************************
...
**************************************************************************/

void send_unit_info(struct player *dest, struct unit *punit, int dosend)
{
  int o;
  struct packet_unit_info info;

  info.id=punit->id;
  info.owner=punit->owner;
  info.x=punit->x;
  info.y=punit->y;
  info.homecity=punit->homecity;
  info.veteran=punit->veteran;
  info.type=punit->type;
  info.movesleft=punit->moves_left;
  info.hp=punit->hp;
  info.activity=punit->activity;
  info.activity_count=punit->activity_count;
  info.unhappiness=punit->unhappiness;
  info.upkeep=punit->upkeep;
  info.bribe_cost=punit->bribe_cost;
  info.ai=punit->ai.control;
    
  for(o=0; o<game.nplayers; o++)           /* dests */
    if(!dest || &game.players[o]==dest)
      if(dosend || map_get_known(info.x, info.y, &game.players[o]))
	 send_packet_unit_info(game.players[o].conn, &info);
}
/**************************************************************************
...
**************************************************************************/

int auto_settler_do_goto(struct player *pplayer, struct unit *punit, int x, int y)
{
  punit->goto_dest_x=map_adjust_x(x);
  punit->goto_dest_y=map_adjust_y(y);
  punit->activity=ACTIVITY_GOTO;
  punit->activity_count=0;
  send_unit_info(0, punit, 0);
  do_unit_goto(pplayer, punit);
  return 1;
}

/**************************************************************************
...
**************************************************************************/
int is_already_assigned(struct unit *myunit, struct player *pplayer, int x, int y)
{
  struct genlist_iterator myiter;
  genlist_iterator_init(&myiter, &map_get_tile(x, y)->units.list, 0);
  x=map_adjust_x(x);
  y=map_adjust_y(y);
  if (same_pos(myunit->x, myunit->y, x, y))
    return 0;
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    struct unit *punit=(struct unit *)ITERATOR_PTR(myiter);
    if (myunit==punit) continue;
    if (punit->owner!=pplayer->player_no)
      return 1;
    if (unit_flag(punit->type, F_SETTLERS) && unit_flag(myunit->type, F_SETTLERS))
      return 1;
  }
  genlist_iterator_init(&myiter, &pplayer->units.list, 0);
  for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
    struct unit *punit=(struct unit *)ITERATOR_PTR(myiter);
    if (myunit==punit) continue;
    if (punit->owner!=pplayer->player_no)
      return 1;
    if (same_pos(punit->goto_dest_x, punit->goto_dest_y, x, y) && unit_flag(punit->type, F_SETTLERS) && unit_flag(myunit->type,F_SETTLERS) && punit->activity==ACTIVITY_GOTO)
      return 1;
    if (same_pos(punit->goto_dest_x, punit->goto_dest_y, x, y) && !unit_flag(myunit->type, F_SETTLERS) && punit->activity==ACTIVITY_GOTO)
      return 1;
  }
  return 0;
}

/**************************************************************************
...
**************************************************************************/

int benefit_pollution(struct player *pplayer, int x, int y)
{
  if (map_get_special(x, y) & S_POLLUTION)
    return 80;
  return 0;
}

/**************************************************************************
...
**************************************************************************/
int benefit_road(struct player *pplayer, int x, int y)
{
  int t;
  int f=0;
  enum tile_special_type spec_t = map_get_special(x, y);
  enum tile_terrain_type tile_t = map_get_terrain(x, y);
  if ((spec_t & S_RAILROAD))
    return 0;
  if (get_invention(pplayer, A_RAILROAD) == TECH_KNOWN) 
    f=1;
  if ((spec_t & S_ROAD) && !f) 
    return 0; 
 if(is_unit_activity_on_tile(ACTIVITY_ROAD, x, y))
   return 0;
 if(is_unit_activity_on_tile(ACTIVITY_RAILROAD, x, y))
   return 0;
 
  switch (tile_t) {
  case T_DESERT:
  case T_GRASSLAND:
  case T_PLAINS:
    t=3;
    if (f || spec_t & S_SPECIAL)
      t+=2;
    break;
  case T_RIVER:
    if (f || get_invention(pplayer, A_BRIDGE) == TECH_KNOWN)
      t=3;
    else
      t=0;
    break;
  case T_OCEAN:
    t=0;
    break;
  case T_MOUNTAINS:
    if (f || (spec_t & S_SPECIAL))
      t=5;
    else
      t=1;
    break;
  default:
    if (spec_t & S_SPECIAL)
      t=3;
    else
      t=1;
    break;
  }
  return (t);
}
/**************************************************************************
...
**************************************************************************/
int benefit_mine(struct player *pplayer, int x, int y)
{
  enum tile_special_type spec_t=map_get_special(x, y);
  enum tile_terrain_type tile_t=map_get_terrain(x, y);
  if ((spec_t & S_MINE))
    return 0;
 if(is_unit_activity_on_tile(ACTIVITY_MINE, x, y))
   return 0;

  switch (tile_t) {
  case T_HILLS:
    if (spec_t & S_SPECIAL) /* prod specials are vital */ 
      return 20;
    else
      return 3;
    break;
  case T_MOUNTAINS:
    if (spec_t & S_SPECIAL) /* prod specials are vital */ 
      return 10;
    else
      return 2;
  default:
    return 0;
  }
}

/**************************************************************************
...
***************************************************************************/
int benefit_irrigate(struct player *pplayer, int x, int y)
{
  enum tile_special_type spec_t=map_get_special(x, y);
  enum tile_terrain_type tile_t=map_get_terrain(x, y);
  if ((spec_t & S_IRRIGATION))
    return 0;
  if (!is_water_adjacent_to_tile(x, y))
    return 0;
 if(is_unit_activity_on_tile(ACTIVITY_IRRIGATE, x, y))
   return 0;

  switch (tile_t) {
  case T_DESERT:
  case T_PLAINS:
  case T_GRASSLAND:
  case T_RIVER:
    return ((get_food_tile_bc(x, y)+1)*2);
  case T_JUNGLE:
  case T_SWAMP:
    if ((spec_t & S_SPECIAL)) /* Maybe this should have other priority? */
      return 0;
    return 3;
  default:
    return 0;
  }
}

/**************************************************************************
...
**************************************************************************/
int ai_calc_pollution(struct unit *punit, struct player *pplayer, int x, int y)
{
  if (is_already_assigned(punit, pplayer, x, y))
      return 0;
  return benefit_pollution(pplayer, x, y);
}

/**************************************************************************
...
**************************************************************************/
int ai_calc_mine(struct unit *punit, struct player *pplayer, int x, int y)
{
  if (is_already_assigned(punit, pplayer, x, y))
      return 0;
  if (!in_city_radius(pplayer, x, y)) 
    return 0;
  if (is_worked_here(x, y))
    return 4*benefit_mine(pplayer, x, y);
  else
    return 3*benefit_mine(pplayer, x, y); 
}

/**************************************************************************
...
**************************************************************************/
int ai_calc_road(struct unit *punit, struct player *pplayer, int x, int y)
{
  if (is_already_assigned(punit, pplayer, x, y))
      return 0;
  if (map_get_city(x, y))
    return 0;
  if (!in_city_radius(pplayer, x, y)) {
    if (benefit_road(pplayer, x, y) > 0) /* build up infrastructure */
      return 1;
    else
      return 0;
  }
  if (is_worked_here(x, y))
    return benefit_road(pplayer, x, y)*8;
  return benefit_road(pplayer, x, y)*4;
}

/*************************************************************************
...
**************************************************************************/
int get_food_tile_bc(int xp, int yp)
{
  int f;
  enum tile_special_type spec_t=map_get_special(xp,yp);
  enum tile_terrain_type tile_t=map_get_terrain(xp,yp);
    if (spec_t & S_SPECIAL) 
    f=get_tile_type(tile_t)->food_special;
  else
    f=get_tile_type(tile_t)->food;
    return f;
}

int is_ok_city_spot(int x, int y)
{
  int dx, dy;
  int i;
  struct city *pcity;
  struct genlist_iterator myiter;
  switch (map_get_terrain(x,y)) {
  case T_OCEAN:
  case T_UNKNOWN:
  case T_MOUNTAINS:
  case T_FOREST:
  case T_HILLS:
  case T_ARCTIC:
  case T_DESERT:
  case T_JUNGLE:
  case T_SWAMP:
  case T_TUNDRA:
  case T_LAST:
    return 0;
  case T_GRASSLAND:
  case T_PLAINS:
  case T_RIVER:
    break;
  default:
    break;
  }
  for (i = 0; i < game.nplayers; i++) {
    genlist_iterator_init(&myiter, &(&game.players[i])->cities.list, 0);
    for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
      pcity=(struct city *)ITERATOR_PTR(myiter);
      if (map_distance(x, y, pcity->x, pcity->y)<=8) { 
        dx=make_dx(pcity->x, x);
        dy=make_dy(pcity->y, y);
        if (dx<=5 && dy<5)
          return 0;
        if (dx<5 && dy<=5)
          return 0;
      }
    }
  }
  return 1;
}

int in_city_radius(struct player *pplayer, int x, int y)
{
  int dx,dy;
  struct city *pcity;
  struct genlist_iterator myiter;
   genlist_iterator_init(&myiter, &pplayer->cities.list, 0);
    for(; ITERATOR_PTR(myiter); ITERATOR_NEXT(myiter)) {
      pcity=(struct city *)ITERATOR_PTR(myiter);
      if (map_distance(x, y, pcity->x, pcity->y)<=4) { 
        dx=make_dx(pcity->x, x);
        dy=make_dy(pcity->y, y);
        if (dx<=2 && dy<2)
          return 1;
        if (dx<2 && dy<=2)
          return 1;
      }
    }
    return 0;
}

int ai_calc_irrigate(struct unit *punit, struct player *pplayer, int x, int y)
{
  if (map_get_city(x, y))
    return 0;
  if (is_already_assigned(punit, pplayer, x, y))
      return 0;
  if (get_government(pplayer->player_no) < G_MONARCHY) 
    return 0;
  if (!in_city_radius(pplayer, x, y)) 
    return 0;
  if (is_worked_here(x, y))
    return 4*benefit_irrigate(pplayer, x, y);
  else
    return 3*benefit_irrigate(pplayer, x, y);
}

int dist_mod(int dist, int val)
{
  if (dist==0) return 14*val;
  if (dist==1) return 12*val;
  if (dist<4) return 10*val;
  if (dist<8) return 6*val;
  if (dist<15) return 3*val;
  return ((2*val * (100-dist))/100);
}

int make_dy(int y1, int y2)
{
  int dy=y2-y1;
  if (dy<0) dy=-dy;
  return dy;
}

int make_dx(int x1, int x2)
{
  int tmp;
  x1=map_adjust_x(x1);
  x2=map_adjust_x(x2);
  if(x1>x2)
    tmp=x1, x1=x2, x2=tmp;

  return MIN(x2-x1, map.xsize-x2+x1);
}

/********************************************************************
...
*********************************************************************/
void auto_settler_findwork(struct player *pplayer, struct unit *punit) 
{
  int gx,gy;
  int co=map_get_continent(punit->x, punit->y);
  int t=0;
  int v=0;
  int v2;
  int x, y,z;
  gx=-1;
  gy=-1;
  for (x=0;x<map.xsize;x++)
    for (y=0;y<map.ysize;y++) 
      if (map_get_continent(x, y)==co) {
	z=map_distance(x,y, punit->x, punit->y);
	v2=dist_mod(z, ai_calc_irrigate(punit, pplayer, x, y));
	if (v2>v) {
	  t=ACTIVITY_IRRIGATE;
	  v=v2; gx=x; gy=y;
	}

	v2=dist_mod(z, ai_calc_road(punit, pplayer, x, y));
	if (v2>v) {
	  if (map_get_special(punit->x, punit->y) & S_ROAD)
	    t=ACTIVITY_RAILROAD;
	  else
	    t=ACTIVITY_ROAD;
	  v=v2; gx=x; gy=y;
	}

	v2=dist_mod(z, ai_calc_mine(punit, pplayer, x, y));
	if (v2>v) {
	  t=ACTIVITY_MINE;
	  v=v2; gx=x; gy=y;
	}

	v2=dist_mod(z, ai_calc_pollution(punit, pplayer, x, y));
	if (v2>v) {
	  t=ACTIVITY_POLLUTION;
	  v=v2; gx=x; gy=y;
	}
      }
  if (same_pos(gx, gy, punit->x, punit->y)) {
    set_unit_activity(punit, t);
    send_unit_info(0, punit, 0);
    return;
  }
  if (gx!=-1 && gy!=-1) 
    auto_settler_do_goto(pplayer, punit,gx, gy);
  else 
    punit->ai.control=0;
}

/**************************************************************************
...
**************************************************************************/
void auto_settlers_player(struct player *pplayer) 
{
  struct unit *punit;
  struct genlist_iterator myiter;
  genlist_iterator_init(&myiter, &pplayer->units.list, 0);
  for(; ITERATOR_PTR(myiter);) {
    punit=(struct unit *)ITERATOR_PTR(myiter);
    ITERATOR_NEXT(myiter);
    if (punit->ai.control) {
      if(punit->activity == ACTIVITY_SENTRY && 
	 !map_get_terrain(punit->x, punit->y) == T_OCEAN &&
	 is_ground_unit(punit))
	set_unit_activity(punit, ACTIVITY_IDLE);
      if (punit->activity == ACTIVITY_IDLE)
	auto_settler_findwork(pplayer, punit);
      if (punit->ai.control && punit->moves_left && punit->activity == ACTIVITY_IDLE) 
	/* fix for the lost turn */
	auto_settler_findwork(pplayer, punit);
    }
  }
}

void auto_settlers()
{
  int i;
  for (i = 0; i < game.nplayers; i++) {
    auto_settlers_player(&game.players[i]);
  }
}


